/*
 * @(#)BasicProgressBarUI.java	1.37 98/04/11
 * 
 * Copyright (c) 1997 Sun Microsystems, Inc. All Rights Reserved.
 * 
 * This software is the confidential and proprietary information of Sun
 * Microsystems, Inc. ("Confidential Information").  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sun.
 * 
 * SUN MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SUN SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 * 
 */

package com.sun.java.swing.plaf.basic;

import java.awt.*;
import com.sun.java.swing.*;
import com.sun.java.swing.event.*;
import com.sun.java.swing.plaf.*;
import java.io.Serializable;


/**
 * A Basic L&F implementation of ProgressBarUI.
 * <p>
 * Warning: serialized objects of this class will not be compatible with
 * future swing releases.  The current serialization support is appropriate
 * for short term storage or RMI between Swing1.0 applications.  It will
 * not be possible to load serialized Swing1.0 objects with future releases
 * of Swing.  The JDK1.2 release of Swing will be the compatibility
 * baseline for the serialized form of Swing objects.
 *
 * @version 1.37 04/11/98
 * @author Michael C. Albers
 */
public class BasicProgressBarUI extends ProgressBarUI
    implements ChangeListener, Serializable {

    private int cachedPercent;
    private static final Dimension PREFERRED_INNER_HORIZONTAL = new Dimension(146, 12);
    private static final Dimension PREFERRED_INNER_VERTICAL = new Dimension(12, 146);
    protected static int cellLength;
    protected static int cellSpacing;


    public static ComponentUI createUI(JComponent x) {
	return new BasicProgressBarUI();
    }

    public void installUI(JComponent c) {
	installDefaults(c);
	installListeners(c);
    }

    public void uninstallUI(JComponent c) {
	uninstallListeners(c);
    }
  
    protected void installDefaults(JComponent c) {
	c.setOpaque(true);
 	LookAndFeel.installBorder(c,"ProgressBar.border");	
	LookAndFeel.installColorsAndFont(c, "ProgressBar.background",
					 "ProgressBar.foreground",
					 "ProgressBar.font");
	cellLength = ((Integer)UIManager.get("ProgressBar.cellLength")).intValue();
	cellSpacing = ((Integer)UIManager.get("ProgressBar.cellSpacing")).intValue();
    }
    
    protected void installListeners(JComponent c) {
	((JProgressBar)c).addChangeListener(this);
    }	

    protected void uninstallListeners(JComponent c) {
	((JProgressBar)c).removeChangeListener(this);
    }
	
    protected void uninstallDefaults(JComponent c) {
 	LookAndFeel.uninstallBorder(c);	
    }

    public Dimension getPreferredInnerHorizontal() {
	return PREFERRED_INNER_HORIZONTAL;
    }

    public Dimension getPreferredInnerVertical() {
	return PREFERRED_INNER_VERTICAL;
    }

    public int getCachedPercent() {
	return cachedPercent;
    }

    public void setCachedPercent(int cachedPercent) {
	this.cachedPercent = cachedPercent;
    }

    /**
     * This determines the amount of the progress bar that should be filled
     * based on the percent done gathered from the model. This is a common
     * operation so it was abstracted out. It assumes that your progress bar
     * is linear. That is, if you are making a circular progress indicator,
     * you will want to override this method.
     */
    public int getAmountFull(JComponent c) {
	int amountFull = 0;
	JProgressBar progressBar = (JProgressBar)c;
	BoundedRangeModel model = progressBar.getModel();
	int width = c.getWidth();
	int height = c.getHeight();
	Insets b = progressBar.getInsets(); // effectively the border
	int x = b.left;
	int y = b.top;

	long span = model.getMaximum() - model.getMinimum();
	if (span != 0) {
	    double bigValue = model.getValue();
	    double fractionComplete = bigValue / span;
	    if (progressBar.getOrientation() == JProgressBar.HORIZONTAL) {
	        double bigWidth = width - (b.left + b.right);
	        amountFull = (int)(bigWidth * fractionComplete);
	    } else {
	        double bigHeight = height - (b.top + b.bottom);
	        amountFull = (int)(bigHeight * fractionComplete);
	    }
	}
	return amountFull;
    }

    /**
     * All purpose paint method that should do the right thing for almost
     * all linear progress bars. By setting a few values in the defaults
     * table, things should work just fine to paint your progress bar.
     * Naturally, override this if you are making a circular or
     * semi-circular progress bar.
     */
    public void paint(Graphics g, JComponent c) {
	JProgressBar progressBar = (JProgressBar)c;
	BoundedRangeModel model = progressBar.getModel();

	Dimension size = progressBar.getSize(); //total size
	Insets i = progressBar.getInsets(); // area for border
	Rectangle barRect = new Rectangle(size); // area to draw progress

	barRect.x += i.left;
	barRect.y += i.top;
	barRect.width -= (i.right + barRect.x);
	barRect.height -= (i.bottom + barRect.y);

	int current;
	int increment = cellLength + cellSpacing; // a cell and its spacing
	int amountFull = getAmountFull(c); // amount of progress to draw

	g.setColor(c.getForeground());
	if (progressBar.getOrientation() == JProgressBar.HORIZONTAL) {
	    // the largest number to draw a cell at
	    int max = (barRect.x + amountFull) - cellLength;
	    
	    // draw the cells
	    for (current = barRect.x; current <= max; current += increment) {
		g.fillRect(current, barRect.y, cellLength, barRect.height);
	    }

	} else {
	    // the smallest number to draw a cell at
	    //  that is, the number at the top
	    int min = barRect.height - amountFull;

	    // draw the cells
	    // note that this has a certain inefficiency at 100% where
	    //  it will draw outside of it's actual area.
	    for (current = barRect.height; current >= min;
		 current -= increment)
	    {
		g.fillRect(barRect.x, current, barRect.width, cellLength);
	    }
	}
    }

    public Dimension getPreferredSize(JComponent c) {
	Dimension	size;
	JProgressBar	progressBar = (JProgressBar)c;
	Insets		border = progressBar.getInsets();

	if (progressBar.getOrientation() == JProgressBar.HORIZONTAL) {
	    size = new Dimension(getPreferredInnerHorizontal());
	} else {
	    size = new Dimension(getPreferredInnerVertical());
	}

	size.width += border.left + border.right;
	size.height += border.top + border.bottom;
	return size;
    }

    public Dimension getMinimumSize(JComponent c) {
	Dimension pref = getPreferredSize(c);
	JProgressBar progressBar = (JProgressBar)c;
	if (progressBar.getOrientation() == JProgressBar.HORIZONTAL) {
	    pref.width = 0;
	} else {
	    pref.height = 0;
	}
	return pref;
    }

    public Dimension getMaximumSize(JComponent c) {
	Dimension pref = getPreferredSize(c);
	JProgressBar progressBar = (JProgressBar)c;
	if (progressBar.getOrientation() == JProgressBar.HORIZONTAL) {
	    pref.width = Short.MAX_VALUE;
	} else {
	    pref.height = Short.MAX_VALUE;
	}
	return pref;
    }


    //
    // Change Events
    //
    public void stateChanged(ChangeEvent e) {
	JProgressBar bar = (JProgressBar)e.getSource();
	BoundedRangeModel model = bar.getModel();
	int newRange = model.getMaximum() - model.getMinimum();
	int newPercent;
	int oldPercent = getCachedPercent();

	if (newRange > 0) {
	    newPercent = (100 * model.getValue()) / newRange;
	} else {
	    newPercent = 0;
	}

	if (newPercent != oldPercent) {
	    setCachedPercent(newPercent);
	    bar.repaint();
	}
    }
}
