/*
 * @(#)BasicOptionPaneUI.java	1.18 98/02/02
 * 
 * Copyright (c) 1997 Sun Microsystems, Inc. All Rights Reserved.
 * 
 * This software is the confidential and proprietary information of Sun
 * Microsystems, Inc. ("Confidential Information").  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sun.
 * 
 * SUN MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SUN SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 * 
 */

package com.sun.java.swing.plaf.basic;

import com.sun.java.swing.border.Border;
import com.sun.java.swing.border.EmptyBorder;
import com.sun.java.swing.*;
import com.sun.java.swing.event.*;
import com.sun.java.swing.plaf.ComponentUI;
import com.sun.java.swing.plaf.OptionPaneUI;
import java.awt.*;
import java.awt.event.*;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.io.Serializable;


/**
 * Provides the basic look and feel for a JOptionPane.
 * <p>
 * Warning: serialized objects of this class will not be compatible with
 * future swing releases.  The current serialization support is appropriate
 * for short term storage or RMI between Swing1.0 applications.  It will
 * not be possible to load serialized Swing1.0 objects with future releases
 * of Swing.  The JDK1.2 release of Swing will be the compatibility
 * baseline for the serialized form of Swing objects.
 * 
 * @version 1.18 02/02/98
 * @author James Gosling
 * @author Scott Violet
 */
public class BasicOptionPaneUI extends AbstractOptionPaneUI implements PropertyChangeListener
{
    /** Minimum width for the Component. */
    public static final int       MIN_WIDTH = 262;
    /** Minimum height for the Component. */
    public static final int       MIN_HEIGHT = 90;

    /** Option names for YES_NO_OPTION. */
    public static final String[] yesNoOptions = { "Yes", "No" };
    /** Options names for YES_NO_CANCEL_OPTION. */
    public static final String[] yesNoCancelOptions = { "Yes", "No", "Cancel"};
    /** Default option names, when none are supplied. */
    public static final String[] defaultOptions = { "OK" };
    /** Option names for OK_CANCEL_OPTION. */
    public static final String[] okCancelOptions = { "OK", "Cancel" };


    /** JOptionPane that the reciever is providing the look and feel for. */
    protected JOptionPane         optionPane;
    /** JComponent provide for input if optionPane.getWantsInput() returns
     * true. */
    protected JComponent          inputComponent;

    protected PropertyChangeListener propListener;


    /**
      * Creates a new BasicOptionPaneUI instance.
      */
    public static ComponentUI createUI(JComponent x) {
	return new BasicOptionPaneUI();
    }

    /**
      * Installs the reciever as the L&F for the passed in JOptionPane
      */
    public void installUI(JComponent c) {
	optionPane = (JOptionPane)c;

        installDefaults(c);

	Dimension            ourMin = getMinimumOptionPaneSize();
	if(ourMin != null)
	    optionPane.setMinimumSize(ourMin);
	validateComponent();

	c.setOpaque(true);
        installListeners(c);        
    }


    /**
      * Removes the receiver from the L&F controller of the passed in split
      * pane.
      */
    public void uninstallUI(JComponent c) {
	if((JOptionPane)c == optionPane) {
	    emptyContainer(optionPane);
            uninstallListeners(c);
            uninstallDefaults(c);
	    optionPane = null;
	}
    }

    protected void installDefaults(JComponent c) {
        LookAndFeel.installColorsAndFont(c, "OptionPane.background", 
                                         "OptionPane.foreground", "OptionPane.font");
	LookAndFeel.installBorder(c, "OptionPane.border");
    }

    protected void uninstallDefaults(JComponent c) {
	LookAndFeel.uninstallBorder(c);
    }

    protected void installListeners(JComponent c) {
        if ((propListener = createPropertyChangeListener(c)) != null) {
            optionPane.addPropertyChangeListener(propListener);
        }
    }

    protected void uninstallListeners(JComponent c) {
        if (propListener != null) {
            optionPane.removePropertyChangeListener(propListener);
            propListener = null;
        }
    }

    protected PropertyChangeListener createPropertyChangeListener(JComponent c) {
        return this;
    }

    /**
     * If the source of the PropertyChangeEvent <code>e</code> equals the
     * optionPane and is one of the ICON_PROPERTY, MESSAGE_PROPERTY,
     * OPTIONS_PROPERTY or INITIAL_VALUE_PROPERTY,
     * validateComponent is invoked.
     */
    public void propertyChange(PropertyChangeEvent e) {
	if(e.getSource() == optionPane) {
	    String         changeName = e.getPropertyName();

	    if(changeName.equals(JOptionPane.OPTIONS_PROPERTY) ||
	       changeName.equals(JOptionPane.INITIAL_VALUE_PROPERTY) ||
	       changeName.equals(JOptionPane.ICON_PROPERTY) ||
	       changeName.equals(JOptionPane.MESSAGE_TYPE_PROPERTY) ||
	       changeName.equals(JOptionPane.OPTION_TYPE_PROPERTY) ||
	       changeName.equals(JOptionPane.MESSAGE_PROPERTY) ||
	       changeName.equals(JOptionPane.SELECTION_VALUES_PROPERTY) ||
	       changeName.equals(JOptionPane.INITIAL_SELECTION_VALUE_PROPERTY) ||
	       changeName.equals(JOptionPane.WANTS_INPUT_PROPERTY))
		validateComponent();
	}
    }


    /**
     * Returns the icon from the JOptionPane the reciever is providing
     * the look and feel for, or the default icon as returned from
     * getDefaultIcon.
     */
    public Icon getIcon() {
	Icon      mIcon = (optionPane == null ? null : optionPane.getIcon());

	if(mIcon == null && optionPane != null)
	    mIcon = getIconForType(optionPane.getMessageType());
	return mIcon;
    }

    /**
     * Returns the icon to use for the passed in type.
     */
    public Icon getIconForType(int messageType) {
	if(messageType < 0 || messageType > 3)
	    return null;
	switch(messageType) {
	case 0:
	    return UIManager.getIcon("OptionPane.errorIcon");
	case 1:
	    return UIManager.getIcon("OptionPane.informationIcon");
	case 2:
	    return UIManager.getIcon("OptionPane.warningIcon");
	case 3:
	    return UIManager.getIcon("OptionPane.questionIcon");
	}
	return null;
    }

    /**
     * Returns the maximum number of characters to place on a line.
     * Default is to return Integer.MAX_VALUE. Concrete implementations
     * may want to return a value that means something.
     */
    public int getMaxCharactersPerLineCount() {
	return optionPane.getMaxCharactersPerLineCount();
    }

    /**
     * Returns the message to display from the JMessagePane the receiver is
     * providing the look and feel for.
     */
    public Object getMessage() {
	inputComponent = null;
	if(optionPane != null) {
	    if(optionPane.getWantsInput()) {
		/* Create a user comopnent to capture the input. If the
		   selectionValues are non null the component and there
		   are < 20 values it'll be a combobox, if non null and
		   >= 20, it'll be a list, otherwise it'll be a textfield. */
		Object             message = optionPane.getMessage();
		Object[]           sValues = optionPane.getSelectionValues();
		Object             inputValue = optionPane
		                           .getInitialSelectionValue();
		JComponent         toAdd;

		if(sValues != null) {
		    if(sValues.length < 20) {
			JComboBox            cBox = new JComboBox();

			for(int counter = 0, maxCounter = sValues.length;
			    counter < maxCounter; counter++)
			    cBox.addItem(sValues[counter]);
			if(inputValue != null)
			    cBox.setSelectedItem(inputValue);
			inputComponent = cBox;
			toAdd = cBox;
		    }
		    else {
			JList                list = new JList(sValues);
			JScrollPane          sp = new JScrollPane(list);

			list.setVisibleRowCount(10);
			if(inputValue != null)
			    list.setSelectedValue(inputValue, true);
			list.addMouseListener(new ListSelectionListener());
			toAdd = sp;
			inputComponent = list;
		    }
		}
		else {
		    JTextField         tf = new JTextField(20);

		    if(inputValue != null)
			tf.setText(inputValue.toString());
		    tf.addActionListener(new TextFieldActionListener());
		    toAdd = inputComponent = tf;
		}

		/* Construct the new message, validateComonent can handle
		   descending arrays, so it isn't a problem to just create
		   another array. */
		Object[]           newMessage;

		if(message == null) {
		    newMessage = new Object[1];
		    newMessage[0] = toAdd;
		}
		else {
		    newMessage = new Object[2];
		    newMessage[0] = message;
		    newMessage[1] = toAdd;
		}
		return newMessage;
	    }
	    return optionPane.getMessage();
	}
	return null;
    }

    /**
     * Returns the buttons to display from the JOptionPane the receiver is
     * providing the look and feel for. If the JOptionPane has options
     * set, they will be provided, otherwise if the optionType is
     * YES_NO_OPTION, yesNoOptions is returned, if the type is
     * YES_NO_CANCEL_OPTION yesNoCancelOptions is returned, otherwise
     * defaultButtons are returned.
     */
    public Object[] getButtons() {
	if(optionPane != null) {
	    Object[]           suppliedOptions = optionPane.getOptions();

	    if(suppliedOptions == null) {
		int            type = optionPane.getOptionType();

		if(type == JOptionPane.YES_NO_OPTION)
		    return BasicOptionPaneUI.yesNoOptions;
		else if(type == JOptionPane.YES_NO_CANCEL_OPTION)
		    return BasicOptionPaneUI.yesNoCancelOptions;
		else if(type == JOptionPane.OK_CANCEL_OPTION)
		    return BasicOptionPaneUI.okCancelOptions;
		return BasicOptionPaneUI.defaultOptions;
	    }
	    return suppliedOptions;
	}
	return null;
    }

    /**
     * Returns the JOptionPane the receiver is providing the look and feel
     * for.
     */
    public Container getContainer() {
	return optionPane;
    }

    /**
     * Returns the initial index into the buttons to select. The index
     * is calculated from the initial value from the JOptionPane and
     * options of the JOptionPane or 0.
     */
    public int getInitialIndex() {
	if(optionPane != null) {
	    Object             iv = optionPane.getInitialValue();
	    Object[]           options = optionPane.getOptions();

	    if(options == null) {
		return 0;
	    }
	    else if(iv != null) {
		for(int counter = options.length - 1; counter >= 0; counter--){
		    if(options[counter].equals(iv))
			return counter;
		}
	    }
	}
	return -1;
    }

    /**
     * Sets the input value in the option pane the receiver is providing
     * the look and feel for based on the value in the inputComponent.
     */
    protected void resetInputValue() {
	if(inputComponent != null && (inputComponent instanceof JTextField))
	    optionPane.setInputValue(((JTextField)inputComponent).getText());
	else if(inputComponent != null &&
		(inputComponent instanceof JComboBox))
	    optionPane.setInputValue(((JComboBox)inputComponent)
				     .getSelectedItem());
	else if(inputComponent != null)
	    optionPane.setInputValue(((JList)inputComponent)
				     .getSelectedValue());
    }

    /**
     * Messaged when a JButton as created from validateComponent is clicked
     * Invokes setValue on the JOptionPane with the appropriate value.
     * <p>
     * If you are creating your own look and feel and subclassing this
     * be sure to set the value to an Integer value representing
     * YES_OPTION, NO_OPTION or CANCEL_OPTION.
     */
    public void createdButtonFired(int buttonIndex) {
	if(optionPane != null) {
	    Object[]           options = optionPane.getOptions();

	    if(options == null) {
		int            messageType = optionPane.getOptionType();

		if(inputComponent != null &&
		   (messageType == JOptionPane.YES_NO_OPTION ||
		    messageType == JOptionPane.YES_NO_CANCEL_OPTION ||
		    messageType == JOptionPane.OK_CANCEL_OPTION) &&
		   buttonIndex == 0)
		    resetInputValue();
		if(messageType == JOptionPane.OK_CANCEL_OPTION &&
		   buttonIndex == 1)
		    optionPane.setValue(new Integer(2));
		else
		    optionPane.setValue(new Integer(buttonIndex));
	    }
	    else
		optionPane.setValue(options[buttonIndex]);
	}
    }

    /**
     * Returns the minimum size the option pane should be. Primarily
     * provided for subclassers wishin to offer a different minimum size.
     */
    public Dimension getMinimumOptionPaneSize() {
	return new Dimension(BasicOptionPaneUI.MIN_WIDTH,
			     BasicOptionPaneUI.MIN_HEIGHT);
    }

    /**
     * If c is the JOptionPane the reciever is contained in, the preferred
     * size that is returned is the maximum of the preferred size of
     * the LayoutManager for the JOptionPane, and
     * <code>getMinimumOptionPaneSize</code>.
     */
    public Dimension getPreferredSize(JComponent c) {
	if((JOptionPane)c == optionPane) {
	    Dimension            ourMin = getMinimumOptionPaneSize();
	    LayoutManager        lm = c.getLayout();

	    if(lm != null) {
		Dimension         lmSize = lm.preferredLayoutSize(c);

		if(ourMin != null)
		    return new Dimension
			(Math.max(lmSize.width, ourMin.width),
			 Math.max(lmSize.height, ourMin.height));
		return lmSize;
	    }
	    return ourMin;
	}
	return null;
    }

    /**
     * Messages getPreferredSize.
     */
    public Dimension getMinimumSize(JComponent c) {
	return getPreferredSize(c);
    }

    /**
     * Messages getPreferredSize.
     */
    public Dimension getMaximumSize(JComponent c) {
	return getPreferredSize(c);
    }

    /**
     * Returns a copy of AbstractOptionPaneUI.defaultInsets.
     */
    public Insets getInsets(JComponent c) {
	return null;
    }

    /**
     * Returns true, basic L&F wants all the buttons to have the same
     * width.
     */
    public boolean getSizeButtonsToSameWidth() {
	return true;
    }

    /**
     * If inputComponent is non-null, the focus is requested on that,
     * otherwise super is messaged.
     */
    public void selectInitialValue() {
	if(inputComponent != null)
	    inputComponent.requestFocus();
	else
	    super.selectInitialValue();
    }

    //
    // Classed used when optionPane.getWantsInput returns true.
    //

    /**
     * Listener when a JList is created to handle input from the user.
     */
    private class ListSelectionListener extends MouseAdapter implements
	Serializable
    {
	public void mousePressed(MouseEvent e) {
	    if(e.getClickCount() == 2) {
		JList     list = (JList)e.getSource();
		int       index = list.locationToIndex(e.getPoint());

		optionPane.setInputValue(list.getModel().getElementAt(index));
	    }
	}
    }

    /**
     * Listener when a JTextField is created to handle input from the user.
     */
    private class TextFieldActionListener implements ActionListener,
	Serializable
    {
	public void actionPerformed(ActionEvent e) {
	    optionPane.setInputValue(((JTextField)e.getSource()).getText());
	}
    }
}
