/*
 * @(#)BasicFileChooserUI.java	1.4 98/04/14
 * 
 * Copyright (c) 1997 Sun Microsystems, Inc. All Rights Reserved.
 * 
 * This software is the confidential and proprietary information of Sun
 * Microsystems, Inc. ("Confidential Information").  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sun.
 * 
 * SUN MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SUN SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 * 
 */

package com.sun.java.swing.plaf.basic;

import com.sun.java.swing.*;
import com.sun.java.swing.preview.*;
import com.sun.java.swing.preview.filechooser.*;
import com.sun.java.swing.event.*;
import com.sun.java.swing.plaf.*;
import com.sun.java.swing.plaf.basic.*;
import java.awt.*;
import java.awt.event.*;
import java.beans.*;
import java.io.File;
import java.io.IOException;
import java.util.*;

/**
 * Basic L&F implementation of a FileChooser.  
 *
 * @version %i% %g%
 * @author Jeff Dinkins
 */
public abstract class BasicFileChooserUI extends FileChooserUI implements PropertyChangeListener {

    /* FileView icons */
    protected Icon directoryIcon = null;
    protected Icon fileIcon = null;
    protected Icon computerIcon = null;
    protected Icon hardDriveIcon = null;
    protected Icon floppyDriveIcon = null;

    protected Icon newFolderIcon = null;
    protected Icon upFolderIcon = null;
    protected Icon homeFolderIcon = null;
    protected Icon listViewIcon = null;
    protected Icon detailsViewIcon = null;

    protected String saveButtonText = null;
    protected String openButtonText = null;
    protected String cancelButtonText = null;
    protected String updateButtonText = null;
    protected String helpButtonText = null;

    protected static String saveButtonToolTipText = null;
    protected static String openButtonToolTipText = null;
    protected static String cancelButtonToolTipText = null;
    protected static String updateButtonToolTipText = null;
    protected static String helpButtonToolTipText = null;

    // Some of the more generic FileChooser functions
    private Action approveSelectionAction = new ApproveSelectionAction();
    private Action cancelSelectionAction = new CancelSelectionAction();
    private Action updateAction = new UpdateAction();
    private Action newFolderAction = new NewFolderAction();
    private Action goHomeAction = new GoHomeAction();
    private Action changeToParentDirectoryAction = new ChangeToParentDirectoryAction();

    private JFileChooser filechooser = null;

    private AcceptAllFileFilter acceptAllFileFilter = new AcceptAllFileFilter();

    /* FileView type descriptions */
    // PENDING(jeff) - I18N - these should be localized
    protected String fileDescription = "Generic File";
    protected String directoryDescription = "Directory";

    protected BasicDirectoryModel model = null;

    protected BasicFileView fileView = new BasicFileView();

    protected JPanel accessoryPanel = null;

    /**
     * Abstract methods
     */
    public abstract String getFileName();
    public abstract void setFileName(String filename);
    public abstract String getDirectoryName();
    public abstract void setDirectoryName(String dirname);
    public abstract void ensureFileIsVisible(File f);
    public abstract void rescanCurrentDirectory();
    public abstract void propertyChange(PropertyChangeEvent e);


    //
    // ComponentUI Interface Implementation methods
    //

    /**
     * returns a BasicFileChooserUI
     */
    // public static ComponentUI createUI(JComponent c) {
    //     return new BasicFileChooserUI((JFileChooser)c);
    // }

    public BasicFileChooserUI(JFileChooser b) {
    }

    public void installUI(JComponent c) {
	accessoryPanel = new JPanel(new BorderLayout());
	filechooser = (JFileChooser) c;

	loadIcons();
	loadStrings();
	createModel();
	installComponents();
	addListeners();
    }

    public void uninstallUI(JComponent c) {
	filechooser.removeAll();

	uninstallListeners();

	if(accessoryPanel != null) {
	    accessoryPanel.removeAll();
	}

	filechooser = null;
	accessoryPanel = null;
    }

    protected void uninstallListeners() {
	if(getFileChooser() != null) {
	    getFileChooser().removeAll();
	    getFileChooser().removePropertyChangeListener(this);
	    getFileChooser().removePropertyChangeListener(model);
	}
    }

    protected void loadIcons() {
	directoryIcon    = UIManager.getIcon("FileView.directoryIcon");
	fileIcon         = UIManager.getIcon("FileView.fileIcon");
	computerIcon     = UIManager.getIcon("FileView.computerIcon");
	hardDriveIcon    = UIManager.getIcon("FileView.hardDriveIcon");
	floppyDriveIcon  = UIManager.getIcon("FileView.floppyDriveIcon");

	newFolderIcon = UIManager.getIcon("FileChooser.newFolderIcon");
	upFolderIcon = UIManager.getIcon("FileChooser.upFolderIcon");
	homeFolderIcon = UIManager.getIcon("FileChooser.homeFolderIcon");
	detailsViewIcon = UIManager.getIcon("FileChooser.detailsViewIcon");
	listViewIcon = UIManager.getIcon("FileChooser.listViewIcon");

    }

    protected void loadStrings() { 
	saveButtonText = UIManager.getString("FileChooser.saveButtonText");
	openButtonText = UIManager.getString("FileChooser.openButtonText");
	cancelButtonText = UIManager.getString("FileChooser.cancelButtonText");
	updateButtonText = UIManager.getString("FileChooser.updateButtonText");
	helpButtonText = UIManager.getString("FileChooser.helpButtonText");

	saveButtonToolTipText = UIManager.getString("FileChooser.saveButtonToolTipText");
	openButtonToolTipText = UIManager.getString("FileChooser.openButtonToolTipText");
	cancelButtonToolTipText = UIManager.getString("FileChooser.cancelButtonToolTipText");
	updateButtonToolTipText = UIManager.getString("FileChooser.updateButtonToolTipText");
	helpButtonToolTipText = UIManager.getString("FileChooser.helpButtonToolTipText");
    }

    protected void createModel() {
	model = new BasicDirectoryModel(getFileChooser());
    }

    public BasicDirectoryModel getModel() {
	return model;
    }

    protected void addListeners() {
	getFileChooser().addPropertyChangeListener(model);
	getFileChooser().addPropertyChangeListener(this);
    }

    
    public JFileChooser getFileChooser() {
	return filechooser;
    }

    public JPanel getAccessoryPanel() {
	return accessoryPanel;
    }


    public ListSelectionListener createListSelectionListener() {
	return new SelectionListener();
    }

    protected class DoubleClickListener extends MouseAdapter {
	JList list;
	public  DoubleClickListener(JList list) {
	    this.list = list;
	}
	
	public void mouseClicked(MouseEvent e) {
	    if (e.getClickCount() == 2) {
		int index = list.locationToIndex(e.getPoint());
		if(index >= 0) {
		    File f = (File) list.getModel().getElementAt(index);
		    if(getFileChooser().isTraversable(f)) {
			list.clearSelection();
			getFileChooser().setCurrentDirectory(f);
		    } else {
			getFileChooser().approveSelection();
		    }
		}
	    }
	}
    }

    protected MouseListener createDoubleClickListener(JList list) {
	return new DoubleClickListener(list);
    }

    protected class SelectionListener implements ListSelectionListener {
	public void valueChanged(ListSelectionEvent e) {
	    if(!e.getValueIsAdjusting()) {
		JList list = (JList) e.getSource();
		File f = (File) list.getSelectedValue();
		if(f != null && ((getFileChooser().isDirectorySelectionEnabled()) ||
				 !getFileChooser().isTraversable(f))) {
		    getFileChooser().setSelectedFile(f);
		}
	    }
	}
    }
    

    /**
     * Returns the default accept all file filter
     */
    public FileFilter getAcceptAllFileFilter() {
	return acceptAllFileFilter;
    }


    public FileView getFileView() {
	return fileView;
    }


    /**
     * Returns the title of this dialog
     */
    public String getDialogTitle() {
	return getApproveButtonText();
    }

    public abstract void installComponents();

    public String getApproveButtonText() {
	String buttonText = getFileChooser().getApproveButtonText();
	if(buttonText != null) {
	    return buttonText;
	}

	if(getFileChooser().getDialogType() == JFileChooser.OPEN_DIALOG) {
	    return openButtonText;
	} else if(getFileChooser().getDialogType() == JFileChooser.SAVE_DIALOG) {
	    return saveButtonText;
	} 
	return null;
    }

    public String getApproveButtonToolTipText() {
	String tooltipText = getFileChooser().getApproveButtonToolTipText();
	if(tooltipText != null) {
	    return tooltipText;
	}

	if(getFileChooser().getDialogType() == JFileChooser.OPEN_DIALOG) {
	    return openButtonToolTipText;
	} else if(getFileChooser().getDialogType() == JFileChooser.SAVE_DIALOG) {
	    return saveButtonToolTipText;
	} 
	return null;
    }

    public Action getNewFolderAction() {
	return newFolderAction;
    }

    public Action getGoHomeAction() {
	return goHomeAction;
    }

    public Action getChangeToParentDirectoryAction() {
	return changeToParentDirectoryAction;
    }

    public Action getApproveSelectionAction() {
	return approveSelectionAction;
    }

    public Action getCancelSelectionAction() {
	return cancelSelectionAction;
    }

    public Action getUpdateAction() {
	return updateAction;
    }


    /**
     * Creates a new folder.
     */
    protected class NewFolderAction extends AbstractAction {
	protected NewFolderAction() {
	    super("New Folder");
	}
	public void actionPerformed(ActionEvent e) {
	    File currentDirectory = getFileChooser().getCurrentDirectory();
	    File newFolder = null;
	    try {
		newFolder = getFileChooser().getFileSystemView().createNewFolder(currentDirectory);
	    } catch (IOException exc) {
		JOptionPane.showMessageDialog(getFileChooser(), "Error creating new folder: " + exc,
					      "Error Creating New Folder", JOptionPane.ERROR_MESSAGE);
		return;
	    } 
	    getFileChooser().rescanCurrentDirectory();
	    getFileChooser().ensureFileIsVisible(newFolder);
	}
    }

    /**
     * Acts on the "home" key event or equivalent event.
     */
    protected class GoHomeAction extends AbstractAction {
	protected GoHomeAction() {
	    super("Go Home");
	}
	public void actionPerformed(ActionEvent e) {
	    getFileChooser().setCurrentDirectory(null);
	}
    }

    protected class ChangeToParentDirectoryAction extends AbstractAction {
	protected ChangeToParentDirectoryAction() {
	    super("Go Up");
	}
	public void actionPerformed(ActionEvent e) {
	    getFileChooser().changeToParentDirectory();
	}
    }

    /**
     * Responds to an Open or Save request
     */
    protected class ApproveSelectionAction extends AbstractAction {
	public void actionPerformed(ActionEvent e) {
	    String filename = getFileName();
	    FileSystemView fs = getFileChooser().getFileSystemView();
	    File dir = getFileChooser().getCurrentDirectory();

	    if(filename != null) {
		// Remove whitespace from beginning and end of filename
		filename = filename.trim();
	    }
	    if(filename != null && !filename.equals("")) {
		// check for directory change action
		File selectedFile = fs.createFileObject(filename);
		if(!selectedFile.isAbsolute()) {
		   selectedFile = fs.createFileObject(dir, filename);
		}
		if(selectedFile.isDirectory() && getFileChooser().isTraversable(selectedFile)
		                              && !getFileChooser().isDirectorySelectionEnabled())
		{
		    getFileChooser().setCurrentDirectory(selectedFile);
		    return;
		} else if ((!selectedFile.isDirectory() && getFileChooser().isFileSelectionEnabled()) || 
			   (selectedFile.isDirectory() && getFileChooser().isDirectorySelectionEnabled())) {
		    getFileChooser().setSelectedFile(selectedFile);
		}
	    } 
	    getFileChooser().approveSelection();
	}
    }


    /**
     * Responds to a cancel request.
     */
    protected class CancelSelectionAction extends AbstractAction {
	public void actionPerformed(ActionEvent e) {
	    getFileChooser().cancelSelection();
	}
    } 

    /**
     * Rescans the files in the current directory
     */
    protected class UpdateAction extends AbstractAction {
	public void actionPerformed(ActionEvent e) {
	    getFileChooser().setCurrentDirectory(
		getFileChooser().getFileSystemView().createFileObject(getDirectoryName())
	    );
	    getFileChooser().rescanCurrentDirectory();
	}
    }

    
    // *********************************************
    // ***** the default AcceptAll file filter *****
    // *********************************************
    protected class AcceptAllFileFilter extends FileFilter {
	
	public AcceptAllFileFilter() {
	}
	
	public boolean accept(File f) {
	    return true;
	}
	
	public String getDescription() {
	    return UIManager.getString("FileChooser.acceptAllFileFilterText");
	}
    }


    // ***********************
    // * FileView operations *
    // ***********************
    protected class BasicFileView extends FileView {
	// PENDING(jeff) - pass in the icon cache size
	protected Hashtable iconCache = new Hashtable();
	
	public BasicFileView() {
	}
	
	public void clearIconCache() {
	    iconCache = new Hashtable();
	}
	
	public String getName(File f) {
	    String fileName = null;
	    if(f != null) {
		fileName = f.getName();
		if (fileName.equals("")) {
		    fileName = f.getPath();
		}
	    }
	    return fileName;
	}
	
	
	public String getDescription(File f) {
	    return f.getName();
	}
	
	public String getTypeDescription(File f) {
	    if (f.isDirectory()) {
		return directoryDescription;
	    } else {
		return fileDescription;
	    }
	}
	
	public Icon getCachedIcon(File f) {
	    return (Icon) iconCache.get(f);
	}
	
	public void cacheIcon(File f, Icon i) {
	    if(f == null || i == null) {
		return;
	    }
	    iconCache.put(f, i);
	}
	
	public Icon getIcon(File f) {
	    Icon icon = getCachedIcon(f);
	    if(icon != null) {
		return icon;
	    }
	    if (f != null && f.isDirectory()) {
		if(getFileChooser().getFileSystemView().isRoot(f)) {
		    icon = hardDriveIcon;
		} else {
		    icon = directoryIcon;
		}
	    } else {
		icon = fileIcon;
	    }
	    cacheIcon(f, icon);
	    return icon;
	}
	
	public Boolean isTraversable(File f) {
	    if (f.isDirectory()) {
		return Boolean.TRUE;
	    } else {
		return Boolean.FALSE;
	    }
	}
	
	public Boolean isHidden(File f) {
	    String name = f.getName();
	    if(name != null && name.charAt(0) == '.') {
		return Boolean.TRUE;
	    } else {
		return Boolean.FALSE;
	    }
	}
    }

}
