/*
 * @(#)BasicComboBoxUI.java	1.71 98/06/22
 * 
 * Copyright (c) 1997 Sun Microsystems, Inc. All Rights Reserved.
 * 
 * This software is the confidential and proprietary information of Sun
 * Microsystems, Inc. ("Confidential Information").  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sun.
 * 
 * SUN MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SUN SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 * 
 */

package com.sun.java.swing.plaf.basic;

import java.awt.*;
import java.awt.event.*;
import com.sun.java.swing.*;
import com.sun.java.swing.FocusManager;
import com.sun.java.swing.plaf.*;
import com.sun.java.swing.border.*;
import com.sun.java.swing.text.*;
import com.sun.java.swing.event.*;
import java.io.Serializable;
import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeEvent;

/**
 * Basic UI for JComboBox.  This class adds and removes components from the
 * JComboBox.  The arrow button and the editor are managed by this object.
 * The popup menu is handled by BasicComboPopup.  BasicComboPopup supplies
 * this class with a MouseListener, MouseMotionListener, and a KeyListener.
 * These listeners are added to the arrow button and the JComboBox by default.
 * Subclasses of BasicComboBoxUI should attach the listeners to whichever
 * components they like.
 *
 * installListeners() is where listeners get added to the JComboBox (and model).
 * configureEditor() is where listeners get added to the editor.
 * configureArrowButton() is where listeners get added to the arrow button.
 * 
 * Inner classes for handling events:
 *    ComboBoxFocusListener
 *    ComboBoxItemListener
 *    ComboBoxListDataListener
 *    ComboBoxPropertyChangeListener
 *    ListMouseListener
 *    SelectionChangeKeyListener
 *
 * <p>
 * Warning: serialized objects of this class will not be compatible with
 * future swing releases.  The current serialization support is appropriate 
 * for short term storage or RMI between Swing1.0 applications.  It will
 * not be possible to load serialized Swing1.0 objects with future releases
 * of Swing.  The JDK1.2 release of Swing will be the compatibility
 * baseline for the serialized form of Swing objects.
 *
 * @version 1.71 06/22/98
 * @author Tom Santos
 */
public class BasicComboBoxUI extends ComboBoxUI implements Serializable {
    protected JComboBox comboBox;
    protected boolean   hasFocus = false;

    // This list is for drawing the current item in the combo box.
    // This listBox is not used by the popup.
    protected JList   listBox;

    // Used to render the currently selected item in the combo box.
    // It doesn't have anything to do with the popup's rendering.
    protected CellRendererPane currentValuePane = new CellRendererPane();

    // The implementation of ComboPopup that is used to show the popup.
    protected ComboPopup popup;

    // The Component that the ComboBoxEditor uses for editing 
    protected Component editor;

    // The arrow button that invokes the popup.
    protected JButton   arrowButton;

    // Listeners that are attached to the JComboBox
    protected KeyListener keyListener;
    protected FocusListener focusListener;
    protected ItemListener itemListener;
    protected PropertyChangeListener propertyChangeListener;

    // Listeners that the ComboPopup produces.
    // These get attached to any component that wishes to invoke the popup.
    protected MouseListener popupMouseListener;
    protected MouseMotionListener popupMouseMotionListener;
    protected KeyListener popupKeyListener;

    // This is used for knowing when to cache the minimum preferred size.
    // If the data in the list changes, the cached value get marked for recalc.
    protected ListDataListener listDataListener;

    // Flag for recalculating the minimum preferred size.
    protected boolean isMinimumSizeDirty = true;

    // Cached minimum preferred size.
    protected Dimension cachedMinimumSize = new Dimension( 0, 0 );


    //========================
    // begin UI Initialization
    //

    public static ComponentUI createUI(JComponent c) {
        return new BasicComboBoxUI();
    }

    public void installUI(JComponent c) {
        isMinimumSizeDirty = true;

        comboBox = (JComboBox)c;
        installDefaults( c );
        listBox = createListBox();
        popup = createPopup();

        popupMouseListener = popup.getMouseListener();
        popupMouseMotionListener = popup.getMouseMotionListener();
        popupKeyListener = popup.getKeyListener();

        keyListener = createKeyListener();
        focusListener = createFocusListener();
        listDataListener = createListDataListener();

        itemListener = createItemListener();
        propertyChangeListener = createPropertyChangeListener();

        addSubComponents();
        installListeners( c );

        comboBox.setLayout( createLayoutManager() );

        comboBox.setRequestFocusEnabled( true );

        // An invokeLater() was used here because updateComponentTree() resets
        // our sub-components after this method is completed.  By delaying, we
        // can set what we need after updateComponentTree() has set all of the
        // values to defaults.
        final JComboBox cBox = comboBox;
        Runnable initializer = new Runnable() {
            public void run(){
		if(comboBox != null) {
		    if ( editor != null ) {
			editor.setFont( cBox.getFont() );
		    }
		    addKeyAccelerators( cBox );
		}
            }
        };
        SwingUtilities.invokeLater( initializer );
    }

    public void uninstallUI(JComponent c) {
        popup.hide();
        popup.uninstallingUI();

        removeKeyAccelerators( c );

        comboBox.setLayout(null);

        comboBox.resetKeyboardActions();
        comboBox.remove(currentValuePane);

        removeSubComponents();
        removeListeners( c );
        comboBox = null;
        popup = null;
        keyListener = null;
        focusListener = null;
        listDataListener = null;
        popupKeyListener = null;
        popupMouseListener = null;
        popupMouseMotionListener = null;
    }

    /**
     * Installs the default colors, default font, default renderer, and default
     * editor into the JComboBox.
     */
    protected void installDefaults( JComponent c ) {
        LookAndFeel.installColorsAndFont( c,
                                          "ComboBox.background",
                                          "ComboBox.foreground",
                                          "ComboBox.font" );
        if ( comboBox.getRenderer() == null || comboBox.getRenderer() instanceof UIResource ) {
            comboBox.setRenderer((ListCellRenderer)(UIManager.get("ComboBox.renderer")));
        }
        if ( comboBox.getEditor() == null || comboBox.getEditor() instanceof UIResource ) {
            comboBox.setEditor((ComboBoxEditor)(UIManager.get("ComboBox.editor")));
        }

        if ( c.getBorder() instanceof UIResource ) {
            c.setBorder( null );
        }
    }

    /**
     * Attaches listeners to the JComboBox and JComboBoxModel.
     */
    protected void installListeners( JComponent c ) {
        comboBox.addItemListener( itemListener );
        comboBox.addPropertyChangeListener( propertyChangeListener );
        comboBox.addKeyListener( keyListener );
        comboBox.addFocusListener( focusListener );
        comboBox.addMouseListener( popupMouseListener );
        comboBox.addMouseMotionListener( popupMouseMotionListener );
        comboBox.addKeyListener( popupKeyListener );
        if ( comboBox.getModel() != null ) {
            comboBox.getModel().addListDataListener( listDataListener );
        }
    }

    /**
     * Removes listeners from the JComboBox and JComboBoxModel.
     */
    protected void removeListeners( JComponent c ) {
        comboBox.removeItemListener( itemListener );
        comboBox.removePropertyChangeListener( propertyChangeListener );
        comboBox.removeKeyListener( keyListener );
        comboBox.removeFocusListener( focusListener );
        comboBox.removeMouseListener( popupMouseListener );
        comboBox.removeMouseMotionListener( popupMouseMotionListener );
        comboBox.removeKeyListener( popupKeyListener );
        if ( comboBox.getModel() != null ) {
            comboBox.getModel().removeListDataListener( listDataListener );
        }
    }

    /**
     * Creates a JList instance that is used by ComboBoxUI.getList().  The list
     * is there to satisfy Accessibility requirements.  The list need not be the
     * list in the implementation of ComboPopup.
     */
    protected JList createListBox() {
        return new JList();
    }

    /**
     * Creates an implementation of the ComboPopup interface.
     * Returns an instance of BasicComboPopup.
     */
    protected ComboPopup createPopup() {
        return new BasicComboPopup( comboBox );
    }

    /**
     * Creates the key listener for handling type-ahead.
     * Returns an instance of BasicComboBoxUI$SelectionChangeKeyListener.
     */
    protected KeyListener createKeyListener() {
        return new SelectionChangeKeyListener();
    } 

    /**
     * Creates the focus listener that hides the popup when the focus is lost.
     * Returns an instance of BasicComboBoxUI$ComboBoxFocusListener.
     */
    protected FocusListener createFocusListener() {
        return new ComboBoxFocusListener();
    }

    /**
     * Creates the list data listener that is used for caching the preferred sizes.
     * Returns an instance of BasicComboBoxUI$ComboListDataListener.
     */
    protected ListDataListener createListDataListener() {
        return new ComboBoxListDataListener();
    }

    /**
     * Creates the item listener that watches for updates in the current selection
     * so that it can update the display.
     * Returns an instance of BasicComboBoxUI$ComboBoxItemListener.
     */
    protected ItemListener createItemListener() {
        return new ComboBoxItemListener();
    }

    /**
     * Creates the list data listener that is used for caching the preferred sizes.
     * Returns an instance of BasicComboBoxUI$ComboBoxPropertyChangeListener.
     */
    protected PropertyChangeListener createPropertyChangeListener() {
        return new ComboBoxPropertyChangeListener();
    }

    /**
     * Creates the standard combo box layout manager that has the arrow button to
     * the right and the editor to the left.
     * Returns an instance of BasicComboBoxUI$ComboBoxLayoutManager.
     */
    protected LayoutManager createLayoutManager() {
        return new ComboBoxLayoutManager();
    }

    //
    // end UI Initialization
    //======================


    //======================
    // begin Inner classes
    //

    /**
     * This listener checks to see if the key event isn't a navigation key.  If
     * it finds a key event that wasn't a navigation key it dispatches it to
     * JComboBox.selectWithKeyChar() so that it can do type-ahead.
     */
    protected class SelectionChangeKeyListener extends KeyAdapter {
        public void keyPressed( KeyEvent e ) {
            if ( comboBox.isEnabled() && !isNavigationKey( e.getKeyCode() ) ) {
                if ( comboBox.selectWithKeyChar(e.getKeyChar()) ) {
                    e.consume();
                }
            }
        }
    }

    /**
     * This listener hides the popup when the focus is lost.  It also repaints
     * when focus is gained or lost.
     */
    protected class ComboBoxFocusListener implements FocusListener {
        public void focusGained( FocusEvent e ) {
            comboBox.repaint();
        }

        public void focusLost( FocusEvent e ) {
            if ( !e.isTemporary() ) {
                popup.hide();
            }
            comboBox.repaint();
        }
    }

    /**
     * This listener hides the popup when the focus is lost.  It also repaints
     * when focus is gained or lost.
     */
    protected class ComboBoxListDataListener implements ListDataListener {
        public void contentsChanged( ListDataEvent e ) {
            isMinimumSizeDirty = true;
        }

        public void intervalAdded( ListDataEvent e ) {
            contentsChanged( e );
        }

        public void intervalRemoved( ListDataEvent e ) {
            contentsChanged( e );
        }
    }

    /**
     * This listener watches for changes to the selection in the combo box and
     * updates the display of the currently selected item.
     */
    protected class ComboBoxItemListener implements ItemListener {
        public void itemStateChanged(ItemEvent e) {
            ComboBoxModel model = comboBox.getModel();
            Object v = model.getSelectedItem();
            if ( editor != null )
                comboBox.configureEditor(comboBox.getEditor(),v);
            else {
                Rectangle r = rectangleForCurrentValue();
                comboBox.repaint(0,r.x,r.y,r.width,r.height);
            }
        }
    }

    /**
     * This listener watches for bound properties that have changed in the JComboBox.
     * It looks for the model and editor being swapped-out and updates appropriately.
     * It also looks for changes in the editable, enabled, and maximumRowCount properties.
     */
    public class ComboBoxPropertyChangeListener implements PropertyChangeListener {
        public void propertyChange(PropertyChangeEvent e) {
            String propertyName = e.getPropertyName();

            if ( propertyName.equals("model") ) {
                if ( listBox != null ) {
                    listBox.setModel(comboBox.getModel());
                    if ( popupIsVisible() )
                        popup.hide();
                }
            }
            else if ( propertyName.equals( "editor" ) && comboBox.isEditable() ) {
                removeEditor();
                addEditor();
            }
            else if ( propertyName.equals( "editable" ) ) {
                if ( comboBox.isEditable() ) {
                    comboBox.setRequestFocusEnabled( false );
                    comboBox.removeKeyListener( popupKeyListener );
                    addEditor();
                }
                else {
                    comboBox.setRequestFocusEnabled( true );
                    comboBox.addKeyListener( popupKeyListener );
                    removeEditor();
                }
            }
            else if ( propertyName.equals( "enabled" ) ) {
                boolean cbIsEnabled = comboBox.isEnabled();
                if ( cbIsEnabled ) {
                    if ( editor != null )
                        editor.setEnabled(true);
                    if ( arrowButton != null )
                        arrowButton.setEnabled(true);
                }
                else {
                    if ( editor != null )
                        editor.setEnabled(false);
                    if ( arrowButton != null )
                        arrowButton.setEnabled(false);
                }
                comboBox.repaint();
            }
            else if ( propertyName.equals( "maximumRowCount" ) ) {
                if ( popupIsVisible() ) {
                    popup.hide();
                    popup.show();
                }
            }
        }     
    }

    /**
     * This layout manager handles the 'standard' layout of combo boxes.  It puts
     * the arrow button to the right and the editor to the left.  If there is no
     * editor it still keeps the arrow button to the right.
     */
    public class ComboBoxLayoutManager implements LayoutManager {
        public void addLayoutComponent(String name, Component comp) {}

        public void removeLayoutComponent(Component comp) {}

        public Dimension preferredLayoutSize(Container parent) {
            JComboBox cb = (JComboBox)parent;
            return parent.getPreferredSize();
        }

        public Dimension minimumLayoutSize(Container parent) {
            JComboBox cb = (JComboBox)parent;
            return parent.getMinimumSize();
        }

        public void layoutContainer(Container parent) {
            JComboBox cb = (JComboBox)parent;
            int width = cb.getWidth();
            int height = cb.getHeight();
            Insets insets = getInsets();
            int buttonSize = height - (insets.top + insets.bottom);
            Rectangle cvb;

            if ( editor != null ) {
                cvb = rectangleForCurrentValue();
                editor.setBounds(cvb);
            }
            if ( arrowButton != null ) {
                arrowButton.setBounds( width - (insets.right + buttonSize),
                                       insets.top,
                                       buttonSize, buttonSize);
            }
        }
    }

    //
    // end Inner classes
    //====================


    //===============================
    // begin Sub-Component Management
    //

    /**
     * The editor and arrow button are added to the JComboBox here.
     */
    protected void addSubComponents() {
        arrowButton = createArrowButton();
        configureArrowButton();
        comboBox.add( arrowButton );

        if ( comboBox.isEditable() ) {
            addEditor();
        }

        comboBox.add( currentValuePane );
    }

    /**
     * The editor and/or arrow button are removed from the JComboBox here.
     * This method calls removeAll() on the JComboBox just to make sure that
     * everything gets removed.
     */
    protected void removeSubComponents() {
        if ( arrowButton != null ) {
            unconfigureArrowButton();
        }
        if ( editor != null ) {
            unconfigureEditor();
        }
        comboBox.removeAll(); // Just to be safe.
        arrowButton = null;
    }

    /**
     * Adds the editor to the JComboBox.  It also calls configureEditor()
     */
    public void addEditor() {
        removeEditor();
        editor = comboBox.getEditor().getEditorComponent();
        comboBox.add(editor);
        if ( editor != null ) {
            configureEditor();
        }
    }

    /**
     * Removes the editor from the JComboBox.  It also calls unconfigureEditor()
     */
    public void removeEditor() {
        if ( editor != null ) {
            unconfigureEditor();
            comboBox.remove( editor );
        }
    }

    /**
     * Configures the editor by setting its font and adding listeners.
     */
    protected void configureEditor() {
        editor.setFont( comboBox.getFont() );
        editor.addKeyListener( popupKeyListener );

        comboBox.configureEditor(comboBox.getEditor(),comboBox.getSelectedItem());
    }

    /**
     * Unconfigures the editor by removing listeners.
     */
    protected void unconfigureEditor() {
        editor.removeKeyListener( popupKeyListener );
    }

    /**
     * Configures the arrow button by adding listeners.
     */
    public void configureArrowButton() {
        if ( arrowButton != null ) {
            arrowButton.setRequestFocusEnabled(false);
            arrowButton.addMouseListener( popupMouseListener );
            arrowButton.addMouseMotionListener( popupMouseMotionListener );
            arrowButton.resetKeyboardActions();
        }
    }

    /**
     * Unconfigures the arrow button by removing listeners.
     */
    public void unconfigureArrowButton() {
        if ( arrowButton != null ) {
            arrowButton.removeMouseListener( popupMouseListener );
            arrowButton.removeMouseMotionListener( popupMouseMotionListener );
        }
    }

    /**
     * Creates the arrow button.  Subclasses can create any button they like.
     * The default behavior of this class is to attach various listeners to the
     * button returned by this method.
     * Returns an instance of BasicArrowButton.
     */
    protected JButton createArrowButton() {
        return new BasicArrowButton(BasicArrowButton.SOUTH);
    }

    //
    // end Sub-Component Management
    //===============================


    //================================
    // begin ComboBoxUI Implementation
    //

    /**
     * Tells if the popup is visible or not.
     */
    public boolean popupIsVisible() {
        return popup.isVisible();
    }

    /**
     * Shows the popup.
     */
    public void showPopup() {
        popup.show();
    }

    /**
     * Hides the popup.
     */
    public void hidePopup() {
        popup.hide();
    }

    /**
     * Determines if the JComboBox is focus traversable.  If the JComboBox is editable
     * this returns false, otherwise it returns true.
     */
    public boolean isFocusTraversable() {
        return comboBox.isEditable() ? false : true;
    }

    /**
     * Returns a list for rendering the items in the ComboBoxModel.  The list
     * is there to satisfy Accessibility requirements.  The list need not be the
     * list in the implementation of ComboPopup.
     */
    public JList getList() {
        return listBox;
    }

    //
    // end ComboBoxUI Implementation
    //==============================


    //=================================
    // begin ComponentUI Implementation

    public void paint( Graphics g, JComponent c ) {
        hasFocus = comboBox.hasFocus();
        if ( !comboBox.isEditable() ) {
            Rectangle r = rectangleForCurrentValue();
            paintCurrentValueBackground(g,r,hasFocus);
            paintCurrentValue(g,r,hasFocus);
        }
    }

    public Dimension getPreferredSize( JComponent c ) {
        Dimension size = getMinimumSize( c );
        size.width += 4; // Added for a little 'elbow room'.
        return size;
    }

    public Dimension getMinimumSize( JComponent c ) {
        if ( !isMinimumSizeDirty ) {
            return new Dimension( cachedMinimumSize );
        }
        Dimension size;
        Insets insets = getInsets();
        size = getDisplaySize();
        size.height += insets.top + insets.bottom;
        int buttonSize = size.height - (insets.top + insets.bottom);
        size.width +=  insets.left + insets.right + buttonSize;

        cachedMinimumSize.setSize( size.width, size.height ); 
        isMinimumSizeDirty = false;

        return size;
    }

    public Dimension getMaximumSize( JComponent c ) {
        Dimension size = getPreferredSize( c );
        size.width = Short.MAX_VALUE;
        return size;
    }

    //
    // end ComponentUI Implementation
    //===============================


    //======================
    // begin Utility Methods
    //

    /**
     * Returns whether or not the supplied keyCode maps to a key that is used for
     * navigation.  This is used for optimizing key input by only passing non-
     * navigation keys to the type-ahead mechanism.  Subclasses should override this
     * if they change the navigation keys.
     */
    protected boolean isNavigationKey( int keyCode ) {
        return keyCode == KeyEvent.VK_UP || keyCode == KeyEvent.VK_DOWN;
    }  

    /**
     * Selects the next item in the list.  It won't change the selection if the
     * currently selected item is already the last item.
     */
    protected void selectNextPossibleValue() {
        int si = comboBox.getSelectedIndex();

        if ( si < comboBox.getModel().getSize() - 1 ) {
            comboBox.setSelectedIndex(si+1);
            comboBox.repaint();
        }
    }

    /**
     * Selects the previous item in the list.  It won't change the selection if the
     * currently selected item is already the first item.
     */
    protected void selectPreviousPossibleValue() {
        int si = comboBox.getSelectedIndex();

        if ( si > 0 ) {
            comboBox.setSelectedIndex(si-1);
            comboBox.repaint();
        }
    }

    /**
     * Hides the popup if it is showing and shows the popup if it is hidden.
     */
    protected void toggleOpenClose() {
        if ( popupIsVisible() )
            popup.hide();
        else
            popup.show();
    }

    /**
     * Returns the area that is reserved for drawing the currently selected item.
     */
    protected Rectangle rectangleForCurrentValue() {
        int width = comboBox.getWidth();
        int height = comboBox.getHeight();
        Insets insets = getInsets();
        int buttonSize = height - (insets.top + insets.bottom);
        if ( arrowButton != null ) {
           buttonSize = arrowButton.getWidth();
        }
        return new Rectangle(insets.left, insets.top,
                             width - (insets.left + insets.right + buttonSize),
                             height - (insets.top + insets.bottom));
    }

    /**
     * Gets the insets from the JComboBox.
     */
    protected Insets getInsets() {
        return comboBox.getInsets();
    }

    //
    // end Utility Methods
    //====================


    //===============================
    // begin Painting Utility Methods
    //

    /**
     * Paints the currently selected item.
     */
    public void paintCurrentValue(Graphics g,Rectangle bounds,boolean hasFocus) {
        ListCellRenderer renderer = comboBox.getRenderer();
        Component c;
        if ( hasFocus && !popupIsVisible() ) {
            c = renderer.getListCellRendererComponent( listBox,
                                                       comboBox.getSelectedItem(),
                                                       -1,
                                                       true,
                                                       false );
        }
        else {
            c = renderer.getListCellRendererComponent( listBox,
                                                       comboBox.getSelectedItem(),
                                                       -1,
                                                       false,
                                                       false );
            c.setBackground(UIManager.getColor("ComboBox.background"));
        }
        c.setFont(comboBox.getFont());
        if ( hasFocus && !popupIsVisible() ) {
            c.setForeground(listBox.getSelectionForeground());
            c.setBackground(listBox.getSelectionBackground());
        }
        else {
            if ( comboBox.isEnabled() ) {
                c.setForeground(comboBox.getForeground());
                c.setBackground(comboBox.getBackground());
            }
            else {
                c.setForeground(UIManager.getColor("ComboBox.disabledForeground"));
                c.setBackground(UIManager.getColor("ComboBox.disabledBackground"));
            }
        }
        currentValuePane.paintComponent(g,c,comboBox,bounds.x,bounds.y,
                                        bounds.width,bounds.height);
    }

    /**
     * Paints the background of the currently selected item.
     */
    public void paintCurrentValueBackground(Graphics g,Rectangle bounds,boolean hasFocus) {
        Color t = g.getColor();
        if ( comboBox.isEnabled() )
            g.setColor(UIManager.getColor("ComboBox.background"));
        else
            g.setColor(UIManager.getColor("ComboBox.disabledBackground"));
        g.fillRect(bounds.x,bounds.y,bounds.width,bounds.height);
        g.setColor(t);
    }

    /**
     * Repaint the currently selected item.
     */
    void repaintCurrentValue() {
        Rectangle r = rectangleForCurrentValue();
        comboBox.repaint(r.x,r.y,r.width,r.height);
    }

    //
    // end Painting Utility Methods
    //=============================


    //===============================
    // begin Size Utility Methods
    //

    /** 
     * Return the dimension the the combo box should have if by default
     * if there is no current value and no value in the list of possible
     * values.
     */
    protected Dimension getDefaultSize() {
        return new Dimension(100,20);
    }

    protected Dimension getDisplaySize() {
        int i,c;
        Dimension result = new Dimension();
        ListCellRenderer renderer = comboBox.getRenderer();
        ComboBoxModel model = comboBox.getModel();
        Component cpn;
        Dimension d;

        if ( renderer != null && model.getSize() > 0 ) {
            for ( i=0,c=model.getSize();i<c;i++ ) {
                cpn = renderer.getListCellRendererComponent(listBox, model.getElementAt(i),-1, false, false);
                currentValuePane.add(cpn);
                cpn.setFont(comboBox.getFont());
                d = cpn.getPreferredSize();
                currentValuePane.remove(cpn);
                result.width = Math.max(result.width,d.width);
                result.height = Math.max(result.height,d.height);
            }

            if ( comboBox.isEditable() ) {
                d = editor.getPreferredSize();
                result.width = Math.max(result.width,d.width);
                result.height = Math.max(result.height,d.height);
            }

            return result;
        }
        else
            return getDefaultSize();

    }

    //
    // end Size Utility Methods
    //=============================


    //=================================
    // begin Keyboard Action Management
    //

    /**
     * Adds keyboard actions to the JComboBox.  Actions on enter and esc are already
     * supplied.  Add more actions as you need them.
     */
    protected void addKeyAccelerators(JComponent comp) {
        AbstractAction enterAction = new AbstractAction() {
            public void actionPerformed(ActionEvent e){
                popup.hide();
            }
            public boolean isEnabled(){
                return comboBox.isEnabled();
            }
        };

        comp.registerKeyboardAction( enterAction,
                                     KeyStroke.getKeyStroke(KeyEvent.VK_ENTER,0),
                                     JComponent.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT);

        AbstractAction escAction = new AbstractAction() {
            public void actionPerformed(ActionEvent e){
                popup.hide();
            }
            public boolean isEnabled(){
                return comboBox.isEnabled();
            }
        };

        comp.registerKeyboardAction( escAction,
                                     KeyStroke.getKeyStroke(KeyEvent.VK_ESCAPE,0),
                                     JComponent.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT);
    }

    /**
     * Removes the keyboard actions that were added by addKeyAccelerators().
     */
    protected void removeKeyAccelerators(JComponent comp) {
        comp.unregisterKeyboardAction(KeyStroke.getKeyStroke(KeyEvent.VK_ENTER,0));
        comp.unregisterKeyboardAction(KeyStroke.getKeyStroke(KeyEvent.VK_ESCAPE,0));
    }

    //
    // end Keyboard Action Management
    //===============================
}
