/*
 * @(#)BasicColorChooserUI.java	1.11 98/02/02
 * 
 * Copyright (c) 1997 Sun Microsystems, Inc. All Rights Reserved.
 * 
 * This software is the confidential and proprietary information of Sun
 * Microsystems, Inc. ("Confidential Information").  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sun.
 * 
 * SUN MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SUN SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 * 
 */

package com.sun.java.swing.plaf.basic;

import com.sun.java.swing.*;
import com.sun.java.swing.preview.*;
import com.sun.java.swing.event.*;
import com.sun.java.swing.plaf.ComponentUI;
import com.sun.java.swing.plaf.ColorChooserUI;
import java.util.*;
import java.awt.*;
import java.awt.image.*;
import java.awt.event.*;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.io.Serializable;


/**
 * Provides the basic look and feel for a JColorChooser.
 * <p>
 * @version 1.11 02/02/98
 * @author Tom Santos
 */

public class BasicColorChooserUI extends ColorChooserUI implements PropertyChangeListener
{
    JColorChooser chooser;
    JTabbedPane tabbedPane;
    Hashtable panels;

    public static ComponentUI createUI(JComponent x) {
	return new BasicColorChooserUI();
    }

    public void installUI( JComponent c ) {
        super.installUI( c );
	installDefaults( c );

        chooser = (JColorChooser)c;
	tabbedPane = new JTabbedPane();
	panels = new Hashtable();

        chooser.setLayout( createSinglePaneLayout() );

        ColorChooserPanel panel = new BasicHSVChooserPanel();
	chooser.addChooserPanel( "HSV", panel );

	panel = new BasicRGBChooserPanel();
	chooser.addChooserPanel( "RGB", panel );

	installListeners( c );
    }

    public void uninstallUI( JComponent c ) {
	chooser.removeChooserPanel( "HSV" );

	panels = null;
	chooser = null;
	tabbedPane = null;

	uninstallDefaults( c );
	uninstallListeners( c );
    }

    public Dimension getPreferredSize( JComponent c ) {
        Dimension result;

        if ( panels.size() == 1 ) {
	    result = ((ColorChooserPanel)panels.elements().nextElement()).getPreferredSize();
	}
	else if ( panels.size() > 1 ) {
	    return tabbedPane.getPreferredSize();
	}
	else {
	    result = new Dimension( 0, 0 );
	}

	return result;
    }

    protected void installDefaults(JComponent c) {
        LookAndFeel.installColorsAndFont(c, "ColorChooser.background", 
                                              "ColorChooser.foreground",
                                              "ColorChooser.font");
    }

    protected void uninstallDefaults(JComponent c) {
    }

    protected void installListeners( JComponent c ) {
	chooser.addPropertyChangeListener( this );
    }

    protected void uninstallListeners( JComponent c ) {
	chooser.removePropertyChangeListener( this );
    }

    // This is only used in propertyChange()
    boolean settingAllPanels = false;

    public void propertyChange(PropertyChangeEvent e) {
	if ( e.getPropertyName().equals( ColorChooserPanel.COLOR_PROPERTY ) && !settingAllPanels ) {
	    if ( chooser.getColor() != e.getNewValue() ) {
	        chooser.setColor( (Color)e.getNewValue() );
	    }
	}
	else if ( e.getPropertyName().equals( JColorChooser.COLOR_PROPERTY ) ) {
	    Enumeration elements = panels.elements();
	    settingAllPanels = true;
	    while ( elements.hasMoreElements() ) {
	        ColorChooserPanel panel = (ColorChooserPanel)elements.nextElement();
	        if ( panel.getColor() != e.getNewValue() ) {
		    panel.setColor( (Color)e.getNewValue() );
		}
	    }
	    settingAllPanels = false;
	}
    }

    /**
     * Adds a color chooser panel to the color chooser.
     *
     * @param name   a string that specifies the name of the panel
     */
    public void addChooserPanel( String name, ColorChooserPanel panel ) {
        if ( panel != null && name != null ) {
	    panel.installChooserPanel();

	    if ( panels.size() == 0 && chooser.getColor() == null ) {
	        chooser.setColor( panel.getColor() );
	    }
	    else {
	        panel.setColor( chooser.getColor() );
	    }

	    panels.put( name, panel );
	    panel.addPropertyChangeListener( this );

	    if ( panels.size() == 2 ) {
	        switchToMultiplePanelSetup();
	    }
	    else if ( panels.size() <= 1 ) {
	        chooser.add( panel, "Center" );
	    }
	    else {
	        JPanel centeredPanel = new JPanel();
		centeredPanel.setLayout( new CenterLayout() );
		centeredPanel.add( panel );
	        tabbedPane.addTab( name, centeredPanel );
	    }
	}
    }

    /**
     * Removes the ColorChooserPanel specified by "name".
     *
     * @param name   a string that specifies the panel to be removed
     */
    public ColorChooserPanel removeChooserPanel( String name ) {
        ColorChooserPanel panel = null;

        if ( name != null ) {
	    panel = (ColorChooserPanel)panels.remove( name );

	    if ( panels.size() > 1 ) {
	        tabbedPane.removeTabAt( tabbedPane.indexOfTab( name ) );
	    }
	    else if ( panels.size() == 0 ) {
	        chooser.remove( panel );
	    }
	    // If we are brought down to 1 panel don't do remove anything from
	    // anywhere.  We'll just let switchToSinglePanelSetup() do what's needed.

	    panel.uninstallChooserPanel();
	    panel.removePropertyChangeListener( this );

	    if ( panel != null && panels.size() == 1 ) {
	        switchToSinglePanelSetup();
	    }
	}

        return panel;
    }


    /**
     * Reorganizes to show multiple ColorChooserPanels.  It uses a tabbed pane
     * show the multiple panels.
     */
    protected void switchToMultiplePanelSetup() {
        chooser.setLayout( createTabbedPaneLayout() );
	
	Enumeration keys = panels.keys();

	while ( keys.hasMoreElements() ) {
	    String key = (String)keys.nextElement();
	    chooser.remove( (Component)panels.get( key ) );
	    JPanel panel = new JPanel();
	    panel.setLayout( new CenterLayout() );
	    panel.add( (Component)panels.get( key ) );
	    tabbedPane.addTab( key, panel );
	}

	chooser.add( tabbedPane, "Center" );
    }


    protected void switchToSinglePanelSetup() {
        chooser.remove( tabbedPane );

	for ( int i = 0; i < panels.size(); ++i ) {
	    tabbedPane.removeTabAt( 0 );
	}

        chooser.setLayout( createSinglePaneLayout() );

	Enumeration elements = panels.elements();

	if ( elements.hasMoreElements() ) {
	    chooser.add( (Component)elements.nextElement(), "Center" );
	}
    }

    protected LayoutManager createSinglePaneLayout() {
        return new CenterLayout();
    }

    protected LayoutManager createTabbedPaneLayout() {
        return new BorderLayout();
    }

    /**
     * Center-positioning layout manager.
     * <p>
     * Warning: serialized objects of this class will not be compatible with
     * future swing releases.  The current serialization support is appropriate
     * for short term storage or RMI between Swing1.0 applications.  It will
     * not be possible to load serialized Swing1.0 objects with future releases
     * of Swing.  The JDK1.2 release of Swing will be the compatibility
     * baseline for the serialized form of Swing objects.
     */
    public static class CenterLayout implements LayoutManager, Serializable {
        public void addLayoutComponent(String name, Component comp) { }
        public void removeLayoutComponent(Component comp) { }

        public Dimension preferredLayoutSize( Container container ) {
	    Component c = container.getComponent( 0 );
	    if ( c != null ) {
	        return c.getPreferredSize();
	    }
	    else {
	        return new Dimension( 0, 0 );
	    }
        }

        public Dimension minimumLayoutSize(Container cont) {
    	    return preferredLayoutSize(cont);
        }

        public void layoutContainer(Container container) {
	    try {
	        Component c = container.getComponent( 0 );
	    
		c.setSize( c.getPreferredSize() );
		Dimension size = c.getSize();
		Dimension containerSize = container.getSize();
		int componentLeft = (containerSize.width / 2) - (size.width / 2);
		int componentTop = (containerSize.height / 2) - (size.height / 2);

		c.setBounds( componentLeft, componentTop, size.width, size.height );
	    }
	    catch( Exception e ) {
	    }
        }
    }
}

