/*
 * @(#)AbstractTreeUI.java	1.31 98/02/05
 * 
 * Copyright (c) 1997 Sun Microsystems, Inc. All Rights Reserved.
 * 
 * This software is the confidential and proprietary information of Sun
 * Microsystems, Inc. ("Confidential Information").  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sun.
 * 
 * SUN MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SUN SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 * 
 */

package com.sun.java.swing.plaf.basic;

import java.awt.*;
import java.util.*;
import com.sun.java.swing.*;
import com.sun.java.swing.event.*;
import com.sun.java.swing.plaf.TreeUI;
import com.sun.java.swing.tree.*;
import java.io.*;

/**
 * AbstractTreeUI can be used as a placeholder for subclassers wishing to
 * create their own Look and Feel, but that don't want to reinvent
 * everything. It will track the nodes that have been expanded and can
 * handle almost all of the TreeUI methods. There are a few methods
 * subclassers must implement, such as determining the size of
 * each of the nodes. Subclassers will need to support drawing of the nodes,
 * editing of the nodes and potentially key and mouse navigation, look
 * to BasicTreeUI for an example of all of this.<p>
 * AbstractTreeUI will use VisibleTreeNode to track everything, unless it
 * has bee configured as a largeModel, in which case LargeTreeModelNode
 * will be used.<p>
 * <p>
 * Warning: serialized objects of this class will not be compatible with
 * future swing releases.  The current serialization support is appropriate 
 * for short term storage or RMI between Swing1.0 applications.  It will
 * not be possible to load serialized Swing1.0 objects with future releases
 * of Swing.  The JDK1.2 release of Swing will be the compatibility
 * baseline for the serialized form of Swing objects.
 *
 * @version 1.31 02/05/98
 * @author Rob Davis
 * @author Ray Ryan
 * @author Scott Violet
 */

public abstract class AbstractTreeUI extends TreeUI implements Serializable,
    TreeModelListener, RowMapper, TreeSelectionListener
{
    /** The model that is currently being compressed. */
    transient protected TreeModel          treeModel;

    /** Is the root being displayed? */
    protected boolean            rootVisible;

    /**
     * The root node of the internal cache of nodes that have been shown.
     * If the treeModel is vending a network rather than a true tree, 
     * there may be one cached node for each path to a modeled node.
     */
    protected VisibleTreeNode	 treeCacheRoot;

    /**
     * The array of nodes that are currently visible, in the order they 
     * are displayed.
     */
    protected Vector            visibleNodes;

    /**
      * When messaged to paint() if this is true updateNodeSizes()
      * will be messaged before painting to relayout the location and
      * sizes of the nodes.
      */
    protected boolean           updateNodeSizes;

    /**
      * Height to use for each row.  If this is <= 0 the renderer will be
      * used to determine the height for each row.
      */
    protected int               rowHeight;

    /**
      * true if there should be handles at the top level.  If the root
      * isn't visible it is highly recommended to make this true.
      */
    protected boolean           showsRootHandles;

    /**
      * Used to determine if a node is selected.
      */
    protected TreeSelectionModel treeSelectionModel;

    /** Root node, used if largeModel is set to true. */
    protected LargeTreeModelNode       largeRoot;

    /** If this is true, VisibleTreeNode is not used, instead
     * LargeTreeModelNode will be used to track the state of the tree.
     * If this is true, rowHeight will always be > 0. */
    protected boolean            largeModel;

    /** Number of rows, only used for largeModel. */
    protected int                largeRowCount;


    public AbstractTreeUI()
    {
	rowHeight = 16;
	visibleNodes = new Vector();
    }

    /** Sets the model the data is going to come from.  Will message
      * rebuild if the new model is different from the current model. */
    public void setModel(TreeModel newModel) {
	TreeModel oldModel = treeModel;

	if (newModel != oldModel) {
	    if (oldModel != null) {
		oldModel.removeTreeModelListener(this);
	    }

	    treeModel = newModel;
	    if(treeModel != null)
		treeModel.addTreeModelListener(this);
	    rebuild();
	}
    }

    /**
      * Returns the TreeModel that is being displayed for.
      */
    public TreeModel getModel() {
	return treeModel;
    }

    /**
     * Configures the reciever to display a largeModel based on
     * <code>largeModel</code>.
     */
    public void setLargeModel(boolean largeModel) {
	if(this.largeModel != largeModel) {
	    this.largeModel = largeModel;
	    if(largeModel && rowHeight <= 0)
		rowHeight = 16;
	    rebuild();
	}
    }

    /**
     * Returns true if the reciever is configured for displaying large
     * models.
     */
    public boolean isLargeModel() {
	return largeModel;
    }

    /**
      * Sets whether or not the root node of the model is to be displayed.
      */
    public void setRootVisible(boolean rootVisible) {
	if (rootVisible != this.rootVisible) {
	    this.rootVisible = rootVisible;
	    /* Remove or add the root to the visible nodes. */
	    if(treeModel != null)
	    {
		if(!largeModel) {
		    if(this.rootVisible) {
			treeCacheRoot.updatePreferredSize(0);
			visibleNodes.insertElementAt(treeCacheRoot, 0);
		    }
		    else if(visibleNodes.size() > 0) {
			visibleNodes.removeElementAt(0);
			if(treeSelectionModel != null)
			    treeSelectionModel.removeSelectionPath
				(treeCacheRoot.getTreePath());
		    }
		    if(treeSelectionModel != null)
			treeSelectionModel.resetRowSelection();
		    if(getRowCount() > 0)
			getNode(0).setYOrigin(0);
		    updateYLocationsFrom(0);
		    this.visibleNodesChanged();
		}
		else {
		    if(this.rootVisible)
			largeRowCount++;
		    else
			largeRowCount--;
		    if(treeSelectionModel != null)
			treeSelectionModel.clearSelection();
		    this.visibleNodesChanged();
		}
	    }
	}
    }

    /**
      * Returns true if root is currently visible.
      */
    public boolean isRootVisible() {
	return rootVisible;
    }

    /** Sets whether or not the root handles are to be displayed.
      */
    public void setShowsRootHandles(boolean newValue) {
	if(showsRootHandles != newValue)
	{
	    showsRootHandles = newValue;
	    this.visibleNodesChanged();
	}
    }

    /** Returns true if handles for the root nodes are displayed.
      */
    public boolean getShowsRootHandles()
    {
	return showsRootHandles;
    }

    /**
      * Sets the height of each to be <I>rowHeight</I>.  If <I>rowHeight</I>
      * is less than or equal to zero the current cell renderer will
      * be queried for each rows height.
      * If largeModel is true and rowHeight is < 1, an
      * IllegalArgumentException will be thrown.
      */
    public void setRowHeight(int rowHeight)
    {
	if(rowHeight != this.rowHeight)
	{
	    if(!largeModel) {
		this.rowHeight = rowHeight;
		if(!isFixedRowHeight() && visibleNodes.size() > 0)
		    updateNodeSizes(true);
		else if(isFixedRowHeight()) {
		    for(int counter = getRowCount() - 1; counter >= 0;
			counter--)
			getNode(counter).getPreferredSize().height = rowHeight;
		}
		this.visibleNodesChanged();
	    }
	    else {
		if(rowHeight <= 0)
		    throw new IllegalArgumentException("AbstractTreeUI.setRowHeight() row height must be > 0 for large models");
		this.rowHeight = rowHeight;
		visibleNodesChanged();
	    }
	}
    }

    /**
      * Returns the height of each row.  If returned value is less than
      * or equal to 0 the height for each row is determined by the
      * renderer.
      */
    public int getRowHeight()
    {
	return rowHeight;
    }

    /**
      * Returns true if the height of each row is a fixed size.
      */
    public boolean isFixedRowHeight()
    {
	return (rowHeight > 0);
    }

    /**
      * Subclassers must implement this method, should return the x
      * origin of where the particular node is goin to drawn at.
      */
    public abstract int getXOriginOfNode(VisibleTreeNode node);

    /**
      * Subclassers must implement this to return the size of node.
      */
    public abstract Dimension getSizeOfNode(VisibleTreeNode node, int row);

    /**
     * Parent is the parent of the node being drawn for. Will be null if
     * getting root bounds.
     */
    protected abstract Rectangle getLargeBoundsOf(LargeTreeModelNode parent,
						  int row,
						  Object childUserObject);

    /**
      * Messaged when the visible nodes have changed (eg when a
      * node is expanded, when nodes are removed from the model...).
      */
    public abstract void visibleNodesChanged();

    /**
      * Messaged from the VisibleTreeNode after it has been expanded.
      */
    protected abstract void pathWasExpanded(TreePath path);

    /**
      * Messaged from the VisibleTreeNode after it has collapsed.
      */
    protected abstract void pathWasCollapsed(TreePath path);

    /**
      * Returns true if path is an instance of AbstractTreePath
      * and was created by this instance (updating it if it is out of date).
      * Otherwise false is returned.
      * Only used if largeModel is false.
      */
    protected boolean isAbstractTreePath(TreePath path, boolean onlyIfVisible,
					 boolean shouldCreate) {
	if(path != null && path instanceof AbstractTreePath) {
	    AbstractTreePath          atPath = (AbstractTreePath)path;

	    if(atPath.node == null || atPath.getUI() == this) {
		if(atPath.node == null || atPath.node.isValid == false) {
		    atPath.node = getNodeForPath(path.getPath(),
						 false, shouldCreate);
		    if(atPath.node == null)
			return false;
		}
		return true;
	    }
	}
	return false;
    }

    /**
      * Returns the VisibleTreeNode identified by path.  This mirrors
      * the behavior of getNodeForPath, but tries to take advantage of
      * path if it is an instance of AbstractTreePath.
      * Only used if largeModel is false.
      */
    protected VisibleTreeNode getNodeForTreePath(TreePath path,
						 boolean onlyIfVisible,
						 boolean shouldCreate) {
	if(isAbstractTreePath(path, onlyIfVisible, shouldCreate)) {
	    VisibleTreeNode      vtn = ((AbstractTreePath)path).node;

	    if(vtn == null)
		return null;
	    if(onlyIfVisible && !vtn.isVisible())
		return null;
	    return vtn;
	}
	if(path != null)
	    return getNodeForPath(path.getPath(), onlyIfVisible, shouldCreate);
	return null;
    }

    /**
      * Returns the path of VisibleTreeNode's to reach the path identified
      * in path.  This mirrors the behavior of getNodesForPath, but tries
      * to take advantage of path if it is an instance of AbstractTreePath.
      * Only used if largeModel is false.
      */
    protected VisibleTreeNode[] getNodesForTreePath(TreePath path,
						    boolean onlyIfVisible,
						    boolean shouldCreate) {
	if(isAbstractTreePath(path, onlyIfVisible, shouldCreate)) {
	    VisibleTreeNode      vtn = ((AbstractTreePath)path).node;

	    if(vtn == null) {
		return null;
	    }
	    if(onlyIfVisible && !vtn.isVisible()) {
		return null;
	    }
	    // PENDING(scott): is this really necessary?  How come
	    // casting doesn't work here?
	    TreeNode[]        tPath = vtn.getPath();
	    VisibleTreeNode[] cPath = new VisibleTreeNode[tPath.length];
	    System.arraycopy(tPath, 0, cPath, 0, tPath.length);
	    return cPath;
	}
	if(path != null)
	    return getNodesForPath(path.getPath(), onlyIfVisible,
				   shouldCreate);
	return null;
    }

    /**
      * Returns an AbstractTreePath for path.  If path is already an
      * instance of AbstractTreePath, it is returned, otherwise an
      * instance of AbstractTreePath is created.
      * Only used if largeModel is false.
      */
    protected TreePath ensurePathIsAbstract(TreePath path,
						  VisibleTreeNode node) {
	if(path != null && !(path instanceof AbstractTreePath))
	    return new AbstractTreePath(path.getPath(), node);
	return path;
    }

    /**
     * Returns the VisibleTreeNode that represents the last item in
     * <I>path</I>.  If <I>onlyIfVisibile</I> is true and any of the nodes in
     * <I>path</I> are not currently visible, null will be returned.
     * <I>shouldCreate</I> is passed into <b>getLoadedChildren</b>, a
     * true value will load the nodes if they have not already been
     * loaded.
      * Only used if largeModel is false.
     */
    public VisibleTreeNode getNodeForPath(Object path[],
					  boolean onlyIfVisible,
					  boolean shouldCreate) {
	VisibleTreeNode[]         nodes = getNodesForPath(path, onlyIfVisible,
							shouldCreate);

	if(nodes != null && nodes.length > 0)
	    return nodes[nodes.length - 1];
	return null;
    }

    /**
     * Returns the VisibleTreeNodes that represent
     * <I>path</I>.  If <I>onlyIfVisibile</I> is true and any of the nodes in
     * <I>path</I> are not currently visible, null will be returned.
     * <I>shouldCreate</I> is passed into <b>getLoadedChildren</b>, a
     * true value will load the nodes if they have not already been
     * loaded.
      * Only used if largeModel is false.
     */
    public VisibleTreeNode[] getNodesForPath(Object path[],
					     boolean onlyIfVisible,
					     boolean shouldCreate) {
	if(path != null && path.length > 0 && treeModel != null)
	{
	    int                           counter;
	    VisibleTreeNode               retNode;
	    VisibleTreeNode               nodePath[];

	    if(treeCacheRoot.getValue().equals(path[0]))
		retNode = treeCacheRoot;
	    else
		return null;
	    counter = 1;
	    if(onlyIfVisible && path.length > 1 && !retNode.isExpanded())
		return null;
	    nodePath = new VisibleTreeNode[path.length];
	    nodePath[0] = retNode;
	    while(retNode != null && counter < path.length)
	    {
		Enumeration               childEnum;
		VisibleTreeNode           newNode = null;

		childEnum = retNode.getLoadedChildren(shouldCreate);
		while(childEnum.hasMoreElements() &&
		      newNode == null)
		{
		    newNode = (VisibleTreeNode)childEnum.nextElement();
		    if(!newNode.getValue().equals(path[counter]))
			newNode = null;
		}
		nodePath[counter] = retNode = newNode;
		counter++;
		if(retNode != null && onlyIfVisible && counter < path.length &&
		   !retNode.isExpanded())
		    retNode = null;
	    }
	    if(retNode != null)
		return nodePath;
	    return null;
	}
	return null;
    }

    /**
      * Updates the y locations of all of the visible nodes after
      * location.
      * Only used if largeModel is false.
      */
    protected void updateYLocationsFrom(int location)
    {
	if(location >= 0 && location < getRowCount())
	{
	    int                    counter, maxCounter, newYOrigin;
	    VisibleTreeNode        aNode;

	    aNode = getNode(location);
	    newYOrigin = aNode.getYOrigin() + aNode.getPreferredSize().height;
	    for(counter = location + 1, maxCounter = visibleNodes.size();
		counter < maxCounter;counter++)
	    {
		aNode = (VisibleTreeNode)visibleNodes.
		    elementAt(counter);
		aNode.setYOrigin(newYOrigin);
		newYOrigin += aNode.getPreferredSize().height;
	    }
	}
    }

    // 
    // TreeModelListener
    //

    /**
     * Invoked after nodes in the tree have changed in some way.  All of
     * the nodes are siblings.  The nodes have not changed locations in
     * the tree or altered their children arrays, but other attributes
     * have changed and may affect the nodes' presentation.
     */
    public synchronized void treeNodesChanged(TreeModelEvent e)
    {
	if(e != null) {
	    if(largeModel) {
		this.visibleNodesChanged();
	    }
	    else {
		int               changedIndexs[];
		VisibleTreeNode   changedNode;

		changedIndexs = e.getChildIndices();
		changedNode = getNodeForPath(e.getPath(), false, false);
		if(changedNode != null)
		{
		    Object            changedValue = changedNode.getValue();

		    /* Update the size of the changed node, as well as all the
		       child indexs that are passed in. */
		       changedNode.updatePreferredSize();
		    if(changedIndexs != null)
		    {
			int                counter;
			VisibleTreeNode    changedChildNode;

			for(counter = 0; counter < changedIndexs.length;
			    counter++) {
			    try {
				changedChildNode = (VisibleTreeNode)changedNode
				    .getChildAt(changedIndexs[counter]);
				/* Reset the user object. */
				changedChildNode.setUserObject
				    (treeModel.getChild(changedValue,
						     changedIndexs[counter]));
				changedChildNode.updatePreferredSize();
			    }
			    catch (Exception ex) {}
			}
		    }
		    if(!isFixedRowHeight()) {
			int          aRow = changedNode.getRow();

			if(aRow != -1)
			    this.updateYLocationsFrom(aRow);
		    }
		    this.visibleNodesChanged();
		}
	    }
	}
    }

    /**
      * Creates a new node to represent the node at <I>childIndex</I> in
      * <I>parent</I>s children.  This should be called if the node doesn't
      * already exist and <I>parent</I> has been expanded at least once.
      * The newly created node will be made visible if <I>parent</I> is
      * currently expanded.  This does not update the position of any
      * cells, nor update the selection if it needs to be.  If succesful
      * in creating the new VisibleTreeNode, it is returned, otherwise
      * null is returned.
      * Only used if largeModel is false.
      */
    protected VisibleTreeNode createNodeAt(VisibleTreeNode parent,
					   int childIndex)
    {
	boolean               isParentRoot;
	Object                 newValue;
	VisibleTreeNode        newChildNode;

	try
	{
	    newValue = treeModel.getChild(parent.getValue(), childIndex);
	    newChildNode = createNodeForValue(newValue, -1);
	    parent.insert(newChildNode, childIndex);
	} catch (Exception ex) { newChildNode = null; }
	isParentRoot = (parent == treeCacheRoot);
	if(newChildNode != null && parent.isExpanded() &&
	   (parent.getRow() != -1 || isParentRoot))
	{
	    int                 newRow;

	    /* Find the new row to insert this newly visible node at. */
	    if(childIndex == 0)
	    {
		if(isParentRoot && !isRootVisible())
		    newRow = 0;
		else
		    newRow = parent.getRow() + 1;
	    }
	    else if(childIndex == parent.getChildCount())
		newRow = parent.getLastVisibleNode().getRow() + 1;
	    else
	    {
		VisibleTreeNode          previousNode;
		
		previousNode = (VisibleTreeNode)parent.
		    getChildAt(childIndex - 1);
		newRow = previousNode.getLastVisibleNode().getRow() + 1;
	    }
	    visibleNodes.insertElementAt(newChildNode, newRow);
	}
	return newChildNode;
    }

    /**
     * Invoked after nodes have been inserted into the tree.
     */
    public synchronized void treeNodesInserted(TreeModelEvent e)
    {
	if(e != null && !largeModel)
	{
	    int               changedIndexs[];
	    VisibleTreeNode   changedParentNode;

	    changedIndexs = e.getChildIndices();
	    changedParentNode = getNodeForPath(e.getPath(), false, false);
	    /* Only need to update the children if the node has been
	       expanded once. */
	    // PENDING(scott): make sure childIndexs is sorted!
	    if(changedParentNode != null && changedIndexs != null &&
	       changedIndexs.length > 0) {
		if(changedParentNode.hasBeenExpanded()) {
		    boolean            makeVisible;
		    int                counter;
		    Object             changedParent;
		    VisibleTreeNode    newNode;
		
		    changedParent = changedParentNode.getValue();
		    makeVisible = ((changedParentNode == treeCacheRoot &&
				    !rootVisible) ||
				   (changedParentNode.getRow() != -1 &&
				    changedParentNode.isExpanded()));
		    for(counter = 0;counter < changedIndexs.length;counter++)
		    {
			newNode = this.createNodeAt(changedParentNode,
						    changedIndexs[counter]);
		    }
		    if(treeSelectionModel != null)
			treeSelectionModel.resetRowSelection();
		    /* Update the y origins from the index of the parent
		       to the end of the visible rows. */
		    if(!isFixedRowHeight() && makeVisible)
		    {
			if(changedParentNode == treeCacheRoot)
			    this.updateYLocationsFrom(0);
			else
			    this.updateYLocationsFrom(changedParentNode.
						      getRow());
			this.visibleNodesChanged();
		    }
		    else if(makeVisible)
			this.visibleNodesChanged();
		}
		else
		    changedParentNode.modelChildCountChanged();
	    }
	}
	else if(e != null) {
	    int                 changedIndexs[];
	    LargeTreeModelNode  changedParent = getLargeTreeModelNodeForPath
		                  (e.getPath(), false, false);
	    int                 maxCounter;

	    changedIndexs = e.getChildIndices();
	    /* Only need to update the children if the node has been
	       expanded once. */
	    // PENDING(scott): make sure childIndexs is sorted!
	    if(changedParent != null && changedIndexs != null &&
	       (maxCounter = changedIndexs.length) > 0) {
		boolean          isVisible =
		    (changedParent.isVisible() &&
		     changedParent.isExpanded());

		for(int counter = 0; counter < maxCounter; counter++) {
		    changedParent.childInsertedAtModelIndex
			(changedIndexs[counter]);
		    if(isVisible)
			largeRowCount++;
		}
		if(isVisible && treeSelectionModel != null)
		    treeSelectionModel.resetRowSelection();
		if(isVisible)
		    this.visibleNodesChanged();
		else
		    changedParent.modelChildCountChanged();
	    }
	}
    }

    /**
     * Invoked after nodes have been removed from the tree.  Note that
     * if a subtree is removed from the tree, this method may only be
     * invoked once for the root of the removed subtree, not once for
     * each individual set of siblings removed.
     */
    public synchronized void treeNodesRemoved(TreeModelEvent e)
    {
	if(e != null && !largeModel)
	{
	    int               changedIndexs[];
	    VisibleTreeNode   changedParentNode;

	    changedIndexs = e.getChildIndices();
	    changedParentNode = getNodeForPath(e.getPath(), false, false);
	    // PENDING(scott): make sure that changedIndexs are sorted in
	    // ascending order.
	    if(changedParentNode != null && changedIndexs != null &&
	       changedIndexs.length > 0) {
		if(changedParentNode.hasBeenExpanded())
		{
		    boolean            makeInvisible;
		    int                counter;
		    int                removedRow;
		    VisibleTreeNode    removedNode;

		    makeInvisible = ((changedParentNode == treeCacheRoot &&
				      !rootVisible) || 
				     (changedParentNode.getRow() != -1 &&
				      changedParentNode.isExpanded()));
		    for(counter = changedIndexs.length - 1;counter >= 0;
			counter--)
		    {
			try
			{
			    removedNode = (VisibleTreeNode)changedParentNode.
				getChildAt(changedIndexs[counter]);
			    if(removedNode.isExpanded())
				removedNode.collapse(false);

			    /* Let the selection model now. */
			    if(makeInvisible) {
				removedRow = removedNode.getRow();
				if(removedRow != -1) {
				    visibleNodes.removeElementAt(removedRow);
				    if(treeSelectionModel != null) {
					TreePath oldPath = removedNode.
					    getTreePath();

					treeSelectionModel.removeSelectionPath
					    (oldPath);
				    }
				}
			    }
			    changedParentNode.remove(removedNode);
			} catch (Exception ex) {
			    System.out.println("Exception removing node" + ex);
			}
		    }
		    if(treeSelectionModel != null)
			treeSelectionModel.resetRowSelection();
		    /* Update the y origins from the index of the parent
		       to the end of the visible rows. */
		    if(!isFixedRowHeight() && makeInvisible)
		    {
			if(changedParentNode == treeCacheRoot)
			{
			    /* It is possible for first row to have been
			       removed if the root isn't visible, in which
			       case ylocations will be off! */
			    if(getRowCount() > 0)
				getNode(0).setYOrigin(0);
			    updateYLocationsFrom(0);
			}
			else
			    updateYLocationsFrom(changedParentNode.getRow());
			this.visibleNodesChanged();
		    }
		    else if(makeInvisible)
			this.visibleNodesChanged();
		}
		else
		    changedParentNode.modelChildCountChanged();
	    }
	}
	else if(e != null) {
	    int                  changedIndexs[];
	    int                  maxCounter;
	    Object[]             parentPath = e.getPath();
	    LargeTreeModelNode changedParentNode = getLargeTreeModelNodeForPath
		                       (parentPath, false, false);

	    changedIndexs = e.getChildIndices();
	    // PENDING(scott): make sure that changedIndexs are sorted in
	    // ascending order.
	    if(changedParentNode != null && changedIndexs != null &&
	       (maxCounter = changedIndexs.length) > 0) {
		Object[]           childPath;
		Object[]           children = e.getChildren();
		int                parentPathLength = parentPath.length;
		boolean            isVisible =
		    (changedParentNode.isVisible() &&
		     changedParentNode.isExpanded());
		LargeTreeModelNode       childNode;

		for(int counter = 0; counter < maxCounter; counter++) {
		    childNode = changedParentNode.childAtModelIndex
			(changedIndexs[counter]);
		    if(childNode != null) {
			childNode.collapse(false);
			changedParentNode.remove(childNode);
		    }
		    if(isVisible)
			largeRowCount--;
		    changedParentNode.childRemovedAtModelIndex
			(changedIndexs[counter]);

		    /* Clean up the selection. */
		    if(treeSelectionModel != null && children != null &&
			children[counter] != null) {
			childPath = new Object[parentPathLength + 1];
			System.arraycopy(parentPath, 0, childPath, 0,
					 parentPathLength);
			childPath[parentPathLength] = children[counter];
			treeSelectionModel.removeSelectionPath
			                   (new TreePath(childPath));
		    }
		}
		if(isVisible) {
		    if(treeSelectionModel != null)
			treeSelectionModel.resetRowSelection();
		    this.visibleNodesChanged();
		}
		else
		    changedParentNode.modelChildCountChanged();
	    }
	    
	}
    }

    /**
     * Invoked after the tree has drastically changed structure.  The event
     * will identify the root of the subtree that has changed.  The
     * behavior of this is to recreate the last node that is identified
     * by the path in the passed in event.  If the node was previously
     * expanded it will be expanded when recreated, but none of its
     * children will be expanded.  If the path returned by e.getPath() is
     * of length one and the first element does not identify the current
     *  root node the first element should become the new root of the tree.
     */
    public synchronized void treeStructureChanged(TreeModelEvent e)
    {
	if(e != null && !largeModel)
	{
	    Object            changedPath[];
	    VisibleTreeNode   changedNode;

	    changedPath = e.getPath();
	    changedNode = getNodeForPath(changedPath, false, false);
	    if(changedNode == null && changedPath != null &&
	       changedPath.length == 1)
		changedNode = treeCacheRoot;
	    if(changedNode != null)
	    {
		boolean                   wasExpanded, wasVisible;
		int                       newIndex;

		wasExpanded = changedNode.isExpanded();
		wasVisible = (changedNode.getRow() != -1);
		if(changedNode == treeCacheRoot) {
		    this.rebuild();
		}
		else {
		    int                              nodeIndex, oldRow;
		    VisibleTreeNode                  newNode, parent;

		    /* Remove the current node and recreate a new one. */
		    parent = (VisibleTreeNode)changedNode.getParent();
		    nodeIndex = parent.getIndex(changedNode);
		    if(wasVisible && wasExpanded)
		    {
			changedNode.collapse(false);
		    }
		    if(wasVisible)
			visibleNodes.removeElement(changedNode);
		    changedNode.removeFromParent();
		    createNodeAt(parent, nodeIndex);
		    newNode = (VisibleTreeNode)parent.getChildAt(nodeIndex);
		    if(wasVisible && wasExpanded)
			newNode.expand(false);
		    newIndex = newNode.getRow();
		    if(!isFixedRowHeight() && wasVisible)
		    {
			if(newIndex == 0)
			    updateYLocationsFrom(newIndex);
			else
			    updateYLocationsFrom(newIndex - 1);
			this.visibleNodesChanged();
		    }
		    else if(wasVisible)
			this.visibleNodesChanged();
		}
	    }
	}
	else if(e != null) {
	    Object            changedPath[] = e.getPath();

	    if(changedPath != null && changedPath.length > 0) {
		if(changedPath.length == 1)
		    rebuild();
		else {
		    LargeTreeModelNode changedNode=getLargeTreeModelNodeForPath
			                        (changedPath, false, false);

		    if(changedNode != null) {
			boolean             wasExpanded, wasVisible;
			int                 newIndex;
			LargeTreeModelNode  parent = (LargeTreeModelNode)
			                       changedNode.getParent();

			wasExpanded = changedNode.isExpanded();
			wasVisible = changedNode.isVisible();
			if(wasVisible && wasExpanded) {
			    changedNode.collapse(false);
			    changedNode.removeFromParent();
			    changedNode = getLargeTreeModelNodeForPath
				(changedPath, false, true);
			    changedNode.expand(false);
			}
			else
			    changedNode.removeFromParent();
			if(treeSelectionModel != null && wasVisible &&
			   wasExpanded)
			    treeSelectionModel.resetRowSelection();
			if(wasVisible)
			    this.visibleNodesChanged();
			else
			    parent.modelChildCountChanged();
		    }
		}
	    }
	}
    }
    
    /**
     * Sent to completely rebuild the visible tree. All nodes are collapsed.
     */
    public void rebuild() {
	if(!largeModel && treeCacheRoot != null)
	    treeCacheRoot.markInvalid();
	if(treeModel != null) {
	    if(!largeModel) {
		largeRoot = null;
		treeCacheRoot = createNodeForValue(treeModel.getRoot(), 0);
		visibleNodes.removeAllElements();
		if (isRootVisible())
		    visibleNodes.addElement(treeCacheRoot);
		if(!treeCacheRoot.isExpanded())
		    treeCacheRoot.expand();
		else {
		    Enumeration cursor = treeCacheRoot.children();
		    while(cursor.hasMoreElements()) {
			visibleNodes.addElement(cursor.nextElement());
		    }
		    if(!isFixedRowHeight())
			updateYLocationsFrom(0);
		}
	    }
	    else {
		treeCacheRoot = null;
		visibleNodes.removeAllElements();
		largeRoot = createLargeTreeModelNodeForValue
		                 (treeModel.getRoot(), 0);
		if(isRootVisible())
		    largeRowCount = 1;
		else
		    largeRowCount = 0;
		largeRoot.expand(true);
	    }
	}
	else {
	    visibleNodes.removeAllElements();
	    treeCacheRoot = null;
	    largeRoot = null;
	    largeRowCount = 0;
	}
	/* Clear out the selection model, might not always be the right
	   thing to do, but the tree is being rebuilt, soooo.... */
	if(treeSelectionModel != null) {
	    treeSelectionModel.clearSelection();
	}
	this.visibleNodesChanged();
    }

    /**
      * Resets the y origin of all the visible nodes as well as messaging
      * all the visible nodes to updatePreferredSize().  You should not
      * normally have to call this.  Expanding and contracting the nodes
      * automaticly adjusts the locations.
      * updateAll determines if updatePreferredSize() is call on all nodes
      * or just those that don't have a valid size.
      * Only used if largeModel is false.
      */
    public void updateNodeSizes(boolean updateAll)
    {
	int                      aY, counter, maxCounter;
	VisibleTreeNode          node;

	updateNodeSizes = false;
	for(aY = counter = 0, maxCounter = visibleNodes.size();
	    counter < maxCounter;counter++)
	{
	    node = (VisibleTreeNode)visibleNodes.elementAt(counter);
	    node.setYOrigin(aY);
	    if(updateAll || !node.hasValidSize())
		node.updatePreferredSize(counter);
	    aY += node.getPreferredSize().height;
	}
    }

    /**
      * Returns the y origin of row.
      */
    public int getYOriginOfRow(int row)
    {
	if(row < 0)
	    return -1;
	if(row >= getRowCount())
	    return -1;
	if(isFixedRowHeight())
	    return (row * getRowHeight());
	return getNode(row).getYOrigin();
    }

    /**
      * Returns the index of the row containing location.  If there
      * are no rows, -1 is returned.  If location is beyond the last
      * row index, the last row index is returned.
      */
    public int getRowContainingYLocation(int location)
    {
	if(isFixedRowHeight())
	{
	    if(getRowCount() == 0)
		return -1;
	    return Math.max(0, Math.min(getRowCount() - 1,
					location / getRowHeight()));
	}
	int                    max, maxY, mid, min, minY;
	VisibleTreeNode        node;

	if((max = getRowCount()) <= 0)
	    return -1;
	mid = min = 0;
	while(min < max)
	{
	    mid = (max - min) / 2 + min;
	    node = (VisibleTreeNode)visibleNodes.elementAt(mid);
	    minY = node.getYOrigin();
	    maxY = minY + node.getPreferredSize().height;
	    if(location < minY)
	    {
		max = mid - 1;
	    }
	    else if(location >= maxY)
	    {
		min = mid + 1;
	    }
	    else
		break;
	}
	if(min == max)
	{
	    mid = min;
	    if(mid >= getRowCount())
		mid = getRowCount() - 1;
	}
	return mid;
    }

    /**
     * Ensures that all the path components in path are expanded, accept
     * for the last component which will only be expanded if expandLast
     * is true.
     * Returns true if succesful in finding the path.
     */
    protected boolean ensureLargePathIsExpanded(TreePath path,
					     boolean expandLast) {
	Object[]               aPath = path.getPath();

	if(aPath != null) {
	    int                    pathLength = aPath.length;

	    if(treeModel.isLeaf(aPath[pathLength - 1])) {
		if(pathLength == 1)
		    aPath = null;
		else {
		    Object[]           tPath = new Object[pathLength - 1];

		    System.arraycopy(aPath, 0, tPath, 0, pathLength - 1);
		    aPath = tPath;
		    expandLast = true;
		}
	    }
	    if(aPath != null) {
		LargeTreeModelNode     lastNode = getLargeTreeModelNodeForPath
		                                 (aPath, false, true);

		if(lastNode != null) {
		    Object[]           expPath = lastNode.getPath();
		    int                maxCounter = expPath.length - 1;

		    for(int counter = 0; counter < maxCounter; counter++) {
			((LargeTreeModelNode)expPath[counter]).expand(true);
		    }
		    if(expandLast)
			((LargeTreeModelNode)expPath[maxCounter]).expand(true);
		    return true;
		}
	    }
	}
	return false;
    }

    /**
      * Returns the VisibleTreeNode instances that are used to
      * represent the user values in <i>path</i>, as well as insuring
      * that all the nodes are visible.  If any component of the path is
      * not valid, null will be returned.<p>
      * Only used if largeModel is false.
      */
    protected VisibleTreeNode[] ensurePathIsExpanded(TreePath path,
						     boolean expandLast)
    {
	VisibleTreeNode[]        nodePath = getNodesForTreePath(path, false,
								true);

	if(nodePath != null) {
	    for (int counter = 0, maxCounter = (expandLast ? nodePath.length :
						nodePath.length - 1);
		 counter < maxCounter; counter++) {
		if(!nodePath[counter].isExpanded())
		    nodePath[counter].expand();
	    }
	}
	return nodePath;
    }

    /**
      * Returns the path to the first selected value, or null if
      * nothing is currently selected.
      */
    public TreePath getSelectionPath() {
	if(treeSelectionModel != null)
	    return treeSelectionModel.getSelectionPath();
	return null;
    }

    /**
      * Returns the path of the selected values, or null if nothing is
      * current selected.
      */
    public TreePath[] getSelectionPaths() {
	if(treeSelectionModel != null)
	    return treeSelectionModel.getSelectionPaths();
	return null;
    }

    /**
      * Returns all of the currently selected rows.
      */
    public int[] getSelectionRows() {
	if(treeSelectionModel != null)
	    return treeSelectionModel.getSelectionRows();
	return null;
    }

    /**
      * Gets the first selected row.
      */
    public int getMinSelectionRow() {
	if(treeSelectionModel != null)
	    return treeSelectionModel.getMinSelectionRow();
	return -1;
    }

    /**
      * Gets the last selected row.
      */
    public int getMaxSelectionRow() {
	if(treeSelectionModel != null)
	    return treeSelectionModel.getMaxSelectionRow();
	return -1;
    }

    /**
      * Returns true if item identified by path is currently selected.
      */
    public boolean isPathSelected(TreePath path) {
	if(treeSelectionModel != null)
	    return treeSelectionModel.isPathSelected(path);
	return false;
    }

    /**
      * Returns true if the row identitifed by row is selected.
      */
    public boolean isRowSelected(int row) {
	if(treeSelectionModel != null)
	    return treeSelectionModel.isRowSelected(row);
	return false;
    }

    /**
      * Returns true if the value identified by path is currently expanded,
      * this will return false if any of the values in path are currently
      * not being displayed.
      */
    public boolean isExpanded(TreePath path) {
	if(!largeModel) {
	    VisibleTreeNode         lastNode = getNodeForTreePath(path, true,
								  false);

	    if(lastNode != null)
		return lastNode.isExpanded();
	}
	else if(path != null) {
	    LargeTreeModelNode     lastNode = getLargeTreeModelNodeForPath
		                      (path.getPath(), true, false);

	    return (lastNode != null && lastNode.isExpanded());
	}
	return false;
    }

    /**
      * Returns true if the value identified by row is currently expanded.
      */
    public boolean isExpanded(int row) {
	if(!largeModel) {
	    return getNode(row).isExpanded();
	}
	LargeTreeModelNode     eNode = getLargeTreeModelNodeForRow(row, false);

	if(eNode != null)
	    return eNode.isExpanded();
	return false;
    }

    /**
      * Returns true if the value identified by path is currently collapsed,
      * this will return false if any of the values in path are currently
      * not being displayed.
      */
    public boolean isCollapsed(TreePath path) {
	return !isExpanded(path);
    }

    /**
      * Returns true if the value identified by row is currently collapsed.
      */
    public boolean isCollapsed(int row) {
	return !isExpanded(row);
    }

    /**
     * Returns true if the node at <code>row</code> is a leaf.
     */
    public boolean isLeaf(int row) {
	if(!largeModel) {
	    VisibleTreeNode            node = getNode(row);

	    if(node != null)
		return node.isLeaf();
	}
	else {
	    if(row == 0 && isRootVisible())
		return treeModel.isLeaf(treeModel.getRoot());
	    else {
		int[]                    childIndex = new int[1];
		LargeTreeModelNode       parent;

		parent = getLargeParentAndChildIndexOfRow(row, childIndex);
		if(parent != null)
		    return treeModel.isLeaf(treeModel.getChild
				     (parent.getUserObject(), childIndex[0]));
	    }
	}
	return true;
    }

    /**
      * Ensures that all of the parents of path are currently expanded.
      * To make sure it is truyly visible, you may wish to call
      * scrollPathToVisible, which will try and scroll the path to be
      * visibile if the JTree is in a scroller.
      */
    public void makeVisible(TreePath path) {
	if(!largeModel)
	    ensurePathIsExpanded(path, false);
	else
	    ensureLargePathIsExpanded(path, false);
    }

    /**
      * Returns true if all the parents of path are currently expanded.
      */
    public boolean isVisible(TreePath path) {
	if(!largeModel) {
	    VisibleTreeNode         lastNode = getNodeForTreePath(path, true,
								  false);

	    if(lastNode != null)
		return true;
	}
	else if(path != null) {
	    Object[]                oPath = path.getPath();
	    if(oPath != null) {
		int                     pathLength = oPath.length;
		if(pathLength > 0) {
		    LargeTreeModelNode        lastNode;

		    if(treeModel.isLeaf(oPath[pathLength - 1]) &&
		       pathLength > 1) {
			Object[]        tempPath = new Object[pathLength - 1];

			System.arraycopy(oPath, 0, tempPath, 0, pathLength- 1);
			lastNode = getLargeTreeModelNodeForPath(tempPath, true,
							  false);
			return (lastNode != null && lastNode.isVisible() &&
				lastNode.isExpanded());
		    }
		    lastNode = getLargeTreeModelNodeForPath(oPath, true,false);
		    return lastNode.isVisible();
		}
	    }
	}
	return false;
    }

    /**
     * Returns the number of visible rows.
     */
    public int getRowCount() {
	if(!largeModel)
	    return visibleNodes.size();
	return largeRowCount;
    }

    /**
      * Returns the Rectangle enclosing the label portion that the
      * last item in path will be drawn into.  Will return null if
      * any component in path is currently valid.
      */
    public Rectangle getPathBounds(TreePath path) {
	if(!largeModel) {
	    VisibleTreeNode      node = getNodeForTreePath(path, true, false);

	    if(node != null)
		return node.getNodeBounds();
	}
	else if(path != null) {
	    return getRowBounds(getRowForPath(path));
	}
	return null;
    }

    /**
      * Returns the Rectangle enclosing the label portion that the
      * item identified by row will be drawn into.
      */
    public Rectangle getRowBounds(int row) {
	if(row >= 0 && row < getRowCount()) {
	    if(!largeModel)
		return getNode(row).getNodeBounds();
	    else {
		if(row == 0 && isRootVisible())
		    return getLargeBoundsOf(null, 0,
					    largeRoot.getUserObject());

		int[]           cIndex = new int[1];
		LargeTreeModelNode parent = getLargeParentAndChildIndexOfRow
		                  (row, cIndex);

		if(parent != null)
		    return getLargeBoundsOf(parent, row, treeModel.
				 getChild(parent.getUserObject(), cIndex[0]));
	    }
	}
	return null;
    }

    /**
      * Returns the path for passed in row.  If row is not visible
      * null is returned.
      */
    public TreePath getPathForRow(int row) {
	if(row >= 0 && row < getRowCount()) {
	    if(!largeModel)
		return getNode(row).getTreePath();
	    else
		return getLargePathForRow(row);
	}
	return null;
    }

    /**
      * Returns the row that the last item identified in path is visible
      * at.  Will return -1 if any of the elements in path are not
      * currently visible.
      */
    public int getRowForPath(TreePath path) {
	if(path == null)
	    return -1;

	if(!largeModel) {
	    VisibleTreeNode    visNode = getNodeForTreePath(path, true, false);

	    if(visNode != null)
		return visNode.getRow();
	    else
		return -1;
	}
	else {
	    return getLargeRowForPath(path.getPath());
	}
    }

    /**
     * Returns the rows that the TreePath instances in <code>path</code>
     * are being displayed at. The receiver should return an array of
     * the same length as that passed in, and if one of the TreePaths
     * in <code>path</code> is not valid its entry in the array should
     * be set to -1.
     */
    public int[] getRowsForPaths(TreePath[] paths) {
	if(paths == null)
	    return null;

	int               numPaths = paths.length;
	int[]             rows = new int[numPaths];

	for(int counter = 0; counter < numPaths; counter++)
	    rows[counter] = getRowForPath(paths[counter]);
	return rows;
    }

    /**
     * Returns the <code>TreePath</code>s for the indices in 
     * <code>rows</code>. The receiver should return an array of the same
     * size as that passed in, and if one of the rows is invalid it
     * should be set to null.
     */
    public TreePath[] getPathsForRows(int[] rows) {
	if(rows == null)
	    return null;

	int              numRows = rows.length;
	TreePath[]       paths = new TreePath[numRows];

	for(int counter = 0; counter < numRows; counter++)
	    paths[counter] = getPathForRow(rows[counter]);
	return paths;
    }

    /**
      * Ensures that the last item identified in path is expanded and
      * visible.
      */
    public void expandPath(TreePath path) {
	if(path != null) {
	    if(!largeModel)
		ensurePathIsExpanded(path, true);
	    else
		ensureLargePathIsExpanded(path, true);
	}
    }

    /**
      * Insures that the item identified by row is expanded.
      */
    public void expandRow(int row) {
	if(row >= 0 && row < getRowCount()) {
	    if(!largeModel)
		getNode(row).expand();
	    else {
		LargeTreeModelNode node =getLargeTreeModelNodeForRow(row,true);

		if(node != null)
		    node.expand(true);
	    }
	}
    }

    /**
      * Insures that the last item identified in path is collapsed and
      * visible.
      */
    public void collapsePath(TreePath path) {
	if(!largeModel) {
	    VisibleTreeNode[]      nodePath = getNodesForTreePath(path, false,
								  true);

	    if(nodePath != null) {
		int             counter, maxCounter;

		for (counter = 0, maxCounter = nodePath.length - 1;
		     counter < maxCounter; counter++) {
		    if(!nodePath[counter].isExpanded())
			nodePath[counter].expand();
		}
		if(nodePath.length > 0)
		    nodePath[counter].collapse();
	    }
	}
	else if(path != null) {
	    LargeTreeModelNode      node = getLargeTreeModelNodeForPath
		                        (path.getPath(), false, true);

	    if(node != null)
		node.collapse(true);
	}
    }

    /**
      * Insures that the item identified by row is collapsed.
      */
    public void collapseRow(int row) {
	if(row >= 0 && row < getRowCount()) {
	    if(!largeModel)
		getNode(row).collapse();
	    else {
		LargeTreeModelNode node = getLargeTreeModelNodeForRow
		                        (row,false);

		if(node != null)
		    node.collapse(true);
	    }
	}
    }


    /**
      * Returns the path to the node that is closest to x,y.  If
      * there is nothing currently visible this will return null, otherwise
      * it'll always return a valid path.  If you need to test if the
      * returned object is exactly at x, y you should get the bounds for
      * the returned path and test x, y against that.
      */
    public TreePath getClosestPathForLocation(int x, int y) {
	if(getRowCount() == 0)
	    return null;
	int                row = getRowContainingYLocation(y);

	if(!largeModel)
	    return getNode(row).getTreePath();
	return getLargePathForRow(row);
    }

    /**
      * Returns the row to the node that is closest to x,y.  If
      * there is nothing currently visible this will return -1, otherwise
      * it'll always return a valid row.  If you need to test if the
      * returned object is exactly at x, y you should get the bounds for
      * the returned row and test x, y against that.
      */
    public int getClosestRowForLocation(int x, int y) {
	if(getRowCount() == 0)
	    return -1;
	return getRowContainingYLocation(y);
    }

    /**
     * Returns the data model object displayed at the given row
     */
    public Object getValue(int row) {
	if(!largeModel)
	    return getNode(row).getValue();
	int[]         cIndex = new int[1];
	LargeTreeModelNode  eNode = getLargeParentAndChildIndexOfRow(row, cIndex);

	if(row == 0 && isRootVisible())
	    return treeModel.getRoot();
	if(eNode != null)
	    return treeModel.getChild(eNode.getUserObject(), cIndex[0]);
	return null;
    }

    /**
     * Returns the AbstractTreeUI.VisibleNode displayed at the given row
      * Only used if largeModel is false.
     */
    public VisibleTreeNode getNode(int row) {
	return (VisibleTreeNode)visibleNodes.elementAt(row);
    }

    /**
     * Returns an enumeration of AbstractTreeUI.VisibleTreeNode.  There is one
     * element per visible row, and this enumerates through the rows from top
     * to bottom.
      * Only used if largeModel is false.
     */
    public Enumeration visibleNodes() {
	return visibleNodes.elements();
    }

    /**
      * Returns the maximum node width. If not largeModel then 
      * this will repeatedly call getXOrigin() on this as well as
      * getPreferredSize() on each node to determine the size.
      * If this is a largeModel, then this will repeatedly get the bounds
      * of each row (getRowBounds), in other words this is very expensive
      * for largeModels.
      */
    public int getMaxNodeWidth() {
	int                     maxWidth = 0;
	if(!largeModel) {
	    int                 counter, nodeMaxX;
	    VisibleTreeNode     node;
	    
	    for(counter = getRowCount() - 1;counter >= 0;counter--) {
		node = this.getNode(counter);
		if((nodeMaxX = node.getPreferredSize().width +
		    this.getXOriginOfNode(node)) > maxWidth)
		    maxWidth = nodeMaxX;
	    }
	}
	else {
	    Rectangle           nodeBounds;

	    for(int counter = getRowCount() - 1;counter >= 0;counter--) {
		nodeBounds = getRowBounds(counter);
		maxWidth = Math.max(maxWidth, nodeBounds.x + nodeBounds.width);
	    }
	}
	return maxWidth;
    }

    /**
     * Sets the TreeSelectionModel used to manage the selection to
     * new LSM.
     */
    public void setSelectionModel(TreeSelectionModel newLSM) {
	if(treeSelectionModel != newLSM) {
	    if(treeSelectionModel != null) {
		treeSelectionModel.removeTreeSelectionListener(this);
		treeSelectionModel.setRowMapper(null);
	    }
	    treeSelectionModel = newLSM;
	    if(treeSelectionModel != null) {
		treeSelectionModel.addTreeSelectionListener(this);
		treeSelectionModel.setRowMapper(this);
		treeSelectionModel.resetRowSelection();
	    }
	}
    }

    /**
     * Returns the model used to maintain the selection.
     */
    public TreeSelectionModel getSelectionModel() {
	return treeSelectionModel;
    }

    /**
     * Returns true if row is selected.
     */
    public boolean isSelectedIndex(int row) {
	if(treeSelectionModel != null)
	    return treeSelectionModel.isRowSelected(row);
	return false;
    }

    /**
     * Notification that the model has changed. Makes sure all the paths
     * in the TreeSelectionEvent <code>e</code> are visible (that is
     * there are parents are expanded).
     */
    public void valueChanged(TreeSelectionEvent e) {
	if(treeSelectionModel != null) {
	    TreePath[]           paths = treeSelectionModel
		                         .getSelectionPaths();

	    if(paths != null) {
		for(int counter = paths.length - 1; counter >= 0; counter--) {
		    makeVisible(paths[counter]);
		}
	    }
	}
    }

    /**
      * Responsible for creating a VisibleTreeNode that will be used
      * to track display information about value.
      */
    protected VisibleTreeNode createNodeForValue(Object value, int index) {
	return new VisibleTreeNode(this, value, index);
    }



    //
    // Following are primarily used for the large data sets.
    //

    /**
     * Creates and returns an instance of LargeTreeModelNode.
     */
    protected LargeTreeModelNode createLargeTreeModelNodeForValue(Object value,
						      int childIndex) {
	return new LargeTreeModelNode(this, value, childIndex);
    }

    /**
     * Returns the parent Expanded node for the passed in row. Will
     * return null for the root, or any rows outside of the current range.
     * Also, returns by refernence in cIndex the child index from
     * the model.
     */
    protected LargeTreeModelNode getLargeParentAndChildIndexOfRow
	                   (int row, int[] cIndex) {
	int[]            rowIndex = new int[1];
	LargeTreeModelNode[]   retNode = new LargeTreeModelNode[1];
	boolean[]        isParent = new boolean[1];

	if(isRootVisible())
	    rowIndex[0] = 0;
	else
	    rowIndex[0] = -1;
	if(largeRoot.getPathForRow(row, rowIndex, null, retNode, isParent,
				   cIndex)) {
	    if(isParent[0])
		return retNode[0];
	    cIndex[0] = retNode[0].getChildIndex();
	    return (LargeTreeModelNode)retNode[0].getParent();
	}
	return null;
    }

    /**
     * Returns the LargeTreeModelNode for the passed in row. This will return
     * null if row is > then number of rows, or if there isn't currently
     * an expanded node for the passed in row.
     */
    protected LargeTreeModelNode getLargeTreeModelNodeForRow(int row,
						 boolean shouldCreate) {
	int[]                  rowIndex = new int[1];
	int[]                  cIndex = new int[1];
	LargeTreeModelNode[]   retNode = new LargeTreeModelNode[1];
	boolean[]              isParent = new boolean[1];

	if(isRootVisible())
	    rowIndex[0] = 0;
	else
	    rowIndex[0] = -1;
	if(largeRoot.getPathForRow(row, rowIndex, null, retNode, isParent,
				   cIndex)) {
	    if(isParent[0]) {
		if(shouldCreate) {
		    Object childUO = treeModel.getChild(retNode[0]
					      .getUserObject(), cIndex[0]);

		    if(treeModel.isLeaf(childUO))
			return null;
		    /* Not a leaf, and no LargeTreeModelNode, need to
		       create one. */
		    LargeTreeModelNode  newNode;

		    newNode = createLargeTreeModelNodeForValue
			               (childUO, cIndex[0]);
		    retNode[0].addLargeTreeModelNode(newNode);
		    return newNode;
		}
		return null;
	    }
	    return retNode[0];
	}
	return null;
    }
    
    /**
     * Returns the TreePath of the passed in row.
     */
    protected TreePath getLargePathForRow(int row) {
	int[]            rowIndex = new int[1];
	TreePath[]       retPath = new TreePath[1];

	if(isRootVisible())
	    rowIndex[0] = 0;
	else
	    rowIndex[0] = -1;
	if(largeRoot.getPathForRow(row, rowIndex, retPath, null, null, null))
	    return retPath[0];
	return null;
    }

    /**
     * Returns the row for the given TreePath.
     */
    protected int getLargeRowForPath(Object[] path) {
	int[]            rowIndex = new int[1];

	if(isRootVisible())
	    rowIndex[0] = 0;
	else
	    rowIndex[0] = -1;
	if(path != null &&
	   largeRoot.getRow(path, 0, path.length, true, rowIndex))
	    return rowIndex[0];
	return -1;
    }

    /**
     * Messages getLageTreeModelNodeForPath(path, onlyIfVisible, shouldCreate,
     * path.length as long as path is non-null and the length is > 0.
     * Otherwise returns null.
     */
    protected LargeTreeModelNode getLargeTreeModelNodeForPath(Object[] path,
						  boolean onlyIfVisible,
						  boolean shouldCreate) {
	if(path != null) {
	    int                  pathLength = path.length;
	    if(pathLength > 0)
		return getLargeTreeModelNodeForPath(path, onlyIfVisible,
					      shouldCreate, pathLength);
	}
	return null;
    }

    /**
     * Returns the LargeTreeModelNode instance for the given path.
     * Path should only contain non-leafs.
     * If onlyIfVisible null will be returned if any items in the path
     * are not expanded.
     * If shouldCreate is true and some of the path is currently expanded,
     * it will be created and set to expanded.
     * 
     */
    protected LargeTreeModelNode getLargeTreeModelNodeForPath(Object[] path,
						  boolean onlyIfVisible,
						  boolean shouldCreate,
						  int pathLength) {
	if(path[0].equals(largeRoot.getUserObject())) {
	    LargeTreeModelNode             aNode;
	    LargeTreeModelNode             beginLastParent;
	    LargeTreeModelNode             lastParent = largeRoot;
	    int                            cCounter, maxCCounter, wantIndex;

	    for(int counter = 1; counter < pathLength; counter++) {
		if(onlyIfVisible && !lastParent.isExpanded)
		    return null;
		wantIndex = treeModel.getIndexOfChild
		            (lastParent.getUserObject(), path[counter]);
		if(wantIndex < 0)
		    throw new RuntimeException("invalid index " + wantIndex +
					       " for path " + path[counter]);
		maxCCounter = lastParent.getChildCount();
		beginLastParent = lastParent;
		for(cCounter = 0; cCounter < maxCCounter; cCounter++) {
		    aNode = (LargeTreeModelNode)lastParent
			           .getChildAt(cCounter);
		    if(aNode.childIndex == wantIndex) {
			lastParent = aNode;
			cCounter = maxCCounter;
		    }
		    else if(aNode.childIndex > wantIndex) {
			if(shouldCreate) {
			    LargeTreeModelNode      newNode;

			    newNode = createLargeTreeModelNodeForValue
				         (path[counter], wantIndex);

			    lastParent.insert(newNode, cCounter);
			    lastParent = newNode;
			    cCounter = maxCCounter;
			}
			else
			    return null;
		    }
		}
		if(beginLastParent == lastParent) {
		    if(shouldCreate) {
			LargeTreeModelNode      newNode;

			newNode = createLargeTreeModelNodeForValue
			                (path[counter], wantIndex);

			lastParent.add(newNode);
			lastParent = newNode;
		    }
		    else
			return null;
		}
	    }
	    return lastParent;
	}
	return null;
    }

    protected AbstractTreePath createTreePathFor(VisibleTreeNode node) {
	return new AbstractTreePath(node.getUserObjectPath(), node);
    }

    // Serialization support.  
    private void writeObject(ObjectOutputStream s) throws IOException {
	Vector      values = new Vector();

	s.defaultWriteObject();
	// Save the treeModel, if its Serializable.
	if(treeModel != null && treeModel instanceof Serializable) {
	    values.addElement("treeModel");
	    values.addElement(treeModel);
	}
	// Save the treeselectionModel, if its Serializable.
	if(treeSelectionModel != null &&
	   treeSelectionModel instanceof Serializable) {
	    values.addElement("treeSelectionModel");
	    values.addElement(treeSelectionModel);
	}
	s.writeObject(values);
    }

    private void readObject(ObjectInputStream s) 
	throws IOException, ClassNotFoundException {
	s.defaultReadObject();

	Vector          values = (Vector)s.readObject();
	int             indexCounter = 0;
	int             maxCounter = values.size();

	if(indexCounter < maxCounter && values.elementAt(indexCounter).
	   equals("treeModel")) {
	    treeModel = (TreeModel)values.elementAt(++indexCounter);
	    indexCounter++;
	}
	if(indexCounter < maxCounter && values.elementAt(indexCounter).
	   equals("treeSelectionModel")) {
	    treeSelectionModel = (TreeSelectionModel)values
		                 .elementAt(++indexCounter);
	    indexCounter++;
	}
    }
}

/**
 * AbstractTreePath extends TreePath by keeping track of the
 * VisibleTreeNode that is used to track the data from the model.
 * This information is used to eficiently track where a path
 * currently is on the screen.
 */
class AbstractTreePath extends TreePath
{
    /** Used to to determine visibility. */
    VisibleTreeNode           node;

    public AbstractTreePath(Object[] newPath, VisibleTreeNode node) {
	super(newPath);
	this.node = node;
    }

    /**
     * Returns the UI the receiver represents.
     */
    public AbstractTreeUI getUI() {
	return node.treeUI;
    }

    public boolean equals(Object o) {
	if((o instanceof AbstractTreePath)) {
	    AbstractTreePath       aTP = (AbstractTreePath)o;

	    /* We really can't make any more assumptions than this. */
	    if(node != null && aTP.node != null &&
	       aTP.node.treeUI == node.treeUI) {
		if(aTP.node == node) {
		    return true;
		}
		if(node.isValid && aTP.node.isValid)
		{
		    return false;
		}
	    }
	}
	return super.equals(o);
    }

    /**
     * Returns the hashCode for the object.  This must be defined
     * here to ensure 100% pure.
     *
     * @return the hashCode for the object
     */
    public int hashCode() { 
	return super.hashCode();
    }
    
    public String toString() {
	StringBuffer    aString = new StringBuffer();

	aString.append("AbstractTreePath: VN ");
	if(node != null)
	    aString.append(node.hashCode() + " {");
	else
	    aString.append("NULL {");
	if(path != null) {
	    for(int counter = 0; counter < path.length; counter++) {
		if(counter > 0)
		    aString.append(", ");
		aString.append(path[counter].toString());
	    }
	}
	aString.append("}");
	return aString.toString();
    }
} // End of class AbstractTreeUI.AbstractTreePath

