/*
 * @(#)FontChooser.java	1.11 01/28/98
 * 
 * Copyright (c) 1997 Sun Microsystems, Inc. All Rights Reserved.
 * 
 * This software is the confidential and proprietary information of Sun
 * Microsystems, Inc. ("Confidential Information").  You shall not disclose
 * such Confidential Information and shall use it only in accordance with the
 * terms of the license agreement you entered into with Sun.
 * 
 * SUN MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE,
 * OR NON-INFRINGEMENT. SUN SHALL NOT BE LIABLE FOR ANY DAMAGES SUFFERED BY
 * LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE OR
 * ITS DERIVATIVES.
 * 
 * @author James Gosling
 */

package com.sun.java.swing;

import java.awt.*;
import java.awt.event.*;
import java.io.*;

import com.sun.java.swing.event.*;

public class FontChooser extends JComponent  implements ListSelectionListener,
           ChangeListener, ItemListener, AdjustmentListener
{
    private String name;
    private int size, maxsize, style;
    private String[] nameList;
    private boolean fontSpecified = false;
    private StringSpinner names;
    private Spinner sizes;
    private JCheckBox italic, bold;
    private Component preview;
    private String previewString = "Toy box";
    private static String[] dfSizes = {"8", "9", "10", "11", "12", "13", "14", "16", "18",
    "20", "24", "28", "32", "36", "48", "72"};


    /** Convenience method to prompt for a Font.
	@param fparent the parent component for the dialog box.
		fparent may be null if a default parent has
		been established with StandardDialog
	@param title a title for the dialog box
	@param initial the initial font
	@param target the ChangeListener that will be informed if the
			OK or Apply buttons are hit.
	@return If target is null,
		the dialog box will be modal and the method
		will return the final result, or null if canceled.
		Otherwise, the
		dialog box is non-modal, the method returns null
		immediatly, and the listener is informed when
		appropriate.  The source of the change event will
		be a FontChooser.
	@see StandardDialog
     */
    public static Font ask(java.awt.Component parent, String title, Font initial,
			        ChangeListener target) {
	StandardDialog tDialog;
	FontChooser tChooser = new FontChooser ();
	tDialog = new StandardDialog(parent, tChooser, target == null);
	if (target != null)
	    tDialog.addChangeListener(target);
	if (title != null)
	    tDialog.setTitle(title);
	if (initial != null)
	    tChooser.setSelectedFont(initial);
	tDialog.start();
	if (target != null)
	    return null;
	Font r = tDialog.isCanceled() ? null : tChooser.getSelectedFont();
	tDialog.dispose();
	return r;
    }

    public FontChooser () {
	super();
	name = "Serif";
	style = 0;
	size = 12;
	maxsize = 24;
	GridBagConstraints cons = new GridBagConstraints();
	setLayout(new GridBagLayout());
	cons.gridy = 0;
	cons.insets = new Insets(3, 3, 3, 3);
	cons.anchor = GridBagConstraints.CENTER;
	String[] s = getToolkit().getFontList();
	if (s != null) {
	    for (int i = s.length; --i >= 1;)
		for (int j = 1; j <= i; j++) {
		    String s1 = s[j - 1];
		    String s2 = s[j];
		    if (s1.compareTo(s2) > 0) {
			s[j - 1] = s2;
			s[j] = s1;
		    }
		}
	}
	names = new StringSpinner(0, null, s);
	int i;
	JComboBox combo = new JComboBox();
	for (int ni = 0; ni < s.length; ni++) {
	    combo.addPossibleValue((Object) (s[ni]));
	}
	combo.setCurrentValueIndex(names.getValue());
	combo.setEditable(true);
	SpinnerComboBoxEditor editor = new SpinnerComboBoxEditor(combo);
	editor.setEditorComponent(names);
	combo.setEditor(editor);
	cons.gridx = GridBagConstraints.RELATIVE;
	nameList = s;
	add(combo, cons);
	names.addAdjustmentListener(this);
	sizes = new Spinner(11);
	sizes.setMinimum(0);
	sizes.setMaximum(999);
	add(sizes, cons);
	sizes.addAdjustmentListener(this);
	bold = new JCheckBox("Bold");
	bold.addChangeListener(this);
	add(bold, cons);

	italic = new JCheckBox("Italic");
	italic.addChangeListener(this);
	add(italic, cons);

	cons.gridx = 0;
	cons.gridy = 1;
	cons.gridheight = GridBagConstraints.REMAINDER;
	cons.gridwidth = GridBagConstraints.REMAINDER;
	cons.fill = GridBagConstraints.BOTH;
	cons.weighty = 5;
	preview = new Component() {
	    int ascent;
	    public void paint(Graphics g) {
		g.setFont(getFont());
		g.drawString(previewString, 0, ascent);
	    }
	    public Dimension getPreferredSize() {
		ascent = maxsize*3/4;
		return new Dimension(maxsize, maxsize);
	    }
	    public Dimension getMinimumSize() {
		return getPreferredSize();
	    }
	};
	preview.setFont(new Font ("TimesRoman", 0, 36));
	add(preview, cons);
    }
    public void stateChanged(ChangeEvent e) {
	recalc();
    }
    public void adjustmentValueChanged(AdjustmentEvent e) {
	recalc();
    }
    public void itemStateChanged(ItemEvent e) {
	recalc();
    }
    public void valueChanged(ListSelectionEvent e) {
	recalc();
    }
    private void recalc() {
	String n = names.getValueName();
	if (n != null)
	    name = n;
	size = sizes.getValue();
	int nstyle = 0;
	if (italic.getModel().isSelected())
	    nstyle |= Font.ITALIC;
	if (bold.getModel().isSelected())
	    nstyle |= Font.BOLD;
	style = nstyle;
	preview.setFont(getSelectedFont());
	if (size > maxsize && maxsize<150) {
	    maxsize = (size+20)/20*20;
	    if (maxsize>150) maxsize = 150;
	    preview.invalidate();
	    Component c = this;
	    while (!(c instanceof java.awt.Window) && c.getParent() != null)
		c = c.getParent();
	    if (c instanceof Window) {
		Window w = (Window) c;
		w.pack();
		w.setSize(w.getPreferredSize());
	    } else
		c.validate();
	}
	preview.repaint();
    }
    public String toString() {
	String s = name + size;
	if ((style & Font.BOLD) !=0)
	    s += "-bold";
	if ((style & Font.ITALIC) !=0)
	    s += "-italic";
	return s;
    }

    public Font getSelectedFont() {
	return new Font (name, style, size>0 ? size : 1);
    }
    public void setSelectedFont(Font c) {
	setSelectedFont(c.getName(), c.getStyle(), c.getSize());
    }
    public void setPreviewString(String s) {
	previewString = s;
	preview.repaint();
    }
    public void setSelectedFont(String n, int sy, int sz) {
	name = n;
	style = sy;
	size = sz;
	italic.getModel().setSelected((sy & Font.ITALIC) !=0);
	bold.getModel().setSelected((sy & Font.BOLD) !=0);
	sizes.setValue(sz);
	String[] nl = nameList;
	for (int i = 0; i < nl.length; i++)
	    if (nl[i].equalsIgnoreCase(n)) {
		names.setValue(i);
		break;
	    }
	recalc();
    }
    private static class SpinnerComboBoxEditor implements ComboBoxEditor {
	private JComboBox myJComboBox;
	private Spinner editor;
	public SpinnerComboBoxEditor (JComboBox jcombobox) {
	    myJComboBox = jcombobox;
	}
	public Component getEditorComponent() {
	    return editor;
	}
	public void setEditorComponent(Spinner newEditor) {
	    editor = newEditor;
	}
	public void setValue(Object anObject) {
	    editor.setValue(myJComboBox.getCurrentValueIndex()
				+ editor.getMinimum());
	}
	public Object getValue() {
	    return editor.getText();
	}
	public void selectAll() {
	}
	public void addActionListener(ActionListener l) {
	}
	public void removeActionListener(ActionListener l) {
	}
    }
    /** A FontChooser.Patch is a small button-like component that
        displays a font sample.  Clicking it pops up a FontChooser
        dialog box to allow the font to be changed. */
    public static class Patch extends Component implements MouseListener {
	private Font c;
	private Dimension d;
	int ascent;
	/** Create a patch with the given font and a standard size */
	public Patch(Font c) {
	    this.c = c;
	    addMouseListener(this);
	}
	public void setSelectedFont(Font c) {
	    if (c != null && !c.equals(this.c)) {
		this.c = c;
		fireStateChanged();
		repaint();
	    }
	}
	public Font getSelectedFont() {
	    return c;
	}

	private ChangeEvent changeEvent;
	private EventListenerList listenerList = new EventListenerList();

	/**
	 * Adds a ChangeListener to the button.
	 */
	public void addChangeListener(ChangeListener l) {
	    listenerList.add(ChangeListener.class, l);
	}
	
	/**
	 * Removes a ChangeListener from the button.
	 */
	public void removeChangeListener(ChangeListener l) {
	    listenerList.remove(ChangeListener.class, l);
	}
	/*
	 * Notify all listeners that have registered interest for
	 * notification on this event type.  The event instance 
	 * is lazily created using the parameters passed into 
	 * the fire method.
	 * @see EventListenerList
	 */
	protected void fireStateChanged() {
	    // Guaranteed to return a non-null array
	    Object[] listeners = listenerList.getListenerList();
	    // Process the listeners last to first, notifying
	    // those that are interested in this event
	    for (int i = listeners.length-2; i>=0; i-=2) {
		if (listeners[i]==ChangeListener.class) {
		    // Lazily create the event:
		    if (changeEvent == null)
			changeEvent = new ChangeEvent(this);
		    ((ChangeListener)listeners[i+1]).stateChanged(changeEvent);
		}	       
	    }
	}	
	public void paint(Graphics g) {
	    Dimension d = getSize();
	    g.setColor(getBackground());
	    g.fillRect(1, 1, d.width - 2, d.height - 2);
	    g.setColor(Color.white);
	    g.drawLine(0, 0, d.width, 0);
	    g.drawLine(0, 0, 0, d.height);
	    g.setColor(Color.black);
	    g.drawLine(1, d.height - 1, d.width, d.height - 1);
	    g.drawLine(d.width - 1, 1, d.width - 1, d.height);
	    g.setColor(getForeground());
	    StringBuffer sb = new StringBuffer();
	    sb.append(c.getName());
	    sb.append(c.getSize());
	    if (c.isBold()) sb.append('b');
	    if (c.isItalic()) sb.append('i');
	    String l = sb.toString();
	    FontMetrics fm = getFontMetrics(getFont());
	    g.drawString(l, (d.width-fm.stringWidth(l))>>1, 
			 ((d.height-fm.getHeight())>>1)+fm.getAscent());
	}
	public void update(Graphics g) {
	    paint(g);
	}
	public Dimension getPreferredSize() {
	    if (d==null) {
	    Font f = getFont();
	    FontMetrics fm = Toolkit.getDefaultToolkit().getFontMetrics(f);
	    d = new Dimension(fm.stringWidth("ZapfChanceryMedium129bi")+2,
			      fm.getHeight()+6);
	    }
	    return d;
	}
	public Dimension getMinimumSize() {
	    return getPreferredSize();
	}
	public void mouseClicked(MouseEvent e) {
	}
	public void mousePressed(MouseEvent e) {
	    setSelectedFont(FontChooser.ask(this, "Pick a new font", c, null));
	}
	public void mouseReleased(MouseEvent e) {
	}
	public void mouseEntered(MouseEvent e) {
	}
	public void mouseExited(MouseEvent e) {
	}
    }
}
