/*
 * @(#)TableModelEvent.java	1.10 98/02/02
 * 
 * Copyright (c) 1997 Sun Microsystems, Inc. All Rights Reserved.
 * 
 * This software is the confidential and proprietary information of Sun
 * Microsystems, Inc. ("Confidential Information").  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sun.
 * 
 * SUN MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SUN SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 * 
 */

package com.sun.java.swing.event;

import java.util.EventObject;
import com.sun.java.swing.table.*;

/**
 * TableModelEvent is used to notify listeners that a table model
 * has changed. The model event describes changes to a TableModel 
 * and all references to rows and columns are in the co-ordinate 
 * system of the model. 
 * Depending on the parameters used in the constructors, the TableModelevent
 * can be used to specify the following types of changes: <p>
 *
 * <pre>
 * TableModelEvent(source);              //  The data, ie. all rows changed 
 * TableModelEvent(source, HEADER_ROW);  //  Structure change, reallcoate TableColumns
 * TableModelEvent(source, 1);           //  Row 1 changed
 * TableModelEvent(source, 3, 6);        //  Rows 3 to 6 inclusive changed
 * TableModelEvent(source, 2, 2, 6);     //  Cell at (2, 6) changed
 * TableModelEvent(source, 3, 6, ALL_COLUMNS, INSERT); // Rows (3, 6) were inserted
 * TableModelEvent(source, 3, 6, ALL_COLUMNS, DELETE); // Rows (3, 6) were deleted
 * </pre>
 *
 * It is possible to use other combinations of the parameters, not all of them 
 * are meaningful. By subclassing, you can add other information, for example: 
 * whether the event WILL happen or DID happen. This makes the specification 
 * of rows in DELETE events more useful but has not been included in 
 * the swing package as the JTable only needs post-event notification. 
 * <p>
 * Warning: serialized objects of this class will not be compatible with
 * future swing releases.  The current serialization support is appropriate
 * for short term storage or RMI between Swing1.0 applications.  It will
 * not be possible to load serialized Swing1.0 objects with future releases
 * of Swing.  The JDK1.2 release of Swing will be the compatibility
 * baseline for the serialized form of Swing objects.
 *
 * @version 1.10 02/02/98
 * @author Alan Chung
 * @author Philip Milne
 * @see TableModel
 */
public class TableModelEvent extends java.util.EventObject
{
    public static final int INSERT =  1;
    public static final int UPDATE =  0;
    public static final int DELETE = -1;
    
    public static final int HEADER_ROW = -1;

    public static final int ALL_COLUMNS = -1;
    
//
//  Instance Variables
//

    protected int       type;
    protected int	firstRow;
    protected int	lastRow;
    protected int	column;

//
// Constructors
//

    /** 
     *  All row data in the table has changed, listeners should discard any state 
     *  that was based on the rows and requery the TableModel to get the new 
     *  row count and all the appropriate values. 
     *  The JTable will repaint the entire visible region on recieving 
     *  this event, querying the model for the cell values that are visble. 
     *  The structure of the table ie, the column names, types and order 
     *  have not changed.  
     */
    public TableModelEvent(TableModel source) {
        // Use Integer.MAX_VALUE instead of getRowCount() in case rows were deleted. 
	this(source, 0, Integer.MAX_VALUE, ALL_COLUMNS, UPDATE);
    }
    
    /**
     *  This row of data has been updated. 
     *  To denote the arrival of a completely new table with a different structure 
     *  use <code>HEADER_ROW</code> as the value for the <I>row</I>. 
     *  When the JTable recieves this event and its <I>autoCreateColumnsFromModel</I> 
     *  flag is set it discards any TableColumns that it had and reallocates 
     *  default ones in the order they appear in the model. This is the 
     *  same as calling <code>setModel(TableModel)</code> on the JTable. 
     */
    public TableModelEvent(TableModel source, int row) {
	this(source, row, row, ALL_COLUMNS, UPDATE);
    }
    
    /**
     *  The data in rows [<I>firstRow</I>, <I>lastRow</I>] have been updated. 
     */
    public TableModelEvent(TableModel source, int firstRow, int lastRow) {
	this(source, firstRow, lastRow, ALL_COLUMNS, UPDATE);
    }
    
    /**
     *  The cells in column <I>column</I> in the range 
     *  [<I>firstRow</I>, <I>lastRow</I>] have been updated. 
     */
    public TableModelEvent(TableModel source, int firstRow, int lastRow, int column) {
	this(source, firstRow, lastRow, column, UPDATE);
    }
    
    /**
     *  The cells from (firstRow, column) to (lastRow, column) have been changed. 
     *  The <I>column</I> refers to the column index of the cell in the model's 
     *  co-ordinate system. When <I>column</I> is ALL_COLUMNS, all cells in the 
     *  specified range of rows are considered changed. 
     *  <p>
     *  The <I>type</I> should be one of: INSERT, UPDATE and DELETE. 
     */
    public TableModelEvent(TableModel source, int firstRow, int lastRow, int column, int type) {
	super(source);
	this.firstRow = firstRow;
	this.lastRow = lastRow;
	this.column = column;
	this.type = type;
    }

//
// Querying Methods
//

   /** Returns the first row that changed.  HEADER_ROW means the meta data, 
     * ie. names, types and order of the columns. 
     */
    public int getFirstRow() { return firstRow; };

    /** Returns the last row that changed. */
    public int getLastRow() { return lastRow; };
    
    /**
     *  Returns the column for the event.  If the return
     *  value is ALL_COLUMNS; it means every column in the specified
     *  rows changed.
     */
    public int getColumn() { return column; }; 
    
    /**
     *  Returns the type of event - one of: INSERT, UPDATE and DELETE.
     */
    public int getType() { return type; }
}
