/*
 * @(#)CompoundBorder.java	1.9 98/02/02
 *
 * Copyright (c) 1997 Sun Microsystems, Inc. All Rights Reserved.
 *
 * This software is the confidential and proprietary information of Sun
 * Microsystems, Inc. ("Confidential Information").  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sun.
 *
 * SUN MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SUN SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 *
 */
package com.sun.java.swing.border;

import java.awt.Graphics;
import java.awt.Insets;
import java.awt.Component;

/**
 * A composite Border class used to compose two Border objects
 * into a single border by nesting an inside Border object within 
 * the insets of an outside Border object.
 *
 * For example, this class may be used to add blank margin space
 * to a component with an existing decorative border:
 * <p>
 * <code><pre>
 *    Border border = comp.getBorder();
 *    Border margin = new EmptyBorder(10,10,10,10);
 *    comp.setBorder(new CompoundBorder(border, margin));
 * </pre></code>
 * <p>
 * Warning: serialized objects of this class will not be compatible with
 * future swing releases.  The current serialization support is appropriate
 * for short term storage or RMI between Swing1.0 applications.  It will
 * not be possible to load serialized Swing1.0 objects with future releases
 * of Swing.  The JDK1.2 release of Swing will be the compatibility
 * baseline for the serialized form of Swing objects.
 *
 * @version 1.9 02/02/98
 * @author David Kloba
 */
public class CompoundBorder extends AbstractBorder {
    protected Border outsideBorder;
    protected Border insideBorder;

    /** 
     * Creates a compound border with null outside and inside borders.
     */
    public CompoundBorder() {
	this.outsideBorder = null;
	this.insideBorder = null;
    }	    

    /** 
     * Creates a compound border with the specified outside and 
     * inside borders.  Either border may be null.
     * @param outsideBorder the outside border
     * @param insideBorder the inside border to be nested
     */
    public CompoundBorder(Border outsideBorder, Border insideBorder) {
	this.outsideBorder = outsideBorder;
	this.insideBorder = insideBorder;
    }	    

    /**
     * Returns whether or not this compound border is opaque.
     * Returns true if both the inside and outside borders are
     * non-null and opaque; returns false otherwise.
     */
    public boolean isBorderOpaque() { 
	return (outsideBorder != null && outsideBorder.isBorderOpaque()) &&
               (insideBorder != null && insideBorder.isBorderOpaque()); 
    }

    /** 
     * Paints the compound border by painting the outside border
     * with the specified position and size and then painting the
     * inside border at the specified position and size offset by
     * the insets of the outside border.
     * @param c the component for which this border is being painted
     * @param g the paint graphics
     * @param x the x position of the painted border
     * @param y the y position of the painted border
     * @param width the width of the painted border
     * @param height the height of the painted border
     */
    public void paintBorder(Component c, Graphics g, int x, int y, int width, int height) {
	Insets  nextInsets;
	int px, py, pw, ph;

	px = x;
	py = y;
	pw = width;
	ph = height;

	if(outsideBorder != null) {
	    outsideBorder.paintBorder(c, g, px, py, pw, ph);		

	    nextInsets = outsideBorder.getBorderInsets(c);
	    px += nextInsets.left;
	    py += nextInsets.top;
	    pw = pw - nextInsets.right - nextInsets.left;
	    ph = ph - nextInsets.bottom - nextInsets.top;
	}
	if(insideBorder != null) 
  	    insideBorder.paintBorder(c, g, px, py, pw, ph);		

    }
     
    /**
     * Returns the insets of the composite border by adding
     * the insets of the outside border to the insets of the
     * inside border.
     * @param c the component for which this border insets value applies
     */
    public Insets getBorderInsets(Component c)       {
	int top, left, bottom, right;
	Insets  nextInsets;

	top = left = right = bottom = 0;
	if(outsideBorder != null) {
	    nextInsets = outsideBorder.getBorderInsets(c);
	    top += nextInsets.top;
	    left += nextInsets.left;
	    right += nextInsets.right;
	    bottom += nextInsets.bottom;
	}
	if(insideBorder != null) {
	    nextInsets = insideBorder.getBorderInsets(c);
	    top += nextInsets.top;
	    left += nextInsets.left;
	    right += nextInsets.right;
	    bottom += nextInsets.bottom;
	}

	return new Insets(top, left, bottom, right);
    }

    /**
     * Returns the outside border object.
     */
    public Border getOutsideBorder() {
        return outsideBorder;
    }

    /**
     * Returns the inside border object.
     */
    public Border getInsideBorder() {
        return insideBorder;
    }
}

