/*
 * @(#)SwingGraphics.java	1.9 98/04/08
 * 
 * Copyright (c) 1997 Sun Microsystems, Inc. All Rights Reserved.
 * 
 * This software is the confidential and proprietary information of Sun
 * Microsystems, Inc. ("Confidential Information").  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sun.
 * 
 * SUN MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SUN SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 * 
 */
package com.sun.java.swing;

/**
 * A private graphics to access clip bounds without creating a new 
 * rectangle
 *
 * @version 1.9 04/08/98
 * @author Arnaud Weber
 */

import java.awt.*;
import java.awt.image.*;
import java.util.Vector;

class SwingGraphics extends Graphics {
    Graphics graphics;
    Rectangle clipRect;
    static final int poolSize = 20;

    public Graphics create() {
        return createSwingGraphics(graphics.create());
    }

    public Graphics subGraphics() {
        return graphics;
    }

    SwingGraphics(Graphics g) {
        //        System.out.println("Need new SwingGraphics");
        //        Thread.currentThread().dumpStack();
        init(g);
    }
    
    void init(Graphics g) {
        if(g == null) {
            Thread.currentThread().dumpStack();
        }
        graphics = g;
        clipRect = g.getClipBounds();
        if(clipRect == null)
            clipRect = new Rectangle(0,0,Integer.MAX_VALUE,Integer.MAX_VALUE);
    }

    public static SwingGraphics createSwingGraphics(Graphics g) {
        if(g == null)
            return null;
        if(g instanceof SwingGraphics)
            return (SwingGraphics)g;
        else {
            SwingGraphics sg = getRecycledSwingGraphics();
            if(sg != null) 
                sg.init(g);
            else
                sg = new SwingGraphics(g);
            return sg;
        }
    }

    public Graphics create(int x,int y,int w,int h) {
        return createSwingGraphics(createGraphics(graphics, x, y, w, h));
    }

    // This variable does not have to be stored in the AppContext.
    private static int needsSetClip = -1;  // tri-state: not initialized.

    /**
     * Create a Graphics from another Graphics object, and set its clip
     * to be the intersection of the first Graphics object's clip rect.
     * Graphics.create() normally does this, but Microsoft's SDK for Java
     * 2.0 doesn't set the clip of the returned object.  Since this method
     * is supposed to emulate what Graphics.create() does, all potential
     * bugs should be first checked with that method before changing the
     * behavior here.
     */
    static Graphics createGraphics(Graphics g, int x, int y, 
                                   int width, int height) {
        Graphics cg = g.create(x, y, width, height);

        if (needsSetClip != 0) {
            Rectangle r = g.getClipBounds();
            int X = (r.x > x) ? (r.x - x) : 0;
            int Y = (r.y > y) ? (r.y - y) : 0;
            int W = Math.min(x + width, r.x + r.width) - Math.max(x, r.x);
            int H = Math.min(y + height, r.y + r.height) - Math.max(y, r.y);

            // Initialize needsSetClip first time.
            if (needsSetClip == -1 && (x > 0 || y > 0)) {
                Rectangle r2 = cg.getClipBounds();
                if (r2.x != X || r2.y != Y || 
                    r2.width != W || r2.height != H) {
                    needsSetClip = 1;
                } else {
                    needsSetClip = 0;
                }
            }

            cg.setClip(X, Y, W, H);
        }

        return cg;
    }

    public void translate(int x,int y) {
        graphics.translate(x,y);
    }

    public Color getColor() {
        return graphics.getColor();
    }

    public void setColor(Color c) {
        graphics.setColor(c);
    }

    public void setPaintMode() {
        graphics.setPaintMode();
    }

    public void setXORMode(Color c1) {
        graphics.setXORMode(c1);
    }

    public Font getFont() {
        return graphics.getFont();
    }

    public void setFont(Font font) {
        graphics.setFont(font);
    }
    
    public FontMetrics getFontMetrics(Font f) {
        return graphics.getFontMetrics(f);
    }

    public Rectangle getClipBounds() {
        return graphics.getClipBounds();
    }

    public boolean isClipIntersecting(Rectangle r) {
        if (clipRect.x >= r.x + r.width || clipRect.x + clipRect.width <= r.x ||
            clipRect.y >= r.y + r.height || clipRect.y + clipRect.height <= r.y) {
            return false;
        }
        return !(clipRect.width == 0 || clipRect.height == 0 || r.width == 0 ||
                 r.height == 0);
    }

    public int getClipX() {
        return clipRect.x;
    }

    public int getClipY() {
        return clipRect.y;
    }

    public int getClipWidth() {
        return clipRect.width;
    }

    public int getClipHeight() {
        return clipRect.height;
    }

    public void clipRect(int x, int y, int width, int height) {
        graphics.clipRect(x,y,width,height);
        _changeClip(x,y,width,height,false);
    }

    public void setClip(int x, int y, int width, int height) {
        graphics.setClip(x,y,width,height);
	_changeClip(x, y, width, height, true);
    }

    public Shape getClip() {
        return graphics.getClip();
    }
    
    public void setClip(Shape clip) {
        graphics.setClip(clip);
        if(clip instanceof Rectangle) {
            Rectangle r = (Rectangle) clip;
            _changeClip(r.x,r.y,r.width,r.height,true);
        }
    }

    public void copyArea(int x, int y, int width, int height,
                         int dx, int dy) {
        graphics.copyArea(x,y,width,height,dx,dy);
    }

    public void drawLine(int x1, int y1, int x2, int y2) {
        graphics.drawLine(x1,y1,x2,y2);
    }

    public void fillRect(int x, int y, int width, int height) {
        graphics.fillRect(x,y,width,height);
    }
    
    public void clearRect(int x, int y, int width, int height) {
        graphics.clearRect(x,y,width,height);
    }

    public void drawRoundRect(int x, int y, int width, int height,
                              int arcWidth, int arcHeight) {
        graphics.drawRoundRect(x,y,width,height,arcWidth,arcHeight);
    }

    public void fillRoundRect(int x, int y, int width, int height,
                              int arcWidth, int arcHeight) {
        graphics.fillRoundRect(x,y,width,height,arcWidth,arcHeight);
    }
    
    public void drawOval(int x, int y, int width, int height) {
        graphics.drawOval(x,y,width,height);
    }

    public void fillOval(int x, int y, int width, int height) {
        graphics.fillOval(x,y,width,height);
    }

    public void drawArc(int x, int y, int width, int height,
                        int startAngle, int arcAngle) {
        graphics.drawArc(x,y,width,height,startAngle,arcAngle);
    }

    public void fillArc(int x, int y, int width, int height,
                        int startAngle, int arcAngle) {
        graphics.fillArc(x,y,width,height,startAngle,arcAngle);
    }

    public void drawPolyline(int xPoints[], int yPoints[],
                             int nPoints) {
        graphics.drawPolyline(xPoints,yPoints,nPoints);
    }

    public void drawPolygon(int xPoints[], int yPoints[],
                            int nPoints) {
        graphics.drawPolygon(xPoints,yPoints,nPoints);
    }

    public void fillPolygon(int xPoints[], int yPoints[],
                            int nPoints) {
        graphics.fillPolygon(xPoints,yPoints,nPoints);
    }

    public void drawString(String str, int x, int y) {
        graphics.drawString(str,x,y);
    }

    public boolean drawImage(Image img, int x, int y, 
                             ImageObserver observer) {
        return graphics.drawImage(img,x,y,observer);
    }

    public boolean drawImage(Image img, int x, int y,
                             int width, int height, 
                             ImageObserver observer) {
        return graphics.drawImage(img,x,y,width,height,observer);
    }

    public boolean drawImage(Image img, int x, int y, 
                             Color bgcolor,
                             ImageObserver observer) {
        return graphics.drawImage(img,x,y,bgcolor,observer);
    }

    public boolean drawImage(Image img, int x, int y,
                             int width, int height, 
                             Color bgcolor,
                             ImageObserver observer) {
        return graphics.drawImage(img,x,y,width,height,bgcolor,observer);
    }

    public boolean drawImage(Image img,
                             int dx1, int dy1, int dx2, int dy2,
                             int sx1, int sy1, int sx2, int sy2,
                             ImageObserver observer) {
        return graphics.drawImage(img,dx1,dy1,dx2,dy2,sx1,sy1,sx2,sy2,observer);
    }

    public boolean drawImage(Image img,
                             int dx1, int dy1, int dx2, int dy2,
                             int sx1, int sy1, int sx2, int sy2,
                             Color bgcolor,
                             ImageObserver observer) {
        return graphics.drawImage(img,dx1,dy1,dx2,dy2,sx1,sy1,sx2,sy2,bgcolor,observer);
    }
    
    public void dispose() {
        if(graphics != null) {
            graphics.dispose();
            graphics = null;
        }
        recycle();
    }

    public void recycle() {
        graphics = null;
        SwingGraphics.recycleSwingGraphics(this);
    }

    private void _changeClip(int x,int y,int w,int h,boolean set) {
        if(set) {
            clipRect.x = x;
            clipRect.y = y;
            clipRect.width = w;
            clipRect.height = h;
        } else 
            SwingUtilities.computeIntersection(x,y,w,h,clipRect);
    }

    private static final Object graphicsPoolKey = SwingGraphics.class;

    private static synchronized void recycleSwingGraphics(SwingGraphics g) {
        Vector pool = (Vector)SwingUtilities.appContextGet(graphicsPoolKey);
        if (pool == null) {
            pool = new Vector(poolSize);
            SwingUtilities.appContextPut(graphicsPoolKey, pool);
        }
        if(pool.size() < poolSize) {
            /**       int index = pool.indexOf(g);
            if(index != -1) {
                System.out.println("Tried to recycle the same graphics twice!");
                Thread.currentThread().dumpStack();
                while(true);
            }**/
            pool.addElement(g);
        }
        //  System.out.println("Pool size is now " + pool.size());
    }
    
    private static synchronized SwingGraphics getRecycledSwingGraphics() {
        int size;
        SwingGraphics r = null;
        Vector pool = (Vector)SwingUtilities.appContextGet(graphicsPoolKey);
        if(pool == null)
            return null;
        else if((size = pool.size()) > 0) {
            r = (SwingGraphics) pool.elementAt(size - 1);
            pool.removeElementAt(size - 1);
        }
        return r;
    }
}








