/*
 * @(#)ProgressMonitorInputStream.java	1.8 98/01/30
 * 
 * Copyright (c) 1997 Sun Microsystems, Inc. All Rights Reserved.
 * 
 * This software is the confidential and proprietary information of Sun
 * Microsystems, Inc. ("Confidential Information").  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sun.
 * 
 * SUN MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SUN SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 * 
 */
 
package com.sun.java.swing;

import java.io.*;
import java.awt.Component;

/** Monitors the progress of reading from some InputStream.  Normally invoked
 * in roughly this form:
 * <pre>
 * InputStream in = new BufferedInputStream(
 * &nbsp;          new ProgressMonitorInputStream(
 * &nbsp;                   parentComponent,
 * &nbsp;                  "Reading "+fileName,
 * &nbsp;                   new FileInputStream(fileName)));
 * </pre><p>
 * This creates a progress monitor to monitor the progress of reading
 * the input stream.  If it's taking a while, a ProgressDialog will
 * be popped up to inform the user.  If the user hits the Cancel button
 * an InterruptedIOException will be thrown on the next read.
 * All the right cleanup is done when the stream is closed.
 * @see ProgressMonitor
 * @see JOptionPane
 * @author James Gosling
 */
public class ProgressMonitorInputStream extends FilterInputStream {
    private int nread = 0;
    private int size = 0;
    private ProgressMonitor monitor;
    /** @param message Descriptive text to be placed in the dialog box
     *        if one is popped up.
     * @param parentComponent The component triggering the operation
     *		being monitored.
     * @param in The input stream to be monitored.
     */
    public ProgressMonitorInputStream(Component parentComponent, Object message, InputStream in) {
        super(in);
        try { size = in.available(); }
        catch(IOException ioe) { size = 0; }
        monitor = new ProgressMonitor(parentComponent,message,null,0,size);
    }
    /* Get the ProgressMonitor object being used by this stream.  Normally
     * this isn't needed unless you want to do something like change the
     * descriptive text partway through reading the file.
     */
    public ProgressMonitor getProgressMonitor() { return monitor; }
    public int read() throws IOException {
        int c = in.read();
        if (c>=0) monitor.setProgress(nread++);
        if (monitor.isCanceled()) {
            InterruptedIOException x = new InterruptedIOException("progress");
            x.bytesTransferred = nread;
            throw x;
        }
        return c;
    }
    public int read(byte b[]) throws IOException {
        int nr = in.read(b);
        if (nr>0) monitor.setProgress(nread+=nr);
        if (monitor.isCanceled()) {
            InterruptedIOException x = new InterruptedIOException("progress");
            x.bytesTransferred = nread;
            throw x;
        }
        return nr;
    }
    public int read(byte b[],
                  int off,
                  int len) throws IOException {
        int nr = in.read(b, off, len);
        if (nr>0) monitor.setProgress(nread+=nr);
        if (monitor.isCanceled()) {
            InterruptedIOException x = new InterruptedIOException("progress");
            x.bytesTransferred = nread;
            throw x;
        }
        return nr;
    }
    public long skip(long n) throws IOException {
        long nr = in.skip(n);
        if (nr>0) monitor.setProgress(nread+=nr);
        return nr;
    }
    public void close() throws IOException {
        in.close();
        monitor.close();
    }
    public synchronized void reset() throws IOException {
        in.reset();
        nread = size-in.available();
        monitor.setProgress(nread);
    }
}
