/*
 * @(#)JToolBar.java	1.41 98/02/16
 * 
 * Copyright (c) 1997 Sun Microsystems, Inc. All Rights Reserved.
 * 
 * This software is the confidential and proprietary information of Sun
 * Microsystems, Inc. ("Confidential Information").  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sun.
 * 
 * SUN MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SUN SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 * 
 */

package com.sun.java.swing;

import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Insets;
import java.awt.event.*;
import java.beans.*;
import java.io.Serializable;

import com.sun.java.swing.border.Border;
import com.sun.java.swing.plaf.*;
import com.sun.java.accessibility.*;

/**
 * JToolBar provides a component which is useful for displaying commonly
 * used Actions or controls.  It can be dragged out into a seperate window
 * by the user (unless the floatable property is set to false).  In order
 * for drag-out to work correctly, it is recommended that you add JToolBar
 * instances to one of the four 'sides' of a container whose layout manager 
 * is a BorderLayout, and do not add children to any of the other four 'sides'.
 * <p>
 * For the keyboard keys used by this component in the standard Look and
 * Feel (L&F) renditions, see the
 * <a href="doc-files/Key-Index.html#JToolBar">JToolBar</a> key assignments.
 * <p>
 * Warning: serialized objects of this class will not be compatible with
 * future swing releases.  The current serialization support is appropriate
 * for short term storage or RMI between Swing1.0 applications.  It will
 * not be possible to load serialized Swing1.0 objects with future releases
 * of Swing.  The JDK1.2 release of Swing will be the compatibility
 * baseline for the serialized form of Swing objects.
 *
 * @version 1.41 02/16/98
 * @author Georges Saab
 * @see Action
 */
public class JToolBar extends JComponent implements Accessible {
    private    boolean   paintBorder              = true;  
    private    Insets    margin                   = null;
    private    boolean   floatable                = true;


    /**
     * Create a new toolbar.
     */
    public JToolBar() {
        // Create the Popup
        this.setLayout(new BoxLayout(this, BoxLayout.X_AXIS));
        updateUI();
    }

    /**
     * Returns the toolbar's current UI.
     * @see setUI
     */
    public ToolBarUI getUI() {
        return (ToolBarUI)ui;
    }
    
    /**
     * Sets the L&F object that renders this component.
     *
     * @param ui  the ToolBarUI L&F object
     * @see UIDefaults#getUI
     * @beaninfo
     * description: The menu item's UI delegate
     *       bound: true
     *      expert: true
     *      hidden: true
     */
    public void setUI(ToolBarUI ui) {
        super.setUI(ui);
    }
    
    /**
     * Notification from the UIFactory that the L&F has changed. 
     * Called to replace the UI with the latest version from the 
     * UIFactory.
     *
     * @see JComponent#updateUI
     */
    public void updateUI() {
        setUI((ToolBarUI)UIManager.getUI(this));
        invalidate();
    }



    /**
     * Returns the name of the L&F class that renders this component.
     *
     * @return "ToolBarUI"
     * @see JComponent#getUIClassID
     * @see UIDefaults#getUI
     */
    public String getUIClassID() {
        return "ToolBarUI";
    }


    /*
     * Returns the index of the specified component.
     * (Note: Separators occupy index positions.)
     *
     * @param c  the Component to find
     * @return an int indicating the component's position, where 0=first
     */
    public int getComponentIndex(Component c) {
        int ncomponents = this.getComponentCount();
        Component[] component = this.getComponents();
        for (int i = 0 ; i < ncomponents ; i++) {
            Component comp = component[i];
            if (comp == c) 
                return i;
        }
        return -1;
    }

    /*
     * Returns the component at the specified index.
     *
     * @param i  the component's position, where 0=first
     * @return   the Component at that position, or null for an
     *           invalid index or if there is a separator at that
     *           position
     */
    public Component getComponentAtIndex(int i) {
        int ncomponents = this.getComponentCount();
        if (i <= ncomponents) {
            Component[] component = this.getComponents();
            return component[i];
        }
        return null;
    }

     /**
      * Sets the margin between the toolbar's border and
      * its buttons. Setting to null causes the toolbar to
      * use the default margins. The toolbar's default Border
      * object uses this value to create the proper margin.
      * However, if a non-default border is set on the toolbar, 
      * it is that Border object's responsibility to create the
      * appropriate margin space (otherwise this property will 
      * effectively be ignored).
      *
      * @param m an Insets object that defines the space between the 
      *          border and the buttons
      * @see Insets
      * @beaninfo
      * description: The margin between the toolbar's border and contents
      *       bound: true
      *      expert: true
      */
     public void setMargin(Insets m) {
         Insets old = margin;
         margin = m;
         firePropertyChange("margin", old, m);
         invalidate();
     }
 
     /**
      * Returns the margin between the toolbar's border and
      * its buttons.
      *
      * @return an Insets object containing the margin values
      * @see Insets
      */
     public Insets getMargin() {
         if(margin == null) {
             return new Insets(0,0,0,0);
         } else {
             return margin;
         }
     }

     /**
      * Checks whether the border should be painted.
      *
      * @return true if the border should be painted, else false
      * @see setBorderPainted
      */
     public boolean isBorderPainted() {
         return paintBorder;
     }
     
 
     /**
      * Sets whether the border should be painted.
      *
      * @param b if true, the border is painted.
      * @see isBorderPainted
      * @beaninfo
      * description: Does the toolbar paint its borders?
      *      expert: true
      */
     public void setBorderPainted(boolean b) {
         paintBorder = b;
         invalidate();
     }
 
     /**
      * Paint the toolbar's border if BorderPainted property is true.
      * 
      * @param g  the Graphics context in which the painting is done
      * @see JComponent#paint
      * @see JComponent#setBorder
      */
     protected void paintBorder(Graphics g) {    
         if (isBorderPainted()) {
             super.paintBorder(g);
         }
     }

    /** 
     * Return true if the Toolbar can be dragged out by the user.
     *
     * @return true if the Toolbar can be dragged out by the user
     */
    public boolean isFloatable() {
        return floatable;
    }

     /**
      * Sets whether the toolbar can be made to float
      *
      * @param b if true, the toolbar can be dragged out
      * @see isFloatable
      * @beaninfo
      * description: Can the toolbar be made to float by the user?
      *       bound: true
      *   preferred: true
      */
    public void setFloatable(boolean b) {
        if (floatable != b) {
            this.floatable = b;
            firePropertyChange("floatable", !floatable, floatable);         
        }
    }

    /**
     * Appends a toolbar-specific separator to the end of the toolbar
     * (not a JSeparator).
     */
    public void addSeparator() {
        JToolBar.Separator s = new JToolBar.Separator();
        add(s);
    }

    /**
     * Add a new JButton which dispatches the action.
     *
     * @param a the Action object to add as a new menu item
     */
    public JButton add(Action a) {
        JButton b = new JButton((String)a.getValue(Action.NAME),
                                (Icon)a.getValue(Action.SMALL_ICON));
        b.setHorizontalTextPosition(JButton.CENTER);
        b.setVerticalTextPosition(JButton.BOTTOM);
        b.setEnabled(a.isEnabled());
        b.addActionListener(a);
        add(b);
        PropertyChangeListener actionPropertyChangeListener = 
            createActionChangeListener(b);
        a.addPropertyChangeListener(actionPropertyChangeListener);
        return b;
    }

    protected PropertyChangeListener createActionChangeListener(JButton b) {
        return new ActionChangedListener(b);
    }

    private class ActionChangedListener implements PropertyChangeListener {
        JButton button;
        
        ActionChangedListener(JButton b) {
            super();
            this.button = b;
        }
        public void propertyChange(PropertyChangeEvent e) {
            String propertyName = e.getPropertyName();
            if (e.getPropertyName().equals(Action.NAME)) {
                String text = (String) e.getNewValue();
                button.setText(text);
                button.repaint();
            } else if (propertyName.equals("enabled")) {
                Boolean enabledState = (Boolean) e.getNewValue();
                button.setEnabled(enabledState.booleanValue());
                button.repaint();
            } else if (e.getPropertyName().equals(Action.SMALL_ICON)) {
                Icon icon = (Icon) e.getNewValue();
                button.setIcon(icon);
                button.invalidate();
                button.repaint();
            } 
        }
    }

    /**
     * A toolbar-specific separator. An object with dimension but
     * no contents used to divide buttons on a toolbar into groups.
     * <p>
     * Warning: serialized objects of this class will not be compatible with
     * future swing releases.  The current serialization support is appropriate
     * for short term storage or RMI between Swing1.0 applications.  It will
     * not be possible to load serialized Swing1.0 objects with future releases
     * of Swing.  The JDK1.2 release of Swing will be the compatibility
     * baseline for the serialized form of Swing objects.
     */
    public class Separator extends Component {
        /** Create the separator */
        public Separator() {
        }

        /** 
         * Return the minimum size for the separator
         *
         * @return the Dimension object containing the separator's
         *         minimum size
         */
        public Dimension getMinimumSize() {
            return new Dimension(10,5);
        }
        /** 
         * Return the maximum size for the separator
         *
         * @return the Dimension object containing the separator's
         *         maximum size
         */
        public Dimension getMaximumSize() {
            return new Dimension(10,5);
        }
        /** 
         * Return the preferred size for the separator
         *
         * @return the Dimension object containing the separator's
         *         preferred size
         */
        public Dimension getPreferredSize() {
            return new Dimension(10,5);
        }
    }

/////////////////
// Accessibility support
////////////////

    /**
     * Get the AccessibleContext associated with this JComponent
     *
     * @return the AccessibleContext of this JComponent
     */
    public AccessibleContext getAccessibleContext() {
        if (accessibleContext == null) {
            accessibleContext = new AccessibleJToolBar();
        }
        return accessibleContext;
    }

    /**
     * The class used to obtain the accessible role for this object.
     * <p>
     * Warning: serialized objects of this class will not be compatible with
     * future swing releases.  The current serialization support is appropriate
     * for short term storage or RMI between Swing1.0 applications.  It will
     * not be possible to load serialized Swing1.0 objects with future releases
     * of Swing.  The JDK1.2 release of Swing will be the compatibility
     * baseline for the serialized form of Swing objects.
     */
    protected class AccessibleJToolBar extends AccessibleJComponent {

        /**
         * Get the state of this object.
         *
         * @return an instance of AccessibleStateSet containing the current 
         * state set of the object
         * @see AccessibleState
         */
        public AccessibleStateSet getAccessibleStateSet() {
            AccessibleStateSet states = SwingUtilities.getAccessibleStateSet(JToolBar.this);
            // FIXME:  [[[WDW - need to add orientation from BoxLayout]]]
            // FIXME:  [[[WDW - need to do SELECTABLE if SelectionModel is added]]]
            return states;
        }
    
        /**
         * Get the role of this object.
         *
         * @return an instance of AccessibleRole describing the role of the object
         */
        public AccessibleRole getAccessibleRole() {
            return AccessibleRole.TOOL_BAR;
        }
    } // inner class AccessibleJToolBar
}
