/*
 * @(#)JProgressBar.java	1.49 98/04/07
 * 
 * Copyright (c) 1997 Sun Microsystems, Inc. All Rights Reserved.
 * 
 * This software is the confidential and proprietary information of Sun
 * Microsystems, Inc. ("Confidential Information").  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sun.
 * 
 * SUN MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SUN SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 * 
 */

package com.sun.java.swing;

import java.awt.Color;
import java.awt.Graphics;
import com.sun.java.swing.event.*;
import com.sun.java.accessibility.*;
import java.io.Serializable;
import com.sun.java.swing.plaf.ProgressBarUI;


/**
 * A component that displays an integer value graphically within a bounded 
 * interval. A progress bar typically communicates the progress of an 
 * event by displaying its percentage of completion. The orientation of 
 * the progress bar depends on its size. If its height is greater than
 * its width, the progress bar is vertical.
 * <p>
 * Warning: serialized objects of this class will not be compatible with
 * future swing releases.  The current serialization support is appropriate
 * for short term storage or RMI between Swing1.0 applications.  It will
 * not be possible to load serialized Swing1.0 objects with future releases
 * of Swing.  The JDK1.2 release of Swing will be the compatibility
 * baseline for the serialized form of Swing objects.
 *
 * @beaninfo
 *      attribute: isContainer false
 *    description:  A component that displays an integer value graphically.
 *
 * @version 1.49 04/07/98
 * @author Rob Davis
 */
public class JProgressBar extends JComponent implements SwingConstants, Accessible
{
    protected int orientation;
    protected boolean paintBorder;
    protected BoundedRangeModel barModel;


    /**
     * Only one ChangeEvent is needed per instance since the
     * event's only interesting property is the immutable source, which
     * is the progress bar.
     */
    protected transient ChangeEvent changeEvent = null;
    protected ChangeListener changeListener = null;

    /**
     * Creates a horizontal progress bar with a border.
     */
    public JProgressBar()
    {
        setModel(new DefaultBoundedRangeModel());
        updateUI();

        orientation = HORIZONTAL;  // documented with set/getOrientation()
        paintBorder = true;        // documented with is/setBorderPainted()
    }

    /**
     * Overridden to call paint without filling the background.
     */
    public void update(Graphics g) {
        this.paint(g);
    }

    /**
     * Returns <code>JProgressBar.VERTICAL</code> or 
     * <code>JProgressBar.HORIZONTAL</code>, depending on the orientation
     * of the progress bar. The default orientation is 
     * <code>HORIZONTAL</code>.
     *
     * @return HORIZONTAL or VERTICAL
     */
    public int getOrientation() {
        return orientation;
    }

    /**
     * Sets the progress bar's orientation to <I>newOrientation</I>, which
     * must be <code>JProgressBar.VERTICAL</code> or 
     * <code>JProgressBar.HORIZONTAL</code>. The default orientation 
     * is <code>HORIZONTAL</code>.
     *
     * @param  newOrientation  HORIZONTAL or VERTICAL
     * @exception      IllegalArgumentException    if <I>newOrientation</I>
     *                                              is an illegal value
     * @beaninfo
     *    preferred: true
     *  description: The progress bar's orientation.
     */
    public void setOrientation(int newOrientation) {
        if (orientation != newOrientation) {
            switch (newOrientation) {
            case VERTICAL:
            case HORIZONTAL:
                int oldorientation = orientation;
                orientation = newOrientation;
                if (accessibleContext != null) {
                    accessibleContext.firePropertyChange(
			    AccessibleContext.ACCESSIBLE_STATE_PROPERTY,
                            ((oldorientation == VERTICAL) 
			     ? AccessibleState.VERTICAL 
			     : AccessibleState.HORIZONTAL),
                            ((orientation == VERTICAL) 
			     ? AccessibleState.VERTICAL 
			     : AccessibleState.HORIZONTAL));
	        }
                break;
            default:
                throw new IllegalArgumentException(newOrientation +
                                             " is not a legal orientation");
            }
            repaint();
        }
    }

    /**
     * Returns true if the progress bar has a border or false if it does not.
     *
     * @return whether the progress bar has a border
     * @see    setBorderPainted
     */
    public boolean isBorderPainted() {
        return paintBorder;
    }

    /**
     * Sets whether the progress bar should have a border.
     *
     * @param   b       true if the progress bar should have a border
     * @see     isBorderPainted
     * @beaninfo
     *  description: Whether the progress bar should have a border.
     */
    public void setBorderPainted(boolean b) {
        paintBorder = b;
        repaint();
    }

    /**
     * Paint the progress bar's border if BorderPainted property is true.
     * 
     * @param g  the Graphics context within which to paint the border
     * @see #paint
     * @see #setBorder
     */
    protected void paintBorder(Graphics g) {    
        if (isBorderPainted()) {
            super.paintBorder(g);
        }
    }


    /**
     * Returns the L&F object that renders this component.
     *
     * @return the ProgressBarUI object that renders this component
     */
    public ProgressBarUI getUI() {
        return (ProgressBarUI)ui;
    }

    /**
     * Sets the L&F object that renders this component.
     *
     * @param ui  the ProgressBarUI L&F object
     * @see UIDefaults#getUI
     * @beaninfo
     *       expert: true
     *  description: The ProgressBarUI implementation that defines the combo box look and feel.
     */
    public void setUI(ProgressBarUI ui) {
        if (this.ui != ui) {
            super.setUI(ui);
            repaint();
        }
    }


    /**
     * Notification from the UIFactory that the L&F has changed. 
     * Called to replace the UI with the latest version from the 
     * UIFactory.
     *
     * @see JComponent#updateUI
     */
    public void updateUI() {
        setUI((ProgressBarUI)UIManager.getUI(this));
    }


    /**
     * Returns the name of the L&F class that renders this component.
     *
     * @return "ProgressBarUI"
     * @see JComponent#getUIClassID
     * @see UIDefaults#getUI
     * @beaninfo
     *        expert: true
     *   description: A string that specifies the name of the L&F class.
     */
    public String getUIClassID() {
        return "ProgressBarUI";
    }


    /* We pass each Change event to the listeners with the
     * the progress bar as the event source.
     * <p>
     * Warning: serialized objects of this class will not be compatible with
     * future swing releases.  The current serialization support is appropriate
     * for short term storage or RMI between Swing1.0 applications.  It will
     * not be possible to load serialized Swing1.0 objects with future releases
     * of Swing.  The JDK1.2 release of Swing will be the compatibility
     * baseline for the serialized form of Swing objects.
     */
    protected class ModelListener implements ChangeListener, Serializable {
        public void stateChanged(ChangeEvent e) {
            fireStateChanged();
        }
    }

    /* Subclasses that want to handle ChangeEvents differently
     * can override this to return a subclass of ModelListener or
     * another ChangeListener implementation.
     */
    protected ChangeListener createChangeListener() {
        return new ModelListener();
    }

    /**
     * Adds a ChangeListener to the button.
     *
     * @param l the ChangeListener to add
     */
    public void addChangeListener(ChangeListener l) {
        listenerList.add(ChangeListener.class, l);
    }
    
    /**
     * Removes a ChangeListener from the button.
     *
     * @param l the ChangeListener to remove
     */
    public void removeChangeListener(ChangeListener l) {
        listenerList.remove(ChangeListener.class, l);
    }
        
    /**
     * Notify all listeners that have registered interest for
     * notification on this event type.  The event instance 
     * is lazily created using the parameters passed into 
     * the fire method.
     * @see EventListenerList
     */
    protected void fireStateChanged() {
        // Guaranteed to return a non-null array
        Object[] listeners = listenerList.getListenerList();
        // Process the listeners last to first, notifying
        // those that are interested in this event
        for (int i = listeners.length-2; i>=0; i-=2) {
            if (listeners[i]==ChangeListener.class) {
                // Lazily create the event:
                if (changeEvent == null)
                    changeEvent = new ChangeEvent(this);
                ((ChangeListener)listeners[i+1]).stateChanged(changeEvent);
            }          
        }
    } 
      
    /**
     * Returns the data model used by the JProgressBar.
     *
     * @return the BoundedRangeModel currently in use
     * @see    BoundedRangeModel
     */
    public BoundedRangeModel getModel() {
        return barModel;
    }

    /**
     * Sets the data model used by the JProgressBar.
     *
     * @param  newModel the BoundedRangeModel to use
     * @see    BoundedRangeModel
     * @beaninfo
     *    expert: true
     * description: The data model used by the JProgressBar.
     */
    public void setModel(BoundedRangeModel newModel) {
        // PENDING(???) setting the same model to multiple bars is broken; listeners
        BoundedRangeModel oldModel = getModel();

        if (newModel != oldModel) {
            if (oldModel != null) {
                oldModel.removeChangeListener(changeListener);
                changeListener = null;
            }

            barModel = newModel;

            if (newModel != null) {
                changeListener = createChangeListener();
                newModel.addChangeListener(changeListener);
            }

	    if (accessibleContext != null) {
                accessibleContext.firePropertyChange(
		        AccessibleContext.ACCESSIBLE_VALUE_PROPERTY,
                        (oldModel== null 
			 ? null : new Integer(oldModel.getValue())),
                        (newModel== null 
			 ? null : new Integer(newModel.getValue())));
	    }

            barModel.setExtent(0);
            repaint();
        }
    }


    /* All of the model methods are implemented by delegation. */

    /**
     * Returns the model's current value. The value is always between the 
     * model's minimum and maximum values, inclusive.
     *
     * @return  the value
     * @see     #setValue
     * @see     BoundedRangeModel
     */
    public int getValue() { return getModel().getValue(); }

    /**
     * Returns the model's minimum value.
     *
     * @return  an int -- the model's minimum
     * @see     #setMinimum
     * @see     BoundedRangeModel
     */
    public int getMinimum() { return getModel().getMinimum(); }

    /**
     * Returns the model's maximum value.
     *
     * @return  an int -- the model's maximum
     * @see     #setMaximum
     * @see     BoundedRangeModel
     */
    public int getMaximum() { return getModel().getMaximum(); }

    /**
     * Sets the model's current value to <I>x</I>. If <I>x</I> is less than the
     * minimum or greater than the maximum, this method throws an
     * <code>IllegalArgumentException</code> and the value is not changed.
     * <p>
     * Notifies any listeners if the data changes.
     *
     * @param   x       the new value
     * @see     #getValue
     * @see     BoundedRangeModel
     * @beaninfo
     *    preferred: true
     *  description: The model's current value.
     */
    public void setValue(int n) { 
        BoundedRangeModel brm = getModel();
	int oldValue = brm.getValue();

	brm.setValue(n);
 
	if (accessibleContext != null) {
            accessibleContext.firePropertyChange(
		    AccessibleContext.ACCESSIBLE_VALUE_PROPERTY,
                    new Integer(oldValue),
                    new Integer(brm.getValue()));
	}
    }

    /**
     * Sets the model's minimum to <I>x</I>. If the maximum value or
     * current value is outside of the new minimum, the maximum or 
     * current value is adjusted accordingly.
     * <p>
     * Notifies any listeners if the data changes.
     *
     * @param  x       the new minimum
     * @see    #getMinimum
     * @see    #addChangeListener
     * @see    BoundedRangeModel
     * @beaninfo
     *  preferred: true
     * description: The model's minimum value.
     */
    public void setMinimum(int n) { getModel().setMinimum(n); }

    /**
     * Sets the model's maximum to <I>x</I>. If the minimum value or
     * current value is outside of the new maximum, the minimum or 
     * current value is adjusted accordingly.
     * <p>
     * Notifies any listeners if the data changes.
     *
     * @param  x       the new maximum
     * @see    #getMaximum
     * @see    #addChangeListener
     * @see    BoundedRangeModel
     * @beaninfo
     *    preferred: true
     *  description: The model's maximum value.
     */
    public void setMaximum(int n) { getModel().setMaximum(n); }

/////////////////
// Accessibility support
////////////////

    /**
     * Get the AccessibleContext associated with this JComponent
     *
     * @return the AccessibleContext of this JComponent
     * @beaninfo
     *       expert: true
     *  description: The AccessibleContext associated with this ProgressBar.
     */
    public AccessibleContext getAccessibleContext() {
        if (accessibleContext == null) {
            accessibleContext = new AccessibleJProgressBar();
        }
        return accessibleContext;
    }

    /**
     * The class used to obtain the accessible role for this object.
     * <p>
     * Warning: serialized objects of this class will not be compatible with
     * future swing releases.  The current serialization support is appropriate
     * for short term storage or RMI between Swing1.0 applications.  It will
     * not be possible to load serialized Swing1.0 objects with future releases
     * of Swing.  The JDK1.2 release of Swing will be the compatibility
     * baseline for the serialized form of Swing objects.
     */
    protected class AccessibleJProgressBar extends AccessibleJComponent
        implements AccessibleValue {

        /**
         * Get the state set of this object.
         *
         * @return an instance of AccessibleState containing the current state 
         * of the object
         * @see AccessibleState
         */
        public AccessibleStateSet getAccessibleStateSet() {
            AccessibleStateSet states = super.getAccessibleStateSet();
            if (getModel().getValueIsAdjusting()) {
                states.add(AccessibleState.BUSY);
            }
            if (getOrientation() == VERTICAL) {
                states.add(AccessibleState.VERTICAL);
            } else {
                states.add(AccessibleState.HORIZONTAL);
            }
            return states;
        }

        /**
         * Get the role of this object.
         *
         * @return an instance of AccessibleRole describing the role of the 
         * object
         */
        public AccessibleRole getAccessibleRole() {
            return AccessibleRole.PROGRESS_BAR;
        }

        /**
         * Get the AccessibleValue associated with this object if one
         * exists.  Otherwise return null.
         */
        public AccessibleValue getAccessibleValue() {
            return this;
        }

        /**
         * Get the accessible value of this object.
         *
         * @return The current value of this object.
         */
        public Number getCurrentAccessibleValue() {
            return new Integer(getValue());
        }

        /**
         * Set the value of this object as a Number.
         *
         * @return True if the value was set.
         */
        public boolean setCurrentAccessibleValue(Number n) {
            if (n instanceof Integer) {
                setValue(n.intValue());
                return true;
            } else {
                return false;
            }
        }

        /**
         * Get the minimum accessible value of this object.
         *
         * @return The minimum value of this object.
         */
        public Number getMinimumAccessibleValue() {
            return new Integer(getMinimum());
        }

        /**
         * Get the maximum accessible value of this object.
         *
         * @return The maximum value of this object.
         */
        public Number getMaximumAccessibleValue() {
            return new Integer(getMaximum());
        }

    } // AccessibleJProgressBar
}
