/*
 * @(#)JMenuItem.java	1.53 98/02/16
 * 
 * Copyright (c) 1997 Sun Microsystems, Inc. All Rights Reserved.
 * 
 * This software is the confidential and proprietary information of Sun
 * Microsystems, Inc. ("Confidential Information").  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sun.
 * 
 * SUN MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SUN SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 * 
 */
package com.sun.java.swing;

import java.util.EventListener;
import java.awt.*;
import java.awt.event.*;
import java.awt.image.*;
import java.io.Serializable;
import com.sun.java.swing.plaf.*;
import com.sun.java.swing.event.*;
import com.sun.java.accessibility.*;

/**
 * An implementation of a MenuItem. A menu item is essentially a button
 * sitting in a list. When the user selects the "button", the action
 * associated with the menu item is performed. A JMenuItem contained
 * in a JPopupMenu performs exactly that function.
 * <p>
 * For the keyboard keys used by this component in the standard Look and
 * Feel (L&F) renditions, see the
 * <a href="doc-files/Key-Index.html#JMenuItem">JMenuItem</a> key assignments.
 * <p>
 * Warning: serialized objects of this class will not be compatible with
 * future swing releases.  The current serialization support is appropriate 
 * for short term storage or RMI between Swing1.0 applications.  It will
 * not be possible to load serialized Swing1.0 objects with future releases
 * of Swing.  The JDK1.2 release of Swing will be the compatibility
 * baseline for the serialized form of Swing objects.
 *
 * @version 1.53 02/16/98
 * @author Georges Saab
 * @author David Karlton
 * @see JPopupMenu
 * @see JMenu
 * @see JCheckBoxMenuItem
 * @see JRadioButtonMenuItem
 */
public class JMenuItem extends AbstractButton implements Accessible,MenuElement  {

    /**
     * Creates a menuItem with no set text or icon.
     */
    public JMenuItem() {
        this(null, (Icon)null);
        setRequestFocusEnabled(false);
    }

    /**
     * Creates a menuItem with an icon.
     *
     * @param icon the icon of the MenuItem.
     */
    public JMenuItem(Icon icon) {
        this(null, icon);
        setRequestFocusEnabled(false);
    }

    /**
     * Creates a menuItem with text.
     *
     * @param text the text of the MenuItem.
     */
    public JMenuItem(String text) {
        this(text, (Icon)null);
    }
    
    /**
     * Creates a menuItem with the supplied text and icon.
     *
     * @param text the text of the MenuItem.
     * @param icon the icon of the MenuItem.
     */
    public JMenuItem(String text, Icon icon) {
        setModel(new DefaultButtonModel());
        init(text, icon);
        setBorderPainted(false);
        setFocusPainted(false);
        setHorizontalTextPosition(JButton.LEFT);
        setHorizontalAlignment(JButton.LEFT);
        updateUI();
    }

    /**
     * Creates a menuItem with the specified text and
     * keyboard mnemonic.
     *
     * @param text the text of the MenuItem.
     * @param mnemonic the keyboard mnemonic for the MenuItem
     */
    public JMenuItem(String text, int mnemonic) {
        setModel(new DefaultButtonModel());
        init(text, null);
        setBorderPainted(false);
        setFocusPainted(false);
        setHorizontalTextPosition(JButton.LEFT);
        setHorizontalAlignment(JButton.LEFT);
        setMnemonic(mnemonic);
        updateUI();
    }

    /**
     * Initialize the menu item with the specified text and icon.
     *
     * @param text the text of the MenuItem.
     * @param icon the icon of the MenuItem.
     */
    protected void init(String text, Icon icon) {
        setLayout(new OverlayLayout(this));

        if(text != null) {
            setText(text);
        }
        
        if(icon != null) {
            setIcon(icon);
        }
        
        // Listen for Focus events
        addFocusListener(new MenuItemFocusListener());
    }

    private static class MenuItemFocusListener implements FocusListener,
        Serializable {
        public void focusGained(FocusEvent event) {}
        public void focusLost(FocusEvent event) {
            // When focus is lost, repaint if 
            // the focus information is painted
            JMenuItem mi = (JMenuItem)event.getSource();
            if(mi.isFocusPainted()) {
                mi.repaint();
            }
        }
    }
        
    
    /**
     * Sets the L&F object that renders this component.
     *
     * @param ui  the MenuItemUI L&F object
     * @see UIDefaults#getUI
     * @beaninfo
     * description: The menu item's UI delegate
     *       bound: true
     *      expert: true
     *      hidden: true
     */
    public void setUI(MenuItemUI ui) {
        super.setUI(ui);
    }
    
    /**
     * Notification from the UIFactory that the L&F has changed. 
     * Called to replace the UI with the latest version from the 
     * UIFactory.
     *
     * @see JComponent#updateUI
     */
    public void updateUI() {
        setUI((MenuItemUI)UIManager.getUI(this));
    }


    /**
     * Returns the name of the L&F class that renders this component.
     *
     * @return "MenuItemUI"
     * @see JComponent#getUIClassID
     * @see UIDefaults#getUI
     */
    public String getUIClassID() {
        return "MenuItemUI";
    }


    /**
     * Identifies the menu item as "armed". If the mouse button is
     * released while it is over this item, the menu's action event
     * will fire. If the mouse button is released elsewhere, the
     * event will not fire and the menu item will be disarmed.
     * 
     * @param b true to arm the menu item so it can be selected
     * @beaninfo
     *    description: Mouse release will fire an action event
     *         hidden: true
     */
    public void setArmed(boolean b) {
        ButtonModel model = (ButtonModel) getModel();

        boolean oldValue = model.isArmed();
        if ((accessibleContext != null) && (oldValue != b)) {
            if (b) {
                accessibleContext.firePropertyChange(
                        AccessibleContext.ACCESSIBLE_STATE_PROPERTY,
                        null, 
                        AccessibleState.ARMED);
            } else {
                accessibleContext.firePropertyChange(
                        AccessibleContext.ACCESSIBLE_STATE_PROPERTY,
                        AccessibleState.ARMED, 
                        null);
            }
        }
        if(model.isArmed() != b) {
            model.setArmed(b);
        }
    }

    /**
     * Returns whether the menu item is "armed".
     * 
     * @return true if the menu item is armed, and it can be selected
     * @see #setArmed
     */
    public boolean isArmed() {
        ButtonModel model = (ButtonModel) getModel();
        return model.isArmed();
    }

    /**
     * Enable or disable the menu item.
     *
     * @param b  true to enable the item
     * @beaninfo
     *    description: Does the component react to user interaction
     *          bound: true
     *      preferred: true
     */
    public void setEnabled(boolean b) {
        // Make sure we aren't armed!
        if (b == false)
            setArmed(false);
        super.setEnabled(b);
    }

    /* The keystroke which acts as the menu item's accelerator
     */
    private KeyStroke accelerator;

    /* Set the key combination which invokes the Menu Item's
     * action listeners without navigating the menu hierarchy.
     *
     * @param keyStroke the KeyStroke which will serve as an accelerator
     * @beaninfo
     *     description: The keystroke combination which will invoke the JMenuItem's
     *                  actionlisteners without navigating the menu hierarchy
     *           bound: true
     *       preferred: true
     */
    public void setAccelerator(KeyStroke keyStroke) {
        if (accelerator != null)
            unregisterKeyboardAction(accelerator);

        // PENDING(ges) change this to a (lighter) ActionListener which implements
        // Serializable
        registerKeyboardAction(new AbstractAction(){
            public void actionPerformed(ActionEvent e) {
                MenuSelectionManager.defaultManager().clearSelectedPath();
                doClick();
            }
        } , keyStroke, WHEN_IN_FOCUSED_WINDOW);
        this.accelerator = keyStroke;
    }

    /* Returns the KeyStroke which serves as an accelerator 
     * for the menu item.
     */
    public KeyStroke getAccelerator() {
        return this.accelerator;
    }

    /**
     * Process a mouse event. event is a MouseEvent with source being the receiving component.
     * componentPath is the path of the receiving MenuElement in the menu
     * hierarchy. manager is the MenuSelectionManager for the menu hierarchy.
     * This method should process the MouseEvent and change the menu selection if necessary
     * by using MenuSelectionManager's API.
     * <p>
     * Note: you do not have to forward the event to sub-components. This is done automatically
     * by the MenuSelectionManager
     */
    public void processMouseEvent(MouseEvent event,MenuElement path[],MenuSelectionManager manager) {
        ((MenuItemUI)getUI()).processMouseEvent(this,event,path,manager);
    }

    /** Implemented to be a MenuElement. This message is forwarded to the UI **/
    public void processKeyEvent(KeyEvent e,MenuElement path[],MenuSelectionManager manager) {
        ((MenuItemUI)getUI()).processKeyEvent(this,e,path,manager);
    }

    /**
     * Called by the MenuSelectionManager when the MenuElement is selected
     * or unselected.
     * 
     * @param isIncluded  true if this menu item is on the part of the menu
     *                    path that changed, false if this menu is part of the
     *                    a menu path that changed, but this particular part of
     *                    that path is still the same
     * @see MenuSelectionManager#setSelectedPath(MenuElement[])
     */
    public void menuSelectionChanged(boolean isIncluded) {
        setArmed(isIncluded);
    }

    /**
     * This method returns an array containing the sub-menu components for this menu component.
     *
     * @return an array of MenuElements
     */
    public MenuElement[] getSubElements() {
        return new MenuElement[0];
    }
    
    /**
     * This method returns the java.awt.Component used to paint this object.
     * The returned component will be used to convert events and detect if an event is inside
     * a menu component.
     *
     * @return the Component that paints this menu item
     */
    public Component getComponent() {
        return this;
    }

/////////////////
// Accessibility support
////////////////

    /**
     * Get the AccessibleContext associated with this JComponent
     *
     * @return the AccessibleContext of this JComponent
     */
    public AccessibleContext getAccessibleContext() {
        if (accessibleContext == null) {
            accessibleContext = new AccessibleJMenuItem();
        }
        return accessibleContext;
    }


    /**
     * The class used to obtain the accessible role for this object.
     * <p>
     * Warning: serialized objects of this class will not be compatible with
     * future swing releases.  The current serialization support is appropriate
     * for short term storage or RMI between Swing1.0 applications.  It will
     * not be possible to load serialized Swing1.0 objects with future releases
     * of Swing.  The JDK1.2 release of Swing will be the compatibility
     * baseline for the serialized form of Swing objects.
     */
    protected class AccessibleJMenuItem extends AccessibleAbstractButton implements ChangeListener {

        AccessibleJMenuItem() {
            super();
            JMenuItem.this.addChangeListener(this);
        }

        /**
         * Get the role of this object.
         *
         * @return an instance of AccessibleRole describing the role of the 
         * object
         */
        public AccessibleRole getAccessibleRole() {
            return AccessibleRole.MENU_ITEM;
        }

        /**
         * Supports the change listener interface and fires property change
         */
        public void stateChanged(ChangeEvent e) {
            firePropertyChange(AccessibleContext.ACCESSIBLE_VISIBLE_DATA_PROPERTY, 
                               new Boolean(false), new Boolean(true));
        }
    } // inner class AccessibleJMenuItem
}

