/*
 * @(#)DefaultButtonModel.java	1.21 98/02/02
 * 
 * Copyright (c) 1997 Sun Microsystems, Inc. All Rights Reserved.
 * 
 * This software is the confidential and proprietary information of Sun
 * Microsystems, Inc. ("Confidential Information").  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sun.
 * 
 * SUN MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SUN SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 * 
 */
package com.sun.java.swing;

import java.awt.*;
import java.awt.event.*;
import java.awt.image.*;
import java.io.Serializable;
import java.util.EventListener;
import com.sun.java.swing.event.*;

/**
 * The default implementation of a Button component's
 * data model.
 * <p>
 * Warning: serialized objects of this class will not be compatible with
 * future swing releases.  The current serialization support is appropriate 
 * for short term storage or RMI between Swing1.0 applications.  It will
 * not be possible to load serialized Swing1.0 objects with future releases
 * of Swing.  The JDK1.2 release of Swing will be the compatibility
 * baseline for the serialized form of Swing objects.
 *
 * @version 1.21 02/02/98
 * @author Jeff Dinkins
 */
public class DefaultButtonModel implements ButtonModel, Serializable {

    protected int stateMask = 0;
    protected String actionCommand = null;
    protected ButtonGroup group = null;
        
    protected int mnemonic = 0;

    /**
     * Only one ChangeEvent is needed per button model instance since the
     * event's only state is the source property.  The source of events
     * generated is always "this".
     */
    protected transient ChangeEvent changeEvent = null;
    protected EventListenerList listenerList = new EventListenerList();
    
        
    /**
     * Constructs a JButtonModel
     *
     */
    public DefaultButtonModel() {
        stateMask = 0;
        setEnabled(true);
    }
        
    /**
     * Indicates if the button can be selected or not by
     * an input device (such as a mouse pointer).
     *
     * @see ButtonModel#setArmed
     */
    public final static int ARMED = 1 << 0;
        
    /**
     * Indicates if the button has been selected. Only needed for
     * certain types of buttons - such as RadioButton or Checkbox.
     */
    public final static int SELECTED = 1 << 1;
        
    /**
     * Indicates partial commitment towards choosing the
     * button.
     */
    public final static int PRESSED = 1 << 2;
        
    /**
     * Indicates whether the button is currently enabled.
     */
    public final static int ENABLED = 1 << 3;

    /**
     * Indicates that the mouse is over the button.
     */
    public final static int ROLLOVER = 1 << 4;
        
    /**
     * Sets the actionCommand that gets sent when the putton is pressed.
     */
    public void setActionCommand(String actionCommand) {
        this.actionCommand = actionCommand;
    }
        
    /**
     * Returns the action command for this button. 
     */
    public String getActionCommand() {
        return actionCommand;
    }

    /**
     * Returns whether the button is "armed".
     * 
     * @return true if the button is armed, and it can be selected
     * @see ButtonModel#setArmed
     */
    public boolean isArmed() {
        return (stateMask & ARMED) != 0;
    }
        
    /**
     * Checks if the button is selected.
     */
    public boolean isSelected() {
        return (stateMask & SELECTED) != 0;
    }
        
    /**
     * Checks if the button is disabled.
     */
    public boolean isEnabled() {
        return (stateMask & ENABLED) != 0;
    }
        
    /**
     * Checks if the button is pressed.
     */
    public boolean isPressed() {
        return (stateMask & PRESSED) != 0;
    }
        
    /**
     * Checks if the button is rolled over.
     */
    public boolean isRollover() {
        return (stateMask & ROLLOVER) != 0;
    }
        
    /**
     * Identifies the button as "armed". 
     *
     * @param b true to arm the button so it can be selected
     * @see ButtonModel#setArmed
     */
    public void setArmed(boolean b) {
        if((isArmed() == b) || !isEnabled()) {
            return;
        }
            
        if (b) {
            stateMask |= ARMED;
        } else {
            stateMask &= ~ARMED;
        }
            
        fireStateChanged();
    }

    /**
     * Sets the button to be enabled or disabled state
     */
    public void setEnabled(boolean b) {
        if(isEnabled() == b) {
            return;
        }
            
        if (b) {
            stateMask |= ENABLED;
        } else {
            stateMask &= ~ENABLED;
        }
            
        fireStateChanged();
    }
        
    /**
     * Sets the selected state of the button.
     * @param b true selects the toggle button,
     *          false deselects the toggle button.
     */
    public void setSelected(boolean b) {
        if (this.isSelected() == b) {
            return;
        }

        if (b) {
            stateMask |= SELECTED;
        } else {
            stateMask &= ~SELECTED;
        }

        fireItemStateChanged(
                new ItemEvent(this,
                              ItemEvent.ITEM_STATE_CHANGED,
                              this,
                              b ?  ItemEvent.SELECTED : ItemEvent.DESELECTED));
        
        fireStateChanged();
        
    }
        
        
    /**
     * Sets the button to pressed state
     */
    public void setPressed(boolean b) {
        if((isPressed() == b) || !isEnabled()) {
            return;
        }
        
        if (b) {
            stateMask |= PRESSED;
        } else {
            stateMask &= ~PRESSED;
        }

        if(!isPressed() && isArmed()) {
            fireActionPerformed(
                new ActionEvent(this, ActionEvent.ACTION_PERFORMED,
                                getActionCommand())
                );
        }
            
        fireStateChanged();
    }   

    /**
     * Sets the button to the rollover state
     */
    public void setRollover(boolean b) {
        if((isRollover() == b) || !isEnabled()) {
            return;
        }
        
        if (b) {
            stateMask |= ROLLOVER;
        } else {
            stateMask &= ~ROLLOVER;
        }

        fireStateChanged();
    }

    /**
     * Sets the keyboard mnemonic for this model
     */	
    public void setMnemonic(int key) {
	mnemonic = key;
	fireStateChanged();
    }

    /**
     * Gets the keyboard mnemonic for this model
     */
    public int getMnemonic() {
	return mnemonic;
    }

    /**
     * Adds a ChangeListener to the button.
     */
    public void addChangeListener(ChangeListener l) {
        listenerList.add(ChangeListener.class, l);
    }
    
    /**
     * Removes a ChangeListener from the button.
     */
    public void removeChangeListener(ChangeListener l) {
        listenerList.remove(ChangeListener.class, l);
    }

    /*
     * Notify all listeners that have registered interest for
     * notification on this event type.  The event instance 
     * is lazily created using the parameters passed into 
     * the fire method.
     * @see EventListenerList
     */
    protected void fireStateChanged() {
        // Guaranteed to return a non-null array
        Object[] listeners = listenerList.getListenerList();
        // Process the listeners last to first, notifying
        // those that are interested in this event
        for (int i = listeners.length-2; i>=0; i-=2) {
            if (listeners[i]==ChangeListener.class) {
                // Lazily create the event:
                if (changeEvent == null)
                    changeEvent = new ChangeEvent(this);
                ((ChangeListener)listeners[i+1]).stateChanged(changeEvent);
            }          
        }
    }   
    
    /**
     * adds an ActionListener to the button
     */
    public void addActionListener(ActionListener l) {
        listenerList.add(ActionListener.class, l);
    }
        
    /**
     * removes an ActionListener from the button
     */
    public void removeActionListener(ActionListener l) {
        listenerList.remove(ActionListener.class, l);
    }

    /*
     * Notify all listeners that have registered interest for
     * notification on this event type.  The event instance 
     * is lazily created using the parameters passed into 
     * the fire method.
     * @see EventListenerList
     */
    protected void fireActionPerformed(ActionEvent e) {
        // Guaranteed to return a non-null array
        Object[] listeners = listenerList.getListenerList();
        // Process the listeners last to first, notifying
        // those that are interested in this event
        for (int i = listeners.length-2; i>=0; i-=2) {
            if (listeners[i]==ActionListener.class) {
                // Lazily create the event:
                // if (changeEvent == null)
                // changeEvent = new ChangeEvent(this);
                ((ActionListener)listeners[i+1]).actionPerformed(e);
            }          
        }
    }   

    /**
     * adds an ItemListener to the button
     */
    public void addItemListener(ItemListener l) {
        listenerList.add(ItemListener.class, l);
    }
        
    /**
     * removes an ItemListener from the button
     */
    public void removeItemListener(ItemListener l) {
        listenerList.remove(ItemListener.class, l);
    }

    /*
     * Notify all listeners that have registered interest for
     * notification on this event type.  The event instance 
     * is lazily created using the parameters passed into 
     * the fire method.
     * @see EventListenerList
     */
    protected void fireItemStateChanged(ItemEvent e) {
        // Guaranteed to return a non-null array
        Object[] listeners = listenerList.getListenerList();
        // Process the listeners last to first, notifying
        // those that are interested in this event
        for (int i = listeners.length-2; i>=0; i-=2) {
            if (listeners[i]==ItemListener.class) {
                // Lazily create the event:
                // if (changeEvent == null)
                // changeEvent = new ChangeEvent(this);
                ((ItemListener)listeners[i+1]).itemStateChanged(e);
            }          
        }
    }   

    public Object[] getSelectedObjects() {
        return null; 
    }

    public void setGroup(ButtonGroup group) {
        this.group = group;
    }

}
