/*
 * @(#)AbstractAction.java	1.18 98/02/02
 * 
 * Copyright (c) 1997 Sun Microsystems, Inc. All Rights Reserved.
 * 
 * This software is the confidential and proprietary information of Sun
 * Microsystems, Inc. ("Confidential Information").  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sun.
 * 
 * SUN MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SUN SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 * 
 */
package com.sun.java.swing;

import java.awt.*;
import java.awt.event.*;
import java.beans.*;
import java.util.Hashtable;
import java.io.Serializable;

/**
 * This class provides default implementations for the JFC Action 
 * interface. Standard behaviors like the get and set methods for
 * Action object properties (icon, text, and enabled) are defined
 * here. The developer need only subclass this abstract class and
 * define the <code>actionPerformed</code> method. 
 * <p>
 * Warning: serialized objects of this class will not be compatible with
 * future swing releases.  The current serialization support is appropriate 
 * for short term storage or RMI between Swing1.0 applications.  It will
 * not be possible to load serialized Swing1.0 objects with future releases
 * of Swing.  The JDK1.2 release of Swing will be the compatibility
 * baseline for the serialized form of Swing objects.
 *
 * @version 1.18 02/02/98
 * @author Georges Saab
 * @see Action
 */
public abstract class AbstractAction implements Action, Cloneable, Serializable 
{
    protected boolean enabled = true;
    // Will be replaced by a lighter weight storage mechanism soon!
    private Hashtable keyTable = new Hashtable(5);

    /**
     * Defines an Action object with a default description string
     * and default icon.
     */
    public AbstractAction() {}
    
    /**
     * Defines an Action object with the specified description string
     * and a default icon.
     */
    public AbstractAction(String name) {
	putValue(Action.NAME, name);
    }

    /**
     * Defines an Action object with the specified description string
     * and a the specified icon.
     */
    public AbstractAction(String name, Icon icon) {
	this(name);
	putValue(Action.SMALL_ICON, icon);
    }

    /** @see Action#getValue */
    // Gets the Object associated with key
    public Object getValue(String key) {
	return keyTable.get(key);
    }

    /** @see Action#putValue */
    // Sets the Value associated with key
    public synchronized void putValue(String key, Object newValue) {
	Object oldValue = null;
	if (keyTable.containsKey(key))
	    oldValue = keyTable.get(key);
	keyTable.put(key,newValue);
	firePropertyChange(key, oldValue, newValue);
    }

    /** @see Action#isEnabled */
    public boolean isEnabled() {
	return enabled;
    }

    /** @see Action#setEnabled */
    public synchronized void setEnabled(boolean newValue) {
	boolean oldValue = this.enabled;
	this.enabled = newValue;
	firePropertyChange("enabled", 
			   new Boolean(oldValue), new Boolean(newValue));
    }

    /*
     * If any PropertyChangeListeners have been registered, the
     * changeSupport field describes them.
     */
    protected java.beans.PropertyChangeSupport changeSupport;

    /**
     * Support for reporting bound property changes.  This method can be called
     * when a bound property has changed and it will send the appropriate
     * PropertyChangeEvent to any registered PropertyChangeListeners.
     */
    protected void firePropertyChange(String propertyName, Object oldValue, Object newValue) {
        if (changeSupport == null) {
            return;
        }
        changeSupport.firePropertyChange(propertyName, oldValue, newValue);
    }


    /**
     * Add a PropertyChangeListener to the listener list.
     * The listener is registered for all properties.
     * <p>
     * A PropertyChangeEvent will get fired in response to setting
     * a bound property, e.g. setFont, setBackground, or setForeground.
     * Note that if the current component is inheriting its foreground, 
     * background, or font from its container, then no event will be 
     * fired in response to a change in the inherited property.
     *
     * @param listener  The PropertyChangeListener to be added
     *
     * @see Action#addPropertyChangeListener 
     */
    public synchronized void addPropertyChangeListener(PropertyChangeListener listener) {
        if (changeSupport == null) {
	    changeSupport = new java.beans.PropertyChangeSupport(this);
        }
        changeSupport.addPropertyChangeListener(listener);
    }


    /**
     * Remove a PropertyChangeListener from the listener list.
     * This removes a PropertyChangeListener that was registered
     * for all properties.
     *
     * @param listener  The PropertyChangeListener to be removed
     *
     * @see Action#removePropertyChangeListener 
     */
    public synchronized void removePropertyChangeListener(PropertyChangeListener listener) {
        if (changeSupport == null) {
            return;
        }
        changeSupport.removePropertyChangeListener(listener);
    }

}
