/* Remote target communications for serial-line targets in custom GDB protocol
   Copyright 1988, 1991, 1992, 1993, 1994, 1995, 1996 Free Software Foundation, Inc.
   Copyright 1993 CONVEX Computer Corporation.

This file is part of GDB.

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  */


/* Remote communication protocol.

req addr size data	description

   			to this request until it takes a breakpoint)
 #                      start the timing function
 ?			stop target and return status
 A			force a task attachment
 C			detach from target (target continues)
 G	   N   yes      write target registers
 M   ADDR  N   yes	write N bytes of data to ADDR
 O			open a channel - done by "target", not "attach"
 a			attach to task
 c			continue target
 g			read target registers
 k			kill target (terminate task or reboot microkernel)
 m   ADDR  N		read N bytes of data from ADDR
 s			step target by machine instructions
 v			request kgdb/tgdb protocol version
 w			poll until target stops (target does not respond
 
*/

#include "defs.h"

#ifdef ANSI_PROTOTYPES
#include <stdarg.h>
#else
#include <varargs.h>
#endif
#include <fcntl.h>
#include <sys/socket.h>
#include <netinet/in.h>		/* struct sockaddr_in */
#include <netinet/tcp.h>	/* TCP_NODELAY */
#include <netdb.h>		/* struct hostent */
#include <sys/time.h>
#include "frame.h"
#include "inferior.h"
#include "target.h"
#include "wait.h"
/*#include "terminal.h"*/
#include "gdbcmd.h"

#include "gdb_packet.h"

/*
 * Protocol history:
 * 
 * 1 - original style.  Kernel doesn't even support 'v' request, but
 *     returns '1' as an error code.
 *
 * 2 - added 'v'.  Extended registers to support all CR registers.
 *
 * 3 - Added 'A'.  If attach with 'a' returns status==2, then the
 *     requested task already has an attached gdb.  'A' means force
 *     the attach.  This is a special version.  Since the attach must
 *     be done *before* the 'v' command, we take status==2 as meaning
 *     version >= 3.
 *
 * 4 - Added the 'node' and 'spare' fields to the kgdb_request struct,
 *     made the data field 1024 instead of 512.  This is revlocked
 *     to a protocol 4 kernel.
 *
 * NOTE: When doing comparisons against target_protocol_version, keep
 * in mind that the version number may eventually change, so you
 * probably don't want to use '==' or '!='.  For example, if you add
 * feature "foo" in version 3, it will probably also exist in versions
 * 4,5,...  So make your code conditional on >=3, not ==3.
 */

#define CURRENT_PROTOCOL 4

#if !defined(DONT_USE_REMOTE)
#ifdef USG
#include <sys/types.h>
#endif

#include <signal.h>

/* Prototypes for local functions */

static int
kgdb_write_bytes PARAMS ((CORE_ADDR, char *, int));

static int
kgdb_read_bytes PARAMS ((CORE_ADDR, char *, int));

static void
kgdb_files_info PARAMS ((struct target_ops *));

static int
kgdb_xfer_memory PARAMS ((CORE_ADDR, char *, int, int, struct target_ops *));

static void 
kgdb_prepare_to_store PARAMS ((void));

static void
kgdb_fetch_registers PARAMS ((int));

static void
kgdb_resume PARAMS ((int, int, enum target_signal));

static void
kgdb_ether_open PARAMS ((char *, int));

static void
print_data PARAMS ((char *data, size_t size, FILE *f));

static void
kgdb_request_dump PARAMS ((char *title, kgdb_request_t request, FILE *f));

static void
kgdb_response_dump PARAMS ((char *title, kgdb_response_t response, FILE *f));

static int
net_clnt_call PARAMS ((int action,
		       unsigned int address,
		       unsigned int data_size,
		       void *data,
		       kgdb_response_t *response,
		       unsigned int *response_size ));

static void
kgdb_ether_close PARAMS ((int quitting));

static void (*kgdb_close) PARAMS ((int quitting)) = NULL;
     
static void
kgdb_store_registers PARAMS ((int));

static int
kgdb_wait PARAMS ((int, struct target_waitstatus *));

static void
kgdb_detach PARAMS ((char *, int));

static void
kgdb_kernel_timing PARAMS ((char *, int));

static void
kgdb_set_mach PARAMS ((char *, int));

static unsigned int
kgdb_get_time PARAMS ((void));

static int
#ifdef ANSI_PROTOTYPES
logprintf PARAMS ((char *fmt, ...));
#else
logprintf(va_dcl va_alist);
#endif
     
static void
kgdb_log_start PARAMS ((void));

static void
kgdb_log PARAMS ((char *d, char *s));

static char *
kgdb_pid_to_str PARAMS ((int));

extern struct target_ops kgdb_ops;	/* Forward decl */

int kiodebug = 0;
static unsigned int kiotimeout = 5; /* 0.5 seconds */
static int usetimeout;			/* 1 means need timeout (e.g. UDP) */

static int kgdb_log_flag = 0;
static FILE *kgdb_log_file = 0;
static char *kgdb_log_filename = "gdb-remote-log";
static int target_set_flag = 0;
/*static int target_protocol_version = -1;*/

/*
** Stuff for user-variables specific to these targets.
*/
/* Trapped internal variables are used to handle special registers.
   A trapped i.v. calls a hook here every time it is dereferenced,
   to provide a new value for the variable, and it calls a hook here
   when a new value is assigned, to do something with the value. */

/* Return 1 if NAME is a trapped internal variable, else 0. */

static int
is_trapped(name)
     char *name;
{
  return STREQ(name, "thread");
}

static value_ptr
value_of_trapped(var)
     struct internalvar *var;
{
  register value_ptr val;

  if (TYPE_CODE (VALUE_TYPE (var->value)) != TYPE_CODE_INT)
    val = value_from_longest (builtin_type_unsigned_int, 0);
  else
    {
      /* Copy code from value_of_internalvar () */

      val = value_copy (var->value);
      if (VALUE_LAZY (val))
	value_fetch_lazy (val);
    }

  VALUE_LVAL (val) = lval_internalvar;
  VALUE_INTERNALVAR (val) = var;
  return val;
}

/* Handle a new value assigned to a trapped internal variable.  We probably
   need to flush some state when changing node, task, or thread. */

static int
set_trapped(var, val, bitpos, bitsize, offset)
     struct internalvar *var;
     value_ptr val;
     int bitpos, bitsize, offset;
{
  int ret = 0;

  if (TYPE_CODE (VALUE_TYPE (val)) != TYPE_CODE_INT
      || value_as_long (val) < 0)
    error ("Unsigned integer type required");

  /* This is copied from values.c:set_internalvar().  We need to update
     the value before fetching new state from the inferior.  I don't know
     whether it is a good idea to update the value twice (once here and
     again after we return. */

  free ((PTR)var->value);
  var->value = value_copy (val);
  /* Force the value to be fetched from the target now, to avoid problems
     later when this internalvar is referenced and the target is gone or
     has changed.  */
  if (VALUE_LAZY (var->value))
    value_fetch_lazy (var->value);
  release_value (var->value);

  purge_gdb_caches();

  return ret;
}

/*
** Global for kernel timing function.
*/
static int is_kgdb_kernel_timing = 0;

void
kgdb_log_open ()
{
  if (kgdb_log_file != 0)
    fclose (kgdb_log_file);

  kgdb_log_file = 0;
  if (kgdb_log_flag)
    {
      kgdb_log_file = fopen (kgdb_log_filename, "a");
      if (!kgdb_log_file)
	{
	  extern char *sys_errlist[];
	  kgdb_log_flag = 0;
	  warning("Could not open log file: %s\n", sys_errlist[errno]);
	  return;
	}
    }
}

void
kgdb_log_close ()
{
  if (kgdb_log_file != 0)
    {
      fflush (kgdb_log_file);
      fclose (kgdb_log_file);
      kgdb_log_file = 0;
    }
}

static int
#ifdef ANSI_PROTOTYPES
logprintf(char *fmt, ...)
{
#else
logprintf(va_alist)
va_dcl
{
      char *fmt;
#endif
  int ret = 0;
  va_list args;

  if ((kgdb_log_file == 0) && kgdb_log_flag) {
      kgdb_log_open ();
      kgdb_log_start ();
  }

  if (kgdb_log_file)
    {
#if ANSI_PROTOTYPES
      va_start(args, fmt);
#else
      va_start (args);
      fmt = va_arg (args, char *);
#endif
      ret = vfprintf(kgdb_log_file, fmt, args);
      va_end(args);
    }
  return ret;
}

static struct timeval start_time, last_time;

static void
kgdb_log_start ()
{
  time_t time(), clock;
  char *ctime();

  if (kgdb_log_file == 0)
    return;

  clock = time ((time_t *) 0);
  gettimeofday (&start_time, 0);
  last_time = start_time;

  fprintf (kgdb_log_file, "\n* Remote debugging started at %s\n",
	   ctime (&clock));
}

static void
kgdb_log (d, s)
     char *d;			/* direction */
     char *s;			/* string sent/received */
{
  struct timeval now;
  double total_usec, incremental_usec;

  if ((kgdb_log_file == 0) && kgdb_log_flag) {
      kgdb_log_open ();
      kgdb_log_start ();
  }

  if (kgdb_log_file == 0)
      return;

  gettimeofday (&now, 0);
  total_usec = ((now.tv_sec * 1e6) + now.tv_usec) -
    ((start_time.tv_sec * 1e6) + start_time.tv_usec);
  incremental_usec = ((now.tv_sec * 1e6) + now.tv_usec) -
    ((last_time.tv_sec * 1e6) + last_time.tv_usec);
  last_time = now;

  fprintf (kgdb_log_file, "\n%10.0f %10.0f %s %s", total_usec,
	   incremental_usec, d, s);
}

/* Descriptor for I/O to remote machine.  Initialize it to -1 so that
   kgdb_ether_open knows that we don't have a file open when the program
   starts.  */
int kgdb_desc = -1;

static void
dump_data(data, size, f)
     char *data;
     size_t size;
     FILE *f;
{
  if (size != 0)
    {
      int i;
      if (isatty(fileno(f)))
	for (i = 0;  i < min(5, size);  ++i)
	  fprintf(f, " %02x", data[i] & 0xff);
      else
	for (i = 0;  i < size;  ++i)
	  {
	    if (i % 16 == 0)
	      fprintf(f, "\n    %04X:", i);
	    fprintf(f, " %02x", data[i] & 0xff);
	  }
    }
}

static void
kgdb_request_dump (title, request, f)
     char *title;
     kgdb_request_t request;
     FILE *f;
{
  fprintf(f, "%s", title);

  if (ntohl (request->type) != 0)
    {
      fprintf (f, "type %d (not a request)\n", ntohl (request->type));
      return;
    }

  fprintf (f, "%08x task %d thread %d %c", ntohl (request->xid),
	  ntohl (request->task_id), ntohl (request->thread_id),
	  ntohl (request->request));
  if (ntohl (request->address) != 0 || request->size != 0)
    fprintf (f, " address %08x", ntohl (request->address));
  if (ntohl (request->request) == 'G' ||
      ntohl (request->request) == 'M' ||
      ntohl (request->request) == 't')
    { 
      fprintf (f, "    size %d", ntohl(request->size));
      dump_data(request->data, ntohl(request->size), f);
    }
  fprintf(f, "\n");
}

static void
kgdb_response_dump (title, response, f)
     char *title;
     kgdb_response_t response;
     FILE *f;
{
  fprintf(f, "%s", title);

  if (ntohl (response->type) != 1)
    {
      fprintf (f, "type %d (not a response)\n", ntohl (response->type));
      return;
    }
      
  fprintf (f, "%08x        thread %d ", ntohl (response->xid),
	   ntohl (response->thread_id));
  fprintf (f, "     code %08x",
	   ntohl (response->return_code));
  fprintf(f, " size %4d", ntohl(response->size));
  dump_data(response->data, ntohl(response->size), f);
  fprintf (f, "\n");
}

int task_id;

static int
net_clnt_call (action, address, data_size, data, response, response_size) 
     int action;
     unsigned int address;
     unsigned int data_size;
     void *data;
     kgdb_response_t *response;
     unsigned int *response_size;
{
  static unsigned int xid = 0;
  static struct kgdb_request req;
  static struct kgdb_response resp;
  kgdb_request_t request = &req;
  unsigned int expected_size;
  unsigned int size;
  unsigned int retries;
  ssize_t cnt;

  retries = usetimeout ? 5 : 1;

  request->xid = htonl (xid++);
  request->type = htonl (0);
  request->task_id = htonl (task_id);
  request->thread_id = htonl
      (value_as_long (value_of_internalvar (lookup_internalvar ("thread"))));
  request->request = htonl (action);
  request->address = htonl (address);
  
  request->size = htonl (data_size);
  if (data != 0)
    bcopy (data, request->data, data_size);

  size = sizeof(*request) - sizeof(request->data) + data_size;

  while (retries-- > 0)
    {
      int status;
      struct timeval timeout;
      fd_set kgdb_set;
      
      if (kiodebug)
	kgdb_request_dump("\nsending  ", request, stderr);
      if (kgdb_log_flag)
	kgdb_request_dump("\nsending  ", request, kgdb_log_file);

      cnt = write (kgdb_desc, request, size);

      if (kiodebug & 2)
	fprintf(stderr, "\twrite %d  ", cnt);

      while (1)
	{
	  timeout.tv_sec = kiotimeout / 10;
	  timeout.tv_usec = (kiotimeout % 10) * 100000;
	  FD_ZERO (&kgdb_set);
	  FD_SET (kgdb_desc, &kgdb_set);

          if (kiodebug & 2)
	    fprintf(stderr, "set 0x%x ", kgdb_set);

#ifdef	__hpux
	  status = select (kgdb_desc + 1, (int *) &kgdb_set, 0, 0, 
			   usetimeout ? &timeout : NULL);
#else
	  status = select (kgdb_desc + 1, (fd_set *) &kgdb_set, 0, 0, 
			   usetimeout ? &timeout : NULL);
#endif
          if (kiodebug & 2)
	    fprintf(stderr, "sel %d  set 0x%x ", status, kgdb_set);

	  if (status < 0)
	    {
	      if (errno == EINTR)
		{
		  printf ("(interrupt)\n");
		  return -1;
		}
	      else
		{
		  printf ("select returned %d, errno %d\n", status, errno);
		  return -1;
		}
	    }
	  
	  if (status == 0)
	    break;		/* timeout */
	  
	  status = read (kgdb_desc, &resp, sizeof (resp));
          if (kiodebug & 2)
	    fprintf(stderr, "read %d errno %d\n", status, errno);
	  if (status <= 0)
	    break;
	  
	  if (kiodebug)
	    kgdb_response_dump("received ", &resp, stderr);
	  if (kgdb_log_flag)
	    kgdb_response_dump("\nreceived ", &resp, kgdb_log_file);
	  
	  /*
	   * If we got a response to a prior packet, we probably ^C'd
	   * out of the previous request, thus getting it's response.
	   * So for this we just loop around and read the next response,
	   * without even counting this as a retry. 
	   */
	  if (resp.xid < request->xid  &&  ntohl(resp.type) == 1)
	    continue;
	  
	  if (resp.xid != request->xid || ntohl (resp.type) != 1)
	    break;
	  
	  resp.task_id = ntohl (resp.task_id);
	  resp.thread_id = ntohl (resp.thread_id);
	  resp.return_code = ntohl (resp.return_code);
	  resp.size = ntohl (resp.size);
	  
	  if (resp.return_code == -1)
	    {
	      printf("\nRemote system requested force exit, going away....\n");
	      exit(1);
	    }
	  
	  if (response == 0)
	    return resp.return_code;
	  
	  *response = &resp;
	  
	  if (response_size == 0)
	    return resp.return_code;
	  else
	    expected_size = *response_size;
	  
	  *response_size = resp.size;
	  if (resp.size != expected_size)
	    return 1;
	  
	  return resp.return_code;
	} /* while 1 */
      
    }

  if (kiodebug)
    fprintf(stderr, "net_clnt_call timeout\n");
  if (kgdb_log_flag)
    fprintf(kgdb_log_file, "net_clnt_call timeout\n");

  if (action != 'w')
        printf("Remote system communication timeout, retries exceeded\n");

  return -1;	/* timeout */
}


/* Clean up connection to a remote debugger.  */

/* ARGSUSED */
static void
kgdb_ether_close (quitting)
     int quitting;
{
  logprintf("\nkgdb_ether_close(quitting=%d)\n", quitting);
  if (kgdb_desc >= 0)
    close (kgdb_desc);
  kgdb_desc = -1;
  kgdb_log_close ();
  kgdb_close = NULL;
}

static void
parse_hostname(name, protocol, dflt_port, addr, hostname, portname)
     char **name;
     char *protocol;
     char *dflt_port;
     struct sockaddr_in *addr;
     char **hostname;
     char **portname;
{
  static char host[64];
  static char port[32];
  
  struct hostent *iphost;
  struct servent *service;
  char *p = *name;
  char *o;

  while (isspace(*p))		/* skip whitespace */
    ++p;

  /* read hostname out of name:port */
  o = host;
  while (!isspace(*p)  &&  *p != ':'  &&  *p)
    *o++ = *p++;
  *o = '\0';

  /* read port name */
  if (*p == ':')
    {
      while (*p == ':')
	p++;
      if (isspace(*p))
	error("Can't parse hostname:port \"%s\"", *name);
      o = port;
      while (!isspace(*p)  &&  *p)
	*o++ = *p++;
      *o = '\0';
    }
  else
    strcpy(port, dflt_port);
  
  bzero(addr, sizeof(*addr));

  /* Convert IP name/number */
  addr->sin_addr.s_addr = inet_addr(host);
  if (addr->sin_addr.s_addr != -1)
    {
      iphost = gethostbyaddr((char*)&addr->sin_addr,
			     sizeof(addr->sin_addr), AF_INET);
      if (iphost)
	strcpy(host, iphost->h_name);
      else
	error("%s: unknown host", host);
    }
  else
    {
      iphost = gethostbyname(host);
      if (iphost)
	bcopy(iphost->h_addr, &addr->sin_addr, iphost->h_length);
      else
	error("%s: unknown host", host);
    }
  addr->sin_family = iphost->h_addrtype; /* should be AF_INET */
  
  /* Convert service name/number */
  addr->sin_port = atoi(port);
  if (addr->sin_port != 0)
    {
      service = getservbyport(addr->sin_port, protocol);
      if (service != 0)
	strcpy(port, service->s_name);
    }
  else
    {
      service = getservbyname(port, protocol);
      if (service != 0)
	addr->sin_port = service->s_port;
      else
	error("%s:bad port name", port);
    }

  if (kiodebug) {
	fprintf(stderr, "sock addr 0x%x port 0x%x\n", addr->sin_addr.s_addr, 
                addr->sin_port);
  }

  *name = p;
  if (hostname)
    *hostname = host;
  if (portname)
    *portname = port;
}

/* Open a connection to a remote debugger.
   NAME is the filename used for communication.  */

char *kgdb_host;
extern char * (*ptr_pid_to_str) PARAMS ((int));
static void
kgdb_ether_open (args, from_tty)
     char *args;
     int from_tty;
{
  char *name;
  char *p;
  int on = 1;
  int status;
  struct sockaddr_in addr;
  char *service_name;

  logprintf("\nkgdb_ether_open(\"%s\", %d)\n", args, from_tty);
  
  if (args == 0)
    error (
"To open a remote debug connection, you need to specify what\n\
network host and service is attached to the remote system\n\
(e.g. \"hostname:remote-debug\").");

#ifdef XYZZY
  if (target_set_flag)
     {
     error ("Target already set, if not correct, detach before re-targeting.");
     }
#endif

  target_preopen (from_tty);

  if (kgdb_close)
    (*kgdb_close)(0);

  kgdb_close = kgdb_ether_close;
  parse_hostname(&args, "udp", "remote-debug", &addr, &name, &service_name);
  kgdb_desc = socket(AF_INET, SOCK_DGRAM, 0);
  if (kgdb_desc < 0)
    perror_with_name (name);
  if (connect(kgdb_desc, (struct sockaddr *) &addr, sizeof(addr)) < 0)
    perror_with_name (name);
  if (from_tty)
    printf("Remote debugging %s:%s\n", name, service_name);
  kgdb_host = savestring (name, strlen (name));

  ptr_pid_to_str = &kgdb_pid_to_str;

  target_set_flag = 1;

  kgdb_log_open ();
  kgdb_log_start ();

#if 0
  status = net_clnt_call ('?', 0, 0, 0, 0, 0);
  if (status != 0)
    error ("Can't attach to remote task");
#endif

  push_target (&kgdb_ops);		/* Switch to using remote target now */
  target_has_execution = 0;
  target_protocol_version = -1;
  usetimeout = 1;

#if 0
  inferior_pid = 1;		/* FIXME */
  start_remote ();		/* Initialize gdb process mechanisms */
#endif
}

extern int attach_flag;

/* Based on infrun.c:child_attach() */

static void
kgdb_attach (args, from_tty)
     char *args;
     int from_tty;
{
  int status;
  char *exec_file;

  logprintf("\nkgdb_attach(\"%s\", %d)\n", args, from_tty);

  dont_repeat();

  if (!args)
    error_no_arg ("task-id to attach");

  task_id = atoi (args);

  if (target_has_execution)
    {
      if (query ("A program is being debugged already.  Kill it? "))
	target_kill ();
      else
	error ("Inferior not killed.");
    }

  exec_file = (char *) get_exec_file (1);

  if (from_tty)
    {
      printf ("Attaching program:\n  %s task %d on %s\n", exec_file,
	      task_id, kgdb_host);
      fflush (stdout);
    }


  /* Make sure internal variable "thread" is initialized */
  set_internalvar (lookup_internalvar ("thread"),
                   value_from_longest (builtin_type_unsigned_int, 0));


  status = net_clnt_call ('a', 0, 0, 0, 0, 0);
  if (status == 2  &&
      query("Connection already open to task %d.  Force the attach? ",
	    task_id))
    status = net_clnt_call ('A', 0, 0, 0, 0, 0);
  if (status != 0)
    error ("Can't attach to task");
  
  attach_flag = 1;

  target_protocol_version = net_clnt_call('v', 0, 0, 0, 0, 0);
  if (target_protocol_version > CURRENT_PROTOCOL)
    {
      warning("This GDB uses a protocol older than what your kernel uses.\n"
	      "Use at your own risk.");
    }
  else if (target_protocol_version < CURRENT_PROTOCOL)
    {
      warning("This GDB is uses a newer protocol version (%d) than the one your\n"
	      "kernel supports (%d).  It can usually make do, but some features may not\n"
	      "work completely.", CURRENT_PROTOCOL, target_protocol_version);
    }
  
  inferior_pid = 1;
  target_has_execution = 1;

  start_remote ();
#if 0
  mark_breakpoints_out ();
  target_terminal_init ();
  clear_proceed_status ();
  stop_soon_quietly = 1;
  target_terminal_inferior ();
  wait_for_inferior ();
  normal_stop ();
#endif
}

/* kgdb_detach()
   takes a program previously attached to and detaches it.
   We better not have left any breakpoints
   in the program or it'll die when it hits one.
   Close the open connection to the remote debugger.
   Use this when you want to detach and do something else
   with your gdb.  */

static void
kgdb_detach (args, from_tty)
     char *args;
     int from_tty;
{
  int status;
#if 0 /* gdb-4.16 */
  char buf[PBUFSIZ];
#endif
 
  if (args)
    error ("Argument given to \"detach\" when remotely debugging.");
  
  logprintf("\nkgdb_detach(\"%s\", %d)\n", args, from_tty);
  
  status = net_clnt_call ('C', 0, 0, 0, 0, 0);
  if (status != 0) 
    {
    if (status == -1)
      {
	printf("\nTimeout on detach, forcing gdb exit...\n");
	exit(0);
      }
    else
      {
        error ("Error while detaching from remote task");
      }
    }
#if 0 /* gdb-4.16 */
  /* Tell the remote target to detach.  */
  strcpy (buf, "D");
  remote_send (buf);
#endif
  pop_target ();

  set_internalvar (lookup_internalvar ("thread"),
		   value_from_longest (builtin_type_unsigned_int, 0));

  attach_flag = 0;
  target_set_flag =  0;  /* Allow reset of target after detach */

  if (from_tty)
    printf ("Detached from remote task.\n");
}

/* kgdb_kill() -- kill a running task (reboot machine) */

static void
kgdb_kill ()
{
  int status;

  if (!attach_flag) error("Command invalid, remote not attached.");
  
  logprintf("\nkgdb_kill()\n");

  printf ("Killing remote task\n");
  status = net_clnt_call ('k', 0, 0, 0, 0, 0);
  if (status != 0)
    error ("Can't kill remote task");
  pop_target ();

  set_internalvar (lookup_internalvar ("thread"),
		   value_from_longest (builtin_type_unsigned_int, 0));

  attach_flag = 0;
  target_set_flag =  0;  /* Allow reset of target after detach */

  printf ("Killed remote task.\n");
}

/* Tell the remote machine to resume.  */

static void
kgdb_resume (pid, step, siggnal)
     int pid, step;
     enum target_signal siggnal;
{
  int status;

  if (!attach_flag) error("Command invalid, remote not attached.");
  
  logprintf("\nkgdb_resume(pid=%d, step=%d, siggnal=%d)\n", pid, step, siggnal);

  if (siggnal)
    error ("Can't send signals to a remote system.  Try `handle %d ignore'.",
	   siggnal);

  status = net_clnt_call (step ? 's' : 'c', 0, 0, 0, 0, 0);
  if (status != 0)
    error ("Can't %s remote thread", step ? "step" : "continue");
}
/* Tell the remote machine to resume a single thread.  */

void
one_cont_kgdb_resume (step, siggnal)
     int step, siggnal;
{
  int status;

  if (!attach_flag) error("Command invalid, remote not attached.");
  
  logprintf("\none_cont_kgdb_resume(step=%d, siggnal=%d)\n", step, siggnal);
  
  if (siggnal)
    error ("Can't send signals to a remote system.  Try `handle %d ignore'.",
	   siggnal);

  status = net_clnt_call ('1', 0, 0, 0, 0, 0);
  if (status != 0)
    error ("Can't continue remote thread");
}

/* Wait until the remote machine stops, then return,
   storing status in STATUS just as `wait' would.
   Returns "pid" (though it's not clear what, if anything, that
   means in the case of this target).  */

extern int inferior_pid;

static int
kgdb_wait (huh, status)
     int huh;
     struct target_waitstatus *status;

{
  kgdb_response_t response;
  int s;
  struct internalvar *thread;
  int siggnal;
  int size, pid;
  
  logprintf("\nkgdb_wait(%p)\n", status);
  
  status->kind = TARGET_WAITKIND_EXITED;
  status->value.integer = 0;

  while (1)
    {
      size = sizeof(siggnal);
      s = net_clnt_call('w', 0, 0, 0, &response, &size);
      if (s == 0)
	break;

      if (quit_flag)
	{
	  quit_flag = 0;
	  size = sizeof(siggnal);
	  s = net_clnt_call('?', 0, 0, 0, &response, &size);
	  if (s == 0)
	    break;

	  fprintf (stderr, "Can't suspend remote task\n");
	  terminal_ours ();
	  if (query ("Can't suspend remote task.  Disconnect from target? "))
	    {
	      target_mourn_inferior();
	      error ("Use the \"target\" command to reconnect.");
	    }
	  else
	    {
	      terminal_inferior();
	      continue;
	    }
        }
    }


  status->kind = TARGET_WAITKIND_STOPPED;
  status->value.sig = *(int*)response->data;

  /*
   * Only set the thread number if the value in the response is
   * different from the current value. 
   */
  thread = lookup_internalvar("thread");
  if (value_as_long(value_of_internalvar(thread)) != response->thread_id)
    set_internalvar(thread, value_from_longest(builtin_type_unsigned_int,
					       response->thread_id));

  pid = task_id ? response->thread_id :
                 (response->task_id << 16) + response->thread_id;

  inferior_pid = pid;
  printf_filtered("[Stopped in %s]\n", target_pid_to_str (pid));

  return ( pid );
}


/* Convert a process ID to a string.  Returns the string in a static
   buffer. This version should work for normal targets and remote
   targets (XXX but it should be in target.c XXX). */

char *
kgdb_pid_to_str (pid)
     int pid;
{
  static char buf[64];
  extern char *normal_pid_to_str(int);

  if (STREQ (current_target.to_shortname, "ether")) {
    if (task_id == 0)
      sprintf (buf, "task %d thread %d", pid >> 16, pid & 0xffff);
    else 
      sprintf (buf, "task %d thread %d", task_id, pid);
  }
  else {
    return(normal_pid_to_str(pid));
  }

  return buf;
}


/* Read the remote registers.  */

static void
kgdb_fetch_registers (regno)
     int regno;
{
  static char zeros[8];

  kgdb_response_t response;
  unsigned int response_size = (target_protocol_version >= 2)
                               ? REGISTER_BYTES
			       : 512;
  int status;
  extern char registers[];
  int r;

  if (!attach_flag) error("Command invalid, remote not attached.");
  
  logprintf("\nkgdb_fetch_registers(regno=%d)\n", regno);
  
  status = net_clnt_call ('g', 0, 0, 0, &response, &response_size);
  if (status != 0)
    error ("Error reading remote registers.  Status = %d", status);
  
  for (r = 0;  REGISTER_BYTE(r) < response_size;  ++r)
    supply_register(r, response->data + REGISTER_BYTE(r));
  /* fill any unsupported registers to zero.  this should only be */
  /* needed on old kernels that don't do the extended registers */
  while (r < NUM_REGS)
    supply_register(r++, zeros);
  
  if (is_kgdb_kernel_timing) {
	printf("kgdb_fetch_registers, getting the registers.\n");
  	printf("Timing to this breakpoint is 0x%x\n", kgdb_get_time());
  }

}

/* Prepare to store registers.  Since we send them all, we have to
   read out the ones we don't want to change first.  */

static void 
kgdb_prepare_to_store ()
{
  logprintf("\nkgdb_prepare_to_store()\n");
  kgdb_fetch_registers (-1);
}

/* Store the remote registers from the contents of the block REGISTERS.  */

static void
kgdb_store_registers (regno)
     int regno;
{
  int status;
  extern char registers[];

  if (!attach_flag) error("Command invalid, remote not attached.");
  
  logprintf("\nkgdb_store_registers(regno=%d)\n", regno);
  
  /* The current kgdb stub in the microkernel copies REGISTER_BYTES of */
  /* data from the packet into it's data strucutes, so it should be */
  /* safe to extend REGISTER_BYTES in gdb without having to update the */
  /* kernel (at least as far as this function is converned). */

  status = net_clnt_call ('G', 0, REGISTER_BYTES, registers, 0, 0);
  if (status != 0)
    error ("Error writing remote registers");

  /*
   * Some of the control registers are masked in, and not directly
   * written.  So we need to re-read them.
   */
  if (target_protocol_version >= 2)
    kgdb_fetch_registers(regno);
}

/* Write memory data directly to the remote machine.
   This does not inform the data cache; the data cache uses this.
   MEMADDR is the address in the remote memory space.
   MYADDR is the address of the buffer in our space.
   LEN is the number of bytes.  */

static int
kgdb_write_bytes (memaddr, myaddr, len)
     CORE_ADDR memaddr;
     char *myaddr;
     int len;
{
  int status;

  if (!attach_flag) error("Command invalid, remote not attached.");
  
  logprintf("\nkgdb_write_bytes(memaddr=%p, myaddr=%p, len=%d)\n",
	    memaddr, myaddr, len);
  
  status = net_clnt_call ('M', memaddr, len, myaddr, 0, 0);
  if (status != 0)
    error ("Error writing remote memory");

  return 0;
}

/* Read memory data directly from the remote machine.
   This does not use the data cache; the data cache uses this.
   MEMADDR is the address in the remote memory space.
   MYADDR is the address of the buffer in our space.
   LEN is the number of bytes.  */

static int
kgdb_read_bytes (memaddr, myaddr, len)
     CORE_ADDR memaddr;
     char *myaddr;
     int len;
{
  kgdb_response_t response;
  unsigned int response_size = len;
  int status;

  if (!attach_flag) error("Command invalid, remote not attached.");
  
  logprintf("\nkgdb_read_bytes(memaddr=%p, myaddr=%p, len=%d)\n",
	    memaddr, myaddr, len);
  
  status = net_clnt_call ('m', memaddr, len, 0, &response, &response_size);
  if (status != 0)
    error ("Error reading remote memory");

  bcopy (response->data, myaddr, len);
  return 0;
}

/* Read or write LEN bytes from inferior memory at MEMADDR, transferring
   to or from debugger address MYADDR.  Write to inferior if SHOULD_WRITE is
   nonzero.  Returns length of data written or read; 0 for error.  */

/* ARGSUSED */
static int
kgdb_xfer_memory(memaddr, myaddr, len, should_write, target)
     CORE_ADDR memaddr;
     char *myaddr;
     int len;
     int should_write;
     struct target_ops *target;			/* ignored */
{
  int origlen = len;
  int xfersize;
  int res;

  logprintf("\nkgdb_xfer_memory(memaddr=%p, myaddr=%p, len=%d,\n"
	    "                 should_write=%d, target='%s')\n",
	    memaddr, myaddr, len, should_write, target->to_shortname);
  
  while (len > 0)
    {
      if (len > 512)
	xfersize = 512;
      else
	xfersize = len;

      if (should_write)
        res = kgdb_write_bytes(memaddr, myaddr, xfersize);
      else
	res = kgdb_read_bytes (memaddr, myaddr, xfersize);
      if (res != 0)
	{
	  errno = EFAULT;
	  return 0;
        }
      memaddr += xfersize;
      myaddr  += xfersize;
      len     -= xfersize;
    }
  return origlen; /* no error possible */
}

static void
kgdb_files_info (ignore)
struct target_ops *ignore;
{
  printf ("Debugging a remote task over a network connection.\n");
}

static unsigned char mk_break_insn[] = {0x00, 0x00, 0xa0, 0x00};
static unsigned char break_insn[] = BREAKPOINT;
static unsigned char break_alt_insn[] = BREAKPOINT_ALT;

int
kgdb_insert_breakpoint (addr, contents_cache)
     CORE_ADDR addr;
     char *contents_cache;
{
  int val;
  int mk = (task_id == 0);

  logprintf("\nkgdb_insert_breakpoint(addr=%p, contents_cache=%p)\n",
	    addr, contents_cache);

  if (!mk && addr >= 0xc0000000U) {
    printf("WARNING: setting user breakpoint in kernel addr range??\n");
  }

  if (mk) {
    /* Use the alternate breakpoint byte codes */
    val = target_read_memory (addr, contents_cache, sizeof break_alt_insn);
    if (val == 0) {
      val = target_write_memory (addr, (char *) break_alt_insn, 
                                 sizeof break_alt_insn);
    }
    else {
      error("kgdb_insert_breakpoint: target read failed");
    }
  }
  else {
    /* Use the regular breakpoint byte codes */
    val = target_read_memory (addr, contents_cache, sizeof break_insn);
    if (val == 0)
      val = target_write_memory (addr, (char *) break_insn, sizeof break_insn);
    else
      error("kgdb_insert_breakpoint: target read failed");
  }

  return val;
}

int
kgdb_remove_breakpoint (addr, contents_cache)
     CORE_ADDR addr;
     char *contents_cache;
{
  logprintf("\nkgdb_remote_breakpoint(addr=%p, contents_cache=%p)\n",
	    addr, contents_cache);
 
  /* Assume user and kernel (std and alt) are the same size */  
  return target_write_memory (addr, contents_cache, sizeof break_insn);
}


static void
tasks_info (args, from_tty)
     char *args;
     int from_tty;
{
  struct task_info 
    {
      unsigned task;
      unsigned map;
      unsigned susp;
      unsigned threads;
    };
  
  kgdb_response_t response;
  unsigned int i;
  int status;
  struct task_info *tip;
  int task_index = 0;
  unsigned int saved_task;

  if (!attach_flag) error("Command invalid, remote not attached.");
  
  logprintf("\ntasks_info(\"%s\", %d)\n", args, from_tty);
  
  status = net_clnt_call ('T', 0, 0, 0, &response, 0);
  if (status != 0)
    error ("Can't get task information");

  printf_filtered("  id        task         map  susp  threads\n");
  do {
    tip = (struct task_info *) response->data;
    for (i = 0; i < response->size / sizeof(*tip); i++, tip++) {
      if (i == 0) {	/* detect old protocol case */
	if (!task_index)
	  saved_task = tip->task;
	else
	  if (saved_task == tip->task)
	    return;
      }
      printf_filtered(" %3d  %#010x  %#010x  %4d  %7d\n", 
		    task_index + i, tip->task, tip->map, tip->susp, tip->threads);
    }
    task_index = task_index+i;
  } while (response->size / sizeof(*tip) == 1024 / sizeof(struct task_info)
	     && net_clnt_call ('T', 0, 4, &task_index, &response, 0) == 0);
}

static void
threads_info (args, from_tty)
     char *args;
     int from_tty;
{
  static char state_name[] = {' ', 'R', 'S', 'W', 'U', 'H'};
  static char flag_name[] = {' ', 'S', 'I'};
  kgdb_response_t response;
  unsigned int i;
  int status;
  unsigned int *dp;
  int thread_info_args[2];
  int thread_index = 0;
  unsigned int saved_thread;
  unsigned int saved_chk_sum;


  if (!args) 
	thread_info_args[0] = task_id;		/* wanted task */
  else
	thread_info_args[0] = atoi(args);	/* wanted task */

  thread_info_args[1] = thread_index;		/* thread index to start */

/* printf("gdb: Trying for task %d\n", wanted_task); */

  if (!attach_flag) error("Command invalid, remote not attached.");

  if (task_id != 0 && thread_info_args[0] != task_id) {
	printf_filtered("warning: unless debugging kernel, can list threads in attached task only\n");
  }
  
  logprintf("\nthreads_info(\"%s\", %d)\n", args, from_tty);
  
  status = net_clnt_call('t', 0, 8,
			 &thread_info_args[0], &response, 0);
  if (status != 0)
    error ("Can't get thread information");

  /*
   *  User task case
   */
  if (task_id != 0) do {
    unsigned int chk_sum = 0;
    
    dp = (int *) response->data;
      
    if (thread_index == 0) {
      printf_filtered ("  Task %d:\n", task_id);
      printf_filtered ("  id state flags susp pc\n");
    }
    for (i = 0; i < response->size / sizeof(int); i++)
      chk_sum += *(((int *)dp)+i);
    if (thread_index == 0)	/* detect old protocol case */
      saved_chk_sum = chk_sum;
    else
      if (saved_chk_sum == chk_sum) 
	return;
    for (i = 0; i < response->size / (sizeof(int) * 4); i++)
      {
	unsigned int state, flags, susp, pc;
	
	state = *dp++;
	flags = *dp++;
	susp = *dp++;
	pc = *dp++;
	printf_filtered (" %3d   %c     %c    %2d  ",
			 i+thread_index, state_name[state],
			 flag_name[flags], susp);
	print_address (pc, stdout);
	printf_filtered ("\n");
      }
    thread_info_args[1] = thread_index = thread_index + i;
  } while (response->size / (sizeof(int) * 4) == 1024 / (sizeof(int) * 4) &&
	   net_clnt_call('t', 0, 8, &thread_info_args[0], &response, 0) == 0);

  /*
   *  Kernel task case
   */
  else do { 
    dp = (int *) response->data;
    
    if (thread_index == 0) {
      printf_filtered("Task %d\n", thread_info_args[0]);
      printf_filtered("  id   thread     kern sp  wait_event             \n");
    }
    for (i = 0; i < response->size / (sizeof(int) * 5); i++) 
      {
	unsigned int thread, sp, pc, wait_ev, return_pc;
	
	thread = *dp++;
	
	if (i == 0) {	/* detect old protocol case */
	  if (!thread_index)
	    saved_thread = thread;
	  else
	    if (saved_thread == thread)
	      return;
	}
  	sp = *dp++;
  	pc = *dp++;
  	wait_ev = *dp++;
  	return_pc = *dp++;
        if (sp & 1)
  	  printf_filtered("*%3d 0x%08x 0x%08x 0x%08x", 
	                  thread_index + i, thread, (sp & ~1), wait_ev);
	else
  	   printf_filtered(" %3d 0x%08x 0x%08x 0x%08x", 
                           thread_index + i, thread, sp, wait_ev);

	print_address_symbolic(wait_ev, stdout, asm_demangle, " ");

	if (return_pc) {
		printf_filtered("\n                           ");
		print_address(return_pc, stdout);
		printf_filtered("\n");
	} 
	else {
		printf_filtered("\n");
	}

      }
    thread_info_args[1] = thread_index = thread_index+i;
  } while (response->size / (sizeof(int) * 5) == 1024 / (sizeof(int) * 5) &&
	   net_clnt_call('t', 0, 8,
			 &thread_info_args[0],
			 &response, 0) == 0);
}


/* Define the target subroutine names */

static char hp700_doc[] = "\
Debug a remote workstation.\n\
Specify the IP # or hostname of the target.  Optional second argument\n\
is the port number of name.  By default, port \"remote-debug\" is used.\n\
\n\
Opens an Ethernet/IP/UDP connection to a remote machine.  The host and target\n\
should be on the same backbone.  You need to have \"remote-debug 468/udp\"\n\
in your /etc/services file.";

/* NOTE: Before changing the shortname, you must also change locations */
/* where current_target->to_shortname is compared to "remote", and */
/* change those, also */

struct target_ops kgdb_ops = {
  "ether",			/* to_shortname */
  "Remote network debugging",	/* to_longname */
  &hp700_doc[0],		/* to_doc */
  kgdb_ether_open,		/* to_open */
  kgdb_ether_close,		/* to_close */
  kgdb_attach,			/* to_attach */
  kgdb_detach,			/* to_detach */
  kgdb_resume,			/* to_resume */
  kgdb_wait,			/* to_wait */
  kgdb_fetch_registers,		/* to_fetch_registers */
  kgdb_store_registers,		/* to_store_registers */
  kgdb_prepare_to_store,	/* to_prepare_to_store */
  kgdb_xfer_memory,		/* to_xfer_memory */
  kgdb_files_info,		/* to_files_info */
  kgdb_insert_breakpoint,	/* to_insert_breakpoint */
  kgdb_remove_breakpoint,	/* to_remove_breakpoint */
  NULL,				/* to_terminal_init */
  NULL,				/* to_terminal_inferior */
  NULL,				/* to_terminal_ours_for_output */
  NULL,				/* to_terminal_ours */
  NULL,				/* to_terminal_info */
  kgdb_kill,			/* to_kill */
  NULL,				/* to_load */
  NULL,				/* to_lookup_symbol */
  NULL,				/* to_create_inferior */
  NULL,				/* to_mourn_inferior */
  NULL,				/* to_can_run */
  NULL,				/* to_notice_signals */      
  NULL,				/* to_thread_alive */
  NULL,				/* to_stop */
  process_stratum,		/* to_stratum */
  NULL,				/* to_next */
  1,				/* to_has_all_memory */
  1,				/* to_has_memory */
  1,				/* to_has_stack */
  1,				/* to_has_registers */
  1,				/* to_has_execution */
  NULL,				/* sections */
  NULL,				/* sections_end */
  is_trapped,			/* to_is_trapped */
  set_trapped,			/* to_set_trapped */
  value_of_trapped,		/* to_get_trapped */
  0,                            /* to_protocol */
  OPS_MAGIC			/* to_magic */
};

/*
** kgdb_kernel_timing:  This routine sends a message to the kernel gdb stubs
**                      to start timing the amount of time it takes from the
**                      time a user executes a continue to the amount of time
**                      it takes for the task to stop.
*/
static void
kgdb_kernel_timing(args, from_tty)
	char *args;
	int from_tty;

{

  int status;

  if (!attach_flag) error("Command invalid, remote not attached.");

  if (args)
    error ("Argument given to \"kernel_timing\" when remotely debugging.");

  logprintf("\nkgdb_kernel_timing(\"%s\", %d)\n", args, from_tty);
  
  if (is_kgdb_kernel_timing) {
	is_kgdb_kernel_timing = 0;
  } else {
	is_kgdb_kernel_timing = 1;
  }
  
  status = net_clnt_call ('#', 0, 0, 0, 0, 0);
  if (status != 0) 
    {
    if (status == -1)
      {
	printf("\nTimeout on kernel_timing, forcing kgdb exit...\n");
	exit(0);
      }
    else
      {
        error ("Error while kernel_timing from remote task");
      }
    }

  if (from_tty) {
    if (is_kgdb_kernel_timing) {
      printf ("Started remote timing.\n");
    } else {
      printf ("Stopped remote timing.\n");
    }
  }

}

static unsigned int
kgdb_get_time()
{

  kgdb_response_t response;
  unsigned int response_size = sizeof(unsigned int);
  unsigned int retval;
  int status;

  if (!attach_flag) error("Command invalid, remote not attached.");

  logprintf("\nkgdb_get_time()\n");
  
  status = net_clnt_call ('$', 0, 0, 0, &response, &response_size);
  if (status != 0)
    error ("Error reading the timing value");
  
  bcopy (response->data, &retval, sizeof(unsigned int));
  return retval;

}


/*
 * Set the current Mach task and thread.
 */

static void
kgdb_mthread(args, from_tty)
        char *args;
        int from_tty;
{
  char *tok;
  char *token = " .\t";
  int task_no, thread_no;
  value_ptr val;
  struct symbol *sym;

  if (!args)
    error_no_arg ("[ thread-id ] or [ thread-id task-id ] to activate");

  /* Find the task and thread id numbers */
  task_no = thread_no = -1;

  tok = strtok(args, token);
  if (tok == NULL)
    error_no_arg ("[ thread-id ] or [ thread-id task-id ] to activate");
  thread_no = atoi(tok);

  tok = strtok(NULL, token);
  if (tok != NULL)
    task_no = atoi(tok);

  if (thread_no < 0 || thread_no > 10000)
    error("invalid thread id");

  if (task_no != -1 && task_id != 0)
    error("must be debugging kernel to set active task");
  if (task_no != -1 && (task_no < 0 || task_no > 10000))
    error("invalid task id");

  /* The convenience variable $thread */
  set_internalvar (lookup_internalvar ("thread"),
                   value_from_longest (builtin_type_unsigned_int, thread_no));

  /* The convenience variable $task */
  if (task_no != -1 && task_id == 0) {
    set_internalvar (lookup_internalvar ("task"),
                     value_from_longest (builtin_type_unsigned_int, task_no));

    /* The global variable "task" in the remote */
    sym = lookup_symbol ("task", 0, VAR_NAMESPACE, 0, NULL);
    write_memory(sym->ginfo.value.address, (char *)&task_no, sizeof(int));
  }

  purge_gdb_caches();
}




void
_initialize_kgdb ()
{
  add_target(&kgdb_ops);

  add_show_from_set (
    add_set_cmd ("etherdebug", no_class, var_zinteger, (char *)&kiodebug,
		   "Set debugging of remote target.\n\
When enabled, each packet exchanged with the remote target\n\
is displayed. Bits control degree of debugging.", &setlist),
	&showlist);

  add_show_from_set (
    add_set_cmd ("timeout", no_class, var_uinteger, (char *)&kiotimeout,
		 "Set timeout for remote network I/O (tenths of seconds).",
		 &setlist), &showlist);
  
  add_show_from_set (
    add_set_cmd ("remotelog", no_class, var_boolean, (char *)&kgdb_log_flag,
		   "Set logging of remote I/O.\n\
When enabled, each packet sent to or received from the remote target\n\
is logged to \"gdb-remote-log\".", &setlist),
	&showlist);

  /*
   * These info commands use the "T" and "t" extensions of the remote
   * gdb protocol to retrieve the Mach task and thread lists. We use
   * mtasks and mthreads to distinguish them from the standard gdb builtin
   * command "info threads" (see thread.c), which we should eventually use.
   */
  add_info ("tasks", tasks_info, "Show active Mach tasks");
  add_info ("mtasks", tasks_info, "Show active Mach tasks");
  add_info ("mthreads", threads_info, "Show active Mach threads in task"); 

  /*
   * Allow the user to set the current Mach task and thread for backtraces.
   * This is a wrapper for setting the $thread and $task variables, and then
   * purging gdb's caches.
   */
  add_cmd("mthread", no_class, kgdb_mthread,
	  "Set current Mach thread.", &cmdlist);

  /*
   * The following is for kernel routine timing.  This stuff will only
   * make sense for in house usage.
   */
  add_cmd("kernel_timing", no_class, kgdb_kernel_timing,
	  "Turn on breakpoint timing function.", &cmdlist);

  /*
   * Check that user and kernel (i.e. std and alt) breakpoints are the
   * same size. Some code in this module depends on this fact.
   */
  if (sizeof(break_insn) != sizeof(break_alt_insn)) {
    error("_initialize_kgdb: user and kernel brkpts different size!");
  }
  ptr_pid_to_str = &kgdb_pid_to_str;
}
#endif  /* !defined(DONT_USE_REMOTE) */
