/* Machine-dependent functions for Mach 3.0 on PowerPC.

   Copyright (C) 1996 Free Software Foundation, Inc.

This file is part of GDB.

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */

#include "defs.h"
#include "inferior.h"
#include <assert.h>
#include <mach_error.h>

#define x(regname) offsetof(struct ppc_thread_state, regname)

/* Offsets in struct ppc_thread_state corresponding to the special
   registers from FIRST_SP_REGNUM to LAST_SP_REGNUM.  */
static int registermap[] = {
    x(srr0),	/* PC */
    x(srr1),	/* PS (MSR) */
    x(cr),	/* CR */
    x(lr),	/* LR */
    x(ctr),	/* CTR */
    x(xer),	/* XER */
    x(mq),	/* MQ */
};

int
ppc_thread_state_offset(regno)
    int regno;
{
    if (regno < FP0_REGNUM)
	return offsetof(struct ppc_thread_state, r0) + regno * sizeof (int);
    if (regno <= FPLAST_REGNUM)
	return offsetof(struct ppc_float_state, fpregs[0]) +
	       (regno - FP0_REGNUM) * sizeof (double);
    assert(regno >= FIRST_SP_REGNUM && regno <= LAST_SP_REGNUM);
    assert(sizeof registermap / sizeof registermap[0] ==
		LAST_SP_REGNUM - FIRST_SP_REGNUM + 1);
    return registermap[regno - FIRST_SP_REGNUM];
}

void
ppc_supply_thread_state_registers(flavor, state)
    thread_state_flavor_t flavor;
    thread_state_t state;
{
    int i, off;

    switch (flavor) {
    case PPC_THREAD_STATE:
	for (i = GP0_REGNUM; i < GP0_REGNUM + 32; i++) {
	    off = ppc_thread_state_offset(i);
	    supply_register(i, ((char *) state) + off);
	}
	for (i = FIRST_SP_REGNUM; i <= LAST_SP_REGNUM; i++) {
	    off = ppc_thread_state_offset(i);
	    supply_register(i, ((char *) state) + off);
	}
	break;
    case PPC_FLOAT_STATE:
	for (i = FP0_REGNUM; i <= FPLAST_REGNUM; i++) {
	    off = ppc_thread_state_offset(i);
	    supply_register(i, ((char *) state) + off);
	}
	break;
    default:
	printf("ppc_supply_thread_state_registers\n");
	abort();
    }
}

/* Construct a thread state of the appropriate flavor from the registers[]
   array if all the contained registers are valid, and return 1 on success.
   Otherwise (some of the registers are invalid) return 0.  */
int
ppc_make_thread_state(flavor, state)
    thread_state_flavor_t flavor;
    thread_state_t state;
{
    int i, off;

    switch (flavor) {
    case PPC_THREAD_STATE:
	i = GP0_REGNUM; break;
    case PPC_FLOAT_STATE:
#if 0
	i = FP0_REGNUM; break;
#else
	/* We can't do this until the FP status register gets added to
	   registers[].  */
	return 0;
#endif
    default:
	printf("ppc_make_thread_state\n");
	abort();
    }
    while (1) {
	if (!register_valid[i])
	    return 0;
	off = ppc_thread_state_offset(i);
	memcpy(((char *) state) + off, &registers[REGISTER_BYTE(i)],
	       REGISTER_RAW_SIZE(i));

	/* Move to next register; continue if there is one, else break. */
	switch (i) {
	case GP0_REGNUM + 31:
	    i = FIRST_SP_REGNUM;
	    continue;
	case FPLAST_REGNUM:
	case LAST_SP_REGNUM:
	    break;
	default:
	    i++;
	    continue;
	}
	break;
    }
    return 1;
}

void
ppc_store_all_registers()
{
    thread_state_data_t state;
    int i, off;
    kern_return_t ret;
    mach_msg_type_number_t count;

    if (!ppc_make_thread_state(PPC_THREAD_STATE, state)) {
	printf("ppc_store_all_registers\n");
	abort();
    }
    ret = thread_set_state(current_thread, PPC_THREAD_STATE,
			   (thread_state_t) state, PPC_THREAD_STATE_COUNT);
    if (ret != KERN_SUCCESS) {
machfail:
	warning("ppc_store_all_registers: %s", mach_error_string(ret));
	return;
    }

    /* Set up the ppc_float_state structure.  Currently the FP status
       register is not mapped anywhere as a gdb register (this is a bug)
       so we must pick up the old float_state before modifying it.  */
    count = PPC_FLOAT_STATE_COUNT;
    ret = thread_get_state(current_thread, PPC_FLOAT_STATE,
			   (thread_state_t) state, &count);
    if (ret != KERN_SUCCESS)
	goto machfail;
    for (i = FP0_REGNUM; i <= FPLAST_REGNUM; i++) {
	off = ppc_thread_state_offset(i);
	memcpy ((char *) state + off, &registers[REGISTER_BYTE (i)],
		REGISTER_RAW_SIZE (i));
    }
    ret = thread_set_state(current_thread, PPC_FLOAT_STATE,
			   (thread_state_t) state, count);
    if (ret != KERN_SUCCESS)
	goto machfail;
}
