/* Remote target communications for serial-line targets in custom GDB protocol
   Copyright 1988, 1991, 1992, 1993, 1994, 1995, 1996 Free Software Foundation, Inc.

This file is part of GDB.

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  */

/* Remote communication protocol.

   A debug packet whose contents are <data>
   is encapsulated for transmission in the form:

	$ <data> # CSUM1 CSUM2

	<data> must be ASCII alphanumeric and cannot include characters
	'$' or '#'.  If <data> starts with two characters followed by
	':', then the existing stubs interpret this as a sequence number.

	CSUM1 and CSUM2 are ascii hex representation of an 8-bit 
	checksum of <data>, the most significant nibble is sent first.
	the hex digits 0-9,a-f are used.

   Receiver responds with:

	+	- if CSUM is correct and ready for next packet
	-	- if CSUM is incorrect

   <data> is as follows:
   Most values are encoded in ascii hex digits.  Signal numbers are according
   to the numbering in target.h.

	Request		Packet

	set thread	Hct...		Set thread for subsequent operations.
					c = 'c' for thread used in step and 
					continue; t... can be -1 for all
					threads.
					c = 'g' for thread used in other
					operations.  If zero, pick a thread,
					any thread.
	reply		OK		for success
			ENN		for an error.

	read registers  g
	read version    v
	reply		XX....X		Each byte of register data
					is described by two hex digits.
					Registers are in the internal order
					for GDB, and the bytes in a register
					are in the same order the machine uses.
			or ENN		for an error.

	write regs	GXX..XX		Each byte of register data
					is described by two hex digits.
	reply		OK		for success
			ENN		for an error

        write reg	Pn...=r...	Write register n... with value r...,
					which contains two hex digits for each
					byte in the register (target byte
					order).
	reply		OK		for success
			ENN		for an error
	(not supported by all stubs).

	read mem	mAA..AA,LL..L	AA..AA is address, LL..LL is length.
	reply		XX..XX		XX..XX is mem contents
					Can be fewer bytes than requested
					if able to read only part of the data.
			or ENN		NN is errno

	write mem	MAA..AA,LL..LL,XX..XX
					AA..AA is address,
					LL....LL is number of bytes,
					XX..XX is data
	reply		OK		for success
			ENN		for an error (this includes the case
					where only part of the data was
					written).

	continue	cAA..AA		AA..AA is address to resume
					If AA..AA is omitted,
					resume at same address.

	step		sAA..AA		AA..AA is address to resume
					If AA..AA is omitted,
					resume at same address.

	continue with	Csig;AA		Continue with signal sig (hex signal
	signal				number).

	step with	Ssig;AA		Like 'C' but step not continue.
	signal

	last signal     ?               Reply the current reason for stopping.
                                        This is the same reply as is generated
					for step or cont : SAA where AA is the
					signal number.

	detach          D               Reply OK.

	There is no immediate reply to step or cont.
	The reply comes when the machine stops.
	It is		SAA		AA is the signal number.

	or...		TAAn...:r...;n...:r...;n...:r...;
					AA = signal number
					n... = register number (hex)
					  r... = register contents
					n... = `thread'
					  r... = thread process ID.  This is
						 a hex integer.
					n... = other string not starting 
					    with valid hex digit.
					  gdb should ignore this n,r pair
					  and go on to the next.  This way
					  we can extend the protocol.
	or...		WAA		The process exited, and AA is
					the exit status.  This is only
					applicable for certains sorts of
					targets.
	or...		XAA		The process terminated with signal
					AA.
        or...           OXX..XX	XX..XX  is hex encoding of ASCII data. This
					can happen at any time while the program is
					running and the debugger should
					continue to wait for 'W', 'T', etc.

	thread alive	TXX		Find out if the thread XX is alive.
	reply		OK		thread is still alive
			ENN		thread is dead
	
	remote restart	RXX		Restart the remote server

	extended ops 	!		Use the extended remote protocol.
					Sticky -- only needs to be set once.

	kill request	k

	toggle debug	d		toggle debug flag (see 386 & 68k stubs)
	reset		r		reset -- see sparc stub.
	reserved	<other>		On other requests, the stub should
					ignore the request and send an empty
					response ($#<checksum>).  This way
					we can extend the protocol and GDB
					can tell whether the stub it is
					talking to uses the old or the new.
	search		tAA:PP,MM	Search backwards starting at address
					AA for a match with pattern PP and
					mask MM.  PP and MM are 4 bytes.
					Not supported by all stubs.

	general query	qXXXX		Request info about XXXX.
	general set	QXXXX=yyyy	Set value of XXXX to yyyy.
	query sect offs	qOffsets	Get section offsets.  Reply is
					Text=xxx;Data=yyy;Bss=zzz

	Responses can be run-length encoded to save space.  A '*' means that
	the next character is an ASCII encoding giving a repeat count which
	stands for that many repititions of the character preceding the '*'.
	The encoding is n+29, yielding a printable character where n >=3 
	(which is where rle starts to win).  Don't use an n > 126.

	So 
	"0* " means the same as "0000".  */

#include <sys/ioctl.h>
#include "defs.h"
#include "gdb_string.h"
#include <fcntl.h>
#include <sys/time.h>
/* ?? *03.06.96 *NY*
#include <unistd.h>
#include <termios.h>
*/
#include "frame.h"
#include "inferior.h"
#include "bfd.h"
#include "symfile.h"
#include "target.h"
#include "wait.h"
/*#include "terminal.h"*/
#include "gdbcmd.h"
/* ?? *03.07.96 *NY**/
#include "swap.h"
#include "gdb_packet.h"
/**/
#include "objfiles.h"
#include "gdb-stabs.h"
#include "thread.h"

#include "dcache.h"

/*
 * START OF EDM DEPENDENT DEFINITIONS
 *
 * The following definitions are also found in edm_kernel/kgdb/gdb_defs.h.
 * They cannot be independently modified.
 */
/* #define PBUFSIZ 1024 *NY* 03.07.96 */
/* Having this larger than 400 causes us to be incompatible with m68k-stub.c
   and i386-stub.c.  Normally, no one would notice because it only matters
   for writing large chunks of memory (e.g. in downloads).  Also, this needs
   to be more than 400 if required to hold the registers (see below, where
   we round it up based on REGISTER_BYTES).  */
#define	PBUFSIZ	400
#define KGDB_PROTOCOL_VERSION 2


#ifndef CROSS_GDB
#define	POLL_DELAY 50000
#endif

/*
 * Communications protocol characters
 */
#define	STX	0x02	/* Start of packet */
#define	ETX	0x03	/* End of packet */
#define ACK	0x06	/* Positive Acknowlegement */
#define NAK	0x15	/* Negative Acknowlegement */
#define QIT	0x1d	/* ^]  local quit character */
#define ITR	0x1f	/* ^_  remote interrupt character */

/* Number of times to retry getting and putting packets.  Set
 *  to 0 if no retries desired.
 */
#ifdef REMOTE_WITH_RETRY
#define MAX_RETRIES	10
#endif

/*
 * Entry/Error flags
 */
#define	KGDB_BREAK_POINT	0	/* Break point trap */
#define	KGDB_SINGLE_STEP	1	/* Single step tran */
#define KGDB_COMMAND		2	/* Command from gdb */
#define KGDB_EXCEPTION		3	/* Unexpected kernel exception */
#define KGDB_KEXCEPTION		4	/* Unexpected kgdb exception */
#define KGDB_KERROR		5	/* Internal kgdb error */
#define KGDB_MEM_ERROR		6	/* Internal kgdb error */

#define KGDB_SIGTRAP		5

#define ERROR_SHIFT 		16
#define ERROR_MASK 		0xFFFF

char *i386_exceptions[] =
{
    "Divide error",
    "Debug exceptions",
    "Non-maskable interrupt",
    "Breakpoint",
    "Overflow",
    "Bounds check",
    "Invalid opcode",
    "Device not available",
    "Double fault",
    "FPU fault",
    "Invalid TSS",
    "Segment not present",
    "Stack fault",
    "General protection",
    "Page fault",
    "Reserved (16)",
    "Floating point error",
    "Alignment check"
};

int max_exceptions = sizeof(i386_exceptions)/sizeof(char *);

/*
 * END OF KGDB DEPENDENT DEFINITIONS
 */

/* #if !defined(DONT_USE_REMOTE) *NY* 03.07.96*/
#ifdef USG
#include <sys/types.h>
#endif

#include <signal.h>
#include "serial.h"

extern char * (*ptr_pid_to_str) PARAMS ((int));

/* Prototypes for local functions */

static void
parse_remote_sl_error PARAMS ((char *));

static int remote_sl_write_bytes PARAMS ((CORE_ADDR memaddr,
				       char *myaddr, int len));

static int remote_sl_read_bytes PARAMS ((CORE_ADDR memaddr,
				      char *myaddr, int len));

static void remote_sl_files_info PARAMS ((struct target_ops *ignore));

static int remote_sl_xfer_memory PARAMS ((CORE_ADDR memaddr, char *myaddr,
				       int len, int should_write,
				       struct target_ops *target));

static void remote_sl_prepare_to_store PARAMS ((void));

static void remote_sl_fetch_registers PARAMS ((int regno));

static void remote_sl_resume PARAMS ((int pid, int step,
				   enum target_signal siggnal));

static int remote_sl_start_remote PARAMS ((char *dummy));

static void remote_sl_open PARAMS ((char *name, int from_tty));

static void extended_remote_sl_open PARAMS ((char *name, int from_tty));

static void remote_sl_open_1 PARAMS ((char *, int, struct target_ops *));

static void remote_sl_close PARAMS ((int quitting));

static void remote_sl_store_registers PARAMS ((int regno));

static void remote_sl_mourn PARAMS ((void));

static void extended_remote_sl_restart PARAMS ((void));

static void extended_remote_sl_mourn PARAMS ((void));

static void extended_remote_sl_create_inferior PARAMS ((char *, char *, char **));

static void remote_sl_mourn_1 PARAMS ((struct target_ops *));

static void
getpkt PARAMS ((char *, int));
static void
new_getpkt PARAMS ((kgdb_packet_t, int));

static int
putpkt PARAMS ((char *));
static int 
new_putpkt PARAMS ((kgdb_packet_t, int));

static void
remote_sl_send PARAMS ((char *));

static int readchar PARAMS ((serial_t, int timeout));
#if 0
static int remote_sl_wait PARAMS ((WAITTYPE *status));
#else
static int remote_sl_wait PARAMS ((int pid, struct target_waitstatus *status));
#endif

static void remote_sl_kill PARAMS ((void));

static int
tohex PARAMS ((int));

static int
fromhex PARAMS ((int));

static void
remote_sl_detach PARAMS ((char *, int));

static void interrupt_query PARAMS ((void));

char *
normal_pid_to_str PARAMS ((int));

extern struct target_ops remote_sl_ops;	/* Forward decl */
extern struct target_ops extended_remote_sl_ops;	/* Forward decl */

static unsigned int kiotimeout = 2; /* 0.2 seconds */
static struct timeval start_time, last_time;

static int usetimeout;           /* 1 means need timeout (e.g. UDP) */
static int remote_sl_timeout = 2;

/*#ifdef __hpux__ */
typedef int             boolean_t;
/*#endif*/

/* This variable chooses whether to send a ^C or a break when the user
   requests program interruption.  Although ^C is usually what remote
   systems expect, and that is the default here, sometimes a break is
   preferable instead.  */

static int remote_sl_break;

/* Descriptor for I/O to remote machine.  Initialize it to NULL so that
   remote_open knows that we don't have a file open when the program
   starts.  */
serial_t remote_sl_desc = NULL;
static /*int*/ serial_t local_desc = NULL;
static int remote_sl_protocol_version = -1;
static int task = 0;

/* Maximum number of bytes to read/write at once.  The value here
   is chosen to fill up a packet (the headers account for the 32).  */
#define MAXBUFBYTES ((PBUFSIZ-32)/2)

/* Round up PBUFSIZ to hold all the registers, at least.  */
/* The blank line after the #if seems to be required to work around a
   bug in HP's PA compiler.  */
#if REGISTER_BYTES > MAXBUFBYTES

#undef	PBUFSIZ
#define	PBUFSIZ	(REGISTER_BYTES * 2 + 32)
#endif

/* Should we try the 'P' request?  If this is set to one when the stub
   doesn't support 'P', the only consequence is some unnecessary traffic.  */
/* That's not the case for hpux */
#if 1 /* mk++ does not support P yet */
#define INIT_STUB_SUPPORTS_P 0
#else
#define INIT_STUB_SUPPORTS_P 1
#endif
static int stub_supports_P = INIT_STUB_SUPPORTS_P;

/* These are the threads which we last sent to the remote system.  -1 for all
   or -2 for not sent yet.  */
int general_thread;
int cont_thread;

static void
set_thread (th, gen)
     int th;
     int gen;
{
  char buf[PBUFSIZ];
  int state = gen ? general_thread : cont_thread;
  if (state == th)
    return;
  buf[0] = 'H';
  buf[1] = gen ? 'g' : 'c';
  if (th == 42000)
    {
      buf[2] = '0';
      buf[3] = '\0';
    }
  else if (th < 0)
    sprintf (&buf[2], "-%x", -th);
  else
    sprintf (&buf[2], "%x", th);
  putpkt (buf);
  getpkt (buf, 0);
  if (gen)
    general_thread = th;
  else
    cont_thread = th;
}

/*  Return nonzero if the thread TH is still alive on the remote system.  */

static int
remote_sl_thread_alive (th)
     int th;
{
  char buf[PBUFSIZ];

  buf[0] = 'T';
  if (th < 0)
    sprintf (&buf[1], "-%x", -th);
  else
    sprintf (&buf[1], "%x", th);
  putpkt (buf);
  getpkt (buf, 0);
  return (buf[0] == 'O' && buf[1] == 'K');
}

/*  Restart the remote side; this is an extended protocol operation.  */

static void
extended_remote_sl_restart ()
{
  char buf[PBUFSIZ];

  /* Send the restart command; for reasons I don't understand the
     remote side really expects a number after the "R".  */
  buf[0] = 'R';
  sprintf (&buf[1], "%x", 0);
  putpkt (buf);

  /* Now query for status so this looks just like we restarted
     gdbserver from scratch.  */
  putpkt ("?");
  getpkt (buf, 0);
}


#define	INBUFSIZE	PBUFSIZ
#if 0 /* *NY* 03.07.96 */
typedef struct buf_cb {
	int	b_index;
	int	b_count;
	char	b_data[INBUFSIZE];
	int	b_fd;
} buf_cb_t;

static /*buf_cb_t*/ struct _serial_t remote_sl_cb;
static /*buf_cb_t*/ struct _serial_t local_cb;
#endif /* 0 */
static serial_ttystate remote_sl_state;
static serial_ttystate local_state;

#ifdef HAVE_TERMIOS
#include <termios.h>

struct hardwire_ttystate {
  struct termios termios;
};
#endif /* termios */

#ifdef HAVE_SGTTY
#include <terminal.h>
/* Needed for the code which uses select().  We would include <sys/select.h>
   too if it existed on all systems.  */
#include <sys/time.h>

struct hardwire_ttystate
{
  struct sgttyb sgttyb;
  struct tchars tc;
  struct ltchars ltc;
  /* Line discipline flags.  */
  int lmode;
};
#endif /* sgtty */

/* #define OLD_IOCTL_COMPAT 1 */

/* Clean up connection to a remote debugger.  */
#ifdef OLD_IOCTL_COMPAT
#ifdef __hpux__
#define TERM_SET_F TCSETATTRF
#define TERM_GET TCGETATTR
#define TERM_SET TCSETATTR
#else
#define TERM_SET_F TIOCSETAF
#define TERM_GET TIOCGETA
#define TERM_SET TIOCSETA
#endif
#endif

/* ARGSUSED */
static void
remote_sl_close (quitting)
     int quitting;
{
  if (remote_sl_desc)
    SERIAL_CLOSE (remote_sl_desc);
  remote_sl_desc = NULL;

  if (local_desc)
    SERIAL_CLOSE (local_desc);
  local_desc = NULL;
}

#define SERIAL_RAW_KGDB(SERIAL_T) go_raw_kgdb(SERIAL_T)

static void
save_local_term(void)
{
  /*serial_ttystate state;*/
  struct hardwire_ttystate state;
  if ((local_state = SERIAL_GET_TTY_STATE (local_desc)) == NULL)
    {
      error ("Unable to get local terminal attributes.");
      perror_with_name("remote-kgdb");
    }

  /* save current local state */
  state = *(struct hardwire_ttystate*) local_state;
#if 0 /* 03.27.96 */
  SERIAL_RAW_KGDB (local_desc);
#else
#ifdef HAVE_TERMIOS
  state.termios.c_cc[VMIN] = 1;
  state.termios.c_cc[VTIME] = 0;
  state.termios.c_lflag &= ~(ISIG|ICANON|ECHO);
  state.termios.c_iflag |= (IXON|IXOFF);
  state.termios.c_iflag |= IGNPAR;
  state.termios.c_iflag &= ~(IGNBRK|BRKINT|INLCR|IGNCR|ICRNL|IUCLC|INPCK|ISTRIP|IXANY);
  state.termios.c_oflag &= ~OPOST;
  state.termios.c_cflag &= ~(CSIZE|PARENB);
  state.termios.c_cflag |= (CS8|CREAD);
#endif /* HAVE_TERMIOS */

  /*if (tcsetattr(local_desc->fd, TCSAFLUSH, &((struct hardwire_ttystate*)state)->termios) < 0)*/
  if (SERIAL_SET_TTY_STATE (local_desc, &state))
    {
      error ("Unable to set local terminal attributes.");
      perror_with_name("remote-kgdb");
    }
#endif /* 0 */
}

static void
restore_local_term()
{
  /*((struct hardwire_ttystate*)local_state)->termios.c_lflag |= ECHO;*/
  if (SERIAL_SET_TTY_STATE (local_desc, local_state))
    {
      error ("Unable to set local terminal attributes.");
      perror_with_name("remote-kgdb");
    }
  free (local_state);
}

/* Stub for catch_errors.  */

static int
remote_sl_start_remote (dummy)
     char *dummy;
{
  char buf[PBUFSIZ] = {0};
  int i;
  unsigned char *r;
  unsigned char *p;
  struct internalvar *thread;

  start_remote ();		/* Initialize gdb process mechanisms */

  immediate_quit = 1;		/* Allow user to interrupt it */

/* first check the protocol version */
  sprintf (buf, "v");

  putpkt (buf);
  getpkt( (char*) buf, 0);

  r  = (unsigned char *)&remote_sl_protocol_version;
  p = &buf[0];

  for (i = 0; i < sizeof(int); i++)
    {
      if (p[0] == 0 || p[1] == 0)
        error ("Remote reply is too short: %s", buf);
        r[i] = fromhex (p[0]) * 16 + fromhex (p[1]);
        p += 2;
    }

  swapinlong (remote_sl_protocol_version);

  if (remote_sl_protocol_version != KGDB_PROTOCOL_VERSION) 
      warning ("This GDB supports remote protocol version %x, the target is version %x.\n", 
	      KGDB_PROTOCOL_VERSION, 
	      remote_sl_protocol_version);
  else
	if (remote_debug)
		printf_filtered ("Remote protocol check successful.\n");

  thread = lookup_internalvar ("thread");
  set_internalvar (thread, 
		   value_from_longest(builtin_type_unsigned_long, 0));

  immediate_quit = 0;
  return 1;
}

static void go_raw_kgdb PARAMS ((serial_t));

static void
go_raw_kgdb(scb)
     serial_t scb;
{
  serial_ttystate state;
  if ((state = SERIAL_GET_TTY_STATE (scb)) == NULL)
    fprintf_unfiltered(gdb_stderr, "remote-kgdb: get_tty_state failed: %s\n", safe_strerror(errno));

#ifdef HAVE_TERMIOS
  ((struct hardwire_ttystate*)state)->termios.c_cc[VMIN] = 1;
  ((struct hardwire_ttystate*)state)->termios.c_cc[VTIME] = 0;
  ((struct hardwire_ttystate*)state)->termios.c_lflag &= ~(ISIG|ICANON|ECHO);
  ((struct hardwire_ttystate*)state)->termios.c_iflag |= (IXON|IXOFF);
  ((struct hardwire_ttystate*)state)->termios.c_iflag &= ~(IGNBRK|IGNPAR|BRKINT|INLCR|IGNCR|ICRNL|IUCLC|INPCK|ISTRIP|IXANY);
  ((struct hardwire_ttystate*)state)->termios.c_oflag &= ~OPOST;
  ((struct hardwire_ttystate*)state)->termios.c_cflag &= ~(CSIZE|PARENB);
  ((struct hardwire_ttystate*)state)->termios.c_cflag |= (CLOCAL);
  ((struct hardwire_ttystate*)state)->termios.c_cflag |= CS8;
#endif

#ifdef HAVE_TERMIO
  ((struct hardwire_ttystate*)state)->termio.c_iflag |= (IXON|IXOFF);
  ((struct hardwire_ttystate*)state)->termios.c_iflag &= ~(IGNBRK|IGNPAR|BRKINT|INLCR|IGNCR|ICRNL|IUCLC|INPCK|ISTRIP|IXANY);
  ((struct hardwire_ttystate*)state)->termio.c_oflag &= ~OPOST;
  ((struct hardwire_ttystate*)state)->termio.c_lflag &= ~(ISIG|ICANON|ECHO);
  ((struct hardwire_ttystate*)state)->termio.c_cflag &= ~(CSIZE|PARENB);
  ((struct hardwire_ttystate*)state)->termio.c_cflag |= CLOCAL;
  ((struct hardwire_ttystate*)state)->termios.c_cflag |= CS8;
  ((struct hardwire_ttystate*)state)->termio.c_cc[VMIN] = 1;
  ((struct hardwire_ttystate*)state)->termio.c_cc[VTIME] = 0;
#endif

#ifdef HAVE_SGTTY
  ((struct hardwire_ttystate*)state)->sgttyb.sg_flags |= RAW | ANYP;
  ((struct hardwire_ttystate*)state)->sgttyb.sg_flags &= ~(CBREAK | ECHO);
#endif

  scb->current_timeout = 0;

  if (SERIAL_SET_TTY_STATE (scb, state))
    fprintf_unfiltered(gdb_stderr, "remote-kgdb: set_tty_state failed: %s\n", safe_strerror(errno));
}

/* Open a connection to a remote debugger.
   NAME is the filename used for communication.  */

static void
remote_sl_open (name, from_tty)
     char *name;
     int from_tty;
{
  remote_sl_open_1 (name, from_tty, &remote_sl_ops);
}

/* Open a connection to a remote debugger using the extended
   remote gdb protocol.  NAME is the filename used for communication.  */

static void
extended_remote_sl_open (name, from_tty)
     char *name;
     int from_tty;
{
  char buf[PBUFSIZ];

  /* Do the basic remote open stuff.  */
  remote_sl_open_1 (name, from_tty, &extended_remote_sl_ops);

  /* Now tell the remote that we're using the extended protocol.  */
  putpkt ("!");
  getpkt (buf, 0);

}

/* Generic code for opening a connection to a remote target.  */
static DCACHE *remote_sl_dcache;

void set_hardwire_open_flags PARAMS ((int));/* do not* use fcntl instead*04.22.96*/
static void
remote_sl_open_1 (name, from_tty, target)
     char *name;
     int from_tty;
     struct target_ops *target;
{
#if 0
  struct termios term;
#endif
  int a_rate, b_rate = 0;
  int baudrate_set = 0;

  if (name == 0)
    error (
"To open a remote debug connection, you need to specify what serial\n\
device is attached to the remote system (e.g. /dev/ttya).");

  target_preopen (from_tty);

  unpush_target (target);

  remote_sl_dcache = dcache_init (remote_sl_read_bytes, remote_sl_write_bytes);

  /*
   * If the default terminal characteristcs are ~CLOCAL the open will
   * not return until CD is asserted on the line.  If the line is truly
   * local and never asserts CD, the open will never complete unless
   * O_NDELAY is selected.
   */
  set_hardwire_open_flags (O_RDWR | O_NDELAY);/* do not*04.22.96: use fcntl*/
  remote_sl_desc = SERIAL_OPEN (name);
  set_hardwire_open_flags (0);
  
  if (!remote_sl_desc)
    perror_with_name (name);

  ptr_pid_to_str = &normal_pid_to_str;
#if 1
  {
    int flags = fcntl (remote_sl_desc->fd, F_GETFL, 0);
    if (flags == -1) 
      perror_with_name (name);
    fcntl (remote_sl_desc->fd, F_SETFL, flags | O_NDELAY);
  }
#endif
  if (baud_rate != -1)
    {
      if (SERIAL_SETBAUDRATE (remote_sl_desc, baud_rate))
	{
	  SERIAL_CLOSE (remote_sl_desc);
	  perror_with_name (name);
	}
    }
#ifdef __hpux__
  else
    /* force baud rate to 9600...the kernel expects this. */
      if (SERIAL_SETBAUDRATE (remote_sl_desc, 9600))
	{
	  SERIAL_CLOSE (remote_sl_desc);
	  perror_with_name (name);
	}
#endif /* __hpux__ */    
#if 0 /* NY * 03.13.96 */
  term.c_cc[VMIN] = 1;
  term.c_cc[VTIME] = 0;
  term.c_lflag &= ~(ISIG|ICANON|ECHO);
  term.c_iflag |= (IXON|IXOFF);
  term.c_iflag &= ~(IGNBRK|IGNPAR|BRKINT|INLCR|IGNCR|ICRNL|IUCLC|INPCK|ISTRIP|IXANY);
  term.c_oflag &= ~OPOST;
  term.c_cflag &= ~(CSIZE|PARENB);
  term.c_cflag |= (CLOCAL|PARENB);	/* Local/Even parity */
  term.c_cflag |= CS7;			/* 7 bit */

#ifdef OLD_IOCTL_COMPAT
#ifdef __i386__
#undef B9600
#define B9600 0x0000000d
#endif

  /* force baud rate to 9600...the kernel expects this. */
  term.c_cflag = (term.c_cflag & ~CBAUD) | B9600;

  if (ioctl (remote_sl_desc, TERM_SET, &term) < 0)
	  perror_with_name("remote-kgdb");
#else
  if (cfsetispeed(&term, B9600) < 0) 
	  perror_with_name("remote-kgdb");

  if (cfsetospeed(&term, B9600) < 0) 
	  perror_with_name("remote-kgdb");

  if (tcsetattr(remote_sl_desc, TCSANOW, &term) < 0) 
	  perror_with_name("remote-kgdb");

  if (tcflush(remote_sl_desc, TCIOFLUSH) < 0) 
	  perror_with_name("remote-kgdb");

#endif /* OLD_IOCTL_COMPAT */
#endif /* 0 */

  /*remote_cb.fd = remote_desc;*/
  SERIAL_RAW_KGDB (remote_sl_desc);

  /* If there is something sitting in the buffer we might take it as a
     response to a command, which would be bad.  */
  SERIAL_FLUSH_INPUT (remote_sl_desc);

  set_hardwire_open_flags (O_RDWR | O_NDELAY);/* do not *use fcntl instead*/
  local_desc = SERIAL_OPEN ("/dev/tty");
  set_hardwire_open_flags (0);

  if (!local_desc)
    perror_with_name ("/dev/tty");
  /*local_cb.fd = local_desc;*/
#if 1
  {
    int flags = fcntl (local_desc->fd, F_GETFL, 0);
    if (flags == -1) 
      perror_with_name (name);
    fcntl (local_desc->fd, F_SETFL, flags | O_NDELAY);
  }
#endif
  if (from_tty)
    {
      puts_filtered ("Remote debugging using ");
      puts_filtered (name);
#if defined (__alpha__)   /* alpha is different in that */
      puts_filtered (". Escape character is '^Shft_' (Ctrl-Shift-Undescore).");
#else
      puts_filtered (". Escape character is '^]'.");
#endif
      puts_filtered ("\n");
    }
  push_target (target);	/* Switch to using remote target now */

  /* Start out by trying the 'P' request to set registers.  We set this each
     time that we open a new target so that if the user switches from one
     stub to another, we can (if the target is closed and reopened) cope.  */
  stub_supports_P = INIT_STUB_SUPPORTS_P;

  general_thread = -2;
  cont_thread = -2;

  /* Without this, some commands which require an active target (such as kill)
     won't work.  This variable serves (at least) double duty as both the pid
     of the target process (if it has such), and as a flag indicating that a
     target is active.  These functions should be split out into seperate
     variables, especially since GDB will someday have a notion of debugging
     several processes.  */

  inferior_pid = 42000;
  /* Start the remote connection; if error (0), discard this target.
     In particular, if the user quits, be sure to discard it
     (we'd be in an inconsistent state otherwise).  */
  if (!catch_errors (remote_sl_start_remote, (char *)0, 
		     "Couldn't establish connection to remote target\n", RETURN_MASK_ALL))
    pop_target();
}

/* This takes a program previously attached to and detaches it.  After
   this is done, GDB can be used to debug some other program.  We
   better not have left any breakpoints in the target program or it'll
   die when it hits one.  */

static void
remote_sl_detach (args, from_tty)
     char *args;
     int from_tty;
{
  char buf[PBUFSIZ];

  if (args)
    error ("Argument given to \"detach\" when remotely debugging.");

  /* Tell the remote target to detach.  */
  strcpy (buf, "D");
  remote_sl_send (buf);

  pop_target ();
  if (from_tty)
    puts_filtered ("Ending remote debugging.\n");
}

/* Convert hex digit A to a number.  */

static int
fromhex (a)
     int a;
{
  if (a >= '0' && a <= '9')
    return a - '0';
  else if (a >= 'a' && a <= 'f')
    return a - 'a' + 10;
  else 
    error ("Reply contains invalid hex digit %d", a);
  return -1;
}

/* Convert number NIB to a hex digit.  */

static int
tohex (nib)
     int nib;
{
  if (nib < 10)
    return '0'+nib;
  else
    return 'a'+nib-10;
}

/* Tell the remote machine to resume.  */

static enum target_signal last_sent_signal = TARGET_SIGNAL_0;
int last_sent_step;

static void
remote_sl_resume (pid, step, siggnal)
     int pid, step;
     enum target_signal siggnal;
{
  char buf[PBUFSIZ];

  if (pid == -1)
    set_thread (inferior_pid, 0);
  else
    set_thread (pid, 0);

  dcache_flush (remote_sl_dcache);

  last_sent_signal = siggnal;
  last_sent_step = step;

  if (siggnal != TARGET_SIGNAL_0)
    {
      buf[0] = step ? 'S' : 'C';
      buf[1] = tohex (((int)siggnal >> 4) & 0xf);
      buf[2] = tohex ((int)siggnal & 0xf);
      buf[3] = '\0';
    }
  else
    strcpy (buf, step ? "s": "c");

  putpkt (buf);
}


/* Ask the user what to do when an interrupt is received.  */

static void
interrupt_query ()
{
  target_terminal_ours ();

  if (query ("Interrupted while waiting for the program.\n\
Give up (and stop debugging it)? "))
    {
      target_mourn_inferior ();
      return_to_top_level (RETURN_QUIT);
    }

  target_terminal_inferior ();
}

/* Number of bytes of registers this stub implements.  */
static int register_bytes_found;

/* If nonzero, ignore the next kill.  */
int kill_kludge;

/* Wait until the remote machine stops, then return,
   storing status in STATUS just as `wait' would.
   Returns "pid" (though it's not clear what, if anything, that
   means in the case of this target).  */

static int
remote_sl_wait (pid, status)
     int pid;
     struct target_waitstatus *status;
{
  unsigned char buf[PBUFSIZ];
  int thread_num = -1;

  unsigned char *r;
  int i;
  unsigned char *p;
  REGISTER_TYPE swap_reg;
  int reason;

  status->kind = TARGET_WAITKIND_EXITED;
  status->value.integer = 0;

  while (1)
    {
	/*
	 * getpkt() may facilitate an indeterminate amount of
	 * simple TTY-style chatter before detecting, collecting
	 * and returning a GDB packet.
	 */
      getpkt ((char *) buf, 1);

      switch (buf[0])
	{
	case 'E':		/* Error of some sort */
	  parse_remote_sl_error (&buf[1]);
	  continue;
	case 'T':		/* Status with PC, SP, FP, ... */
	  {
	    unsigned char regs[REGISTER_BYTES];

      /* Reply describes registers byte by byte, each byte encoded as two
         hex characters.  Suck them all up, then supply them to the
         register cacheing/storage mechanism.  */

	    p = &buf[1];		/* after T */
	    while (*p)
	      {
		unsigned char *p1;
		char *p_temp;
		unsigned LONGEST val;
#if 0
		/*
		 * 03.26.96
		 * for now, keep old protocol;
		 * switch later to the new one.
		 */
		/* Convert a string to a register value: */
		regno = strtol ((const char *) p, &p_temp, 16);
		p1 = (unsigned char *)p_temp;

		if (p1 == p)
		  {
		    p1 = (unsigned char *) strchr ((const char *) p, ':');
		    if (p1 == NULL)
		      warning ("Malformed packet (missing colon): %s\n\
Packet: '%s'\n", p, buf);
		    if (strncmp ((const char *) p, "thread", p1 - p) == 0)
		      {
			thread_num = strtol ((const char *) ++p1, &p_temp, 16);
			p = (unsigned char *)p_temp;
		      }
		  }
		else
#else                                                               /* #if 0 */
		p1 = p;
#endif                                                        /* #else #if 0 */
		  {
		    p = p1;
		    val = 0L;

		    register_bytes_found = REGISTER_BYTES;
		    for (i = 0; i < REGISTER_BYTES; i++)
		      {
			if (p[0] == 0 || p[1] == 0)
			  {
			    register_bytes_found = i;
			    warning ("Remote reply is too short: %s", buf);
			  }
			regs[i] = fromhex (p[0]) * 16 + fromhex (p[1]);
			p += 2;
		      }

		    for (i = 0; i < NUM_REGS; i++) {
		      supply_register (i, &regs[REGISTER_BYTE(i)]);
		    }
#if 1 /* to change it later, for new protocol * 03.26.96 *OSF*/

		    status->kind = TARGET_WAITKIND_STOPPED;
		    status->value.sig = (enum target_signal)
		      (((fromhex (p[0])) << 4) + (fromhex (p[1])));

		    goto got_status;
#endif
		  }
	      }
	  }
#ifdef	MikeDEBUG
	  goto got_status;
#else                                                    /* #ifdef MikeDEBUG */
	  /* fall through */
#endif                                             /* #else #ifdef MikeDEBUG */
	case 'S':		/* Old style status, just signal only */
	  status->kind = TARGET_WAITKIND_STOPPED;
	  status->value.sig = (enum target_signal)
	    (((fromhex (buf[1])) << 4) + (fromhex (buf[2])));

	  goto got_status;
	case 'Q':
#if 0 /* NY */
	    WSETSTOP ((*status), 0);
	    return 0;
#else
	    goto got_status;
#endif
	case 'W':		/* Target exited */
	  {
	    /* The remote process exited.  */
	    status->kind = TARGET_WAITKIND_EXITED;
	    status->value.integer = (fromhex (buf[1]) << 4) + fromhex (buf[2]);
	    goto got_status;
	  }
	case 'X':
	  status->kind = TARGET_WAITKIND_SIGNALLED;
	  status->value.sig = (enum target_signal)
	    (((fromhex (buf[1])) << 4) + (fromhex (buf[2])));
	  kill_kludge = 1;

	  goto got_status;
	case 'O':		/* Console output */
          target_terminal_ours_for_output ();
	  printf_filtered( "%s", &(buf[ 1 ])  );
          target_terminal_inferior ();
	  continue;
	case '\0':
	  if (last_sent_signal != TARGET_SIGNAL_0)
	    {
	      /* Zero length reply means that we tried 'S' or 'C' and
		 the remote system doesn't support it.  */
	      target_terminal_ours_for_output ();
	      printf_filtered
		("Can't send signals to this remote system.  %s not sent.\n",
		 target_signal_to_name (last_sent_signal));
	      last_sent_signal = TARGET_SIGNAL_0;
	      target_terminal_inferior ();

	      strcpy ((char *) buf, last_sent_step ? "s" : "c");
	      putpkt ((char *) buf);
	      continue;
	    }
	  /* else fallthrough */
	default:
	  warning ("Invalid remote reply: %s", buf);
	  continue;
	}
    }

  r = (char *)&reason;

  for (i = 0; i < sizeof(int); i++)
    {
      if (p[0] == 0 || p[1] == 0)
        error ("Remote reply is too short: %s", buf);
        r[i] = fromhex (p[0]) * 16 + fromhex (p[1]);
        p += 2;
    }

  swapinlong(reason);

  if ((reason & ERROR_MASK) != KGDB_SIGTRAP)
    {
      i = reason >> ERROR_SHIFT;
      if (i < max_exceptions)
        printf_filtered ("Unexpected remote exception %d: %s\n", i, 
          i386_exceptions[i]);
      else
        printf_filtered ("Unexpected remote exception %d", i);
      reason = KGDB_SIGTRAP;
    }
/*  WSETSTOP ((*status), reason); * *?? NY*/

 got_status:
  if (thread_num != -1)
    {
      /* Initial thread value can only be acquired via wait, so deal with
	 this marker which is used before the first thread value is
	 acquired.  */
      if (inferior_pid == 42000)
	{
	  inferior_pid = thread_num;
	  add_thread (inferior_pid);
	}
      return thread_num;
    }
  return inferior_pid;
}

static void
parse_remote_sl_error(
	char *buf)
{
  char *p = buf;
  char *r;
  int reason;
  int i;

  r = (char *)&reason;

  for (i = 0; i < sizeof(int); i++)
    {
      if (p[0] == 0 || p[1] == 0)
        error ("Remote reply is too short: %s", buf - 1);
        r[i] = fromhex (p[0]) * 16 + fromhex (p[1]);
        p += 2;
    }

  swapinlong(reason);

  switch (reason & ERROR_MASK)
    {
      case KGDB_MEM_ERROR:
/* Does this really need to be an error ?? */
        error ("Invalid remote address"); 
        break;
      case KGDB_KERROR:
        error ("Remote internal error");
        break;
      case KGDB_KEXCEPTION:
	i = reason >> ERROR_SHIFT;
	if (i < max_exceptions)
          error ("Remote internal exception %d: %s", i, i386_exceptions[i]);
	else
          error ("Remote internal exception %d", i);
        break;
      default:
        error ("Kgdb unknown error code %d (0x%x)", reason, reason);
        break;
    }
}

/* Read the remote registers into the block REGS.  */
/* Currently we just read all the registers, so we don't use regno.  */
/* ARGSUSED */
static void
remote_sl_fetch_registers (regno)
     int regno;
{

  struct kgdb_packet pkt_buf;
  unsigned int response_size = REGISTER_BYTES;

  int i, status;
  REGISTER_TYPE swap_reg;
  char *p;
  char regs[REGISTER_BYTES];

  set_thread (inferior_pid, 1);

 pkt_buf.cmd = 'g';
  pkt_buf.address = 0;
  pkt_buf.data[0] = (unsigned char) 0;
  pkt_buf.size = 0;
  pkt_buf.task_id = task;
  pkt_buf.thread_id = value_as_pointer(value_of_internalvar(lookup_internalvar("thread")));

  swapinlong(pkt_buf.task_id);
  swapinlong(pkt_buf.thread_id);

  new_putpkt (&pkt_buf, 1);
  new_getpkt (&pkt_buf, 0);

  if (pkt_buf.cmd == 'E')
    parse_remote_sl_error (&pkt_buf.cmd);

#if 0
  remote_sl_send (&pkt_buf);
#endif

  /* Reply describes registers byte by byte, each byte encoded as two
     hex characters.  Suck them all up, then supply them to the
     register cacheing/storage mechanism.  */

  p = pkt_buf.data;
  for (i = 0; i < REGISTER_BYTES; i++) {
      if (p[0] == 0 || p[1] == 0)
	error ("Remote reply is too short: %s", pkt_buf);
      regs[i] = fromhex (p[0]) * 16 + fromhex (p[1]);
      p += 2;
  }
  register_bytes_found = i;
  for (i = 0; i < NUM_REGS; i++) {
    supply_register (i, &regs[REGISTER_BYTE(i)]);
  }

}

/* Prepare to store registers.  Since we may send them all (using a
   'G' request), we have to read out the ones we don't want to change
   first.  */

static void 
remote_sl_prepare_to_store ()
{
  /* Make sure the entire registers array is valid.  */
  read_register_bytes (0, (char *)NULL, REGISTER_BYTES);
}

/* Store register REGNO, or all registers if REGNO == -1, from the contents
   of REGISTERS.  FIXME: ignores errors.  */

static void
remote_sl_store_registers (regno)
     int regno;
{
  char buf[PBUFSIZ];
  int i;
  char *p;

  set_thread (inferior_pid, 1);

  if (regno >= 0 && stub_supports_P)
    {
      /* Try storing a single register.  */
      char *regp;

      sprintf (buf, "P%x=", regno);
      p = buf + strlen (buf);
      regp = &registers[REGISTER_BYTE (regno)];
      for (i = 0; i < REGISTER_RAW_SIZE (regno); ++i)
	{
	  *p++ = tohex ((regp[i] >> 4) & 0xf);
	  *p++ = tohex (regp[i] & 0xf);
	}
      *p = '\0';
      remote_sl_send (buf);
      if (buf[0] != '\0')
	{
	  /* The stub understands the 'P' request.  We are done.  */
	  return;
	}

      /* The stub does not support the 'P' request.  Use 'G' instead,
	 and don't try using 'P' in the future (it will just waste our
	 time).  */
      stub_supports_P = 0;
    }

  buf[0] = 'G';
  
  /* Command describes registers byte by byte,
     each byte encoded as two hex characters.  */

  p = buf + 1;
  /* remote_prepare_to_store insures that register_bytes_found gets set.  */
  if (!register_bytes_found) 
    remote_sl_fetch_registers (1);           /* Arg currently unimportant. */

  for (i = 0; i < register_bytes_found; i++)
    {
      *p++ = tohex ((registers[i] >> 4) & 0xf);
      *p++ = tohex (registers[i] & 0xf);
    }
  *p = '\0';

  remote_sl_send (buf);
}
/* 
   Use of the data cache *used* to be disabled because it loses for looking at
   and changing hardware I/O ports and the like.  Accepting `volatile'
   would perhaps be one way to fix it.  Another idea would be to use the
   executable file for the text segment (for all SEC_CODE sections?
   For all SEC_READONLY sections?).  This has problems if you want to
   actually see what the memory contains (e.g. self-modifying code,
   clobbered memory, user downloaded the wrong thing).  

   Because it speeds so much up, it's now enabled, if you're playing
   with registers you turn it of (set remotecache 0)
*/

/* Read a word from remote address ADDR and return it.
   This goes through the data cache.  */

#if 0	/* unused? */
int
remote_sl_fetch_word (addr)
     CORE_ADDR addr;
{
  return dcache_fetch (remote_sl_dcache, addr);
}

/* Write a word WORD into remote address ADDR.
   This goes through the data cache.  */

void
remote_sl_store_word (addr, word)
     CORE_ADDR addr;
     int word;
{
  dcache_poke (remote_sl_dcache, addr, word);
}
#endif /* 0 (unused?) */

/* Write memory data directly to the remote machine.
   This does not inform the data cache; the data cache uses this.
   MEMADDR is the address in the remote memory space.
   MYADDR is the address of the buffer in our space.
   LEN is the number of bytes.  

   Returns number of bytes transferred, or 0 for error. */

static int
remote_sl_write_bytes (memaddr, myaddr, len)
     CORE_ADDR memaddr;
     char *myaddr;
     int len;
{
  char buf[PBUFSIZ];
  int i;
  char *p;
  int unswap_todo;
  char *b;

  int done;
  CORE_ADDR memaddr_done = memaddr;
  /* Chop the transfer down if necessary */

  done = 0;
  while (done < len)
    {
      int todo = len - done;
      int cando = PBUFSIZ /2 - 32; /* number of bytes that will fit. */
      if (todo > cando)
	todo = cando;
      unswap_todo = todo;
  p = buf;

  /* Encode the command */
  *p++ = 'M';

      memaddr_done = memaddr + done;
  swapinlong (memaddr_done);
  swapinlong (todo);

  /* Encode the addrress */
  b = (char *) &memaddr_done;
  for (i = 0; i < sizeof (memaddr); i++)
    {
      *p++ = tohex ((b[i] >> 4) & 0xf);
      *p++ = tohex (b[i] & 0xf);
    }
  *p++ = ',';

 /* Encode the byte count */
  b = (char *) &todo; /* &len */
  for (i = 0; i < sizeof(todo); i++)
    {
      *p++ = tohex ((b[i] >> 4) & 0xf);
      *p++ = tohex (b[i] & 0xf);
    }
  *p++ = ',';

  /* Encode the data */
  for (i = 0; i < unswap_todo /*unswap_len*/; i++)
    {
      *p++ = tohex ((myaddr[i + done] >> 4) & 0xf);
      *p++ = tohex (myaddr[i + done] & 0xf);
    }
  *p = '\0';

  putpkt (buf);
  getpkt (buf, 0);

  if (buf[0] == 'E')
    {
      /* There is no correspondance between what the remote protocol uses
	 for errors and errno codes.  We would like a cleaner way of
	 representing errors (big enough to include errno codes, bfd_error
	 codes, and others).  But for now just return EIO.  */
      errno = EIO;
      return 0;
    }
      done += unswap_todo;
    }
  return len;
}

/* Read memory data directly from the remote machine.
   This does not use the data cache; the data cache uses this.
   MEMADDR is the address in the remote memory space.
   MYADDR is the address of the buffer in our space.
   LEN is the number of bytes. 

   Returns number of bytes transferred, or 0 for error. */

static int
remote_sl_read_bytes (memaddr, myaddr, len)
     CORE_ADDR memaddr;
     char *myaddr;
     int len;
{
  char buf[PBUFSIZ];
  int i;
  char *p;
  char *b;
  int unswap_todo;
#if 1 /* code from gdb_4_16 remote.c; should be here? there's more */
  int done;
  CORE_ADDR memaddr_done = memaddr;
  /* Chop transfer down if neccessary */
#endif

#if 0
  if (len > PBUFSIZ / 2 - 1)
    abort ();
#endif

  done = 0;
  while (done < len)
    {
      int todo = len - done;      
      int cando = PBUFSIZ /2 - 32; /* number of bytes that will fit. */
      if (todo > cando)
	todo = cando;
      unswap_todo = todo;
  
  p = buf;

  /* Encode the command */
  *p++ = 'm';

      memaddr_done = memaddr + done;
  /* Encode the addrress */
  b = (char *) &memaddr_done;

  swapinlong (memaddr_done);
  swapinlong (todo);  /*len*/

  for (i = 0; i < sizeof(memaddr); i++)
    {
      *p++ = tohex ((b[i] >> 4) & 0xf);
      *p++ = tohex (b[i] & 0xf);
    }
  *p++ = ',';

 /* Encode the byte count */
  b = (char *) &todo;

  for (i = 0; i < sizeof(todo); i++)
    {
      *p++ = tohex ((b[i] >> 4) & 0xf);
      *p++ = tohex (b[i] & 0xf);
    }
  *p = '\0';

  putpkt (buf);
  getpkt (buf, 0);

  if (buf[0] == 'E')
    {
      /* There is no correspondance between what the remote protocol uses
	 for errors and errno codes.  We would like a cleaner way of
	 representing errors (big enough to include errno codes, bfd_error
	 codes, and others).  But for now just return EIO.  */
      errno = EIO;
      return 0;
    }

  /* Reply describes memory byte by byte,
     each byte encoded as two hex characters.  */

  p = buf;

  for (i = 0; i < unswap_todo; i++)
    {
      if (p[0] == 0 || p[1] == 0)
	/* Reply is short.  This means that we were able to read only part
	   of what we wanted to.  */
	break;
      myaddr[i+done] = fromhex (p[0]) * 16 + fromhex (p[1]);
      p += 2;
    }
      done += unswap_todo;
    }
  return len;
}

/* Read or write LEN bytes from inferior memory at MEMADDR, transferring
   to or from debugger address MYADDR.  Write to inferior if SHOULD_WRITE is
   nonzero.  Returns length of data written or read; 0 for error.  */

/* ARGSUSED */
static int
remote_sl_xfer_memory(memaddr, myaddr, len, should_write, target)
     CORE_ADDR memaddr;
     char *myaddr;
     int len;
     int should_write;
     struct target_ops *target;			/* ignored */
{
  return dcache_xfer_memory (remote_sl_dcache, memaddr, myaddr, len, should_write);
}

static void
remote_sl_files_info (ignore)
     struct target_ops *ignore;
{
  puts_filtered ("Debugging a target over a serial line.\n");
}

/* Stuff for dealing with the packets which are part of this protocol.
   See comment at top of file for details.  */

/* Read a single character from the remote end, masking it down to 7 bits. */

static int
readchar (desc, timeout)
     serial_t desc;
     int timeout;
{
  int ch;
  char *p = (desc == remote_sl_desc ? "Remote" : "Local");

  ch = SERIAL_READCHAR (desc, -1);

  switch (ch)
    {
    case SERIAL_EOF:
      error ("%s connection closed", p);
    case SERIAL_ERROR:
#if defined (__alpha__)   /* read on alpha is different in that */
      if (errno == EWOULDBLOCK)
	return 0;
#endif
      error ("%s communication error", p);
    case SERIAL_TIMEOUT:
      return 0;
    default:
      return ch & 0x7f;
    }
}

/* Send the command in BUF to the remote machine,
   and read the reply into BUF.
   Report an error if we get an error reply.  */

static void
remote_sl_send (buf)
     char *buf;
{
  putpkt (buf);
  getpkt (buf, 0);

  if (buf[0] == 'E')
    parse_remote_sl_error(&buf[1]);
}

/* Send a packet to the remote machine, with error checking.
   The data of the packet is in BUF.  */

static int
putpkt (buf)
     char *buf;
{
  int i;
  unsigned char csum = 0;
  char buf2[PBUFSIZ];
  int cnt = strlen (buf);
  int c = 0;
  int tcount = 0;
  char *p;
  int retries = 0;

  /* Copy the packet into buffer BUF2, encapsulating it
     and giving it a checksum.  */

  if (cnt > (int) sizeof (buf2) - 5)		/* Prosanity check */
    abort();

  p = buf2;
  *p++ = STX;

  for (i = 0; i < cnt; i++)
    {
      csum += buf[i];
      *p++ = buf[i];
    }
  *p++ = ETX;
  *p++ = tohex ((csum >> 4) & 0xf);
  *p++ = tohex (csum & 0xf);

  /* Send it over and over until we get a positive ack.  */

  while (1)
    {
      int started_error_output = 0;
      int next_chunk = 0;

try_putpkt_again:
      if (remote_debug)
	{
	  *p = '\0';
	  printf_unfiltered ("Sending packet: %s...", buf2);
	  gdb_flush(gdb_stdout);
	}
      if (SERIAL_WRITE (remote_sl_desc, buf2, p - buf2))
	perror_with_name ("putpkt: write failed");

      /* read until either an ACK is read */
      while (1)
	{
	  while ((c = readchar (remote_sl_desc, remote_sl_timeout)) == 0)
#ifdef POLL_DELAY
	    usleep (POLL_DELAY);
#else
	  ;
#endif
 	  if (remote_debug)
	    {
	      switch (c)
		{
		case ACK:
		case SERIAL_TIMEOUT:
		case '$':
		  if (started_error_output)
		    {
		      putchar_unfiltered ('\n');
		      started_error_output = 0;
		    }
		  break;
		}
	    }

	  switch (c)
	    {
	    case ACK:
	      if (remote_debug)
		printf_unfiltered("Ack\n");
	      return 1;
	    case SERIAL_TIMEOUT:
	      tcount ++;
	      if (tcount > 3)
		return 0;
	      break;		/* Retransmit buffer */
		  /* If we got a NAK...return and get an error packet */
	    case NAK:
	      {
		if (++retries < MAX_RETRIES) {
		  if (remote_debug)
		    printf("\nGot a NAK...try #%d: %s\n", retries, buf2);
		  continue;
		}
		else
		  return 0;
	      }
	    default:
	      if (remote_debug)
		{
		  if (!started_error_output)
		    {
		      started_error_output = 1;
		      printf_unfiltered ("putpkt: Junk: ");
		    }
		  putchar_unfiltered (c & 0177);
		}
	      continue;
	    }
	  break;		/* Here to retransmit */
	}

#if 0
      /* This is wrong.  If doing a long backtrace, the user should be
	 able to get out next time we call QUIT, without anything as violent
	 as interrupt_query.  If we want to provide a way out of here
	 without getting to the next QUIT, it should be based on hitting
	 ^C twice as in remote_wait.  */
      if (quit_flag)
	{
	  quit_flag = 0;
	  interrupt_query ();
	}
#endif
    }
}

/* Send a packet to the remote machine, with error checking.
   The data of the packet is in BUF.  */

static int
new_putpkt (kgdb_packet_t buf, int need_thread)
{
  int i;
  int retries = 0;
  unsigned char csum = 0;
  char buf2[PBUFSIZ];
  int cnt;
  char c = 0;
  int tcount = 0;

  char *p;
  char *q;

  /* Copy the packet into buffer BUF2, encapsulating it
     and giving it a checksum.  */

  cnt = buf->size + sizeof(buf->cmd);
  if (need_thread)
	  cnt += sizeof(buf->thread_id) + sizeof(buf->task_id);

  if (cnt > sizeof(buf2) - 5)		/* Prosanity check */
    abort();

  p = buf2;
  *p++ = STX;

  /* move the cmd */
  *p++ = buf->cmd;
  csum = buf->cmd;

 
  if (need_thread) {
      char c;

      /* move the task_id */
      q = (char*)&buf->task_id;
      for (i = 0; i < sizeof(buf->task_id); i++) {
	  c = tohex((*q >> 4) & 0xf);
          csum += c;
	  *p++ = c;
	  c = tohex(*q++ & 0xf);
          csum += c;
	  *p++ = c;
      }

      q = (char*)&buf->thread_id;
      for (i = 0; i < sizeof(buf->thread_id); i++) {
	  c = tohex((*q >> 4) & 0xf);
          csum += c;
	  *p++ = c;
	  c = tohex(*q++ & 0xf);
          csum += c;
	  *p++ = c;
      }
  }


  /* move the data a byte at a time and calculate the check sum*/
  q = (char*) buf->data;
  for (i = 0; i < buf->size; i++)
    {
      csum += *q;
      *p++ = *q++;
    }

  *p++ = ETX;
  *p++ =  tohex ((csum >> 4) & 0xf); 
  *p++ = tohex (csum & 0xf);

  /* Send it over and over until we get a positive ack.  */
  while (1)
    {
      int started_error_output = 0;
    if (remote_debug)
      {
	*p = '\0'; 
	printf_unfiltered ("Sending packet: %s...", buf2);
	gdb_flush(gdb_stdout);
      }

try_new_putpkt_again:
      if (SERIAL_WRITE (remote_sl_desc, buf2, p - buf2))
	perror_with_name ("putpkt: write failed");

      /* read until either an ACK is read */
      while (1)
	{
	  while ((c = (unsigned char) readchar (remote_sl_desc, remote_sl_timeout)) == 0)
#ifdef POLL_DELAY
	    usleep (POLL_DELAY);
#else
	  ;
#endif
	  if (remote_debug) 
	    {
	      switch (c)
		{
		case ACK:
		case SERIAL_TIMEOUT:
		case '$':
		  if (started_error_output)
		    {
		      putchar_unfiltered ('\n');
		      started_error_output = 0;
		    }
		}
	    }
	  
	  switch (c)
	    {
	    case ACK:
	      if (remote_debug)
		printf_unfiltered ("Ack\n");
	      return 1;
	    case SERIAL_TIMEOUT:
	      tcount ++;
	      if (tcount > 3)
		return 0;
	      break;		/* Retransmit buffer */

	      /* If we got a NAK...see if we want to try again, otherwise
		 return and get an error packet */
	    case NAK:
	      {
		if (++retries < MAX_RETRIES) {
		  if (remote_debug)
		    printf("\nGot a NAK...try #%d: %s\n", retries, buf2);
		  continue;
		}
		else
		  return 0; 
	      }
	    default:
	      if (remote_debug)
		{
		  if (!started_error_output)
		    {
		      started_error_output = 1;
		      printf_unfiltered ("putpkt: Junk: ");
		    }
		  putchar_unfiltered (c & 0177);
		}
	      continue;
	    }
	  break;		/* Here to retransmit */
	}
    }
}

/* Come here after finding the start of the frame.  Collect the rest into BUF,
   verifying the checksum, length, and handling run-length compression.
   Returns 0 on any error, 1 on success.  */

static int
read_frame (buf, is_kgdb_buf)
     char *buf;
     int is_kgdb_buf;
{
  unsigned char csum;
  unsigned char c1, c2;
  char *bp;
  int c;
  int count = 0;

  csum = 0;
  if (is_kgdb_buf)
    bp = &((kgdb_packet_t) buf)->data[0];
  else
  bp = buf;

  while (1)
    {
      while ((c = readchar (remote_sl_desc, remote_sl_timeout)) == 0)
#ifdef POLL_DELAY
	    usleep (POLL_DELAY);
#else
	  ;
#endif
      count++;

      switch (c)
	{
	case SERIAL_TIMEOUT:
	  if (remote_debug)
	    puts_filtered ("Timeout in mid-packet, retrying\n");
	  return 0;
	case STX:
	  if (remote_debug)
	    puts_filtered ("Saw new packet start in middle of old one\n");
	  return 0;		/* Start a new packet, count retries */
	case ETX:
	  {
	    unsigned char pktcsum;

	    *bp = '\000';

	    while ((c = readchar (remote_sl_desc, remote_sl_timeout)) == 0)
#ifdef POLL_DELAY
	      usleep (POLL_DELAY);
#else
	    ;
#endif
	    pktcsum = fromhex (c) << 4;
	    while ((c = readchar (remote_sl_desc, remote_sl_timeout)) == 0)
#ifdef POLL_DELAY
	      usleep (POLL_DELAY);
#else
	    ;
#endif
	    pktcsum |= fromhex (c);

	    if (csum == pktcsum)
	      return 1;

	    if (remote_debug) 
	      {
		printf_filtered ("Bad checksum, sentsum=0x%x, csum=0x%x, buf=",
				 pktcsum, csum);
		puts_filtered (buf);
		puts_filtered ("\n");
	      }
	    return 0;
	  }
#if 0 /* ?? */
	case '*':		/* Run length encoding */
	  csum += c;
	  c = readchar (remote_sl_desc, remote_sl_timeout);
	  csum += c;
	  c = c - ' ' + 3;	/* Compute repeat count */


	  if (c > 0 && c < 255 && bp + c - 1 < buf + PBUFSIZ - 1)
	    {
	      memset (bp, *(bp - 1), c);
	      bp += c;
	      continue;
	    }

	  *bp = '\0';
	  printf_filtered ("Repeat count %d too large for buffer: ", c);
	  puts_filtered (((kgdb_packet_t) buf)->data);
	  puts_filtered ("\n");
	  return 0;
#endif /* 0 */
	default:
	  if (bp < buf + PBUFSIZ - 1)
	    {
	      *bp++ = c;
	      csum += c;
	      continue;
	    }

	  printf_filtered ("bp=%08x, buf=%08x, c=%d count=%d\n",bp,buf,c,count);
	  puts_filtered (bp);
	  *bp = '\0';
	  puts_filtered ("Remote packet too long: ");
	  puts_filtered (buf);
	  puts_filtered ("\n");

	  return 0;
	}
    }
}

#ifndef FALSE
#define FALSE 0
#endif
#ifndef TRUE
#define TRUE 1
#endif
/* Read a packet from the remote machine, with error checking,
   and store it in BUF.  BUF is expected to be of size PBUFSIZ.
   If FOREVER, wait forever rather than timing out; this is used
   while the target is executing user code.  */

static void
getpkt (buf, forever)
     char *buf;
     int forever;
{
  char ack = ACK;
  char nak = NAK;
  char outbuf[INBUFSIZE];
  boolean_t quit;
  boolean_t interrupt=FALSE;
  boolean_t packet;
  int i;
  char itr = ITR;

  char c;
  int tries = 0;
  int timeout;
  int val;

  strcpy (buf,"timeout");

  if (forever)
    {
#ifdef MAINTENANCE_CMDS
      timeout = watchdog > 0 ? watchdog : -1;
#else
      timeout = -1;
#endif
    }
  else
    timeout = remote_sl_timeout;

#define MAX_TRIES 3

  for (tries = 1; tries <= MAX_TRIES; tries++)
    {
      save_local_term();
      while (1)
        {
          /* Poll the local keyboard */
          quit = FALSE;
          i = 0;
          while ((c = readchar (local_desc, timeout)) != 0)
            {
              if (c == QIT)
                {
                  quit = TRUE;
                  break;
                }
              else if (c == ITR)
                {
                  interrupt = TRUE;
                  break;
                }
	      else
                {
                  outbuf[i++] = c;
		  if (i >= sizeof (outbuf))
		    break;
                }
            }
          if (i > 0)
            SERIAL_WRITE (remote_sl_desc, outbuf, i);
          if (interrupt) {
  	    SERIAL_WRITE (remote_sl_desc, &itr, 1);
	    if (remote_debug) printf("getpkt: send interrupt\n");
	    interrupt = FALSE;
	  }

          if (quit)
            break;
    
          /* Poll the serial line */
          packet = FALSE;
          i = 0;
          while ((c = readchar(remote_sl_desc, remote_sl_timeout)) != 0)
	    {
              if (c == STX)
                {
                  packet = TRUE;
                  break;
                }
	      else if (c == ACK)
	        {
		  SERIAL_WRITE (remote_sl_desc, &c, 1); 
		}
              else
                {
                  outbuf[i++] = c;
		  if (i >= sizeof (outbuf))
		    break;
                }
            }
          if (i > 0)
            SERIAL_WRITE (local_desc, outbuf, i);
          if (packet)
            break;
#ifdef POLL_DELAY
	  usleep (POLL_DELAY);
#endif
        }
      restore_local_term();
	
      if (quit)
	{
	  buf[0] = 'Q';
	  return;
	}

      val = read_frame (buf, 0);

      if (val == 1)
	{
	  if (remote_debug)
	    fprintf_unfiltered (gdb_stderr, "Packet received: %s\n", buf);
	  SERIAL_WRITE (remote_sl_desc, &ack, 1);
	  return;
	}

      /* Try the whole thing again.  */
retry:
      SERIAL_WRITE (remote_sl_desc, &nak, 1);
    }
  /* We have tried hard enough, and just can't receive the packet.  Give up. */

  printf_unfiltered ("Ignoring packet error, continuing...\n");
  SERIAL_WRITE (remote_sl_desc, &ack, 1);
}

/* Read a packet from the remote machine, with error checking,
   and store it in BUF.  BUF is expected to be of size PBUFSIZ.
   If FOREVER, wait forever rather than timing out; this is used
   while the target is executing user code.  */

static void
new_getpkt (pkt_buf, forever)
     kgdb_packet_t pkt_buf;
     int forever;
{
  char ack = ACK;
  char nak = NAK;
  char outbuf[INBUFSIZE];
  boolean_t quit;
  boolean_t interrupt=FALSE;
  boolean_t packet;
  int i;
  char itr = ITR;

  char c;
  int tries = 0;
  int timeout;
  int val;

  strcpy (pkt_buf->data, "timeout");

  if (forever)
    {
#ifdef MAINTENANCE_CMDS
      timeout = watchdog > 0 ? watchdog : -1;
#else
      timeout = -1;
#endif
    }
  else
    timeout = remote_sl_timeout;

#define MAX_TRIES 3

  for (tries = 1; tries <= MAX_TRIES; tries++)
    {
      save_local_term();
      while (1)
        {
          /* Poll the local keyboard */
          quit = FALSE;
          i = 0;
          while ((c = readchar (local_desc, timeout)) != 0)
            {
              if (c == QIT)
                {
                  quit = TRUE;
                  break;
                }
              else if (c == ITR)
                {
                  interrupt = TRUE;
                  break;
                }
	      else
                {
                  outbuf[i++] = c;
                }
            }
          if (i > 0)
            SERIAL_WRITE (remote_sl_desc, outbuf, i);
          if (interrupt) {
  	    SERIAL_WRITE (remote_sl_desc, &itr, 1);
	    if (remote_debug) printf("getpkt: interrupt\n");
	    interrupt = FALSE;
	  }

          if (quit)
            break;
    
          /* Poll the serial line */
          packet = FALSE;
          i = 0;
          while ((c = readchar (remote_sl_desc, remote_sl_timeout)) != 0)
	    {
              if (c == STX)
                {
                  packet = TRUE;
                  break;
                }
	      else if (c == ACK)
	        {
		  SERIAL_WRITE (remote_sl_desc, &c, 1); 
		}
              else
                {
                  outbuf[i++] = c;
                }
            }
          if (i > 0)
            SERIAL_WRITE (local_desc, outbuf, i);
          if (packet)
            break;
#ifdef POLL_DELAY
	  usleep (POLL_DELAY);
#endif
        }
      restore_local_term();
	
      if (quit)
	{
	  pkt_buf->cmd = 'Q';
	  return;
	}

      val = read_frame (pkt_buf, 1);

      if (val == 1)
	{
	  if (remote_debug)
	    fprintf_unfiltered (gdb_stderr, "Packet received: %s\n", (char*)pkt_buf);
	  SERIAL_WRITE (remote_sl_desc, &ack, 1);
	  return;
	}

      /* Try the whole thing again.  */
retry:
      SERIAL_WRITE (remote_sl_desc, &nak, 1);
    }
  /* We have tried hard enough, and just can't receive the packet.  Give up. */

  printf_unfiltered ("Ignoring packet error, continuing...\n");
  SERIAL_WRITE (remote_sl_desc, &ack, 1);
}

/* The data cache leads to incorrect results because it doesn't know about
   volatile variables, thus making it impossible to debug functions which
   use hardware registers.  Therefore it is #if 0'd out.  Effect on
   performance is some, for backtraces of functions with a few
   arguments each.  For functions with many arguments, the stack
   frames don't fit in the cache blocks, which makes the cache less
   helpful.  Disabling the cache is a big performance win for fetching
   large structures, because the cache code fetched data in 16-byte
   chunks.  */
#if 0
/* The data cache records all the data read from the remote machine
   since the last time it stopped.

   Each cache block holds 16 bytes of data
   starting at a multiple-of-16 address.  */

#define DCACHE_SIZE 64		/* Number of cache blocks */

struct dcache_block {
	struct dcache_block *next, *last;
	unsigned int addr;	/* Address for which data is recorded.  */
	int data[4];
};

struct dcache_block dcache_free, dcache_valid;

/* Free all the data cache blocks, thus discarding all cached data.  */ 

static void
dcache_flush ()
{
  register struct dcache_block *db;

  while ((db = dcache_valid.next) != &dcache_valid)
    {
      remque (db);
      insque (db, &dcache_free);
    }
}

/*
 * If addr is present in the dcache, return the address of the block 
 * containing it.
 */

struct dcache_block *
dcache_hit (addr)
{
  register struct dcache_block *db;

  if (addr & 3)
    abort ();

  /* Search all cache blocks for one that is at this address.  */
  db = dcache_valid.next;
  while (db != &dcache_valid)
    {
      if ((addr & 0xfffffff0) == db->addr)
	return db;
      db = db->next;
    }
  return NULL;
}

/*  Return the int data at address ADDR in dcache block DC.  */

int
dcache_value (db, addr)
     struct dcache_block *db;
     unsigned int addr;
{
  if (addr & 3)
    abort ();
  return (db->data[(addr>>2)&3]);
}

/* Get a free cache block, put it on the valid list,
   and return its address.  The caller should store into the block
   the address and data that it describes.  */

struct dcache_block *
dcache_alloc ()
{
  register struct dcache_block *db;

  if ((db = dcache_free.next) == &dcache_free)
    /* If we can't get one from the free list, take last valid */
    db = dcache_valid.last;

  remque (db);
  insque (db, &dcache_valid);
  return (db);
}

/* Return the contents of the word at address ADDR in the remote machine,
   using the data cache.  */

int
dcache_fetch (addr)
     CORE_ADDR addr;
{
  register struct dcache_block *db;

  db = dcache_hit (addr);
  if (db == 0)
    {
      db = dcache_alloc ();
      remote_sl_read_bytes (addr & ~0xf, db->data, 16);
      db->addr = addr & ~0xf;
    }
  return (dcache_value (db, addr));
}

/* Write the word at ADDR both in the data cache and in the remote machine.  */

dcache_poke (addr, data)
     CORE_ADDR addr;
     int data;
{
  register struct dcache_block *db;

  /* First make sure the word is IN the cache.  DB is its cache block.  */
  db = dcache_hit (addr);
  if (db == 0)
    {
      db = dcache_alloc ();
      remote_sl_read_bytes (addr & ~0xf, db->data, 16);
      db->addr = addr & ~0xf;
    }

  /* Modify the word in the cache.  */
  db->data[(addr>>2)&3] = data;

  /* Send the changed word.  */
  remote_sl_write_bytes (addr, &data, 4);
}

/* Initialize the data cache.  */

dcache_init ()
{
  register i;
  register struct dcache_block *db;

  db = (struct dcache_block *) xmalloc (sizeof (struct dcache_block) * 
					DCACHE_SIZE);
  dcache_free.next = dcache_free.last = &dcache_free;
  dcache_valid.next = dcache_valid.last = &dcache_valid;
  for (i=0;i<DCACHE_SIZE;i++,db++)
    insque (db, &dcache_free);
}

#endif /* 0 */

/*
** Stuff for user-variables specific to these targets.
*/
/* Trapped internal variables are used to handle special registers.
   A trapped i.v. calls a hook here every time it is dereferenced,
   to provide a new value for the variable, and it calls a hook here
   when a new value is assigned, to do something with the value. */

/* Return 1 if NAME is a trapped internal variable, else 0. */

int
is_trapped(name)
     char *name;
{
  if (STREQ(name, "thread")) 
    return 1;

  return 0;
}

value_ptr
value_is_trapped(var)
     struct internalvar *var;
{
  register value_ptr val;

  if (TYPE_CODE (VALUE_TYPE (var->value)) != TYPE_CODE_INT)
    val = value_from_longest (builtin_type_unsigned_int, 0);
  else
    {
      /* Copy code from value_of_internalvar () */

      val = value_copy (var->value);
      if (VALUE_LAZY (val))
	value_fetch_lazy (val);
    }

  VALUE_LVAL (val) = lval_internalvar;
  VALUE_INTERNALVAR (val) = var;

  return val;
}

/* Handle a new value assigned to a trapped internal variable.  We probably
   need to flush some state when changing node, task, or thread. */

int
set_trapped(var, val, bitpos, bitsize, offset)
     struct internalvar *var;
     value_ptr val;
     int bitpos, bitsize, offset;
{
  int ret = 0;

  if (TYPE_CODE (VALUE_TYPE (val)) != TYPE_CODE_INT
      || value_as_pointer (val) < 0)
    error ("Unsigned integer type required");

  /* This is copied from values.c:set_internalvar().  We need to update
     the value before fetching new state from the inferior.  I don't know
     whether it is a good idea to update the value twice (once here and
     again after we return. */

  free ((PTR)var->value);
  var->value = value_copy (val);
  /* Force the value to be fetched from the target now, to avoid problems
     later when this internalvar is referenced and the target is gone or
     has changed.  */
  if (VALUE_LAZY (var->value))
    value_fetch_lazy (var->value);
  release_value (var->value);

  purge_gdb_caches();

  return ret;
}


static void
remote_sl_kill ()
{
  /* For some mysterious reason, wait_for_inferior calls kill instead of
     mourn after it gets TARGET_WAITKIND_SIGNALLED.  Work around it.  */
  if (kill_kludge)
    {
      kill_kludge = 0;
      target_mourn_inferior ();
      return;
    }

  /* Use catch_errors so the user can quit from gdb even when we aren't on
     speaking terms with the remote system.  */
  catch_errors (putpkt, "k", "", RETURN_MASK_ERROR);

  /* Don't wait for it to die.  I'm not really sure it matters whether
     we do or not.  For the existing stubs, kill is a noop.  */
  target_mourn_inferior ();
}

static void
remote_sl_mourn ()
{
  remote_sl_mourn_1 (&remote_sl_ops);
}

static void
extended_remote_sl_mourn ()
{
  /* We do _not_ want to mourn the target like this; this will
     remove the extended remote target  from the target stack,
     and the next time the user says "run" it'll fail. 

     FIXME: What is the right thing to do here?  */
#if 0
  remote_sl_mourn_1 (&extended_remote_sl_ops);
#endif
}

/* Worker function for remote_mourn.  */
static void
remote_sl_mourn_1 (target)
     struct target_ops *target;
{
  unpush_target (target);
  generic_mourn_inferior ();
}

/* In the extended protocol we want to be able to do things like
   "run" and have them basically work as expected.  So we need
   a special create_inferior function. 

   FIXME: One day add support for changing the exec file
   we're debugging, arguments and an environment.  */

static void
extended_remote_sl_create_inferior (exec_file, args, env)
     char *exec_file;
     char *args;
     char **env;
{
  /* Rip out the breakpoints; we'll reinsert them after restarting
     the remote server.  */
  remove_breakpoints ();

  /* Now restart the remote server.  */
  extended_remote_sl_restart ();

  /* Now put the breakpoints back in.  This way we're safe if the
     restart function works via a unix fork on the remote side.  */
  insert_breakpoints ();

  /* Clean up from the last time we were running.  */
  clear_proceed_status ();

  /* Let the remote process run.  */
  proceed (-1, TARGET_SIGNAL_0, 0);
}


#ifdef REMOTE_BREAKPOINT

/* On some machines, e.g. 68k, we may use a different breakpoint instruction
   than other targets.  */
static unsigned char break_insn[] = REMOTE_BREAKPOINT;

#else /* No REMOTE_BREAKPOINT.  */

/* Same old breakpoint instruction.  This code does nothing different
   than mem-break.c.  */
static unsigned char break_insn[] = BREAKPOINT;

#endif /* No REMOTE_BREAKPOINT.  */

/* Insert a breakpoint on targets that don't have any better breakpoint
   support.  We read the contents of the target location and stash it,
   then overwrite it with a breakpoint instruction.  ADDR is the target
   location in the target machine.  CONTENTS_CACHE is a pointer to 
   memory allocated for saving the target contents.  It is guaranteed
   by the caller to be long enough to save sizeof BREAKPOINT bytes (this
   is accomplished via BREAKPOINT_MAX).  */

static int
remote_sl_insert_breakpoint (addr, contents_cache)
     CORE_ADDR addr;
     char *contents_cache;
{
  int val;

  val = target_read_memory (addr, contents_cache, sizeof break_insn);

  if (val == 0)
    val = target_write_memory (addr, (char *)break_insn, sizeof break_insn);

  return val;
}

static int
remote_sl_remove_breakpoint (addr, contents_cache)
     CORE_ADDR addr;
     char *contents_cache;
{
  return target_write_memory (addr, contents_cache, sizeof break_insn);
}


/* Define the target subroutine names */

struct target_ops remote_sl_ops = {
  "remote-serial",			/* to_shortname */
  "Remote serial target in gdb-specific protocol",	/* to_longname */
  "Use a remote computer via a serial line, using a gdb-specific protocol.\n\
Specify the serial device it is connected to (e.g. /dev/ttya).",  /* to_doc */
  remote_sl_open,			/* to_open */
  remote_sl_close,			/* to_close */
  NULL,				/* to_attach */
  remote_sl_detach,		/* to_detach */
  remote_sl_resume,		/* to_resume */
  remote_sl_wait,			/* to_wait */
  remote_sl_fetch_registers,	/* to_fetch_registers */
  remote_sl_store_registers,	/* to_store_registers */
  remote_sl_prepare_to_store,	/* to_prepare_to_store */
  remote_sl_xfer_memory,		/* to_xfer_memory */
  remote_sl_files_info,		/* to_files_info */

  remote_sl_insert_breakpoint,	/* to_insert_breakpoint */
  remote_sl_remove_breakpoint,	/* to_remove_breakpoint */

  NULL,				/* to_terminal_init */
  NULL,				/* to_terminal_inferior */
  NULL,				/* to_terminal_ours_for_output */
  NULL,				/* to_terminal_ours */
  NULL,				/* to_terminal_info */
  remote_sl_kill,			/* to_kill */
  generic_load,			/* to_load */
  NULL,				/* to_lookup_symbol */
  NULL,				/* to_create_inferior */
  remote_sl_mourn,			/* to_mourn_inferior */
  0,				/* to_can_run */
  0,				/* to_notice_signals */
  remote_sl_thread_alive,          /* to_thread_alive */
  0,                          /* to_stop */
  process_stratum,		/* to_stratum */
  NULL,				/* to_next */
  1,				/* to_has_all_memory */
  1,				/* to_has_memory */
  1,				/* to_has_stack */
  1,				/* to_has_registers */
  1,				/* to_has_execution */
  NULL,				/* sections */
  NULL,				/* sections_end */
  /* target operations added by OSF */ 
  is_trapped,                   /* to_is_trapped */ 
  set_trapped,                  /* to_set_trapped*/ 
  value_is_trapped,             /* to_get_trapped */
  0,                            /* to_protocol_version */
  OPS_MAGIC			/* to_magic */
};

struct target_ops extended_remote_sl_ops = {
  "extended-remote-serial",			/* to_shortname */
  "Extended remote serial target in gdb-specific protocol",/* to_longname */
  "Use a remote computer via a serial line, using a gdb-specific protocol.\n\
Specify the serial device it is connected to (e.g. /dev/ttya).",  /* to_doc */
  extended_remote_sl_open,			/* to_open */
  remote_sl_close,			/* to_close */
  NULL,				/* to_attach */
  remote_sl_detach,		/* to_detach */
  remote_sl_resume,		/* to_resume */
  remote_sl_wait,			/* to_wait */
  remote_sl_fetch_registers,	/* to_fetch_registers */
  remote_sl_store_registers,	/* to_store_registers */
  remote_sl_prepare_to_store,	/* to_prepare_to_store */
  remote_sl_xfer_memory,		/* to_xfer_memory */
  remote_sl_files_info,		/* to_files_info */
  remote_sl_insert_breakpoint,	/* to_insert_breakpoint */
  remote_sl_remove_breakpoint,	/* to_remove_breakpoint */

  NULL,				/* to_terminal_init */
  NULL,				/* to_terminal_inferior */
  NULL,				/* to_terminal_ours_for_output */
  NULL,				/* to_terminal_ours */
  NULL,				/* to_terminal_info */
  remote_sl_kill,			/* to_kill */
  generic_load,			/* to_load */
  NULL,				/* to_lookup_symbol */
  extended_remote_sl_create_inferior,/* to_create_inferior */
  extended_remote_sl_mourn,	/* to_mourn_inferior */
  0,				/* to_can_run */
  0,				/* to_notice_signals */
  remote_sl_thread_alive,		/* to_thread_alive */
  0,				/* to_stop */
  process_stratum,		/* to_stratum */
  NULL,				/* to_next */
  1,				/* to_has_all_memory */
  1,				/* to_has_memory */
  1,				/* to_has_stack */
  1,				/* to_has_registers */
  1,				/* to_has_execution */
  NULL,				/* sections */
  NULL,				/* sections_end */
  /* target operations added by OSF */ 
  is_trapped,                   /* to_is_trapped */ 
  set_trapped,                  /* to_set_trapped*/ 
  value_is_trapped,             /* to_get_trapped */
  0,                            /* to_protocol_version */
  OPS_MAGIC			/* to_magic */
};

void
_initialize_remote_sl ()
{
  add_target (&remote_sl_ops);

  add_show_from_set (
    add_set_cmd ("remotedebug", no_class, var_boolean, (char *)&remote_debug,
		   "Set debugging of remote serial I/O.\n\
When enabled, each packet sent or received with the remote target\n\
is displayed.", &setlist),
	&showlist);

#if 1 /* added: gdb-4.16 */
  add_show_from_set (add_set_cmd ("remotetimeout", no_class,
				  var_integer, (char *)&remote_sl_timeout,
				  "Set timeout value for remote read.\n", &setlist),
		     &showlist);

  add_show_from_set (add_set_cmd ("remotebreak", no_class,
				  var_integer, (char *)&remote_sl_break,
				  "Set whether to send break if interrupted.\n", &setlist),
		     &showlist);
#endif
  
  ptr_pid_to_str = &normal_pid_to_str;
}

/* #endif *NY* 03.07.96 */

/*
 * @OSF_COPYRIGHT@
 * 
 */
/*
 * HISTORY
 * $Log: remote-kgdb.c,v $
 * Revision 1.1.2.2  1995/09/08  17:55:42  duthie
 * 	Add some RETRY code that can be turned on with
 * 	REMOTE_WITH_RETRY.  Make this the default for now in
 * 	remote.mk.
 * 	[1995/09/08  17:50:00  duthie]
 *
 * Revision 1.1.2.1  1995/05/12  19:39:07  duthie
 * 	New support for remote kernel debugging.
 * 	[1995/05/12  19:38:51  duthie]
 * 
 * 	define boolean_t for hpux.
 * 	[1995/01/26  16:53:05  duthie]
 * 
 * 	Change ioctl mechanism to more portable xpg4 flavor of
 * 	tcsetattr(), cfsetispeed(),cfsetospeed() functions.
 * 	New file for debugging remote osf kernels.
 * 	Fix termio for HP.   Fix endianness problems.
 * 	Make 9600 baud the default baud rate.
 * 	[1995/01/25  18:36:07  duthie]
 * 
 * 	Fixed remote interrupt support.
 * 	[1994/08/05  10:55:47  rod]
 * 
 * 	Call usleep while polling so we don't hog the local CPU.
 * 	[1994/06/24  19:33:56  rod]
 * 
 * 	Remote interrupt support.
 * 	[1994/06/24  16:32:19  rod]
 * 
 * 	Fix up fatal error processing.
 * 	[1994/05/09  20:08:51  rod]
 * 
 * 	Cloned from remote.c for trusted Mach remote kernel debugging.
 * 	[1994/05/06  14:13:36  rod]
 * 
 * $EndLog$
 */
