/* OSF extension - added from Intel sources */
/* Intel Paragon target interface, for GDB when running under OSF/1 AD.
   Copyright (C) 1986, 1987, 1989, 1991 Free Software Foundation, Inc.

This file is part of GDB.

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */

#include "defs.h"
#include "frame.h"
#include "dis-asm.h"


/* debug prints */
#if 0
#define IDEBUG(x) printf x
#else
#define IDEBUG(x)
#endif

/* return values */
#define OK (0)
#define FAILED (-1)

/* 
 * Duplicated from breakpoint.c because (at least for now) this is a
 * machine dependent data structure.
 */
static char break_insn[] = BREAKPOINT;

/*
 *  Breakpoint structure for single stepping
 */
static struct {
    CORE_ADDR     address;
    unsigned long shadow_contents;
} step_bkpt;

/* Non-zero if we just simulated a single-step ptrace call.  This is
   needed because we cannot remove the breakpoints in the inferior
   process until after the `wait' in `wait_for_inferior'.  Used for
   i860. */

int one_stepped=0;


/*************************************************************************
 **  This file contains i860 dependent routines. These are purposely
 **  simple-minded for now. For example, we do not worry about when the
 **  floating-point half of a dual-instruction-mode instruction gets
 **  executed when there is a breakpoint on that instruction. In general,
 **  if your program has been optimized that much things like line numbers
 **  and symbolic memory access will not work correctly anyway. KISS for now.
 *************************************************************************/

/*****************************************************************************
 *                      SINGLE-STEP ROUTINES
 ****************************************************************************/

/*
 *  Macros for disassembly in single-step routines
 *  See i860 Programmer's Reference Manual instruction format
 */

#define CC_set    (read_register(PS_REGNUM) & 0x00000004)
#define LCC_set   (read_register(PS_REGNUM) & 0x00000008)
#define DIM_set   (read_register(PS_REGNUM) & 0x00004000)
#define DS_set    (read_register(PS_REGNUM) & 0x00002000)
#define BRX(x)    ((long)((x) << 6) >> 4)
#define RVAL(x)   (read_register(x))
#define SRC1(x)   (((unsigned long)(x) >> 11) & 0x1f)
#define SRC2(x)   (((unsigned long)(x) >> 21) & 0x1f)
#define DEST(x)   (((unsigned long)(x) >> 16) & 0x1f)
#define I(x)      ((x) & 0x04000000)
#define OFFSET(x) ((((x) >> 5) & 0x0000f800) | ((x) & 0x000007ff))
#define SX(x)     (((x) & 0x8000) ? (x) | 0xffff0000 : (x))


/************************************************************************
 *
 *      Calling Sequence:
 *            transfer_inst( caddr, istransfer, next, step_over )
 *
 *      Description:
 *            Check if current instruction is a transfer intruction
 *
 *      Parameters:
 *            caddr:      - Address of current instruction. If in dual mode, 
 *                          then addr points to floating point instruction 
 *                          of the pair.
 *            istransfer: - Pointer to boolean return; TRUE if next instruction
 *                          is a transfer instruction.
 *            next:       - Pointer to target address of transfer instruction.
 *                          address of next instruction to be executed.
 *            step_over:  - Flag denoting step over calls, for calculating
 *                          target address.
 *
 *      Returns:
 *            OK on success, FAILED otherwise.
 *
 ************************************************************************/

int
transfer_inst ( caddr, istransfer, next, step_over)
	CORE_ADDR         caddr;
	boolean_t	       *istransfer;
	CORE_ADDR        *next;
	boolean_t         step_over;
{
	CORE_ADDR         addr;
	CORE_ADDR         inst;
	long            val;
	int             dual;		  /* next addr to put breakpoint in
					   * normal command */
	int             dual_t;		  /* next addr to put breakpoint in
					   * 'taken' commands */
	int             addr_inc;	  /* adjustment to core part of inst
					   * in dual mode */
        int             status;

	/*
	 *  get current value in addr 
	 */
	addr = caddr;
	if (DIM_set)
		addr += 4;

	if ( (status = read_memory( addr, 
                                    (char *) &inst,
                                    sizeof( break_insn )) ) != 0 ) {
	    return status;
	}

	/* 
	 *  dual mode pairs float and core instruction
	 *  traps can be set only on the core instruction 
	 */
	if (DIM_set)
		if (DS_set) {
			dual = 8;
			dual_t = 12;
			addr_inc = 0;
		} else {
			dual = 12;
			dual_t = 20;
			addr_inc = 4;
		}
	else if (DS_set) {
		dual = 8;
		dual_t = 16;
		addr_inc = 4;
	} else {
		dual = 4;
		dual_t = 8;
		addr_inc = 0;
	}

	/* 
	 *  check for CTRL format instruction 
	 */
	if ((inst & 0xe0000000) == 0x60000000) {


		addr_inc = BRX (inst);
#if !OLD_BRADDR
		if (DIM_set)
			addr = caddr + 8;
		else
			addr = caddr + 4;
#endif

		switch (((int) (inst >> 16) >> 10) & 7) {
		    case 2:		  /* br   */
			*next = addr + addr_inc;
			*istransfer = TRUE;
			return OK;

		    case 3:		  /* call */
			if (!step_over) {
				*next = addr + addr_inc;
			} else
				*next = caddr + dual_t;
			*istransfer = TRUE;
			return OK;

		    case 4:		  /* bc   */
			if (CC_set)
				*next = addr + addr_inc;
			else
				*next = caddr + dual;
			*istransfer = TRUE;
			return OK;

		    case 6:		  /* bnc  */
			if (!CC_set)
				*next = addr + addr_inc;
			else
				*next = caddr + dual;
			*istransfer = TRUE;
			return OK;

		    case 5:		  /* bc.t */
			if (CC_set) {
				*next = addr + addr_inc;
#if 0
				if (addr == caddr) {
					*next = caddr + dual_t;
				}
#endif
			} else
				*next = caddr + dual_t;
			*istransfer = TRUE;
			return OK;

		    case 7:		  /* bnc.t */
			if (!CC_set) {
				*next = addr + addr_inc;
#if 0
				if (addr == caddr) {
					*next = caddr + dual_t;
				}
#endif
			} else
				*next = caddr + dual_t;
			*istransfer = TRUE;
			return OK;
		}
	}

	/* 
	 *  check for calli instruction 
	 */
	if ((inst & 0xfc00001f) == 0x4c000002) {
		addr = RVAL (SRC1 (inst));
#if OLD_BRADDR
		addr |= 0xf0000000;
#endif
		if (!step_over) {
			*next = addr + addr_inc;
		} else
			*next = caddr + dual_t;
		*istransfer = TRUE;
		return OK;
	}

	/* 
	 *  check for bri instruction 
	 */
	if ((inst & 0xfc000000) == 0x40000000) {
		long            cur_psr;

		addr = RVAL (SRC1 (inst));

#if OLD_BRADDR
		addr |= 0xf0000000;
#endif
		/*donb cur_psr = app_ps->upsr;*/
		cur_psr = read_register(PS_REGNUM);
		if ((cur_psr & 0x00001f00) == 0)	/* check trap bits */
			*next = addr + addr_inc;	/* no trap bits set */
		else if (((cur_psr & 0x00004000) != 0) &&
			 ((cur_psr & 0x00002000) == 0))	/* check DIM bit */
			*next = addr + 4; /* DIM bit is set */
		else
			*next = addr;	  /* DIM bit is clear */


		*istransfer = TRUE;
		return OK;
	}

	/* 
	 *  check for bte & btne instructions 
	 */
	if ((inst & 0xf0000000) == 0x50000000) {
		long            cmp;

		if (I (inst))		  /* immediate flag set */
			cmp = SRC1 (inst);
		else
			cmp = RVAL (SRC1 (inst));

		if (DIM_set)
			*next = caddr + 8;
		else
			*next = caddr + 4;

		val = RVAL (SRC2 (inst));
		if (inst & 0x08000000) {  /* check if bte */
			if (val == cmp)
				*next += (SX (OFFSET (inst)) << 2) + addr_inc;
			else
				*next = caddr + dual;
		} else {		  /* instruction is btne */
			if (val != cmp)
				*next += (SX (OFFSET (inst)) << 2) + addr_inc;
			else
				*next = caddr + dual;
		}
		*istransfer = TRUE;
		return OK;
	}

	/* 
	 *  check for bla instruction 
	 */
	if ((inst & 0xfc000000) == 0xb4000000) {
		if (DIM_set)
			*next = caddr + 8;
		else
			*next = caddr + 4;
		if (LCC_set) {
			addr = *next + (SX (OFFSET (inst)) << 2);
			*next = addr + addr_inc;
			if (addr == caddr) {
				*next = caddr + dual_t;
			}
		} else
			*next = caddr + dual_t;
		*istransfer = TRUE;
		return OK;
	}

	/* 
	 *  the instruction is not a transfer instruction 
	 */
	*istransfer = FALSE;
	return OK;

      error_exit:
	return FAILED;
}

/************************************************************************
 *
 *      Calling Sequence:
 *            next_inst (addr, step_over, next_addr)
 *
 *      Description:
 *            Find address of next instruction to be executed. This
 *            routine is from the i860 simulator adb.
 *
 *      Parameters:
 *            addr:           Address of current instruction
 *            step_over:      Flag which tells to skip over procedures.
 *            next_addr:      Ptr to address of next instruction.
 *
 *      Returns:
 *            OK on success, FAILED otherwise.
 *
 ************************************************************************/

int
next_inst( addr, step_over, next_addr )
	CORE_ADDR         addr;
	boolean_t         step_over;
	CORE_ADDR         *next_addr;
{
	boolean_t		istransfer;

	/* 
	 *  check if transfer instruction 
	 */
	if ( transfer_inst( addr, 
                           &istransfer, 
                            next_addr, 
                            step_over ) == FAILED ) {
	    goto error_exit;
	}

	if (istransfer)
	    return OK;

	/* 
	 *  Non-transfer: next sequential instruction to be executed 
	 */
	if (DIM_set) {
		if (DS_set) {
			*next_addr = addr + 8;
		} else {
			*next_addr = addr + 12;
		}
	} else if (DS_set) {
		*next_addr = addr + 8;
	} else {
		*next_addr = addr + 4;
	}

	return OK;

      error_exit:
	return FAILED;
}

/************************************************************************
 *
 *      Calling Sequence:
 *            single_step( signal )
 *
 *      Description:
 *	      Step "one instruction".
 *
 *	Parameters:
 *	      signal:         - Signal to pass to the process
 *
 *	Returns:
 *            0 if instruction stepped with no errors, errno otherwise.
 *
 ************************************************************************/


int 
single_step( signal )
    int signal;
{
    CORE_ADDR          pc;	/* current program counter   */
    CORE_ADDR          next_pc;	/* where to set next ss bkpt */
    int                status;

IDEBUG(( "single_step( signal = %d )\n", signal ));

    pc = read_register( PC_REGNUM );

    if ( !one_stepped ) {
IDEBUG(( "single_step: !one_stepped\n" ));
        /*
         *  Get the address of the next trappable instruction.
         *  (not fp half of dual mode, not shadow of delayed branch,
         *  target of branch and call restrictions accounted for)
         */
        if ( next_inst( pc,
    		        FALSE,  /* Do not step over calls */
    		       &next_pc ) == -1 ) {
            goto error_exit;
        }
IDEBUG(( "single_step: pc 0x%08lx, next_pc 0x%08lx\n", pc, next_pc ));

        /* Set the breakpoint */
        step_bkpt.address = next_pc;
        if ((status = read_memory( step_bkpt.address, 
                                   &step_bkpt.shadow_contents, 
                                   sizeof( break_insn ))) != 0 ) {
             return status;
        }
IDEBUG(( "single_step: read instr 0x%08lx\n", step_bkpt.shadow_contents ));
        if ((status = write_memory( step_bkpt.address, 
                                    break_insn, 
                                    sizeof( break_insn ))) != 0 ) {
             return status;
        }
IDEBUG(( "single_step: planted breakpoint\n" ));

        /* Will let it go in the caller */
        one_stepped = 1;

    } else {  /* one_stepped TRUE, just remove old single-step breakpoint */
IDEBUG(( "single_step: one_stepped: remove old ss bkpt\n" ));

        if ( (status = write_memory( step_bkpt.address, 
                                     &step_bkpt.shadow_contents, 
                                     sizeof( break_insn ) )) != 0 ) {
            return status;
        }
        one_stepped = 0;
    }

    return 0;

  error_exit:
    return status;
}

/*************************************************************************
/* Written for i860 by Jim Hanko (hanko@orc.olivetti.com) */
/* This code was based on SPARC code written by Gary Beihl (beihl@mcc.com),
 *  by Michael Tiemann (tiemann@corto.inria.fr). 
 */

struct command_line *get_breakpoint_commands ();

CORE_ADDR 
i860_skip_prologue (pc)
     CORE_ADDR pc;
{
  long instr;
  int regno;

IDEBUG(( "skip_prologue( %08lx )\n", pc ));

  instr = read_memory_integer (pc, 4);

  /* Recognize "addu|adds -X,sp,sp" insn. */
  if ((instr & 0xEFFF0000) == 0x84420000)
    {
      pc += 4;
      instr = read_memory_integer (pc, 4);
    }
  else
    return(pc);					/* No frame! */

  /* Recognize store of return addr and frame pointer into frame */
  while (1)
    {
      if ((instr & 0xFFE0F801) == 0x1C400801 ||  /* st.l r1,X(sp) */
          (instr & 0xFFE0F801) == 0x1C401801)    /* st.l fp,X(sp) */
        {
          pc += 4;
          instr = read_memory_integer (pc, 4);
        }
      else
 	break;
    }

  /* Recognize "addu|adds X,sp,fp" insn. */
  if ((instr & 0xEFFF0000) == 0x84430000)
    {
      pc += 4;
      instr = read_memory_integer (pc, 4);
    }

  /* Now recognize stores into the frame from the registers. */

  while (1)
    {
      if ((instr & 0xFFA00003) == 0x1C200001 ||	/* st.l rn,X(fp|sp) */
          (instr & 0xFFA00001) == 0x4C200000)	/* fst.y fn,X(fp|sp) */
        {
	  regno = (instr >> 11) & 0x1f;
	  if (regno == 0)			/* source reg == 0? quit */
	    break;
          pc += 4;
          instr = read_memory_integer (pc, 4);
        }
      else
        break;
    }

  return(pc);
}
/* set in call_function() [valops.c] to the address of the "call dummy" code
   so dummy frames can be easily recognized; also used in wait_for_inferior() 
   [infrun.c]. When not used, it points into the ABI's 'reserved area' */

CORE_ADDR call_dummy_set = 0;	/* true if dummy call being done */
CORE_ADDR call_dummy_start;	/* address of call dummy code */

i860_frame_find_saved_regs(frame_info, frame_saved_regs)
    struct frame_info *frame_info;
    struct frame_saved_regs *frame_saved_regs;
{
  register CORE_ADDR pc;
  long instr, spdelta = 0, offset;
  int i, size, reg;
  int r1_off = -1, fp_off = -1;
  int framesize;

IDEBUG(( "frame_find_saved_regs( %08lx, %08lx )\n", frame_info, frame_saved_regs ));

  bzero (frame_saved_regs, sizeof(*frame_saved_regs));

  if (call_dummy_set && frame_info->pc >= call_dummy_start && 
	frame_info->pc <= call_dummy_start + CALL_DUMMY_LENGTH)
    {
      /* DUMMY frame - all registers stored in order at fp; old sp is
	 at fp + NUM_REGS*4 */
      for (i = 1; i < NUM_REGS; i++) /* skip reg 0 */ {
if (i == FP_REGNUM)
	if (i != SP_REGNUM && i != F0_REGNUM && i != F0_REGNUM + 1)
	  frame_saved_regs->regs[i] = frame_info->frame + i*4;
      }
      frame_saved_regs->regs[SP_REGNUM] = frame_info->frame + NUM_REGS*4; 
      call_dummy_set = 0;

      return;
    }

  pc = get_pc_function_start (frame_info->pc); 

  instr = read_memory_integer (pc, 4);
  /* Recognize "addu|adds -X,sp,sp" insn. */
  if ((instr & 0xEFFF0000) == 0x84420000)
    {
      framesize = -SIGN_EXT16(instr & 0x0000FFFF);
      pc += 4;
      instr = read_memory_integer (pc, 4);
    }
  else
    goto punt;					/* No frame! */

  /* Recognize store of return addr and frame pointer into frame */
  while (1)
    {
      if ((instr & 0xFFE0F801) == 0x1C400801)  /* st.l r1,X(sp) */
        {
	  r1_off = SIGN_EXT16(((instr&0x001F0000) >> 5) | (instr&0x000007FE));
          pc += 4;
          instr = read_memory_integer (pc, 4);
        }
      else if ((instr & 0xFFE0F801) == 0x1C401801)    /* st.l fp,X(sp) */
        {
	  fp_off = SIGN_EXT16(((instr&0x001F0000) >> 5) | (instr&0x000007FE));
          pc += 4;
          instr = read_memory_integer (pc, 4);
        }
      else
 	break;
    }

  /* Recognize "addu|adds X,sp,fp" insn. */
  if ((instr & 0xEFFF0000) == 0x84430000)
    {
      spdelta = SIGN_EXT16(instr & 0x0000FFFF);
      pc += 4;
      instr = read_memory_integer (pc, 4);
    }

  /* Now recognize stores into the frame from the registers. */

  while (1)
    {
      if ((instr & 0xFFC00003) == 0x1C400001)	/* st.l rn,X(fp|sp) */
        {
	  offset = SIGN_EXT16(((instr&0x001F0000) >> 5) | (instr&0x000007FE));
	  reg = (instr >> 11) & 0x1F;
	  if (reg == 0)
	    break;
	  if ((instr & 0x00200000) == 0)	/* was this using sp? */
	    if (spdelta)			/* and we know sp-fp delta */
	      offset -= spdelta;		/* if so, adjust the offset */
	    else
	      break;				/* if not, give up */


	  /* Handle the case where the return address is stored after the fp 
	     is adjusted */

	  if (reg == 1)
	    frame_saved_regs->regs[PC_REGNUM] = frame_info->frame + offset;
	  else
	    frame_saved_regs->regs[reg] = frame_info->frame + offset;

          pc += 4;
          instr = read_memory_integer (pc, 4);
        }
      else if ((instr & 0xFFC00001) == 0x2C400000) /* fst.y fn,X(fp|sp) */
        {
	  /*
	   * The number of words in a floating store based on 3 LSB of instr
	   */
	  static int fst_sizes[] = {2, 0, 1, 0, 4, 0, 1, 0};

	  size = fst_sizes[instr & 7];
	  reg = ((instr >> 16) & 0x1F) + F0_REGNUM;
	  if (reg == 0)
	    break;

	  if (size > 1)					/* align the offset */
	    offset = SIGN_EXT16(instr & 0x0000FFF8);	/* drop 3 bits */
	  else
	    offset = SIGN_EXT16(instr & 0x0000FFFC);	/* drop 2 bits */

	  if ((instr & 0x00200000) == 0)	/* was this using sp? */
	    if (spdelta)			/* and we know sp-fp delta */
	      offset -= spdelta;		/* if so, adjust the offset */
	    else
	      break;				/* if not, give up */

	  for (i = 0; i < size; i++)
	    {
	      frame_saved_regs->regs[reg] = frame_info->frame + offset;

	      offset += 4;
	      reg++;
	    }

          pc += 4;
          instr = read_memory_integer (pc, 4);
        }
      else
        break;
    }

punt: ;
  if (framesize != 0 && spdelta != 0)
    frame_saved_regs->regs[SP_REGNUM] = frame_info->frame+(framesize-spdelta);
  else
    frame_saved_regs->regs[SP_REGNUM] = frame_info->frame + 8;

  if (spdelta && fp_off != -1)
    frame_saved_regs->regs[FP_REGNUM] = frame_info->frame - spdelta + fp_off;
  else
    frame_saved_regs->regs[FP_REGNUM] = frame_info->frame;

  if (spdelta && r1_off != -1)
    frame_saved_regs->regs[PC_REGNUM] = frame_info->frame - spdelta + r1_off;
  else
    frame_saved_regs->regs[PC_REGNUM] = frame_info->frame + 4;
}


void
_initialize_i860_tdep ()
{
  tm_print_insn = print_insn_i860;
}
