/* Machine-dependent functions for OSF Mach 3.0 on i386.

   Copyright (C) 1996 Free Software Foundation, Inc.

This file is part of GDB.

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */

#include "defs.h"
#include "inferior.h"
#include <assert.h>
#include <mach_error.h>

#define x(regname) offsetof(struct i386_thread_state, regname)

/* Offsets in struct i386_thread_state corresponding to the non-FP registers. */
static int registermap[] = {
    x(eax), x(ecx), x(edx), x(ebx), x(uesp), x(ebp), x(esi), x(edi),
    x(eip), x(efl), x(cs), x(ss), x(ds), x(es), x(fs), x(gs),
};

/* Magic definitions that should be tm-i386.h.  */
#define LAST_INT_REGNUM 15
#define LAST_FLOAT_REGNUM 23
#define env387_reg(n) (28 + (n) * 10)

int
i386_thread_state_offset(regno)
    int regno;
{
    if (regno < FP0_REGNUM)
	return registermap[regno];
    return offsetof(struct i386_float_state, hw_state) +
	   env387_reg(regno - FP0_REGNUM);
}

void
i386_supply_thread_state_registers(flavor, state)
    thread_state_flavor_t flavor;
    thread_state_t state;
{
    int i, off;

    switch (flavor) {
    case i386_THREAD_STATE:
	for (i = 0; i <= LAST_INT_REGNUM; i++) {
	    off = registermap[i];
	    supply_register(i, ((char *) state) + off);
	}
	break;
    case i386_FLOAT_STATE:
	for (i = FP0_REGNUM; i <= LAST_FLOAT_REGNUM; i++) {
	    off = i386_thread_state_offset(i);
	    supply_register(i, ((char *) state) + off);
	}
	break;
    default:
	printf("i386_supply_thread_state_registers\n");
	abort();
    }
}

/* Construct a thread state of the appropriate flavor from the registers[]
   array if all the contained registers are valid, and return 1 on success.
   Otherwise (some of the registers are invalid) return 0.  */
int
i386_make_thread_state(flavor, state)
    thread_state_flavor_t flavor;
    thread_state_t state;
{
    int i, off;

    switch (flavor) {
    case i386_THREAD_STATE:
	for (i = 0; i <= LAST_INT_REGNUM; i++) {
	    if (!register_valid[i])
		return 0;
	    off = registermap[i];
	    memcpy(((char *) state) + off, &registers[REGISTER_BYTE(i)],
		   REGISTER_RAW_SIZE(i));
	}
	return 1;
    case i386_FLOAT_STATE:
	/* This state contains lots of information apart from the values
	   of the floating-point registers, but that other information is
	   not contained in registers[], so we can never do this.  */
	return 0;
    default:
	printf("i386_make_thread_state\n");
	abort();
    }
}

void
i386_store_all_registers()
{
    thread_state_data_t state;
    int i, off;
    kern_return_t ret;
    mach_msg_type_number_t count;

    if (!i386_make_thread_state(i386_THREAD_STATE, state)) {
	printf("i386_store_all_registers\n");
	abort();
    }
    ret = thread_set_state(current_thread, i386_THREAD_STATE,
			   (thread_state_t) state, i386_THREAD_STATE_COUNT);
    if (ret != KERN_SUCCESS) {
machfail:
	warning("i386_store_all_registers: %s", mach_error_string(ret));
	return;
    }

    /* Set up the i386_float_state structure.  Not all FP state is contained
       in registers[] so we must obtain the old state before modifying.  */
    count = i386_FLOAT_STATE_COUNT;
    ret = thread_get_state(current_thread, i386_FLOAT_STATE,
			   (thread_state_t) state, &count);
    if (ret != KERN_SUCCESS)
	goto machfail;
    for (i = FP0_REGNUM; i <= LAST_FLOAT_REGNUM; i++) {
	off = i386_thread_state_offset(i);
	memcpy ((char *) state + off, &registers[REGISTER_BYTE (i)],
		REGISTER_RAW_SIZE (i));
    }
    ret = thread_set_state(current_thread, i386_FLOAT_STATE,
			   (thread_state_t) state, count);
    if (ret != KERN_SUCCESS)
	goto machfail;
}

/* We could just copy this out of i386m3-nat.c, but the resultant code
   duplication makes me queasy.  So leave it out until we do it properly.  */
void
i386_mach3_float_info()
{
    error ("`info float' is not yet implemented on this platform.");
}
