#pragma once

/**
 * \file
 * Contains declarations of all functions and types which represent a public TDLib interface.
 */
#include "td/tl/TlObject.h"

#include <string>

#include <cstdint>
#include <utility>
#include <vector>

namespace td {
class TlStorerToString;

namespace td_api {

/**
 * This type is used to store 32-bit signed integers, which can be represented as Number in JSON.
 */
using int32 = std::int32_t;
/**
 * This type is used to store 53-bit signed integers, which can be represented as Number in JSON.
 */
using int53 = std::int64_t;
/**
 * This type is used to store 64-bit signed integers, which can't be represented as Number in JSON and are represented as String instead.
 */
using int64 = std::int64_t;

/**
 * This type is used to store UTF-8 strings.
 */
using string = std::string;

/**
 * This type is used to store arbitrary sequences of bytes. In JSON interface the bytes are base64-encoded.
 */
using bytes = std::string;

/**
 * This type is used to store a list of objects of any type and is represented as Array in JSON.
 */
template <class Type>
using array = std::vector<Type>;

/**
 * This class is a base class for all TDLib API classes and functions.
 */
using BaseObject = ::td::TlObject;

/**
 * A smart wrapper to store a pointer to a TDLib API object. Can be treated as an analogue of std::unique_ptr.
 */
template <class Type>
using object_ptr = ::td::tl_object_ptr<Type>;

/**
 * A function to create a dynamically allocated TDLib API object. Can be treated as an analogue of std::make_unique.
 * Usage example:
 * \code
 * auto get_authorization_state_request = td::td_api::make_object<td::td_api::getAuthorizationState>();
 * auto message_text = td::td_api::make_object<td::td_api::formattedText>("Hello, world!!!",
 *                     td::td_api::array<td::td_api::object_ptr<td::td_api::textEntity>>());
 * auto send_message_request = td::td_api::make_object<td::td_api::sendMessage>(chat_id, 0, 0, nullptr, nullptr,
 *      td::td_api::make_object<td::td_api::inputMessageText>(std::move(message_text), false, true));
 * \endcode
 *
 * \tparam Type Type of an object to construct.
 * \param[in] args Arguments to pass to the object constructor.
 * \return Wrapped pointer to the created object.
 */
template <class Type, class... Args>
object_ptr<Type> make_object(Args &&... args) {
  return object_ptr<Type>(new Type(std::forward<Args>(args)...));
}

/**
 * A function to cast a wrapped in td::td_api::object_ptr TDLib API object to its subclass or superclass.
 * Casting an object to an incorrect type will lead to undefined behaviour.
 * Usage example:
 * \code
 * td::td_api::object_ptr<td::td_api::callState> call_state = ...;
 * switch (call_state->get_id()) {
 *   case td::td_api::callStatePending::ID: {
 *     auto state = td::td_api::move_object_as<td::td_api::callStatePending>(call_state);
 *     // use state
 *     break;
 *   }
 *   case td::td_api::callStateExchangingKeys::ID: {
 *     // no additional fields, no casting is needed
 *     break;
 *   }
 *   case td::td_api::callStateReady::ID: {
 *     auto state = td::td_api::move_object_as<td::td_api::callStateReady>(call_state);
 *     // use state
 *     break;
 *   }
 *   case td::td_api::callStateHangingUp::ID: {
 *     // no additional fields, no casting is needed
 *     break;
 *   }
 *   case td::td_api::callStateDiscarded::ID: {
 *     auto state = td::td_api::move_object_as<td::td_api::callStateDiscarded>(call_state);
 *     // use state
 *     break;
 *   }
 *   case td::td_api::callStateError::ID: {
 *     auto state = td::td_api::move_object_as<td::td_api::callStateError>(call_state);
 *     // use state
 *     break;
 *   }
 *   default:
 *     assert(false);
 * }
 * \endcode
 *
 * \tparam ToType Type of a TDLib API object to move to.
 * \tparam FromType Type of a TDLib API object to move from, this is auto-deduced.
 * \param[in] from Wrapped in td::td_api::object_ptr pointer to a TDLib API object.
 */
template <class ToType, class FromType>
object_ptr<ToType> move_object_as(FromType &&from) {
  return object_ptr<ToType>(static_cast<ToType *>(from.release()));
}

/**
 * Returns a string representation of a TDLib API object.
 * \param[in] value The object.
 * \return Object string representation.
 */
std::string to_string(const BaseObject &value);

/**
 * Returns a string representation of a TDLib API object.
 * \tparam T Object type, auto-deduced.
 * \param[in] value The object.
 * \return Object string representation.
 */
template <class T>
std::string to_string(const object_ptr<T> &value) {
  if (value == nullptr) {
    return "null";
  }

  return to_string(*value);
}

/**
 * Returns a string representation of a list of TDLib API objects.
 * \tparam T Object type, auto-deduced.
 * \param[in] values The objects.
 * \return Objects string representation.
 */
template <class T>
std::string to_string(const std::vector<object_ptr<T>> &values) {
  std::string result = "{\n";
  for (const auto &value : values) {
    if (value == nullptr) {
      result += "null\n";
    } else {
      result += to_string(*value);
    }
  }
  result += "}\n";
  return result;
}

class accountTtl;

class address;

class animatedChatPhoto;

class animatedEmoji;

class animation;

class animations;

class audio;

class authenticationCodeInfo;

class AuthenticationCodeType;

class AuthorizationState;

class autoDownloadSettings;

class autoDownloadSettingsPresets;

class background;

class BackgroundFill;

class BackgroundType;

class backgrounds;

class bankCardActionOpenUrl;

class bankCardInfo;

class basicGroup;

class basicGroupFullInfo;

class botCommand;

class BotCommandScope;

class botCommands;

class call;

class CallDiscardReason;

class callId;

class CallProblem;

class callProtocol;

class callServer;

class CallServerType;

class CallState;

class callbackQueryAnswer;

class CallbackQueryPayload;

class CanTransferOwnershipResult;

class chat;

class ChatAction;

class ChatActionBar;

class chatAdministrator;

class chatAdministrators;

class chatEvent;

class ChatEventAction;

class chatEventLogFilters;

class chatEvents;

class chatFilter;

class chatFilterInfo;

class chatInviteLink;

class chatInviteLinkCount;

class chatInviteLinkCounts;

class chatInviteLinkInfo;

class chatInviteLinkMember;

class chatInviteLinkMembers;

class chatInviteLinks;

class chatJoinRequest;

class chatJoinRequests;

class chatJoinRequestsInfo;

class ChatList;

class chatLists;

class chatLocation;

class chatMember;

class ChatMemberStatus;

class chatMembers;

class ChatMembersFilter;

class chatNearby;

class chatNotificationSettings;

class chatPermissions;

class chatPhoto;

class chatPhotoInfo;

class chatPhotos;

class chatPosition;

class ChatReportReason;

class ChatSource;

class ChatStatistics;

class chatStatisticsAdministratorActionsInfo;

class chatStatisticsInviterInfo;

class chatStatisticsMessageInteractionInfo;

class chatStatisticsMessageSenderInfo;

class chatTheme;

class ChatType;

class chats;

class chatsNearby;

class CheckChatUsernameResult;

class CheckStickerSetNameResult;

class closedVectorPath;

class connectedWebsite;

class connectedWebsites;

class ConnectionState;

class contact;

class count;

class countries;

class countryInfo;

class customRequestResult;

class databaseStatistics;

class date;

class dateRange;

class datedFile;

class deepLinkInfo;

class DeviceToken;

class DiceStickers;

class document;

class draftMessage;

class emailAddressAuthenticationCodeInfo;

class emojis;

class encryptedCredentials;

class encryptedPassportElement;

class error;

class file;

class filePart;

class FileType;

class formattedText;

class foundMessages;

class game;

class gameHighScore;

class gameHighScores;

class groupCall;

class groupCallId;

class groupCallParticipant;

class groupCallParticipantVideoInfo;

class groupCallRecentSpeaker;

class GroupCallVideoQuality;

class groupCallVideoSourceGroup;

class hashtags;

class httpUrl;

class identityDocument;

class importedContacts;

class inlineKeyboardButton;

class InlineKeyboardButtonType;

class InlineQueryResult;

class inlineQueryResults;

class InputBackground;

class InputChatPhoto;

class InputCredentials;

class InputFile;

class inputIdentityDocument;

class InputInlineQueryResult;

class InputMessageContent;

class InputPassportElement;

class inputPassportElementError;

class InputPassportElementErrorSource;

class inputPersonalDocument;

class InputSticker;

class inputThumbnail;

class InternalLinkType;

class invoice;

class jsonObjectMember;

class JsonValue;

class keyboardButton;

class KeyboardButtonType;

class labeledPricePart;

class languagePackInfo;

class languagePackString;

class LanguagePackStringValue;

class languagePackStrings;

class localFile;

class localizationTargetInfo;

class location;

class LogStream;

class logTags;

class logVerbosityLevel;

class LoginUrlInfo;

class MaskPoint;

class maskPosition;

class message;

class messageCalendar;

class messageCalendarDay;

class MessageContent;

class messageCopyOptions;

class MessageFileType;

class messageForwardInfo;

class MessageForwardOrigin;

class messageInteractionInfo;

class messageLink;

class messageLinkInfo;

class messagePosition;

class messagePositions;

class messageReplyInfo;

class MessageSchedulingState;

class messageSendOptions;

class MessageSender;

class messageSenders;

class MessageSendingState;

class messageStatistics;

class messageThreadInfo;

class messages;

class minithumbnail;

class networkStatistics;

class NetworkStatisticsEntry;

class NetworkType;

class notification;

class notificationGroup;

class NotificationGroupType;

class NotificationSettingsScope;

class NotificationType;

class ok;

class OptionValue;

class orderInfo;

class PageBlock;

class pageBlockCaption;

class PageBlockHorizontalAlignment;

class pageBlockListItem;

class pageBlockRelatedArticle;

class pageBlockTableCell;

class PageBlockVerticalAlignment;

class passportAuthorizationForm;

class PassportElement;

class passportElementError;

class PassportElementErrorSource;

class PassportElementType;

class passportElements;

class passportElementsWithErrors;

class passportRequiredElement;

class passportSuitableElement;

class passwordState;

class paymentForm;

class paymentFormTheme;

class paymentReceipt;

class paymentResult;

class paymentsProviderStripe;

class personalDetails;

class personalDocument;

class phoneNumberAuthenticationSettings;

class phoneNumberInfo;

class photo;

class photoSize;

class point;

class poll;

class pollOption;

class PollType;

class profilePhoto;

class proxies;

class proxy;

class ProxyType;

class PublicChatType;

class PushMessageContent;

class pushReceiverId;

class recommendedChatFilter;

class recommendedChatFilters;

class recoveryEmailAddress;

class remoteFile;

class ReplyMarkup;

class ResetPasswordResult;

class RichText;

class savedCredentials;

class scopeNotificationSettings;

class SearchMessagesFilter;

class seconds;

class secretChat;

class SecretChatState;

class session;

class sessions;

class shippingOption;

class sponsoredMessage;

class StatisticalGraph;

class statisticalValue;

class sticker;

class stickerSet;

class stickerSetInfo;

class stickerSets;

class stickers;

class storageStatistics;

class storageStatisticsByChat;

class storageStatisticsByFileType;

class storageStatisticsFast;

class SuggestedAction;

class supergroup;

class supergroupFullInfo;

class SupergroupMembersFilter;

class tMeUrl;

class TMeUrlType;

class tMeUrls;

class tdlibParameters;

class temporaryPasswordState;

class termsOfService;

class testBytes;

class testInt;

class testString;

class testVectorInt;

class testVectorIntObject;

class testVectorString;

class testVectorStringObject;

class text;

class textEntities;

class textEntity;

class TextEntityType;

class TextParseMode;

class themeSettings;

class thumbnail;

class ThumbnailFormat;

class TopChatCategory;

class Update;

class updates;

class user;

class userFullInfo;

class UserPrivacySetting;

class UserPrivacySettingRule;

class userPrivacySettingRules;

class UserStatus;

class UserType;

class users;

class validatedOrderInfo;

class VectorPathCommand;

class venue;

class video;

class videoChat;

class videoNote;

class voiceNote;

class webPage;

class webPageInstantView;

class Object;

/**
 * This class is a base class for all TDLib API classes.
 */
class Object: public TlObject {
 public:
};

/**
 * This class is a base class for all TDLib API functions.
 */
class Function: public TlObject {
 public:
};

/**
 * Contains information about the period of inactivity after which the current user&\#039;s account will automatically be deleted.
 */
class accountTtl final : public Object {
 public:
  /// Number of days of inactivity before the account will be flagged for deletion; 30-366 days.
  int32 days_;

  /**
   * Contains information about the period of inactivity after which the current user&\#039;s account will automatically be deleted.
   */
  accountTtl();

  /**
   * Contains information about the period of inactivity after which the current user&\#039;s account will automatically be deleted.
   *
   * \param[in] days_ Number of days of inactivity before the account will be flagged for deletion; 30-366 days.
   */
  explicit accountTtl(int32 days_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1324495492;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Describes an address.
 */
class address final : public Object {
 public:
  /// A two-letter ISO 3166-1 alpha-2 country code.
  string country_code_;
  /// State, if applicable.
  string state_;
  /// City.
  string city_;
  /// First line of the address.
  string street_line1_;
  /// Second line of the address.
  string street_line2_;
  /// Address postal code.
  string postal_code_;

  /**
   * Describes an address.
   */
  address();

  /**
   * Describes an address.
   *
   * \param[in] country_code_ A two-letter ISO 3166-1 alpha-2 country code.
   * \param[in] state_ State, if applicable.
   * \param[in] city_ City.
   * \param[in] street_line1_ First line of the address.
   * \param[in] street_line2_ Second line of the address.
   * \param[in] postal_code_ Address postal code.
   */
  address(string const &country_code_, string const &state_, string const &city_, string const &street_line1_, string const &street_line2_, string const &postal_code_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -2043654342;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Animated variant of a chat photo in MPEG4 format.
 */
class animatedChatPhoto final : public Object {
 public:
  /// Animation width and height.
  int32 length_;
  /// Information about the animation file.
  object_ptr<file> file_;
  /// Timestamp of the frame, used as a static chat photo.
  double main_frame_timestamp_;

  /**
   * Animated variant of a chat photo in MPEG4 format.
   */
  animatedChatPhoto();

  /**
   * Animated variant of a chat photo in MPEG4 format.
   *
   * \param[in] length_ Animation width and height.
   * \param[in] file_ Information about the animation file.
   * \param[in] main_frame_timestamp_ Timestamp of the frame, used as a static chat photo.
   */
  animatedChatPhoto(int32 length_, object_ptr<file> &&file_, double main_frame_timestamp_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 191994926;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Describes an animated representation of an emoji.
 */
class animatedEmoji final : public Object {
 public:
  /// Animated sticker for the emoji.
  object_ptr<sticker> sticker_;
  /// Emoji modifier fitzpatrick type; 0-6; 0 if none.
  int32 fitzpatrick_type_;
  /// File containing the sound to be played when the animated emoji is clicked if any; may be null. The sound is encoded with the Opus codec, and stored inside an OGG container.
  object_ptr<file> sound_;

  /**
   * Describes an animated representation of an emoji.
   */
  animatedEmoji();

  /**
   * Describes an animated representation of an emoji.
   *
   * \param[in] sticker_ Animated sticker for the emoji.
   * \param[in] fitzpatrick_type_ Emoji modifier fitzpatrick type; 0-6; 0 if none.
   * \param[in] sound_ File containing the sound to be played when the animated emoji is clicked if any; may be null. The sound is encoded with the Opus codec, and stored inside an OGG container.
   */
  animatedEmoji(object_ptr<sticker> &&sticker_, int32 fitzpatrick_type_, object_ptr<file> &&sound_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1816658231;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Describes an animation file. The animation must be encoded in GIF or MPEG4 format.
 */
class animation final : public Object {
 public:
  /// Duration of the animation, in seconds; as defined by the sender.
  int32 duration_;
  /// Width of the animation.
  int32 width_;
  /// Height of the animation.
  int32 height_;
  /// Original name of the file; as defined by the sender.
  string file_name_;
  /// MIME type of the file, usually &quot;image/gif&quot; or &quot;video/mp4&quot;.
  string mime_type_;
  /// True, if stickers were added to the animation. The list of corresponding sticker set can be received using getAttachedStickerSets.
  bool has_stickers_;
  /// Animation minithumbnail; may be null.
  object_ptr<minithumbnail> minithumbnail_;
  /// Animation thumbnail in JPEG or MPEG4 format; may be null.
  object_ptr<thumbnail> thumbnail_;
  /// File containing the animation.
  object_ptr<file> animation_;

  /**
   * Describes an animation file. The animation must be encoded in GIF or MPEG4 format.
   */
  animation();

  /**
   * Describes an animation file. The animation must be encoded in GIF or MPEG4 format.
   *
   * \param[in] duration_ Duration of the animation, in seconds; as defined by the sender.
   * \param[in] width_ Width of the animation.
   * \param[in] height_ Height of the animation.
   * \param[in] file_name_ Original name of the file; as defined by the sender.
   * \param[in] mime_type_ MIME type of the file, usually &quot;image/gif&quot; or &quot;video/mp4&quot;.
   * \param[in] has_stickers_ True, if stickers were added to the animation. The list of corresponding sticker set can be received using getAttachedStickerSets.
   * \param[in] minithumbnail_ Animation minithumbnail; may be null.
   * \param[in] thumbnail_ Animation thumbnail in JPEG or MPEG4 format; may be null.
   * \param[in] animation_ File containing the animation.
   */
  animation(int32 duration_, int32 width_, int32 height_, string const &file_name_, string const &mime_type_, bool has_stickers_, object_ptr<minithumbnail> &&minithumbnail_, object_ptr<thumbnail> &&thumbnail_, object_ptr<file> &&animation_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -872359106;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Represents a list of animations.
 */
class animations final : public Object {
 public:
  /// List of animations.
  array<object_ptr<animation>> animations_;

  /**
   * Represents a list of animations.
   */
  animations();

  /**
   * Represents a list of animations.
   *
   * \param[in] animations_ List of animations.
   */
  explicit animations(array<object_ptr<animation>> &&animations_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 344216945;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Describes an audio file. Audio is usually in MP3 or M4A format.
 */
class audio final : public Object {
 public:
  /// Duration of the audio, in seconds; as defined by the sender.
  int32 duration_;
  /// Title of the audio; as defined by the sender.
  string title_;
  /// Performer of the audio; as defined by the sender.
  string performer_;
  /// Original name of the file; as defined by the sender.
  string file_name_;
  /// The MIME type of the file; as defined by the sender.
  string mime_type_;
  /// The minithumbnail of the album cover; may be null.
  object_ptr<minithumbnail> album_cover_minithumbnail_;
  /// The thumbnail of the album cover in JPEG format; as defined by the sender. The full size thumbnail is supposed to be extracted from the downloaded file; may be null.
  object_ptr<thumbnail> album_cover_thumbnail_;
  /// File containing the audio.
  object_ptr<file> audio_;

  /**
   * Describes an audio file. Audio is usually in MP3 or M4A format.
   */
  audio();

  /**
   * Describes an audio file. Audio is usually in MP3 or M4A format.
   *
   * \param[in] duration_ Duration of the audio, in seconds; as defined by the sender.
   * \param[in] title_ Title of the audio; as defined by the sender.
   * \param[in] performer_ Performer of the audio; as defined by the sender.
   * \param[in] file_name_ Original name of the file; as defined by the sender.
   * \param[in] mime_type_ The MIME type of the file; as defined by the sender.
   * \param[in] album_cover_minithumbnail_ The minithumbnail of the album cover; may be null.
   * \param[in] album_cover_thumbnail_ The thumbnail of the album cover in JPEG format; as defined by the sender. The full size thumbnail is supposed to be extracted from the downloaded file; may be null.
   * \param[in] audio_ File containing the audio.
   */
  audio(int32 duration_, string const &title_, string const &performer_, string const &file_name_, string const &mime_type_, object_ptr<minithumbnail> &&album_cover_minithumbnail_, object_ptr<thumbnail> &&album_cover_thumbnail_, object_ptr<file> &&audio_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1179334690;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Information about the authentication code that was sent.
 */
class authenticationCodeInfo final : public Object {
 public:
  /// A phone number that is being authenticated.
  string phone_number_;
  /// The way the code was sent to the user.
  object_ptr<AuthenticationCodeType> type_;
  /// The way the next code will be sent to the user; may be null.
  object_ptr<AuthenticationCodeType> next_type_;
  /// Timeout before the code can be re-sent, in seconds.
  int32 timeout_;

  /**
   * Information about the authentication code that was sent.
   */
  authenticationCodeInfo();

  /**
   * Information about the authentication code that was sent.
   *
   * \param[in] phone_number_ A phone number that is being authenticated.
   * \param[in] type_ The way the code was sent to the user.
   * \param[in] next_type_ The way the next code will be sent to the user; may be null.
   * \param[in] timeout_ Timeout before the code can be re-sent, in seconds.
   */
  authenticationCodeInfo(string const &phone_number_, object_ptr<AuthenticationCodeType> &&type_, object_ptr<AuthenticationCodeType> &&next_type_, int32 timeout_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -860345416;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Provides information about the method by which an authentication code is delivered to the user.
 */
class AuthenticationCodeType: public Object {
 public:
};

/**
 * An authentication code is delivered via a private Telegram message, which can be viewed from another active session.
 */
class authenticationCodeTypeTelegramMessage final : public AuthenticationCodeType {
 public:
  /// Length of the code.
  int32 length_;

  /**
   * An authentication code is delivered via a private Telegram message, which can be viewed from another active session.
   */
  authenticationCodeTypeTelegramMessage();

  /**
   * An authentication code is delivered via a private Telegram message, which can be viewed from another active session.
   *
   * \param[in] length_ Length of the code.
   */
  explicit authenticationCodeTypeTelegramMessage(int32 length_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 2079628074;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * An authentication code is delivered via an SMS message to the specified phone number.
 */
class authenticationCodeTypeSms final : public AuthenticationCodeType {
 public:
  /// Length of the code.
  int32 length_;

  /**
   * An authentication code is delivered via an SMS message to the specified phone number.
   */
  authenticationCodeTypeSms();

  /**
   * An authentication code is delivered via an SMS message to the specified phone number.
   *
   * \param[in] length_ Length of the code.
   */
  explicit authenticationCodeTypeSms(int32 length_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 962650760;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * An authentication code is delivered via a phone call to the specified phone number.
 */
class authenticationCodeTypeCall final : public AuthenticationCodeType {
 public:
  /// Length of the code.
  int32 length_;

  /**
   * An authentication code is delivered via a phone call to the specified phone number.
   */
  authenticationCodeTypeCall();

  /**
   * An authentication code is delivered via a phone call to the specified phone number.
   *
   * \param[in] length_ Length of the code.
   */
  explicit authenticationCodeTypeCall(int32 length_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1636265063;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * An authentication code is delivered by an immediately canceled call to the specified phone number. The phone number that calls is the code that must be entered automatically.
 */
class authenticationCodeTypeFlashCall final : public AuthenticationCodeType {
 public:
  /// Pattern of the phone number from which the call will be made.
  string pattern_;

  /**
   * An authentication code is delivered by an immediately canceled call to the specified phone number. The phone number that calls is the code that must be entered automatically.
   */
  authenticationCodeTypeFlashCall();

  /**
   * An authentication code is delivered by an immediately canceled call to the specified phone number. The phone number that calls is the code that must be entered automatically.
   *
   * \param[in] pattern_ Pattern of the phone number from which the call will be made.
   */
  explicit authenticationCodeTypeFlashCall(string const &pattern_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1395882402;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * An authentication code is delivered by an immediately canceled call to the specified phone number. The last digits of the phone number that calls are the code that must be entered manually by the user.
 */
class authenticationCodeTypeMissedCall final : public AuthenticationCodeType {
 public:
  /// Prefix of the phone number from which the call will be made.
  string phone_number_prefix_;
  /// Number of digits in the code, excluding the prefix.
  int32 length_;

  /**
   * An authentication code is delivered by an immediately canceled call to the specified phone number. The last digits of the phone number that calls are the code that must be entered manually by the user.
   */
  authenticationCodeTypeMissedCall();

  /**
   * An authentication code is delivered by an immediately canceled call to the specified phone number. The last digits of the phone number that calls are the code that must be entered manually by the user.
   *
   * \param[in] phone_number_prefix_ Prefix of the phone number from which the call will be made.
   * \param[in] length_ Number of digits in the code, excluding the prefix.
   */
  authenticationCodeTypeMissedCall(string const &phone_number_prefix_, int32 length_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 700123783;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Represents the current authorization state of the TDLib client.
 */
class AuthorizationState: public Object {
 public:
};

/**
 * TDLib needs TdlibParameters for initialization.
 */
class authorizationStateWaitTdlibParameters final : public AuthorizationState {
 public:

  /**
   * TDLib needs TdlibParameters for initialization.
   */
  authorizationStateWaitTdlibParameters();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 904720988;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * TDLib needs an encryption key to decrypt the local database.
 */
class authorizationStateWaitEncryptionKey final : public AuthorizationState {
 public:
  /// True, if the database is currently encrypted.
  bool is_encrypted_;

  /**
   * TDLib needs an encryption key to decrypt the local database.
   */
  authorizationStateWaitEncryptionKey();

  /**
   * TDLib needs an encryption key to decrypt the local database.
   *
   * \param[in] is_encrypted_ True, if the database is currently encrypted.
   */
  explicit authorizationStateWaitEncryptionKey(bool is_encrypted_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 612103496;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * TDLib needs the user&\#039;s phone number to authorize. Call `setAuthenticationPhoneNumber` to provide the phone number, or use `requestQrCodeAuthentication`, or `checkAuthenticationBotToken` for other authentication options.
 */
class authorizationStateWaitPhoneNumber final : public AuthorizationState {
 public:

  /**
   * TDLib needs the user&\#039;s phone number to authorize. Call `setAuthenticationPhoneNumber` to provide the phone number, or use `requestQrCodeAuthentication`, or `checkAuthenticationBotToken` for other authentication options.
   */
  authorizationStateWaitPhoneNumber();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 306402531;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * TDLib needs the user&\#039;s authentication code to authorize.
 */
class authorizationStateWaitCode final : public AuthorizationState {
 public:
  /// Information about the authorization code that was sent.
  object_ptr<authenticationCodeInfo> code_info_;

  /**
   * TDLib needs the user&\#039;s authentication code to authorize.
   */
  authorizationStateWaitCode();

  /**
   * TDLib needs the user&\#039;s authentication code to authorize.
   *
   * \param[in] code_info_ Information about the authorization code that was sent.
   */
  explicit authorizationStateWaitCode(object_ptr<authenticationCodeInfo> &&code_info_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 52643073;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The user needs to confirm authorization on another logged in device by scanning a QR code with the provided link.
 */
class authorizationStateWaitOtherDeviceConfirmation final : public AuthorizationState {
 public:
  /// A tg:// URL for the QR code. The link will be updated frequently.
  string link_;

  /**
   * The user needs to confirm authorization on another logged in device by scanning a QR code with the provided link.
   */
  authorizationStateWaitOtherDeviceConfirmation();

  /**
   * The user needs to confirm authorization on another logged in device by scanning a QR code with the provided link.
   *
   * \param[in] link_ A tg:// URL for the QR code. The link will be updated frequently.
   */
  explicit authorizationStateWaitOtherDeviceConfirmation(string const &link_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 860166378;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The user is unregistered and need to accept terms of service and enter their first name and last name to finish registration.
 */
class authorizationStateWaitRegistration final : public AuthorizationState {
 public:
  /// Telegram terms of service.
  object_ptr<termsOfService> terms_of_service_;

  /**
   * The user is unregistered and need to accept terms of service and enter their first name and last name to finish registration.
   */
  authorizationStateWaitRegistration();

  /**
   * The user is unregistered and need to accept terms of service and enter their first name and last name to finish registration.
   *
   * \param[in] terms_of_service_ Telegram terms of service.
   */
  explicit authorizationStateWaitRegistration(object_ptr<termsOfService> &&terms_of_service_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 550350511;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The user has been authorized, but needs to enter a password to start using the application.
 */
class authorizationStateWaitPassword final : public AuthorizationState {
 public:
  /// Hint for the password; may be empty.
  string password_hint_;
  /// True, if a recovery email address has been set up.
  bool has_recovery_email_address_;
  /// Pattern of the email address to which the recovery email was sent; empty until a recovery email has been sent.
  string recovery_email_address_pattern_;

  /**
   * The user has been authorized, but needs to enter a password to start using the application.
   */
  authorizationStateWaitPassword();

  /**
   * The user has been authorized, but needs to enter a password to start using the application.
   *
   * \param[in] password_hint_ Hint for the password; may be empty.
   * \param[in] has_recovery_email_address_ True, if a recovery email address has been set up.
   * \param[in] recovery_email_address_pattern_ Pattern of the email address to which the recovery email was sent; empty until a recovery email has been sent.
   */
  authorizationStateWaitPassword(string const &password_hint_, bool has_recovery_email_address_, string const &recovery_email_address_pattern_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 187548796;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The user has been successfully authorized. TDLib is now ready to answer queries.
 */
class authorizationStateReady final : public AuthorizationState {
 public:

  /**
   * The user has been successfully authorized. TDLib is now ready to answer queries.
   */
  authorizationStateReady();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1834871737;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The user is currently logging out.
 */
class authorizationStateLoggingOut final : public AuthorizationState {
 public:

  /**
   * The user is currently logging out.
   */
  authorizationStateLoggingOut();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 154449270;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * TDLib is closing, all subsequent queries will be answered with the error 500. Note that closing TDLib can take a while. All resources will be freed only after authorizationStateClosed has been received.
 */
class authorizationStateClosing final : public AuthorizationState {
 public:

  /**
   * TDLib is closing, all subsequent queries will be answered with the error 500. Note that closing TDLib can take a while. All resources will be freed only after authorizationStateClosed has been received.
   */
  authorizationStateClosing();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 445855311;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * TDLib client is in its final state. All databases are closed and all resources are released. No other updates will be received after this. All queries will be responded to with error code 500. To continue working, one must create a new instance of the TDLib client.
 */
class authorizationStateClosed final : public AuthorizationState {
 public:

  /**
   * TDLib client is in its final state. All databases are closed and all resources are released. No other updates will be received after this. All queries will be responded to with error code 500. To continue working, one must create a new instance of the TDLib client.
   */
  authorizationStateClosed();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1526047584;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains auto-download settings.
 */
class autoDownloadSettings final : public Object {
 public:
  /// True, if the auto-download is enabled.
  bool is_auto_download_enabled_;
  /// The maximum size of a photo file to be auto-downloaded, in bytes.
  int32 max_photo_file_size_;
  /// The maximum size of a video file to be auto-downloaded, in bytes.
  int32 max_video_file_size_;
  /// The maximum size of other file types to be auto-downloaded, in bytes.
  int32 max_other_file_size_;
  /// The maximum suggested bitrate for uploaded videos, in kbit/s.
  int32 video_upload_bitrate_;
  /// True, if the beginning of video files needs to be preloaded for instant playback.
  bool preload_large_videos_;
  /// True, if the next audio track needs to be preloaded while the user is listening to an audio file.
  bool preload_next_audio_;
  /// True, if &quot;use less data for calls&quot; option needs to be enabled.
  bool use_less_data_for_calls_;

  /**
   * Contains auto-download settings.
   */
  autoDownloadSettings();

  /**
   * Contains auto-download settings.
   *
   * \param[in] is_auto_download_enabled_ True, if the auto-download is enabled.
   * \param[in] max_photo_file_size_ The maximum size of a photo file to be auto-downloaded, in bytes.
   * \param[in] max_video_file_size_ The maximum size of a video file to be auto-downloaded, in bytes.
   * \param[in] max_other_file_size_ The maximum size of other file types to be auto-downloaded, in bytes.
   * \param[in] video_upload_bitrate_ The maximum suggested bitrate for uploaded videos, in kbit/s.
   * \param[in] preload_large_videos_ True, if the beginning of video files needs to be preloaded for instant playback.
   * \param[in] preload_next_audio_ True, if the next audio track needs to be preloaded while the user is listening to an audio file.
   * \param[in] use_less_data_for_calls_ True, if &quot;use less data for calls&quot; option needs to be enabled.
   */
  autoDownloadSettings(bool is_auto_download_enabled_, int32 max_photo_file_size_, int32 max_video_file_size_, int32 max_other_file_size_, int32 video_upload_bitrate_, bool preload_large_videos_, bool preload_next_audio_, bool use_less_data_for_calls_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -2144418333;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains auto-download settings presets for the current user.
 */
class autoDownloadSettingsPresets final : public Object {
 public:
  /// Preset with lowest settings; supposed to be used by default when roaming.
  object_ptr<autoDownloadSettings> low_;
  /// Preset with medium settings; supposed to be used by default when using mobile data.
  object_ptr<autoDownloadSettings> medium_;
  /// Preset with highest settings; supposed to be used by default when connected on Wi-Fi.
  object_ptr<autoDownloadSettings> high_;

  /**
   * Contains auto-download settings presets for the current user.
   */
  autoDownloadSettingsPresets();

  /**
   * Contains auto-download settings presets for the current user.
   *
   * \param[in] low_ Preset with lowest settings; supposed to be used by default when roaming.
   * \param[in] medium_ Preset with medium settings; supposed to be used by default when using mobile data.
   * \param[in] high_ Preset with highest settings; supposed to be used by default when connected on Wi-Fi.
   */
  autoDownloadSettingsPresets(object_ptr<autoDownloadSettings> &&low_, object_ptr<autoDownloadSettings> &&medium_, object_ptr<autoDownloadSettings> &&high_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -782099166;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Describes a chat background.
 */
class background final : public Object {
 public:
  /// Unique background identifier.
  int64 id_;
  /// True, if this is one of default backgrounds.
  bool is_default_;
  /// True, if the background is dark and is recommended to be used with dark theme.
  bool is_dark_;
  /// Unique background name.
  string name_;
  /// Document with the background; may be null. Null only for filled backgrounds.
  object_ptr<document> document_;
  /// Type of the background.
  object_ptr<BackgroundType> type_;

  /**
   * Describes a chat background.
   */
  background();

  /**
   * Describes a chat background.
   *
   * \param[in] id_ Unique background identifier.
   * \param[in] is_default_ True, if this is one of default backgrounds.
   * \param[in] is_dark_ True, if the background is dark and is recommended to be used with dark theme.
   * \param[in] name_ Unique background name.
   * \param[in] document_ Document with the background; may be null. Null only for filled backgrounds.
   * \param[in] type_ Type of the background.
   */
  background(int64 id_, bool is_default_, bool is_dark_, string const &name_, object_ptr<document> &&document_, object_ptr<BackgroundType> &&type_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -429971172;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Describes a fill of a background.
 */
class BackgroundFill: public Object {
 public:
};

/**
 * Describes a solid fill of a background.
 */
class backgroundFillSolid final : public BackgroundFill {
 public:
  /// A color of the background in the RGB24 format.
  int32 color_;

  /**
   * Describes a solid fill of a background.
   */
  backgroundFillSolid();

  /**
   * Describes a solid fill of a background.
   *
   * \param[in] color_ A color of the background in the RGB24 format.
   */
  explicit backgroundFillSolid(int32 color_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1010678813;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Describes a gradient fill of a background.
 */
class backgroundFillGradient final : public BackgroundFill {
 public:
  /// A top color of the background in the RGB24 format.
  int32 top_color_;
  /// A bottom color of the background in the RGB24 format.
  int32 bottom_color_;
  /// Clockwise rotation angle of the gradient, in degrees; 0-359. Must be always divisible by 45.
  int32 rotation_angle_;

  /**
   * Describes a gradient fill of a background.
   */
  backgroundFillGradient();

  /**
   * Describes a gradient fill of a background.
   *
   * \param[in] top_color_ A top color of the background in the RGB24 format.
   * \param[in] bottom_color_ A bottom color of the background in the RGB24 format.
   * \param[in] rotation_angle_ Clockwise rotation angle of the gradient, in degrees; 0-359. Must be always divisible by 45.
   */
  backgroundFillGradient(int32 top_color_, int32 bottom_color_, int32 rotation_angle_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1839206017;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Describes a freeform gradient fill of a background.
 */
class backgroundFillFreeformGradient final : public BackgroundFill {
 public:
  /// A list of 3 or 4 colors of the freeform gradients in the RGB24 format.
  array<int32> colors_;

  /**
   * Describes a freeform gradient fill of a background.
   */
  backgroundFillFreeformGradient();

  /**
   * Describes a freeform gradient fill of a background.
   *
   * \param[in] colors_ A list of 3 or 4 colors of the freeform gradients in the RGB24 format.
   */
  explicit backgroundFillFreeformGradient(array<int32> &&colors_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1145469255;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Describes the type of a background.
 */
class BackgroundType: public Object {
 public:
};

/**
 * A wallpaper in JPEG format.
 */
class backgroundTypeWallpaper final : public BackgroundType {
 public:
  /// True, if the wallpaper must be downscaled to fit in 450x450 square and then box-blurred with radius 12.
  bool is_blurred_;
  /// True, if the background needs to be slightly moved when device is tilted.
  bool is_moving_;

  /**
   * A wallpaper in JPEG format.
   */
  backgroundTypeWallpaper();

  /**
   * A wallpaper in JPEG format.
   *
   * \param[in] is_blurred_ True, if the wallpaper must be downscaled to fit in 450x450 square and then box-blurred with radius 12.
   * \param[in] is_moving_ True, if the background needs to be slightly moved when device is tilted.
   */
  backgroundTypeWallpaper(bool is_blurred_, bool is_moving_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1972128891;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A PNG or TGV (gzipped subset of SVG with MIME type &quot;application/x-tgwallpattern&quot;) pattern to be combined with the background fill chosen by the user.
 */
class backgroundTypePattern final : public BackgroundType {
 public:
  /// Fill of the background.
  object_ptr<BackgroundFill> fill_;
  /// Intensity of the pattern when it is shown above the filled background; 0-100.
  int32 intensity_;
  /// True, if the background fill must be applied only to the pattern itself. All other pixels are black in this case. For dark themes only.
  bool is_inverted_;
  /// True, if the background needs to be slightly moved when device is tilted.
  bool is_moving_;

  /**
   * A PNG or TGV (gzipped subset of SVG with MIME type &quot;application/x-tgwallpattern&quot;) pattern to be combined with the background fill chosen by the user.
   */
  backgroundTypePattern();

  /**
   * A PNG or TGV (gzipped subset of SVG with MIME type &quot;application/x-tgwallpattern&quot;) pattern to be combined with the background fill chosen by the user.
   *
   * \param[in] fill_ Fill of the background.
   * \param[in] intensity_ Intensity of the pattern when it is shown above the filled background; 0-100.
   * \param[in] is_inverted_ True, if the background fill must be applied only to the pattern itself. All other pixels are black in this case. For dark themes only.
   * \param[in] is_moving_ True, if the background needs to be slightly moved when device is tilted.
   */
  backgroundTypePattern(object_ptr<BackgroundFill> &&fill_, int32 intensity_, bool is_inverted_, bool is_moving_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1290213117;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A filled background.
 */
class backgroundTypeFill final : public BackgroundType {
 public:
  /// The background fill.
  object_ptr<BackgroundFill> fill_;

  /**
   * A filled background.
   */
  backgroundTypeFill();

  /**
   * A filled background.
   *
   * \param[in] fill_ The background fill.
   */
  explicit backgroundTypeFill(object_ptr<BackgroundFill> &&fill_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 993008684;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains a list of backgrounds.
 */
class backgrounds final : public Object {
 public:
  /// A list of backgrounds.
  array<object_ptr<background>> backgrounds_;

  /**
   * Contains a list of backgrounds.
   */
  backgrounds();

  /**
   * Contains a list of backgrounds.
   *
   * \param[in] backgrounds_ A list of backgrounds.
   */
  explicit backgrounds(array<object_ptr<background>> &&backgrounds_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 724728704;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Describes an action associated with a bank card number.
 */
class bankCardActionOpenUrl final : public Object {
 public:
  /// Action text.
  string text_;
  /// The URL to be opened.
  string url_;

  /**
   * Describes an action associated with a bank card number.
   */
  bankCardActionOpenUrl();

  /**
   * Describes an action associated with a bank card number.
   *
   * \param[in] text_ Action text.
   * \param[in] url_ The URL to be opened.
   */
  bankCardActionOpenUrl(string const &text_, string const &url_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -196454267;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Information about a bank card.
 */
class bankCardInfo final : public Object {
 public:
  /// Title of the bank card description.
  string title_;
  /// Actions that can be done with the bank card number.
  array<object_ptr<bankCardActionOpenUrl>> actions_;

  /**
   * Information about a bank card.
   */
  bankCardInfo();

  /**
   * Information about a bank card.
   *
   * \param[in] title_ Title of the bank card description.
   * \param[in] actions_ Actions that can be done with the bank card number.
   */
  bankCardInfo(string const &title_, array<object_ptr<bankCardActionOpenUrl>> &&actions_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -2116647730;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Represents a basic group of 0-200 users (must be upgraded to a supergroup to accommodate more than 200 users).
 */
class basicGroup final : public Object {
 public:
  /// Group identifier.
  int53 id_;
  /// Number of members in the group.
  int32 member_count_;
  /// Status of the current user in the group.
  object_ptr<ChatMemberStatus> status_;
  /// True, if the group is active.
  bool is_active_;
  /// Identifier of the supergroup to which this group was upgraded; 0 if none.
  int53 upgraded_to_supergroup_id_;

  /**
   * Represents a basic group of 0-200 users (must be upgraded to a supergroup to accommodate more than 200 users).
   */
  basicGroup();

  /**
   * Represents a basic group of 0-200 users (must be upgraded to a supergroup to accommodate more than 200 users).
   *
   * \param[in] id_ Group identifier.
   * \param[in] member_count_ Number of members in the group.
   * \param[in] status_ Status of the current user in the group.
   * \param[in] is_active_ True, if the group is active.
   * \param[in] upgraded_to_supergroup_id_ Identifier of the supergroup to which this group was upgraded; 0 if none.
   */
  basicGroup(int53 id_, int32 member_count_, object_ptr<ChatMemberStatus> &&status_, bool is_active_, int53 upgraded_to_supergroup_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -194767217;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains full information about a basic group.
 */
class basicGroupFullInfo final : public Object {
 public:
  /// Chat photo; may be null.
  object_ptr<chatPhoto> photo_;
  /// Group description. Updated only after the basic group is opened.
  string description_;
  /// User identifier of the creator of the group; 0 if unknown.
  int53 creator_user_id_;
  /// Group members.
  array<object_ptr<chatMember>> members_;
  /// Primary invite link for this group; may be null. For chat administrators with can_invite_users right only. Updated only after the basic group is opened.
  object_ptr<chatInviteLink> invite_link_;
  /// List of commands of bots in the group.
  array<object_ptr<botCommands>> bot_commands_;

  /**
   * Contains full information about a basic group.
   */
  basicGroupFullInfo();

  /**
   * Contains full information about a basic group.
   *
   * \param[in] photo_ Chat photo; may be null.
   * \param[in] description_ Group description. Updated only after the basic group is opened.
   * \param[in] creator_user_id_ User identifier of the creator of the group; 0 if unknown.
   * \param[in] members_ Group members.
   * \param[in] invite_link_ Primary invite link for this group; may be null. For chat administrators with can_invite_users right only. Updated only after the basic group is opened.
   * \param[in] bot_commands_ List of commands of bots in the group.
   */
  basicGroupFullInfo(object_ptr<chatPhoto> &&photo_, string const &description_, int53 creator_user_id_, array<object_ptr<chatMember>> &&members_, object_ptr<chatInviteLink> &&invite_link_, array<object_ptr<botCommands>> &&bot_commands_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 2022233397;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Represents a command supported by a bot.
 */
class botCommand final : public Object {
 public:
  /// Text of the bot command.
  string command_;
  /// Description of the bot command.
  string description_;

  /**
   * Represents a command supported by a bot.
   */
  botCommand();

  /**
   * Represents a command supported by a bot.
   *
   * \param[in] command_ Text of the bot command.
   * \param[in] description_ Description of the bot command.
   */
  botCommand(string const &command_, string const &description_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1032140601;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Represents the scope to which bot commands are relevant.
 */
class BotCommandScope: public Object {
 public:
};

/**
 * A scope covering all users.
 */
class botCommandScopeDefault final : public BotCommandScope {
 public:

  /**
   * A scope covering all users.
   */
  botCommandScopeDefault();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 795652779;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A scope covering all private chats.
 */
class botCommandScopeAllPrivateChats final : public BotCommandScope {
 public:

  /**
   * A scope covering all private chats.
   */
  botCommandScopeAllPrivateChats();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -344889543;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A scope covering all group and supergroup chats.
 */
class botCommandScopeAllGroupChats final : public BotCommandScope {
 public:

  /**
   * A scope covering all group and supergroup chats.
   */
  botCommandScopeAllGroupChats();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -981088162;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A scope covering all group and supergroup chat administrators.
 */
class botCommandScopeAllChatAdministrators final : public BotCommandScope {
 public:

  /**
   * A scope covering all group and supergroup chat administrators.
   */
  botCommandScopeAllChatAdministrators();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1998329169;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A scope covering all members of a chat.
 */
class botCommandScopeChat final : public BotCommandScope {
 public:
  /// Chat identifier.
  int53 chat_id_;

  /**
   * A scope covering all members of a chat.
   */
  botCommandScopeChat();

  /**
   * A scope covering all members of a chat.
   *
   * \param[in] chat_id_ Chat identifier.
   */
  explicit botCommandScopeChat(int53 chat_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -430234971;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A scope covering all administrators of a chat.
 */
class botCommandScopeChatAdministrators final : public BotCommandScope {
 public:
  /// Chat identifier.
  int53 chat_id_;

  /**
   * A scope covering all administrators of a chat.
   */
  botCommandScopeChatAdministrators();

  /**
   * A scope covering all administrators of a chat.
   *
   * \param[in] chat_id_ Chat identifier.
   */
  explicit botCommandScopeChatAdministrators(int53 chat_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1119682126;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A scope covering a member of a chat.
 */
class botCommandScopeChatMember final : public BotCommandScope {
 public:
  /// Chat identifier.
  int53 chat_id_;
  /// User identifier.
  int53 user_id_;

  /**
   * A scope covering a member of a chat.
   */
  botCommandScopeChatMember();

  /**
   * A scope covering a member of a chat.
   *
   * \param[in] chat_id_ Chat identifier.
   * \param[in] user_id_ User identifier.
   */
  botCommandScopeChatMember(int53 chat_id_, int53 user_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -211380494;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains a list of bot commands.
 */
class botCommands final : public Object {
 public:
  /// Bot&\#039;s user identifier.
  int53 bot_user_id_;
  /// List of bot commands.
  array<object_ptr<botCommand>> commands_;

  /**
   * Contains a list of bot commands.
   */
  botCommands();

  /**
   * Contains a list of bot commands.
   *
   * \param[in] bot_user_id_ Bot&\#039;s user identifier.
   * \param[in] commands_ List of bot commands.
   */
  botCommands(int53 bot_user_id_, array<object_ptr<botCommand>> &&commands_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1741364468;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Describes a call.
 */
class call final : public Object {
 public:
  /// Call identifier, not persistent.
  int32 id_;
  /// Peer user identifier.
  int53 user_id_;
  /// True, if the call is outgoing.
  bool is_outgoing_;
  /// True, if the call is a video call.
  bool is_video_;
  /// Call state.
  object_ptr<CallState> state_;

  /**
   * Describes a call.
   */
  call();

  /**
   * Describes a call.
   *
   * \param[in] id_ Call identifier, not persistent.
   * \param[in] user_id_ Peer user identifier.
   * \param[in] is_outgoing_ True, if the call is outgoing.
   * \param[in] is_video_ True, if the call is a video call.
   * \param[in] state_ Call state.
   */
  call(int32 id_, int53 user_id_, bool is_outgoing_, bool is_video_, object_ptr<CallState> &&state_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 920360804;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Describes the reason why a call was discarded.
 */
class CallDiscardReason: public Object {
 public:
};

/**
 * The call wasn&\#039;t discarded, or the reason is unknown.
 */
class callDiscardReasonEmpty final : public CallDiscardReason {
 public:

  /**
   * The call wasn&\#039;t discarded, or the reason is unknown.
   */
  callDiscardReasonEmpty();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1258917949;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The call was ended before the conversation started. It was canceled by the caller or missed by the other party.
 */
class callDiscardReasonMissed final : public CallDiscardReason {
 public:

  /**
   * The call was ended before the conversation started. It was canceled by the caller or missed by the other party.
   */
  callDiscardReasonMissed();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1680358012;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The call was ended before the conversation started. It was declined by the other party.
 */
class callDiscardReasonDeclined final : public CallDiscardReason {
 public:

  /**
   * The call was ended before the conversation started. It was declined by the other party.
   */
  callDiscardReasonDeclined();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1729926094;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The call was ended during the conversation because the users were disconnected.
 */
class callDiscardReasonDisconnected final : public CallDiscardReason {
 public:

  /**
   * The call was ended during the conversation because the users were disconnected.
   */
  callDiscardReasonDisconnected();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1342872670;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The call was ended because one of the parties hung up.
 */
class callDiscardReasonHungUp final : public CallDiscardReason {
 public:

  /**
   * The call was ended because one of the parties hung up.
   */
  callDiscardReasonHungUp();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 438216166;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains the call identifier.
 */
class callId final : public Object {
 public:
  /// Call identifier.
  int32 id_;

  /**
   * Contains the call identifier.
   */
  callId();

  /**
   * Contains the call identifier.
   *
   * \param[in] id_ Call identifier.
   */
  explicit callId(int32 id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 65717769;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Describes the exact type of a problem with a call.
 */
class CallProblem: public Object {
 public:
};

/**
 * The user heard their own voice.
 */
class callProblemEcho final : public CallProblem {
 public:

  /**
   * The user heard their own voice.
   */
  callProblemEcho();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 801116548;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The user heard background noise.
 */
class callProblemNoise final : public CallProblem {
 public:

  /**
   * The user heard background noise.
   */
  callProblemNoise();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1053065359;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The other side kept disappearing.
 */
class callProblemInterruptions final : public CallProblem {
 public:

  /**
   * The other side kept disappearing.
   */
  callProblemInterruptions();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1119493218;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The speech was distorted.
 */
class callProblemDistortedSpeech final : public CallProblem {
 public:

  /**
   * The speech was distorted.
   */
  callProblemDistortedSpeech();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 379960581;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The user couldn&\#039;t hear the other side.
 */
class callProblemSilentLocal final : public CallProblem {
 public:

  /**
   * The user couldn&\#039;t hear the other side.
   */
  callProblemSilentLocal();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 253652790;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The other side couldn&\#039;t hear the user.
 */
class callProblemSilentRemote final : public CallProblem {
 public:

  /**
   * The other side couldn&\#039;t hear the user.
   */
  callProblemSilentRemote();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 573634714;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The call ended unexpectedly.
 */
class callProblemDropped final : public CallProblem {
 public:

  /**
   * The call ended unexpectedly.
   */
  callProblemDropped();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1207311487;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The video was distorted.
 */
class callProblemDistortedVideo final : public CallProblem {
 public:

  /**
   * The video was distorted.
   */
  callProblemDistortedVideo();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 385245706;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The video was pixelated.
 */
class callProblemPixelatedVideo final : public CallProblem {
 public:

  /**
   * The video was pixelated.
   */
  callProblemPixelatedVideo();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 2115315411;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Specifies the supported call protocols.
 */
class callProtocol final : public Object {
 public:
  /// True, if UDP peer-to-peer connections are supported.
  bool udp_p2p_;
  /// True, if connection through UDP reflectors is supported.
  bool udp_reflector_;
  /// The minimum supported API layer; use 65.
  int32 min_layer_;
  /// The maximum supported API layer; use 65.
  int32 max_layer_;
  /// List of supported tgcalls versions.
  array<string> library_versions_;

  /**
   * Specifies the supported call protocols.
   */
  callProtocol();

  /**
   * Specifies the supported call protocols.
   *
   * \param[in] udp_p2p_ True, if UDP peer-to-peer connections are supported.
   * \param[in] udp_reflector_ True, if connection through UDP reflectors is supported.
   * \param[in] min_layer_ The minimum supported API layer; use 65.
   * \param[in] max_layer_ The maximum supported API layer; use 65.
   * \param[in] library_versions_ List of supported tgcalls versions.
   */
  callProtocol(bool udp_p2p_, bool udp_reflector_, int32 min_layer_, int32 max_layer_, array<string> &&library_versions_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1075562897;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Describes a server for relaying call data.
 */
class callServer final : public Object {
 public:
  /// Server identifier.
  int64 id_;
  /// Server IPv4 address.
  string ip_address_;
  /// Server IPv6 address.
  string ipv6_address_;
  /// Server port number.
  int32 port_;
  /// Server type.
  object_ptr<CallServerType> type_;

  /**
   * Describes a server for relaying call data.
   */
  callServer();

  /**
   * Describes a server for relaying call data.
   *
   * \param[in] id_ Server identifier.
   * \param[in] ip_address_ Server IPv4 address.
   * \param[in] ipv6_address_ Server IPv6 address.
   * \param[in] port_ Server port number.
   * \param[in] type_ Server type.
   */
  callServer(int64 id_, string const &ip_address_, string const &ipv6_address_, int32 port_, object_ptr<CallServerType> &&type_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1865932695;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Describes the type of a call server.
 */
class CallServerType: public Object {
 public:
};

/**
 * A Telegram call reflector.
 */
class callServerTypeTelegramReflector final : public CallServerType {
 public:
  /// A peer tag to be used with the reflector.
  bytes peer_tag_;

  /**
   * A Telegram call reflector.
   */
  callServerTypeTelegramReflector();

  /**
   * A Telegram call reflector.
   *
   * \param[in] peer_tag_ A peer tag to be used with the reflector.
   */
  explicit callServerTypeTelegramReflector(bytes const &peer_tag_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1507850700;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A WebRTC server.
 */
class callServerTypeWebrtc final : public CallServerType {
 public:
  /// Username to be used for authentication.
  string username_;
  /// Authentication password.
  string password_;
  /// True, if the server supports TURN.
  bool supports_turn_;
  /// True, if the server supports STUN.
  bool supports_stun_;

  /**
   * A WebRTC server.
   */
  callServerTypeWebrtc();

  /**
   * A WebRTC server.
   *
   * \param[in] username_ Username to be used for authentication.
   * \param[in] password_ Authentication password.
   * \param[in] supports_turn_ True, if the server supports TURN.
   * \param[in] supports_stun_ True, if the server supports STUN.
   */
  callServerTypeWebrtc(string const &username_, string const &password_, bool supports_turn_, bool supports_stun_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1250622821;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Describes the current call state.
 */
class CallState: public Object {
 public:
};

/**
 * The call is pending, waiting to be accepted by a user.
 */
class callStatePending final : public CallState {
 public:
  /// True, if the call has already been created by the server.
  bool is_created_;
  /// True, if the call has already been received by the other party.
  bool is_received_;

  /**
   * The call is pending, waiting to be accepted by a user.
   */
  callStatePending();

  /**
   * The call is pending, waiting to be accepted by a user.
   *
   * \param[in] is_created_ True, if the call has already been created by the server.
   * \param[in] is_received_ True, if the call has already been received by the other party.
   */
  callStatePending(bool is_created_, bool is_received_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1073048620;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The call has been answered and encryption keys are being exchanged.
 */
class callStateExchangingKeys final : public CallState {
 public:

  /**
   * The call has been answered and encryption keys are being exchanged.
   */
  callStateExchangingKeys();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1848149403;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The call is ready to use.
 */
class callStateReady final : public CallState {
 public:
  /// Call protocols supported by the peer.
  object_ptr<callProtocol> protocol_;
  /// List of available call servers.
  array<object_ptr<callServer>> servers_;
  /// A JSON-encoded call config.
  string config_;
  /// Call encryption key.
  bytes encryption_key_;
  /// Encryption key emojis fingerprint.
  array<string> emojis_;
  /// True, if peer-to-peer connection is allowed by users privacy settings.
  bool allow_p2p_;

  /**
   * The call is ready to use.
   */
  callStateReady();

  /**
   * The call is ready to use.
   *
   * \param[in] protocol_ Call protocols supported by the peer.
   * \param[in] servers_ List of available call servers.
   * \param[in] config_ A JSON-encoded call config.
   * \param[in] encryption_key_ Call encryption key.
   * \param[in] emojis_ Encryption key emojis fingerprint.
   * \param[in] allow_p2p_ True, if peer-to-peer connection is allowed by users privacy settings.
   */
  callStateReady(object_ptr<callProtocol> &&protocol_, array<object_ptr<callServer>> &&servers_, string const &config_, bytes const &encryption_key_, array<string> &&emojis_, bool allow_p2p_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -2000107571;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The call is hanging up after discardCall has been called.
 */
class callStateHangingUp final : public CallState {
 public:

  /**
   * The call is hanging up after discardCall has been called.
   */
  callStateHangingUp();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -2133790038;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The call has ended successfully.
 */
class callStateDiscarded final : public CallState {
 public:
  /// The reason, why the call has ended.
  object_ptr<CallDiscardReason> reason_;
  /// True, if the call rating must be sent to the server.
  bool need_rating_;
  /// True, if the call debug information must be sent to the server.
  bool need_debug_information_;

  /**
   * The call has ended successfully.
   */
  callStateDiscarded();

  /**
   * The call has ended successfully.
   *
   * \param[in] reason_ The reason, why the call has ended.
   * \param[in] need_rating_ True, if the call rating must be sent to the server.
   * \param[in] need_debug_information_ True, if the call debug information must be sent to the server.
   */
  callStateDiscarded(object_ptr<CallDiscardReason> &&reason_, bool need_rating_, bool need_debug_information_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -190853167;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The call has ended with an error.
 */
class callStateError final : public CallState {
 public:
  /// Error. An error with the code 4005000 will be returned if an outgoing call is missed because of an expired timeout.
  object_ptr<error> error_;

  /**
   * The call has ended with an error.
   */
  callStateError();

  /**
   * The call has ended with an error.
   *
   * \param[in] error_ Error. An error with the code 4005000 will be returned if an outgoing call is missed because of an expired timeout.
   */
  explicit callStateError(object_ptr<error> &&error_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -975215467;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains a bot&\#039;s answer to a callback query.
 */
class callbackQueryAnswer final : public Object {
 public:
  /// Text of the answer.
  string text_;
  /// True, if an alert must be shown to the user instead of a toast notification.
  bool show_alert_;
  /// URL to be opened.
  string url_;

  /**
   * Contains a bot&\#039;s answer to a callback query.
   */
  callbackQueryAnswer();

  /**
   * Contains a bot&\#039;s answer to a callback query.
   *
   * \param[in] text_ Text of the answer.
   * \param[in] show_alert_ True, if an alert must be shown to the user instead of a toast notification.
   * \param[in] url_ URL to be opened.
   */
  callbackQueryAnswer(string const &text_, bool show_alert_, string const &url_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 360867933;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Represents a payload of a callback query.
 */
class CallbackQueryPayload: public Object {
 public:
};

/**
 * The payload for a general callback button.
 */
class callbackQueryPayloadData final : public CallbackQueryPayload {
 public:
  /// Data that was attached to the callback button.
  bytes data_;

  /**
   * The payload for a general callback button.
   */
  callbackQueryPayloadData();

  /**
   * The payload for a general callback button.
   *
   * \param[in] data_ Data that was attached to the callback button.
   */
  explicit callbackQueryPayloadData(bytes const &data_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1977729946;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The payload for a callback button requiring password.
 */
class callbackQueryPayloadDataWithPassword final : public CallbackQueryPayload {
 public:
  /// The password for the current user.
  string password_;
  /// Data that was attached to the callback button.
  bytes data_;

  /**
   * The payload for a callback button requiring password.
   */
  callbackQueryPayloadDataWithPassword();

  /**
   * The payload for a callback button requiring password.
   *
   * \param[in] password_ The password for the current user.
   * \param[in] data_ Data that was attached to the callback button.
   */
  callbackQueryPayloadDataWithPassword(string const &password_, bytes const &data_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1340266738;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The payload for a game callback button.
 */
class callbackQueryPayloadGame final : public CallbackQueryPayload {
 public:
  /// A short name of the game that was attached to the callback button.
  string game_short_name_;

  /**
   * The payload for a game callback button.
   */
  callbackQueryPayloadGame();

  /**
   * The payload for a game callback button.
   *
   * \param[in] game_short_name_ A short name of the game that was attached to the callback button.
   */
  explicit callbackQueryPayloadGame(string const &game_short_name_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1303571512;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Represents result of checking whether the current session can be used to transfer a chat ownership to another user.
 */
class CanTransferOwnershipResult: public Object {
 public:
};

/**
 * The session can be used.
 */
class canTransferOwnershipResultOk final : public CanTransferOwnershipResult {
 public:

  /**
   * The session can be used.
   */
  canTransferOwnershipResultOk();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -89881021;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The 2-step verification needs to be enabled first.
 */
class canTransferOwnershipResultPasswordNeeded final : public CanTransferOwnershipResult {
 public:

  /**
   * The 2-step verification needs to be enabled first.
   */
  canTransferOwnershipResultPasswordNeeded();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1548372703;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The 2-step verification was enabled recently, user needs to wait.
 */
class canTransferOwnershipResultPasswordTooFresh final : public CanTransferOwnershipResult {
 public:
  /// Time left before the session can be used to transfer ownership of a chat, in seconds.
  int32 retry_after_;

  /**
   * The 2-step verification was enabled recently, user needs to wait.
   */
  canTransferOwnershipResultPasswordTooFresh();

  /**
   * The 2-step verification was enabled recently, user needs to wait.
   *
   * \param[in] retry_after_ Time left before the session can be used to transfer ownership of a chat, in seconds.
   */
  explicit canTransferOwnershipResultPasswordTooFresh(int32 retry_after_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 811440913;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The session was created recently, user needs to wait.
 */
class canTransferOwnershipResultSessionTooFresh final : public CanTransferOwnershipResult {
 public:
  /// Time left before the session can be used to transfer ownership of a chat, in seconds.
  int32 retry_after_;

  /**
   * The session was created recently, user needs to wait.
   */
  canTransferOwnershipResultSessionTooFresh();

  /**
   * The session was created recently, user needs to wait.
   *
   * \param[in] retry_after_ Time left before the session can be used to transfer ownership of a chat, in seconds.
   */
  explicit canTransferOwnershipResultSessionTooFresh(int32 retry_after_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 984664289;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A chat. (Can be a private chat, basic group, supergroup, or secret chat.)
 */
class chat final : public Object {
 public:
  /// Chat unique identifier.
  int53 id_;
  /// Type of the chat.
  object_ptr<ChatType> type_;
  /// Chat title.
  string title_;
  /// Chat photo; may be null.
  object_ptr<chatPhotoInfo> photo_;
  /// Actions that non-administrator chat members are allowed to take in the chat.
  object_ptr<chatPermissions> permissions_;
  /// Last message in the chat; may be null.
  object_ptr<message> last_message_;
  /// Positions of the chat in chat lists.
  array<object_ptr<chatPosition>> positions_;
  /// Identifier of a user or chat that is selected to send messages in the chat; may be null if the user can&\#039;t change message sender.
  object_ptr<MessageSender> message_sender_id_;
  /// True, if chat content can&\#039;t be saved locally, forwarded, or copied.
  bool has_protected_content_;
  /// True, if the chat is marked as unread.
  bool is_marked_as_unread_;
  /// True, if the chat is blocked by the current user and private messages from the chat can&\#039;t be received.
  bool is_blocked_;
  /// True, if the chat has scheduled messages.
  bool has_scheduled_messages_;
  /// True, if the chat messages can be deleted only for the current user while other users will continue to see the messages.
  bool can_be_deleted_only_for_self_;
  /// True, if the chat messages can be deleted for all users.
  bool can_be_deleted_for_all_users_;
  /// True, if the chat can be reported to Telegram moderators through reportChat or reportChatPhoto.
  bool can_be_reported_;
  /// Default value of the disable_notification parameter, used when a message is sent to the chat.
  bool default_disable_notification_;
  /// Number of unread messages in the chat.
  int32 unread_count_;
  /// Identifier of the last read incoming message.
  int53 last_read_inbox_message_id_;
  /// Identifier of the last read outgoing message.
  int53 last_read_outbox_message_id_;
  /// Number of unread messages with a mention/reply in the chat.
  int32 unread_mention_count_;
  /// Notification settings for this chat.
  object_ptr<chatNotificationSettings> notification_settings_;
  /// Current message Time To Live setting (self-destruct timer) for the chat; 0 if not defined. TTL is counted from the time message or its content is viewed in secret chats and from the send date in other chats.
  int32 message_ttl_;
  /// If non-empty, name of a theme, set for the chat.
  string theme_name_;
  /// Information about actions which must be possible to do through the chat action bar; may be null.
  object_ptr<ChatActionBar> action_bar_;
  /// Information about video chat of the chat.
  object_ptr<videoChat> video_chat_;
  /// Information about pending join requests; may be null.
  object_ptr<chatJoinRequestsInfo> pending_join_requests_;
  /// Identifier of the message from which reply markup needs to be used; 0 if there is no default custom reply markup in the chat.
  int53 reply_markup_message_id_;
  /// A draft of a message in the chat; may be null.
  object_ptr<draftMessage> draft_message_;
  /// Application-specific data associated with the chat. (For example, the chat scroll position or local chat notification settings can be stored here.) Persistent if the message database is used.
  string client_data_;

  /**
   * A chat. (Can be a private chat, basic group, supergroup, or secret chat.)
   */
  chat();

  /**
   * A chat. (Can be a private chat, basic group, supergroup, or secret chat.)
   *
   * \param[in] id_ Chat unique identifier.
   * \param[in] type_ Type of the chat.
   * \param[in] title_ Chat title.
   * \param[in] photo_ Chat photo; may be null.
   * \param[in] permissions_ Actions that non-administrator chat members are allowed to take in the chat.
   * \param[in] last_message_ Last message in the chat; may be null.
   * \param[in] positions_ Positions of the chat in chat lists.
   * \param[in] message_sender_id_ Identifier of a user or chat that is selected to send messages in the chat; may be null if the user can&\#039;t change message sender.
   * \param[in] has_protected_content_ True, if chat content can&\#039;t be saved locally, forwarded, or copied.
   * \param[in] is_marked_as_unread_ True, if the chat is marked as unread.
   * \param[in] is_blocked_ True, if the chat is blocked by the current user and private messages from the chat can&\#039;t be received.
   * \param[in] has_scheduled_messages_ True, if the chat has scheduled messages.
   * \param[in] can_be_deleted_only_for_self_ True, if the chat messages can be deleted only for the current user while other users will continue to see the messages.
   * \param[in] can_be_deleted_for_all_users_ True, if the chat messages can be deleted for all users.
   * \param[in] can_be_reported_ True, if the chat can be reported to Telegram moderators through reportChat or reportChatPhoto.
   * \param[in] default_disable_notification_ Default value of the disable_notification parameter, used when a message is sent to the chat.
   * \param[in] unread_count_ Number of unread messages in the chat.
   * \param[in] last_read_inbox_message_id_ Identifier of the last read incoming message.
   * \param[in] last_read_outbox_message_id_ Identifier of the last read outgoing message.
   * \param[in] unread_mention_count_ Number of unread messages with a mention/reply in the chat.
   * \param[in] notification_settings_ Notification settings for this chat.
   * \param[in] message_ttl_ Current message Time To Live setting (self-destruct timer) for the chat; 0 if not defined. TTL is counted from the time message or its content is viewed in secret chats and from the send date in other chats.
   * \param[in] theme_name_ If non-empty, name of a theme, set for the chat.
   * \param[in] action_bar_ Information about actions which must be possible to do through the chat action bar; may be null.
   * \param[in] video_chat_ Information about video chat of the chat.
   * \param[in] pending_join_requests_ Information about pending join requests; may be null.
   * \param[in] reply_markup_message_id_ Identifier of the message from which reply markup needs to be used; 0 if there is no default custom reply markup in the chat.
   * \param[in] draft_message_ A draft of a message in the chat; may be null.
   * \param[in] client_data_ Application-specific data associated with the chat. (For example, the chat scroll position or local chat notification settings can be stored here.) Persistent if the message database is used.
   */
  chat(int53 id_, object_ptr<ChatType> &&type_, string const &title_, object_ptr<chatPhotoInfo> &&photo_, object_ptr<chatPermissions> &&permissions_, object_ptr<message> &&last_message_, array<object_ptr<chatPosition>> &&positions_, object_ptr<MessageSender> &&message_sender_id_, bool has_protected_content_, bool is_marked_as_unread_, bool is_blocked_, bool has_scheduled_messages_, bool can_be_deleted_only_for_self_, bool can_be_deleted_for_all_users_, bool can_be_reported_, bool default_disable_notification_, int32 unread_count_, int53 last_read_inbox_message_id_, int53 last_read_outbox_message_id_, int32 unread_mention_count_, object_ptr<chatNotificationSettings> &&notification_settings_, int32 message_ttl_, string const &theme_name_, object_ptr<ChatActionBar> &&action_bar_, object_ptr<videoChat> &&video_chat_, object_ptr<chatJoinRequestsInfo> &&pending_join_requests_, int53 reply_markup_message_id_, object_ptr<draftMessage> &&draft_message_, string const &client_data_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1617921149;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Describes the different types of activity in a chat.
 */
class ChatAction: public Object {
 public:
};

/**
 * The user is typing a message.
 */
class chatActionTyping final : public ChatAction {
 public:

  /**
   * The user is typing a message.
   */
  chatActionTyping();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 380122167;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The user is recording a video.
 */
class chatActionRecordingVideo final : public ChatAction {
 public:

  /**
   * The user is recording a video.
   */
  chatActionRecordingVideo();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 216553362;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The user is uploading a video.
 */
class chatActionUploadingVideo final : public ChatAction {
 public:
  /// Upload progress, as a percentage.
  int32 progress_;

  /**
   * The user is uploading a video.
   */
  chatActionUploadingVideo();

  /**
   * The user is uploading a video.
   *
   * \param[in] progress_ Upload progress, as a percentage.
   */
  explicit chatActionUploadingVideo(int32 progress_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1234185270;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The user is recording a voice note.
 */
class chatActionRecordingVoiceNote final : public ChatAction {
 public:

  /**
   * The user is recording a voice note.
   */
  chatActionRecordingVoiceNote();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -808850058;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The user is uploading a voice note.
 */
class chatActionUploadingVoiceNote final : public ChatAction {
 public:
  /// Upload progress, as a percentage.
  int32 progress_;

  /**
   * The user is uploading a voice note.
   */
  chatActionUploadingVoiceNote();

  /**
   * The user is uploading a voice note.
   *
   * \param[in] progress_ Upload progress, as a percentage.
   */
  explicit chatActionUploadingVoiceNote(int32 progress_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -613643666;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The user is uploading a photo.
 */
class chatActionUploadingPhoto final : public ChatAction {
 public:
  /// Upload progress, as a percentage.
  int32 progress_;

  /**
   * The user is uploading a photo.
   */
  chatActionUploadingPhoto();

  /**
   * The user is uploading a photo.
   *
   * \param[in] progress_ Upload progress, as a percentage.
   */
  explicit chatActionUploadingPhoto(int32 progress_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 654240583;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The user is uploading a document.
 */
class chatActionUploadingDocument final : public ChatAction {
 public:
  /// Upload progress, as a percentage.
  int32 progress_;

  /**
   * The user is uploading a document.
   */
  chatActionUploadingDocument();

  /**
   * The user is uploading a document.
   *
   * \param[in] progress_ Upload progress, as a percentage.
   */
  explicit chatActionUploadingDocument(int32 progress_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 167884362;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The user is picking a sticker to send.
 */
class chatActionChoosingSticker final : public ChatAction {
 public:

  /**
   * The user is picking a sticker to send.
   */
  chatActionChoosingSticker();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 372753697;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The user is picking a location or venue to send.
 */
class chatActionChoosingLocation final : public ChatAction {
 public:

  /**
   * The user is picking a location or venue to send.
   */
  chatActionChoosingLocation();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -2017893596;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The user is picking a contact to send.
 */
class chatActionChoosingContact final : public ChatAction {
 public:

  /**
   * The user is picking a contact to send.
   */
  chatActionChoosingContact();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1222507496;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The user has started to play a game.
 */
class chatActionStartPlayingGame final : public ChatAction {
 public:

  /**
   * The user has started to play a game.
   */
  chatActionStartPlayingGame();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -865884164;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The user is recording a video note.
 */
class chatActionRecordingVideoNote final : public ChatAction {
 public:

  /**
   * The user is recording a video note.
   */
  chatActionRecordingVideoNote();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 16523393;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The user is uploading a video note.
 */
class chatActionUploadingVideoNote final : public ChatAction {
 public:
  /// Upload progress, as a percentage.
  int32 progress_;

  /**
   * The user is uploading a video note.
   */
  chatActionUploadingVideoNote();

  /**
   * The user is uploading a video note.
   *
   * \param[in] progress_ Upload progress, as a percentage.
   */
  explicit chatActionUploadingVideoNote(int32 progress_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1172364918;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The user is watching animations sent by the other party by clicking on an animated emoji.
 */
class chatActionWatchingAnimations final : public ChatAction {
 public:
  /// The animated emoji.
  string emoji_;

  /**
   * The user is watching animations sent by the other party by clicking on an animated emoji.
   */
  chatActionWatchingAnimations();

  /**
   * The user is watching animations sent by the other party by clicking on an animated emoji.
   *
   * \param[in] emoji_ The animated emoji.
   */
  explicit chatActionWatchingAnimations(string const &emoji_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 2052990641;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The user has canceled the previous action.
 */
class chatActionCancel final : public ChatAction {
 public:

  /**
   * The user has canceled the previous action.
   */
  chatActionCancel();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1160523958;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Describes actions which must be possible to do through a chat action bar.
 */
class ChatActionBar: public Object {
 public:
};

/**
 * The chat can be reported as spam using the method reportChat with the reason chatReportReasonSpam.
 */
class chatActionBarReportSpam final : public ChatActionBar {
 public:
  /// If true, the chat was automatically archived and can be moved back to the main chat list using addChatToList simultaneously with setting chat notification settings to default using setChatNotificationSettings.
  bool can_unarchive_;

  /**
   * The chat can be reported as spam using the method reportChat with the reason chatReportReasonSpam.
   */
  chatActionBarReportSpam();

  /**
   * The chat can be reported as spam using the method reportChat with the reason chatReportReasonSpam.
   *
   * \param[in] can_unarchive_ If true, the chat was automatically archived and can be moved back to the main chat list using addChatToList simultaneously with setting chat notification settings to default using setChatNotificationSettings.
   */
  explicit chatActionBarReportSpam(bool can_unarchive_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1312758246;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The chat is a location-based supergroup, which can be reported as having unrelated location using the method reportChat with the reason chatReportReasonUnrelatedLocation.
 */
class chatActionBarReportUnrelatedLocation final : public ChatActionBar {
 public:

  /**
   * The chat is a location-based supergroup, which can be reported as having unrelated location using the method reportChat with the reason chatReportReasonUnrelatedLocation.
   */
  chatActionBarReportUnrelatedLocation();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 758175489;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The chat is a recently created group chat to which new members can be invited.
 */
class chatActionBarInviteMembers final : public ChatActionBar {
 public:

  /**
   * The chat is a recently created group chat to which new members can be invited.
   */
  chatActionBarInviteMembers();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1985313904;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The chat is a private or secret chat, which can be reported using the method reportChat, or the other user can be blocked using the method toggleMessageSenderIsBlocked, or the other user can be added to the contact list using the method addContact.
 */
class chatActionBarReportAddBlock final : public ChatActionBar {
 public:
  /// If true, the chat was automatically archived and can be moved back to the main chat list using addChatToList simultaneously with setting chat notification settings to default using setChatNotificationSettings.
  bool can_unarchive_;
  /// If non-negative, the current user was found by the peer through searchChatsNearby and this is the distance between the users.
  int32 distance_;

  /**
   * The chat is a private or secret chat, which can be reported using the method reportChat, or the other user can be blocked using the method toggleMessageSenderIsBlocked, or the other user can be added to the contact list using the method addContact.
   */
  chatActionBarReportAddBlock();

  /**
   * The chat is a private or secret chat, which can be reported using the method reportChat, or the other user can be blocked using the method toggleMessageSenderIsBlocked, or the other user can be added to the contact list using the method addContact.
   *
   * \param[in] can_unarchive_ If true, the chat was automatically archived and can be moved back to the main chat list using addChatToList simultaneously with setting chat notification settings to default using setChatNotificationSettings.
   * \param[in] distance_ If non-negative, the current user was found by the peer through searchChatsNearby and this is the distance between the users.
   */
  chatActionBarReportAddBlock(bool can_unarchive_, int32 distance_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -914150419;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The chat is a private or secret chat and the other user can be added to the contact list using the method addContact.
 */
class chatActionBarAddContact final : public ChatActionBar {
 public:

  /**
   * The chat is a private or secret chat and the other user can be added to the contact list using the method addContact.
   */
  chatActionBarAddContact();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -733325295;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The chat is a private or secret chat with a mutual contact and the user&\#039;s phone number can be shared with the other user using the method sharePhoneNumber.
 */
class chatActionBarSharePhoneNumber final : public ChatActionBar {
 public:

  /**
   * The chat is a private or secret chat with a mutual contact and the user&\#039;s phone number can be shared with the other user using the method sharePhoneNumber.
   */
  chatActionBarSharePhoneNumber();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 35188697;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The chat is a private chat with an administrator of a chat to which the user sent join request.
 */
class chatActionBarJoinRequest final : public ChatActionBar {
 public:
  /// Title of the chat to which the join request was sent.
  string title_;
  /// True, if the join request was sent to a channel chat.
  bool is_channel_;
  /// Point in time (Unix timestamp) when the join request was sent.
  int32 request_date_;

  /**
   * The chat is a private chat with an administrator of a chat to which the user sent join request.
   */
  chatActionBarJoinRequest();

  /**
   * The chat is a private chat with an administrator of a chat to which the user sent join request.
   *
   * \param[in] title_ Title of the chat to which the join request was sent.
   * \param[in] is_channel_ True, if the join request was sent to a channel chat.
   * \param[in] request_date_ Point in time (Unix timestamp) when the join request was sent.
   */
  chatActionBarJoinRequest(string const &title_, bool is_channel_, int32 request_date_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1037140744;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains information about a chat administrator.
 */
class chatAdministrator final : public Object {
 public:
  /// User identifier of the administrator.
  int53 user_id_;
  /// Custom title of the administrator.
  string custom_title_;
  /// True, if the user is the owner of the chat.
  bool is_owner_;

  /**
   * Contains information about a chat administrator.
   */
  chatAdministrator();

  /**
   * Contains information about a chat administrator.
   *
   * \param[in] user_id_ User identifier of the administrator.
   * \param[in] custom_title_ Custom title of the administrator.
   * \param[in] is_owner_ True, if the user is the owner of the chat.
   */
  chatAdministrator(int53 user_id_, string const &custom_title_, bool is_owner_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1920449836;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Represents a list of chat administrators.
 */
class chatAdministrators final : public Object {
 public:
  /// A list of chat administrators.
  array<object_ptr<chatAdministrator>> administrators_;

  /**
   * Represents a list of chat administrators.
   */
  chatAdministrators();

  /**
   * Represents a list of chat administrators.
   *
   * \param[in] administrators_ A list of chat administrators.
   */
  explicit chatAdministrators(array<object_ptr<chatAdministrator>> &&administrators_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -2126186435;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Represents a chat event.
 */
class chatEvent final : public Object {
 public:
  /// Chat event identifier.
  int64 id_;
  /// Point in time (Unix timestamp) when the event happened.
  int32 date_;
  /// Identifier of the user or chat who performed the action.
  object_ptr<MessageSender> member_id_;
  /// The action.
  object_ptr<ChatEventAction> action_;

  /**
   * Represents a chat event.
   */
  chatEvent();

  /**
   * Represents a chat event.
   *
   * \param[in] id_ Chat event identifier.
   * \param[in] date_ Point in time (Unix timestamp) when the event happened.
   * \param[in] member_id_ Identifier of the user or chat who performed the action.
   * \param[in] action_ The action.
   */
  chatEvent(int64 id_, int32 date_, object_ptr<MessageSender> &&member_id_, object_ptr<ChatEventAction> &&action_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -652102704;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Represents a chat event.
 */
class ChatEventAction: public Object {
 public:
};

/**
 * A message was edited.
 */
class chatEventMessageEdited final : public ChatEventAction {
 public:
  /// The original message before the edit.
  object_ptr<message> old_message_;
  /// The message after it was edited.
  object_ptr<message> new_message_;

  /**
   * A message was edited.
   */
  chatEventMessageEdited();

  /**
   * A message was edited.
   *
   * \param[in] old_message_ The original message before the edit.
   * \param[in] new_message_ The message after it was edited.
   */
  chatEventMessageEdited(object_ptr<message> &&old_message_, object_ptr<message> &&new_message_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -430967304;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A message was deleted.
 */
class chatEventMessageDeleted final : public ChatEventAction {
 public:
  /// Deleted message.
  object_ptr<message> message_;

  /**
   * A message was deleted.
   */
  chatEventMessageDeleted();

  /**
   * A message was deleted.
   *
   * \param[in] message_ Deleted message.
   */
  explicit chatEventMessageDeleted(object_ptr<message> &&message_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -892974601;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A poll in a message was stopped.
 */
class chatEventPollStopped final : public ChatEventAction {
 public:
  /// The message with the poll.
  object_ptr<message> message_;

  /**
   * A poll in a message was stopped.
   */
  chatEventPollStopped();

  /**
   * A poll in a message was stopped.
   *
   * \param[in] message_ The message with the poll.
   */
  explicit chatEventPollStopped(object_ptr<message> &&message_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 2009893861;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A message was pinned.
 */
class chatEventMessagePinned final : public ChatEventAction {
 public:
  /// Pinned message.
  object_ptr<message> message_;

  /**
   * A message was pinned.
   */
  chatEventMessagePinned();

  /**
   * A message was pinned.
   *
   * \param[in] message_ Pinned message.
   */
  explicit chatEventMessagePinned(object_ptr<message> &&message_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 438742298;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A message was unpinned.
 */
class chatEventMessageUnpinned final : public ChatEventAction {
 public:
  /// Unpinned message.
  object_ptr<message> message_;

  /**
   * A message was unpinned.
   */
  chatEventMessageUnpinned();

  /**
   * A message was unpinned.
   *
   * \param[in] message_ Unpinned message.
   */
  explicit chatEventMessageUnpinned(object_ptr<message> &&message_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -376161513;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A new member joined the chat.
 */
class chatEventMemberJoined final : public ChatEventAction {
 public:

  /**
   * A new member joined the chat.
   */
  chatEventMemberJoined();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -235468508;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A new member joined the chat via an invite link.
 */
class chatEventMemberJoinedByInviteLink final : public ChatEventAction {
 public:
  /// Invite link used to join the chat.
  object_ptr<chatInviteLink> invite_link_;

  /**
   * A new member joined the chat via an invite link.
   */
  chatEventMemberJoinedByInviteLink();

  /**
   * A new member joined the chat via an invite link.
   *
   * \param[in] invite_link_ Invite link used to join the chat.
   */
  explicit chatEventMemberJoinedByInviteLink(object_ptr<chatInviteLink> &&invite_link_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -253307459;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A new member was accepted to the chat by an administrator.
 */
class chatEventMemberJoinedByRequest final : public ChatEventAction {
 public:
  /// User identifier of the chat administrator, approved user join request.
  int53 approver_user_id_;
  /// Invite link used to join the chat; may be null.
  object_ptr<chatInviteLink> invite_link_;

  /**
   * A new member was accepted to the chat by an administrator.
   */
  chatEventMemberJoinedByRequest();

  /**
   * A new member was accepted to the chat by an administrator.
   *
   * \param[in] approver_user_id_ User identifier of the chat administrator, approved user join request.
   * \param[in] invite_link_ Invite link used to join the chat; may be null.
   */
  chatEventMemberJoinedByRequest(int53 approver_user_id_, object_ptr<chatInviteLink> &&invite_link_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1647804865;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A member left the chat.
 */
class chatEventMemberLeft final : public ChatEventAction {
 public:

  /**
   * A member left the chat.
   */
  chatEventMemberLeft();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -948420593;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A new chat member was invited.
 */
class chatEventMemberInvited final : public ChatEventAction {
 public:
  /// New member user identifier.
  int53 user_id_;
  /// New member status.
  object_ptr<ChatMemberStatus> status_;

  /**
   * A new chat member was invited.
   */
  chatEventMemberInvited();

  /**
   * A new chat member was invited.
   *
   * \param[in] user_id_ New member user identifier.
   * \param[in] status_ New member status.
   */
  chatEventMemberInvited(int53 user_id_, object_ptr<ChatMemberStatus> &&status_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 953663433;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A chat member has gained/lost administrator status, or the list of their administrator privileges has changed.
 */
class chatEventMemberPromoted final : public ChatEventAction {
 public:
  /// Affected chat member user identifier.
  int53 user_id_;
  /// Previous status of the chat member.
  object_ptr<ChatMemberStatus> old_status_;
  /// New status of the chat member.
  object_ptr<ChatMemberStatus> new_status_;

  /**
   * A chat member has gained/lost administrator status, or the list of their administrator privileges has changed.
   */
  chatEventMemberPromoted();

  /**
   * A chat member has gained/lost administrator status, or the list of their administrator privileges has changed.
   *
   * \param[in] user_id_ Affected chat member user identifier.
   * \param[in] old_status_ Previous status of the chat member.
   * \param[in] new_status_ New status of the chat member.
   */
  chatEventMemberPromoted(int53 user_id_, object_ptr<ChatMemberStatus> &&old_status_, object_ptr<ChatMemberStatus> &&new_status_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 525297761;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A chat member was restricted/unrestricted or banned/unbanned, or the list of their restrictions has changed.
 */
class chatEventMemberRestricted final : public ChatEventAction {
 public:
  /// Affected chat member identifier.
  object_ptr<MessageSender> member_id_;
  /// Previous status of the chat member.
  object_ptr<ChatMemberStatus> old_status_;
  /// New status of the chat member.
  object_ptr<ChatMemberStatus> new_status_;

  /**
   * A chat member was restricted/unrestricted or banned/unbanned, or the list of their restrictions has changed.
   */
  chatEventMemberRestricted();

  /**
   * A chat member was restricted/unrestricted or banned/unbanned, or the list of their restrictions has changed.
   *
   * \param[in] member_id_ Affected chat member identifier.
   * \param[in] old_status_ Previous status of the chat member.
   * \param[in] new_status_ New status of the chat member.
   */
  chatEventMemberRestricted(object_ptr<MessageSender> &&member_id_, object_ptr<ChatMemberStatus> &&old_status_, object_ptr<ChatMemberStatus> &&new_status_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1603608069;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The chat title was changed.
 */
class chatEventTitleChanged final : public ChatEventAction {
 public:
  /// Previous chat title.
  string old_title_;
  /// New chat title.
  string new_title_;

  /**
   * The chat title was changed.
   */
  chatEventTitleChanged();

  /**
   * The chat title was changed.
   *
   * \param[in] old_title_ Previous chat title.
   * \param[in] new_title_ New chat title.
   */
  chatEventTitleChanged(string const &old_title_, string const &new_title_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1134103250;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The chat permissions was changed.
 */
class chatEventPermissionsChanged final : public ChatEventAction {
 public:
  /// Previous chat permissions.
  object_ptr<chatPermissions> old_permissions_;
  /// New chat permissions.
  object_ptr<chatPermissions> new_permissions_;

  /**
   * The chat permissions was changed.
   */
  chatEventPermissionsChanged();

  /**
   * The chat permissions was changed.
   *
   * \param[in] old_permissions_ Previous chat permissions.
   * \param[in] new_permissions_ New chat permissions.
   */
  chatEventPermissionsChanged(object_ptr<chatPermissions> &&old_permissions_, object_ptr<chatPermissions> &&new_permissions_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1311557720;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The chat description was changed.
 */
class chatEventDescriptionChanged final : public ChatEventAction {
 public:
  /// Previous chat description.
  string old_description_;
  /// New chat description.
  string new_description_;

  /**
   * The chat description was changed.
   */
  chatEventDescriptionChanged();

  /**
   * The chat description was changed.
   *
   * \param[in] old_description_ Previous chat description.
   * \param[in] new_description_ New chat description.
   */
  chatEventDescriptionChanged(string const &old_description_, string const &new_description_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 39112478;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The chat username was changed.
 */
class chatEventUsernameChanged final : public ChatEventAction {
 public:
  /// Previous chat username.
  string old_username_;
  /// New chat username.
  string new_username_;

  /**
   * The chat username was changed.
   */
  chatEventUsernameChanged();

  /**
   * The chat username was changed.
   *
   * \param[in] old_username_ Previous chat username.
   * \param[in] new_username_ New chat username.
   */
  chatEventUsernameChanged(string const &old_username_, string const &new_username_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1728558443;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The chat photo was changed.
 */
class chatEventPhotoChanged final : public ChatEventAction {
 public:
  /// Previous chat photo value; may be null.
  object_ptr<chatPhoto> old_photo_;
  /// New chat photo value; may be null.
  object_ptr<chatPhoto> new_photo_;

  /**
   * The chat photo was changed.
   */
  chatEventPhotoChanged();

  /**
   * The chat photo was changed.
   *
   * \param[in] old_photo_ Previous chat photo value; may be null.
   * \param[in] new_photo_ New chat photo value; may be null.
   */
  chatEventPhotoChanged(object_ptr<chatPhoto> &&old_photo_, object_ptr<chatPhoto> &&new_photo_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -811572541;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The can_invite_users permission of a supergroup chat was toggled.
 */
class chatEventInvitesToggled final : public ChatEventAction {
 public:
  /// New value of can_invite_users permission.
  bool can_invite_users_;

  /**
   * The can_invite_users permission of a supergroup chat was toggled.
   */
  chatEventInvitesToggled();

  /**
   * The can_invite_users permission of a supergroup chat was toggled.
   *
   * \param[in] can_invite_users_ New value of can_invite_users permission.
   */
  explicit chatEventInvitesToggled(bool can_invite_users_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -62548373;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The linked chat of a supergroup was changed.
 */
class chatEventLinkedChatChanged final : public ChatEventAction {
 public:
  /// Previous supergroup linked chat identifier.
  int53 old_linked_chat_id_;
  /// New supergroup linked chat identifier.
  int53 new_linked_chat_id_;

  /**
   * The linked chat of a supergroup was changed.
   */
  chatEventLinkedChatChanged();

  /**
   * The linked chat of a supergroup was changed.
   *
   * \param[in] old_linked_chat_id_ Previous supergroup linked chat identifier.
   * \param[in] new_linked_chat_id_ New supergroup linked chat identifier.
   */
  chatEventLinkedChatChanged(int53 old_linked_chat_id_, int53 new_linked_chat_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1797419439;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The slow_mode_delay setting of a supergroup was changed.
 */
class chatEventSlowModeDelayChanged final : public ChatEventAction {
 public:
  /// Previous value of slow_mode_delay, in seconds.
  int32 old_slow_mode_delay_;
  /// New value of slow_mode_delay, in seconds.
  int32 new_slow_mode_delay_;

  /**
   * The slow_mode_delay setting of a supergroup was changed.
   */
  chatEventSlowModeDelayChanged();

  /**
   * The slow_mode_delay setting of a supergroup was changed.
   *
   * \param[in] old_slow_mode_delay_ Previous value of slow_mode_delay, in seconds.
   * \param[in] new_slow_mode_delay_ New value of slow_mode_delay, in seconds.
   */
  chatEventSlowModeDelayChanged(int32 old_slow_mode_delay_, int32 new_slow_mode_delay_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1653195765;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The message TTL was changed.
 */
class chatEventMessageTtlChanged final : public ChatEventAction {
 public:
  /// Previous value of message_ttl.
  int32 old_message_ttl_;
  /// New value of message_ttl.
  int32 new_message_ttl_;

  /**
   * The message TTL was changed.
   */
  chatEventMessageTtlChanged();

  /**
   * The message TTL was changed.
   *
   * \param[in] old_message_ttl_ Previous value of message_ttl.
   * \param[in] new_message_ttl_ New value of message_ttl.
   */
  chatEventMessageTtlChanged(int32 old_message_ttl_, int32 new_message_ttl_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 776386995;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The sign_messages setting of a channel was toggled.
 */
class chatEventSignMessagesToggled final : public ChatEventAction {
 public:
  /// New value of sign_messages.
  bool sign_messages_;

  /**
   * The sign_messages setting of a channel was toggled.
   */
  chatEventSignMessagesToggled();

  /**
   * The sign_messages setting of a channel was toggled.
   *
   * \param[in] sign_messages_ New value of sign_messages.
   */
  explicit chatEventSignMessagesToggled(bool sign_messages_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1313265634;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The has_protected_content setting of a channel was toggled.
 */
class chatEventHasProtectedContentToggled final : public ChatEventAction {
 public:
  /// New value of has_protected_content.
  bool has_protected_content_;

  /**
   * The has_protected_content setting of a channel was toggled.
   */
  chatEventHasProtectedContentToggled();

  /**
   * The has_protected_content setting of a channel was toggled.
   *
   * \param[in] has_protected_content_ New value of has_protected_content.
   */
  explicit chatEventHasProtectedContentToggled(bool has_protected_content_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -184270335;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The supergroup sticker set was changed.
 */
class chatEventStickerSetChanged final : public ChatEventAction {
 public:
  /// Previous identifier of the chat sticker set; 0 if none.
  int64 old_sticker_set_id_;
  /// New identifier of the chat sticker set; 0 if none.
  int64 new_sticker_set_id_;

  /**
   * The supergroup sticker set was changed.
   */
  chatEventStickerSetChanged();

  /**
   * The supergroup sticker set was changed.
   *
   * \param[in] old_sticker_set_id_ Previous identifier of the chat sticker set; 0 if none.
   * \param[in] new_sticker_set_id_ New identifier of the chat sticker set; 0 if none.
   */
  chatEventStickerSetChanged(int64 old_sticker_set_id_, int64 new_sticker_set_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1243130481;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The supergroup location was changed.
 */
class chatEventLocationChanged final : public ChatEventAction {
 public:
  /// Previous location; may be null.
  object_ptr<chatLocation> old_location_;
  /// New location; may be null.
  object_ptr<chatLocation> new_location_;

  /**
   * The supergroup location was changed.
   */
  chatEventLocationChanged();

  /**
   * The supergroup location was changed.
   *
   * \param[in] old_location_ Previous location; may be null.
   * \param[in] new_location_ New location; may be null.
   */
  chatEventLocationChanged(object_ptr<chatLocation> &&old_location_, object_ptr<chatLocation> &&new_location_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -405930674;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The is_all_history_available setting of a supergroup was toggled.
 */
class chatEventIsAllHistoryAvailableToggled final : public ChatEventAction {
 public:
  /// New value of is_all_history_available.
  bool is_all_history_available_;

  /**
   * The is_all_history_available setting of a supergroup was toggled.
   */
  chatEventIsAllHistoryAvailableToggled();

  /**
   * The is_all_history_available setting of a supergroup was toggled.
   *
   * \param[in] is_all_history_available_ New value of is_all_history_available.
   */
  explicit chatEventIsAllHistoryAvailableToggled(bool is_all_history_available_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1599063019;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A chat invite link was edited.
 */
class chatEventInviteLinkEdited final : public ChatEventAction {
 public:
  /// Previous information about the invite link.
  object_ptr<chatInviteLink> old_invite_link_;
  /// New information about the invite link.
  object_ptr<chatInviteLink> new_invite_link_;

  /**
   * A chat invite link was edited.
   */
  chatEventInviteLinkEdited();

  /**
   * A chat invite link was edited.
   *
   * \param[in] old_invite_link_ Previous information about the invite link.
   * \param[in] new_invite_link_ New information about the invite link.
   */
  chatEventInviteLinkEdited(object_ptr<chatInviteLink> &&old_invite_link_, object_ptr<chatInviteLink> &&new_invite_link_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -460190366;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A chat invite link was revoked.
 */
class chatEventInviteLinkRevoked final : public ChatEventAction {
 public:
  /// The invite link.
  object_ptr<chatInviteLink> invite_link_;

  /**
   * A chat invite link was revoked.
   */
  chatEventInviteLinkRevoked();

  /**
   * A chat invite link was revoked.
   *
   * \param[in] invite_link_ The invite link.
   */
  explicit chatEventInviteLinkRevoked(object_ptr<chatInviteLink> &&invite_link_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1579417629;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A revoked chat invite link was deleted.
 */
class chatEventInviteLinkDeleted final : public ChatEventAction {
 public:
  /// The invite link.
  object_ptr<chatInviteLink> invite_link_;

  /**
   * A revoked chat invite link was deleted.
   */
  chatEventInviteLinkDeleted();

  /**
   * A revoked chat invite link was deleted.
   *
   * \param[in] invite_link_ The invite link.
   */
  explicit chatEventInviteLinkDeleted(object_ptr<chatInviteLink> &&invite_link_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1394974361;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A video chat was created.
 */
class chatEventVideoChatCreated final : public ChatEventAction {
 public:
  /// Identifier of the video chat. The video chat can be received through the method getGroupCall.
  int32 group_call_id_;

  /**
   * A video chat was created.
   */
  chatEventVideoChatCreated();

  /**
   * A video chat was created.
   *
   * \param[in] group_call_id_ Identifier of the video chat. The video chat can be received through the method getGroupCall.
   */
  explicit chatEventVideoChatCreated(int32 group_call_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1822853755;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A video chat was ended.
 */
class chatEventVideoChatEnded final : public ChatEventAction {
 public:
  /// Identifier of the video chat. The video chat can be received through the method getGroupCall.
  int32 group_call_id_;

  /**
   * A video chat was ended.
   */
  chatEventVideoChatEnded();

  /**
   * A video chat was ended.
   *
   * \param[in] group_call_id_ Identifier of the video chat. The video chat can be received through the method getGroupCall.
   */
  explicit chatEventVideoChatEnded(int32 group_call_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1630039112;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A video chat participant was muted or unmuted.
 */
class chatEventVideoChatParticipantIsMutedToggled final : public ChatEventAction {
 public:
  /// Identifier of the affected group call participant.
  object_ptr<MessageSender> participant_id_;
  /// New value of is_muted.
  bool is_muted_;

  /**
   * A video chat participant was muted or unmuted.
   */
  chatEventVideoChatParticipantIsMutedToggled();

  /**
   * A video chat participant was muted or unmuted.
   *
   * \param[in] participant_id_ Identifier of the affected group call participant.
   * \param[in] is_muted_ New value of is_muted.
   */
  chatEventVideoChatParticipantIsMutedToggled(object_ptr<MessageSender> &&participant_id_, bool is_muted_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 521165047;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A video chat participant volume level was changed.
 */
class chatEventVideoChatParticipantVolumeLevelChanged final : public ChatEventAction {
 public:
  /// Identifier of the affected group call participant.
  object_ptr<MessageSender> participant_id_;
  /// New value of volume_level; 1-20000 in hundreds of percents.
  int32 volume_level_;

  /**
   * A video chat participant volume level was changed.
   */
  chatEventVideoChatParticipantVolumeLevelChanged();

  /**
   * A video chat participant volume level was changed.
   *
   * \param[in] participant_id_ Identifier of the affected group call participant.
   * \param[in] volume_level_ New value of volume_level; 1-20000 in hundreds of percents.
   */
  chatEventVideoChatParticipantVolumeLevelChanged(object_ptr<MessageSender> &&participant_id_, int32 volume_level_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1131385534;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The mute_new_participants setting of a video chat was toggled.
 */
class chatEventVideoChatMuteNewParticipantsToggled final : public ChatEventAction {
 public:
  /// New value of the mute_new_participants setting.
  bool mute_new_participants_;

  /**
   * The mute_new_participants setting of a video chat was toggled.
   */
  chatEventVideoChatMuteNewParticipantsToggled();

  /**
   * The mute_new_participants setting of a video chat was toggled.
   *
   * \param[in] mute_new_participants_ New value of the mute_new_participants setting.
   */
  explicit chatEventVideoChatMuteNewParticipantsToggled(bool mute_new_participants_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -126547970;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Represents a set of filters used to obtain a chat event log.
 */
class chatEventLogFilters final : public Object {
 public:
  /// True, if message edits need to be returned.
  bool message_edits_;
  /// True, if message deletions need to be returned.
  bool message_deletions_;
  /// True, if pin/unpin events need to be returned.
  bool message_pins_;
  /// True, if members joining events need to be returned.
  bool member_joins_;
  /// True, if members leaving events need to be returned.
  bool member_leaves_;
  /// True, if invited member events need to be returned.
  bool member_invites_;
  /// True, if member promotion/demotion events need to be returned.
  bool member_promotions_;
  /// True, if member restricted/unrestricted/banned/unbanned events need to be returned.
  bool member_restrictions_;
  /// True, if changes in chat information need to be returned.
  bool info_changes_;
  /// True, if changes in chat settings need to be returned.
  bool setting_changes_;
  /// True, if changes to invite links need to be returned.
  bool invite_link_changes_;
  /// True, if video chat actions need to be returned.
  bool video_chat_changes_;

  /**
   * Represents a set of filters used to obtain a chat event log.
   */
  chatEventLogFilters();

  /**
   * Represents a set of filters used to obtain a chat event log.
   *
   * \param[in] message_edits_ True, if message edits need to be returned.
   * \param[in] message_deletions_ True, if message deletions need to be returned.
   * \param[in] message_pins_ True, if pin/unpin events need to be returned.
   * \param[in] member_joins_ True, if members joining events need to be returned.
   * \param[in] member_leaves_ True, if members leaving events need to be returned.
   * \param[in] member_invites_ True, if invited member events need to be returned.
   * \param[in] member_promotions_ True, if member promotion/demotion events need to be returned.
   * \param[in] member_restrictions_ True, if member restricted/unrestricted/banned/unbanned events need to be returned.
   * \param[in] info_changes_ True, if changes in chat information need to be returned.
   * \param[in] setting_changes_ True, if changes in chat settings need to be returned.
   * \param[in] invite_link_changes_ True, if changes to invite links need to be returned.
   * \param[in] video_chat_changes_ True, if video chat actions need to be returned.
   */
  chatEventLogFilters(bool message_edits_, bool message_deletions_, bool message_pins_, bool member_joins_, bool member_leaves_, bool member_invites_, bool member_promotions_, bool member_restrictions_, bool info_changes_, bool setting_changes_, bool invite_link_changes_, bool video_chat_changes_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1251197299;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains a list of chat events.
 */
class chatEvents final : public Object {
 public:
  /// List of events.
  array<object_ptr<chatEvent>> events_;

  /**
   * Contains a list of chat events.
   */
  chatEvents();

  /**
   * Contains a list of chat events.
   *
   * \param[in] events_ List of events.
   */
  explicit chatEvents(array<object_ptr<chatEvent>> &&events_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -585329664;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Represents a filter of user chats.
 */
class chatFilter final : public Object {
 public:
  /// The title of the filter; 1-12 characters without line feeds.
  string title_;
  /// The chosen icon name for short filter representation. If non-empty, must be one of &quot;All&quot;, &quot;Unread&quot;, &quot;Unmuted&quot;, &quot;Bots&quot;, &quot;Channels&quot;, &quot;Groups&quot;, &quot;Private&quot;, &quot;Custom&quot;, &quot;Setup&quot;, &quot;Cat&quot;, &quot;Crown&quot;, &quot;Favorite&quot;, &quot;Flower&quot;, &quot;Game&quot;, &quot;Home&quot;, &quot;Love&quot;, &quot;Mask&quot;, &quot;Party&quot;, &quot;Sport&quot;, &quot;Study&quot;, &quot;Trade&quot;, &quot;Travel&quot;, &quot;Work&quot;. If empty, use getChatFilterDefaultIconName to get default icon name for the filter.
  string icon_name_;
  /// The chat identifiers of pinned chats in the filtered chat list.
  array<int53> pinned_chat_ids_;
  /// The chat identifiers of always included chats in the filtered chat list.
  array<int53> included_chat_ids_;
  /// The chat identifiers of always excluded chats in the filtered chat list.
  array<int53> excluded_chat_ids_;
  /// True, if muted chats need to be excluded.
  bool exclude_muted_;
  /// True, if read chats need to be excluded.
  bool exclude_read_;
  /// True, if archived chats need to be excluded.
  bool exclude_archived_;
  /// True, if contacts need to be included.
  bool include_contacts_;
  /// True, if non-contact users need to be included.
  bool include_non_contacts_;
  /// True, if bots need to be included.
  bool include_bots_;
  /// True, if basic groups and supergroups need to be included.
  bool include_groups_;
  /// True, if channels need to be included.
  bool include_channels_;

  /**
   * Represents a filter of user chats.
   */
  chatFilter();

  /**
   * Represents a filter of user chats.
   *
   * \param[in] title_ The title of the filter; 1-12 characters without line feeds.
   * \param[in] icon_name_ The chosen icon name for short filter representation. If non-empty, must be one of &quot;All&quot;, &quot;Unread&quot;, &quot;Unmuted&quot;, &quot;Bots&quot;, &quot;Channels&quot;, &quot;Groups&quot;, &quot;Private&quot;, &quot;Custom&quot;, &quot;Setup&quot;, &quot;Cat&quot;, &quot;Crown&quot;, &quot;Favorite&quot;, &quot;Flower&quot;, &quot;Game&quot;, &quot;Home&quot;, &quot;Love&quot;, &quot;Mask&quot;, &quot;Party&quot;, &quot;Sport&quot;, &quot;Study&quot;, &quot;Trade&quot;, &quot;Travel&quot;, &quot;Work&quot;. If empty, use getChatFilterDefaultIconName to get default icon name for the filter.
   * \param[in] pinned_chat_ids_ The chat identifiers of pinned chats in the filtered chat list.
   * \param[in] included_chat_ids_ The chat identifiers of always included chats in the filtered chat list.
   * \param[in] excluded_chat_ids_ The chat identifiers of always excluded chats in the filtered chat list.
   * \param[in] exclude_muted_ True, if muted chats need to be excluded.
   * \param[in] exclude_read_ True, if read chats need to be excluded.
   * \param[in] exclude_archived_ True, if archived chats need to be excluded.
   * \param[in] include_contacts_ True, if contacts need to be included.
   * \param[in] include_non_contacts_ True, if non-contact users need to be included.
   * \param[in] include_bots_ True, if bots need to be included.
   * \param[in] include_groups_ True, if basic groups and supergroups need to be included.
   * \param[in] include_channels_ True, if channels need to be included.
   */
  chatFilter(string const &title_, string const &icon_name_, array<int53> &&pinned_chat_ids_, array<int53> &&included_chat_ids_, array<int53> &&excluded_chat_ids_, bool exclude_muted_, bool exclude_read_, bool exclude_archived_, bool include_contacts_, bool include_non_contacts_, bool include_bots_, bool include_groups_, bool include_channels_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -664815123;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains basic information about a chat filter.
 */
class chatFilterInfo final : public Object {
 public:
  /// Unique chat filter identifier.
  int32 id_;
  /// The title of the filter; 1-12 characters without line feeds.
  string title_;
  /// The chosen or default icon name for short filter representation. One of &quot;All&quot;, &quot;Unread&quot;, &quot;Unmuted&quot;, &quot;Bots&quot;, &quot;Channels&quot;, &quot;Groups&quot;, &quot;Private&quot;, &quot;Custom&quot;, &quot;Setup&quot;, &quot;Cat&quot;, &quot;Crown&quot;, &quot;Favorite&quot;, &quot;Flower&quot;, &quot;Game&quot;, &quot;Home&quot;, &quot;Love&quot;, &quot;Mask&quot;, &quot;Party&quot;, &quot;Sport&quot;, &quot;Study&quot;, &quot;Trade&quot;, &quot;Travel&quot;, &quot;Work&quot;.
  string icon_name_;

  /**
   * Contains basic information about a chat filter.
   */
  chatFilterInfo();

  /**
   * Contains basic information about a chat filter.
   *
   * \param[in] id_ Unique chat filter identifier.
   * \param[in] title_ The title of the filter; 1-12 characters without line feeds.
   * \param[in] icon_name_ The chosen or default icon name for short filter representation. One of &quot;All&quot;, &quot;Unread&quot;, &quot;Unmuted&quot;, &quot;Bots&quot;, &quot;Channels&quot;, &quot;Groups&quot;, &quot;Private&quot;, &quot;Custom&quot;, &quot;Setup&quot;, &quot;Cat&quot;, &quot;Crown&quot;, &quot;Favorite&quot;, &quot;Flower&quot;, &quot;Game&quot;, &quot;Home&quot;, &quot;Love&quot;, &quot;Mask&quot;, &quot;Party&quot;, &quot;Sport&quot;, &quot;Study&quot;, &quot;Trade&quot;, &quot;Travel&quot;, &quot;Work&quot;.
   */
  chatFilterInfo(int32 id_, string const &title_, string const &icon_name_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -943721165;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains a chat invite link.
 */
class chatInviteLink final : public Object {
 public:
  /// Chat invite link.
  string invite_link_;
  /// Name of the link.
  string name_;
  /// User identifier of an administrator created the link.
  int53 creator_user_id_;
  /// Point in time (Unix timestamp) when the link was created.
  int32 date_;
  /// Point in time (Unix timestamp) when the link was last edited; 0 if never or unknown.
  int32 edit_date_;
  /// Point in time (Unix timestamp) when the link will expire; 0 if never.
  int32 expiration_date_;
  /// The maximum number of members, which can join the chat using the link simultaneously; 0 if not limited. Always 0 if the link requires approval.
  int32 member_limit_;
  /// Number of chat members, which joined the chat using the link.
  int32 member_count_;
  /// Number of pending join requests created using this link.
  int32 pending_join_request_count_;
  /// True, if the link only creates join request. If true, total number of joining members will be unlimited.
  bool creates_join_request_;
  /// True, if the link is primary. Primary invite link can&\#039;t have name, expiration date, or usage limit. There is exactly one primary invite link for each administrator with can_invite_users right at a given time.
  bool is_primary_;
  /// True, if the link was revoked.
  bool is_revoked_;

  /**
   * Contains a chat invite link.
   */
  chatInviteLink();

  /**
   * Contains a chat invite link.
   *
   * \param[in] invite_link_ Chat invite link.
   * \param[in] name_ Name of the link.
   * \param[in] creator_user_id_ User identifier of an administrator created the link.
   * \param[in] date_ Point in time (Unix timestamp) when the link was created.
   * \param[in] edit_date_ Point in time (Unix timestamp) when the link was last edited; 0 if never or unknown.
   * \param[in] expiration_date_ Point in time (Unix timestamp) when the link will expire; 0 if never.
   * \param[in] member_limit_ The maximum number of members, which can join the chat using the link simultaneously; 0 if not limited. Always 0 if the link requires approval.
   * \param[in] member_count_ Number of chat members, which joined the chat using the link.
   * \param[in] pending_join_request_count_ Number of pending join requests created using this link.
   * \param[in] creates_join_request_ True, if the link only creates join request. If true, total number of joining members will be unlimited.
   * \param[in] is_primary_ True, if the link is primary. Primary invite link can&\#039;t have name, expiration date, or usage limit. There is exactly one primary invite link for each administrator with can_invite_users right at a given time.
   * \param[in] is_revoked_ True, if the link was revoked.
   */
  chatInviteLink(string const &invite_link_, string const &name_, int53 creator_user_id_, int32 date_, int32 edit_date_, int32 expiration_date_, int32 member_limit_, int32 member_count_, int32 pending_join_request_count_, bool creates_join_request_, bool is_primary_, bool is_revoked_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -205812476;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Describes a chat administrator with a number of active and revoked chat invite links.
 */
class chatInviteLinkCount final : public Object {
 public:
  /// Administrator&\#039;s user identifier.
  int53 user_id_;
  /// Number of active invite links.
  int32 invite_link_count_;
  /// Number of revoked invite links.
  int32 revoked_invite_link_count_;

  /**
   * Describes a chat administrator with a number of active and revoked chat invite links.
   */
  chatInviteLinkCount();

  /**
   * Describes a chat administrator with a number of active and revoked chat invite links.
   *
   * \param[in] user_id_ Administrator&\#039;s user identifier.
   * \param[in] invite_link_count_ Number of active invite links.
   * \param[in] revoked_invite_link_count_ Number of revoked invite links.
   */
  chatInviteLinkCount(int53 user_id_, int32 invite_link_count_, int32 revoked_invite_link_count_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1021999210;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains a list of chat invite link counts.
 */
class chatInviteLinkCounts final : public Object {
 public:
  /// List of invite link counts.
  array<object_ptr<chatInviteLinkCount>> invite_link_counts_;

  /**
   * Contains a list of chat invite link counts.
   */
  chatInviteLinkCounts();

  /**
   * Contains a list of chat invite link counts.
   *
   * \param[in] invite_link_counts_ List of invite link counts.
   */
  explicit chatInviteLinkCounts(array<object_ptr<chatInviteLinkCount>> &&invite_link_counts_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 920326637;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains information about a chat invite link.
 */
class chatInviteLinkInfo final : public Object {
 public:
  /// Chat identifier of the invite link; 0 if the user has no access to the chat before joining.
  int53 chat_id_;
  /// If non-zero, the amount of time for which read access to the chat will remain available, in seconds.
  int32 accessible_for_;
  /// Type of the chat.
  object_ptr<ChatType> type_;
  /// Title of the chat.
  string title_;
  /// Chat photo; may be null.
  object_ptr<chatPhotoInfo> photo_;
  /// Chat description.
  string description_;
  /// Number of members in the chat.
  int32 member_count_;
  /// User identifiers of some chat members that may be known to the current user.
  array<int53> member_user_ids_;
  /// True, if the link only creates join request.
  bool creates_join_request_;
  /// True, if the chat is a public supergroup or channel, i.e. it has a username or it is a location-based supergroup.
  bool is_public_;

  /**
   * Contains information about a chat invite link.
   */
  chatInviteLinkInfo();

  /**
   * Contains information about a chat invite link.
   *
   * \param[in] chat_id_ Chat identifier of the invite link; 0 if the user has no access to the chat before joining.
   * \param[in] accessible_for_ If non-zero, the amount of time for which read access to the chat will remain available, in seconds.
   * \param[in] type_ Type of the chat.
   * \param[in] title_ Title of the chat.
   * \param[in] photo_ Chat photo; may be null.
   * \param[in] description_ Chat description.
   * \param[in] member_count_ Number of members in the chat.
   * \param[in] member_user_ids_ User identifiers of some chat members that may be known to the current user.
   * \param[in] creates_join_request_ True, if the link only creates join request.
   * \param[in] is_public_ True, if the chat is a public supergroup or channel, i.e. it has a username or it is a location-based supergroup.
   */
  chatInviteLinkInfo(int53 chat_id_, int32 accessible_for_, object_ptr<ChatType> &&type_, string const &title_, object_ptr<chatPhotoInfo> &&photo_, string const &description_, int32 member_count_, array<int53> &&member_user_ids_, bool creates_join_request_, bool is_public_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 546234276;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Describes a chat member joined a chat via an invite link.
 */
class chatInviteLinkMember final : public Object {
 public:
  /// User identifier.
  int53 user_id_;
  /// Point in time (Unix timestamp) when the user joined the chat.
  int32 joined_chat_date_;
  /// User identifier of the chat administrator, approved user join request.
  int53 approver_user_id_;

  /**
   * Describes a chat member joined a chat via an invite link.
   */
  chatInviteLinkMember();

  /**
   * Describes a chat member joined a chat via an invite link.
   *
   * \param[in] user_id_ User identifier.
   * \param[in] joined_chat_date_ Point in time (Unix timestamp) when the user joined the chat.
   * \param[in] approver_user_id_ User identifier of the chat administrator, approved user join request.
   */
  chatInviteLinkMember(int53 user_id_, int32 joined_chat_date_, int53 approver_user_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1409060582;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains a list of chat members joined a chat via an invite link.
 */
class chatInviteLinkMembers final : public Object {
 public:
  /// Approximate total count of chat members found.
  int32 total_count_;
  /// List of chat members, joined a chat via an invite link.
  array<object_ptr<chatInviteLinkMember>> members_;

  /**
   * Contains a list of chat members joined a chat via an invite link.
   */
  chatInviteLinkMembers();

  /**
   * Contains a list of chat members joined a chat via an invite link.
   *
   * \param[in] total_count_ Approximate total count of chat members found.
   * \param[in] members_ List of chat members, joined a chat via an invite link.
   */
  chatInviteLinkMembers(int32 total_count_, array<object_ptr<chatInviteLinkMember>> &&members_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 315635051;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains a list of chat invite links.
 */
class chatInviteLinks final : public Object {
 public:
  /// Approximate total count of chat invite links found.
  int32 total_count_;
  /// List of invite links.
  array<object_ptr<chatInviteLink>> invite_links_;

  /**
   * Contains a list of chat invite links.
   */
  chatInviteLinks();

  /**
   * Contains a list of chat invite links.
   *
   * \param[in] total_count_ Approximate total count of chat invite links found.
   * \param[in] invite_links_ List of invite links.
   */
  chatInviteLinks(int32 total_count_, array<object_ptr<chatInviteLink>> &&invite_links_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 112891427;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Describes a user that sent a join request and waits for administrator approval.
 */
class chatJoinRequest final : public Object {
 public:
  /// User identifier.
  int53 user_id_;
  /// Point in time (Unix timestamp) when the user sent the join request.
  int32 date_;
  /// A short bio of the user.
  string bio_;

  /**
   * Describes a user that sent a join request and waits for administrator approval.
   */
  chatJoinRequest();

  /**
   * Describes a user that sent a join request and waits for administrator approval.
   *
   * \param[in] user_id_ User identifier.
   * \param[in] date_ Point in time (Unix timestamp) when the user sent the join request.
   * \param[in] bio_ A short bio of the user.
   */
  chatJoinRequest(int53 user_id_, int32 date_, string const &bio_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 59341416;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains a list of requests to join a chat.
 */
class chatJoinRequests final : public Object {
 public:
  /// Approximate total count of requests found.
  int32 total_count_;
  /// List of the requests.
  array<object_ptr<chatJoinRequest>> requests_;

  /**
   * Contains a list of requests to join a chat.
   */
  chatJoinRequests();

  /**
   * Contains a list of requests to join a chat.
   *
   * \param[in] total_count_ Approximate total count of requests found.
   * \param[in] requests_ List of the requests.
   */
  chatJoinRequests(int32 total_count_, array<object_ptr<chatJoinRequest>> &&requests_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1291680519;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains information about pending join requests for a chat.
 */
class chatJoinRequestsInfo final : public Object {
 public:
  /// Total number of pending join requests.
  int32 total_count_;
  /// Identifiers of at most 3 users sent the newest pending join requests.
  array<int53> user_ids_;

  /**
   * Contains information about pending join requests for a chat.
   */
  chatJoinRequestsInfo();

  /**
   * Contains information about pending join requests for a chat.
   *
   * \param[in] total_count_ Total number of pending join requests.
   * \param[in] user_ids_ Identifiers of at most 3 users sent the newest pending join requests.
   */
  chatJoinRequestsInfo(int32 total_count_, array<int53> &&user_ids_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 888534463;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Describes a list of chats.
 */
class ChatList: public Object {
 public:
};

/**
 * A main list of chats.
 */
class chatListMain final : public ChatList {
 public:

  /**
   * A main list of chats.
   */
  chatListMain();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -400991316;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A list of chats usually located at the top of the main chat list. Unmuted chats are automatically moved from the Archive to the Main chat list when a new message arrives.
 */
class chatListArchive final : public ChatList {
 public:

  /**
   * A list of chats usually located at the top of the main chat list. Unmuted chats are automatically moved from the Archive to the Main chat list when a new message arrives.
   */
  chatListArchive();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 362770115;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A list of chats belonging to a chat filter.
 */
class chatListFilter final : public ChatList {
 public:
  /// Chat filter identifier.
  int32 chat_filter_id_;

  /**
   * A list of chats belonging to a chat filter.
   */
  chatListFilter();

  /**
   * A list of chats belonging to a chat filter.
   *
   * \param[in] chat_filter_id_ Chat filter identifier.
   */
  explicit chatListFilter(int32 chat_filter_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -2022707655;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains a list of chat lists.
 */
class chatLists final : public Object {
 public:
  /// List of chat lists.
  array<object_ptr<ChatList>> chat_lists_;

  /**
   * Contains a list of chat lists.
   */
  chatLists();

  /**
   * Contains a list of chat lists.
   *
   * \param[in] chat_lists_ List of chat lists.
   */
  explicit chatLists(array<object_ptr<ChatList>> &&chat_lists_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -258292771;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Represents a location to which a chat is connected.
 */
class chatLocation final : public Object {
 public:
  /// The location.
  object_ptr<location> location_;
  /// Location address; 1-64 characters, as defined by the chat owner.
  string address_;

  /**
   * Represents a location to which a chat is connected.
   */
  chatLocation();

  /**
   * Represents a location to which a chat is connected.
   *
   * \param[in] location_ The location.
   * \param[in] address_ Location address; 1-64 characters, as defined by the chat owner.
   */
  chatLocation(object_ptr<location> &&location_, string const &address_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1566863583;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Describes a user or a chat as a member of another chat.
 */
class chatMember final : public Object {
 public:
  /// Identifier of the chat member. Currently, other chats can be only Left or Banned. Only supergroups and channels can have other chats as Left or Banned members and these chats must be supergroups or channels.
  object_ptr<MessageSender> member_id_;
  /// Identifier of a user that invited/promoted/banned this member in the chat; 0 if unknown.
  int53 inviter_user_id_;
  /// Point in time (Unix timestamp) when the user joined the chat.
  int32 joined_chat_date_;
  /// Status of the member in the chat.
  object_ptr<ChatMemberStatus> status_;

  /**
   * Describes a user or a chat as a member of another chat.
   */
  chatMember();

  /**
   * Describes a user or a chat as a member of another chat.
   *
   * \param[in] member_id_ Identifier of the chat member. Currently, other chats can be only Left or Banned. Only supergroups and channels can have other chats as Left or Banned members and these chats must be supergroups or channels.
   * \param[in] inviter_user_id_ Identifier of a user that invited/promoted/banned this member in the chat; 0 if unknown.
   * \param[in] joined_chat_date_ Point in time (Unix timestamp) when the user joined the chat.
   * \param[in] status_ Status of the member in the chat.
   */
  chatMember(object_ptr<MessageSender> &&member_id_, int53 inviter_user_id_, int32 joined_chat_date_, object_ptr<ChatMemberStatus> &&status_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1829953909;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Provides information about the status of a member in a chat.
 */
class ChatMemberStatus: public Object {
 public:
};

/**
 * The user is the owner of the chat and has all the administrator privileges.
 */
class chatMemberStatusCreator final : public ChatMemberStatus {
 public:
  /// A custom title of the owner; 0-16 characters without emojis; applicable to supergroups only.
  string custom_title_;
  /// True, if the creator isn&\#039;t shown in the chat member list and sends messages anonymously; applicable to supergroups only.
  bool is_anonymous_;
  /// True, if the user is a member of the chat.
  bool is_member_;

  /**
   * The user is the owner of the chat and has all the administrator privileges.
   */
  chatMemberStatusCreator();

  /**
   * The user is the owner of the chat and has all the administrator privileges.
   *
   * \param[in] custom_title_ A custom title of the owner; 0-16 characters without emojis; applicable to supergroups only.
   * \param[in] is_anonymous_ True, if the creator isn&\#039;t shown in the chat member list and sends messages anonymously; applicable to supergroups only.
   * \param[in] is_member_ True, if the user is a member of the chat.
   */
  chatMemberStatusCreator(string const &custom_title_, bool is_anonymous_, bool is_member_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -160019714;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The user is a member of the chat and has some additional privileges. In basic groups, administrators can edit and delete messages sent by others, add new members, ban unprivileged members, and manage video chats. In supergroups and channels, there are more detailed options for administrator privileges.
 */
class chatMemberStatusAdministrator final : public ChatMemberStatus {
 public:
  /// A custom title of the administrator; 0-16 characters without emojis; applicable to supergroups only.
  string custom_title_;
  /// True, if the current user can edit the administrator privileges for the called user.
  bool can_be_edited_;
  /// True, if the administrator can get chat event log, get chat statistics, get message statistics in channels, get channel members, see anonymous administrators in supergroups and ignore slow mode. Implied by any other privilege; applicable to supergroups and channels only.
  bool can_manage_chat_;
  /// True, if the administrator can change the chat title, photo, and other settings.
  bool can_change_info_;
  /// True, if the administrator can create channel posts; applicable to channels only.
  bool can_post_messages_;
  /// True, if the administrator can edit messages of other users and pin messages; applicable to channels only.
  bool can_edit_messages_;
  /// True, if the administrator can delete messages of other users.
  bool can_delete_messages_;
  /// True, if the administrator can invite new users to the chat.
  bool can_invite_users_;
  /// True, if the administrator can restrict, ban, or unban chat members; always true for channels.
  bool can_restrict_members_;
  /// True, if the administrator can pin messages; applicable to basic groups and supergroups only.
  bool can_pin_messages_;
  /// True, if the administrator can add new administrators with a subset of their own privileges or demote administrators that were directly or indirectly promoted by them.
  bool can_promote_members_;
  /// True, if the administrator can manage video chats.
  bool can_manage_video_chats_;
  /// True, if the administrator isn&\#039;t shown in the chat member list and sends messages anonymously; applicable to supergroups only.
  bool is_anonymous_;

  /**
   * The user is a member of the chat and has some additional privileges. In basic groups, administrators can edit and delete messages sent by others, add new members, ban unprivileged members, and manage video chats. In supergroups and channels, there are more detailed options for administrator privileges.
   */
  chatMemberStatusAdministrator();

  /**
   * The user is a member of the chat and has some additional privileges. In basic groups, administrators can edit and delete messages sent by others, add new members, ban unprivileged members, and manage video chats. In supergroups and channels, there are more detailed options for administrator privileges.
   *
   * \param[in] custom_title_ A custom title of the administrator; 0-16 characters without emojis; applicable to supergroups only.
   * \param[in] can_be_edited_ True, if the current user can edit the administrator privileges for the called user.
   * \param[in] can_manage_chat_ True, if the administrator can get chat event log, get chat statistics, get message statistics in channels, get channel members, see anonymous administrators in supergroups and ignore slow mode. Implied by any other privilege; applicable to supergroups and channels only.
   * \param[in] can_change_info_ True, if the administrator can change the chat title, photo, and other settings.
   * \param[in] can_post_messages_ True, if the administrator can create channel posts; applicable to channels only.
   * \param[in] can_edit_messages_ True, if the administrator can edit messages of other users and pin messages; applicable to channels only.
   * \param[in] can_delete_messages_ True, if the administrator can delete messages of other users.
   * \param[in] can_invite_users_ True, if the administrator can invite new users to the chat.
   * \param[in] can_restrict_members_ True, if the administrator can restrict, ban, or unban chat members; always true for channels.
   * \param[in] can_pin_messages_ True, if the administrator can pin messages; applicable to basic groups and supergroups only.
   * \param[in] can_promote_members_ True, if the administrator can add new administrators with a subset of their own privileges or demote administrators that were directly or indirectly promoted by them.
   * \param[in] can_manage_video_chats_ True, if the administrator can manage video chats.
   * \param[in] is_anonymous_ True, if the administrator isn&\#039;t shown in the chat member list and sends messages anonymously; applicable to supergroups only.
   */
  chatMemberStatusAdministrator(string const &custom_title_, bool can_be_edited_, bool can_manage_chat_, bool can_change_info_, bool can_post_messages_, bool can_edit_messages_, bool can_delete_messages_, bool can_invite_users_, bool can_restrict_members_, bool can_pin_messages_, bool can_promote_members_, bool can_manage_video_chats_, bool is_anonymous_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 82243562;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The user is a member of the chat, without any additional privileges or restrictions.
 */
class chatMemberStatusMember final : public ChatMemberStatus {
 public:

  /**
   * The user is a member of the chat, without any additional privileges or restrictions.
   */
  chatMemberStatusMember();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 844723285;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The user is under certain restrictions in the chat. Not supported in basic groups and channels.
 */
class chatMemberStatusRestricted final : public ChatMemberStatus {
 public:
  /// True, if the user is a member of the chat.
  bool is_member_;
  /// Point in time (Unix timestamp) when restrictions will be lifted from the user; 0 if never. If the user is restricted for more than 366 days or for less than 30 seconds from the current time, the user is considered to be restricted forever.
  int32 restricted_until_date_;
  /// User permissions in the chat.
  object_ptr<chatPermissions> permissions_;

  /**
   * The user is under certain restrictions in the chat. Not supported in basic groups and channels.
   */
  chatMemberStatusRestricted();

  /**
   * The user is under certain restrictions in the chat. Not supported in basic groups and channels.
   *
   * \param[in] is_member_ True, if the user is a member of the chat.
   * \param[in] restricted_until_date_ Point in time (Unix timestamp) when restrictions will be lifted from the user; 0 if never. If the user is restricted for more than 366 days or for less than 30 seconds from the current time, the user is considered to be restricted forever.
   * \param[in] permissions_ User permissions in the chat.
   */
  chatMemberStatusRestricted(bool is_member_, int32 restricted_until_date_, object_ptr<chatPermissions> &&permissions_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1661432998;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The user or the chat is not a chat member.
 */
class chatMemberStatusLeft final : public ChatMemberStatus {
 public:

  /**
   * The user or the chat is not a chat member.
   */
  chatMemberStatusLeft();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -5815259;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The user or the chat was banned (and hence is not a member of the chat). Implies the user can&\#039;t return to the chat, view messages, or be used as a participant identifier to join a video chat of the chat.
 */
class chatMemberStatusBanned final : public ChatMemberStatus {
 public:
  /// Point in time (Unix timestamp) when the user will be unbanned; 0 if never. If the user is banned for more than 366 days or for less than 30 seconds from the current time, the user is considered to be banned forever. Always 0 in basic groups.
  int32 banned_until_date_;

  /**
   * The user or the chat was banned (and hence is not a member of the chat). Implies the user can&\#039;t return to the chat, view messages, or be used as a participant identifier to join a video chat of the chat.
   */
  chatMemberStatusBanned();

  /**
   * The user or the chat was banned (and hence is not a member of the chat). Implies the user can&\#039;t return to the chat, view messages, or be used as a participant identifier to join a video chat of the chat.
   *
   * \param[in] banned_until_date_ Point in time (Unix timestamp) when the user will be unbanned; 0 if never. If the user is banned for more than 366 days or for less than 30 seconds from the current time, the user is considered to be banned forever. Always 0 in basic groups.
   */
  explicit chatMemberStatusBanned(int32 banned_until_date_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1653518666;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains a list of chat members.
 */
class chatMembers final : public Object {
 public:
  /// Approximate total count of chat members found.
  int32 total_count_;
  /// A list of chat members.
  array<object_ptr<chatMember>> members_;

  /**
   * Contains a list of chat members.
   */
  chatMembers();

  /**
   * Contains a list of chat members.
   *
   * \param[in] total_count_ Approximate total count of chat members found.
   * \param[in] members_ A list of chat members.
   */
  chatMembers(int32 total_count_, array<object_ptr<chatMember>> &&members_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -497558622;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Specifies the kind of chat members to return in searchChatMembers.
 */
class ChatMembersFilter: public Object {
 public:
};

/**
 * Returns contacts of the user.
 */
class chatMembersFilterContacts final : public ChatMembersFilter {
 public:

  /**
   * Returns contacts of the user.
   */
  chatMembersFilterContacts();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1774485671;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Returns the owner and administrators.
 */
class chatMembersFilterAdministrators final : public ChatMembersFilter {
 public:

  /**
   * Returns the owner and administrators.
   */
  chatMembersFilterAdministrators();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1266893796;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Returns all chat members, including restricted chat members.
 */
class chatMembersFilterMembers final : public ChatMembersFilter {
 public:

  /**
   * Returns all chat members, including restricted chat members.
   */
  chatMembersFilterMembers();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 670504342;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Returns users which can be mentioned in the chat.
 */
class chatMembersFilterMention final : public ChatMembersFilter {
 public:
  /// If non-zero, the identifier of the current message thread.
  int53 message_thread_id_;

  /**
   * Returns users which can be mentioned in the chat.
   */
  chatMembersFilterMention();

  /**
   * Returns users which can be mentioned in the chat.
   *
   * \param[in] message_thread_id_ If non-zero, the identifier of the current message thread.
   */
  explicit chatMembersFilterMention(int53 message_thread_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 856419831;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Returns users under certain restrictions in the chat; can be used only by administrators in a supergroup.
 */
class chatMembersFilterRestricted final : public ChatMembersFilter {
 public:

  /**
   * Returns users under certain restrictions in the chat; can be used only by administrators in a supergroup.
   */
  chatMembersFilterRestricted();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1256282813;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Returns users banned from the chat; can be used only by administrators in a supergroup or in a channel.
 */
class chatMembersFilterBanned final : public ChatMembersFilter {
 public:

  /**
   * Returns users banned from the chat; can be used only by administrators in a supergroup or in a channel.
   */
  chatMembersFilterBanned();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1863102648;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Returns bot members of the chat.
 */
class chatMembersFilterBots final : public ChatMembersFilter {
 public:

  /**
   * Returns bot members of the chat.
   */
  chatMembersFilterBots();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1422567288;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Describes a chat located nearby.
 */
class chatNearby final : public Object {
 public:
  /// Chat identifier.
  int53 chat_id_;
  /// Distance to the chat location, in meters.
  int32 distance_;

  /**
   * Describes a chat located nearby.
   */
  chatNearby();

  /**
   * Describes a chat located nearby.
   *
   * \param[in] chat_id_ Chat identifier.
   * \param[in] distance_ Distance to the chat location, in meters.
   */
  chatNearby(int53 chat_id_, int32 distance_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 48120405;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains information about notification settings for a chat.
 */
class chatNotificationSettings final : public Object {
 public:
  /// If true, mute_for is ignored and the value for the relevant type of chat is used instead.
  bool use_default_mute_for_;
  /// Time left before notifications will be unmuted, in seconds.
  int32 mute_for_;
  /// If true, sound is ignored and the value for the relevant type of chat is used instead.
  bool use_default_sound_;
  /// The name of an audio file to be used for notification sounds; only applies to iOS applications.
  string sound_;
  /// If true, show_preview is ignored and the value for the relevant type of chat is used instead.
  bool use_default_show_preview_;
  /// True, if message content must be displayed in notifications.
  bool show_preview_;
  /// If true, disable_pinned_message_notifications is ignored and the value for the relevant type of chat is used instead.
  bool use_default_disable_pinned_message_notifications_;
  /// If true, notifications for incoming pinned messages will be created as for an ordinary unread message.
  bool disable_pinned_message_notifications_;
  /// If true, disable_mention_notifications is ignored and the value for the relevant type of chat is used instead.
  bool use_default_disable_mention_notifications_;
  /// If true, notifications for messages with mentions will be created as for an ordinary unread message.
  bool disable_mention_notifications_;

  /**
   * Contains information about notification settings for a chat.
   */
  chatNotificationSettings();

  /**
   * Contains information about notification settings for a chat.
   *
   * \param[in] use_default_mute_for_ If true, mute_for is ignored and the value for the relevant type of chat is used instead.
   * \param[in] mute_for_ Time left before notifications will be unmuted, in seconds.
   * \param[in] use_default_sound_ If true, sound is ignored and the value for the relevant type of chat is used instead.
   * \param[in] sound_ The name of an audio file to be used for notification sounds; only applies to iOS applications.
   * \param[in] use_default_show_preview_ If true, show_preview is ignored and the value for the relevant type of chat is used instead.
   * \param[in] show_preview_ True, if message content must be displayed in notifications.
   * \param[in] use_default_disable_pinned_message_notifications_ If true, disable_pinned_message_notifications is ignored and the value for the relevant type of chat is used instead.
   * \param[in] disable_pinned_message_notifications_ If true, notifications for incoming pinned messages will be created as for an ordinary unread message.
   * \param[in] use_default_disable_mention_notifications_ If true, disable_mention_notifications is ignored and the value for the relevant type of chat is used instead.
   * \param[in] disable_mention_notifications_ If true, notifications for messages with mentions will be created as for an ordinary unread message.
   */
  chatNotificationSettings(bool use_default_mute_for_, int32 mute_for_, bool use_default_sound_, string const &sound_, bool use_default_show_preview_, bool show_preview_, bool use_default_disable_pinned_message_notifications_, bool disable_pinned_message_notifications_, bool use_default_disable_mention_notifications_, bool disable_mention_notifications_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1503183218;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Describes actions that a user is allowed to take in a chat.
 */
class chatPermissions final : public Object {
 public:
  /// True, if the user can send text messages, contacts, locations, and venues.
  bool can_send_messages_;
  /// True, if the user can send audio files, documents, photos, videos, video notes, and voice notes. Implies can_send_messages permissions.
  bool can_send_media_messages_;
  /// True, if the user can send polls. Implies can_send_messages permissions.
  bool can_send_polls_;
  /// True, if the user can send animations, games, stickers, and dice and use inline bots. Implies can_send_messages permissions.
  bool can_send_other_messages_;
  /// True, if the user may add a web page preview to their messages. Implies can_send_messages permissions.
  bool can_add_web_page_previews_;
  /// True, if the user can change the chat title, photo, and other settings.
  bool can_change_info_;
  /// True, if the user can invite new users to the chat.
  bool can_invite_users_;
  /// True, if the user can pin messages.
  bool can_pin_messages_;

  /**
   * Describes actions that a user is allowed to take in a chat.
   */
  chatPermissions();

  /**
   * Describes actions that a user is allowed to take in a chat.
   *
   * \param[in] can_send_messages_ True, if the user can send text messages, contacts, locations, and venues.
   * \param[in] can_send_media_messages_ True, if the user can send audio files, documents, photos, videos, video notes, and voice notes. Implies can_send_messages permissions.
   * \param[in] can_send_polls_ True, if the user can send polls. Implies can_send_messages permissions.
   * \param[in] can_send_other_messages_ True, if the user can send animations, games, stickers, and dice and use inline bots. Implies can_send_messages permissions.
   * \param[in] can_add_web_page_previews_ True, if the user may add a web page preview to their messages. Implies can_send_messages permissions.
   * \param[in] can_change_info_ True, if the user can change the chat title, photo, and other settings.
   * \param[in] can_invite_users_ True, if the user can invite new users to the chat.
   * \param[in] can_pin_messages_ True, if the user can pin messages.
   */
  chatPermissions(bool can_send_messages_, bool can_send_media_messages_, bool can_send_polls_, bool can_send_other_messages_, bool can_add_web_page_previews_, bool can_change_info_, bool can_invite_users_, bool can_pin_messages_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1584650463;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Describes a chat or user profile photo.
 */
class chatPhoto final : public Object {
 public:
  /// Unique photo identifier.
  int64 id_;
  /// Point in time (Unix timestamp) when the photo has been added.
  int32 added_date_;
  /// Photo minithumbnail; may be null.
  object_ptr<minithumbnail> minithumbnail_;
  /// Available variants of the photo in JPEG format, in different size.
  array<object_ptr<photoSize>> sizes_;
  /// Animated variant of the photo in MPEG4 format; may be null.
  object_ptr<animatedChatPhoto> animation_;

  /**
   * Describes a chat or user profile photo.
   */
  chatPhoto();

  /**
   * Describes a chat or user profile photo.
   *
   * \param[in] id_ Unique photo identifier.
   * \param[in] added_date_ Point in time (Unix timestamp) when the photo has been added.
   * \param[in] minithumbnail_ Photo minithumbnail; may be null.
   * \param[in] sizes_ Available variants of the photo in JPEG format, in different size.
   * \param[in] animation_ Animated variant of the photo in MPEG4 format; may be null.
   */
  chatPhoto(int64 id_, int32 added_date_, object_ptr<minithumbnail> &&minithumbnail_, array<object_ptr<photoSize>> &&sizes_, object_ptr<animatedChatPhoto> &&animation_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -113003577;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains basic information about the photo of a chat.
 */
class chatPhotoInfo final : public Object {
 public:
  /// A small (160x160) chat photo variant in JPEG format. The file can be downloaded only before the photo is changed.
  object_ptr<file> small_;
  /// A big (640x640) chat photo variant in JPEG format. The file can be downloaded only before the photo is changed.
  object_ptr<file> big_;
  /// Chat photo minithumbnail; may be null.
  object_ptr<minithumbnail> minithumbnail_;
  /// True, if the photo has animated variant.
  bool has_animation_;

  /**
   * Contains basic information about the photo of a chat.
   */
  chatPhotoInfo();

  /**
   * Contains basic information about the photo of a chat.
   *
   * \param[in] small_ A small (160x160) chat photo variant in JPEG format. The file can be downloaded only before the photo is changed.
   * \param[in] big_ A big (640x640) chat photo variant in JPEG format. The file can be downloaded only before the photo is changed.
   * \param[in] minithumbnail_ Chat photo minithumbnail; may be null.
   * \param[in] has_animation_ True, if the photo has animated variant.
   */
  chatPhotoInfo(object_ptr<file> &&small_, object_ptr<file> &&big_, object_ptr<minithumbnail> &&minithumbnail_, bool has_animation_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 167058358;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains a list of chat or user profile photos.
 */
class chatPhotos final : public Object {
 public:
  /// Total number of photos.
  int32 total_count_;
  /// List of photos.
  array<object_ptr<chatPhoto>> photos_;

  /**
   * Contains a list of chat or user profile photos.
   */
  chatPhotos();

  /**
   * Contains a list of chat or user profile photos.
   *
   * \param[in] total_count_ Total number of photos.
   * \param[in] photos_ List of photos.
   */
  chatPhotos(int32 total_count_, array<object_ptr<chatPhoto>> &&photos_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1510699180;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Describes a position of a chat in a chat list.
 */
class chatPosition final : public Object {
 public:
  /// The chat list.
  object_ptr<ChatList> list_;
  /// A parameter used to determine order of the chat in the chat list. Chats must be sorted by the pair (order, chat.id) in descending order.
  int64 order_;
  /// True, if the chat is pinned in the chat list.
  bool is_pinned_;
  /// Source of the chat in the chat list; may be null.
  object_ptr<ChatSource> source_;

  /**
   * Describes a position of a chat in a chat list.
   */
  chatPosition();

  /**
   * Describes a position of a chat in a chat list.
   *
   * \param[in] list_ The chat list.
   * \param[in] order_ A parameter used to determine order of the chat in the chat list. Chats must be sorted by the pair (order, chat.id) in descending order.
   * \param[in] is_pinned_ True, if the chat is pinned in the chat list.
   * \param[in] source_ Source of the chat in the chat list; may be null.
   */
  chatPosition(object_ptr<ChatList> &&list_, int64 order_, bool is_pinned_, object_ptr<ChatSource> &&source_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -622557355;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Describes the reason why a chat is reported.
 */
class ChatReportReason: public Object {
 public:
};

/**
 * The chat contains spam messages.
 */
class chatReportReasonSpam final : public ChatReportReason {
 public:

  /**
   * The chat contains spam messages.
   */
  chatReportReasonSpam();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -510848863;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The chat promotes violence.
 */
class chatReportReasonViolence final : public ChatReportReason {
 public:

  /**
   * The chat promotes violence.
   */
  chatReportReasonViolence();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1330235395;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The chat contains pornographic messages.
 */
class chatReportReasonPornography final : public ChatReportReason {
 public:

  /**
   * The chat contains pornographic messages.
   */
  chatReportReasonPornography();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 722614385;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The chat has child abuse related content.
 */
class chatReportReasonChildAbuse final : public ChatReportReason {
 public:

  /**
   * The chat has child abuse related content.
   */
  chatReportReasonChildAbuse();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1070686531;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The chat contains copyrighted content.
 */
class chatReportReasonCopyright final : public ChatReportReason {
 public:

  /**
   * The chat contains copyrighted content.
   */
  chatReportReasonCopyright();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 986898080;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The location-based chat is unrelated to its stated location.
 */
class chatReportReasonUnrelatedLocation final : public ChatReportReason {
 public:

  /**
   * The location-based chat is unrelated to its stated location.
   */
  chatReportReasonUnrelatedLocation();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 2632403;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The chat represents a fake account.
 */
class chatReportReasonFake final : public ChatReportReason {
 public:

  /**
   * The chat represents a fake account.
   */
  chatReportReasonFake();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1713230446;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A custom reason provided by the user.
 */
class chatReportReasonCustom final : public ChatReportReason {
 public:

  /**
   * A custom reason provided by the user.
   */
  chatReportReasonCustom();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1288925974;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Describes a reason why an external chat is shown in a chat list.
 */
class ChatSource: public Object {
 public:
};

/**
 * The chat is sponsored by the user&\#039;s MTProxy server.
 */
class chatSourceMtprotoProxy final : public ChatSource {
 public:

  /**
   * The chat is sponsored by the user&\#039;s MTProxy server.
   */
  chatSourceMtprotoProxy();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 394074115;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The chat contains a public service announcement.
 */
class chatSourcePublicServiceAnnouncement final : public ChatSource {
 public:
  /// The type of the announcement.
  string type_;
  /// The text of the announcement.
  string text_;

  /**
   * The chat contains a public service announcement.
   */
  chatSourcePublicServiceAnnouncement();

  /**
   * The chat contains a public service announcement.
   *
   * \param[in] type_ The type of the announcement.
   * \param[in] text_ The text of the announcement.
   */
  chatSourcePublicServiceAnnouncement(string const &type_, string const &text_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -328571244;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Contains a detailed statistics about a chat.
 */
class ChatStatistics: public Object {
 public:
};

/**
 * A detailed statistics about a supergroup chat.
 */
class chatStatisticsSupergroup final : public ChatStatistics {
 public:
  /// A period to which the statistics applies.
  object_ptr<dateRange> period_;
  /// Number of members in the chat.
  object_ptr<statisticalValue> member_count_;
  /// Number of messages sent to the chat.
  object_ptr<statisticalValue> message_count_;
  /// Number of users who viewed messages in the chat.
  object_ptr<statisticalValue> viewer_count_;
  /// Number of users who sent messages to the chat.
  object_ptr<statisticalValue> sender_count_;
  /// A graph containing number of members in the chat.
  object_ptr<StatisticalGraph> member_count_graph_;
  /// A graph containing number of members joined and left the chat.
  object_ptr<StatisticalGraph> join_graph_;
  /// A graph containing number of new member joins per source.
  object_ptr<StatisticalGraph> join_by_source_graph_;
  /// A graph containing distribution of active users per language.
  object_ptr<StatisticalGraph> language_graph_;
  /// A graph containing distribution of sent messages by content type.
  object_ptr<StatisticalGraph> message_content_graph_;
  /// A graph containing number of different actions in the chat.
  object_ptr<StatisticalGraph> action_graph_;
  /// A graph containing distribution of message views per hour.
  object_ptr<StatisticalGraph> day_graph_;
  /// A graph containing distribution of message views per day of week.
  object_ptr<StatisticalGraph> week_graph_;
  /// List of users sent most messages in the last week.
  array<object_ptr<chatStatisticsMessageSenderInfo>> top_senders_;
  /// List of most active administrators in the last week.
  array<object_ptr<chatStatisticsAdministratorActionsInfo>> top_administrators_;
  /// List of most active inviters of new members in the last week.
  array<object_ptr<chatStatisticsInviterInfo>> top_inviters_;

  /**
   * A detailed statistics about a supergroup chat.
   */
  chatStatisticsSupergroup();

  /**
   * A detailed statistics about a supergroup chat.
   *
   * \param[in] period_ A period to which the statistics applies.
   * \param[in] member_count_ Number of members in the chat.
   * \param[in] message_count_ Number of messages sent to the chat.
   * \param[in] viewer_count_ Number of users who viewed messages in the chat.
   * \param[in] sender_count_ Number of users who sent messages to the chat.
   * \param[in] member_count_graph_ A graph containing number of members in the chat.
   * \param[in] join_graph_ A graph containing number of members joined and left the chat.
   * \param[in] join_by_source_graph_ A graph containing number of new member joins per source.
   * \param[in] language_graph_ A graph containing distribution of active users per language.
   * \param[in] message_content_graph_ A graph containing distribution of sent messages by content type.
   * \param[in] action_graph_ A graph containing number of different actions in the chat.
   * \param[in] day_graph_ A graph containing distribution of message views per hour.
   * \param[in] week_graph_ A graph containing distribution of message views per day of week.
   * \param[in] top_senders_ List of users sent most messages in the last week.
   * \param[in] top_administrators_ List of most active administrators in the last week.
   * \param[in] top_inviters_ List of most active inviters of new members in the last week.
   */
  chatStatisticsSupergroup(object_ptr<dateRange> &&period_, object_ptr<statisticalValue> &&member_count_, object_ptr<statisticalValue> &&message_count_, object_ptr<statisticalValue> &&viewer_count_, object_ptr<statisticalValue> &&sender_count_, object_ptr<StatisticalGraph> &&member_count_graph_, object_ptr<StatisticalGraph> &&join_graph_, object_ptr<StatisticalGraph> &&join_by_source_graph_, object_ptr<StatisticalGraph> &&language_graph_, object_ptr<StatisticalGraph> &&message_content_graph_, object_ptr<StatisticalGraph> &&action_graph_, object_ptr<StatisticalGraph> &&day_graph_, object_ptr<StatisticalGraph> &&week_graph_, array<object_ptr<chatStatisticsMessageSenderInfo>> &&top_senders_, array<object_ptr<chatStatisticsAdministratorActionsInfo>> &&top_administrators_, array<object_ptr<chatStatisticsInviterInfo>> &&top_inviters_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -17244633;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A detailed statistics about a channel chat.
 */
class chatStatisticsChannel final : public ChatStatistics {
 public:
  /// A period to which the statistics applies.
  object_ptr<dateRange> period_;
  /// Number of members in the chat.
  object_ptr<statisticalValue> member_count_;
  /// Mean number of times the recently sent messages was viewed.
  object_ptr<statisticalValue> mean_view_count_;
  /// Mean number of times the recently sent messages was shared.
  object_ptr<statisticalValue> mean_share_count_;
  /// A percentage of users with enabled notifications for the chat.
  double enabled_notifications_percentage_;
  /// A graph containing number of members in the chat.
  object_ptr<StatisticalGraph> member_count_graph_;
  /// A graph containing number of members joined and left the chat.
  object_ptr<StatisticalGraph> join_graph_;
  /// A graph containing number of members muted and unmuted the chat.
  object_ptr<StatisticalGraph> mute_graph_;
  /// A graph containing number of message views in a given hour in the last two weeks.
  object_ptr<StatisticalGraph> view_count_by_hour_graph_;
  /// A graph containing number of message views per source.
  object_ptr<StatisticalGraph> view_count_by_source_graph_;
  /// A graph containing number of new member joins per source.
  object_ptr<StatisticalGraph> join_by_source_graph_;
  /// A graph containing number of users viewed chat messages per language.
  object_ptr<StatisticalGraph> language_graph_;
  /// A graph containing number of chat message views and shares.
  object_ptr<StatisticalGraph> message_interaction_graph_;
  /// A graph containing number of views of associated with the chat instant views.
  object_ptr<StatisticalGraph> instant_view_interaction_graph_;
  /// Detailed statistics about number of views and shares of recently sent messages.
  array<object_ptr<chatStatisticsMessageInteractionInfo>> recent_message_interactions_;

  /**
   * A detailed statistics about a channel chat.
   */
  chatStatisticsChannel();

  /**
   * A detailed statistics about a channel chat.
   *
   * \param[in] period_ A period to which the statistics applies.
   * \param[in] member_count_ Number of members in the chat.
   * \param[in] mean_view_count_ Mean number of times the recently sent messages was viewed.
   * \param[in] mean_share_count_ Mean number of times the recently sent messages was shared.
   * \param[in] enabled_notifications_percentage_ A percentage of users with enabled notifications for the chat.
   * \param[in] member_count_graph_ A graph containing number of members in the chat.
   * \param[in] join_graph_ A graph containing number of members joined and left the chat.
   * \param[in] mute_graph_ A graph containing number of members muted and unmuted the chat.
   * \param[in] view_count_by_hour_graph_ A graph containing number of message views in a given hour in the last two weeks.
   * \param[in] view_count_by_source_graph_ A graph containing number of message views per source.
   * \param[in] join_by_source_graph_ A graph containing number of new member joins per source.
   * \param[in] language_graph_ A graph containing number of users viewed chat messages per language.
   * \param[in] message_interaction_graph_ A graph containing number of chat message views and shares.
   * \param[in] instant_view_interaction_graph_ A graph containing number of views of associated with the chat instant views.
   * \param[in] recent_message_interactions_ Detailed statistics about number of views and shares of recently sent messages.
   */
  chatStatisticsChannel(object_ptr<dateRange> &&period_, object_ptr<statisticalValue> &&member_count_, object_ptr<statisticalValue> &&mean_view_count_, object_ptr<statisticalValue> &&mean_share_count_, double enabled_notifications_percentage_, object_ptr<StatisticalGraph> &&member_count_graph_, object_ptr<StatisticalGraph> &&join_graph_, object_ptr<StatisticalGraph> &&mute_graph_, object_ptr<StatisticalGraph> &&view_count_by_hour_graph_, object_ptr<StatisticalGraph> &&view_count_by_source_graph_, object_ptr<StatisticalGraph> &&join_by_source_graph_, object_ptr<StatisticalGraph> &&language_graph_, object_ptr<StatisticalGraph> &&message_interaction_graph_, object_ptr<StatisticalGraph> &&instant_view_interaction_graph_, array<object_ptr<chatStatisticsMessageInteractionInfo>> &&recent_message_interactions_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -825434183;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains statistics about administrator actions done by a user.
 */
class chatStatisticsAdministratorActionsInfo final : public Object {
 public:
  /// Administrator user identifier.
  int53 user_id_;
  /// Number of messages deleted by the administrator.
  int32 deleted_message_count_;
  /// Number of users banned by the administrator.
  int32 banned_user_count_;
  /// Number of users restricted by the administrator.
  int32 restricted_user_count_;

  /**
   * Contains statistics about administrator actions done by a user.
   */
  chatStatisticsAdministratorActionsInfo();

  /**
   * Contains statistics about administrator actions done by a user.
   *
   * \param[in] user_id_ Administrator user identifier.
   * \param[in] deleted_message_count_ Number of messages deleted by the administrator.
   * \param[in] banned_user_count_ Number of users banned by the administrator.
   * \param[in] restricted_user_count_ Number of users restricted by the administrator.
   */
  chatStatisticsAdministratorActionsInfo(int53 user_id_, int32 deleted_message_count_, int32 banned_user_count_, int32 restricted_user_count_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -406467202;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains statistics about number of new members invited by a user.
 */
class chatStatisticsInviterInfo final : public Object {
 public:
  /// User identifier.
  int53 user_id_;
  /// Number of new members invited by the user.
  int32 added_member_count_;

  /**
   * Contains statistics about number of new members invited by a user.
   */
  chatStatisticsInviterInfo();

  /**
   * Contains statistics about number of new members invited by a user.
   *
   * \param[in] user_id_ User identifier.
   * \param[in] added_member_count_ Number of new members invited by the user.
   */
  chatStatisticsInviterInfo(int53 user_id_, int32 added_member_count_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 629396619;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains statistics about interactions with a message.
 */
class chatStatisticsMessageInteractionInfo final : public Object {
 public:
  /// Message identifier.
  int53 message_id_;
  /// Number of times the message was viewed.
  int32 view_count_;
  /// Number of times the message was forwarded.
  int32 forward_count_;

  /**
   * Contains statistics about interactions with a message.
   */
  chatStatisticsMessageInteractionInfo();

  /**
   * Contains statistics about interactions with a message.
   *
   * \param[in] message_id_ Message identifier.
   * \param[in] view_count_ Number of times the message was viewed.
   * \param[in] forward_count_ Number of times the message was forwarded.
   */
  chatStatisticsMessageInteractionInfo(int53 message_id_, int32 view_count_, int32 forward_count_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -765580756;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains statistics about messages sent by a user.
 */
class chatStatisticsMessageSenderInfo final : public Object {
 public:
  /// User identifier.
  int53 user_id_;
  /// Number of sent messages.
  int32 sent_message_count_;
  /// Average number of characters in sent messages; 0 if unknown.
  int32 average_character_count_;

  /**
   * Contains statistics about messages sent by a user.
   */
  chatStatisticsMessageSenderInfo();

  /**
   * Contains statistics about messages sent by a user.
   *
   * \param[in] user_id_ User identifier.
   * \param[in] sent_message_count_ Number of sent messages.
   * \param[in] average_character_count_ Average number of characters in sent messages; 0 if unknown.
   */
  chatStatisticsMessageSenderInfo(int53 user_id_, int32 sent_message_count_, int32 average_character_count_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1762295371;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Describes a chat theme.
 */
class chatTheme final : public Object {
 public:
  /// Theme name.
  string name_;
  /// Theme settings for a light chat theme.
  object_ptr<themeSettings> light_settings_;
  /// Theme settings for a dark chat theme.
  object_ptr<themeSettings> dark_settings_;

  /**
   * Describes a chat theme.
   */
  chatTheme();

  /**
   * Describes a chat theme.
   *
   * \param[in] name_ Theme name.
   * \param[in] light_settings_ Theme settings for a light chat theme.
   * \param[in] dark_settings_ Theme settings for a dark chat theme.
   */
  chatTheme(string const &name_, object_ptr<themeSettings> &&light_settings_, object_ptr<themeSettings> &&dark_settings_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -113218503;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Describes the type of a chat.
 */
class ChatType: public Object {
 public:
};

/**
 * An ordinary chat with a user.
 */
class chatTypePrivate final : public ChatType {
 public:
  /// User identifier.
  int53 user_id_;

  /**
   * An ordinary chat with a user.
   */
  chatTypePrivate();

  /**
   * An ordinary chat with a user.
   *
   * \param[in] user_id_ User identifier.
   */
  explicit chatTypePrivate(int53 user_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1579049844;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A basic group (a chat with 0-200 other users).
 */
class chatTypeBasicGroup final : public ChatType {
 public:
  /// Basic group identifier.
  int53 basic_group_id_;

  /**
   * A basic group (a chat with 0-200 other users).
   */
  chatTypeBasicGroup();

  /**
   * A basic group (a chat with 0-200 other users).
   *
   * \param[in] basic_group_id_ Basic group identifier.
   */
  explicit chatTypeBasicGroup(int53 basic_group_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 973884508;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A supergroup or channel (with unlimited members).
 */
class chatTypeSupergroup final : public ChatType {
 public:
  /// Supergroup or channel identifier.
  int53 supergroup_id_;
  /// True, if the supergroup is a channel.
  bool is_channel_;

  /**
   * A supergroup or channel (with unlimited members).
   */
  chatTypeSupergroup();

  /**
   * A supergroup or channel (with unlimited members).
   *
   * \param[in] supergroup_id_ Supergroup or channel identifier.
   * \param[in] is_channel_ True, if the supergroup is a channel.
   */
  chatTypeSupergroup(int53 supergroup_id_, bool is_channel_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1472570774;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A secret chat with a user.
 */
class chatTypeSecret final : public ChatType {
 public:
  /// Secret chat identifier.
  int32 secret_chat_id_;
  /// User identifier of the secret chat peer.
  int53 user_id_;

  /**
   * A secret chat with a user.
   */
  chatTypeSecret();

  /**
   * A secret chat with a user.
   *
   * \param[in] secret_chat_id_ Secret chat identifier.
   * \param[in] user_id_ User identifier of the secret chat peer.
   */
  chatTypeSecret(int32 secret_chat_id_, int53 user_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 862366513;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Represents a list of chats.
 */
class chats final : public Object {
 public:
  /// Approximate total count of chats found.
  int32 total_count_;
  /// List of chat identifiers.
  array<int53> chat_ids_;

  /**
   * Represents a list of chats.
   */
  chats();

  /**
   * Represents a list of chats.
   *
   * \param[in] total_count_ Approximate total count of chats found.
   * \param[in] chat_ids_ List of chat identifiers.
   */
  chats(int32 total_count_, array<int53> &&chat_ids_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1809654812;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Represents a list of chats located nearby.
 */
class chatsNearby final : public Object {
 public:
  /// List of users nearby.
  array<object_ptr<chatNearby>> users_nearby_;
  /// List of location-based supergroups nearby.
  array<object_ptr<chatNearby>> supergroups_nearby_;

  /**
   * Represents a list of chats located nearby.
   */
  chatsNearby();

  /**
   * Represents a list of chats located nearby.
   *
   * \param[in] users_nearby_ List of users nearby.
   * \param[in] supergroups_nearby_ List of location-based supergroups nearby.
   */
  chatsNearby(array<object_ptr<chatNearby>> &&users_nearby_, array<object_ptr<chatNearby>> &&supergroups_nearby_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 187746081;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Represents result of checking whether a username can be set for a chat.
 */
class CheckChatUsernameResult: public Object {
 public:
};

/**
 * The username can be set.
 */
class checkChatUsernameResultOk final : public CheckChatUsernameResult {
 public:

  /**
   * The username can be set.
   */
  checkChatUsernameResultOk();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1498956964;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The username is invalid.
 */
class checkChatUsernameResultUsernameInvalid final : public CheckChatUsernameResult {
 public:

  /**
   * The username is invalid.
   */
  checkChatUsernameResultUsernameInvalid();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -636979370;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The username is occupied.
 */
class checkChatUsernameResultUsernameOccupied final : public CheckChatUsernameResult {
 public:

  /**
   * The username is occupied.
   */
  checkChatUsernameResultUsernameOccupied();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1320892201;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The user has too much chats with username, one of them must be made private first.
 */
class checkChatUsernameResultPublicChatsTooMuch final : public CheckChatUsernameResult {
 public:

  /**
   * The user has too much chats with username, one of them must be made private first.
   */
  checkChatUsernameResultPublicChatsTooMuch();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 858247741;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The user can&\#039;t be a member of a public supergroup.
 */
class checkChatUsernameResultPublicGroupsUnavailable final : public CheckChatUsernameResult {
 public:

  /**
   * The user can&\#039;t be a member of a public supergroup.
   */
  checkChatUsernameResultPublicGroupsUnavailable();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -51833641;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Represents result of checking whether a name can be used for a new sticker set.
 */
class CheckStickerSetNameResult: public Object {
 public:
};

/**
 * The name can be set.
 */
class checkStickerSetNameResultOk final : public CheckStickerSetNameResult {
 public:

  /**
   * The name can be set.
   */
  checkStickerSetNameResultOk();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1404308904;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The name is invalid.
 */
class checkStickerSetNameResultNameInvalid final : public CheckStickerSetNameResult {
 public:

  /**
   * The name is invalid.
   */
  checkStickerSetNameResultNameInvalid();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 177992244;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The name is occupied.
 */
class checkStickerSetNameResultNameOccupied final : public CheckStickerSetNameResult {
 public:

  /**
   * The name is occupied.
   */
  checkStickerSetNameResultNameOccupied();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1012980872;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Represents a closed vector path. The path begins at the end point of the last command.
 */
class closedVectorPath final : public Object {
 public:
  /// List of vector path commands.
  array<object_ptr<VectorPathCommand>> commands_;

  /**
   * Represents a closed vector path. The path begins at the end point of the last command.
   */
  closedVectorPath();

  /**
   * Represents a closed vector path. The path begins at the end point of the last command.
   *
   * \param[in] commands_ List of vector path commands.
   */
  explicit closedVectorPath(array<object_ptr<VectorPathCommand>> &&commands_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 589951657;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains information about one website the current user is logged in with Telegram.
 */
class connectedWebsite final : public Object {
 public:
  /// Website identifier.
  int64 id_;
  /// The domain name of the website.
  string domain_name_;
  /// User identifier of a bot linked with the website.
  int53 bot_user_id_;
  /// The version of a browser used to log in.
  string browser_;
  /// Operating system the browser is running on.
  string platform_;
  /// Point in time (Unix timestamp) when the user was logged in.
  int32 log_in_date_;
  /// Point in time (Unix timestamp) when obtained authorization was last used.
  int32 last_active_date_;
  /// IP address from which the user was logged in, in human-readable format.
  string ip_;
  /// Human-readable description of a country and a region, from which the user was logged in, based on the IP address.
  string location_;

  /**
   * Contains information about one website the current user is logged in with Telegram.
   */
  connectedWebsite();

  /**
   * Contains information about one website the current user is logged in with Telegram.
   *
   * \param[in] id_ Website identifier.
   * \param[in] domain_name_ The domain name of the website.
   * \param[in] bot_user_id_ User identifier of a bot linked with the website.
   * \param[in] browser_ The version of a browser used to log in.
   * \param[in] platform_ Operating system the browser is running on.
   * \param[in] log_in_date_ Point in time (Unix timestamp) when the user was logged in.
   * \param[in] last_active_date_ Point in time (Unix timestamp) when obtained authorization was last used.
   * \param[in] ip_ IP address from which the user was logged in, in human-readable format.
   * \param[in] location_ Human-readable description of a country and a region, from which the user was logged in, based on the IP address.
   */
  connectedWebsite(int64 id_, string const &domain_name_, int53 bot_user_id_, string const &browser_, string const &platform_, int32 log_in_date_, int32 last_active_date_, string const &ip_, string const &location_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 844014445;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains a list of websites the current user is logged in with Telegram.
 */
class connectedWebsites final : public Object {
 public:
  /// List of connected websites.
  array<object_ptr<connectedWebsite>> websites_;

  /**
   * Contains a list of websites the current user is logged in with Telegram.
   */
  connectedWebsites();

  /**
   * Contains a list of websites the current user is logged in with Telegram.
   *
   * \param[in] websites_ List of connected websites.
   */
  explicit connectedWebsites(array<object_ptr<connectedWebsite>> &&websites_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1727949694;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Describes the current state of the connection to Telegram servers.
 */
class ConnectionState: public Object {
 public:
};

/**
 * Currently waiting for the network to become available. Use setNetworkType to change the available network type.
 */
class connectionStateWaitingForNetwork final : public ConnectionState {
 public:

  /**
   * Currently waiting for the network to become available. Use setNetworkType to change the available network type.
   */
  connectionStateWaitingForNetwork();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1695405912;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Currently establishing a connection with a proxy server.
 */
class connectionStateConnectingToProxy final : public ConnectionState {
 public:

  /**
   * Currently establishing a connection with a proxy server.
   */
  connectionStateConnectingToProxy();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -93187239;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Currently establishing a connection to the Telegram servers.
 */
class connectionStateConnecting final : public ConnectionState {
 public:

  /**
   * Currently establishing a connection to the Telegram servers.
   */
  connectionStateConnecting();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1298400670;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Downloading data received while the application was offline.
 */
class connectionStateUpdating final : public ConnectionState {
 public:

  /**
   * Downloading data received while the application was offline.
   */
  connectionStateUpdating();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -188104009;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * There is a working connection to the Telegram servers.
 */
class connectionStateReady final : public ConnectionState {
 public:

  /**
   * There is a working connection to the Telegram servers.
   */
  connectionStateReady();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 48608492;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Describes a user contact.
 */
class contact final : public Object {
 public:
  /// Phone number of the user.
  string phone_number_;
  /// First name of the user; 1-255 characters in length.
  string first_name_;
  /// Last name of the user.
  string last_name_;
  /// Additional data about the user in a form of vCard; 0-2048 bytes in length.
  string vcard_;
  /// Identifier of the user, if known; otherwise 0.
  int53 user_id_;

  /**
   * Describes a user contact.
   */
  contact();

  /**
   * Describes a user contact.
   *
   * \param[in] phone_number_ Phone number of the user.
   * \param[in] first_name_ First name of the user; 1-255 characters in length.
   * \param[in] last_name_ Last name of the user.
   * \param[in] vcard_ Additional data about the user in a form of vCard; 0-2048 bytes in length.
   * \param[in] user_id_ Identifier of the user, if known; otherwise 0.
   */
  contact(string const &phone_number_, string const &first_name_, string const &last_name_, string const &vcard_, int53 user_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1993844876;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains a counter.
 */
class count final : public Object {
 public:
  /// Count.
  int32 count_;

  /**
   * Contains a counter.
   */
  count();

  /**
   * Contains a counter.
   *
   * \param[in] count_ Count.
   */
  explicit count(int32 count_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1295577348;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains information about countries.
 */
class countries final : public Object {
 public:
  /// The list of countries.
  array<object_ptr<countryInfo>> countries_;

  /**
   * Contains information about countries.
   */
  countries();

  /**
   * Contains information about countries.
   *
   * \param[in] countries_ The list of countries.
   */
  explicit countries(array<object_ptr<countryInfo>> &&countries_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1854211813;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains information about a country.
 */
class countryInfo final : public Object {
 public:
  /// A two-letter ISO 3166-1 alpha-2 country code.
  string country_code_;
  /// Native name of the country.
  string name_;
  /// English name of the country.
  string english_name_;
  /// True, if the country must be hidden from the list of all countries.
  bool is_hidden_;
  /// List of country calling codes.
  array<string> calling_codes_;

  /**
   * Contains information about a country.
   */
  countryInfo();

  /**
   * Contains information about a country.
   *
   * \param[in] country_code_ A two-letter ISO 3166-1 alpha-2 country code.
   * \param[in] name_ Native name of the country.
   * \param[in] english_name_ English name of the country.
   * \param[in] is_hidden_ True, if the country must be hidden from the list of all countries.
   * \param[in] calling_codes_ List of country calling codes.
   */
  countryInfo(string const &country_code_, string const &name_, string const &english_name_, bool is_hidden_, array<string> &&calling_codes_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1617195722;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains the result of a custom request.
 */
class customRequestResult final : public Object {
 public:
  /// A JSON-serialized result.
  string result_;

  /**
   * Contains the result of a custom request.
   */
  customRequestResult();

  /**
   * Contains the result of a custom request.
   *
   * \param[in] result_ A JSON-serialized result.
   */
  explicit customRequestResult(string const &result_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -2009960452;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains database statistics.
 */
class databaseStatistics final : public Object {
 public:
  /// Database statistics in an unspecified human-readable format.
  string statistics_;

  /**
   * Contains database statistics.
   */
  databaseStatistics();

  /**
   * Contains database statistics.
   *
   * \param[in] statistics_ Database statistics in an unspecified human-readable format.
   */
  explicit databaseStatistics(string const &statistics_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1123912880;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Represents a date according to the Gregorian calendar.
 */
class date final : public Object {
 public:
  /// Day of the month; 1-31.
  int32 day_;
  /// Month; 1-12.
  int32 month_;
  /// Year; 1-9999.
  int32 year_;

  /**
   * Represents a date according to the Gregorian calendar.
   */
  date();

  /**
   * Represents a date according to the Gregorian calendar.
   *
   * \param[in] day_ Day of the month; 1-31.
   * \param[in] month_ Month; 1-12.
   * \param[in] year_ Year; 1-9999.
   */
  date(int32 day_, int32 month_, int32 year_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -277956960;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Represents a date range.
 */
class dateRange final : public Object {
 public:
  /// Point in time (Unix timestamp) at which the date range begins.
  int32 start_date_;
  /// Point in time (Unix timestamp) at which the date range ends.
  int32 end_date_;

  /**
   * Represents a date range.
   */
  dateRange();

  /**
   * Represents a date range.
   *
   * \param[in] start_date_ Point in time (Unix timestamp) at which the date range begins.
   * \param[in] end_date_ Point in time (Unix timestamp) at which the date range ends.
   */
  dateRange(int32 start_date_, int32 end_date_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1360333926;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * File with the date it was uploaded.
 */
class datedFile final : public Object {
 public:
  /// The file.
  object_ptr<file> file_;
  /// Point in time (Unix timestamp) when the file was uploaded.
  int32 date_;

  /**
   * File with the date it was uploaded.
   */
  datedFile();

  /**
   * File with the date it was uploaded.
   *
   * \param[in] file_ The file.
   * \param[in] date_ Point in time (Unix timestamp) when the file was uploaded.
   */
  datedFile(object_ptr<file> &&file_, int32 date_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1840795491;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains information about a tg: deep link.
 */
class deepLinkInfo final : public Object {
 public:
  /// Text to be shown to the user.
  object_ptr<formattedText> text_;
  /// True, if the user must be asked to update the application.
  bool need_update_application_;

  /**
   * Contains information about a tg: deep link.
   */
  deepLinkInfo();

  /**
   * Contains information about a tg: deep link.
   *
   * \param[in] text_ Text to be shown to the user.
   * \param[in] need_update_application_ True, if the user must be asked to update the application.
   */
  deepLinkInfo(object_ptr<formattedText> &&text_, bool need_update_application_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1864081662;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Represents a data needed to subscribe for push notifications through registerDevice method. To use specific push notification service, the correct application platform must be specified and a valid server authentication data must be uploaded at https://my.telegram.org.
 */
class DeviceToken: public Object {
 public:
};

/**
 * A token for Firebase Cloud Messaging.
 */
class deviceTokenFirebaseCloudMessaging final : public DeviceToken {
 public:
  /// Device registration token; may be empty to deregister a device.
  string token_;
  /// True, if push notifications must be additionally encrypted.
  bool encrypt_;

  /**
   * A token for Firebase Cloud Messaging.
   */
  deviceTokenFirebaseCloudMessaging();

  /**
   * A token for Firebase Cloud Messaging.
   *
   * \param[in] token_ Device registration token; may be empty to deregister a device.
   * \param[in] encrypt_ True, if push notifications must be additionally encrypted.
   */
  deviceTokenFirebaseCloudMessaging(string const &token_, bool encrypt_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -797881849;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A token for Apple Push Notification service.
 */
class deviceTokenApplePush final : public DeviceToken {
 public:
  /// Device token; may be empty to deregister a device.
  string device_token_;
  /// True, if App Sandbox is enabled.
  bool is_app_sandbox_;

  /**
   * A token for Apple Push Notification service.
   */
  deviceTokenApplePush();

  /**
   * A token for Apple Push Notification service.
   *
   * \param[in] device_token_ Device token; may be empty to deregister a device.
   * \param[in] is_app_sandbox_ True, if App Sandbox is enabled.
   */
  deviceTokenApplePush(string const &device_token_, bool is_app_sandbox_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 387541955;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A token for Apple Push Notification service VoIP notifications.
 */
class deviceTokenApplePushVoIP final : public DeviceToken {
 public:
  /// Device token; may be empty to deregister a device.
  string device_token_;
  /// True, if App Sandbox is enabled.
  bool is_app_sandbox_;
  /// True, if push notifications must be additionally encrypted.
  bool encrypt_;

  /**
   * A token for Apple Push Notification service VoIP notifications.
   */
  deviceTokenApplePushVoIP();

  /**
   * A token for Apple Push Notification service VoIP notifications.
   *
   * \param[in] device_token_ Device token; may be empty to deregister a device.
   * \param[in] is_app_sandbox_ True, if App Sandbox is enabled.
   * \param[in] encrypt_ True, if push notifications must be additionally encrypted.
   */
  deviceTokenApplePushVoIP(string const &device_token_, bool is_app_sandbox_, bool encrypt_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 804275689;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A token for Windows Push Notification Services.
 */
class deviceTokenWindowsPush final : public DeviceToken {
 public:
  /// The access token that will be used to send notifications; may be empty to deregister a device.
  string access_token_;

  /**
   * A token for Windows Push Notification Services.
   */
  deviceTokenWindowsPush();

  /**
   * A token for Windows Push Notification Services.
   *
   * \param[in] access_token_ The access token that will be used to send notifications; may be empty to deregister a device.
   */
  explicit deviceTokenWindowsPush(string const &access_token_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1410514289;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A token for Microsoft Push Notification Service.
 */
class deviceTokenMicrosoftPush final : public DeviceToken {
 public:
  /// Push notification channel URI; may be empty to deregister a device.
  string channel_uri_;

  /**
   * A token for Microsoft Push Notification Service.
   */
  deviceTokenMicrosoftPush();

  /**
   * A token for Microsoft Push Notification Service.
   *
   * \param[in] channel_uri_ Push notification channel URI; may be empty to deregister a device.
   */
  explicit deviceTokenMicrosoftPush(string const &channel_uri_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1224269900;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A token for Microsoft Push Notification Service VoIP channel.
 */
class deviceTokenMicrosoftPushVoIP final : public DeviceToken {
 public:
  /// Push notification channel URI; may be empty to deregister a device.
  string channel_uri_;

  /**
   * A token for Microsoft Push Notification Service VoIP channel.
   */
  deviceTokenMicrosoftPushVoIP();

  /**
   * A token for Microsoft Push Notification Service VoIP channel.
   *
   * \param[in] channel_uri_ Push notification channel URI; may be empty to deregister a device.
   */
  explicit deviceTokenMicrosoftPushVoIP(string const &channel_uri_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -785603759;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A token for web Push API.
 */
class deviceTokenWebPush final : public DeviceToken {
 public:
  /// Absolute URL exposed by the push service where the application server can send push messages; may be empty to deregister a device.
  string endpoint_;
  /// Base64url-encoded P-256 elliptic curve Diffie-Hellman public key.
  string p256dh_base64url_;
  /// Base64url-encoded authentication secret.
  string auth_base64url_;

  /**
   * A token for web Push API.
   */
  deviceTokenWebPush();

  /**
   * A token for web Push API.
   *
   * \param[in] endpoint_ Absolute URL exposed by the push service where the application server can send push messages; may be empty to deregister a device.
   * \param[in] p256dh_base64url_ Base64url-encoded P-256 elliptic curve Diffie-Hellman public key.
   * \param[in] auth_base64url_ Base64url-encoded authentication secret.
   */
  deviceTokenWebPush(string const &endpoint_, string const &p256dh_base64url_, string const &auth_base64url_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1694507273;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A token for Simple Push API for Firefox OS.
 */
class deviceTokenSimplePush final : public DeviceToken {
 public:
  /// Absolute URL exposed by the push service where the application server can send push messages; may be empty to deregister a device.
  string endpoint_;

  /**
   * A token for Simple Push API for Firefox OS.
   */
  deviceTokenSimplePush();

  /**
   * A token for Simple Push API for Firefox OS.
   *
   * \param[in] endpoint_ Absolute URL exposed by the push service where the application server can send push messages; may be empty to deregister a device.
   */
  explicit deviceTokenSimplePush(string const &endpoint_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 49584736;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A token for Ubuntu Push Client service.
 */
class deviceTokenUbuntuPush final : public DeviceToken {
 public:
  /// Token; may be empty to deregister a device.
  string token_;

  /**
   * A token for Ubuntu Push Client service.
   */
  deviceTokenUbuntuPush();

  /**
   * A token for Ubuntu Push Client service.
   *
   * \param[in] token_ Token; may be empty to deregister a device.
   */
  explicit deviceTokenUbuntuPush(string const &token_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1782320422;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A token for BlackBerry Push Service.
 */
class deviceTokenBlackBerryPush final : public DeviceToken {
 public:
  /// Token; may be empty to deregister a device.
  string token_;

  /**
   * A token for BlackBerry Push Service.
   */
  deviceTokenBlackBerryPush();

  /**
   * A token for BlackBerry Push Service.
   *
   * \param[in] token_ Token; may be empty to deregister a device.
   */
  explicit deviceTokenBlackBerryPush(string const &token_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1559167234;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A token for Tizen Push Service.
 */
class deviceTokenTizenPush final : public DeviceToken {
 public:
  /// Push service registration identifier; may be empty to deregister a device.
  string reg_id_;

  /**
   * A token for Tizen Push Service.
   */
  deviceTokenTizenPush();

  /**
   * A token for Tizen Push Service.
   *
   * \param[in] reg_id_ Push service registration identifier; may be empty to deregister a device.
   */
  explicit deviceTokenTizenPush(string const &reg_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1359947213;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Contains animated stickers which must be used for dice animation rendering.
 */
class DiceStickers: public Object {
 public:
};

/**
 * A regular animated sticker.
 */
class diceStickersRegular final : public DiceStickers {
 public:
  /// The animated sticker with the dice animation.
  object_ptr<sticker> sticker_;

  /**
   * A regular animated sticker.
   */
  diceStickersRegular();

  /**
   * A regular animated sticker.
   *
   * \param[in] sticker_ The animated sticker with the dice animation.
   */
  explicit diceStickersRegular(object_ptr<sticker> &&sticker_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -740299570;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Animated stickers to be combined into a slot machine.
 */
class diceStickersSlotMachine final : public DiceStickers {
 public:
  /// The animated sticker with the slot machine background. The background animation must start playing after all reel animations finish.
  object_ptr<sticker> background_;
  /// The animated sticker with the lever animation. The lever animation must play once in the initial dice state.
  object_ptr<sticker> lever_;
  /// The animated sticker with the left reel.
  object_ptr<sticker> left_reel_;
  /// The animated sticker with the center reel.
  object_ptr<sticker> center_reel_;
  /// The animated sticker with the right reel.
  object_ptr<sticker> right_reel_;

  /**
   * Animated stickers to be combined into a slot machine.
   */
  diceStickersSlotMachine();

  /**
   * Animated stickers to be combined into a slot machine.
   *
   * \param[in] background_ The animated sticker with the slot machine background. The background animation must start playing after all reel animations finish.
   * \param[in] lever_ The animated sticker with the lever animation. The lever animation must play once in the initial dice state.
   * \param[in] left_reel_ The animated sticker with the left reel.
   * \param[in] center_reel_ The animated sticker with the center reel.
   * \param[in] right_reel_ The animated sticker with the right reel.
   */
  diceStickersSlotMachine(object_ptr<sticker> &&background_, object_ptr<sticker> &&lever_, object_ptr<sticker> &&left_reel_, object_ptr<sticker> &&center_reel_, object_ptr<sticker> &&right_reel_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -375223124;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Describes a document of any type.
 */
class document final : public Object {
 public:
  /// Original name of the file; as defined by the sender.
  string file_name_;
  /// MIME type of the file; as defined by the sender.
  string mime_type_;
  /// Document minithumbnail; may be null.
  object_ptr<minithumbnail> minithumbnail_;
  /// Document thumbnail in JPEG or PNG format (PNG will be used only for background patterns); as defined by the sender; may be null.
  object_ptr<thumbnail> thumbnail_;
  /// File containing the document.
  object_ptr<file> document_;

  /**
   * Describes a document of any type.
   */
  document();

  /**
   * Describes a document of any type.
   *
   * \param[in] file_name_ Original name of the file; as defined by the sender.
   * \param[in] mime_type_ MIME type of the file; as defined by the sender.
   * \param[in] minithumbnail_ Document minithumbnail; may be null.
   * \param[in] thumbnail_ Document thumbnail in JPEG or PNG format (PNG will be used only for background patterns); as defined by the sender; may be null.
   * \param[in] document_ File containing the document.
   */
  document(string const &file_name_, string const &mime_type_, object_ptr<minithumbnail> &&minithumbnail_, object_ptr<thumbnail> &&thumbnail_, object_ptr<file> &&document_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1357271080;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains information about a message draft.
 */
class draftMessage final : public Object {
 public:
  /// Identifier of the message to reply to; 0 if none.
  int53 reply_to_message_id_;
  /// Point in time (Unix timestamp) when the draft was created.
  int32 date_;
  /// Content of the message draft; must be of the type inputMessageText.
  object_ptr<InputMessageContent> input_message_text_;

  /**
   * Contains information about a message draft.
   */
  draftMessage();

  /**
   * Contains information about a message draft.
   *
   * \param[in] reply_to_message_id_ Identifier of the message to reply to; 0 if none.
   * \param[in] date_ Point in time (Unix timestamp) when the draft was created.
   * \param[in] input_message_text_ Content of the message draft; must be of the type inputMessageText.
   */
  draftMessage(int53 reply_to_message_id_, int32 date_, object_ptr<InputMessageContent> &&input_message_text_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1373050112;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Information about the email address authentication code that was sent.
 */
class emailAddressAuthenticationCodeInfo final : public Object {
 public:
  /// Pattern of the email address to which an authentication code was sent.
  string email_address_pattern_;
  /// Length of the code; 0 if unknown.
  int32 length_;

  /**
   * Information about the email address authentication code that was sent.
   */
  emailAddressAuthenticationCodeInfo();

  /**
   * Information about the email address authentication code that was sent.
   *
   * \param[in] email_address_pattern_ Pattern of the email address to which an authentication code was sent.
   * \param[in] length_ Length of the code; 0 if unknown.
   */
  emailAddressAuthenticationCodeInfo(string const &email_address_pattern_, int32 length_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1151066659;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Represents a list of emoji.
 */
class emojis final : public Object {
 public:
  /// List of emojis.
  array<string> emojis_;

  /**
   * Represents a list of emoji.
   */
  emojis();

  /**
   * Represents a list of emoji.
   *
   * \param[in] emojis_ List of emojis.
   */
  explicit emojis(array<string> &&emojis_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 950339552;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains encrypted Telegram Passport data credentials.
 */
class encryptedCredentials final : public Object {
 public:
  /// The encrypted credentials.
  bytes data_;
  /// The decrypted data hash.
  bytes hash_;
  /// Secret for data decryption, encrypted with the service&\#039;s public key.
  bytes secret_;

  /**
   * Contains encrypted Telegram Passport data credentials.
   */
  encryptedCredentials();

  /**
   * Contains encrypted Telegram Passport data credentials.
   *
   * \param[in] data_ The encrypted credentials.
   * \param[in] hash_ The decrypted data hash.
   * \param[in] secret_ Secret for data decryption, encrypted with the service&\#039;s public key.
   */
  encryptedCredentials(bytes const &data_, bytes const &hash_, bytes const &secret_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1331106766;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains information about an encrypted Telegram Passport element; for bots only.
 */
class encryptedPassportElement final : public Object {
 public:
  /// Type of Telegram Passport element.
  object_ptr<PassportElementType> type_;
  /// Encrypted JSON-encoded data about the user.
  bytes data_;
  /// The front side of an identity document.
  object_ptr<datedFile> front_side_;
  /// The reverse side of an identity document; may be null.
  object_ptr<datedFile> reverse_side_;
  /// Selfie with the document; may be null.
  object_ptr<datedFile> selfie_;
  /// List of files containing a certified English translation of the document.
  array<object_ptr<datedFile>> translation_;
  /// List of attached files.
  array<object_ptr<datedFile>> files_;
  /// Unencrypted data, phone number or email address.
  string value_;
  /// Hash of the entire element.
  string hash_;

  /**
   * Contains information about an encrypted Telegram Passport element; for bots only.
   */
  encryptedPassportElement();

  /**
   * Contains information about an encrypted Telegram Passport element; for bots only.
   *
   * \param[in] type_ Type of Telegram Passport element.
   * \param[in] data_ Encrypted JSON-encoded data about the user.
   * \param[in] front_side_ The front side of an identity document.
   * \param[in] reverse_side_ The reverse side of an identity document; may be null.
   * \param[in] selfie_ Selfie with the document; may be null.
   * \param[in] translation_ List of files containing a certified English translation of the document.
   * \param[in] files_ List of attached files.
   * \param[in] value_ Unencrypted data, phone number or email address.
   * \param[in] hash_ Hash of the entire element.
   */
  encryptedPassportElement(object_ptr<PassportElementType> &&type_, bytes const &data_, object_ptr<datedFile> &&front_side_, object_ptr<datedFile> &&reverse_side_, object_ptr<datedFile> &&selfie_, array<object_ptr<datedFile>> &&translation_, array<object_ptr<datedFile>> &&files_, string const &value_, string const &hash_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 2002386193;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * An object of this type can be returned on every function call, in case of an error.
 */
class error final : public Object {
 public:
  /// Error code; subject to future changes. If the error code is 406, the error message must not be processed in any way and must not be displayed to the user.
  int32 code_;
  /// Error message; subject to future changes.
  string message_;

  /**
   * An object of this type can be returned on every function call, in case of an error.
   */
  error();

  /**
   * An object of this type can be returned on every function call, in case of an error.
   *
   * \param[in] code_ Error code; subject to future changes. If the error code is 406, the error message must not be processed in any way and must not be displayed to the user.
   * \param[in] message_ Error message; subject to future changes.
   */
  error(int32 code_, string const &message_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1679978726;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Represents a file.
 */
class file final : public Object {
 public:
  /// Unique file identifier.
  int32 id_;
  /// File size, in bytes; 0 if unknown.
  int32 size_;
  /// Approximate file size in bytes in case the exact file size is unknown. Can be used to show download/upload progress.
  int32 expected_size_;
  /// Information about the local copy of the file.
  object_ptr<localFile> local_;
  /// Information about the remote copy of the file.
  object_ptr<remoteFile> remote_;

  /**
   * Represents a file.
   */
  file();

  /**
   * Represents a file.
   *
   * \param[in] id_ Unique file identifier.
   * \param[in] size_ File size, in bytes; 0 if unknown.
   * \param[in] expected_size_ Approximate file size in bytes in case the exact file size is unknown. Can be used to show download/upload progress.
   * \param[in] local_ Information about the local copy of the file.
   * \param[in] remote_ Information about the remote copy of the file.
   */
  file(int32 id_, int32 size_, int32 expected_size_, object_ptr<localFile> &&local_, object_ptr<remoteFile> &&remote_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 766337656;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains a part of a file.
 */
class filePart final : public Object {
 public:
  /// File bytes.
  bytes data_;

  /**
   * Contains a part of a file.
   */
  filePart();

  /**
   * Contains a part of a file.
   *
   * \param[in] data_ File bytes.
   */
  explicit filePart(bytes const &data_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 911821878;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Represents the type of a file.
 */
class FileType: public Object {
 public:
};

/**
 * The data is not a file.
 */
class fileTypeNone final : public FileType {
 public:

  /**
   * The data is not a file.
   */
  fileTypeNone();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 2003009189;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The file is an animation.
 */
class fileTypeAnimation final : public FileType {
 public:

  /**
   * The file is an animation.
   */
  fileTypeAnimation();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -290816582;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The file is an audio file.
 */
class fileTypeAudio final : public FileType {
 public:

  /**
   * The file is an audio file.
   */
  fileTypeAudio();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -709112160;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The file is a document.
 */
class fileTypeDocument final : public FileType {
 public:

  /**
   * The file is a document.
   */
  fileTypeDocument();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -564722929;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The file is a photo.
 */
class fileTypePhoto final : public FileType {
 public:

  /**
   * The file is a photo.
   */
  fileTypePhoto();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1718914651;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The file is a profile photo.
 */
class fileTypeProfilePhoto final : public FileType {
 public:

  /**
   * The file is a profile photo.
   */
  fileTypeProfilePhoto();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1795089315;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The file was sent to a secret chat (the file type is not known to the server).
 */
class fileTypeSecret final : public FileType {
 public:

  /**
   * The file was sent to a secret chat (the file type is not known to the server).
   */
  fileTypeSecret();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1871899401;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The file is a thumbnail of a file from a secret chat.
 */
class fileTypeSecretThumbnail final : public FileType {
 public:

  /**
   * The file is a thumbnail of a file from a secret chat.
   */
  fileTypeSecretThumbnail();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1401326026;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The file is a file from Secure storage used for storing Telegram Passport files.
 */
class fileTypeSecure final : public FileType {
 public:

  /**
   * The file is a file from Secure storage used for storing Telegram Passport files.
   */
  fileTypeSecure();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1419133146;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The file is a sticker.
 */
class fileTypeSticker final : public FileType {
 public:

  /**
   * The file is a sticker.
   */
  fileTypeSticker();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 475233385;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The file is a thumbnail of another file.
 */
class fileTypeThumbnail final : public FileType {
 public:

  /**
   * The file is a thumbnail of another file.
   */
  fileTypeThumbnail();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -12443298;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The file type is not yet known.
 */
class fileTypeUnknown final : public FileType {
 public:

  /**
   * The file type is not yet known.
   */
  fileTypeUnknown();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -2011566768;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The file is a video.
 */
class fileTypeVideo final : public FileType {
 public:

  /**
   * The file is a video.
   */
  fileTypeVideo();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1430816539;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The file is a video note.
 */
class fileTypeVideoNote final : public FileType {
 public:

  /**
   * The file is a video note.
   */
  fileTypeVideoNote();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -518412385;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The file is a voice note.
 */
class fileTypeVoiceNote final : public FileType {
 public:

  /**
   * The file is a voice note.
   */
  fileTypeVoiceNote();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -588681661;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The file is a wallpaper or a background pattern.
 */
class fileTypeWallpaper final : public FileType {
 public:

  /**
   * The file is a wallpaper or a background pattern.
   */
  fileTypeWallpaper();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1854930076;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A text with some entities.
 */
class formattedText final : public Object {
 public:
  /// The text.
  string text_;
  /// Entities contained in the text. Entities can be nested, but must not mutually intersect with each other. Pre, Code and PreCode entities can&\#039;t contain other entities. Bold, Italic, Underline and Strikethrough entities can contain and to be contained in all other entities. All other entities can&\#039;t contain each other.
  array<object_ptr<textEntity>> entities_;

  /**
   * A text with some entities.
   */
  formattedText();

  /**
   * A text with some entities.
   *
   * \param[in] text_ The text.
   * \param[in] entities_ Entities contained in the text. Entities can be nested, but must not mutually intersect with each other. Pre, Code and PreCode entities can&\#039;t contain other entities. Bold, Italic, Underline and Strikethrough entities can contain and to be contained in all other entities. All other entities can&\#039;t contain each other.
   */
  formattedText(string const &text_, array<object_ptr<textEntity>> &&entities_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -252624564;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains a list of messages found by a search.
 */
class foundMessages final : public Object {
 public:
  /// Approximate total count of messages found; -1 if unknown.
  int32 total_count_;
  /// List of messages.
  array<object_ptr<message>> messages_;
  /// The offset for the next request. If empty, there are no more results.
  string next_offset_;

  /**
   * Contains a list of messages found by a search.
   */
  foundMessages();

  /**
   * Contains a list of messages found by a search.
   *
   * \param[in] total_count_ Approximate total count of messages found; -1 if unknown.
   * \param[in] messages_ List of messages.
   * \param[in] next_offset_ The offset for the next request. If empty, there are no more results.
   */
  foundMessages(int32 total_count_, array<object_ptr<message>> &&messages_, string const &next_offset_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -529809608;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Describes a game.
 */
class game final : public Object {
 public:
  /// Game ID.
  int64 id_;
  /// Game short name. To share a game use the URL https://t.me/{bot_username}?game={game_short_name}.
  string short_name_;
  /// Game title.
  string title_;
  /// Game text, usually containing scoreboards for a game.
  object_ptr<formattedText> text_;
  /// Game description.
  string description_;
  /// Game photo.
  object_ptr<photo> photo_;
  /// Game animation; may be null.
  object_ptr<animation> animation_;

  /**
   * Describes a game.
   */
  game();

  /**
   * Describes a game.
   *
   * \param[in] id_ Game ID.
   * \param[in] short_name_ Game short name. To share a game use the URL https://t.me/{bot_username}?game={game_short_name}.
   * \param[in] title_ Game title.
   * \param[in] text_ Game text, usually containing scoreboards for a game.
   * \param[in] description_ Game description.
   * \param[in] photo_ Game photo.
   * \param[in] animation_ Game animation; may be null.
   */
  game(int64 id_, string const &short_name_, string const &title_, object_ptr<formattedText> &&text_, string const &description_, object_ptr<photo> &&photo_, object_ptr<animation> &&animation_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1565597752;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains one row of the game high score table.
 */
class gameHighScore final : public Object {
 public:
  /// Position in the high score table.
  int32 position_;
  /// User identifier.
  int53 user_id_;
  /// User score.
  int32 score_;

  /**
   * Contains one row of the game high score table.
   */
  gameHighScore();

  /**
   * Contains one row of the game high score table.
   *
   * \param[in] position_ Position in the high score table.
   * \param[in] user_id_ User identifier.
   * \param[in] score_ User score.
   */
  gameHighScore(int32 position_, int53 user_id_, int32 score_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 342871838;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains a list of game high scores.
 */
class gameHighScores final : public Object {
 public:
  /// A list of game high scores.
  array<object_ptr<gameHighScore>> scores_;

  /**
   * Contains a list of game high scores.
   */
  gameHighScores();

  /**
   * Contains a list of game high scores.
   *
   * \param[in] scores_ A list of game high scores.
   */
  explicit gameHighScores(array<object_ptr<gameHighScore>> &&scores_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -725770727;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Describes a group call.
 */
class groupCall final : public Object {
 public:
  /// Group call identifier.
  int32 id_;
  /// Group call title.
  string title_;
  /// Point in time (Unix timestamp) when the group call is supposed to be started by an administrator; 0 if it is already active or was ended.
  int32 scheduled_start_date_;
  /// True, if the group call is scheduled and the current user will receive a notification when the group call will start.
  bool enabled_start_notification_;
  /// True, if the call is active.
  bool is_active_;
  /// True, if the call is joined.
  bool is_joined_;
  /// True, if user was kicked from the call because of network loss and the call needs to be rejoined.
  bool need_rejoin_;
  /// True, if the current user can manage the group call.
  bool can_be_managed_;
  /// Number of participants in the group call.
  int32 participant_count_;
  /// True, if all group call participants are loaded.
  bool loaded_all_participants_;
  /// At most 3 recently speaking users in the group call.
  array<object_ptr<groupCallRecentSpeaker>> recent_speakers_;
  /// True, if the current user&\#039;s video is enabled.
  bool is_my_video_enabled_;
  /// True, if the current user&\#039;s video is paused.
  bool is_my_video_paused_;
  /// True, if the current user can broadcast video or share screen.
  bool can_enable_video_;
  /// True, if only group call administrators can unmute new participants.
  bool mute_new_participants_;
  /// True, if the current user can enable or disable mute_new_participants setting.
  bool can_toggle_mute_new_participants_;
  /// Duration of the ongoing group call recording, in seconds; 0 if none. An updateGroupCall update is not triggered when value of this field changes, but the same recording goes on.
  int32 record_duration_;
  /// True, if a video file is being recorded for the call.
  bool is_video_recorded_;
  /// Call duration, in seconds; for ended calls only.
  int32 duration_;

  /**
   * Describes a group call.
   */
  groupCall();

  /**
   * Describes a group call.
   *
   * \param[in] id_ Group call identifier.
   * \param[in] title_ Group call title.
   * \param[in] scheduled_start_date_ Point in time (Unix timestamp) when the group call is supposed to be started by an administrator; 0 if it is already active or was ended.
   * \param[in] enabled_start_notification_ True, if the group call is scheduled and the current user will receive a notification when the group call will start.
   * \param[in] is_active_ True, if the call is active.
   * \param[in] is_joined_ True, if the call is joined.
   * \param[in] need_rejoin_ True, if user was kicked from the call because of network loss and the call needs to be rejoined.
   * \param[in] can_be_managed_ True, if the current user can manage the group call.
   * \param[in] participant_count_ Number of participants in the group call.
   * \param[in] loaded_all_participants_ True, if all group call participants are loaded.
   * \param[in] recent_speakers_ At most 3 recently speaking users in the group call.
   * \param[in] is_my_video_enabled_ True, if the current user&\#039;s video is enabled.
   * \param[in] is_my_video_paused_ True, if the current user&\#039;s video is paused.
   * \param[in] can_enable_video_ True, if the current user can broadcast video or share screen.
   * \param[in] mute_new_participants_ True, if only group call administrators can unmute new participants.
   * \param[in] can_toggle_mute_new_participants_ True, if the current user can enable or disable mute_new_participants setting.
   * \param[in] record_duration_ Duration of the ongoing group call recording, in seconds; 0 if none. An updateGroupCall update is not triggered when value of this field changes, but the same recording goes on.
   * \param[in] is_video_recorded_ True, if a video file is being recorded for the call.
   * \param[in] duration_ Call duration, in seconds; for ended calls only.
   */
  groupCall(int32 id_, string const &title_, int32 scheduled_start_date_, bool enabled_start_notification_, bool is_active_, bool is_joined_, bool need_rejoin_, bool can_be_managed_, int32 participant_count_, bool loaded_all_participants_, array<object_ptr<groupCallRecentSpeaker>> &&recent_speakers_, bool is_my_video_enabled_, bool is_my_video_paused_, bool can_enable_video_, bool mute_new_participants_, bool can_toggle_mute_new_participants_, int32 record_duration_, bool is_video_recorded_, int32 duration_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1548892209;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains the group call identifier.
 */
class groupCallId final : public Object {
 public:
  /// Group call identifier.
  int32 id_;

  /**
   * Contains the group call identifier.
   */
  groupCallId();

  /**
   * Contains the group call identifier.
   *
   * \param[in] id_ Group call identifier.
   */
  explicit groupCallId(int32 id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 350534469;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Represents a group call participant.
 */
class groupCallParticipant final : public Object {
 public:
  /// Identifier of the group call participant.
  object_ptr<MessageSender> participant_id_;
  /// User&\#039;s audio channel synchronization source identifier.
  int32 audio_source_id_;
  /// User&\#039;s screen sharing audio channel synchronization source identifier.
  int32 screen_sharing_audio_source_id_;
  /// Information about user&\#039;s video channel; may be null if there is no active video.
  object_ptr<groupCallParticipantVideoInfo> video_info_;
  /// Information about user&\#039;s screen sharing video channel; may be null if there is no active screen sharing video.
  object_ptr<groupCallParticipantVideoInfo> screen_sharing_video_info_;
  /// The participant user&\#039;s bio or the participant chat&\#039;s description.
  string bio_;
  /// True, if the participant is the current user.
  bool is_current_user_;
  /// True, if the participant is speaking as set by setGroupCallParticipantIsSpeaking.
  bool is_speaking_;
  /// True, if the participant hand is raised.
  bool is_hand_raised_;
  /// True, if the current user can mute the participant for all other group call participants.
  bool can_be_muted_for_all_users_;
  /// True, if the current user can allow the participant to unmute themselves or unmute the participant (if the participant is the current user).
  bool can_be_unmuted_for_all_users_;
  /// True, if the current user can mute the participant only for self.
  bool can_be_muted_for_current_user_;
  /// True, if the current user can unmute the participant for self.
  bool can_be_unmuted_for_current_user_;
  /// True, if the participant is muted for all users.
  bool is_muted_for_all_users_;
  /// True, if the participant is muted for the current user.
  bool is_muted_for_current_user_;
  /// True, if the participant is muted for all users, but can unmute themselves.
  bool can_unmute_self_;
  /// Participant&\#039;s volume level; 1-20000 in hundreds of percents.
  int32 volume_level_;
  /// User&\#039;s order in the group call participant list. Orders must be compared lexicographically. The bigger is order, the higher is user in the list. If order is empty, the user must be removed from the participant list.
  string order_;

  /**
   * Represents a group call participant.
   */
  groupCallParticipant();

  /**
   * Represents a group call participant.
   *
   * \param[in] participant_id_ Identifier of the group call participant.
   * \param[in] audio_source_id_ User&\#039;s audio channel synchronization source identifier.
   * \param[in] screen_sharing_audio_source_id_ User&\#039;s screen sharing audio channel synchronization source identifier.
   * \param[in] video_info_ Information about user&\#039;s video channel; may be null if there is no active video.
   * \param[in] screen_sharing_video_info_ Information about user&\#039;s screen sharing video channel; may be null if there is no active screen sharing video.
   * \param[in] bio_ The participant user&\#039;s bio or the participant chat&\#039;s description.
   * \param[in] is_current_user_ True, if the participant is the current user.
   * \param[in] is_speaking_ True, if the participant is speaking as set by setGroupCallParticipantIsSpeaking.
   * \param[in] is_hand_raised_ True, if the participant hand is raised.
   * \param[in] can_be_muted_for_all_users_ True, if the current user can mute the participant for all other group call participants.
   * \param[in] can_be_unmuted_for_all_users_ True, if the current user can allow the participant to unmute themselves or unmute the participant (if the participant is the current user).
   * \param[in] can_be_muted_for_current_user_ True, if the current user can mute the participant only for self.
   * \param[in] can_be_unmuted_for_current_user_ True, if the current user can unmute the participant for self.
   * \param[in] is_muted_for_all_users_ True, if the participant is muted for all users.
   * \param[in] is_muted_for_current_user_ True, if the participant is muted for the current user.
   * \param[in] can_unmute_self_ True, if the participant is muted for all users, but can unmute themselves.
   * \param[in] volume_level_ Participant&\#039;s volume level; 1-20000 in hundreds of percents.
   * \param[in] order_ User&\#039;s order in the group call participant list. Orders must be compared lexicographically. The bigger is order, the higher is user in the list. If order is empty, the user must be removed from the participant list.
   */
  groupCallParticipant(object_ptr<MessageSender> &&participant_id_, int32 audio_source_id_, int32 screen_sharing_audio_source_id_, object_ptr<groupCallParticipantVideoInfo> &&video_info_, object_ptr<groupCallParticipantVideoInfo> &&screen_sharing_video_info_, string const &bio_, bool is_current_user_, bool is_speaking_, bool is_hand_raised_, bool can_be_muted_for_all_users_, bool can_be_unmuted_for_all_users_, bool can_be_muted_for_current_user_, bool can_be_unmuted_for_current_user_, bool is_muted_for_all_users_, bool is_muted_for_current_user_, bool can_unmute_self_, int32 volume_level_, string const &order_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 2059182571;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains information about a group call participant&\#039;s video channel.
 */
class groupCallParticipantVideoInfo final : public Object {
 public:
  /// List of synchronization source groups of the video.
  array<object_ptr<groupCallVideoSourceGroup>> source_groups_;
  /// Video channel endpoint identifier.
  string endpoint_id_;
  /// True if the video is paused. This flag needs to be ignored, if new video frames are received.
  bool is_paused_;

  /**
   * Contains information about a group call participant&\#039;s video channel.
   */
  groupCallParticipantVideoInfo();

  /**
   * Contains information about a group call participant&\#039;s video channel.
   *
   * \param[in] source_groups_ List of synchronization source groups of the video.
   * \param[in] endpoint_id_ Video channel endpoint identifier.
   * \param[in] is_paused_ True if the video is paused. This flag needs to be ignored, if new video frames are received.
   */
  groupCallParticipantVideoInfo(array<object_ptr<groupCallVideoSourceGroup>> &&source_groups_, string const &endpoint_id_, bool is_paused_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -14294645;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Describes a recently speaking participant in a group call.
 */
class groupCallRecentSpeaker final : public Object {
 public:
  /// Group call participant identifier.
  object_ptr<MessageSender> participant_id_;
  /// True, is the user has spoken recently.
  bool is_speaking_;

  /**
   * Describes a recently speaking participant in a group call.
   */
  groupCallRecentSpeaker();

  /**
   * Describes a recently speaking participant in a group call.
   *
   * \param[in] participant_id_ Group call participant identifier.
   * \param[in] is_speaking_ True, is the user has spoken recently.
   */
  groupCallRecentSpeaker(object_ptr<MessageSender> &&participant_id_, bool is_speaking_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1819519436;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Describes the quality of a group call video.
 */
class GroupCallVideoQuality: public Object {
 public:
};

/**
 * The worst available video quality.
 */
class groupCallVideoQualityThumbnail final : public GroupCallVideoQuality {
 public:

  /**
   * The worst available video quality.
   */
  groupCallVideoQualityThumbnail();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -379186304;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The medium video quality.
 */
class groupCallVideoQualityMedium final : public GroupCallVideoQuality {
 public:

  /**
   * The medium video quality.
   */
  groupCallVideoQualityMedium();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 394968234;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The best available video quality.
 */
class groupCallVideoQualityFull final : public GroupCallVideoQuality {
 public:

  /**
   * The best available video quality.
   */
  groupCallVideoQualityFull();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -2125916617;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Describes a group of video synchronization source identifiers.
 */
class groupCallVideoSourceGroup final : public Object {
 public:
  /// The semantics of sources, one of &quot;SIM&quot; or &quot;FID&quot;.
  string semantics_;
  /// The list of synchronization source identifiers.
  array<int32> source_ids_;

  /**
   * Describes a group of video synchronization source identifiers.
   */
  groupCallVideoSourceGroup();

  /**
   * Describes a group of video synchronization source identifiers.
   *
   * \param[in] semantics_ The semantics of sources, one of &quot;SIM&quot; or &quot;FID&quot;.
   * \param[in] source_ids_ The list of synchronization source identifiers.
   */
  groupCallVideoSourceGroup(string const &semantics_, array<int32> &&source_ids_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1190900785;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains a list of hashtags.
 */
class hashtags final : public Object {
 public:
  /// A list of hashtags.
  array<string> hashtags_;

  /**
   * Contains a list of hashtags.
   */
  hashtags();

  /**
   * Contains a list of hashtags.
   *
   * \param[in] hashtags_ A list of hashtags.
   */
  explicit hashtags(array<string> &&hashtags_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 676798885;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains an HTTP URL.
 */
class httpUrl final : public Object {
 public:
  /// The URL.
  string url_;

  /**
   * Contains an HTTP URL.
   */
  httpUrl();

  /**
   * Contains an HTTP URL.
   *
   * \param[in] url_ The URL.
   */
  explicit httpUrl(string const &url_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -2018019930;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * An identity document.
 */
class identityDocument final : public Object {
 public:
  /// Document number; 1-24 characters.
  string number_;
  /// Document expiry date; may be null if not applicable.
  object_ptr<date> expiry_date_;
  /// Front side of the document.
  object_ptr<datedFile> front_side_;
  /// Reverse side of the document; only for driver license and identity card; may be null.
  object_ptr<datedFile> reverse_side_;
  /// Selfie with the document; may be null.
  object_ptr<datedFile> selfie_;
  /// List of files containing a certified English translation of the document.
  array<object_ptr<datedFile>> translation_;

  /**
   * An identity document.
   */
  identityDocument();

  /**
   * An identity document.
   *
   * \param[in] number_ Document number; 1-24 characters.
   * \param[in] expiry_date_ Document expiry date; may be null if not applicable.
   * \param[in] front_side_ Front side of the document.
   * \param[in] reverse_side_ Reverse side of the document; only for driver license and identity card; may be null.
   * \param[in] selfie_ Selfie with the document; may be null.
   * \param[in] translation_ List of files containing a certified English translation of the document.
   */
  identityDocument(string const &number_, object_ptr<date> &&expiry_date_, object_ptr<datedFile> &&front_side_, object_ptr<datedFile> &&reverse_side_, object_ptr<datedFile> &&selfie_, array<object_ptr<datedFile>> &&translation_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 445952972;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Represents the result of an ImportContacts request.
 */
class importedContacts final : public Object {
 public:
  /// User identifiers of the imported contacts in the same order as they were specified in the request; 0 if the contact is not yet a registered user.
  array<int53> user_ids_;
  /// The number of users that imported the corresponding contact; 0 for already registered users or if unavailable.
  array<int32> importer_count_;

  /**
   * Represents the result of an ImportContacts request.
   */
  importedContacts();

  /**
   * Represents the result of an ImportContacts request.
   *
   * \param[in] user_ids_ User identifiers of the imported contacts in the same order as they were specified in the request; 0 if the contact is not yet a registered user.
   * \param[in] importer_count_ The number of users that imported the corresponding contact; 0 for already registered users or if unavailable.
   */
  importedContacts(array<int53> &&user_ids_, array<int32> &&importer_count_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 2068432290;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Represents a single button in an inline keyboard.
 */
class inlineKeyboardButton final : public Object {
 public:
  /// Text of the button.
  string text_;
  /// Type of the button.
  object_ptr<InlineKeyboardButtonType> type_;

  /**
   * Represents a single button in an inline keyboard.
   */
  inlineKeyboardButton();

  /**
   * Represents a single button in an inline keyboard.
   *
   * \param[in] text_ Text of the button.
   * \param[in] type_ Type of the button.
   */
  inlineKeyboardButton(string const &text_, object_ptr<InlineKeyboardButtonType> &&type_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -372105704;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Describes the type of an inline keyboard button.
 */
class InlineKeyboardButtonType: public Object {
 public:
};

/**
 * A button that opens a specified URL.
 */
class inlineKeyboardButtonTypeUrl final : public InlineKeyboardButtonType {
 public:
  /// HTTP or tg:// URL to open.
  string url_;

  /**
   * A button that opens a specified URL.
   */
  inlineKeyboardButtonTypeUrl();

  /**
   * A button that opens a specified URL.
   *
   * \param[in] url_ HTTP or tg:// URL to open.
   */
  explicit inlineKeyboardButtonTypeUrl(string const &url_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1130741420;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A button that opens a specified URL and automatically authorize the current user if allowed to do so.
 */
class inlineKeyboardButtonTypeLoginUrl final : public InlineKeyboardButtonType {
 public:
  /// An HTTP URL to open.
  string url_;
  /// Unique button identifier.
  int53 id_;
  /// If non-empty, new text of the button in forwarded messages.
  string forward_text_;

  /**
   * A button that opens a specified URL and automatically authorize the current user if allowed to do so.
   */
  inlineKeyboardButtonTypeLoginUrl();

  /**
   * A button that opens a specified URL and automatically authorize the current user if allowed to do so.
   *
   * \param[in] url_ An HTTP URL to open.
   * \param[in] id_ Unique button identifier.
   * \param[in] forward_text_ If non-empty, new text of the button in forwarded messages.
   */
  inlineKeyboardButtonTypeLoginUrl(string const &url_, int53 id_, string const &forward_text_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1203413081;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A button that sends a callback query to a bot.
 */
class inlineKeyboardButtonTypeCallback final : public InlineKeyboardButtonType {
 public:
  /// Data to be sent to the bot via a callback query.
  bytes data_;

  /**
   * A button that sends a callback query to a bot.
   */
  inlineKeyboardButtonTypeCallback();

  /**
   * A button that sends a callback query to a bot.
   *
   * \param[in] data_ Data to be sent to the bot via a callback query.
   */
  explicit inlineKeyboardButtonTypeCallback(bytes const &data_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1127515139;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A button that asks for password of the current user and then sends a callback query to a bot.
 */
class inlineKeyboardButtonTypeCallbackWithPassword final : public InlineKeyboardButtonType {
 public:
  /// Data to be sent to the bot via a callback query.
  bytes data_;

  /**
   * A button that asks for password of the current user and then sends a callback query to a bot.
   */
  inlineKeyboardButtonTypeCallbackWithPassword();

  /**
   * A button that asks for password of the current user and then sends a callback query to a bot.
   *
   * \param[in] data_ Data to be sent to the bot via a callback query.
   */
  explicit inlineKeyboardButtonTypeCallbackWithPassword(bytes const &data_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 908018248;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A button with a game that sends a callback query to a bot. This button must be in the first column and row of the keyboard and can be attached only to a message with content of the type messageGame.
 */
class inlineKeyboardButtonTypeCallbackGame final : public InlineKeyboardButtonType {
 public:

  /**
   * A button with a game that sends a callback query to a bot. This button must be in the first column and row of the keyboard and can be attached only to a message with content of the type messageGame.
   */
  inlineKeyboardButtonTypeCallbackGame();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -383429528;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A button that forces an inline query to the bot to be inserted in the input field.
 */
class inlineKeyboardButtonTypeSwitchInline final : public InlineKeyboardButtonType {
 public:
  /// Inline query to be sent to the bot.
  string query_;
  /// True, if the inline query must be sent from the current chat.
  bool in_current_chat_;

  /**
   * A button that forces an inline query to the bot to be inserted in the input field.
   */
  inlineKeyboardButtonTypeSwitchInline();

  /**
   * A button that forces an inline query to the bot to be inserted in the input field.
   *
   * \param[in] query_ Inline query to be sent to the bot.
   * \param[in] in_current_chat_ True, if the inline query must be sent from the current chat.
   */
  inlineKeyboardButtonTypeSwitchInline(string const &query_, bool in_current_chat_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -2035563307;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A button to buy something. This button must be in the first column and row of the keyboard and can be attached only to a message with content of the type messageInvoice.
 */
class inlineKeyboardButtonTypeBuy final : public InlineKeyboardButtonType {
 public:

  /**
   * A button to buy something. This button must be in the first column and row of the keyboard and can be attached only to a message with content of the type messageInvoice.
   */
  inlineKeyboardButtonTypeBuy();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1360739440;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A button with a user reference to be handled in the same way as textEntityTypeMentionName entities.
 */
class inlineKeyboardButtonTypeUser final : public InlineKeyboardButtonType {
 public:
  /// User identifier.
  int53 user_id_;

  /**
   * A button with a user reference to be handled in the same way as textEntityTypeMentionName entities.
   */
  inlineKeyboardButtonTypeUser();

  /**
   * A button with a user reference to be handled in the same way as textEntityTypeMentionName entities.
   *
   * \param[in] user_id_ User identifier.
   */
  explicit inlineKeyboardButtonTypeUser(int53 user_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1836574114;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Represents a single result of an inline query.
 */
class InlineQueryResult: public Object {
 public:
};

/**
 * Represents a link to an article or web page.
 */
class inlineQueryResultArticle final : public InlineQueryResult {
 public:
  /// Unique identifier of the query result.
  string id_;
  /// URL of the result, if it exists.
  string url_;
  /// True, if the URL must be not shown.
  bool hide_url_;
  /// Title of the result.
  string title_;
  /// A short description of the result.
  string description_;
  /// Result thumbnail in JPEG format; may be null.
  object_ptr<thumbnail> thumbnail_;

  /**
   * Represents a link to an article or web page.
   */
  inlineQueryResultArticle();

  /**
   * Represents a link to an article or web page.
   *
   * \param[in] id_ Unique identifier of the query result.
   * \param[in] url_ URL of the result, if it exists.
   * \param[in] hide_url_ True, if the URL must be not shown.
   * \param[in] title_ Title of the result.
   * \param[in] description_ A short description of the result.
   * \param[in] thumbnail_ Result thumbnail in JPEG format; may be null.
   */
  inlineQueryResultArticle(string const &id_, string const &url_, bool hide_url_, string const &title_, string const &description_, object_ptr<thumbnail> &&thumbnail_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 206340825;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Represents a user contact.
 */
class inlineQueryResultContact final : public InlineQueryResult {
 public:
  /// Unique identifier of the query result.
  string id_;
  /// A user contact.
  object_ptr<contact> contact_;
  /// Result thumbnail in JPEG format; may be null.
  object_ptr<thumbnail> thumbnail_;

  /**
   * Represents a user contact.
   */
  inlineQueryResultContact();

  /**
   * Represents a user contact.
   *
   * \param[in] id_ Unique identifier of the query result.
   * \param[in] contact_ A user contact.
   * \param[in] thumbnail_ Result thumbnail in JPEG format; may be null.
   */
  inlineQueryResultContact(string const &id_, object_ptr<contact> &&contact_, object_ptr<thumbnail> &&thumbnail_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -181960174;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Represents a point on the map.
 */
class inlineQueryResultLocation final : public InlineQueryResult {
 public:
  /// Unique identifier of the query result.
  string id_;
  /// Location result.
  object_ptr<location> location_;
  /// Title of the result.
  string title_;
  /// Result thumbnail in JPEG format; may be null.
  object_ptr<thumbnail> thumbnail_;

  /**
   * Represents a point on the map.
   */
  inlineQueryResultLocation();

  /**
   * Represents a point on the map.
   *
   * \param[in] id_ Unique identifier of the query result.
   * \param[in] location_ Location result.
   * \param[in] title_ Title of the result.
   * \param[in] thumbnail_ Result thumbnail in JPEG format; may be null.
   */
  inlineQueryResultLocation(string const &id_, object_ptr<location> &&location_, string const &title_, object_ptr<thumbnail> &&thumbnail_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 466004752;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Represents information about a venue.
 */
class inlineQueryResultVenue final : public InlineQueryResult {
 public:
  /// Unique identifier of the query result.
  string id_;
  /// Venue result.
  object_ptr<venue> venue_;
  /// Result thumbnail in JPEG format; may be null.
  object_ptr<thumbnail> thumbnail_;

  /**
   * Represents information about a venue.
   */
  inlineQueryResultVenue();

  /**
   * Represents information about a venue.
   *
   * \param[in] id_ Unique identifier of the query result.
   * \param[in] venue_ Venue result.
   * \param[in] thumbnail_ Result thumbnail in JPEG format; may be null.
   */
  inlineQueryResultVenue(string const &id_, object_ptr<venue> &&venue_, object_ptr<thumbnail> &&thumbnail_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1281036382;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Represents information about a game.
 */
class inlineQueryResultGame final : public InlineQueryResult {
 public:
  /// Unique identifier of the query result.
  string id_;
  /// Game result.
  object_ptr<game> game_;

  /**
   * Represents information about a game.
   */
  inlineQueryResultGame();

  /**
   * Represents information about a game.
   *
   * \param[in] id_ Unique identifier of the query result.
   * \param[in] game_ Game result.
   */
  inlineQueryResultGame(string const &id_, object_ptr<game> &&game_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1706916987;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Represents an animation file.
 */
class inlineQueryResultAnimation final : public InlineQueryResult {
 public:
  /// Unique identifier of the query result.
  string id_;
  /// Animation file.
  object_ptr<animation> animation_;
  /// Animation title.
  string title_;

  /**
   * Represents an animation file.
   */
  inlineQueryResultAnimation();

  /**
   * Represents an animation file.
   *
   * \param[in] id_ Unique identifier of the query result.
   * \param[in] animation_ Animation file.
   * \param[in] title_ Animation title.
   */
  inlineQueryResultAnimation(string const &id_, object_ptr<animation> &&animation_, string const &title_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 2009984267;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Represents an audio file.
 */
class inlineQueryResultAudio final : public InlineQueryResult {
 public:
  /// Unique identifier of the query result.
  string id_;
  /// Audio file.
  object_ptr<audio> audio_;

  /**
   * Represents an audio file.
   */
  inlineQueryResultAudio();

  /**
   * Represents an audio file.
   *
   * \param[in] id_ Unique identifier of the query result.
   * \param[in] audio_ Audio file.
   */
  inlineQueryResultAudio(string const &id_, object_ptr<audio> &&audio_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 842650360;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Represents a document.
 */
class inlineQueryResultDocument final : public InlineQueryResult {
 public:
  /// Unique identifier of the query result.
  string id_;
  /// Document.
  object_ptr<document> document_;
  /// Document title.
  string title_;
  /// Document description.
  string description_;

  /**
   * Represents a document.
   */
  inlineQueryResultDocument();

  /**
   * Represents a document.
   *
   * \param[in] id_ Unique identifier of the query result.
   * \param[in] document_ Document.
   * \param[in] title_ Document title.
   * \param[in] description_ Document description.
   */
  inlineQueryResultDocument(string const &id_, object_ptr<document> &&document_, string const &title_, string const &description_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1491268539;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Represents a photo.
 */
class inlineQueryResultPhoto final : public InlineQueryResult {
 public:
  /// Unique identifier of the query result.
  string id_;
  /// Photo.
  object_ptr<photo> photo_;
  /// Title of the result, if known.
  string title_;
  /// A short description of the result, if known.
  string description_;

  /**
   * Represents a photo.
   */
  inlineQueryResultPhoto();

  /**
   * Represents a photo.
   *
   * \param[in] id_ Unique identifier of the query result.
   * \param[in] photo_ Photo.
   * \param[in] title_ Title of the result, if known.
   * \param[in] description_ A short description of the result, if known.
   */
  inlineQueryResultPhoto(string const &id_, object_ptr<photo> &&photo_, string const &title_, string const &description_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1848319440;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Represents a sticker.
 */
class inlineQueryResultSticker final : public InlineQueryResult {
 public:
  /// Unique identifier of the query result.
  string id_;
  /// Sticker.
  object_ptr<sticker> sticker_;

  /**
   * Represents a sticker.
   */
  inlineQueryResultSticker();

  /**
   * Represents a sticker.
   *
   * \param[in] id_ Unique identifier of the query result.
   * \param[in] sticker_ Sticker.
   */
  inlineQueryResultSticker(string const &id_, object_ptr<sticker> &&sticker_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1848224245;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Represents a video.
 */
class inlineQueryResultVideo final : public InlineQueryResult {
 public:
  /// Unique identifier of the query result.
  string id_;
  /// Video.
  object_ptr<video> video_;
  /// Title of the video.
  string title_;
  /// Description of the video.
  string description_;

  /**
   * Represents a video.
   */
  inlineQueryResultVideo();

  /**
   * Represents a video.
   *
   * \param[in] id_ Unique identifier of the query result.
   * \param[in] video_ Video.
   * \param[in] title_ Title of the video.
   * \param[in] description_ Description of the video.
   */
  inlineQueryResultVideo(string const &id_, object_ptr<video> &&video_, string const &title_, string const &description_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1373158683;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Represents a voice note.
 */
class inlineQueryResultVoiceNote final : public InlineQueryResult {
 public:
  /// Unique identifier of the query result.
  string id_;
  /// Voice note.
  object_ptr<voiceNote> voice_note_;
  /// Title of the voice note.
  string title_;

  /**
   * Represents a voice note.
   */
  inlineQueryResultVoiceNote();

  /**
   * Represents a voice note.
   *
   * \param[in] id_ Unique identifier of the query result.
   * \param[in] voice_note_ Voice note.
   * \param[in] title_ Title of the voice note.
   */
  inlineQueryResultVoiceNote(string const &id_, object_ptr<voiceNote> &&voice_note_, string const &title_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1897393105;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Represents the results of the inline query. Use sendInlineQueryResultMessage to send the result of the query.
 */
class inlineQueryResults final : public Object {
 public:
  /// Unique identifier of the inline query.
  int64 inline_query_id_;
  /// The offset for the next request. If empty, there are no more results.
  string next_offset_;
  /// Results of the query.
  array<object_ptr<InlineQueryResult>> results_;
  /// If non-empty, this text must be shown on the button, which opens a private chat with the bot and sends the bot a start message with the switch_pm_parameter.
  string switch_pm_text_;
  /// Parameter for the bot start message.
  string switch_pm_parameter_;

  /**
   * Represents the results of the inline query. Use sendInlineQueryResultMessage to send the result of the query.
   */
  inlineQueryResults();

  /**
   * Represents the results of the inline query. Use sendInlineQueryResultMessage to send the result of the query.
   *
   * \param[in] inline_query_id_ Unique identifier of the inline query.
   * \param[in] next_offset_ The offset for the next request. If empty, there are no more results.
   * \param[in] results_ Results of the query.
   * \param[in] switch_pm_text_ If non-empty, this text must be shown on the button, which opens a private chat with the bot and sends the bot a start message with the switch_pm_parameter.
   * \param[in] switch_pm_parameter_ Parameter for the bot start message.
   */
  inlineQueryResults(int64 inline_query_id_, string const &next_offset_, array<object_ptr<InlineQueryResult>> &&results_, string const &switch_pm_text_, string const &switch_pm_parameter_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1000709656;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Contains information about background to set.
 */
class InputBackground: public Object {
 public:
};

/**
 * A background from a local file.
 */
class inputBackgroundLocal final : public InputBackground {
 public:
  /// Background file to use. Only inputFileLocal and inputFileGenerated are supported. The file must be in JPEG format for wallpapers and in PNG format for patterns.
  object_ptr<InputFile> background_;

  /**
   * A background from a local file.
   */
  inputBackgroundLocal();

  /**
   * A background from a local file.
   *
   * \param[in] background_ Background file to use. Only inputFileLocal and inputFileGenerated are supported. The file must be in JPEG format for wallpapers and in PNG format for patterns.
   */
  explicit inputBackgroundLocal(object_ptr<InputFile> &&background_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1747094364;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A background from the server.
 */
class inputBackgroundRemote final : public InputBackground {
 public:
  /// The background identifier.
  int64 background_id_;

  /**
   * A background from the server.
   */
  inputBackgroundRemote();

  /**
   * A background from the server.
   *
   * \param[in] background_id_ The background identifier.
   */
  explicit inputBackgroundRemote(int64 background_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -274976231;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Describes a photo to be set as a user profile or chat photo.
 */
class InputChatPhoto: public Object {
 public:
};

/**
 * A previously used profile photo of the current user.
 */
class inputChatPhotoPrevious final : public InputChatPhoto {
 public:
  /// Identifier of the current user&\#039;s profile photo to reuse.
  int64 chat_photo_id_;

  /**
   * A previously used profile photo of the current user.
   */
  inputChatPhotoPrevious();

  /**
   * A previously used profile photo of the current user.
   *
   * \param[in] chat_photo_id_ Identifier of the current user&\#039;s profile photo to reuse.
   */
  explicit inputChatPhotoPrevious(int64 chat_photo_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 23128529;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A static photo in JPEG format.
 */
class inputChatPhotoStatic final : public InputChatPhoto {
 public:
  /// Photo to be set as profile photo. Only inputFileLocal and inputFileGenerated are allowed.
  object_ptr<InputFile> photo_;

  /**
   * A static photo in JPEG format.
   */
  inputChatPhotoStatic();

  /**
   * A static photo in JPEG format.
   *
   * \param[in] photo_ Photo to be set as profile photo. Only inputFileLocal and inputFileGenerated are allowed.
   */
  explicit inputChatPhotoStatic(object_ptr<InputFile> &&photo_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1979179699;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * An animation in MPEG4 format; must be square, at most 10 seconds long, have width between 160 and 800 and be at most 2MB in size.
 */
class inputChatPhotoAnimation final : public InputChatPhoto {
 public:
  /// Animation to be set as profile photo. Only inputFileLocal and inputFileGenerated are allowed.
  object_ptr<InputFile> animation_;
  /// Timestamp of the frame, which will be used as static chat photo.
  double main_frame_timestamp_;

  /**
   * An animation in MPEG4 format; must be square, at most 10 seconds long, have width between 160 and 800 and be at most 2MB in size.
   */
  inputChatPhotoAnimation();

  /**
   * An animation in MPEG4 format; must be square, at most 10 seconds long, have width between 160 and 800 and be at most 2MB in size.
   *
   * \param[in] animation_ Animation to be set as profile photo. Only inputFileLocal and inputFileGenerated are allowed.
   * \param[in] main_frame_timestamp_ Timestamp of the frame, which will be used as static chat photo.
   */
  inputChatPhotoAnimation(object_ptr<InputFile> &&animation_, double main_frame_timestamp_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 90846242;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Contains information about the payment method chosen by the user.
 */
class InputCredentials: public Object {
 public:
};

/**
 * Applies if a user chooses some previously saved payment credentials. To use their previously saved credentials, the user must have a valid temporary password.
 */
class inputCredentialsSaved final : public InputCredentials {
 public:
  /// Identifier of the saved credentials.
  string saved_credentials_id_;

  /**
   * Applies if a user chooses some previously saved payment credentials. To use their previously saved credentials, the user must have a valid temporary password.
   */
  inputCredentialsSaved();

  /**
   * Applies if a user chooses some previously saved payment credentials. To use their previously saved credentials, the user must have a valid temporary password.
   *
   * \param[in] saved_credentials_id_ Identifier of the saved credentials.
   */
  explicit inputCredentialsSaved(string const &saved_credentials_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -2034385364;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Applies if a user enters new credentials on a payment provider website.
 */
class inputCredentialsNew final : public InputCredentials {
 public:
  /// JSON-encoded data with the credential identifier from the payment provider.
  string data_;
  /// True, if the credential identifier can be saved on the server side.
  bool allow_save_;

  /**
   * Applies if a user enters new credentials on a payment provider website.
   */
  inputCredentialsNew();

  /**
   * Applies if a user enters new credentials on a payment provider website.
   *
   * \param[in] data_ JSON-encoded data with the credential identifier from the payment provider.
   * \param[in] allow_save_ True, if the credential identifier can be saved on the server side.
   */
  inputCredentialsNew(string const &data_, bool allow_save_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -829689558;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Applies if a user enters new credentials using Apple Pay.
 */
class inputCredentialsApplePay final : public InputCredentials {
 public:
  /// JSON-encoded data with the credential identifier.
  string data_;

  /**
   * Applies if a user enters new credentials using Apple Pay.
   */
  inputCredentialsApplePay();

  /**
   * Applies if a user enters new credentials using Apple Pay.
   *
   * \param[in] data_ JSON-encoded data with the credential identifier.
   */
  explicit inputCredentialsApplePay(string const &data_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1246570799;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Applies if a user enters new credentials using Google Pay.
 */
class inputCredentialsGooglePay final : public InputCredentials {
 public:
  /// JSON-encoded data with the credential identifier.
  string data_;

  /**
   * Applies if a user enters new credentials using Google Pay.
   */
  inputCredentialsGooglePay();

  /**
   * Applies if a user enters new credentials using Google Pay.
   *
   * \param[in] data_ JSON-encoded data with the credential identifier.
   */
  explicit inputCredentialsGooglePay(string const &data_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 844384100;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Points to a file.
 */
class InputFile: public Object {
 public:
};

/**
 * A file defined by its unique ID.
 */
class inputFileId final : public InputFile {
 public:
  /// Unique file identifier.
  int32 id_;

  /**
   * A file defined by its unique ID.
   */
  inputFileId();

  /**
   * A file defined by its unique ID.
   *
   * \param[in] id_ Unique file identifier.
   */
  explicit inputFileId(int32 id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1788906253;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A file defined by its remote ID. The remote ID is guaranteed to be usable only if the corresponding file is still accessible to the user and known to TDLib. For example, if the file is from a message, then the message must be not deleted and accessible to the user. If the file database is disabled, then the corresponding object with the file must be preloaded by the application.
 */
class inputFileRemote final : public InputFile {
 public:
  /// Remote file identifier.
  string id_;

  /**
   * A file defined by its remote ID. The remote ID is guaranteed to be usable only if the corresponding file is still accessible to the user and known to TDLib. For example, if the file is from a message, then the message must be not deleted and accessible to the user. If the file database is disabled, then the corresponding object with the file must be preloaded by the application.
   */
  inputFileRemote();

  /**
   * A file defined by its remote ID. The remote ID is guaranteed to be usable only if the corresponding file is still accessible to the user and known to TDLib. For example, if the file is from a message, then the message must be not deleted and accessible to the user. If the file database is disabled, then the corresponding object with the file must be preloaded by the application.
   *
   * \param[in] id_ Remote file identifier.
   */
  explicit inputFileRemote(string const &id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -107574466;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A file defined by a local path.
 */
class inputFileLocal final : public InputFile {
 public:
  /// Local path to the file.
  string path_;

  /**
   * A file defined by a local path.
   */
  inputFileLocal();

  /**
   * A file defined by a local path.
   *
   * \param[in] path_ Local path to the file.
   */
  explicit inputFileLocal(string const &path_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 2056030919;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A file generated by the application.
 */
class inputFileGenerated final : public InputFile {
 public:
  /// Local path to a file from which the file is generated; may be empty if there is no such file.
  string original_path_;
  /// String specifying the conversion applied to the original file; must be persistent across application restarts. Conversions beginning with &\#039;\#&\#039; are reserved for internal TDLib usage.
  string conversion_;
  /// Expected size of the generated file, in bytes; 0 if unknown.
  int32 expected_size_;

  /**
   * A file generated by the application.
   */
  inputFileGenerated();

  /**
   * A file generated by the application.
   *
   * \param[in] original_path_ Local path to a file from which the file is generated; may be empty if there is no such file.
   * \param[in] conversion_ String specifying the conversion applied to the original file; must be persistent across application restarts. Conversions beginning with &\#039;\#&\#039; are reserved for internal TDLib usage.
   * \param[in] expected_size_ Expected size of the generated file, in bytes; 0 if unknown.
   */
  inputFileGenerated(string const &original_path_, string const &conversion_, int32 expected_size_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1781351885;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * An identity document to be saved to Telegram Passport.
 */
class inputIdentityDocument final : public Object {
 public:
  /// Document number; 1-24 characters.
  string number_;
  /// Document expiry date; pass null if not applicable.
  object_ptr<date> expiry_date_;
  /// Front side of the document.
  object_ptr<InputFile> front_side_;
  /// Reverse side of the document; only for driver license and identity card; pass null otherwise.
  object_ptr<InputFile> reverse_side_;
  /// Selfie with the document; pass null if unavailable.
  object_ptr<InputFile> selfie_;
  /// List of files containing a certified English translation of the document.
  array<object_ptr<InputFile>> translation_;

  /**
   * An identity document to be saved to Telegram Passport.
   */
  inputIdentityDocument();

  /**
   * An identity document to be saved to Telegram Passport.
   *
   * \param[in] number_ Document number; 1-24 characters.
   * \param[in] expiry_date_ Document expiry date; pass null if not applicable.
   * \param[in] front_side_ Front side of the document.
   * \param[in] reverse_side_ Reverse side of the document; only for driver license and identity card; pass null otherwise.
   * \param[in] selfie_ Selfie with the document; pass null if unavailable.
   * \param[in] translation_ List of files containing a certified English translation of the document.
   */
  inputIdentityDocument(string const &number_, object_ptr<date> &&expiry_date_, object_ptr<InputFile> &&front_side_, object_ptr<InputFile> &&reverse_side_, object_ptr<InputFile> &&selfie_, array<object_ptr<InputFile>> &&translation_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -381776063;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Represents a single result of an inline query; for bots only.
 */
class InputInlineQueryResult: public Object {
 public:
};

/**
 * Represents a link to an animated GIF or an animated (i.e., without sound) H.264/MPEG-4 AVC video.
 */
class inputInlineQueryResultAnimation final : public InputInlineQueryResult {
 public:
  /// Unique identifier of the query result.
  string id_;
  /// Title of the query result.
  string title_;
  /// URL of the result thumbnail (JPEG, GIF, or MPEG4), if it exists.
  string thumbnail_url_;
  /// MIME type of the video thumbnail. If non-empty, must be one of &quot;image/jpeg&quot;, &quot;image/gif&quot; and &quot;video/mp4&quot;.
  string thumbnail_mime_type_;
  /// The URL of the video file (file size must not exceed 1MB).
  string video_url_;
  /// MIME type of the video file. Must be one of &quot;image/gif&quot; and &quot;video/mp4&quot;.
  string video_mime_type_;
  /// Duration of the video, in seconds.
  int32 video_duration_;
  /// Width of the video.
  int32 video_width_;
  /// Height of the video.
  int32 video_height_;
  /// The message reply markup; pass null if none. Must be of type replyMarkupInlineKeyboard or null.
  object_ptr<ReplyMarkup> reply_markup_;
  /// The content of the message to be sent. Must be one of the following types: inputMessageText, inputMessageAnimation, inputMessageInvoice, inputMessageLocation, inputMessageVenue or inputMessageContact.
  object_ptr<InputMessageContent> input_message_content_;

  /**
   * Represents a link to an animated GIF or an animated (i.e., without sound) H.264/MPEG-4 AVC video.
   */
  inputInlineQueryResultAnimation();

  /**
   * Represents a link to an animated GIF or an animated (i.e., without sound) H.264/MPEG-4 AVC video.
   *
   * \param[in] id_ Unique identifier of the query result.
   * \param[in] title_ Title of the query result.
   * \param[in] thumbnail_url_ URL of the result thumbnail (JPEG, GIF, or MPEG4), if it exists.
   * \param[in] thumbnail_mime_type_ MIME type of the video thumbnail. If non-empty, must be one of &quot;image/jpeg&quot;, &quot;image/gif&quot; and &quot;video/mp4&quot;.
   * \param[in] video_url_ The URL of the video file (file size must not exceed 1MB).
   * \param[in] video_mime_type_ MIME type of the video file. Must be one of &quot;image/gif&quot; and &quot;video/mp4&quot;.
   * \param[in] video_duration_ Duration of the video, in seconds.
   * \param[in] video_width_ Width of the video.
   * \param[in] video_height_ Height of the video.
   * \param[in] reply_markup_ The message reply markup; pass null if none. Must be of type replyMarkupInlineKeyboard or null.
   * \param[in] input_message_content_ The content of the message to be sent. Must be one of the following types: inputMessageText, inputMessageAnimation, inputMessageInvoice, inputMessageLocation, inputMessageVenue or inputMessageContact.
   */
  inputInlineQueryResultAnimation(string const &id_, string const &title_, string const &thumbnail_url_, string const &thumbnail_mime_type_, string const &video_url_, string const &video_mime_type_, int32 video_duration_, int32 video_width_, int32 video_height_, object_ptr<ReplyMarkup> &&reply_markup_, object_ptr<InputMessageContent> &&input_message_content_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1489808874;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Represents a link to an article or web page.
 */
class inputInlineQueryResultArticle final : public InputInlineQueryResult {
 public:
  /// Unique identifier of the query result.
  string id_;
  /// URL of the result, if it exists.
  string url_;
  /// True, if the URL must be not shown.
  bool hide_url_;
  /// Title of the result.
  string title_;
  /// A short description of the result.
  string description_;
  /// URL of the result thumbnail, if it exists.
  string thumbnail_url_;
  /// Thumbnail width, if known.
  int32 thumbnail_width_;
  /// Thumbnail height, if known.
  int32 thumbnail_height_;
  /// The message reply markup; pass null if none. Must be of type replyMarkupInlineKeyboard or null.
  object_ptr<ReplyMarkup> reply_markup_;
  /// The content of the message to be sent. Must be one of the following types: inputMessageText, inputMessageInvoice, inputMessageLocation, inputMessageVenue or inputMessageContact.
  object_ptr<InputMessageContent> input_message_content_;

  /**
   * Represents a link to an article or web page.
   */
  inputInlineQueryResultArticle();

  /**
   * Represents a link to an article or web page.
   *
   * \param[in] id_ Unique identifier of the query result.
   * \param[in] url_ URL of the result, if it exists.
   * \param[in] hide_url_ True, if the URL must be not shown.
   * \param[in] title_ Title of the result.
   * \param[in] description_ A short description of the result.
   * \param[in] thumbnail_url_ URL of the result thumbnail, if it exists.
   * \param[in] thumbnail_width_ Thumbnail width, if known.
   * \param[in] thumbnail_height_ Thumbnail height, if known.
   * \param[in] reply_markup_ The message reply markup; pass null if none. Must be of type replyMarkupInlineKeyboard or null.
   * \param[in] input_message_content_ The content of the message to be sent. Must be one of the following types: inputMessageText, inputMessageInvoice, inputMessageLocation, inputMessageVenue or inputMessageContact.
   */
  inputInlineQueryResultArticle(string const &id_, string const &url_, bool hide_url_, string const &title_, string const &description_, string const &thumbnail_url_, int32 thumbnail_width_, int32 thumbnail_height_, object_ptr<ReplyMarkup> &&reply_markup_, object_ptr<InputMessageContent> &&input_message_content_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1973670156;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Represents a link to an MP3 audio file.
 */
class inputInlineQueryResultAudio final : public InputInlineQueryResult {
 public:
  /// Unique identifier of the query result.
  string id_;
  /// Title of the audio file.
  string title_;
  /// Performer of the audio file.
  string performer_;
  /// The URL of the audio file.
  string audio_url_;
  /// Audio file duration, in seconds.
  int32 audio_duration_;
  /// The message reply markup; pass null if none. Must be of type replyMarkupInlineKeyboard or null.
  object_ptr<ReplyMarkup> reply_markup_;
  /// The content of the message to be sent. Must be one of the following types: inputMessageText, inputMessageAudio, inputMessageInvoice, inputMessageLocation, inputMessageVenue or inputMessageContact.
  object_ptr<InputMessageContent> input_message_content_;

  /**
   * Represents a link to an MP3 audio file.
   */
  inputInlineQueryResultAudio();

  /**
   * Represents a link to an MP3 audio file.
   *
   * \param[in] id_ Unique identifier of the query result.
   * \param[in] title_ Title of the audio file.
   * \param[in] performer_ Performer of the audio file.
   * \param[in] audio_url_ The URL of the audio file.
   * \param[in] audio_duration_ Audio file duration, in seconds.
   * \param[in] reply_markup_ The message reply markup; pass null if none. Must be of type replyMarkupInlineKeyboard or null.
   * \param[in] input_message_content_ The content of the message to be sent. Must be one of the following types: inputMessageText, inputMessageAudio, inputMessageInvoice, inputMessageLocation, inputMessageVenue or inputMessageContact.
   */
  inputInlineQueryResultAudio(string const &id_, string const &title_, string const &performer_, string const &audio_url_, int32 audio_duration_, object_ptr<ReplyMarkup> &&reply_markup_, object_ptr<InputMessageContent> &&input_message_content_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1260139988;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Represents a user contact.
 */
class inputInlineQueryResultContact final : public InputInlineQueryResult {
 public:
  /// Unique identifier of the query result.
  string id_;
  /// User contact.
  object_ptr<contact> contact_;
  /// URL of the result thumbnail, if it exists.
  string thumbnail_url_;
  /// Thumbnail width, if known.
  int32 thumbnail_width_;
  /// Thumbnail height, if known.
  int32 thumbnail_height_;
  /// The message reply markup; pass null if none. Must be of type replyMarkupInlineKeyboard or null.
  object_ptr<ReplyMarkup> reply_markup_;
  /// The content of the message to be sent. Must be one of the following types: inputMessageText, inputMessageInvoice, inputMessageLocation, inputMessageVenue or inputMessageContact.
  object_ptr<InputMessageContent> input_message_content_;

  /**
   * Represents a user contact.
   */
  inputInlineQueryResultContact();

  /**
   * Represents a user contact.
   *
   * \param[in] id_ Unique identifier of the query result.
   * \param[in] contact_ User contact.
   * \param[in] thumbnail_url_ URL of the result thumbnail, if it exists.
   * \param[in] thumbnail_width_ Thumbnail width, if known.
   * \param[in] thumbnail_height_ Thumbnail height, if known.
   * \param[in] reply_markup_ The message reply markup; pass null if none. Must be of type replyMarkupInlineKeyboard or null.
   * \param[in] input_message_content_ The content of the message to be sent. Must be one of the following types: inputMessageText, inputMessageInvoice, inputMessageLocation, inputMessageVenue or inputMessageContact.
   */
  inputInlineQueryResultContact(string const &id_, object_ptr<contact> &&contact_, string const &thumbnail_url_, int32 thumbnail_width_, int32 thumbnail_height_, object_ptr<ReplyMarkup> &&reply_markup_, object_ptr<InputMessageContent> &&input_message_content_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1846064594;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Represents a link to a file.
 */
class inputInlineQueryResultDocument final : public InputInlineQueryResult {
 public:
  /// Unique identifier of the query result.
  string id_;
  /// Title of the resulting file.
  string title_;
  /// Short description of the result, if known.
  string description_;
  /// URL of the file.
  string document_url_;
  /// MIME type of the file content; only &quot;application/pdf&quot; and &quot;application/zip&quot; are currently allowed.
  string mime_type_;
  /// The URL of the file thumbnail, if it exists.
  string thumbnail_url_;
  /// Width of the thumbnail.
  int32 thumbnail_width_;
  /// Height of the thumbnail.
  int32 thumbnail_height_;
  /// The message reply markup; pass null if none. Must be of type replyMarkupInlineKeyboard or null.
  object_ptr<ReplyMarkup> reply_markup_;
  /// The content of the message to be sent. Must be one of the following types: inputMessageText, inputMessageDocument, inputMessageInvoice, inputMessageLocation, inputMessageVenue or inputMessageContact.
  object_ptr<InputMessageContent> input_message_content_;

  /**
   * Represents a link to a file.
   */
  inputInlineQueryResultDocument();

  /**
   * Represents a link to a file.
   *
   * \param[in] id_ Unique identifier of the query result.
   * \param[in] title_ Title of the resulting file.
   * \param[in] description_ Short description of the result, if known.
   * \param[in] document_url_ URL of the file.
   * \param[in] mime_type_ MIME type of the file content; only &quot;application/pdf&quot; and &quot;application/zip&quot; are currently allowed.
   * \param[in] thumbnail_url_ The URL of the file thumbnail, if it exists.
   * \param[in] thumbnail_width_ Width of the thumbnail.
   * \param[in] thumbnail_height_ Height of the thumbnail.
   * \param[in] reply_markup_ The message reply markup; pass null if none. Must be of type replyMarkupInlineKeyboard or null.
   * \param[in] input_message_content_ The content of the message to be sent. Must be one of the following types: inputMessageText, inputMessageDocument, inputMessageInvoice, inputMessageLocation, inputMessageVenue or inputMessageContact.
   */
  inputInlineQueryResultDocument(string const &id_, string const &title_, string const &description_, string const &document_url_, string const &mime_type_, string const &thumbnail_url_, int32 thumbnail_width_, int32 thumbnail_height_, object_ptr<ReplyMarkup> &&reply_markup_, object_ptr<InputMessageContent> &&input_message_content_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 578801869;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Represents a game.
 */
class inputInlineQueryResultGame final : public InputInlineQueryResult {
 public:
  /// Unique identifier of the query result.
  string id_;
  /// Short name of the game.
  string game_short_name_;
  /// The message reply markup; pass null if none. Must be of type replyMarkupInlineKeyboard or null.
  object_ptr<ReplyMarkup> reply_markup_;

  /**
   * Represents a game.
   */
  inputInlineQueryResultGame();

  /**
   * Represents a game.
   *
   * \param[in] id_ Unique identifier of the query result.
   * \param[in] game_short_name_ Short name of the game.
   * \param[in] reply_markup_ The message reply markup; pass null if none. Must be of type replyMarkupInlineKeyboard or null.
   */
  inputInlineQueryResultGame(string const &id_, string const &game_short_name_, object_ptr<ReplyMarkup> &&reply_markup_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 966074327;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Represents a point on the map.
 */
class inputInlineQueryResultLocation final : public InputInlineQueryResult {
 public:
  /// Unique identifier of the query result.
  string id_;
  /// Location result.
  object_ptr<location> location_;
  /// Amount of time relative to the message sent time until the location can be updated, in seconds.
  int32 live_period_;
  /// Title of the result.
  string title_;
  /// URL of the result thumbnail, if it exists.
  string thumbnail_url_;
  /// Thumbnail width, if known.
  int32 thumbnail_width_;
  /// Thumbnail height, if known.
  int32 thumbnail_height_;
  /// The message reply markup; pass null if none. Must be of type replyMarkupInlineKeyboard or null.
  object_ptr<ReplyMarkup> reply_markup_;
  /// The content of the message to be sent. Must be one of the following types: inputMessageText, inputMessageInvoice, inputMessageLocation, inputMessageVenue or inputMessageContact.
  object_ptr<InputMessageContent> input_message_content_;

  /**
   * Represents a point on the map.
   */
  inputInlineQueryResultLocation();

  /**
   * Represents a point on the map.
   *
   * \param[in] id_ Unique identifier of the query result.
   * \param[in] location_ Location result.
   * \param[in] live_period_ Amount of time relative to the message sent time until the location can be updated, in seconds.
   * \param[in] title_ Title of the result.
   * \param[in] thumbnail_url_ URL of the result thumbnail, if it exists.
   * \param[in] thumbnail_width_ Thumbnail width, if known.
   * \param[in] thumbnail_height_ Thumbnail height, if known.
   * \param[in] reply_markup_ The message reply markup; pass null if none. Must be of type replyMarkupInlineKeyboard or null.
   * \param[in] input_message_content_ The content of the message to be sent. Must be one of the following types: inputMessageText, inputMessageInvoice, inputMessageLocation, inputMessageVenue or inputMessageContact.
   */
  inputInlineQueryResultLocation(string const &id_, object_ptr<location> &&location_, int32 live_period_, string const &title_, string const &thumbnail_url_, int32 thumbnail_width_, int32 thumbnail_height_, object_ptr<ReplyMarkup> &&reply_markup_, object_ptr<InputMessageContent> &&input_message_content_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1887650218;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Represents link to a JPEG image.
 */
class inputInlineQueryResultPhoto final : public InputInlineQueryResult {
 public:
  /// Unique identifier of the query result.
  string id_;
  /// Title of the result, if known.
  string title_;
  /// A short description of the result, if known.
  string description_;
  /// URL of the photo thumbnail, if it exists.
  string thumbnail_url_;
  /// The URL of the JPEG photo (photo size must not exceed 5MB).
  string photo_url_;
  /// Width of the photo.
  int32 photo_width_;
  /// Height of the photo.
  int32 photo_height_;
  /// The message reply markup; pass null if none. Must be of type replyMarkupInlineKeyboard or null.
  object_ptr<ReplyMarkup> reply_markup_;
  /// The content of the message to be sent. Must be one of the following types: inputMessageText, inputMessagePhoto, inputMessageInvoice, inputMessageLocation, inputMessageVenue or inputMessageContact.
  object_ptr<InputMessageContent> input_message_content_;

  /**
   * Represents link to a JPEG image.
   */
  inputInlineQueryResultPhoto();

  /**
   * Represents link to a JPEG image.
   *
   * \param[in] id_ Unique identifier of the query result.
   * \param[in] title_ Title of the result, if known.
   * \param[in] description_ A short description of the result, if known.
   * \param[in] thumbnail_url_ URL of the photo thumbnail, if it exists.
   * \param[in] photo_url_ The URL of the JPEG photo (photo size must not exceed 5MB).
   * \param[in] photo_width_ Width of the photo.
   * \param[in] photo_height_ Height of the photo.
   * \param[in] reply_markup_ The message reply markup; pass null if none. Must be of type replyMarkupInlineKeyboard or null.
   * \param[in] input_message_content_ The content of the message to be sent. Must be one of the following types: inputMessageText, inputMessagePhoto, inputMessageInvoice, inputMessageLocation, inputMessageVenue or inputMessageContact.
   */
  inputInlineQueryResultPhoto(string const &id_, string const &title_, string const &description_, string const &thumbnail_url_, string const &photo_url_, int32 photo_width_, int32 photo_height_, object_ptr<ReplyMarkup> &&reply_markup_, object_ptr<InputMessageContent> &&input_message_content_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1123338721;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Represents a link to a WEBP or TGS sticker.
 */
class inputInlineQueryResultSticker final : public InputInlineQueryResult {
 public:
  /// Unique identifier of the query result.
  string id_;
  /// URL of the sticker thumbnail, if it exists.
  string thumbnail_url_;
  /// The URL of the WEBP or TGS sticker (sticker file size must not exceed 5MB).
  string sticker_url_;
  /// Width of the sticker.
  int32 sticker_width_;
  /// Height of the sticker.
  int32 sticker_height_;
  /// The message reply markup; pass null if none. Must be of type replyMarkupInlineKeyboard or null.
  object_ptr<ReplyMarkup> reply_markup_;
  /// The content of the message to be sent. Must be one of the following types: inputMessageText, inputMessageSticker, inputMessageInvoice, inputMessageLocation, inputMessageVenue or inputMessageContact.
  object_ptr<InputMessageContent> input_message_content_;

  /**
   * Represents a link to a WEBP or TGS sticker.
   */
  inputInlineQueryResultSticker();

  /**
   * Represents a link to a WEBP or TGS sticker.
   *
   * \param[in] id_ Unique identifier of the query result.
   * \param[in] thumbnail_url_ URL of the sticker thumbnail, if it exists.
   * \param[in] sticker_url_ The URL of the WEBP or TGS sticker (sticker file size must not exceed 5MB).
   * \param[in] sticker_width_ Width of the sticker.
   * \param[in] sticker_height_ Height of the sticker.
   * \param[in] reply_markup_ The message reply markup; pass null if none. Must be of type replyMarkupInlineKeyboard or null.
   * \param[in] input_message_content_ The content of the message to be sent. Must be one of the following types: inputMessageText, inputMessageSticker, inputMessageInvoice, inputMessageLocation, inputMessageVenue or inputMessageContact.
   */
  inputInlineQueryResultSticker(string const &id_, string const &thumbnail_url_, string const &sticker_url_, int32 sticker_width_, int32 sticker_height_, object_ptr<ReplyMarkup> &&reply_markup_, object_ptr<InputMessageContent> &&input_message_content_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 274007129;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Represents information about a venue.
 */
class inputInlineQueryResultVenue final : public InputInlineQueryResult {
 public:
  /// Unique identifier of the query result.
  string id_;
  /// Venue result.
  object_ptr<venue> venue_;
  /// URL of the result thumbnail, if it exists.
  string thumbnail_url_;
  /// Thumbnail width, if known.
  int32 thumbnail_width_;
  /// Thumbnail height, if known.
  int32 thumbnail_height_;
  /// The message reply markup; pass null if none. Must be of type replyMarkupInlineKeyboard or null.
  object_ptr<ReplyMarkup> reply_markup_;
  /// The content of the message to be sent. Must be one of the following types: inputMessageText, inputMessageInvoice, inputMessageLocation, inputMessageVenue or inputMessageContact.
  object_ptr<InputMessageContent> input_message_content_;

  /**
   * Represents information about a venue.
   */
  inputInlineQueryResultVenue();

  /**
   * Represents information about a venue.
   *
   * \param[in] id_ Unique identifier of the query result.
   * \param[in] venue_ Venue result.
   * \param[in] thumbnail_url_ URL of the result thumbnail, if it exists.
   * \param[in] thumbnail_width_ Thumbnail width, if known.
   * \param[in] thumbnail_height_ Thumbnail height, if known.
   * \param[in] reply_markup_ The message reply markup; pass null if none. Must be of type replyMarkupInlineKeyboard or null.
   * \param[in] input_message_content_ The content of the message to be sent. Must be one of the following types: inputMessageText, inputMessageInvoice, inputMessageLocation, inputMessageVenue or inputMessageContact.
   */
  inputInlineQueryResultVenue(string const &id_, object_ptr<venue> &&venue_, string const &thumbnail_url_, int32 thumbnail_width_, int32 thumbnail_height_, object_ptr<ReplyMarkup> &&reply_markup_, object_ptr<InputMessageContent> &&input_message_content_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 541704509;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Represents a link to a page containing an embedded video player or a video file.
 */
class inputInlineQueryResultVideo final : public InputInlineQueryResult {
 public:
  /// Unique identifier of the query result.
  string id_;
  /// Title of the result.
  string title_;
  /// A short description of the result, if known.
  string description_;
  /// The URL of the video thumbnail (JPEG), if it exists.
  string thumbnail_url_;
  /// URL of the embedded video player or video file.
  string video_url_;
  /// MIME type of the content of the video URL, only &quot;text/html&quot; or &quot;video/mp4&quot; are currently supported.
  string mime_type_;
  /// Width of the video.
  int32 video_width_;
  /// Height of the video.
  int32 video_height_;
  /// Video duration, in seconds.
  int32 video_duration_;
  /// The message reply markup; pass null if none. Must be of type replyMarkupInlineKeyboard or null.
  object_ptr<ReplyMarkup> reply_markup_;
  /// The content of the message to be sent. Must be one of the following types: inputMessageText, inputMessageVideo, inputMessageInvoice, inputMessageLocation, inputMessageVenue or inputMessageContact.
  object_ptr<InputMessageContent> input_message_content_;

  /**
   * Represents a link to a page containing an embedded video player or a video file.
   */
  inputInlineQueryResultVideo();

  /**
   * Represents a link to a page containing an embedded video player or a video file.
   *
   * \param[in] id_ Unique identifier of the query result.
   * \param[in] title_ Title of the result.
   * \param[in] description_ A short description of the result, if known.
   * \param[in] thumbnail_url_ The URL of the video thumbnail (JPEG), if it exists.
   * \param[in] video_url_ URL of the embedded video player or video file.
   * \param[in] mime_type_ MIME type of the content of the video URL, only &quot;text/html&quot; or &quot;video/mp4&quot; are currently supported.
   * \param[in] video_width_ Width of the video.
   * \param[in] video_height_ Height of the video.
   * \param[in] video_duration_ Video duration, in seconds.
   * \param[in] reply_markup_ The message reply markup; pass null if none. Must be of type replyMarkupInlineKeyboard or null.
   * \param[in] input_message_content_ The content of the message to be sent. Must be one of the following types: inputMessageText, inputMessageVideo, inputMessageInvoice, inputMessageLocation, inputMessageVenue or inputMessageContact.
   */
  inputInlineQueryResultVideo(string const &id_, string const &title_, string const &description_, string const &thumbnail_url_, string const &video_url_, string const &mime_type_, int32 video_width_, int32 video_height_, int32 video_duration_, object_ptr<ReplyMarkup> &&reply_markup_, object_ptr<InputMessageContent> &&input_message_content_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1724073191;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Represents a link to an opus-encoded audio file within an OGG container, single channel audio.
 */
class inputInlineQueryResultVoiceNote final : public InputInlineQueryResult {
 public:
  /// Unique identifier of the query result.
  string id_;
  /// Title of the voice note.
  string title_;
  /// The URL of the voice note file.
  string voice_note_url_;
  /// Duration of the voice note, in seconds.
  int32 voice_note_duration_;
  /// The message reply markup; pass null if none. Must be of type replyMarkupInlineKeyboard or null.
  object_ptr<ReplyMarkup> reply_markup_;
  /// The content of the message to be sent. Must be one of the following types: inputMessageText, inputMessageVoiceNote, inputMessageInvoice, inputMessageLocation, inputMessageVenue or inputMessageContact.
  object_ptr<InputMessageContent> input_message_content_;

  /**
   * Represents a link to an opus-encoded audio file within an OGG container, single channel audio.
   */
  inputInlineQueryResultVoiceNote();

  /**
   * Represents a link to an opus-encoded audio file within an OGG container, single channel audio.
   *
   * \param[in] id_ Unique identifier of the query result.
   * \param[in] title_ Title of the voice note.
   * \param[in] voice_note_url_ The URL of the voice note file.
   * \param[in] voice_note_duration_ Duration of the voice note, in seconds.
   * \param[in] reply_markup_ The message reply markup; pass null if none. Must be of type replyMarkupInlineKeyboard or null.
   * \param[in] input_message_content_ The content of the message to be sent. Must be one of the following types: inputMessageText, inputMessageVoiceNote, inputMessageInvoice, inputMessageLocation, inputMessageVenue or inputMessageContact.
   */
  inputInlineQueryResultVoiceNote(string const &id_, string const &title_, string const &voice_note_url_, int32 voice_note_duration_, object_ptr<ReplyMarkup> &&reply_markup_, object_ptr<InputMessageContent> &&input_message_content_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1790072503;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * The content of a message to send.
 */
class InputMessageContent: public Object {
 public:
};

/**
 * A text message.
 */
class inputMessageText final : public InputMessageContent {
 public:
  /// Formatted text to be sent; 1-GetOption(&quot;message_text_length_max&quot;) characters. Only Bold, Italic, Underline, Strikethrough, Code, Pre, PreCode, TextUrl and MentionName entities are allowed to be specified manually.
  object_ptr<formattedText> text_;
  /// True, if rich web page previews for URLs in the message text must be disabled.
  bool disable_web_page_preview_;
  /// True, if a chat message draft must be deleted.
  bool clear_draft_;

  /**
   * A text message.
   */
  inputMessageText();

  /**
   * A text message.
   *
   * \param[in] text_ Formatted text to be sent; 1-GetOption(&quot;message_text_length_max&quot;) characters. Only Bold, Italic, Underline, Strikethrough, Code, Pre, PreCode, TextUrl and MentionName entities are allowed to be specified manually.
   * \param[in] disable_web_page_preview_ True, if rich web page previews for URLs in the message text must be disabled.
   * \param[in] clear_draft_ True, if a chat message draft must be deleted.
   */
  inputMessageText(object_ptr<formattedText> &&text_, bool disable_web_page_preview_, bool clear_draft_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 247050392;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * An animation message (GIF-style).
 */
class inputMessageAnimation final : public InputMessageContent {
 public:
  /// Animation file to be sent.
  object_ptr<InputFile> animation_;
  /// Animation thumbnail; pass null to skip thumbnail uploading.
  object_ptr<inputThumbnail> thumbnail_;
  /// File identifiers of the stickers added to the animation, if applicable.
  array<int32> added_sticker_file_ids_;
  /// Duration of the animation, in seconds.
  int32 duration_;
  /// Width of the animation; may be replaced by the server.
  int32 width_;
  /// Height of the animation; may be replaced by the server.
  int32 height_;
  /// Animation caption; pass null to use an empty caption; 0-GetOption(&quot;message_caption_length_max&quot;) characters.
  object_ptr<formattedText> caption_;

  /**
   * An animation message (GIF-style).
   */
  inputMessageAnimation();

  /**
   * An animation message (GIF-style).
   *
   * \param[in] animation_ Animation file to be sent.
   * \param[in] thumbnail_ Animation thumbnail; pass null to skip thumbnail uploading.
   * \param[in] added_sticker_file_ids_ File identifiers of the stickers added to the animation, if applicable.
   * \param[in] duration_ Duration of the animation, in seconds.
   * \param[in] width_ Width of the animation; may be replaced by the server.
   * \param[in] height_ Height of the animation; may be replaced by the server.
   * \param[in] caption_ Animation caption; pass null to use an empty caption; 0-GetOption(&quot;message_caption_length_max&quot;) characters.
   */
  inputMessageAnimation(object_ptr<InputFile> &&animation_, object_ptr<inputThumbnail> &&thumbnail_, array<int32> &&added_sticker_file_ids_, int32 duration_, int32 width_, int32 height_, object_ptr<formattedText> &&caption_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1208433535;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * An audio message.
 */
class inputMessageAudio final : public InputMessageContent {
 public:
  /// Audio file to be sent.
  object_ptr<InputFile> audio_;
  /// Thumbnail of the cover for the album; pass null to skip thumbnail uploading.
  object_ptr<inputThumbnail> album_cover_thumbnail_;
  /// Duration of the audio, in seconds; may be replaced by the server.
  int32 duration_;
  /// Title of the audio; 0-64 characters; may be replaced by the server.
  string title_;
  /// Performer of the audio; 0-64 characters, may be replaced by the server.
  string performer_;
  /// Audio caption; pass null to use an empty caption; 0-GetOption(&quot;message_caption_length_max&quot;) characters.
  object_ptr<formattedText> caption_;

  /**
   * An audio message.
   */
  inputMessageAudio();

  /**
   * An audio message.
   *
   * \param[in] audio_ Audio file to be sent.
   * \param[in] album_cover_thumbnail_ Thumbnail of the cover for the album; pass null to skip thumbnail uploading.
   * \param[in] duration_ Duration of the audio, in seconds; may be replaced by the server.
   * \param[in] title_ Title of the audio; 0-64 characters; may be replaced by the server.
   * \param[in] performer_ Performer of the audio; 0-64 characters, may be replaced by the server.
   * \param[in] caption_ Audio caption; pass null to use an empty caption; 0-GetOption(&quot;message_caption_length_max&quot;) characters.
   */
  inputMessageAudio(object_ptr<InputFile> &&audio_, object_ptr<inputThumbnail> &&album_cover_thumbnail_, int32 duration_, string const &title_, string const &performer_, object_ptr<formattedText> &&caption_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -626786126;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A document message (general file).
 */
class inputMessageDocument final : public InputMessageContent {
 public:
  /// Document to be sent.
  object_ptr<InputFile> document_;
  /// Document thumbnail; pass null to skip thumbnail uploading.
  object_ptr<inputThumbnail> thumbnail_;
  /// If true, automatic file type detection will be disabled and the document will be always sent as file. Always true for files sent to secret chats.
  bool disable_content_type_detection_;
  /// Document caption; pass null to use an empty caption; 0-GetOption(&quot;message_caption_length_max&quot;) characters.
  object_ptr<formattedText> caption_;

  /**
   * A document message (general file).
   */
  inputMessageDocument();

  /**
   * A document message (general file).
   *
   * \param[in] document_ Document to be sent.
   * \param[in] thumbnail_ Document thumbnail; pass null to skip thumbnail uploading.
   * \param[in] disable_content_type_detection_ If true, automatic file type detection will be disabled and the document will be always sent as file. Always true for files sent to secret chats.
   * \param[in] caption_ Document caption; pass null to use an empty caption; 0-GetOption(&quot;message_caption_length_max&quot;) characters.
   */
  inputMessageDocument(object_ptr<InputFile> &&document_, object_ptr<inputThumbnail> &&thumbnail_, bool disable_content_type_detection_, object_ptr<formattedText> &&caption_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1633383097;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A photo message.
 */
class inputMessagePhoto final : public InputMessageContent {
 public:
  /// Photo to send.
  object_ptr<InputFile> photo_;
  /// Photo thumbnail to be sent; pass null to skip thumbnail uploading. The thumbnail is sent to the other party only in secret chats.
  object_ptr<inputThumbnail> thumbnail_;
  /// File identifiers of the stickers added to the photo, if applicable.
  array<int32> added_sticker_file_ids_;
  /// Photo width.
  int32 width_;
  /// Photo height.
  int32 height_;
  /// Photo caption; pass null to use an empty caption; 0-GetOption(&quot;message_caption_length_max&quot;) characters.
  object_ptr<formattedText> caption_;
  /// Photo TTL (Time To Live), in seconds (0-60). A non-zero TTL can be specified only in private chats.
  int32 ttl_;

  /**
   * A photo message.
   */
  inputMessagePhoto();

  /**
   * A photo message.
   *
   * \param[in] photo_ Photo to send.
   * \param[in] thumbnail_ Photo thumbnail to be sent; pass null to skip thumbnail uploading. The thumbnail is sent to the other party only in secret chats.
   * \param[in] added_sticker_file_ids_ File identifiers of the stickers added to the photo, if applicable.
   * \param[in] width_ Photo width.
   * \param[in] height_ Photo height.
   * \param[in] caption_ Photo caption; pass null to use an empty caption; 0-GetOption(&quot;message_caption_length_max&quot;) characters.
   * \param[in] ttl_ Photo TTL (Time To Live), in seconds (0-60). A non-zero TTL can be specified only in private chats.
   */
  inputMessagePhoto(object_ptr<InputFile> &&photo_, object_ptr<inputThumbnail> &&thumbnail_, array<int32> &&added_sticker_file_ids_, int32 width_, int32 height_, object_ptr<formattedText> &&caption_, int32 ttl_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1648801584;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A sticker message.
 */
class inputMessageSticker final : public InputMessageContent {
 public:
  /// Sticker to be sent.
  object_ptr<InputFile> sticker_;
  /// Sticker thumbnail; pass null to skip thumbnail uploading.
  object_ptr<inputThumbnail> thumbnail_;
  /// Sticker width.
  int32 width_;
  /// Sticker height.
  int32 height_;
  /// Emoji used to choose the sticker.
  string emoji_;

  /**
   * A sticker message.
   */
  inputMessageSticker();

  /**
   * A sticker message.
   *
   * \param[in] sticker_ Sticker to be sent.
   * \param[in] thumbnail_ Sticker thumbnail; pass null to skip thumbnail uploading.
   * \param[in] width_ Sticker width.
   * \param[in] height_ Sticker height.
   * \param[in] emoji_ Emoji used to choose the sticker.
   */
  inputMessageSticker(object_ptr<InputFile> &&sticker_, object_ptr<inputThumbnail> &&thumbnail_, int32 width_, int32 height_, string const &emoji_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1072805625;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A video message.
 */
class inputMessageVideo final : public InputMessageContent {
 public:
  /// Video to be sent.
  object_ptr<InputFile> video_;
  /// Video thumbnail; pass null to skip thumbnail uploading.
  object_ptr<inputThumbnail> thumbnail_;
  /// File identifiers of the stickers added to the video, if applicable.
  array<int32> added_sticker_file_ids_;
  /// Duration of the video, in seconds.
  int32 duration_;
  /// Video width.
  int32 width_;
  /// Video height.
  int32 height_;
  /// True, if the video is supposed to be streamed.
  bool supports_streaming_;
  /// Video caption; pass null to use an empty caption; 0-GetOption(&quot;message_caption_length_max&quot;) characters.
  object_ptr<formattedText> caption_;
  /// Video TTL (Time To Live), in seconds (0-60). A non-zero TTL can be specified only in private chats.
  int32 ttl_;

  /**
   * A video message.
   */
  inputMessageVideo();

  /**
   * A video message.
   *
   * \param[in] video_ Video to be sent.
   * \param[in] thumbnail_ Video thumbnail; pass null to skip thumbnail uploading.
   * \param[in] added_sticker_file_ids_ File identifiers of the stickers added to the video, if applicable.
   * \param[in] duration_ Duration of the video, in seconds.
   * \param[in] width_ Video width.
   * \param[in] height_ Video height.
   * \param[in] supports_streaming_ True, if the video is supposed to be streamed.
   * \param[in] caption_ Video caption; pass null to use an empty caption; 0-GetOption(&quot;message_caption_length_max&quot;) characters.
   * \param[in] ttl_ Video TTL (Time To Live), in seconds (0-60). A non-zero TTL can be specified only in private chats.
   */
  inputMessageVideo(object_ptr<InputFile> &&video_, object_ptr<inputThumbnail> &&thumbnail_, array<int32> &&added_sticker_file_ids_, int32 duration_, int32 width_, int32 height_, bool supports_streaming_, object_ptr<formattedText> &&caption_, int32 ttl_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -2108486755;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A video note message.
 */
class inputMessageVideoNote final : public InputMessageContent {
 public:
  /// Video note to be sent.
  object_ptr<InputFile> video_note_;
  /// Video thumbnail; pass null to skip thumbnail uploading.
  object_ptr<inputThumbnail> thumbnail_;
  /// Duration of the video, in seconds.
  int32 duration_;
  /// Video width and height; must be positive and not greater than 640.
  int32 length_;

  /**
   * A video note message.
   */
  inputMessageVideoNote();

  /**
   * A video note message.
   *
   * \param[in] video_note_ Video note to be sent.
   * \param[in] thumbnail_ Video thumbnail; pass null to skip thumbnail uploading.
   * \param[in] duration_ Duration of the video, in seconds.
   * \param[in] length_ Video width and height; must be positive and not greater than 640.
   */
  inputMessageVideoNote(object_ptr<InputFile> &&video_note_, object_ptr<inputThumbnail> &&thumbnail_, int32 duration_, int32 length_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 279108859;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A voice note message.
 */
class inputMessageVoiceNote final : public InputMessageContent {
 public:
  /// Voice note to be sent.
  object_ptr<InputFile> voice_note_;
  /// Duration of the voice note, in seconds.
  int32 duration_;
  /// Waveform representation of the voice note, in 5-bit format.
  bytes waveform_;
  /// Voice note caption; pass null to use an empty caption; 0-GetOption(&quot;message_caption_length_max&quot;) characters.
  object_ptr<formattedText> caption_;

  /**
   * A voice note message.
   */
  inputMessageVoiceNote();

  /**
   * A voice note message.
   *
   * \param[in] voice_note_ Voice note to be sent.
   * \param[in] duration_ Duration of the voice note, in seconds.
   * \param[in] waveform_ Waveform representation of the voice note, in 5-bit format.
   * \param[in] caption_ Voice note caption; pass null to use an empty caption; 0-GetOption(&quot;message_caption_length_max&quot;) characters.
   */
  inputMessageVoiceNote(object_ptr<InputFile> &&voice_note_, int32 duration_, bytes const &waveform_, object_ptr<formattedText> &&caption_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 2136519657;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A message with a location.
 */
class inputMessageLocation final : public InputMessageContent {
 public:
  /// Location to be sent.
  object_ptr<location> location_;
  /// Period for which the location can be updated, in seconds; must be between 60 and 86400 for a live location and 0 otherwise.
  int32 live_period_;
  /// For live locations, a direction in which the location moves, in degrees; 1-360. Pass 0 if unknown.
  int32 heading_;
  /// For live locations, a maximum distance to another chat member for proximity alerts, in meters (0-100000). Pass 0 if the notification is disabled. Can&\#039;t be enabled in channels and Saved Messages.
  int32 proximity_alert_radius_;

  /**
   * A message with a location.
   */
  inputMessageLocation();

  /**
   * A message with a location.
   *
   * \param[in] location_ Location to be sent.
   * \param[in] live_period_ Period for which the location can be updated, in seconds; must be between 60 and 86400 for a live location and 0 otherwise.
   * \param[in] heading_ For live locations, a direction in which the location moves, in degrees; 1-360. Pass 0 if unknown.
   * \param[in] proximity_alert_radius_ For live locations, a maximum distance to another chat member for proximity alerts, in meters (0-100000). Pass 0 if the notification is disabled. Can&\#039;t be enabled in channels and Saved Messages.
   */
  inputMessageLocation(object_ptr<location> &&location_, int32 live_period_, int32 heading_, int32 proximity_alert_radius_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 648735088;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A message with information about a venue.
 */
class inputMessageVenue final : public InputMessageContent {
 public:
  /// Venue to send.
  object_ptr<venue> venue_;

  /**
   * A message with information about a venue.
   */
  inputMessageVenue();

  /**
   * A message with information about a venue.
   *
   * \param[in] venue_ Venue to send.
   */
  explicit inputMessageVenue(object_ptr<venue> &&venue_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1447926269;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A message containing a user contact.
 */
class inputMessageContact final : public InputMessageContent {
 public:
  /// Contact to send.
  object_ptr<contact> contact_;

  /**
   * A message containing a user contact.
   */
  inputMessageContact();

  /**
   * A message containing a user contact.
   *
   * \param[in] contact_ Contact to send.
   */
  explicit inputMessageContact(object_ptr<contact> &&contact_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -982446849;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A dice message.
 */
class inputMessageDice final : public InputMessageContent {
 public:
  /// Emoji on which the dice throw animation is based.
  string emoji_;
  /// True, if the chat message draft must be deleted.
  bool clear_draft_;

  /**
   * A dice message.
   */
  inputMessageDice();

  /**
   * A dice message.
   *
   * \param[in] emoji_ Emoji on which the dice throw animation is based.
   * \param[in] clear_draft_ True, if the chat message draft must be deleted.
   */
  inputMessageDice(string const &emoji_, bool clear_draft_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 841574313;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A message with a game; not supported for channels or secret chats.
 */
class inputMessageGame final : public InputMessageContent {
 public:
  /// User identifier of the bot that owns the game.
  int53 bot_user_id_;
  /// Short name of the game.
  string game_short_name_;

  /**
   * A message with a game; not supported for channels or secret chats.
   */
  inputMessageGame();

  /**
   * A message with a game; not supported for channels or secret chats.
   *
   * \param[in] bot_user_id_ User identifier of the bot that owns the game.
   * \param[in] game_short_name_ Short name of the game.
   */
  inputMessageGame(int53 bot_user_id_, string const &game_short_name_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1252944610;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A message with an invoice; can be used only by bots.
 */
class inputMessageInvoice final : public InputMessageContent {
 public:
  /// Invoice.
  object_ptr<invoice> invoice_;
  /// Product title; 1-32 characters.
  string title_;
  /// Product description; 0-255 characters.
  string description_;
  /// Product photo URL; optional.
  string photo_url_;
  /// Product photo size.
  int32 photo_size_;
  /// Product photo width.
  int32 photo_width_;
  /// Product photo height.
  int32 photo_height_;
  /// The invoice payload.
  bytes payload_;
  /// Payment provider token.
  string provider_token_;
  /// JSON-encoded data about the invoice, which will be shared with the payment provider.
  string provider_data_;
  /// Unique invoice bot deep link parameter for the generation of this invoice. If empty, it would be possible to pay directly from forwards of the invoice message.
  string start_parameter_;

  /**
   * A message with an invoice; can be used only by bots.
   */
  inputMessageInvoice();

  /**
   * A message with an invoice; can be used only by bots.
   *
   * \param[in] invoice_ Invoice.
   * \param[in] title_ Product title; 1-32 characters.
   * \param[in] description_ Product description; 0-255 characters.
   * \param[in] photo_url_ Product photo URL; optional.
   * \param[in] photo_size_ Product photo size.
   * \param[in] photo_width_ Product photo width.
   * \param[in] photo_height_ Product photo height.
   * \param[in] payload_ The invoice payload.
   * \param[in] provider_token_ Payment provider token.
   * \param[in] provider_data_ JSON-encoded data about the invoice, which will be shared with the payment provider.
   * \param[in] start_parameter_ Unique invoice bot deep link parameter for the generation of this invoice. If empty, it would be possible to pay directly from forwards of the invoice message.
   */
  inputMessageInvoice(object_ptr<invoice> &&invoice_, string const &title_, string const &description_, string const &photo_url_, int32 photo_size_, int32 photo_width_, int32 photo_height_, bytes const &payload_, string const &provider_token_, string const &provider_data_, string const &start_parameter_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1038812175;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A message with a poll. Polls can&\#039;t be sent to secret chats. Polls can be sent only to a private chat with a bot.
 */
class inputMessagePoll final : public InputMessageContent {
 public:
  /// Poll question; 1-255 characters (up to 300 characters for bots).
  string question_;
  /// List of poll answer options, 2-10 strings 1-100 characters each.
  array<string> options_;
  /// True, if the poll voters are anonymous. Non-anonymous polls can&\#039;t be sent or forwarded to channels.
  bool is_anonymous_;
  /// Type of the poll.
  object_ptr<PollType> type_;
  /// Amount of time the poll will be active after creation, in seconds; for bots only.
  int32 open_period_;
  /// Point in time (Unix timestamp) when the poll will automatically be closed; for bots only.
  int32 close_date_;
  /// True, if the poll needs to be sent already closed; for bots only.
  bool is_closed_;

  /**
   * A message with a poll. Polls can&\#039;t be sent to secret chats. Polls can be sent only to a private chat with a bot.
   */
  inputMessagePoll();

  /**
   * A message with a poll. Polls can&\#039;t be sent to secret chats. Polls can be sent only to a private chat with a bot.
   *
   * \param[in] question_ Poll question; 1-255 characters (up to 300 characters for bots).
   * \param[in] options_ List of poll answer options, 2-10 strings 1-100 characters each.
   * \param[in] is_anonymous_ True, if the poll voters are anonymous. Non-anonymous polls can&\#039;t be sent or forwarded to channels.
   * \param[in] type_ Type of the poll.
   * \param[in] open_period_ Amount of time the poll will be active after creation, in seconds; for bots only.
   * \param[in] close_date_ Point in time (Unix timestamp) when the poll will automatically be closed; for bots only.
   * \param[in] is_closed_ True, if the poll needs to be sent already closed; for bots only.
   */
  inputMessagePoll(string const &question_, array<string> &&options_, bool is_anonymous_, object_ptr<PollType> &&type_, int32 open_period_, int32 close_date_, bool is_closed_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 2054629900;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A forwarded message.
 */
class inputMessageForwarded final : public InputMessageContent {
 public:
  /// Identifier for the chat this forwarded message came from.
  int53 from_chat_id_;
  /// Identifier of the message to forward.
  int53 message_id_;
  /// True, if a game message is being shared from a launched game; applies only to game messages.
  bool in_game_share_;
  /// Options to be used to copy content of the message without reference to the original sender; pass null to forward the message as usual.
  object_ptr<messageCopyOptions> copy_options_;

  /**
   * A forwarded message.
   */
  inputMessageForwarded();

  /**
   * A forwarded message.
   *
   * \param[in] from_chat_id_ Identifier for the chat this forwarded message came from.
   * \param[in] message_id_ Identifier of the message to forward.
   * \param[in] in_game_share_ True, if a game message is being shared from a launched game; applies only to game messages.
   * \param[in] copy_options_ Options to be used to copy content of the message without reference to the original sender; pass null to forward the message as usual.
   */
  inputMessageForwarded(int53 from_chat_id_, int53 message_id_, bool in_game_share_, object_ptr<messageCopyOptions> &&copy_options_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1696232440;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Contains information about a Telegram Passport element to be saved.
 */
class InputPassportElement: public Object {
 public:
};

/**
 * A Telegram Passport element to be saved containing the user&\#039;s personal details.
 */
class inputPassportElementPersonalDetails final : public InputPassportElement {
 public:
  /// Personal details of the user.
  object_ptr<personalDetails> personal_details_;

  /**
   * A Telegram Passport element to be saved containing the user&\#039;s personal details.
   */
  inputPassportElementPersonalDetails();

  /**
   * A Telegram Passport element to be saved containing the user&\#039;s personal details.
   *
   * \param[in] personal_details_ Personal details of the user.
   */
  explicit inputPassportElementPersonalDetails(object_ptr<personalDetails> &&personal_details_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 164791359;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A Telegram Passport element to be saved containing the user&\#039;s passport.
 */
class inputPassportElementPassport final : public InputPassportElement {
 public:
  /// The passport to be saved.
  object_ptr<inputIdentityDocument> passport_;

  /**
   * A Telegram Passport element to be saved containing the user&\#039;s passport.
   */
  inputPassportElementPassport();

  /**
   * A Telegram Passport element to be saved containing the user&\#039;s passport.
   *
   * \param[in] passport_ The passport to be saved.
   */
  explicit inputPassportElementPassport(object_ptr<inputIdentityDocument> &&passport_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -497011356;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A Telegram Passport element to be saved containing the user&\#039;s driver license.
 */
class inputPassportElementDriverLicense final : public InputPassportElement {
 public:
  /// The driver license to be saved.
  object_ptr<inputIdentityDocument> driver_license_;

  /**
   * A Telegram Passport element to be saved containing the user&\#039;s driver license.
   */
  inputPassportElementDriverLicense();

  /**
   * A Telegram Passport element to be saved containing the user&\#039;s driver license.
   *
   * \param[in] driver_license_ The driver license to be saved.
   */
  explicit inputPassportElementDriverLicense(object_ptr<inputIdentityDocument> &&driver_license_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 304813264;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A Telegram Passport element to be saved containing the user&\#039;s identity card.
 */
class inputPassportElementIdentityCard final : public InputPassportElement {
 public:
  /// The identity card to be saved.
  object_ptr<inputIdentityDocument> identity_card_;

  /**
   * A Telegram Passport element to be saved containing the user&\#039;s identity card.
   */
  inputPassportElementIdentityCard();

  /**
   * A Telegram Passport element to be saved containing the user&\#039;s identity card.
   *
   * \param[in] identity_card_ The identity card to be saved.
   */
  explicit inputPassportElementIdentityCard(object_ptr<inputIdentityDocument> &&identity_card_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -9963390;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A Telegram Passport element to be saved containing the user&\#039;s internal passport.
 */
class inputPassportElementInternalPassport final : public InputPassportElement {
 public:
  /// The internal passport to be saved.
  object_ptr<inputIdentityDocument> internal_passport_;

  /**
   * A Telegram Passport element to be saved containing the user&\#039;s internal passport.
   */
  inputPassportElementInternalPassport();

  /**
   * A Telegram Passport element to be saved containing the user&\#039;s internal passport.
   *
   * \param[in] internal_passport_ The internal passport to be saved.
   */
  explicit inputPassportElementInternalPassport(object_ptr<inputIdentityDocument> &&internal_passport_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 715360043;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A Telegram Passport element to be saved containing the user&\#039;s address.
 */
class inputPassportElementAddress final : public InputPassportElement {
 public:
  /// The address to be saved.
  object_ptr<address> address_;

  /**
   * A Telegram Passport element to be saved containing the user&\#039;s address.
   */
  inputPassportElementAddress();

  /**
   * A Telegram Passport element to be saved containing the user&\#039;s address.
   *
   * \param[in] address_ The address to be saved.
   */
  explicit inputPassportElementAddress(object_ptr<address> &&address_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 461630480;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A Telegram Passport element to be saved containing the user&\#039;s utility bill.
 */
class inputPassportElementUtilityBill final : public InputPassportElement {
 public:
  /// The utility bill to be saved.
  object_ptr<inputPersonalDocument> utility_bill_;

  /**
   * A Telegram Passport element to be saved containing the user&\#039;s utility bill.
   */
  inputPassportElementUtilityBill();

  /**
   * A Telegram Passport element to be saved containing the user&\#039;s utility bill.
   *
   * \param[in] utility_bill_ The utility bill to be saved.
   */
  explicit inputPassportElementUtilityBill(object_ptr<inputPersonalDocument> &&utility_bill_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1389203841;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A Telegram Passport element to be saved containing the user&\#039;s bank statement.
 */
class inputPassportElementBankStatement final : public InputPassportElement {
 public:
  /// The bank statement to be saved.
  object_ptr<inputPersonalDocument> bank_statement_;

  /**
   * A Telegram Passport element to be saved containing the user&\#039;s bank statement.
   */
  inputPassportElementBankStatement();

  /**
   * A Telegram Passport element to be saved containing the user&\#039;s bank statement.
   *
   * \param[in] bank_statement_ The bank statement to be saved.
   */
  explicit inputPassportElementBankStatement(object_ptr<inputPersonalDocument> &&bank_statement_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -26585208;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A Telegram Passport element to be saved containing the user&\#039;s rental agreement.
 */
class inputPassportElementRentalAgreement final : public InputPassportElement {
 public:
  /// The rental agreement to be saved.
  object_ptr<inputPersonalDocument> rental_agreement_;

  /**
   * A Telegram Passport element to be saved containing the user&\#039;s rental agreement.
   */
  inputPassportElementRentalAgreement();

  /**
   * A Telegram Passport element to be saved containing the user&\#039;s rental agreement.
   *
   * \param[in] rental_agreement_ The rental agreement to be saved.
   */
  explicit inputPassportElementRentalAgreement(object_ptr<inputPersonalDocument> &&rental_agreement_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1736154155;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A Telegram Passport element to be saved containing the user&\#039;s passport registration.
 */
class inputPassportElementPassportRegistration final : public InputPassportElement {
 public:
  /// The passport registration page to be saved.
  object_ptr<inputPersonalDocument> passport_registration_;

  /**
   * A Telegram Passport element to be saved containing the user&\#039;s passport registration.
   */
  inputPassportElementPassportRegistration();

  /**
   * A Telegram Passport element to be saved containing the user&\#039;s passport registration.
   *
   * \param[in] passport_registration_ The passport registration page to be saved.
   */
  explicit inputPassportElementPassportRegistration(object_ptr<inputPersonalDocument> &&passport_registration_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1314562128;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A Telegram Passport element to be saved containing the user&\#039;s temporary registration.
 */
class inputPassportElementTemporaryRegistration final : public InputPassportElement {
 public:
  /// The temporary registration document to be saved.
  object_ptr<inputPersonalDocument> temporary_registration_;

  /**
   * A Telegram Passport element to be saved containing the user&\#039;s temporary registration.
   */
  inputPassportElementTemporaryRegistration();

  /**
   * A Telegram Passport element to be saved containing the user&\#039;s temporary registration.
   *
   * \param[in] temporary_registration_ The temporary registration document to be saved.
   */
  explicit inputPassportElementTemporaryRegistration(object_ptr<inputPersonalDocument> &&temporary_registration_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1913238047;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A Telegram Passport element to be saved containing the user&\#039;s phone number.
 */
class inputPassportElementPhoneNumber final : public InputPassportElement {
 public:
  /// The phone number to be saved.
  string phone_number_;

  /**
   * A Telegram Passport element to be saved containing the user&\#039;s phone number.
   */
  inputPassportElementPhoneNumber();

  /**
   * A Telegram Passport element to be saved containing the user&\#039;s phone number.
   *
   * \param[in] phone_number_ The phone number to be saved.
   */
  explicit inputPassportElementPhoneNumber(string const &phone_number_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1319357497;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A Telegram Passport element to be saved containing the user&\#039;s email address.
 */
class inputPassportElementEmailAddress final : public InputPassportElement {
 public:
  /// The email address to be saved.
  string email_address_;

  /**
   * A Telegram Passport element to be saved containing the user&\#039;s email address.
   */
  inputPassportElementEmailAddress();

  /**
   * A Telegram Passport element to be saved containing the user&\#039;s email address.
   *
   * \param[in] email_address_ The email address to be saved.
   */
  explicit inputPassportElementEmailAddress(string const &email_address_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -248605659;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains the description of an error in a Telegram Passport element; for bots only.
 */
class inputPassportElementError final : public Object {
 public:
  /// Type of Telegram Passport element that has the error.
  object_ptr<PassportElementType> type_;
  /// Error message.
  string message_;
  /// Error source.
  object_ptr<InputPassportElementErrorSource> source_;

  /**
   * Contains the description of an error in a Telegram Passport element; for bots only.
   */
  inputPassportElementError();

  /**
   * Contains the description of an error in a Telegram Passport element; for bots only.
   *
   * \param[in] type_ Type of Telegram Passport element that has the error.
   * \param[in] message_ Error message.
   * \param[in] source_ Error source.
   */
  inputPassportElementError(object_ptr<PassportElementType> &&type_, string const &message_, object_ptr<InputPassportElementErrorSource> &&source_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 285756898;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Contains the description of an error in a Telegram Passport element; for bots only.
 */
class InputPassportElementErrorSource: public Object {
 public:
};

/**
 * The element contains an error in an unspecified place. The error will be considered resolved when new data is added.
 */
class inputPassportElementErrorSourceUnspecified final : public InputPassportElementErrorSource {
 public:
  /// Current hash of the entire element.
  bytes element_hash_;

  /**
   * The element contains an error in an unspecified place. The error will be considered resolved when new data is added.
   */
  inputPassportElementErrorSourceUnspecified();

  /**
   * The element contains an error in an unspecified place. The error will be considered resolved when new data is added.
   *
   * \param[in] element_hash_ Current hash of the entire element.
   */
  explicit inputPassportElementErrorSourceUnspecified(bytes const &element_hash_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 267230319;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A data field contains an error. The error is considered resolved when the field&\#039;s value changes.
 */
class inputPassportElementErrorSourceDataField final : public InputPassportElementErrorSource {
 public:
  /// Field name.
  string field_name_;
  /// Current data hash.
  bytes data_hash_;

  /**
   * A data field contains an error. The error is considered resolved when the field&\#039;s value changes.
   */
  inputPassportElementErrorSourceDataField();

  /**
   * A data field contains an error. The error is considered resolved when the field&\#039;s value changes.
   *
   * \param[in] field_name_ Field name.
   * \param[in] data_hash_ Current data hash.
   */
  inputPassportElementErrorSourceDataField(string const &field_name_, bytes const &data_hash_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -426795002;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The front side of the document contains an error. The error is considered resolved when the file with the front side of the document changes.
 */
class inputPassportElementErrorSourceFrontSide final : public InputPassportElementErrorSource {
 public:
  /// Current hash of the file containing the front side.
  bytes file_hash_;

  /**
   * The front side of the document contains an error. The error is considered resolved when the file with the front side of the document changes.
   */
  inputPassportElementErrorSourceFrontSide();

  /**
   * The front side of the document contains an error. The error is considered resolved when the file with the front side of the document changes.
   *
   * \param[in] file_hash_ Current hash of the file containing the front side.
   */
  explicit inputPassportElementErrorSourceFrontSide(bytes const &file_hash_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 588023741;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The reverse side of the document contains an error. The error is considered resolved when the file with the reverse side of the document changes.
 */
class inputPassportElementErrorSourceReverseSide final : public InputPassportElementErrorSource {
 public:
  /// Current hash of the file containing the reverse side.
  bytes file_hash_;

  /**
   * The reverse side of the document contains an error. The error is considered resolved when the file with the reverse side of the document changes.
   */
  inputPassportElementErrorSourceReverseSide();

  /**
   * The reverse side of the document contains an error. The error is considered resolved when the file with the reverse side of the document changes.
   *
   * \param[in] file_hash_ Current hash of the file containing the reverse side.
   */
  explicit inputPassportElementErrorSourceReverseSide(bytes const &file_hash_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 413072891;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The selfie contains an error. The error is considered resolved when the file with the selfie changes.
 */
class inputPassportElementErrorSourceSelfie final : public InputPassportElementErrorSource {
 public:
  /// Current hash of the file containing the selfie.
  bytes file_hash_;

  /**
   * The selfie contains an error. The error is considered resolved when the file with the selfie changes.
   */
  inputPassportElementErrorSourceSelfie();

  /**
   * The selfie contains an error. The error is considered resolved when the file with the selfie changes.
   *
   * \param[in] file_hash_ Current hash of the file containing the selfie.
   */
  explicit inputPassportElementErrorSourceSelfie(bytes const &file_hash_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -773575528;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * One of the files containing the translation of the document contains an error. The error is considered resolved when the file with the translation changes.
 */
class inputPassportElementErrorSourceTranslationFile final : public InputPassportElementErrorSource {
 public:
  /// Current hash of the file containing the translation.
  bytes file_hash_;

  /**
   * One of the files containing the translation of the document contains an error. The error is considered resolved when the file with the translation changes.
   */
  inputPassportElementErrorSourceTranslationFile();

  /**
   * One of the files containing the translation of the document contains an error. The error is considered resolved when the file with the translation changes.
   *
   * \param[in] file_hash_ Current hash of the file containing the translation.
   */
  explicit inputPassportElementErrorSourceTranslationFile(bytes const &file_hash_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 505842299;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The translation of the document contains an error. The error is considered resolved when the list of files changes.
 */
class inputPassportElementErrorSourceTranslationFiles final : public InputPassportElementErrorSource {
 public:
  /// Current hashes of all files with the translation.
  array<bytes> file_hashes_;

  /**
   * The translation of the document contains an error. The error is considered resolved when the list of files changes.
   */
  inputPassportElementErrorSourceTranslationFiles();

  /**
   * The translation of the document contains an error. The error is considered resolved when the list of files changes.
   *
   * \param[in] file_hashes_ Current hashes of all files with the translation.
   */
  explicit inputPassportElementErrorSourceTranslationFiles(array<bytes> &&file_hashes_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -527254048;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The file contains an error. The error is considered resolved when the file changes.
 */
class inputPassportElementErrorSourceFile final : public InputPassportElementErrorSource {
 public:
  /// Current hash of the file which has the error.
  bytes file_hash_;

  /**
   * The file contains an error. The error is considered resolved when the file changes.
   */
  inputPassportElementErrorSourceFile();

  /**
   * The file contains an error. The error is considered resolved when the file changes.
   *
   * \param[in] file_hash_ Current hash of the file which has the error.
   */
  explicit inputPassportElementErrorSourceFile(bytes const &file_hash_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -298492469;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The list of attached files contains an error. The error is considered resolved when the file list changes.
 */
class inputPassportElementErrorSourceFiles final : public InputPassportElementErrorSource {
 public:
  /// Current hashes of all attached files.
  array<bytes> file_hashes_;

  /**
   * The list of attached files contains an error. The error is considered resolved when the file list changes.
   */
  inputPassportElementErrorSourceFiles();

  /**
   * The list of attached files contains an error. The error is considered resolved when the file list changes.
   *
   * \param[in] file_hashes_ Current hashes of all attached files.
   */
  explicit inputPassportElementErrorSourceFiles(array<bytes> &&file_hashes_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -2008541640;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A personal document to be saved to Telegram Passport.
 */
class inputPersonalDocument final : public Object {
 public:
  /// List of files containing the pages of the document.
  array<object_ptr<InputFile>> files_;
  /// List of files containing a certified English translation of the document.
  array<object_ptr<InputFile>> translation_;

  /**
   * A personal document to be saved to Telegram Passport.
   */
  inputPersonalDocument();

  /**
   * A personal document to be saved to Telegram Passport.
   *
   * \param[in] files_ List of files containing the pages of the document.
   * \param[in] translation_ List of files containing a certified English translation of the document.
   */
  inputPersonalDocument(array<object_ptr<InputFile>> &&files_, array<object_ptr<InputFile>> &&translation_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1676966826;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Describes a sticker that needs to be added to a sticker set.
 */
class InputSticker: public Object {
 public:
};

/**
 * A static sticker in PNG format, which will be converted to WEBP server-side.
 */
class inputStickerStatic final : public InputSticker {
 public:
  /// PNG image with the sticker; must be up to 512 KB in size and fit in a 512x512 square.
  object_ptr<InputFile> sticker_;
  /// Emojis corresponding to the sticker.
  string emojis_;
  /// For masks, position where the mask is placed; pass null if unspecified.
  object_ptr<maskPosition> mask_position_;

  /**
   * A static sticker in PNG format, which will be converted to WEBP server-side.
   */
  inputStickerStatic();

  /**
   * A static sticker in PNG format, which will be converted to WEBP server-side.
   *
   * \param[in] sticker_ PNG image with the sticker; must be up to 512 KB in size and fit in a 512x512 square.
   * \param[in] emojis_ Emojis corresponding to the sticker.
   * \param[in] mask_position_ For masks, position where the mask is placed; pass null if unspecified.
   */
  inputStickerStatic(object_ptr<InputFile> &&sticker_, string const &emojis_, object_ptr<maskPosition> &&mask_position_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1409680603;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * An animated sticker in TGS format.
 */
class inputStickerAnimated final : public InputSticker {
 public:
  /// File with the animated sticker. Only local or uploaded within a week files are supported. See https://core.telegram.org/animated_stickers\#technical-requirements for technical requirements.
  object_ptr<InputFile> sticker_;
  /// Emojis corresponding to the sticker.
  string emojis_;

  /**
   * An animated sticker in TGS format.
   */
  inputStickerAnimated();

  /**
   * An animated sticker in TGS format.
   *
   * \param[in] sticker_ File with the animated sticker. Only local or uploaded within a week files are supported. See https://core.telegram.org/animated_stickers\#technical-requirements for technical requirements.
   * \param[in] emojis_ Emojis corresponding to the sticker.
   */
  inputStickerAnimated(object_ptr<InputFile> &&sticker_, string const &emojis_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1127265952;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A thumbnail to be sent along with a file; must be in JPEG or WEBP format for stickers, and less than 200 KB in size.
 */
class inputThumbnail final : public Object {
 public:
  /// Thumbnail file to send. Sending thumbnails by file_id is currently not supported.
  object_ptr<InputFile> thumbnail_;
  /// Thumbnail width, usually shouldn&\#039;t exceed 320. Use 0 if unknown.
  int32 width_;
  /// Thumbnail height, usually shouldn&\#039;t exceed 320. Use 0 if unknown.
  int32 height_;

  /**
   * A thumbnail to be sent along with a file; must be in JPEG or WEBP format for stickers, and less than 200 KB in size.
   */
  inputThumbnail();

  /**
   * A thumbnail to be sent along with a file; must be in JPEG or WEBP format for stickers, and less than 200 KB in size.
   *
   * \param[in] thumbnail_ Thumbnail file to send. Sending thumbnails by file_id is currently not supported.
   * \param[in] width_ Thumbnail width, usually shouldn&\#039;t exceed 320. Use 0 if unknown.
   * \param[in] height_ Thumbnail height, usually shouldn&\#039;t exceed 320. Use 0 if unknown.
   */
  inputThumbnail(object_ptr<InputFile> &&thumbnail_, int32 width_, int32 height_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1582387236;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Describes an internal https://t.me or tg: link, which must be processed by the app in a special way.
 */
class InternalLinkType: public Object {
 public:
};

/**
 * The link is a link to the active sessions section of the app. Use getActiveSessions to handle the link.
 */
class internalLinkTypeActiveSessions final : public InternalLinkType {
 public:

  /**
   * The link is a link to the active sessions section of the app. Use getActiveSessions to handle the link.
   */
  internalLinkTypeActiveSessions();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1886108589;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The link contains an authentication code. Call checkAuthenticationCode with the code if the current authorization state is authorizationStateWaitCode.
 */
class internalLinkTypeAuthenticationCode final : public InternalLinkType {
 public:
  /// The authentication code.
  string code_;

  /**
   * The link contains an authentication code. Call checkAuthenticationCode with the code if the current authorization state is authorizationStateWaitCode.
   */
  internalLinkTypeAuthenticationCode();

  /**
   * The link contains an authentication code. Call checkAuthenticationCode with the code if the current authorization state is authorizationStateWaitCode.
   *
   * \param[in] code_ The authentication code.
   */
  explicit internalLinkTypeAuthenticationCode(string const &code_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -209235982;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The link is a link to a background. Call searchBackground with the given background name to process the link.
 */
class internalLinkTypeBackground final : public InternalLinkType {
 public:
  /// Name of the background.
  string background_name_;

  /**
   * The link is a link to a background. Call searchBackground with the given background name to process the link.
   */
  internalLinkTypeBackground();

  /**
   * The link is a link to a background. Call searchBackground with the given background name to process the link.
   *
   * \param[in] background_name_ Name of the background.
   */
  explicit internalLinkTypeBackground(string const &background_name_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 185411848;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The link is a link to a chat with a Telegram bot. Call searchPublicChat with the given bot username, check that the user is a bot, show START button in the chat with the bot, and then call sendBotStartMessage with the given start parameter after the button is pressed.
 */
class internalLinkTypeBotStart final : public InternalLinkType {
 public:
  /// Username of the bot.
  string bot_username_;
  /// The parameter to be passed to sendBotStartMessage.
  string start_parameter_;

  /**
   * The link is a link to a chat with a Telegram bot. Call searchPublicChat with the given bot username, check that the user is a bot, show START button in the chat with the bot, and then call sendBotStartMessage with the given start parameter after the button is pressed.
   */
  internalLinkTypeBotStart();

  /**
   * The link is a link to a chat with a Telegram bot. Call searchPublicChat with the given bot username, check that the user is a bot, show START button in the chat with the bot, and then call sendBotStartMessage with the given start parameter after the button is pressed.
   *
   * \param[in] bot_username_ Username of the bot.
   * \param[in] start_parameter_ The parameter to be passed to sendBotStartMessage.
   */
  internalLinkTypeBotStart(string const &bot_username_, string const &start_parameter_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1206724291;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The link is a link to a Telegram bot, which is supposed to be added to a group chat. Call searchPublicChat with the given bot username, check that the user is a bot and can be added to groups, ask the current user to select a group to add the bot to, and then call sendBotStartMessage with the given start parameter and the chosen group chat. Bots can be added to a public group only by administrators of the group.
 */
class internalLinkTypeBotStartInGroup final : public InternalLinkType {
 public:
  /// Username of the bot.
  string bot_username_;
  /// The parameter to be passed to sendBotStartMessage.
  string start_parameter_;

  /**
   * The link is a link to a Telegram bot, which is supposed to be added to a group chat. Call searchPublicChat with the given bot username, check that the user is a bot and can be added to groups, ask the current user to select a group to add the bot to, and then call sendBotStartMessage with the given start parameter and the chosen group chat. Bots can be added to a public group only by administrators of the group.
   */
  internalLinkTypeBotStartInGroup();

  /**
   * The link is a link to a Telegram bot, which is supposed to be added to a group chat. Call searchPublicChat with the given bot username, check that the user is a bot and can be added to groups, ask the current user to select a group to add the bot to, and then call sendBotStartMessage with the given start parameter and the chosen group chat. Bots can be added to a public group only by administrators of the group.
   *
   * \param[in] bot_username_ Username of the bot.
   * \param[in] start_parameter_ The parameter to be passed to sendBotStartMessage.
   */
  internalLinkTypeBotStartInGroup(string const &bot_username_, string const &start_parameter_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1040096100;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The link is a link to the change phone number section of the app.
 */
class internalLinkTypeChangePhoneNumber final : public InternalLinkType {
 public:

  /**
   * The link is a link to the change phone number section of the app.
   */
  internalLinkTypeChangePhoneNumber();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -265856255;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The link is a chat invite link. Call checkChatInviteLink with the given invite link to process the link.
 */
class internalLinkTypeChatInvite final : public InternalLinkType {
 public:
  /// Internal representation of the invite link.
  string invite_link_;

  /**
   * The link is a chat invite link. Call checkChatInviteLink with the given invite link to process the link.
   */
  internalLinkTypeChatInvite();

  /**
   * The link is a chat invite link. Call checkChatInviteLink with the given invite link to process the link.
   *
   * \param[in] invite_link_ Internal representation of the invite link.
   */
  explicit internalLinkTypeChatInvite(string const &invite_link_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 428621017;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The link is a link to the filter settings section of the app.
 */
class internalLinkTypeFilterSettings final : public InternalLinkType {
 public:

  /**
   * The link is a link to the filter settings section of the app.
   */
  internalLinkTypeFilterSettings();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1501632411;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The link is a link to a game. Call searchPublicChat with the given bot username, check that the user is a bot, ask the current user to select a chat to send the game, and then call sendMessage with inputMessageGame.
 */
class internalLinkTypeGame final : public InternalLinkType {
 public:
  /// Username of the bot that owns the game.
  string bot_username_;
  /// Short name of the game.
  string game_short_name_;

  /**
   * The link is a link to a game. Call searchPublicChat with the given bot username, check that the user is a bot, ask the current user to select a chat to send the game, and then call sendMessage with inputMessageGame.
   */
  internalLinkTypeGame();

  /**
   * The link is a link to a game. Call searchPublicChat with the given bot username, check that the user is a bot, ask the current user to select a chat to send the game, and then call sendMessage with inputMessageGame.
   *
   * \param[in] bot_username_ Username of the bot that owns the game.
   * \param[in] game_short_name_ Short name of the game.
   */
  internalLinkTypeGame(string const &bot_username_, string const &game_short_name_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -260788787;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The link is a link to a language pack. Call getLanguagePackInfo with the given language pack identifier to process the link.
 */
class internalLinkTypeLanguagePack final : public InternalLinkType {
 public:
  /// Language pack identifier.
  string language_pack_id_;

  /**
   * The link is a link to a language pack. Call getLanguagePackInfo with the given language pack identifier to process the link.
   */
  internalLinkTypeLanguagePack();

  /**
   * The link is a link to a language pack. Call getLanguagePackInfo with the given language pack identifier to process the link.
   *
   * \param[in] language_pack_id_ Language pack identifier.
   */
  explicit internalLinkTypeLanguagePack(string const &language_pack_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1450766996;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The link is a link to a Telegram message. Call getMessageLinkInfo with the given URL to process the link.
 */
class internalLinkTypeMessage final : public InternalLinkType {
 public:
  /// URL to be passed to getMessageLinkInfo.
  string url_;

  /**
   * The link is a link to a Telegram message. Call getMessageLinkInfo with the given URL to process the link.
   */
  internalLinkTypeMessage();

  /**
   * The link is a link to a Telegram message. Call getMessageLinkInfo with the given URL to process the link.
   *
   * \param[in] url_ URL to be passed to getMessageLinkInfo.
   */
  explicit internalLinkTypeMessage(string const &url_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 978541650;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The link contains a message draft text. A share screen needs to be shown to the user, then the chosen chat must be opened and the text is added to the input field.
 */
class internalLinkTypeMessageDraft final : public InternalLinkType {
 public:
  /// Message draft text.
  object_ptr<formattedText> text_;
  /// True, if the first line of the text contains a link. If true, the input field needs to be focused and the text after the link must be selected.
  bool contains_link_;

  /**
   * The link contains a message draft text. A share screen needs to be shown to the user, then the chosen chat must be opened and the text is added to the input field.
   */
  internalLinkTypeMessageDraft();

  /**
   * The link contains a message draft text. A share screen needs to be shown to the user, then the chosen chat must be opened and the text is added to the input field.
   *
   * \param[in] text_ Message draft text.
   * \param[in] contains_link_ True, if the first line of the text contains a link. If true, the input field needs to be focused and the text after the link must be selected.
   */
  internalLinkTypeMessageDraft(object_ptr<formattedText> &&text_, bool contains_link_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 661633749;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The link contains a request of Telegram passport data. Call getPassportAuthorizationForm with the given parameters to process the link if the link was received from outside of the app, otherwise ignore it.
 */
class internalLinkTypePassportDataRequest final : public InternalLinkType {
 public:
  /// User identifier of the service&\#039;s bot.
  int53 bot_user_id_;
  /// Telegram Passport element types requested by the service.
  string scope_;
  /// Service&\#039;s public key.
  string public_key_;
  /// Unique request identifier provided by the service.
  string nonce_;
  /// An HTTP URL to open once the request is finished or canceled with the parameter tg_passport=success or tg_passport=cancel respectively. If empty, then the link tgbot{bot_user_id}://passport/success or tgbot{bot_user_id}://passport/cancel needs to be opened instead.
  string callback_url_;

  /**
   * The link contains a request of Telegram passport data. Call getPassportAuthorizationForm with the given parameters to process the link if the link was received from outside of the app, otherwise ignore it.
   */
  internalLinkTypePassportDataRequest();

  /**
   * The link contains a request of Telegram passport data. Call getPassportAuthorizationForm with the given parameters to process the link if the link was received from outside of the app, otherwise ignore it.
   *
   * \param[in] bot_user_id_ User identifier of the service&\#039;s bot.
   * \param[in] scope_ Telegram Passport element types requested by the service.
   * \param[in] public_key_ Service&\#039;s public key.
   * \param[in] nonce_ Unique request identifier provided by the service.
   * \param[in] callback_url_ An HTTP URL to open once the request is finished or canceled with the parameter tg_passport=success or tg_passport=cancel respectively. If empty, then the link tgbot{bot_user_id}://passport/success or tgbot{bot_user_id}://passport/cancel needs to be opened instead.
   */
  internalLinkTypePassportDataRequest(int53 bot_user_id_, string const &scope_, string const &public_key_, string const &nonce_, string const &callback_url_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -988819839;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The link can be used to confirm ownership of a phone number to prevent account deletion. Call sendPhoneNumberConfirmationCode with the given hash and phone number to process the link.
 */
class internalLinkTypePhoneNumberConfirmation final : public InternalLinkType {
 public:
  /// Hash value from the link.
  string hash_;
  /// Phone number value from the link.
  string phone_number_;

  /**
   * The link can be used to confirm ownership of a phone number to prevent account deletion. Call sendPhoneNumberConfirmationCode with the given hash and phone number to process the link.
   */
  internalLinkTypePhoneNumberConfirmation();

  /**
   * The link can be used to confirm ownership of a phone number to prevent account deletion. Call sendPhoneNumberConfirmationCode with the given hash and phone number to process the link.
   *
   * \param[in] hash_ Hash value from the link.
   * \param[in] phone_number_ Phone number value from the link.
   */
  internalLinkTypePhoneNumberConfirmation(string const &hash_, string const &phone_number_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1757375254;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The link is a link to a proxy. Call addProxy with the given parameters to process the link and add the proxy.
 */
class internalLinkTypeProxy final : public InternalLinkType {
 public:
  /// Proxy server IP address.
  string server_;
  /// Proxy server port.
  int32 port_;
  /// Type of the proxy.
  object_ptr<ProxyType> type_;

  /**
   * The link is a link to a proxy. Call addProxy with the given parameters to process the link and add the proxy.
   */
  internalLinkTypeProxy();

  /**
   * The link is a link to a proxy. Call addProxy with the given parameters to process the link and add the proxy.
   *
   * \param[in] server_ Proxy server IP address.
   * \param[in] port_ Proxy server port.
   * \param[in] type_ Type of the proxy.
   */
  internalLinkTypeProxy(string const &server_, int32 port_, object_ptr<ProxyType> &&type_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1313788694;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The link is a link to a chat by its username. Call searchPublicChat with the given chat username to process the link.
 */
class internalLinkTypePublicChat final : public InternalLinkType {
 public:
  /// Username of the chat.
  string chat_username_;

  /**
   * The link is a link to a chat by its username. Call searchPublicChat with the given chat username to process the link.
   */
  internalLinkTypePublicChat();

  /**
   * The link is a link to a chat by its username. Call searchPublicChat with the given chat username to process the link.
   *
   * \param[in] chat_username_ Username of the chat.
   */
  explicit internalLinkTypePublicChat(string const &chat_username_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1485547717;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The link can be used to login the current user on another device, but it must be scanned from QR-code using in-app camera. An alert similar to &quot;This code can be used to allow someone to log in to your Telegram account. To confirm Telegram login, please go to Settings &gt; Devices &gt; Scan QR and scan the code&quot; needs to be shown.
 */
class internalLinkTypeQrCodeAuthentication final : public InternalLinkType {
 public:

  /**
   * The link can be used to login the current user on another device, but it must be scanned from QR-code using in-app camera. An alert similar to &quot;This code can be used to allow someone to log in to your Telegram account. To confirm Telegram login, please go to Settings &gt; Devices &gt; Scan QR and scan the code&quot; needs to be shown.
   */
  internalLinkTypeQrCodeAuthentication();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1089332956;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The link is a link to app settings.
 */
class internalLinkTypeSettings final : public InternalLinkType {
 public:

  /**
   * The link is a link to app settings.
   */
  internalLinkTypeSettings();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 393561524;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The link is a link to a sticker set. Call searchStickerSet with the given sticker set name to process the link and show the sticker set.
 */
class internalLinkTypeStickerSet final : public InternalLinkType {
 public:
  /// Name of the sticker set.
  string sticker_set_name_;

  /**
   * The link is a link to a sticker set. Call searchStickerSet with the given sticker set name to process the link and show the sticker set.
   */
  internalLinkTypeStickerSet();

  /**
   * The link is a link to a sticker set. Call searchStickerSet with the given sticker set name to process the link and show the sticker set.
   *
   * \param[in] sticker_set_name_ Name of the sticker set.
   */
  explicit internalLinkTypeStickerSet(string const &sticker_set_name_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -529639751;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The link is a link to a theme. TDLib has no theme support yet.
 */
class internalLinkTypeTheme final : public InternalLinkType {
 public:
  /// Name of the theme.
  string theme_name_;

  /**
   * The link is a link to a theme. TDLib has no theme support yet.
   */
  internalLinkTypeTheme();

  /**
   * The link is a link to a theme. TDLib has no theme support yet.
   *
   * \param[in] theme_name_ Name of the theme.
   */
  explicit internalLinkTypeTheme(string const &theme_name_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -200935417;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The link is a link to the theme settings section of the app.
 */
class internalLinkTypeThemeSettings final : public InternalLinkType {
 public:

  /**
   * The link is a link to the theme settings section of the app.
   */
  internalLinkTypeThemeSettings();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1051903722;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The link is an unknown tg: link. Call getDeepLinkInfo to process the link.
 */
class internalLinkTypeUnknownDeepLink final : public InternalLinkType {
 public:
  /// Link to be passed to getDeepLinkInfo.
  string link_;

  /**
   * The link is an unknown tg: link. Call getDeepLinkInfo to process the link.
   */
  internalLinkTypeUnknownDeepLink();

  /**
   * The link is an unknown tg: link. Call getDeepLinkInfo to process the link.
   *
   * \param[in] link_ Link to be passed to getDeepLinkInfo.
   */
  explicit internalLinkTypeUnknownDeepLink(string const &link_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 625596379;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The link is a link to an unsupported proxy. An alert can be shown to the user.
 */
class internalLinkTypeUnsupportedProxy final : public InternalLinkType {
 public:

  /**
   * The link is a link to an unsupported proxy. An alert can be shown to the user.
   */
  internalLinkTypeUnsupportedProxy();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -566649079;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The link is a link to a video chat. Call searchPublicChat with the given chat username, and then joinGoupCall with the given invite hash to process the link.
 */
class internalLinkTypeVideoChat final : public InternalLinkType {
 public:
  /// Username of the chat with the video chat.
  string chat_username_;
  /// If non-empty, invite hash to be used to join the video chat without being muted by administrators.
  string invite_hash_;
  /// True, if the video chat is expected to be a live stream in a channel or a broadcast group.
  bool is_live_stream_;

  /**
   * The link is a link to a video chat. Call searchPublicChat with the given chat username, and then joinGoupCall with the given invite hash to process the link.
   */
  internalLinkTypeVideoChat();

  /**
   * The link is a link to a video chat. Call searchPublicChat with the given chat username, and then joinGoupCall with the given invite hash to process the link.
   *
   * \param[in] chat_username_ Username of the chat with the video chat.
   * \param[in] invite_hash_ If non-empty, invite hash to be used to join the video chat without being muted by administrators.
   * \param[in] is_live_stream_ True, if the video chat is expected to be a live stream in a channel or a broadcast group.
   */
  internalLinkTypeVideoChat(string const &chat_username_, string const &invite_hash_, bool is_live_stream_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -2020149068;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Product invoice.
 */
class invoice final : public Object {
 public:
  /// ISO 4217 currency code.
  string currency_;
  /// A list of objects used to calculate the total price of the product.
  array<object_ptr<labeledPricePart>> price_parts_;
  /// The maximum allowed amount of tip in the smallest units of the currency.
  int53 max_tip_amount_;
  /// Suggested amounts of tip in the smallest units of the currency.
  array<int53> suggested_tip_amounts_;
  /// True, if the payment is a test payment.
  bool is_test_;
  /// True, if the user&\#039;s name is needed for payment.
  bool need_name_;
  /// True, if the user&\#039;s phone number is needed for payment.
  bool need_phone_number_;
  /// True, if the user&\#039;s email address is needed for payment.
  bool need_email_address_;
  /// True, if the user&\#039;s shipping address is needed for payment.
  bool need_shipping_address_;
  /// True, if the user&\#039;s phone number will be sent to the provider.
  bool send_phone_number_to_provider_;
  /// True, if the user&\#039;s email address will be sent to the provider.
  bool send_email_address_to_provider_;
  /// True, if the total price depends on the shipping method.
  bool is_flexible_;

  /**
   * Product invoice.
   */
  invoice();

  /**
   * Product invoice.
   *
   * \param[in] currency_ ISO 4217 currency code.
   * \param[in] price_parts_ A list of objects used to calculate the total price of the product.
   * \param[in] max_tip_amount_ The maximum allowed amount of tip in the smallest units of the currency.
   * \param[in] suggested_tip_amounts_ Suggested amounts of tip in the smallest units of the currency.
   * \param[in] is_test_ True, if the payment is a test payment.
   * \param[in] need_name_ True, if the user&\#039;s name is needed for payment.
   * \param[in] need_phone_number_ True, if the user&\#039;s phone number is needed for payment.
   * \param[in] need_email_address_ True, if the user&\#039;s email address is needed for payment.
   * \param[in] need_shipping_address_ True, if the user&\#039;s shipping address is needed for payment.
   * \param[in] send_phone_number_to_provider_ True, if the user&\#039;s phone number will be sent to the provider.
   * \param[in] send_email_address_to_provider_ True, if the user&\#039;s email address will be sent to the provider.
   * \param[in] is_flexible_ True, if the total price depends on the shipping method.
   */
  invoice(string const &currency_, array<object_ptr<labeledPricePart>> &&price_parts_, int53 max_tip_amount_, array<int53> &&suggested_tip_amounts_, bool is_test_, bool need_name_, bool need_phone_number_, bool need_email_address_, bool need_shipping_address_, bool send_phone_number_to_provider_, bool send_email_address_to_provider_, bool is_flexible_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1479250728;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Represents one member of a JSON object.
 */
class jsonObjectMember final : public Object {
 public:
  /// Member&\#039;s key.
  string key_;
  /// Member&\#039;s value.
  object_ptr<JsonValue> value_;

  /**
   * Represents one member of a JSON object.
   */
  jsonObjectMember();

  /**
   * Represents one member of a JSON object.
   *
   * \param[in] key_ Member&\#039;s key.
   * \param[in] value_ Member&\#039;s value.
   */
  jsonObjectMember(string const &key_, object_ptr<JsonValue> &&value_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1803309418;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Represents a JSON value.
 */
class JsonValue: public Object {
 public:
};

/**
 * Represents a null JSON value.
 */
class jsonValueNull final : public JsonValue {
 public:

  /**
   * Represents a null JSON value.
   */
  jsonValueNull();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -92872499;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Represents a boolean JSON value.
 */
class jsonValueBoolean final : public JsonValue {
 public:
  /// The value.
  bool value_;

  /**
   * Represents a boolean JSON value.
   */
  jsonValueBoolean();

  /**
   * Represents a boolean JSON value.
   *
   * \param[in] value_ The value.
   */
  explicit jsonValueBoolean(bool value_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -2142186576;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Represents a numeric JSON value.
 */
class jsonValueNumber final : public JsonValue {
 public:
  /// The value.
  double value_;

  /**
   * Represents a numeric JSON value.
   */
  jsonValueNumber();

  /**
   * Represents a numeric JSON value.
   *
   * \param[in] value_ The value.
   */
  explicit jsonValueNumber(double value_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1010822033;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Represents a string JSON value.
 */
class jsonValueString final : public JsonValue {
 public:
  /// The value.
  string value_;

  /**
   * Represents a string JSON value.
   */
  jsonValueString();

  /**
   * Represents a string JSON value.
   *
   * \param[in] value_ The value.
   */
  explicit jsonValueString(string const &value_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1597947313;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Represents a JSON array.
 */
class jsonValueArray final : public JsonValue {
 public:
  /// The list of array elements.
  array<object_ptr<JsonValue>> values_;

  /**
   * Represents a JSON array.
   */
  jsonValueArray();

  /**
   * Represents a JSON array.
   *
   * \param[in] values_ The list of array elements.
   */
  explicit jsonValueArray(array<object_ptr<JsonValue>> &&values_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -183913546;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Represents a JSON object.
 */
class jsonValueObject final : public JsonValue {
 public:
  /// The list of object members.
  array<object_ptr<jsonObjectMember>> members_;

  /**
   * Represents a JSON object.
   */
  jsonValueObject();

  /**
   * Represents a JSON object.
   *
   * \param[in] members_ The list of object members.
   */
  explicit jsonValueObject(array<object_ptr<jsonObjectMember>> &&members_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 520252026;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Represents a single button in a bot keyboard.
 */
class keyboardButton final : public Object {
 public:
  /// Text of the button.
  string text_;
  /// Type of the button.
  object_ptr<KeyboardButtonType> type_;

  /**
   * Represents a single button in a bot keyboard.
   */
  keyboardButton();

  /**
   * Represents a single button in a bot keyboard.
   *
   * \param[in] text_ Text of the button.
   * \param[in] type_ Type of the button.
   */
  keyboardButton(string const &text_, object_ptr<KeyboardButtonType> &&type_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -2069836172;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Describes a keyboard button type.
 */
class KeyboardButtonType: public Object {
 public:
};

/**
 * A simple button, with text that must be sent when the button is pressed.
 */
class keyboardButtonTypeText final : public KeyboardButtonType {
 public:

  /**
   * A simple button, with text that must be sent when the button is pressed.
   */
  keyboardButtonTypeText();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1773037256;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A button that sends the user&\#039;s phone number when pressed; available only in private chats.
 */
class keyboardButtonTypeRequestPhoneNumber final : public KeyboardButtonType {
 public:

  /**
   * A button that sends the user&\#039;s phone number when pressed; available only in private chats.
   */
  keyboardButtonTypeRequestPhoneNumber();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1529235527;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A button that sends the user&\#039;s location when pressed; available only in private chats.
 */
class keyboardButtonTypeRequestLocation final : public KeyboardButtonType {
 public:

  /**
   * A button that sends the user&\#039;s location when pressed; available only in private chats.
   */
  keyboardButtonTypeRequestLocation();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -125661955;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A button that allows the user to create and send a poll when pressed; available only in private chats.
 */
class keyboardButtonTypeRequestPoll final : public KeyboardButtonType {
 public:
  /// If true, only regular polls must be allowed to create.
  bool force_regular_;
  /// If true, only polls in quiz mode must be allowed to create.
  bool force_quiz_;

  /**
   * A button that allows the user to create and send a poll when pressed; available only in private chats.
   */
  keyboardButtonTypeRequestPoll();

  /**
   * A button that allows the user to create and send a poll when pressed; available only in private chats.
   *
   * \param[in] force_regular_ If true, only regular polls must be allowed to create.
   * \param[in] force_quiz_ If true, only polls in quiz mode must be allowed to create.
   */
  keyboardButtonTypeRequestPoll(bool force_regular_, bool force_quiz_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1902435512;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Portion of the price of a product (e.g., &quot;delivery cost&quot;, &quot;tax amount&quot;).
 */
class labeledPricePart final : public Object {
 public:
  /// Label for this portion of the product price.
  string label_;
  /// Currency amount in the smallest units of the currency.
  int53 amount_;

  /**
   * Portion of the price of a product (e.g., &quot;delivery cost&quot;, &quot;tax amount&quot;).
   */
  labeledPricePart();

  /**
   * Portion of the price of a product (e.g., &quot;delivery cost&quot;, &quot;tax amount&quot;).
   *
   * \param[in] label_ Label for this portion of the product price.
   * \param[in] amount_ Currency amount in the smallest units of the currency.
   */
  labeledPricePart(string const &label_, int53 amount_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 552789798;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains information about a language pack.
 */
class languagePackInfo final : public Object {
 public:
  /// Unique language pack identifier.
  string id_;
  /// Identifier of a base language pack; may be empty. If a string is missed in the language pack, then it must be fetched from base language pack. Unsupported in custom language packs.
  string base_language_pack_id_;
  /// Language name.
  string name_;
  /// Name of the language in that language.
  string native_name_;
  /// A language code to be used to apply plural forms. See https://www.unicode.org/cldr/charts/latest/supplemental/language_plural_rules.html for more info.
  string plural_code_;
  /// True, if the language pack is official.
  bool is_official_;
  /// True, if the language pack strings are RTL.
  bool is_rtl_;
  /// True, if the language pack is a beta language pack.
  bool is_beta_;
  /// True, if the language pack is installed by the current user.
  bool is_installed_;
  /// Total number of non-deleted strings from the language pack.
  int32 total_string_count_;
  /// Total number of translated strings from the language pack.
  int32 translated_string_count_;
  /// Total number of non-deleted strings from the language pack available locally.
  int32 local_string_count_;
  /// Link to language translation interface; empty for custom local language packs.
  string translation_url_;

  /**
   * Contains information about a language pack.
   */
  languagePackInfo();

  /**
   * Contains information about a language pack.
   *
   * \param[in] id_ Unique language pack identifier.
   * \param[in] base_language_pack_id_ Identifier of a base language pack; may be empty. If a string is missed in the language pack, then it must be fetched from base language pack. Unsupported in custom language packs.
   * \param[in] name_ Language name.
   * \param[in] native_name_ Name of the language in that language.
   * \param[in] plural_code_ A language code to be used to apply plural forms. See https://www.unicode.org/cldr/charts/latest/supplemental/language_plural_rules.html for more info.
   * \param[in] is_official_ True, if the language pack is official.
   * \param[in] is_rtl_ True, if the language pack strings are RTL.
   * \param[in] is_beta_ True, if the language pack is a beta language pack.
   * \param[in] is_installed_ True, if the language pack is installed by the current user.
   * \param[in] total_string_count_ Total number of non-deleted strings from the language pack.
   * \param[in] translated_string_count_ Total number of translated strings from the language pack.
   * \param[in] local_string_count_ Total number of non-deleted strings from the language pack available locally.
   * \param[in] translation_url_ Link to language translation interface; empty for custom local language packs.
   */
  languagePackInfo(string const &id_, string const &base_language_pack_id_, string const &name_, string const &native_name_, string const &plural_code_, bool is_official_, bool is_rtl_, bool is_beta_, bool is_installed_, int32 total_string_count_, int32 translated_string_count_, int32 local_string_count_, string const &translation_url_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 542199642;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Represents one language pack string.
 */
class languagePackString final : public Object {
 public:
  /// String key.
  string key_;
  /// String value; pass null if the string needs to be taken from the built-in English language pack.
  object_ptr<LanguagePackStringValue> value_;

  /**
   * Represents one language pack string.
   */
  languagePackString();

  /**
   * Represents one language pack string.
   *
   * \param[in] key_ String key.
   * \param[in] value_ String value; pass null if the string needs to be taken from the built-in English language pack.
   */
  languagePackString(string const &key_, object_ptr<LanguagePackStringValue> &&value_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1307632736;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Represents the value of a string in a language pack.
 */
class LanguagePackStringValue: public Object {
 public:
};

/**
 * An ordinary language pack string.
 */
class languagePackStringValueOrdinary final : public LanguagePackStringValue {
 public:
  /// String value.
  string value_;

  /**
   * An ordinary language pack string.
   */
  languagePackStringValueOrdinary();

  /**
   * An ordinary language pack string.
   *
   * \param[in] value_ String value.
   */
  explicit languagePackStringValueOrdinary(string const &value_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -249256352;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A language pack string which has different forms based on the number of some object it mentions. See https://www.unicode.org/cldr/charts/latest/supplemental/language_plural_rules.html for more info.
 */
class languagePackStringValuePluralized final : public LanguagePackStringValue {
 public:
  /// Value for zero objects.
  string zero_value_;
  /// Value for one object.
  string one_value_;
  /// Value for two objects.
  string two_value_;
  /// Value for few objects.
  string few_value_;
  /// Value for many objects.
  string many_value_;
  /// Default value.
  string other_value_;

  /**
   * A language pack string which has different forms based on the number of some object it mentions. See https://www.unicode.org/cldr/charts/latest/supplemental/language_plural_rules.html for more info.
   */
  languagePackStringValuePluralized();

  /**
   * A language pack string which has different forms based on the number of some object it mentions. See https://www.unicode.org/cldr/charts/latest/supplemental/language_plural_rules.html for more info.
   *
   * \param[in] zero_value_ Value for zero objects.
   * \param[in] one_value_ Value for one object.
   * \param[in] two_value_ Value for two objects.
   * \param[in] few_value_ Value for few objects.
   * \param[in] many_value_ Value for many objects.
   * \param[in] other_value_ Default value.
   */
  languagePackStringValuePluralized(string const &zero_value_, string const &one_value_, string const &two_value_, string const &few_value_, string const &many_value_, string const &other_value_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1906840261;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A deleted language pack string, the value must be taken from the built-in English language pack.
 */
class languagePackStringValueDeleted final : public LanguagePackStringValue {
 public:

  /**
   * A deleted language pack string, the value must be taken from the built-in English language pack.
   */
  languagePackStringValueDeleted();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1834792698;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains a list of language pack strings.
 */
class languagePackStrings final : public Object {
 public:
  /// A list of language pack strings.
  array<object_ptr<languagePackString>> strings_;

  /**
   * Contains a list of language pack strings.
   */
  languagePackStrings();

  /**
   * Contains a list of language pack strings.
   *
   * \param[in] strings_ A list of language pack strings.
   */
  explicit languagePackStrings(array<object_ptr<languagePackString>> &&strings_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1172082922;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Represents a local file.
 */
class localFile final : public Object {
 public:
  /// Local path to the locally available file part; may be empty.
  string path_;
  /// True, if it is possible to download or generate the file.
  bool can_be_downloaded_;
  /// True, if the file can be deleted.
  bool can_be_deleted_;
  /// True, if the file is currently being downloaded (or a local copy is being generated by some other means).
  bool is_downloading_active_;
  /// True, if the local copy is fully available.
  bool is_downloading_completed_;
  /// Download will be started from this offset. downloaded_prefix_size is calculated from this offset.
  int32 download_offset_;
  /// If is_downloading_completed is false, then only some prefix of the file starting from download_offset is ready to be read. downloaded_prefix_size is the size of that prefix in bytes.
  int32 downloaded_prefix_size_;
  /// Total downloaded file size, in bytes. Can be used only for calculating download progress. The actual file size may be bigger, and some parts of it may contain garbage.
  int32 downloaded_size_;

  /**
   * Represents a local file.
   */
  localFile();

  /**
   * Represents a local file.
   *
   * \param[in] path_ Local path to the locally available file part; may be empty.
   * \param[in] can_be_downloaded_ True, if it is possible to download or generate the file.
   * \param[in] can_be_deleted_ True, if the file can be deleted.
   * \param[in] is_downloading_active_ True, if the file is currently being downloaded (or a local copy is being generated by some other means).
   * \param[in] is_downloading_completed_ True, if the local copy is fully available.
   * \param[in] download_offset_ Download will be started from this offset. downloaded_prefix_size is calculated from this offset.
   * \param[in] downloaded_prefix_size_ If is_downloading_completed is false, then only some prefix of the file starting from download_offset is ready to be read. downloaded_prefix_size is the size of that prefix in bytes.
   * \param[in] downloaded_size_ Total downloaded file size, in bytes. Can be used only for calculating download progress. The actual file size may be bigger, and some parts of it may contain garbage.
   */
  localFile(string const &path_, bool can_be_downloaded_, bool can_be_deleted_, bool is_downloading_active_, bool is_downloading_completed_, int32 download_offset_, int32 downloaded_prefix_size_, int32 downloaded_size_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1166400317;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains information about the current localization target.
 */
class localizationTargetInfo final : public Object {
 public:
  /// List of available language packs for this application.
  array<object_ptr<languagePackInfo>> language_packs_;

  /**
   * Contains information about the current localization target.
   */
  localizationTargetInfo();

  /**
   * Contains information about the current localization target.
   *
   * \param[in] language_packs_ List of available language packs for this application.
   */
  explicit localizationTargetInfo(array<object_ptr<languagePackInfo>> &&language_packs_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -2048670809;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Describes a location on planet Earth.
 */
class location final : public Object {
 public:
  /// Latitude of the location in degrees; as defined by the sender.
  double latitude_;
  /// Longitude of the location, in degrees; as defined by the sender.
  double longitude_;
  /// The estimated horizontal accuracy of the location, in meters; as defined by the sender. 0 if unknown.
  double horizontal_accuracy_;

  /**
   * Describes a location on planet Earth.
   */
  location();

  /**
   * Describes a location on planet Earth.
   *
   * \param[in] latitude_ Latitude of the location in degrees; as defined by the sender.
   * \param[in] longitude_ Longitude of the location, in degrees; as defined by the sender.
   * \param[in] horizontal_accuracy_ The estimated horizontal accuracy of the location, in meters; as defined by the sender. 0 if unknown.
   */
  location(double latitude_, double longitude_, double horizontal_accuracy_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -443392141;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Describes a stream to which TDLib internal log is written.
 */
class LogStream: public Object {
 public:
};

/**
 * The log is written to stderr or an OS specific log.
 */
class logStreamDefault final : public LogStream {
 public:

  /**
   * The log is written to stderr or an OS specific log.
   */
  logStreamDefault();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1390581436;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The log is written to a file.
 */
class logStreamFile final : public LogStream {
 public:
  /// Path to the file to where the internal TDLib log will be written.
  string path_;
  /// The maximum size of the file to where the internal TDLib log is written before the file will automatically be rotated, in bytes.
  int53 max_file_size_;
  /// Pass true to additionally redirect stderr to the log file. Ignored on Windows.
  bool redirect_stderr_;

  /**
   * The log is written to a file.
   */
  logStreamFile();

  /**
   * The log is written to a file.
   *
   * \param[in] path_ Path to the file to where the internal TDLib log will be written.
   * \param[in] max_file_size_ The maximum size of the file to where the internal TDLib log is written before the file will automatically be rotated, in bytes.
   * \param[in] redirect_stderr_ Pass true to additionally redirect stderr to the log file. Ignored on Windows.
   */
  logStreamFile(string const &path_, int53 max_file_size_, bool redirect_stderr_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1532136933;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The log is written nowhere.
 */
class logStreamEmpty final : public LogStream {
 public:

  /**
   * The log is written nowhere.
   */
  logStreamEmpty();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -499912244;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains a list of available TDLib internal log tags.
 */
class logTags final : public Object {
 public:
  /// List of log tags.
  array<string> tags_;

  /**
   * Contains a list of available TDLib internal log tags.
   */
  logTags();

  /**
   * Contains a list of available TDLib internal log tags.
   *
   * \param[in] tags_ List of log tags.
   */
  explicit logTags(array<string> &&tags_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1604930601;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains a TDLib internal log verbosity level.
 */
class logVerbosityLevel final : public Object {
 public:
  /// Log verbosity level.
  int32 verbosity_level_;

  /**
   * Contains a TDLib internal log verbosity level.
   */
  logVerbosityLevel();

  /**
   * Contains a TDLib internal log verbosity level.
   *
   * \param[in] verbosity_level_ Log verbosity level.
   */
  explicit logVerbosityLevel(int32 verbosity_level_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1734624234;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Contains information about an inline button of type inlineKeyboardButtonTypeLoginUrl.
 */
class LoginUrlInfo: public Object {
 public:
};

/**
 * An HTTP url needs to be open.
 */
class loginUrlInfoOpen final : public LoginUrlInfo {
 public:
  /// The URL to open.
  string url_;
  /// True, if there is no need to show an ordinary open URL confirm.
  bool skip_confirm_;

  /**
   * An HTTP url needs to be open.
   */
  loginUrlInfoOpen();

  /**
   * An HTTP url needs to be open.
   *
   * \param[in] url_ The URL to open.
   * \param[in] skip_confirm_ True, if there is no need to show an ordinary open URL confirm.
   */
  loginUrlInfoOpen(string const &url_, bool skip_confirm_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1079045420;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * An authorization confirmation dialog needs to be shown to the user.
 */
class loginUrlInfoRequestConfirmation final : public LoginUrlInfo {
 public:
  /// An HTTP URL to be opened.
  string url_;
  /// A domain of the URL.
  string domain_;
  /// User identifier of a bot linked with the website.
  int53 bot_user_id_;
  /// True, if the user needs to be requested to give the permission to the bot to send them messages.
  bool request_write_access_;

  /**
   * An authorization confirmation dialog needs to be shown to the user.
   */
  loginUrlInfoRequestConfirmation();

  /**
   * An authorization confirmation dialog needs to be shown to the user.
   *
   * \param[in] url_ An HTTP URL to be opened.
   * \param[in] domain_ A domain of the URL.
   * \param[in] bot_user_id_ User identifier of a bot linked with the website.
   * \param[in] request_write_access_ True, if the user needs to be requested to give the permission to the bot to send them messages.
   */
  loginUrlInfoRequestConfirmation(string const &url_, string const &domain_, int53 bot_user_id_, bool request_write_access_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 2128290863;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Part of the face, relative to which a mask is placed.
 */
class MaskPoint: public Object {
 public:
};

/**
 * The mask is placed relatively to the forehead.
 */
class maskPointForehead final : public MaskPoint {
 public:

  /**
   * The mask is placed relatively to the forehead.
   */
  maskPointForehead();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1027512005;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The mask is placed relatively to the eyes.
 */
class maskPointEyes final : public MaskPoint {
 public:

  /**
   * The mask is placed relatively to the eyes.
   */
  maskPointEyes();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1748310861;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The mask is placed relatively to the mouth.
 */
class maskPointMouth final : public MaskPoint {
 public:

  /**
   * The mask is placed relatively to the mouth.
   */
  maskPointMouth();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 411773406;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The mask is placed relatively to the chin.
 */
class maskPointChin final : public MaskPoint {
 public:

  /**
   * The mask is placed relatively to the chin.
   */
  maskPointChin();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 534995335;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Position on a photo where a mask is placed.
 */
class maskPosition final : public Object {
 public:
  /// Part of the face, relative to which the mask is placed.
  object_ptr<MaskPoint> point_;
  /// Shift by X-axis measured in widths of the mask scaled to the face size, from left to right. (For example, -1.0 will place the mask just to the left of the default mask position.)
  double x_shift_;
  /// Shift by Y-axis measured in heights of the mask scaled to the face size, from top to bottom. (For example, 1.0 will place the mask just below the default mask position.)
  double y_shift_;
  /// Mask scaling coefficient. (For example, 2.0 means a doubled size.)
  double scale_;

  /**
   * Position on a photo where a mask is placed.
   */
  maskPosition();

  /**
   * Position on a photo where a mask is placed.
   *
   * \param[in] point_ Part of the face, relative to which the mask is placed.
   * \param[in] x_shift_ Shift by X-axis measured in widths of the mask scaled to the face size, from left to right. (For example, -1.0 will place the mask just to the left of the default mask position.)
   * \param[in] y_shift_ Shift by Y-axis measured in heights of the mask scaled to the face size, from top to bottom. (For example, 1.0 will place the mask just below the default mask position.)
   * \param[in] scale_ Mask scaling coefficient. (For example, 2.0 means a doubled size.)
   */
  maskPosition(object_ptr<MaskPoint> &&point_, double x_shift_, double y_shift_, double scale_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -2097433026;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Describes a message.
 */
class message final : public Object {
 public:
  /// Message identifier; unique for the chat to which the message belongs.
  int53 id_;
  /// Identifier of the sender of the message.
  object_ptr<MessageSender> sender_id_;
  /// Chat identifier.
  int53 chat_id_;
  /// The sending state of the message; may be null.
  object_ptr<MessageSendingState> sending_state_;
  /// The scheduling state of the message; may be null.
  object_ptr<MessageSchedulingState> scheduling_state_;
  /// True, if the message is outgoing.
  bool is_outgoing_;
  /// True, if the message is pinned.
  bool is_pinned_;
  /// True, if the message can be edited. For live location and poll messages this fields shows whether editMessageLiveLocation or stopPoll can be used with this message by the application.
  bool can_be_edited_;
  /// True, if the message can be forwarded.
  bool can_be_forwarded_;
  /// True, if content of the message can be saved locally or copied.
  bool can_be_saved_;
  /// True, if the message can be deleted only for the current user while other users will continue to see it.
  bool can_be_deleted_only_for_self_;
  /// True, if the message can be deleted for all users.
  bool can_be_deleted_for_all_users_;
  /// True, if the message statistics are available.
  bool can_get_statistics_;
  /// True, if the message thread info is available.
  bool can_get_message_thread_;
  /// True, if chat members already viewed the message can be received through getMessageViewers.
  bool can_get_viewers_;
  /// True, if media timestamp links can be generated for media timestamp entities in the message text, caption or web page description.
  bool can_get_media_timestamp_links_;
  /// True, if media timestamp entities refers to a media in this message as opposed to a media in the replied message.
  bool has_timestamped_media_;
  /// True, if the message is a channel post. All messages to channels are channel posts, all other messages are not channel posts.
  bool is_channel_post_;
  /// True, if the message contains an unread mention for the current user.
  bool contains_unread_mention_;
  /// Point in time (Unix timestamp) when the message was sent.
  int32 date_;
  /// Point in time (Unix timestamp) when the message was last edited.
  int32 edit_date_;
  /// Information about the initial message sender; may be null.
  object_ptr<messageForwardInfo> forward_info_;
  /// Information about interactions with the message; may be null.
  object_ptr<messageInteractionInfo> interaction_info_;
  /// If non-zero, the identifier of the chat to which the replied message belongs; Currently, only messages in the Replies chat can have different reply_in_chat_id and chat_id.
  int53 reply_in_chat_id_;
  /// If non-zero, the identifier of the message this message is replying to; can be the identifier of a deleted message.
  int53 reply_to_message_id_;
  /// If non-zero, the identifier of the message thread the message belongs to; unique within the chat to which the message belongs.
  int53 message_thread_id_;
  /// For self-destructing messages, the message&\#039;s TTL (Time To Live), in seconds; 0 if none. TDLib will send updateDeleteMessages or updateMessageContent once the TTL expires.
  int32 ttl_;
  /// Time left before the message expires, in seconds. If the TTL timer isn&\#039;t started yet, equals to the value of the ttl field.
  double ttl_expires_in_;
  /// If non-zero, the user identifier of the bot through which this message was sent.
  int53 via_bot_user_id_;
  /// For channel posts and anonymous group messages, optional author signature.
  string author_signature_;
  /// Unique identifier of an album this message belongs to. Only audios, documents, photos and videos can be grouped together in albums.
  int64 media_album_id_;
  /// If non-empty, contains a human-readable description of the reason why access to this message must be restricted.
  string restriction_reason_;
  /// Content of the message.
  object_ptr<MessageContent> content_;
  /// Reply markup for the message; may be null.
  object_ptr<ReplyMarkup> reply_markup_;

  /**
   * Describes a message.
   */
  message();

  /**
   * Describes a message.
   *
   * \param[in] id_ Message identifier; unique for the chat to which the message belongs.
   * \param[in] sender_id_ Identifier of the sender of the message.
   * \param[in] chat_id_ Chat identifier.
   * \param[in] sending_state_ The sending state of the message; may be null.
   * \param[in] scheduling_state_ The scheduling state of the message; may be null.
   * \param[in] is_outgoing_ True, if the message is outgoing.
   * \param[in] is_pinned_ True, if the message is pinned.
   * \param[in] can_be_edited_ True, if the message can be edited. For live location and poll messages this fields shows whether editMessageLiveLocation or stopPoll can be used with this message by the application.
   * \param[in] can_be_forwarded_ True, if the message can be forwarded.
   * \param[in] can_be_saved_ True, if content of the message can be saved locally or copied.
   * \param[in] can_be_deleted_only_for_self_ True, if the message can be deleted only for the current user while other users will continue to see it.
   * \param[in] can_be_deleted_for_all_users_ True, if the message can be deleted for all users.
   * \param[in] can_get_statistics_ True, if the message statistics are available.
   * \param[in] can_get_message_thread_ True, if the message thread info is available.
   * \param[in] can_get_viewers_ True, if chat members already viewed the message can be received through getMessageViewers.
   * \param[in] can_get_media_timestamp_links_ True, if media timestamp links can be generated for media timestamp entities in the message text, caption or web page description.
   * \param[in] has_timestamped_media_ True, if media timestamp entities refers to a media in this message as opposed to a media in the replied message.
   * \param[in] is_channel_post_ True, if the message is a channel post. All messages to channels are channel posts, all other messages are not channel posts.
   * \param[in] contains_unread_mention_ True, if the message contains an unread mention for the current user.
   * \param[in] date_ Point in time (Unix timestamp) when the message was sent.
   * \param[in] edit_date_ Point in time (Unix timestamp) when the message was last edited.
   * \param[in] forward_info_ Information about the initial message sender; may be null.
   * \param[in] interaction_info_ Information about interactions with the message; may be null.
   * \param[in] reply_in_chat_id_ If non-zero, the identifier of the chat to which the replied message belongs; Currently, only messages in the Replies chat can have different reply_in_chat_id and chat_id.
   * \param[in] reply_to_message_id_ If non-zero, the identifier of the message this message is replying to; can be the identifier of a deleted message.
   * \param[in] message_thread_id_ If non-zero, the identifier of the message thread the message belongs to; unique within the chat to which the message belongs.
   * \param[in] ttl_ For self-destructing messages, the message&\#039;s TTL (Time To Live), in seconds; 0 if none. TDLib will send updateDeleteMessages or updateMessageContent once the TTL expires.
   * \param[in] ttl_expires_in_ Time left before the message expires, in seconds. If the TTL timer isn&\#039;t started yet, equals to the value of the ttl field.
   * \param[in] via_bot_user_id_ If non-zero, the user identifier of the bot through which this message was sent.
   * \param[in] author_signature_ For channel posts and anonymous group messages, optional author signature.
   * \param[in] media_album_id_ Unique identifier of an album this message belongs to. Only audios, documents, photos and videos can be grouped together in albums.
   * \param[in] restriction_reason_ If non-empty, contains a human-readable description of the reason why access to this message must be restricted.
   * \param[in] content_ Content of the message.
   * \param[in] reply_markup_ Reply markup for the message; may be null.
   */
  message(int53 id_, object_ptr<MessageSender> &&sender_id_, int53 chat_id_, object_ptr<MessageSendingState> &&sending_state_, object_ptr<MessageSchedulingState> &&scheduling_state_, bool is_outgoing_, bool is_pinned_, bool can_be_edited_, bool can_be_forwarded_, bool can_be_saved_, bool can_be_deleted_only_for_self_, bool can_be_deleted_for_all_users_, bool can_get_statistics_, bool can_get_message_thread_, bool can_get_viewers_, bool can_get_media_timestamp_links_, bool has_timestamped_media_, bool is_channel_post_, bool contains_unread_mention_, int32 date_, int32 edit_date_, object_ptr<messageForwardInfo> &&forward_info_, object_ptr<messageInteractionInfo> &&interaction_info_, int53 reply_in_chat_id_, int53 reply_to_message_id_, int53 message_thread_id_, int32 ttl_, double ttl_expires_in_, int53 via_bot_user_id_, string const &author_signature_, int64 media_album_id_, string const &restriction_reason_, object_ptr<MessageContent> &&content_, object_ptr<ReplyMarkup> &&reply_markup_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -961280585;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains information about found messages, split by days according to the option &quot;utc_time_offset&quot;.
 */
class messageCalendar final : public Object {
 public:
  /// Total number of found messages.
  int32 total_count_;
  /// Information about messages sent.
  array<object_ptr<messageCalendarDay>> days_;

  /**
   * Contains information about found messages, split by days according to the option &quot;utc_time_offset&quot;.
   */
  messageCalendar();

  /**
   * Contains information about found messages, split by days according to the option &quot;utc_time_offset&quot;.
   *
   * \param[in] total_count_ Total number of found messages.
   * \param[in] days_ Information about messages sent.
   */
  messageCalendar(int32 total_count_, array<object_ptr<messageCalendarDay>> &&days_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1682890519;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains information about found messages sent on a specific day.
 */
class messageCalendarDay final : public Object {
 public:
  /// Total number of found messages sent on the day.
  int32 total_count_;
  /// First message sent on the day.
  object_ptr<message> message_;

  /**
   * Contains information about found messages sent on a specific day.
   */
  messageCalendarDay();

  /**
   * Contains information about found messages sent on a specific day.
   *
   * \param[in] total_count_ Total number of found messages sent on the day.
   * \param[in] message_ First message sent on the day.
   */
  messageCalendarDay(int32 total_count_, object_ptr<message> &&message_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -376467614;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Contains the content of a message.
 */
class MessageContent: public Object {
 public:
};

/**
 * A text message.
 */
class messageText final : public MessageContent {
 public:
  /// Text of the message.
  object_ptr<formattedText> text_;
  /// A preview of the web page that&\#039;s mentioned in the text; may be null.
  object_ptr<webPage> web_page_;

  /**
   * A text message.
   */
  messageText();

  /**
   * A text message.
   *
   * \param[in] text_ Text of the message.
   * \param[in] web_page_ A preview of the web page that&\#039;s mentioned in the text; may be null.
   */
  messageText(object_ptr<formattedText> &&text_, object_ptr<webPage> &&web_page_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1989037971;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * An animation message (GIF-style).
 */
class messageAnimation final : public MessageContent {
 public:
  /// The animation description.
  object_ptr<animation> animation_;
  /// Animation caption.
  object_ptr<formattedText> caption_;
  /// True, if the animation thumbnail must be blurred and the animation must be shown only while tapped.
  bool is_secret_;

  /**
   * An animation message (GIF-style).
   */
  messageAnimation();

  /**
   * An animation message (GIF-style).
   *
   * \param[in] animation_ The animation description.
   * \param[in] caption_ Animation caption.
   * \param[in] is_secret_ True, if the animation thumbnail must be blurred and the animation must be shown only while tapped.
   */
  messageAnimation(object_ptr<animation> &&animation_, object_ptr<formattedText> &&caption_, bool is_secret_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1306939396;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * An audio message.
 */
class messageAudio final : public MessageContent {
 public:
  /// The audio description.
  object_ptr<audio> audio_;
  /// Audio caption.
  object_ptr<formattedText> caption_;

  /**
   * An audio message.
   */
  messageAudio();

  /**
   * An audio message.
   *
   * \param[in] audio_ The audio description.
   * \param[in] caption_ Audio caption.
   */
  messageAudio(object_ptr<audio> &&audio_, object_ptr<formattedText> &&caption_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 276722716;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A document message (general file).
 */
class messageDocument final : public MessageContent {
 public:
  /// The document description.
  object_ptr<document> document_;
  /// Document caption.
  object_ptr<formattedText> caption_;

  /**
   * A document message (general file).
   */
  messageDocument();

  /**
   * A document message (general file).
   *
   * \param[in] document_ The document description.
   * \param[in] caption_ Document caption.
   */
  messageDocument(object_ptr<document> &&document_, object_ptr<formattedText> &&caption_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 596945783;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A photo message.
 */
class messagePhoto final : public MessageContent {
 public:
  /// The photo description.
  object_ptr<photo> photo_;
  /// Photo caption.
  object_ptr<formattedText> caption_;
  /// True, if the photo must be blurred and must be shown only while tapped.
  bool is_secret_;

  /**
   * A photo message.
   */
  messagePhoto();

  /**
   * A photo message.
   *
   * \param[in] photo_ The photo description.
   * \param[in] caption_ Photo caption.
   * \param[in] is_secret_ True, if the photo must be blurred and must be shown only while tapped.
   */
  messagePhoto(object_ptr<photo> &&photo_, object_ptr<formattedText> &&caption_, bool is_secret_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1851395174;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * An expired photo message (self-destructed after TTL has elapsed).
 */
class messageExpiredPhoto final : public MessageContent {
 public:

  /**
   * An expired photo message (self-destructed after TTL has elapsed).
   */
  messageExpiredPhoto();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1404641801;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A sticker message.
 */
class messageSticker final : public MessageContent {
 public:
  /// The sticker description.
  object_ptr<sticker> sticker_;

  /**
   * A sticker message.
   */
  messageSticker();

  /**
   * A sticker message.
   *
   * \param[in] sticker_ The sticker description.
   */
  explicit messageSticker(object_ptr<sticker> &&sticker_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1779022878;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A video message.
 */
class messageVideo final : public MessageContent {
 public:
  /// The video description.
  object_ptr<video> video_;
  /// Video caption.
  object_ptr<formattedText> caption_;
  /// True, if the video thumbnail must be blurred and the video must be shown only while tapped.
  bool is_secret_;

  /**
   * A video message.
   */
  messageVideo();

  /**
   * A video message.
   *
   * \param[in] video_ The video description.
   * \param[in] caption_ Video caption.
   * \param[in] is_secret_ True, if the video thumbnail must be blurred and the video must be shown only while tapped.
   */
  messageVideo(object_ptr<video> &&video_, object_ptr<formattedText> &&caption_, bool is_secret_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 2021281344;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * An expired video message (self-destructed after TTL has elapsed).
 */
class messageExpiredVideo final : public MessageContent {
 public:

  /**
   * An expired video message (self-destructed after TTL has elapsed).
   */
  messageExpiredVideo();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1212209981;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A video note message.
 */
class messageVideoNote final : public MessageContent {
 public:
  /// The video note description.
  object_ptr<videoNote> video_note_;
  /// True, if at least one of the recipients has viewed the video note.
  bool is_viewed_;
  /// True, if the video note thumbnail must be blurred and the video note must be shown only while tapped.
  bool is_secret_;

  /**
   * A video note message.
   */
  messageVideoNote();

  /**
   * A video note message.
   *
   * \param[in] video_note_ The video note description.
   * \param[in] is_viewed_ True, if at least one of the recipients has viewed the video note.
   * \param[in] is_secret_ True, if the video note thumbnail must be blurred and the video note must be shown only while tapped.
   */
  messageVideoNote(object_ptr<videoNote> &&video_note_, bool is_viewed_, bool is_secret_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 963323014;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A voice note message.
 */
class messageVoiceNote final : public MessageContent {
 public:
  /// The voice note description.
  object_ptr<voiceNote> voice_note_;
  /// Voice note caption.
  object_ptr<formattedText> caption_;
  /// True, if at least one of the recipients has listened to the voice note.
  bool is_listened_;

  /**
   * A voice note message.
   */
  messageVoiceNote();

  /**
   * A voice note message.
   *
   * \param[in] voice_note_ The voice note description.
   * \param[in] caption_ Voice note caption.
   * \param[in] is_listened_ True, if at least one of the recipients has listened to the voice note.
   */
  messageVoiceNote(object_ptr<voiceNote> &&voice_note_, object_ptr<formattedText> &&caption_, bool is_listened_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 527777781;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A message with a location.
 */
class messageLocation final : public MessageContent {
 public:
  /// The location description.
  object_ptr<location> location_;
  /// Time relative to the message send date, for which the location can be updated, in seconds.
  int32 live_period_;
  /// Left time for which the location can be updated, in seconds. updateMessageContent is not sent when this field changes.
  int32 expires_in_;
  /// For live locations, a direction in which the location moves, in degrees; 1-360. If 0 the direction is unknown.
  int32 heading_;
  /// For live locations, a maximum distance to another chat member for proximity alerts, in meters (0-100000). 0 if the notification is disabled. Available only for the message sender.
  int32 proximity_alert_radius_;

  /**
   * A message with a location.
   */
  messageLocation();

  /**
   * A message with a location.
   *
   * \param[in] location_ The location description.
   * \param[in] live_period_ Time relative to the message send date, for which the location can be updated, in seconds.
   * \param[in] expires_in_ Left time for which the location can be updated, in seconds. updateMessageContent is not sent when this field changes.
   * \param[in] heading_ For live locations, a direction in which the location moves, in degrees; 1-360. If 0 the direction is unknown.
   * \param[in] proximity_alert_radius_ For live locations, a maximum distance to another chat member for proximity alerts, in meters (0-100000). 0 if the notification is disabled. Available only for the message sender.
   */
  messageLocation(object_ptr<location> &&location_, int32 live_period_, int32 expires_in_, int32 heading_, int32 proximity_alert_radius_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 303973492;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A message with information about a venue.
 */
class messageVenue final : public MessageContent {
 public:
  /// The venue description.
  object_ptr<venue> venue_;

  /**
   * A message with information about a venue.
   */
  messageVenue();

  /**
   * A message with information about a venue.
   *
   * \param[in] venue_ The venue description.
   */
  explicit messageVenue(object_ptr<venue> &&venue_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -2146492043;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A message with a user contact.
 */
class messageContact final : public MessageContent {
 public:
  /// The contact description.
  object_ptr<contact> contact_;

  /**
   * A message with a user contact.
   */
  messageContact();

  /**
   * A message with a user contact.
   *
   * \param[in] contact_ The contact description.
   */
  explicit messageContact(object_ptr<contact> &&contact_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -512684966;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A message with an animated emoji.
 */
class messageAnimatedEmoji final : public MessageContent {
 public:
  /// The animated emoji.
  object_ptr<animatedEmoji> animated_emoji_;
  /// The corresponding emoji.
  string emoji_;

  /**
   * A message with an animated emoji.
   */
  messageAnimatedEmoji();

  /**
   * A message with an animated emoji.
   *
   * \param[in] animated_emoji_ The animated emoji.
   * \param[in] emoji_ The corresponding emoji.
   */
  messageAnimatedEmoji(object_ptr<animatedEmoji> &&animated_emoji_, string const &emoji_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 908195298;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A dice message. The dice value is randomly generated by the server.
 */
class messageDice final : public MessageContent {
 public:
  /// The animated stickers with the initial dice animation; may be null if unknown. updateMessageContent will be sent when the sticker became known.
  object_ptr<DiceStickers> initial_state_;
  /// The animated stickers with the final dice animation; may be null if unknown. updateMessageContent will be sent when the sticker became known.
  object_ptr<DiceStickers> final_state_;
  /// Emoji on which the dice throw animation is based.
  string emoji_;
  /// The dice value. If the value is 0, the dice don&\#039;t have final state yet.
  int32 value_;
  /// Number of frame after which a success animation like a shower of confetti needs to be shown on updateMessageSendSucceeded.
  int32 success_animation_frame_number_;

  /**
   * A dice message. The dice value is randomly generated by the server.
   */
  messageDice();

  /**
   * A dice message. The dice value is randomly generated by the server.
   *
   * \param[in] initial_state_ The animated stickers with the initial dice animation; may be null if unknown. updateMessageContent will be sent when the sticker became known.
   * \param[in] final_state_ The animated stickers with the final dice animation; may be null if unknown. updateMessageContent will be sent when the sticker became known.
   * \param[in] emoji_ Emoji on which the dice throw animation is based.
   * \param[in] value_ The dice value. If the value is 0, the dice don&\#039;t have final state yet.
   * \param[in] success_animation_frame_number_ Number of frame after which a success animation like a shower of confetti needs to be shown on updateMessageSendSucceeded.
   */
  messageDice(object_ptr<DiceStickers> &&initial_state_, object_ptr<DiceStickers> &&final_state_, string const &emoji_, int32 value_, int32 success_animation_frame_number_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1115779641;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A message with a game.
 */
class messageGame final : public MessageContent {
 public:
  /// The game description.
  object_ptr<game> game_;

  /**
   * A message with a game.
   */
  messageGame();

  /**
   * A message with a game.
   *
   * \param[in] game_ The game description.
   */
  explicit messageGame(object_ptr<game> &&game_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -69441162;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A message with a poll.
 */
class messagePoll final : public MessageContent {
 public:
  /// The poll description.
  object_ptr<poll> poll_;

  /**
   * A message with a poll.
   */
  messagePoll();

  /**
   * A message with a poll.
   *
   * \param[in] poll_ The poll description.
   */
  explicit messagePoll(object_ptr<poll> &&poll_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -662130099;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A message with an invoice from a bot.
 */
class messageInvoice final : public MessageContent {
 public:
  /// Product title.
  string title_;
  /// Product description.
  string description_;
  /// Product photo; may be null.
  object_ptr<photo> photo_;
  /// Currency for the product price.
  string currency_;
  /// Product total price in the smallest units of the currency.
  int53 total_amount_;
  /// Unique invoice bot start_parameter. To share an invoice use the URL https://t.me/{bot_username}?start={start_parameter}.
  string start_parameter_;
  /// True, if the invoice is a test invoice.
  bool is_test_;
  /// True, if the shipping address must be specified.
  bool need_shipping_address_;
  /// The identifier of the message with the receipt, after the product has been purchased.
  int53 receipt_message_id_;

  /**
   * A message with an invoice from a bot.
   */
  messageInvoice();

  /**
   * A message with an invoice from a bot.
   *
   * \param[in] title_ Product title.
   * \param[in] description_ Product description.
   * \param[in] photo_ Product photo; may be null.
   * \param[in] currency_ Currency for the product price.
   * \param[in] total_amount_ Product total price in the smallest units of the currency.
   * \param[in] start_parameter_ Unique invoice bot start_parameter. To share an invoice use the URL https://t.me/{bot_username}?start={start_parameter}.
   * \param[in] is_test_ True, if the invoice is a test invoice.
   * \param[in] need_shipping_address_ True, if the shipping address must be specified.
   * \param[in] receipt_message_id_ The identifier of the message with the receipt, after the product has been purchased.
   */
  messageInvoice(string const &title_, string const &description_, object_ptr<photo> &&photo_, string const &currency_, int53 total_amount_, string const &start_parameter_, bool is_test_, bool need_shipping_address_, int53 receipt_message_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1916671476;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A message with information about an ended call.
 */
class messageCall final : public MessageContent {
 public:
  /// True, if the call was a video call.
  bool is_video_;
  /// Reason why the call was discarded.
  object_ptr<CallDiscardReason> discard_reason_;
  /// Call duration, in seconds.
  int32 duration_;

  /**
   * A message with information about an ended call.
   */
  messageCall();

  /**
   * A message with information about an ended call.
   *
   * \param[in] is_video_ True, if the call was a video call.
   * \param[in] discard_reason_ Reason why the call was discarded.
   * \param[in] duration_ Call duration, in seconds.
   */
  messageCall(bool is_video_, object_ptr<CallDiscardReason> &&discard_reason_, int32 duration_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 538893824;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A new video chat was scheduled.
 */
class messageVideoChatScheduled final : public MessageContent {
 public:
  /// Identifier of the video chat. The video chat can be received through the method getGroupCall.
  int32 group_call_id_;
  /// Point in time (Unix timestamp) when the group call is supposed to be started by an administrator.
  int32 start_date_;

  /**
   * A new video chat was scheduled.
   */
  messageVideoChatScheduled();

  /**
   * A new video chat was scheduled.
   *
   * \param[in] group_call_id_ Identifier of the video chat. The video chat can be received through the method getGroupCall.
   * \param[in] start_date_ Point in time (Unix timestamp) when the group call is supposed to be started by an administrator.
   */
  messageVideoChatScheduled(int32 group_call_id_, int32 start_date_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1855185481;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A newly created video chat.
 */
class messageVideoChatStarted final : public MessageContent {
 public:
  /// Identifier of the video chat. The video chat can be received through the method getGroupCall.
  int32 group_call_id_;

  /**
   * A newly created video chat.
   */
  messageVideoChatStarted();

  /**
   * A newly created video chat.
   *
   * \param[in] group_call_id_ Identifier of the video chat. The video chat can be received through the method getGroupCall.
   */
  explicit messageVideoChatStarted(int32 group_call_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 521225561;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A message with information about an ended video chat.
 */
class messageVideoChatEnded final : public MessageContent {
 public:
  /// Call duration, in seconds.
  int32 duration_;

  /**
   * A message with information about an ended video chat.
   */
  messageVideoChatEnded();

  /**
   * A message with information about an ended video chat.
   *
   * \param[in] duration_ Call duration, in seconds.
   */
  explicit messageVideoChatEnded(int32 duration_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 2032544855;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A message with information about an invite to a video chat.
 */
class messageInviteVideoChatParticipants final : public MessageContent {
 public:
  /// Identifier of the video chat. The video chat can be received through the method getGroupCall.
  int32 group_call_id_;
  /// Invited user identifiers.
  array<int53> user_ids_;

  /**
   * A message with information about an invite to a video chat.
   */
  messageInviteVideoChatParticipants();

  /**
   * A message with information about an invite to a video chat.
   *
   * \param[in] group_call_id_ Identifier of the video chat. The video chat can be received through the method getGroupCall.
   * \param[in] user_ids_ Invited user identifiers.
   */
  messageInviteVideoChatParticipants(int32 group_call_id_, array<int53> &&user_ids_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1459065585;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A newly created basic group.
 */
class messageBasicGroupChatCreate final : public MessageContent {
 public:
  /// Title of the basic group.
  string title_;
  /// User identifiers of members in the basic group.
  array<int53> member_user_ids_;

  /**
   * A newly created basic group.
   */
  messageBasicGroupChatCreate();

  /**
   * A newly created basic group.
   *
   * \param[in] title_ Title of the basic group.
   * \param[in] member_user_ids_ User identifiers of members in the basic group.
   */
  messageBasicGroupChatCreate(string const &title_, array<int53> &&member_user_ids_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 795404060;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A newly created supergroup or channel.
 */
class messageSupergroupChatCreate final : public MessageContent {
 public:
  /// Title of the supergroup or channel.
  string title_;

  /**
   * A newly created supergroup or channel.
   */
  messageSupergroupChatCreate();

  /**
   * A newly created supergroup or channel.
   *
   * \param[in] title_ Title of the supergroup or channel.
   */
  explicit messageSupergroupChatCreate(string const &title_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -434325733;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * An updated chat title.
 */
class messageChatChangeTitle final : public MessageContent {
 public:
  /// New chat title.
  string title_;

  /**
   * An updated chat title.
   */
  messageChatChangeTitle();

  /**
   * An updated chat title.
   *
   * \param[in] title_ New chat title.
   */
  explicit messageChatChangeTitle(string const &title_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 748272449;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * An updated chat photo.
 */
class messageChatChangePhoto final : public MessageContent {
 public:
  /// New chat photo.
  object_ptr<chatPhoto> photo_;

  /**
   * An updated chat photo.
   */
  messageChatChangePhoto();

  /**
   * An updated chat photo.
   *
   * \param[in] photo_ New chat photo.
   */
  explicit messageChatChangePhoto(object_ptr<chatPhoto> &&photo_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -813415093;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A deleted chat photo.
 */
class messageChatDeletePhoto final : public MessageContent {
 public:

  /**
   * A deleted chat photo.
   */
  messageChatDeletePhoto();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -184374809;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * New chat members were added.
 */
class messageChatAddMembers final : public MessageContent {
 public:
  /// User identifiers of the new members.
  array<int53> member_user_ids_;

  /**
   * New chat members were added.
   */
  messageChatAddMembers();

  /**
   * New chat members were added.
   *
   * \param[in] member_user_ids_ User identifiers of the new members.
   */
  explicit messageChatAddMembers(array<int53> &&member_user_ids_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1701117908;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A new member joined the chat via an invite link.
 */
class messageChatJoinByLink final : public MessageContent {
 public:

  /**
   * A new member joined the chat via an invite link.
   */
  messageChatJoinByLink();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1846493311;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A new member was accepted to the chat by an administrator.
 */
class messageChatJoinByRequest final : public MessageContent {
 public:

  /**
   * A new member was accepted to the chat by an administrator.
   */
  messageChatJoinByRequest();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1195428732;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A chat member was deleted.
 */
class messageChatDeleteMember final : public MessageContent {
 public:
  /// User identifier of the deleted chat member.
  int53 user_id_;

  /**
   * A chat member was deleted.
   */
  messageChatDeleteMember();

  /**
   * A chat member was deleted.
   *
   * \param[in] user_id_ User identifier of the deleted chat member.
   */
  explicit messageChatDeleteMember(int53 user_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 938029481;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A basic group was upgraded to a supergroup and was deactivated as the result.
 */
class messageChatUpgradeTo final : public MessageContent {
 public:
  /// Identifier of the supergroup to which the basic group was upgraded.
  int53 supergroup_id_;

  /**
   * A basic group was upgraded to a supergroup and was deactivated as the result.
   */
  messageChatUpgradeTo();

  /**
   * A basic group was upgraded to a supergroup and was deactivated as the result.
   *
   * \param[in] supergroup_id_ Identifier of the supergroup to which the basic group was upgraded.
   */
  explicit messageChatUpgradeTo(int53 supergroup_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 104813723;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A supergroup has been created from a basic group.
 */
class messageChatUpgradeFrom final : public MessageContent {
 public:
  /// Title of the newly created supergroup.
  string title_;
  /// The identifier of the original basic group.
  int53 basic_group_id_;

  /**
   * A supergroup has been created from a basic group.
   */
  messageChatUpgradeFrom();

  /**
   * A supergroup has been created from a basic group.
   *
   * \param[in] title_ Title of the newly created supergroup.
   * \param[in] basic_group_id_ The identifier of the original basic group.
   */
  messageChatUpgradeFrom(string const &title_, int53 basic_group_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 325954268;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A message has been pinned.
 */
class messagePinMessage final : public MessageContent {
 public:
  /// Identifier of the pinned message, can be an identifier of a deleted message or 0.
  int53 message_id_;

  /**
   * A message has been pinned.
   */
  messagePinMessage();

  /**
   * A message has been pinned.
   *
   * \param[in] message_id_ Identifier of the pinned message, can be an identifier of a deleted message or 0.
   */
  explicit messagePinMessage(int53 message_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 953503801;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A screenshot of a message in the chat has been taken.
 */
class messageScreenshotTaken final : public MessageContent {
 public:

  /**
   * A screenshot of a message in the chat has been taken.
   */
  messageScreenshotTaken();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1564971605;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A theme in the chat has been changed.
 */
class messageChatSetTheme final : public MessageContent {
 public:
  /// If non-empty, name of a new theme, set for the chat. Otherwise chat theme was reset to the default one.
  string theme_name_;

  /**
   * A theme in the chat has been changed.
   */
  messageChatSetTheme();

  /**
   * A theme in the chat has been changed.
   *
   * \param[in] theme_name_ If non-empty, name of a new theme, set for the chat. Otherwise chat theme was reset to the default one.
   */
  explicit messageChatSetTheme(string const &theme_name_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1716612088;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The TTL (Time To Live) setting for messages in the chat has been changed.
 */
class messageChatSetTtl final : public MessageContent {
 public:
  /// New message TTL.
  int32 ttl_;

  /**
   * The TTL (Time To Live) setting for messages in the chat has been changed.
   */
  messageChatSetTtl();

  /**
   * The TTL (Time To Live) setting for messages in the chat has been changed.
   *
   * \param[in] ttl_ New message TTL.
   */
  explicit messageChatSetTtl(int32 ttl_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1810060209;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A non-standard action has happened in the chat.
 */
class messageCustomServiceAction final : public MessageContent {
 public:
  /// Message text to be shown in the chat.
  string text_;

  /**
   * A non-standard action has happened in the chat.
   */
  messageCustomServiceAction();

  /**
   * A non-standard action has happened in the chat.
   *
   * \param[in] text_ Message text to be shown in the chat.
   */
  explicit messageCustomServiceAction(string const &text_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1435879282;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A new high score was achieved in a game.
 */
class messageGameScore final : public MessageContent {
 public:
  /// Identifier of the message with the game, can be an identifier of a deleted message.
  int53 game_message_id_;
  /// Identifier of the game; may be different from the games presented in the message with the game.
  int64 game_id_;
  /// New score.
  int32 score_;

  /**
   * A new high score was achieved in a game.
   */
  messageGameScore();

  /**
   * A new high score was achieved in a game.
   *
   * \param[in] game_message_id_ Identifier of the message with the game, can be an identifier of a deleted message.
   * \param[in] game_id_ Identifier of the game; may be different from the games presented in the message with the game.
   * \param[in] score_ New score.
   */
  messageGameScore(int53 game_message_id_, int64 game_id_, int32 score_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1344904575;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A payment has been completed.
 */
class messagePaymentSuccessful final : public MessageContent {
 public:
  /// Identifier of the chat, containing the corresponding invoice message; 0 if unknown.
  int53 invoice_chat_id_;
  /// Identifier of the message with the corresponding invoice; can be an identifier of a deleted message.
  int53 invoice_message_id_;
  /// Currency for the price of the product.
  string currency_;
  /// Total price for the product, in the smallest units of the currency.
  int53 total_amount_;

  /**
   * A payment has been completed.
   */
  messagePaymentSuccessful();

  /**
   * A payment has been completed.
   *
   * \param[in] invoice_chat_id_ Identifier of the chat, containing the corresponding invoice message; 0 if unknown.
   * \param[in] invoice_message_id_ Identifier of the message with the corresponding invoice; can be an identifier of a deleted message.
   * \param[in] currency_ Currency for the price of the product.
   * \param[in] total_amount_ Total price for the product, in the smallest units of the currency.
   */
  messagePaymentSuccessful(int53 invoice_chat_id_, int53 invoice_message_id_, string const &currency_, int53 total_amount_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1442934098;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A payment has been completed; for bots only.
 */
class messagePaymentSuccessfulBot final : public MessageContent {
 public:
  /// Currency for price of the product.
  string currency_;
  /// Total price for the product, in the smallest units of the currency.
  int53 total_amount_;
  /// Invoice payload.
  bytes invoice_payload_;
  /// Identifier of the shipping option chosen by the user; may be empty if not applicable.
  string shipping_option_id_;
  /// Information about the order; may be null.
  object_ptr<orderInfo> order_info_;
  /// Telegram payment identifier.
  string telegram_payment_charge_id_;
  /// Provider payment identifier.
  string provider_payment_charge_id_;

  /**
   * A payment has been completed; for bots only.
   */
  messagePaymentSuccessfulBot();

  /**
   * A payment has been completed; for bots only.
   *
   * \param[in] currency_ Currency for price of the product.
   * \param[in] total_amount_ Total price for the product, in the smallest units of the currency.
   * \param[in] invoice_payload_ Invoice payload.
   * \param[in] shipping_option_id_ Identifier of the shipping option chosen by the user; may be empty if not applicable.
   * \param[in] order_info_ Information about the order; may be null.
   * \param[in] telegram_payment_charge_id_ Telegram payment identifier.
   * \param[in] provider_payment_charge_id_ Provider payment identifier.
   */
  messagePaymentSuccessfulBot(string const &currency_, int53 total_amount_, bytes const &invoice_payload_, string const &shipping_option_id_, object_ptr<orderInfo> &&order_info_, string const &telegram_payment_charge_id_, string const &provider_payment_charge_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -438430050;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A contact has registered with Telegram.
 */
class messageContactRegistered final : public MessageContent {
 public:

  /**
   * A contact has registered with Telegram.
   */
  messageContactRegistered();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1502020353;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The current user has connected a website by logging in using Telegram Login Widget on it.
 */
class messageWebsiteConnected final : public MessageContent {
 public:
  /// Domain name of the connected website.
  string domain_name_;

  /**
   * The current user has connected a website by logging in using Telegram Login Widget on it.
   */
  messageWebsiteConnected();

  /**
   * The current user has connected a website by logging in using Telegram Login Widget on it.
   *
   * \param[in] domain_name_ Domain name of the connected website.
   */
  explicit messageWebsiteConnected(string const &domain_name_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1074551800;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Telegram Passport data has been sent.
 */
class messagePassportDataSent final : public MessageContent {
 public:
  /// List of Telegram Passport element types sent.
  array<object_ptr<PassportElementType>> types_;

  /**
   * Telegram Passport data has been sent.
   */
  messagePassportDataSent();

  /**
   * Telegram Passport data has been sent.
   *
   * \param[in] types_ List of Telegram Passport element types sent.
   */
  explicit messagePassportDataSent(array<object_ptr<PassportElementType>> &&types_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1017405171;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Telegram Passport data has been received; for bots only.
 */
class messagePassportDataReceived final : public MessageContent {
 public:
  /// List of received Telegram Passport elements.
  array<object_ptr<encryptedPassportElement>> elements_;
  /// Encrypted data credentials.
  object_ptr<encryptedCredentials> credentials_;

  /**
   * Telegram Passport data has been received; for bots only.
   */
  messagePassportDataReceived();

  /**
   * Telegram Passport data has been received; for bots only.
   *
   * \param[in] elements_ List of received Telegram Passport elements.
   * \param[in] credentials_ Encrypted data credentials.
   */
  messagePassportDataReceived(array<object_ptr<encryptedPassportElement>> &&elements_, object_ptr<encryptedCredentials> &&credentials_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1367863624;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A user in the chat came within proximity alert range.
 */
class messageProximityAlertTriggered final : public MessageContent {
 public:
  /// The identifier of a user or chat that triggered the proximity alert.
  object_ptr<MessageSender> traveler_id_;
  /// The identifier of a user or chat that subscribed for the proximity alert.
  object_ptr<MessageSender> watcher_id_;
  /// The distance between the users.
  int32 distance_;

  /**
   * A user in the chat came within proximity alert range.
   */
  messageProximityAlertTriggered();

  /**
   * A user in the chat came within proximity alert range.
   *
   * \param[in] traveler_id_ The identifier of a user or chat that triggered the proximity alert.
   * \param[in] watcher_id_ The identifier of a user or chat that subscribed for the proximity alert.
   * \param[in] distance_ The distance between the users.
   */
  messageProximityAlertTriggered(object_ptr<MessageSender> &&traveler_id_, object_ptr<MessageSender> &&watcher_id_, int32 distance_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 67761875;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Message content that is not supported in the current TDLib version.
 */
class messageUnsupported final : public MessageContent {
 public:

  /**
   * Message content that is not supported in the current TDLib version.
   */
  messageUnsupported();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1816726139;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Options to be used when a message content is copied without reference to the original sender. Service messages and messageInvoice can&\#039;t be copied.
 */
class messageCopyOptions final : public Object {
 public:
  /// True, if content of the message needs to be copied without reference to the original sender. Always true if the message is forwarded to a secret chat or is local.
  bool send_copy_;
  /// True, if media caption of the message copy needs to be replaced. Ignored if send_copy is false.
  bool replace_caption_;
  /// New message caption; pass null to copy message without caption. Ignored if replace_caption is false.
  object_ptr<formattedText> new_caption_;

  /**
   * Options to be used when a message content is copied without reference to the original sender. Service messages and messageInvoice can&\#039;t be copied.
   */
  messageCopyOptions();

  /**
   * Options to be used when a message content is copied without reference to the original sender. Service messages and messageInvoice can&\#039;t be copied.
   *
   * \param[in] send_copy_ True, if content of the message needs to be copied without reference to the original sender. Always true if the message is forwarded to a secret chat or is local.
   * \param[in] replace_caption_ True, if media caption of the message copy needs to be replaced. Ignored if send_copy is false.
   * \param[in] new_caption_ New message caption; pass null to copy message without caption. Ignored if replace_caption is false.
   */
  messageCopyOptions(bool send_copy_, bool replace_caption_, object_ptr<formattedText> &&new_caption_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1208442937;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Contains information about a file with messages exported from another app.
 */
class MessageFileType: public Object {
 public:
};

/**
 * The messages was exported from a private chat.
 */
class messageFileTypePrivate final : public MessageFileType {
 public:
  /// Name of the other party; may be empty if unrecognized.
  string name_;

  /**
   * The messages was exported from a private chat.
   */
  messageFileTypePrivate();

  /**
   * The messages was exported from a private chat.
   *
   * \param[in] name_ Name of the other party; may be empty if unrecognized.
   */
  explicit messageFileTypePrivate(string const &name_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -521908524;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The messages was exported from a group chat.
 */
class messageFileTypeGroup final : public MessageFileType {
 public:
  /// Title of the group chat; may be empty if unrecognized.
  string title_;

  /**
   * The messages was exported from a group chat.
   */
  messageFileTypeGroup();

  /**
   * The messages was exported from a group chat.
   *
   * \param[in] title_ Title of the group chat; may be empty if unrecognized.
   */
  explicit messageFileTypeGroup(string const &title_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -219836568;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The messages was exported from a chat of unknown type.
 */
class messageFileTypeUnknown final : public MessageFileType {
 public:

  /**
   * The messages was exported from a chat of unknown type.
   */
  messageFileTypeUnknown();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1176353458;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains information about a forwarded message.
 */
class messageForwardInfo final : public Object {
 public:
  /// Origin of a forwarded message.
  object_ptr<MessageForwardOrigin> origin_;
  /// Point in time (Unix timestamp) when the message was originally sent.
  int32 date_;
  /// The type of a public service announcement for the forwarded message.
  string public_service_announcement_type_;
  /// For messages forwarded to the chat with the current user (Saved Messages), to the Replies bot chat, or to the channel&\#039;s discussion group, the identifier of the chat from which the message was forwarded last time; 0 if unknown.
  int53 from_chat_id_;
  /// For messages forwarded to the chat with the current user (Saved Messages), to the Replies bot chat, or to the channel&\#039;s discussion group, the identifier of the original message from which the new message was forwarded last time; 0 if unknown.
  int53 from_message_id_;

  /**
   * Contains information about a forwarded message.
   */
  messageForwardInfo();

  /**
   * Contains information about a forwarded message.
   *
   * \param[in] origin_ Origin of a forwarded message.
   * \param[in] date_ Point in time (Unix timestamp) when the message was originally sent.
   * \param[in] public_service_announcement_type_ The type of a public service announcement for the forwarded message.
   * \param[in] from_chat_id_ For messages forwarded to the chat with the current user (Saved Messages), to the Replies bot chat, or to the channel&\#039;s discussion group, the identifier of the chat from which the message was forwarded last time; 0 if unknown.
   * \param[in] from_message_id_ For messages forwarded to the chat with the current user (Saved Messages), to the Replies bot chat, or to the channel&\#039;s discussion group, the identifier of the original message from which the new message was forwarded last time; 0 if unknown.
   */
  messageForwardInfo(object_ptr<MessageForwardOrigin> &&origin_, int32 date_, string const &public_service_announcement_type_, int53 from_chat_id_, int53 from_message_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -327300408;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Contains information about the origin of a forwarded message.
 */
class MessageForwardOrigin: public Object {
 public:
};

/**
 * The message was originally sent by a known user.
 */
class messageForwardOriginUser final : public MessageForwardOrigin {
 public:
  /// Identifier of the user that originally sent the message.
  int53 sender_user_id_;

  /**
   * The message was originally sent by a known user.
   */
  messageForwardOriginUser();

  /**
   * The message was originally sent by a known user.
   *
   * \param[in] sender_user_id_ Identifier of the user that originally sent the message.
   */
  explicit messageForwardOriginUser(int53 sender_user_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -355174191;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The message was originally sent on behalf of a chat.
 */
class messageForwardOriginChat final : public MessageForwardOrigin {
 public:
  /// Identifier of the chat that originally sent the message.
  int53 sender_chat_id_;
  /// For messages originally sent by an anonymous chat administrator, original message author signature.
  string author_signature_;

  /**
   * The message was originally sent on behalf of a chat.
   */
  messageForwardOriginChat();

  /**
   * The message was originally sent on behalf of a chat.
   *
   * \param[in] sender_chat_id_ Identifier of the chat that originally sent the message.
   * \param[in] author_signature_ For messages originally sent by an anonymous chat administrator, original message author signature.
   */
  messageForwardOriginChat(int53 sender_chat_id_, string const &author_signature_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1526010724;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The message was originally sent by a user, which is hidden by their privacy settings.
 */
class messageForwardOriginHiddenUser final : public MessageForwardOrigin {
 public:
  /// Name of the sender.
  string sender_name_;

  /**
   * The message was originally sent by a user, which is hidden by their privacy settings.
   */
  messageForwardOriginHiddenUser();

  /**
   * The message was originally sent by a user, which is hidden by their privacy settings.
   *
   * \param[in] sender_name_ Name of the sender.
   */
  explicit messageForwardOriginHiddenUser(string const &sender_name_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -271257885;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The message was originally a post in a channel.
 */
class messageForwardOriginChannel final : public MessageForwardOrigin {
 public:
  /// Identifier of the chat from which the message was originally forwarded.
  int53 chat_id_;
  /// Message identifier of the original message.
  int53 message_id_;
  /// Original post author signature.
  string author_signature_;

  /**
   * The message was originally a post in a channel.
   */
  messageForwardOriginChannel();

  /**
   * The message was originally a post in a channel.
   *
   * \param[in] chat_id_ Identifier of the chat from which the message was originally forwarded.
   * \param[in] message_id_ Message identifier of the original message.
   * \param[in] author_signature_ Original post author signature.
   */
  messageForwardOriginChannel(int53 chat_id_, int53 message_id_, string const &author_signature_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1490730723;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The message was imported from an exported message history.
 */
class messageForwardOriginMessageImport final : public MessageForwardOrigin {
 public:
  /// Name of the sender.
  string sender_name_;

  /**
   * The message was imported from an exported message history.
   */
  messageForwardOriginMessageImport();

  /**
   * The message was imported from an exported message history.
   *
   * \param[in] sender_name_ Name of the sender.
   */
  explicit messageForwardOriginMessageImport(string const &sender_name_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -739561951;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains information about interactions with a message.
 */
class messageInteractionInfo final : public Object {
 public:
  /// Number of times the message was viewed.
  int32 view_count_;
  /// Number of times the message was forwarded.
  int32 forward_count_;
  /// Information about direct or indirect replies to the message; may be null. Currently, available only in channels with a discussion supergroup and discussion supergroups for messages, which are not replies itself.
  object_ptr<messageReplyInfo> reply_info_;

  /**
   * Contains information about interactions with a message.
   */
  messageInteractionInfo();

  /**
   * Contains information about interactions with a message.
   *
   * \param[in] view_count_ Number of times the message was viewed.
   * \param[in] forward_count_ Number of times the message was forwarded.
   * \param[in] reply_info_ Information about direct or indirect replies to the message; may be null. Currently, available only in channels with a discussion supergroup and discussion supergroups for messages, which are not replies itself.
   */
  messageInteractionInfo(int32 view_count_, int32 forward_count_, object_ptr<messageReplyInfo> &&reply_info_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -620714966;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains an HTTPS link to a message in a supergroup or channel.
 */
class messageLink final : public Object {
 public:
  /// Message link.
  string link_;
  /// True, if the link will work for non-members of the chat.
  bool is_public_;

  /**
   * Contains an HTTPS link to a message in a supergroup or channel.
   */
  messageLink();

  /**
   * Contains an HTTPS link to a message in a supergroup or channel.
   *
   * \param[in] link_ Message link.
   * \param[in] is_public_ True, if the link will work for non-members of the chat.
   */
  messageLink(string const &link_, bool is_public_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1354089818;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains information about a link to a message in a chat.
 */
class messageLinkInfo final : public Object {
 public:
  /// True, if the link is a public link for a message in a chat.
  bool is_public_;
  /// If found, identifier of the chat to which the message belongs, 0 otherwise.
  int53 chat_id_;
  /// If found, the linked message; may be null.
  object_ptr<message> message_;
  /// Timestamp from which the video/audio/video note/voice note playing must start, in seconds; 0 if not specified. The media can be in the message content or in its web page preview.
  int32 media_timestamp_;
  /// True, if the whole media album to which the message belongs is linked.
  bool for_album_;
  /// True, if the message is linked as a channel post comment or from a message thread.
  bool for_comment_;

  /**
   * Contains information about a link to a message in a chat.
   */
  messageLinkInfo();

  /**
   * Contains information about a link to a message in a chat.
   *
   * \param[in] is_public_ True, if the link is a public link for a message in a chat.
   * \param[in] chat_id_ If found, identifier of the chat to which the message belongs, 0 otherwise.
   * \param[in] message_ If found, the linked message; may be null.
   * \param[in] media_timestamp_ Timestamp from which the video/audio/video note/voice note playing must start, in seconds; 0 if not specified. The media can be in the message content or in its web page preview.
   * \param[in] for_album_ True, if the whole media album to which the message belongs is linked.
   * \param[in] for_comment_ True, if the message is linked as a channel post comment or from a message thread.
   */
  messageLinkInfo(bool is_public_, int53 chat_id_, object_ptr<message> &&message_, int32 media_timestamp_, bool for_album_, bool for_comment_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -981646294;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains information about a message in a specific position.
 */
class messagePosition final : public Object {
 public:
  /// 0-based message position in the full list of suitable messages.
  int32 position_;
  /// Message identifier.
  int53 message_id_;
  /// Point in time (Unix timestamp) when the message was sent.
  int32 date_;

  /**
   * Contains information about a message in a specific position.
   */
  messagePosition();

  /**
   * Contains information about a message in a specific position.
   *
   * \param[in] position_ 0-based message position in the full list of suitable messages.
   * \param[in] message_id_ Message identifier.
   * \param[in] date_ Point in time (Unix timestamp) when the message was sent.
   */
  messagePosition(int32 position_, int53 message_id_, int32 date_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1292189935;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains a list of message positions.
 */
class messagePositions final : public Object {
 public:
  /// Total count of messages found.
  int32 total_count_;
  /// List of message positions.
  array<object_ptr<messagePosition>> positions_;

  /**
   * Contains a list of message positions.
   */
  messagePositions();

  /**
   * Contains a list of message positions.
   *
   * \param[in] total_count_ Total count of messages found.
   * \param[in] positions_ List of message positions.
   */
  messagePositions(int32 total_count_, array<object_ptr<messagePosition>> &&positions_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1930466649;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains information about replies to a message.
 */
class messageReplyInfo final : public Object {
 public:
  /// Number of times the message was directly or indirectly replied.
  int32 reply_count_;
  /// Identifiers of at most 3 recent repliers to the message; available in channels with a discussion supergroup. The users and chats are expected to be inaccessible: only their photo and name will be available.
  array<object_ptr<MessageSender>> recent_replier_ids_;
  /// Identifier of the last read incoming reply to the message.
  int53 last_read_inbox_message_id_;
  /// Identifier of the last read outgoing reply to the message.
  int53 last_read_outbox_message_id_;
  /// Identifier of the last reply to the message.
  int53 last_message_id_;

  /**
   * Contains information about replies to a message.
   */
  messageReplyInfo();

  /**
   * Contains information about replies to a message.
   *
   * \param[in] reply_count_ Number of times the message was directly or indirectly replied.
   * \param[in] recent_replier_ids_ Identifiers of at most 3 recent repliers to the message; available in channels with a discussion supergroup. The users and chats are expected to be inaccessible: only their photo and name will be available.
   * \param[in] last_read_inbox_message_id_ Identifier of the last read incoming reply to the message.
   * \param[in] last_read_outbox_message_id_ Identifier of the last read outgoing reply to the message.
   * \param[in] last_message_id_ Identifier of the last reply to the message.
   */
  messageReplyInfo(int32 reply_count_, array<object_ptr<MessageSender>> &&recent_replier_ids_, int53 last_read_inbox_message_id_, int53 last_read_outbox_message_id_, int53 last_message_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -2093702263;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Contains information about the time when a scheduled message will be sent.
 */
class MessageSchedulingState: public Object {
 public:
};

/**
 * The message will be sent at the specified date.
 */
class messageSchedulingStateSendAtDate final : public MessageSchedulingState {
 public:
  /// Date the message will be sent. The date must be within 367 days in the future.
  int32 send_date_;

  /**
   * The message will be sent at the specified date.
   */
  messageSchedulingStateSendAtDate();

  /**
   * The message will be sent at the specified date.
   *
   * \param[in] send_date_ Date the message will be sent. The date must be within 367 days in the future.
   */
  explicit messageSchedulingStateSendAtDate(int32 send_date_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1485570073;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The message will be sent when the peer will be online. Applicable to private chats only and when the exact online status of the peer is known.
 */
class messageSchedulingStateSendWhenOnline final : public MessageSchedulingState {
 public:

  /**
   * The message will be sent when the peer will be online. Applicable to private chats only and when the exact online status of the peer is known.
   */
  messageSchedulingStateSendWhenOnline();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 2092947464;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Options to be used when a message is sent.
 */
class messageSendOptions final : public Object {
 public:
  /// Pass true to disable notification for the message.
  bool disable_notification_;
  /// Pass true if the message is sent from the background.
  bool from_background_;
  /// Message scheduling state; pass null to send message immediately. Messages sent to a secret chat, live location messages and self-destructing messages can&\#039;t be scheduled.
  object_ptr<MessageSchedulingState> scheduling_state_;

  /**
   * Options to be used when a message is sent.
   */
  messageSendOptions();

  /**
   * Options to be used when a message is sent.
   *
   * \param[in] disable_notification_ Pass true to disable notification for the message.
   * \param[in] from_background_ Pass true if the message is sent from the background.
   * \param[in] scheduling_state_ Message scheduling state; pass null to send message immediately. Messages sent to a secret chat, live location messages and self-destructing messages can&\#039;t be scheduled.
   */
  messageSendOptions(bool disable_notification_, bool from_background_, object_ptr<MessageSchedulingState> &&scheduling_state_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 914544314;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Contains information about the sender of a message.
 */
class MessageSender: public Object {
 public:
};

/**
 * The message was sent by a known user.
 */
class messageSenderUser final : public MessageSender {
 public:
  /// Identifier of the user that sent the message.
  int53 user_id_;

  /**
   * The message was sent by a known user.
   */
  messageSenderUser();

  /**
   * The message was sent by a known user.
   *
   * \param[in] user_id_ Identifier of the user that sent the message.
   */
  explicit messageSenderUser(int53 user_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -336109341;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The message was sent on behalf of a chat.
 */
class messageSenderChat final : public MessageSender {
 public:
  /// Identifier of the chat that sent the message.
  int53 chat_id_;

  /**
   * The message was sent on behalf of a chat.
   */
  messageSenderChat();

  /**
   * The message was sent on behalf of a chat.
   *
   * \param[in] chat_id_ Identifier of the chat that sent the message.
   */
  explicit messageSenderChat(int53 chat_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -239660751;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Represents a list of message senders.
 */
class messageSenders final : public Object {
 public:
  /// Approximate total count of messages senders found.
  int32 total_count_;
  /// List of message senders.
  array<object_ptr<MessageSender>> senders_;

  /**
   * Represents a list of message senders.
   */
  messageSenders();

  /**
   * Represents a list of message senders.
   *
   * \param[in] total_count_ Approximate total count of messages senders found.
   * \param[in] senders_ List of message senders.
   */
  messageSenders(int32 total_count_, array<object_ptr<MessageSender>> &&senders_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -690158467;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Contains information about the sending state of the message.
 */
class MessageSendingState: public Object {
 public:
};

/**
 * The message is being sent now, but has not yet been delivered to the server.
 */
class messageSendingStatePending final : public MessageSendingState {
 public:

  /**
   * The message is being sent now, but has not yet been delivered to the server.
   */
  messageSendingStatePending();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1381803582;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The message failed to be sent.
 */
class messageSendingStateFailed final : public MessageSendingState {
 public:
  /// An error code; 0 if unknown.
  int32 error_code_;
  /// Error message.
  string error_message_;
  /// True, if the message can be re-sent.
  bool can_retry_;
  /// True, if the message can be re-sent only on behalf of a different sender.
  bool need_another_sender_;
  /// Time left before the message can be re-sent, in seconds. No update is sent when this field changes.
  double retry_after_;

  /**
   * The message failed to be sent.
   */
  messageSendingStateFailed();

  /**
   * The message failed to be sent.
   *
   * \param[in] error_code_ An error code; 0 if unknown.
   * \param[in] error_message_ Error message.
   * \param[in] can_retry_ True, if the message can be re-sent.
   * \param[in] need_another_sender_ True, if the message can be re-sent only on behalf of a different sender.
   * \param[in] retry_after_ Time left before the message can be re-sent, in seconds. No update is sent when this field changes.
   */
  messageSendingStateFailed(int32 error_code_, string const &error_message_, bool can_retry_, bool need_another_sender_, double retry_after_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1741887228;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A detailed statistics about a message.
 */
class messageStatistics final : public Object {
 public:
  /// A graph containing number of message views and shares.
  object_ptr<StatisticalGraph> message_interaction_graph_;

  /**
   * A detailed statistics about a message.
   */
  messageStatistics();

  /**
   * A detailed statistics about a message.
   *
   * \param[in] message_interaction_graph_ A graph containing number of message views and shares.
   */
  explicit messageStatistics(object_ptr<StatisticalGraph> &&message_interaction_graph_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1011383888;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains information about a message thread.
 */
class messageThreadInfo final : public Object {
 public:
  /// Identifier of the chat to which the message thread belongs.
  int53 chat_id_;
  /// Message thread identifier, unique within the chat.
  int53 message_thread_id_;
  /// Information about the message thread.
  object_ptr<messageReplyInfo> reply_info_;
  /// Approximate number of unread messages in the message thread.
  int32 unread_message_count_;
  /// The messages from which the thread starts. The messages are returned in a reverse chronological order (i.e., in order of decreasing message_id).
  array<object_ptr<message>> messages_;
  /// A draft of a message in the message thread; may be null.
  object_ptr<draftMessage> draft_message_;

  /**
   * Contains information about a message thread.
   */
  messageThreadInfo();

  /**
   * Contains information about a message thread.
   *
   * \param[in] chat_id_ Identifier of the chat to which the message thread belongs.
   * \param[in] message_thread_id_ Message thread identifier, unique within the chat.
   * \param[in] reply_info_ Information about the message thread.
   * \param[in] unread_message_count_ Approximate number of unread messages in the message thread.
   * \param[in] messages_ The messages from which the thread starts. The messages are returned in a reverse chronological order (i.e., in order of decreasing message_id).
   * \param[in] draft_message_ A draft of a message in the message thread; may be null.
   */
  messageThreadInfo(int53 chat_id_, int53 message_thread_id_, object_ptr<messageReplyInfo> &&reply_info_, int32 unread_message_count_, array<object_ptr<message>> &&messages_, object_ptr<draftMessage> &&draft_message_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -248536056;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains a list of messages.
 */
class messages final : public Object {
 public:
  /// Approximate total count of messages found.
  int32 total_count_;
  /// List of messages; messages may be null.
  array<object_ptr<message>> messages_;

  /**
   * Contains a list of messages.
   */
  messages();

  /**
   * Contains a list of messages.
   *
   * \param[in] total_count_ Approximate total count of messages found.
   * \param[in] messages_ List of messages; messages may be null.
   */
  messages(int32 total_count_, array<object_ptr<message>> &&messages_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -16498159;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Thumbnail image of a very poor quality and low resolution.
 */
class minithumbnail final : public Object {
 public:
  /// Thumbnail width, usually doesn&\#039;t exceed 40.
  int32 width_;
  /// Thumbnail height, usually doesn&\#039;t exceed 40.
  int32 height_;
  /// The thumbnail in JPEG format.
  bytes data_;

  /**
   * Thumbnail image of a very poor quality and low resolution.
   */
  minithumbnail();

  /**
   * Thumbnail image of a very poor quality and low resolution.
   *
   * \param[in] width_ Thumbnail width, usually doesn&\#039;t exceed 40.
   * \param[in] height_ Thumbnail height, usually doesn&\#039;t exceed 40.
   * \param[in] data_ The thumbnail in JPEG format.
   */
  minithumbnail(int32 width_, int32 height_, bytes const &data_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -328540758;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A full list of available network statistic entries.
 */
class networkStatistics final : public Object {
 public:
  /// Point in time (Unix timestamp) from which the statistics are collected.
  int32 since_date_;
  /// Network statistics entries.
  array<object_ptr<NetworkStatisticsEntry>> entries_;

  /**
   * A full list of available network statistic entries.
   */
  networkStatistics();

  /**
   * A full list of available network statistic entries.
   *
   * \param[in] since_date_ Point in time (Unix timestamp) from which the statistics are collected.
   * \param[in] entries_ Network statistics entries.
   */
  networkStatistics(int32 since_date_, array<object_ptr<NetworkStatisticsEntry>> &&entries_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1615554212;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Contains statistics about network usage.
 */
class NetworkStatisticsEntry: public Object {
 public:
};

/**
 * Contains information about the total amount of data that was used to send and receive files.
 */
class networkStatisticsEntryFile final : public NetworkStatisticsEntry {
 public:
  /// Type of the file the data is part of; pass null if the data isn&\#039;t related to files.
  object_ptr<FileType> file_type_;
  /// Type of the network the data was sent through. Call setNetworkType to maintain the actual network type.
  object_ptr<NetworkType> network_type_;
  /// Total number of bytes sent.
  int53 sent_bytes_;
  /// Total number of bytes received.
  int53 received_bytes_;

  /**
   * Contains information about the total amount of data that was used to send and receive files.
   */
  networkStatisticsEntryFile();

  /**
   * Contains information about the total amount of data that was used to send and receive files.
   *
   * \param[in] file_type_ Type of the file the data is part of; pass null if the data isn&\#039;t related to files.
   * \param[in] network_type_ Type of the network the data was sent through. Call setNetworkType to maintain the actual network type.
   * \param[in] sent_bytes_ Total number of bytes sent.
   * \param[in] received_bytes_ Total number of bytes received.
   */
  networkStatisticsEntryFile(object_ptr<FileType> &&file_type_, object_ptr<NetworkType> &&network_type_, int53 sent_bytes_, int53 received_bytes_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 188452706;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains information about the total amount of data that was used for calls.
 */
class networkStatisticsEntryCall final : public NetworkStatisticsEntry {
 public:
  /// Type of the network the data was sent through. Call setNetworkType to maintain the actual network type.
  object_ptr<NetworkType> network_type_;
  /// Total number of bytes sent.
  int53 sent_bytes_;
  /// Total number of bytes received.
  int53 received_bytes_;
  /// Total call duration, in seconds.
  double duration_;

  /**
   * Contains information about the total amount of data that was used for calls.
   */
  networkStatisticsEntryCall();

  /**
   * Contains information about the total amount of data that was used for calls.
   *
   * \param[in] network_type_ Type of the network the data was sent through. Call setNetworkType to maintain the actual network type.
   * \param[in] sent_bytes_ Total number of bytes sent.
   * \param[in] received_bytes_ Total number of bytes received.
   * \param[in] duration_ Total call duration, in seconds.
   */
  networkStatisticsEntryCall(object_ptr<NetworkType> &&network_type_, int53 sent_bytes_, int53 received_bytes_, double duration_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 737000365;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Represents the type of a network.
 */
class NetworkType: public Object {
 public:
};

/**
 * The network is not available.
 */
class networkTypeNone final : public NetworkType {
 public:

  /**
   * The network is not available.
   */
  networkTypeNone();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1971691759;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A mobile network.
 */
class networkTypeMobile final : public NetworkType {
 public:

  /**
   * A mobile network.
   */
  networkTypeMobile();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 819228239;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A mobile roaming network.
 */
class networkTypeMobileRoaming final : public NetworkType {
 public:

  /**
   * A mobile roaming network.
   */
  networkTypeMobileRoaming();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1435199760;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A Wi-Fi network.
 */
class networkTypeWiFi final : public NetworkType {
 public:

  /**
   * A Wi-Fi network.
   */
  networkTypeWiFi();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -633872070;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A different network type (e.g., Ethernet network).
 */
class networkTypeOther final : public NetworkType {
 public:

  /**
   * A different network type (e.g., Ethernet network).
   */
  networkTypeOther();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1942128539;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains information about a notification.
 */
class notification final : public Object {
 public:
  /// Unique persistent identifier of this notification.
  int32 id_;
  /// Notification date.
  int32 date_;
  /// True, if the notification was initially silent.
  bool is_silent_;
  /// Notification type.
  object_ptr<NotificationType> type_;

  /**
   * Contains information about a notification.
   */
  notification();

  /**
   * Contains information about a notification.
   *
   * \param[in] id_ Unique persistent identifier of this notification.
   * \param[in] date_ Notification date.
   * \param[in] is_silent_ True, if the notification was initially silent.
   * \param[in] type_ Notification type.
   */
  notification(int32 id_, int32 date_, bool is_silent_, object_ptr<NotificationType> &&type_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 788743120;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Describes a group of notifications.
 */
class notificationGroup final : public Object {
 public:
  /// Unique persistent auto-incremented from 1 identifier of the notification group.
  int32 id_;
  /// Type of the group.
  object_ptr<NotificationGroupType> type_;
  /// Identifier of a chat to which all notifications in the group belong.
  int53 chat_id_;
  /// Total number of active notifications in the group.
  int32 total_count_;
  /// The list of active notifications.
  array<object_ptr<notification>> notifications_;

  /**
   * Describes a group of notifications.
   */
  notificationGroup();

  /**
   * Describes a group of notifications.
   *
   * \param[in] id_ Unique persistent auto-incremented from 1 identifier of the notification group.
   * \param[in] type_ Type of the group.
   * \param[in] chat_id_ Identifier of a chat to which all notifications in the group belong.
   * \param[in] total_count_ Total number of active notifications in the group.
   * \param[in] notifications_ The list of active notifications.
   */
  notificationGroup(int32 id_, object_ptr<NotificationGroupType> &&type_, int53 chat_id_, int32 total_count_, array<object_ptr<notification>> &&notifications_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 780691541;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Describes the type of notifications in a notification group.
 */
class NotificationGroupType: public Object {
 public:
};

/**
 * A group containing notifications of type notificationTypeNewMessage and notificationTypeNewPushMessage with ordinary unread messages.
 */
class notificationGroupTypeMessages final : public NotificationGroupType {
 public:

  /**
   * A group containing notifications of type notificationTypeNewMessage and notificationTypeNewPushMessage with ordinary unread messages.
   */
  notificationGroupTypeMessages();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1702481123;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A group containing notifications of type notificationTypeNewMessage and notificationTypeNewPushMessage with unread mentions of the current user, replies to their messages, or a pinned message.
 */
class notificationGroupTypeMentions final : public NotificationGroupType {
 public:

  /**
   * A group containing notifications of type notificationTypeNewMessage and notificationTypeNewPushMessage with unread mentions of the current user, replies to their messages, or a pinned message.
   */
  notificationGroupTypeMentions();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -2050324051;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A group containing a notification of type notificationTypeNewSecretChat.
 */
class notificationGroupTypeSecretChat final : public NotificationGroupType {
 public:

  /**
   * A group containing a notification of type notificationTypeNewSecretChat.
   */
  notificationGroupTypeSecretChat();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1390759476;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A group containing notifications of type notificationTypeNewCall.
 */
class notificationGroupTypeCalls final : public NotificationGroupType {
 public:

  /**
   * A group containing notifications of type notificationTypeNewCall.
   */
  notificationGroupTypeCalls();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1379123538;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Describes the types of chats to which notification settings are relevant.
 */
class NotificationSettingsScope: public Object {
 public:
};

/**
 * Notification settings applied to all private and secret chats when the corresponding chat setting has a default value.
 */
class notificationSettingsScopePrivateChats final : public NotificationSettingsScope {
 public:

  /**
   * Notification settings applied to all private and secret chats when the corresponding chat setting has a default value.
   */
  notificationSettingsScopePrivateChats();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 937446759;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Notification settings applied to all basic groups and supergroups when the corresponding chat setting has a default value.
 */
class notificationSettingsScopeGroupChats final : public NotificationSettingsScope {
 public:

  /**
   * Notification settings applied to all basic groups and supergroups when the corresponding chat setting has a default value.
   */
  notificationSettingsScopeGroupChats();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1212142067;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Notification settings applied to all channels when the corresponding chat setting has a default value.
 */
class notificationSettingsScopeChannelChats final : public NotificationSettingsScope {
 public:

  /**
   * Notification settings applied to all channels when the corresponding chat setting has a default value.
   */
  notificationSettingsScopeChannelChats();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 548013448;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Contains detailed information about a notification.
 */
class NotificationType: public Object {
 public:
};

/**
 * New message was received.
 */
class notificationTypeNewMessage final : public NotificationType {
 public:
  /// The message.
  object_ptr<message> message_;

  /**
   * New message was received.
   */
  notificationTypeNewMessage();

  /**
   * New message was received.
   *
   * \param[in] message_ The message.
   */
  explicit notificationTypeNewMessage(object_ptr<message> &&message_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1885935159;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * New secret chat was created.
 */
class notificationTypeNewSecretChat final : public NotificationType {
 public:

  /**
   * New secret chat was created.
   */
  notificationTypeNewSecretChat();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1198638768;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * New call was received.
 */
class notificationTypeNewCall final : public NotificationType {
 public:
  /// Call identifier.
  int32 call_id_;

  /**
   * New call was received.
   */
  notificationTypeNewCall();

  /**
   * New call was received.
   *
   * \param[in] call_id_ Call identifier.
   */
  explicit notificationTypeNewCall(int32 call_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1712734585;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * New message was received through a push notification.
 */
class notificationTypeNewPushMessage final : public NotificationType {
 public:
  /// The message identifier. The message will not be available in the chat history, but the ID can be used in viewMessages, or as reply_to_message_id.
  int53 message_id_;
  /// Identifier of the sender of the message. Corresponding user or chat may be inaccessible.
  object_ptr<MessageSender> sender_id_;
  /// Name of the sender.
  string sender_name_;
  /// True, if the message is outgoing.
  bool is_outgoing_;
  /// Push message content.
  object_ptr<PushMessageContent> content_;

  /**
   * New message was received through a push notification.
   */
  notificationTypeNewPushMessage();

  /**
   * New message was received through a push notification.
   *
   * \param[in] message_id_ The message identifier. The message will not be available in the chat history, but the ID can be used in viewMessages, or as reply_to_message_id.
   * \param[in] sender_id_ Identifier of the sender of the message. Corresponding user or chat may be inaccessible.
   * \param[in] sender_name_ Name of the sender.
   * \param[in] is_outgoing_ True, if the message is outgoing.
   * \param[in] content_ Push message content.
   */
  notificationTypeNewPushMessage(int53 message_id_, object_ptr<MessageSender> &&sender_id_, string const &sender_name_, bool is_outgoing_, object_ptr<PushMessageContent> &&content_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -711680462;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * An object of this type is returned on a successful function call for certain functions.
 */
class ok final : public Object {
 public:

  /**
   * An object of this type is returned on a successful function call for certain functions.
   */
  ok();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -722616727;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Represents the value of an option.
 */
class OptionValue: public Object {
 public:
};

/**
 * Represents a boolean option.
 */
class optionValueBoolean final : public OptionValue {
 public:
  /// The value of the option.
  bool value_;

  /**
   * Represents a boolean option.
   */
  optionValueBoolean();

  /**
   * Represents a boolean option.
   *
   * \param[in] value_ The value of the option.
   */
  explicit optionValueBoolean(bool value_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 63135518;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Represents an unknown option or an option which has a default value.
 */
class optionValueEmpty final : public OptionValue {
 public:

  /**
   * Represents an unknown option or an option which has a default value.
   */
  optionValueEmpty();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 918955155;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Represents an integer option.
 */
class optionValueInteger final : public OptionValue {
 public:
  /// The value of the option.
  int64 value_;

  /**
   * Represents an integer option.
   */
  optionValueInteger();

  /**
   * Represents an integer option.
   *
   * \param[in] value_ The value of the option.
   */
  explicit optionValueInteger(int64 value_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -186858780;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Represents a string option.
 */
class optionValueString final : public OptionValue {
 public:
  /// The value of the option.
  string value_;

  /**
   * Represents a string option.
   */
  optionValueString();

  /**
   * Represents a string option.
   *
   * \param[in] value_ The value of the option.
   */
  explicit optionValueString(string const &value_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 756248212;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Order information.
 */
class orderInfo final : public Object {
 public:
  /// Name of the user.
  string name_;
  /// Phone number of the user.
  string phone_number_;
  /// Email address of the user.
  string email_address_;
  /// Shipping address for this order; may be null.
  object_ptr<address> shipping_address_;

  /**
   * Order information.
   */
  orderInfo();

  /**
   * Order information.
   *
   * \param[in] name_ Name of the user.
   * \param[in] phone_number_ Phone number of the user.
   * \param[in] email_address_ Email address of the user.
   * \param[in] shipping_address_ Shipping address for this order; may be null.
   */
  orderInfo(string const &name_, string const &phone_number_, string const &email_address_, object_ptr<address> &&shipping_address_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 783997294;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Describes a block of an instant view web page.
 */
class PageBlock: public Object {
 public:
};

/**
 * The title of a page.
 */
class pageBlockTitle final : public PageBlock {
 public:
  /// Title.
  object_ptr<RichText> title_;

  /**
   * The title of a page.
   */
  pageBlockTitle();

  /**
   * The title of a page.
   *
   * \param[in] title_ Title.
   */
  explicit pageBlockTitle(object_ptr<RichText> &&title_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1629664784;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The subtitle of a page.
 */
class pageBlockSubtitle final : public PageBlock {
 public:
  /// Subtitle.
  object_ptr<RichText> subtitle_;

  /**
   * The subtitle of a page.
   */
  pageBlockSubtitle();

  /**
   * The subtitle of a page.
   *
   * \param[in] subtitle_ Subtitle.
   */
  explicit pageBlockSubtitle(object_ptr<RichText> &&subtitle_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 264524263;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The author and publishing date of a page.
 */
class pageBlockAuthorDate final : public PageBlock {
 public:
  /// Author.
  object_ptr<RichText> author_;
  /// Point in time (Unix timestamp) when the article was published; 0 if unknown.
  int32 publish_date_;

  /**
   * The author and publishing date of a page.
   */
  pageBlockAuthorDate();

  /**
   * The author and publishing date of a page.
   *
   * \param[in] author_ Author.
   * \param[in] publish_date_ Point in time (Unix timestamp) when the article was published; 0 if unknown.
   */
  pageBlockAuthorDate(object_ptr<RichText> &&author_, int32 publish_date_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1300231184;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A header.
 */
class pageBlockHeader final : public PageBlock {
 public:
  /// Header.
  object_ptr<RichText> header_;

  /**
   * A header.
   */
  pageBlockHeader();

  /**
   * A header.
   *
   * \param[in] header_ Header.
   */
  explicit pageBlockHeader(object_ptr<RichText> &&header_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1402854811;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A subheader.
 */
class pageBlockSubheader final : public PageBlock {
 public:
  /// Subheader.
  object_ptr<RichText> subheader_;

  /**
   * A subheader.
   */
  pageBlockSubheader();

  /**
   * A subheader.
   *
   * \param[in] subheader_ Subheader.
   */
  explicit pageBlockSubheader(object_ptr<RichText> &&subheader_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1263956774;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A kicker.
 */
class pageBlockKicker final : public PageBlock {
 public:
  /// Kicker.
  object_ptr<RichText> kicker_;

  /**
   * A kicker.
   */
  pageBlockKicker();

  /**
   * A kicker.
   *
   * \param[in] kicker_ Kicker.
   */
  explicit pageBlockKicker(object_ptr<RichText> &&kicker_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1361282635;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A text paragraph.
 */
class pageBlockParagraph final : public PageBlock {
 public:
  /// Paragraph text.
  object_ptr<RichText> text_;

  /**
   * A text paragraph.
   */
  pageBlockParagraph();

  /**
   * A text paragraph.
   *
   * \param[in] text_ Paragraph text.
   */
  explicit pageBlockParagraph(object_ptr<RichText> &&text_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1182402406;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A preformatted text paragraph.
 */
class pageBlockPreformatted final : public PageBlock {
 public:
  /// Paragraph text.
  object_ptr<RichText> text_;
  /// Programming language for which the text needs to be formatted.
  string language_;

  /**
   * A preformatted text paragraph.
   */
  pageBlockPreformatted();

  /**
   * A preformatted text paragraph.
   *
   * \param[in] text_ Paragraph text.
   * \param[in] language_ Programming language for which the text needs to be formatted.
   */
  pageBlockPreformatted(object_ptr<RichText> &&text_, string const &language_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1066346178;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The footer of a page.
 */
class pageBlockFooter final : public PageBlock {
 public:
  /// Footer.
  object_ptr<RichText> footer_;

  /**
   * The footer of a page.
   */
  pageBlockFooter();

  /**
   * The footer of a page.
   *
   * \param[in] footer_ Footer.
   */
  explicit pageBlockFooter(object_ptr<RichText> &&footer_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 886429480;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * An empty block separating a page.
 */
class pageBlockDivider final : public PageBlock {
 public:

  /**
   * An empty block separating a page.
   */
  pageBlockDivider();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -618614392;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * An invisible anchor on a page, which can be used in a URL to open the page from the specified anchor.
 */
class pageBlockAnchor final : public PageBlock {
 public:
  /// Name of the anchor.
  string name_;

  /**
   * An invisible anchor on a page, which can be used in a URL to open the page from the specified anchor.
   */
  pageBlockAnchor();

  /**
   * An invisible anchor on a page, which can be used in a URL to open the page from the specified anchor.
   *
   * \param[in] name_ Name of the anchor.
   */
  explicit pageBlockAnchor(string const &name_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -837994576;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A list of data blocks.
 */
class pageBlockList final : public PageBlock {
 public:
  /// The items of the list.
  array<object_ptr<pageBlockListItem>> items_;

  /**
   * A list of data blocks.
   */
  pageBlockList();

  /**
   * A list of data blocks.
   *
   * \param[in] items_ The items of the list.
   */
  explicit pageBlockList(array<object_ptr<pageBlockListItem>> &&items_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1037074852;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A block quote.
 */
class pageBlockBlockQuote final : public PageBlock {
 public:
  /// Quote text.
  object_ptr<RichText> text_;
  /// Quote credit.
  object_ptr<RichText> credit_;

  /**
   * A block quote.
   */
  pageBlockBlockQuote();

  /**
   * A block quote.
   *
   * \param[in] text_ Quote text.
   * \param[in] credit_ Quote credit.
   */
  pageBlockBlockQuote(object_ptr<RichText> &&text_, object_ptr<RichText> &&credit_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1657834142;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A pull quote.
 */
class pageBlockPullQuote final : public PageBlock {
 public:
  /// Quote text.
  object_ptr<RichText> text_;
  /// Quote credit.
  object_ptr<RichText> credit_;

  /**
   * A pull quote.
   */
  pageBlockPullQuote();

  /**
   * A pull quote.
   *
   * \param[in] text_ Quote text.
   * \param[in] credit_ Quote credit.
   */
  pageBlockPullQuote(object_ptr<RichText> &&text_, object_ptr<RichText> &&credit_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 490242317;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * An animation.
 */
class pageBlockAnimation final : public PageBlock {
 public:
  /// Animation file; may be null.
  object_ptr<animation> animation_;
  /// Animation caption.
  object_ptr<pageBlockCaption> caption_;
  /// True, if the animation must be played automatically.
  bool need_autoplay_;

  /**
   * An animation.
   */
  pageBlockAnimation();

  /**
   * An animation.
   *
   * \param[in] animation_ Animation file; may be null.
   * \param[in] caption_ Animation caption.
   * \param[in] need_autoplay_ True, if the animation must be played automatically.
   */
  pageBlockAnimation(object_ptr<animation> &&animation_, object_ptr<pageBlockCaption> &&caption_, bool need_autoplay_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1355669513;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * An audio file.
 */
class pageBlockAudio final : public PageBlock {
 public:
  /// Audio file; may be null.
  object_ptr<audio> audio_;
  /// Audio file caption.
  object_ptr<pageBlockCaption> caption_;

  /**
   * An audio file.
   */
  pageBlockAudio();

  /**
   * An audio file.
   *
   * \param[in] audio_ Audio file; may be null.
   * \param[in] caption_ Audio file caption.
   */
  pageBlockAudio(object_ptr<audio> &&audio_, object_ptr<pageBlockCaption> &&caption_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -63371245;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A photo.
 */
class pageBlockPhoto final : public PageBlock {
 public:
  /// Photo file; may be null.
  object_ptr<photo> photo_;
  /// Photo caption.
  object_ptr<pageBlockCaption> caption_;
  /// URL that needs to be opened when the photo is clicked.
  string url_;

  /**
   * A photo.
   */
  pageBlockPhoto();

  /**
   * A photo.
   *
   * \param[in] photo_ Photo file; may be null.
   * \param[in] caption_ Photo caption.
   * \param[in] url_ URL that needs to be opened when the photo is clicked.
   */
  pageBlockPhoto(object_ptr<photo> &&photo_, object_ptr<pageBlockCaption> &&caption_, string const &url_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 417601156;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A video.
 */
class pageBlockVideo final : public PageBlock {
 public:
  /// Video file; may be null.
  object_ptr<video> video_;
  /// Video caption.
  object_ptr<pageBlockCaption> caption_;
  /// True, if the video must be played automatically.
  bool need_autoplay_;
  /// True, if the video must be looped.
  bool is_looped_;

  /**
   * A video.
   */
  pageBlockVideo();

  /**
   * A video.
   *
   * \param[in] video_ Video file; may be null.
   * \param[in] caption_ Video caption.
   * \param[in] need_autoplay_ True, if the video must be played automatically.
   * \param[in] is_looped_ True, if the video must be looped.
   */
  pageBlockVideo(object_ptr<video> &&video_, object_ptr<pageBlockCaption> &&caption_, bool need_autoplay_, bool is_looped_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 510041394;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A voice note.
 */
class pageBlockVoiceNote final : public PageBlock {
 public:
  /// Voice note; may be null.
  object_ptr<voiceNote> voice_note_;
  /// Voice note caption.
  object_ptr<pageBlockCaption> caption_;

  /**
   * A voice note.
   */
  pageBlockVoiceNote();

  /**
   * A voice note.
   *
   * \param[in] voice_note_ Voice note; may be null.
   * \param[in] caption_ Voice note caption.
   */
  pageBlockVoiceNote(object_ptr<voiceNote> &&voice_note_, object_ptr<pageBlockCaption> &&caption_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1823310463;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A page cover.
 */
class pageBlockCover final : public PageBlock {
 public:
  /// Cover.
  object_ptr<PageBlock> cover_;

  /**
   * A page cover.
   */
  pageBlockCover();

  /**
   * A page cover.
   *
   * \param[in] cover_ Cover.
   */
  explicit pageBlockCover(object_ptr<PageBlock> &&cover_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 972174080;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * An embedded web page.
 */
class pageBlockEmbedded final : public PageBlock {
 public:
  /// Web page URL, if available.
  string url_;
  /// HTML-markup of the embedded page.
  string html_;
  /// Poster photo, if available; may be null.
  object_ptr<photo> poster_photo_;
  /// Block width; 0 if unknown.
  int32 width_;
  /// Block height; 0 if unknown.
  int32 height_;
  /// Block caption.
  object_ptr<pageBlockCaption> caption_;
  /// True, if the block must be full width.
  bool is_full_width_;
  /// True, if scrolling needs to be allowed.
  bool allow_scrolling_;

  /**
   * An embedded web page.
   */
  pageBlockEmbedded();

  /**
   * An embedded web page.
   *
   * \param[in] url_ Web page URL, if available.
   * \param[in] html_ HTML-markup of the embedded page.
   * \param[in] poster_photo_ Poster photo, if available; may be null.
   * \param[in] width_ Block width; 0 if unknown.
   * \param[in] height_ Block height; 0 if unknown.
   * \param[in] caption_ Block caption.
   * \param[in] is_full_width_ True, if the block must be full width.
   * \param[in] allow_scrolling_ True, if scrolling needs to be allowed.
   */
  pageBlockEmbedded(string const &url_, string const &html_, object_ptr<photo> &&poster_photo_, int32 width_, int32 height_, object_ptr<pageBlockCaption> &&caption_, bool is_full_width_, bool allow_scrolling_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1942577763;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * An embedded post.
 */
class pageBlockEmbeddedPost final : public PageBlock {
 public:
  /// Web page URL.
  string url_;
  /// Post author.
  string author_;
  /// Post author photo; may be null.
  object_ptr<photo> author_photo_;
  /// Point in time (Unix timestamp) when the post was created; 0 if unknown.
  int32 date_;
  /// Post content.
  array<object_ptr<PageBlock>> page_blocks_;
  /// Post caption.
  object_ptr<pageBlockCaption> caption_;

  /**
   * An embedded post.
   */
  pageBlockEmbeddedPost();

  /**
   * An embedded post.
   *
   * \param[in] url_ Web page URL.
   * \param[in] author_ Post author.
   * \param[in] author_photo_ Post author photo; may be null.
   * \param[in] date_ Point in time (Unix timestamp) when the post was created; 0 if unknown.
   * \param[in] page_blocks_ Post content.
   * \param[in] caption_ Post caption.
   */
  pageBlockEmbeddedPost(string const &url_, string const &author_, object_ptr<photo> &&author_photo_, int32 date_, array<object_ptr<PageBlock>> &&page_blocks_, object_ptr<pageBlockCaption> &&caption_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 397600949;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A collage.
 */
class pageBlockCollage final : public PageBlock {
 public:
  /// Collage item contents.
  array<object_ptr<PageBlock>> page_blocks_;
  /// Block caption.
  object_ptr<pageBlockCaption> caption_;

  /**
   * A collage.
   */
  pageBlockCollage();

  /**
   * A collage.
   *
   * \param[in] page_blocks_ Collage item contents.
   * \param[in] caption_ Block caption.
   */
  pageBlockCollage(array<object_ptr<PageBlock>> &&page_blocks_, object_ptr<pageBlockCaption> &&caption_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1163760110;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A slideshow.
 */
class pageBlockSlideshow final : public PageBlock {
 public:
  /// Slideshow item contents.
  array<object_ptr<PageBlock>> page_blocks_;
  /// Block caption.
  object_ptr<pageBlockCaption> caption_;

  /**
   * A slideshow.
   */
  pageBlockSlideshow();

  /**
   * A slideshow.
   *
   * \param[in] page_blocks_ Slideshow item contents.
   * \param[in] caption_ Block caption.
   */
  pageBlockSlideshow(array<object_ptr<PageBlock>> &&page_blocks_, object_ptr<pageBlockCaption> &&caption_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 539217375;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A link to a chat.
 */
class pageBlockChatLink final : public PageBlock {
 public:
  /// Chat title.
  string title_;
  /// Chat photo; may be null.
  object_ptr<chatPhotoInfo> photo_;
  /// Chat username, by which all other information about the chat can be resolved.
  string username_;

  /**
   * A link to a chat.
   */
  pageBlockChatLink();

  /**
   * A link to a chat.
   *
   * \param[in] title_ Chat title.
   * \param[in] photo_ Chat photo; may be null.
   * \param[in] username_ Chat username, by which all other information about the chat can be resolved.
   */
  pageBlockChatLink(string const &title_, object_ptr<chatPhotoInfo> &&photo_, string const &username_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -202091253;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A table.
 */
class pageBlockTable final : public PageBlock {
 public:
  /// Table caption.
  object_ptr<RichText> caption_;
  /// Table cells.
  array<array<object_ptr<pageBlockTableCell>>> cells_;
  /// True, if the table is bordered.
  bool is_bordered_;
  /// True, if the table is striped.
  bool is_striped_;

  /**
   * A table.
   */
  pageBlockTable();

  /**
   * A table.
   *
   * \param[in] caption_ Table caption.
   * \param[in] cells_ Table cells.
   * \param[in] is_bordered_ True, if the table is bordered.
   * \param[in] is_striped_ True, if the table is striped.
   */
  pageBlockTable(object_ptr<RichText> &&caption_, array<array<object_ptr<pageBlockTableCell>>> &&cells_, bool is_bordered_, bool is_striped_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -942649288;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A collapsible block.
 */
class pageBlockDetails final : public PageBlock {
 public:
  /// Always visible heading for the block.
  object_ptr<RichText> header_;
  /// Block contents.
  array<object_ptr<PageBlock>> page_blocks_;
  /// True, if the block is open by default.
  bool is_open_;

  /**
   * A collapsible block.
   */
  pageBlockDetails();

  /**
   * A collapsible block.
   *
   * \param[in] header_ Always visible heading for the block.
   * \param[in] page_blocks_ Block contents.
   * \param[in] is_open_ True, if the block is open by default.
   */
  pageBlockDetails(object_ptr<RichText> &&header_, array<object_ptr<PageBlock>> &&page_blocks_, bool is_open_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1599869809;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Related articles.
 */
class pageBlockRelatedArticles final : public PageBlock {
 public:
  /// Block header.
  object_ptr<RichText> header_;
  /// List of related articles.
  array<object_ptr<pageBlockRelatedArticle>> articles_;

  /**
   * Related articles.
   */
  pageBlockRelatedArticles();

  /**
   * Related articles.
   *
   * \param[in] header_ Block header.
   * \param[in] articles_ List of related articles.
   */
  pageBlockRelatedArticles(object_ptr<RichText> &&header_, array<object_ptr<pageBlockRelatedArticle>> &&articles_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1807324374;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A map.
 */
class pageBlockMap final : public PageBlock {
 public:
  /// Location of the map center.
  object_ptr<location> location_;
  /// Map zoom level.
  int32 zoom_;
  /// Map width.
  int32 width_;
  /// Map height.
  int32 height_;
  /// Block caption.
  object_ptr<pageBlockCaption> caption_;

  /**
   * A map.
   */
  pageBlockMap();

  /**
   * A map.
   *
   * \param[in] location_ Location of the map center.
   * \param[in] zoom_ Map zoom level.
   * \param[in] width_ Map width.
   * \param[in] height_ Map height.
   * \param[in] caption_ Block caption.
   */
  pageBlockMap(object_ptr<location> &&location_, int32 zoom_, int32 width_, int32 height_, object_ptr<pageBlockCaption> &&caption_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1510961171;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains a caption of an instant view web page block, consisting of a text and a trailing credit.
 */
class pageBlockCaption final : public Object {
 public:
  /// Content of the caption.
  object_ptr<RichText> text_;
  /// Block credit (like HTML tag &lt;cite&gt;).
  object_ptr<RichText> credit_;

  /**
   * Contains a caption of an instant view web page block, consisting of a text and a trailing credit.
   */
  pageBlockCaption();

  /**
   * Contains a caption of an instant view web page block, consisting of a text and a trailing credit.
   *
   * \param[in] text_ Content of the caption.
   * \param[in] credit_ Block credit (like HTML tag &lt;cite&gt;).
   */
  pageBlockCaption(object_ptr<RichText> &&text_, object_ptr<RichText> &&credit_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1180064650;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Describes a horizontal alignment of a table cell content.
 */
class PageBlockHorizontalAlignment: public Object {
 public:
};

/**
 * The content must be left-aligned.
 */
class pageBlockHorizontalAlignmentLeft final : public PageBlockHorizontalAlignment {
 public:

  /**
   * The content must be left-aligned.
   */
  pageBlockHorizontalAlignmentLeft();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 848701417;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The content must be center-aligned.
 */
class pageBlockHorizontalAlignmentCenter final : public PageBlockHorizontalAlignment {
 public:

  /**
   * The content must be center-aligned.
   */
  pageBlockHorizontalAlignmentCenter();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1009203990;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The content must be right-aligned.
 */
class pageBlockHorizontalAlignmentRight final : public PageBlockHorizontalAlignment {
 public:

  /**
   * The content must be right-aligned.
   */
  pageBlockHorizontalAlignmentRight();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1371369214;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Describes an item of a list page block.
 */
class pageBlockListItem final : public Object {
 public:
  /// Item label.
  string label_;
  /// Item blocks.
  array<object_ptr<PageBlock>> page_blocks_;

  /**
   * Describes an item of a list page block.
   */
  pageBlockListItem();

  /**
   * Describes an item of a list page block.
   *
   * \param[in] label_ Item label.
   * \param[in] page_blocks_ Item blocks.
   */
  pageBlockListItem(string const &label_, array<object_ptr<PageBlock>> &&page_blocks_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 323186259;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains information about a related article.
 */
class pageBlockRelatedArticle final : public Object {
 public:
  /// Related article URL.
  string url_;
  /// Article title; may be empty.
  string title_;
  /// Article description; may be empty.
  string description_;
  /// Article photo; may be null.
  object_ptr<photo> photo_;
  /// Article author; may be empty.
  string author_;
  /// Point in time (Unix timestamp) when the article was published; 0 if unknown.
  int32 publish_date_;

  /**
   * Contains information about a related article.
   */
  pageBlockRelatedArticle();

  /**
   * Contains information about a related article.
   *
   * \param[in] url_ Related article URL.
   * \param[in] title_ Article title; may be empty.
   * \param[in] description_ Article description; may be empty.
   * \param[in] photo_ Article photo; may be null.
   * \param[in] author_ Article author; may be empty.
   * \param[in] publish_date_ Point in time (Unix timestamp) when the article was published; 0 if unknown.
   */
  pageBlockRelatedArticle(string const &url_, string const &title_, string const &description_, object_ptr<photo> &&photo_, string const &author_, int32 publish_date_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 481199251;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Represents a cell of a table.
 */
class pageBlockTableCell final : public Object {
 public:
  /// Cell text; may be null. If the text is null, then the cell must be invisible.
  object_ptr<RichText> text_;
  /// True, if it is a header cell.
  bool is_header_;
  /// The number of columns the cell spans.
  int32 colspan_;
  /// The number of rows the cell spans.
  int32 rowspan_;
  /// Horizontal cell content alignment.
  object_ptr<PageBlockHorizontalAlignment> align_;
  /// Vertical cell content alignment.
  object_ptr<PageBlockVerticalAlignment> valign_;

  /**
   * Represents a cell of a table.
   */
  pageBlockTableCell();

  /**
   * Represents a cell of a table.
   *
   * \param[in] text_ Cell text; may be null. If the text is null, then the cell must be invisible.
   * \param[in] is_header_ True, if it is a header cell.
   * \param[in] colspan_ The number of columns the cell spans.
   * \param[in] rowspan_ The number of rows the cell spans.
   * \param[in] align_ Horizontal cell content alignment.
   * \param[in] valign_ Vertical cell content alignment.
   */
  pageBlockTableCell(object_ptr<RichText> &&text_, bool is_header_, int32 colspan_, int32 rowspan_, object_ptr<PageBlockHorizontalAlignment> &&align_, object_ptr<PageBlockVerticalAlignment> &&valign_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1417658214;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Describes a Vertical alignment of a table cell content.
 */
class PageBlockVerticalAlignment: public Object {
 public:
};

/**
 * The content must be top-aligned.
 */
class pageBlockVerticalAlignmentTop final : public PageBlockVerticalAlignment {
 public:

  /**
   * The content must be top-aligned.
   */
  pageBlockVerticalAlignmentTop();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 195500454;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The content must be middle-aligned.
 */
class pageBlockVerticalAlignmentMiddle final : public PageBlockVerticalAlignment {
 public:

  /**
   * The content must be middle-aligned.
   */
  pageBlockVerticalAlignmentMiddle();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -2123096587;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The content must be bottom-aligned.
 */
class pageBlockVerticalAlignmentBottom final : public PageBlockVerticalAlignment {
 public:

  /**
   * The content must be bottom-aligned.
   */
  pageBlockVerticalAlignmentBottom();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 2092531158;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains information about a Telegram Passport authorization form that was requested.
 */
class passportAuthorizationForm final : public Object {
 public:
  /// Unique identifier of the authorization form.
  int32 id_;
  /// Telegram Passport elements that must be provided to complete the form.
  array<object_ptr<passportRequiredElement>> required_elements_;
  /// URL for the privacy policy of the service; may be empty.
  string privacy_policy_url_;

  /**
   * Contains information about a Telegram Passport authorization form that was requested.
   */
  passportAuthorizationForm();

  /**
   * Contains information about a Telegram Passport authorization form that was requested.
   *
   * \param[in] id_ Unique identifier of the authorization form.
   * \param[in] required_elements_ Telegram Passport elements that must be provided to complete the form.
   * \param[in] privacy_policy_url_ URL for the privacy policy of the service; may be empty.
   */
  passportAuthorizationForm(int32 id_, array<object_ptr<passportRequiredElement>> &&required_elements_, string const &privacy_policy_url_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1070673218;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Contains information about a Telegram Passport element.
 */
class PassportElement: public Object {
 public:
};

/**
 * A Telegram Passport element containing the user&\#039;s personal details.
 */
class passportElementPersonalDetails final : public PassportElement {
 public:
  /// Personal details of the user.
  object_ptr<personalDetails> personal_details_;

  /**
   * A Telegram Passport element containing the user&\#039;s personal details.
   */
  passportElementPersonalDetails();

  /**
   * A Telegram Passport element containing the user&\#039;s personal details.
   *
   * \param[in] personal_details_ Personal details of the user.
   */
  explicit passportElementPersonalDetails(object_ptr<personalDetails> &&personal_details_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1217724035;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A Telegram Passport element containing the user&\#039;s passport.
 */
class passportElementPassport final : public PassportElement {
 public:
  /// Passport.
  object_ptr<identityDocument> passport_;

  /**
   * A Telegram Passport element containing the user&\#039;s passport.
   */
  passportElementPassport();

  /**
   * A Telegram Passport element containing the user&\#039;s passport.
   *
   * \param[in] passport_ Passport.
   */
  explicit passportElementPassport(object_ptr<identityDocument> &&passport_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -263985373;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A Telegram Passport element containing the user&\#039;s driver license.
 */
class passportElementDriverLicense final : public PassportElement {
 public:
  /// Driver license.
  object_ptr<identityDocument> driver_license_;

  /**
   * A Telegram Passport element containing the user&\#039;s driver license.
   */
  passportElementDriverLicense();

  /**
   * A Telegram Passport element containing the user&\#039;s driver license.
   *
   * \param[in] driver_license_ Driver license.
   */
  explicit passportElementDriverLicense(object_ptr<identityDocument> &&driver_license_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1643580589;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A Telegram Passport element containing the user&\#039;s identity card.
 */
class passportElementIdentityCard final : public PassportElement {
 public:
  /// Identity card.
  object_ptr<identityDocument> identity_card_;

  /**
   * A Telegram Passport element containing the user&\#039;s identity card.
   */
  passportElementIdentityCard();

  /**
   * A Telegram Passport element containing the user&\#039;s identity card.
   *
   * \param[in] identity_card_ Identity card.
   */
  explicit passportElementIdentityCard(object_ptr<identityDocument> &&identity_card_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 2083775797;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A Telegram Passport element containing the user&\#039;s internal passport.
 */
class passportElementInternalPassport final : public PassportElement {
 public:
  /// Internal passport.
  object_ptr<identityDocument> internal_passport_;

  /**
   * A Telegram Passport element containing the user&\#039;s internal passport.
   */
  passportElementInternalPassport();

  /**
   * A Telegram Passport element containing the user&\#039;s internal passport.
   *
   * \param[in] internal_passport_ Internal passport.
   */
  explicit passportElementInternalPassport(object_ptr<identityDocument> &&internal_passport_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 36220295;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A Telegram Passport element containing the user&\#039;s address.
 */
class passportElementAddress final : public PassportElement {
 public:
  /// Address.
  object_ptr<address> address_;

  /**
   * A Telegram Passport element containing the user&\#039;s address.
   */
  passportElementAddress();

  /**
   * A Telegram Passport element containing the user&\#039;s address.
   *
   * \param[in] address_ Address.
   */
  explicit passportElementAddress(object_ptr<address> &&address_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -782625232;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A Telegram Passport element containing the user&\#039;s utility bill.
 */
class passportElementUtilityBill final : public PassportElement {
 public:
  /// Utility bill.
  object_ptr<personalDocument> utility_bill_;

  /**
   * A Telegram Passport element containing the user&\#039;s utility bill.
   */
  passportElementUtilityBill();

  /**
   * A Telegram Passport element containing the user&\#039;s utility bill.
   *
   * \param[in] utility_bill_ Utility bill.
   */
  explicit passportElementUtilityBill(object_ptr<personalDocument> &&utility_bill_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -234611246;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A Telegram Passport element containing the user&\#039;s bank statement.
 */
class passportElementBankStatement final : public PassportElement {
 public:
  /// Bank statement.
  object_ptr<personalDocument> bank_statement_;

  /**
   * A Telegram Passport element containing the user&\#039;s bank statement.
   */
  passportElementBankStatement();

  /**
   * A Telegram Passport element containing the user&\#039;s bank statement.
   *
   * \param[in] bank_statement_ Bank statement.
   */
  explicit passportElementBankStatement(object_ptr<personalDocument> &&bank_statement_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -366464408;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A Telegram Passport element containing the user&\#039;s rental agreement.
 */
class passportElementRentalAgreement final : public PassportElement {
 public:
  /// Rental agreement.
  object_ptr<personalDocument> rental_agreement_;

  /**
   * A Telegram Passport element containing the user&\#039;s rental agreement.
   */
  passportElementRentalAgreement();

  /**
   * A Telegram Passport element containing the user&\#039;s rental agreement.
   *
   * \param[in] rental_agreement_ Rental agreement.
   */
  explicit passportElementRentalAgreement(object_ptr<personalDocument> &&rental_agreement_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -290141400;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A Telegram Passport element containing the user&\#039;s passport registration pages.
 */
class passportElementPassportRegistration final : public PassportElement {
 public:
  /// Passport registration pages.
  object_ptr<personalDocument> passport_registration_;

  /**
   * A Telegram Passport element containing the user&\#039;s passport registration pages.
   */
  passportElementPassportRegistration();

  /**
   * A Telegram Passport element containing the user&\#039;s passport registration pages.
   *
   * \param[in] passport_registration_ Passport registration pages.
   */
  explicit passportElementPassportRegistration(object_ptr<personalDocument> &&passport_registration_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 618323071;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A Telegram Passport element containing the user&\#039;s temporary registration.
 */
class passportElementTemporaryRegistration final : public PassportElement {
 public:
  /// Temporary registration.
  object_ptr<personalDocument> temporary_registration_;

  /**
   * A Telegram Passport element containing the user&\#039;s temporary registration.
   */
  passportElementTemporaryRegistration();

  /**
   * A Telegram Passport element containing the user&\#039;s temporary registration.
   *
   * \param[in] temporary_registration_ Temporary registration.
   */
  explicit passportElementTemporaryRegistration(object_ptr<personalDocument> &&temporary_registration_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1237626864;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A Telegram Passport element containing the user&\#039;s phone number.
 */
class passportElementPhoneNumber final : public PassportElement {
 public:
  /// Phone number.
  string phone_number_;

  /**
   * A Telegram Passport element containing the user&\#039;s phone number.
   */
  passportElementPhoneNumber();

  /**
   * A Telegram Passport element containing the user&\#039;s phone number.
   *
   * \param[in] phone_number_ Phone number.
   */
  explicit passportElementPhoneNumber(string const &phone_number_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1320118375;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A Telegram Passport element containing the user&\#039;s email address.
 */
class passportElementEmailAddress final : public PassportElement {
 public:
  /// Email address.
  string email_address_;

  /**
   * A Telegram Passport element containing the user&\#039;s email address.
   */
  passportElementEmailAddress();

  /**
   * A Telegram Passport element containing the user&\#039;s email address.
   *
   * \param[in] email_address_ Email address.
   */
  explicit passportElementEmailAddress(string const &email_address_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1528129531;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains the description of an error in a Telegram Passport element.
 */
class passportElementError final : public Object {
 public:
  /// Type of the Telegram Passport element which has the error.
  object_ptr<PassportElementType> type_;
  /// Error message.
  string message_;
  /// Error source.
  object_ptr<PassportElementErrorSource> source_;

  /**
   * Contains the description of an error in a Telegram Passport element.
   */
  passportElementError();

  /**
   * Contains the description of an error in a Telegram Passport element.
   *
   * \param[in] type_ Type of the Telegram Passport element which has the error.
   * \param[in] message_ Error message.
   * \param[in] source_ Error source.
   */
  passportElementError(object_ptr<PassportElementType> &&type_, string const &message_, object_ptr<PassportElementErrorSource> &&source_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1861902395;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Contains the description of an error in a Telegram Passport element.
 */
class PassportElementErrorSource: public Object {
 public:
};

/**
 * The element contains an error in an unspecified place. The error will be considered resolved when new data is added.
 */
class passportElementErrorSourceUnspecified final : public PassportElementErrorSource {
 public:

  /**
   * The element contains an error in an unspecified place. The error will be considered resolved when new data is added.
   */
  passportElementErrorSourceUnspecified();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -378320830;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * One of the data fields contains an error. The error will be considered resolved when the value of the field changes.
 */
class passportElementErrorSourceDataField final : public PassportElementErrorSource {
 public:
  /// Field name.
  string field_name_;

  /**
   * One of the data fields contains an error. The error will be considered resolved when the value of the field changes.
   */
  passportElementErrorSourceDataField();

  /**
   * One of the data fields contains an error. The error will be considered resolved when the value of the field changes.
   *
   * \param[in] field_name_ Field name.
   */
  explicit passportElementErrorSourceDataField(string const &field_name_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -308650776;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The front side of the document contains an error. The error will be considered resolved when the file with the front side changes.
 */
class passportElementErrorSourceFrontSide final : public PassportElementErrorSource {
 public:

  /**
   * The front side of the document contains an error. The error will be considered resolved when the file with the front side changes.
   */
  passportElementErrorSourceFrontSide();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1895658292;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The reverse side of the document contains an error. The error will be considered resolved when the file with the reverse side changes.
 */
class passportElementErrorSourceReverseSide final : public PassportElementErrorSource {
 public:

  /**
   * The reverse side of the document contains an error. The error will be considered resolved when the file with the reverse side changes.
   */
  passportElementErrorSourceReverseSide();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1918630391;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The selfie with the document contains an error. The error will be considered resolved when the file with the selfie changes.
 */
class passportElementErrorSourceSelfie final : public PassportElementErrorSource {
 public:

  /**
   * The selfie with the document contains an error. The error will be considered resolved when the file with the selfie changes.
   */
  passportElementErrorSourceSelfie();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -797043672;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * One of files with the translation of the document contains an error. The error will be considered resolved when the file changes.
 */
class passportElementErrorSourceTranslationFile final : public PassportElementErrorSource {
 public:
  /// Index of a file with the error.
  int32 file_index_;

  /**
   * One of files with the translation of the document contains an error. The error will be considered resolved when the file changes.
   */
  passportElementErrorSourceTranslationFile();

  /**
   * One of files with the translation of the document contains an error. The error will be considered resolved when the file changes.
   *
   * \param[in] file_index_ Index of a file with the error.
   */
  explicit passportElementErrorSourceTranslationFile(int32 file_index_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -689621228;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The translation of the document contains an error. The error will be considered resolved when the list of translation files changes.
 */
class passportElementErrorSourceTranslationFiles final : public PassportElementErrorSource {
 public:

  /**
   * The translation of the document contains an error. The error will be considered resolved when the list of translation files changes.
   */
  passportElementErrorSourceTranslationFiles();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 581280796;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The file contains an error. The error will be considered resolved when the file changes.
 */
class passportElementErrorSourceFile final : public PassportElementErrorSource {
 public:
  /// Index of a file with the error.
  int32 file_index_;

  /**
   * The file contains an error. The error will be considered resolved when the file changes.
   */
  passportElementErrorSourceFile();

  /**
   * The file contains an error. The error will be considered resolved when the file changes.
   *
   * \param[in] file_index_ Index of a file with the error.
   */
  explicit passportElementErrorSourceFile(int32 file_index_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 2020358960;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The list of attached files contains an error. The error will be considered resolved when the list of files changes.
 */
class passportElementErrorSourceFiles final : public PassportElementErrorSource {
 public:

  /**
   * The list of attached files contains an error. The error will be considered resolved when the list of files changes.
   */
  passportElementErrorSourceFiles();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1894164178;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Contains the type of a Telegram Passport element.
 */
class PassportElementType: public Object {
 public:
};

/**
 * A Telegram Passport element containing the user&\#039;s personal details.
 */
class passportElementTypePersonalDetails final : public PassportElementType {
 public:

  /**
   * A Telegram Passport element containing the user&\#039;s personal details.
   */
  passportElementTypePersonalDetails();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1032136365;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A Telegram Passport element containing the user&\#039;s passport.
 */
class passportElementTypePassport final : public PassportElementType {
 public:

  /**
   * A Telegram Passport element containing the user&\#039;s passport.
   */
  passportElementTypePassport();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -436360376;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A Telegram Passport element containing the user&\#039;s driver license.
 */
class passportElementTypeDriverLicense final : public PassportElementType {
 public:

  /**
   * A Telegram Passport element containing the user&\#039;s driver license.
   */
  passportElementTypeDriverLicense();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1827298379;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A Telegram Passport element containing the user&\#039;s identity card.
 */
class passportElementTypeIdentityCard final : public PassportElementType {
 public:

  /**
   * A Telegram Passport element containing the user&\#039;s identity card.
   */
  passportElementTypeIdentityCard();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -502356132;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A Telegram Passport element containing the user&\#039;s internal passport.
 */
class passportElementTypeInternalPassport final : public PassportElementType {
 public:

  /**
   * A Telegram Passport element containing the user&\#039;s internal passport.
   */
  passportElementTypeInternalPassport();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -793781959;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A Telegram Passport element containing the user&\#039;s address.
 */
class passportElementTypeAddress final : public PassportElementType {
 public:

  /**
   * A Telegram Passport element containing the user&\#039;s address.
   */
  passportElementTypeAddress();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 496327874;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A Telegram Passport element containing the user&\#039;s utility bill.
 */
class passportElementTypeUtilityBill final : public PassportElementType {
 public:

  /**
   * A Telegram Passport element containing the user&\#039;s utility bill.
   */
  passportElementTypeUtilityBill();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 627084906;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A Telegram Passport element containing the user&\#039;s bank statement.
 */
class passportElementTypeBankStatement final : public PassportElementType {
 public:

  /**
   * A Telegram Passport element containing the user&\#039;s bank statement.
   */
  passportElementTypeBankStatement();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 574095667;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A Telegram Passport element containing the user&\#039;s rental agreement.
 */
class passportElementTypeRentalAgreement final : public PassportElementType {
 public:

  /**
   * A Telegram Passport element containing the user&\#039;s rental agreement.
   */
  passportElementTypeRentalAgreement();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -2060583280;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A Telegram Passport element containing the registration page of the user&\#039;s passport.
 */
class passportElementTypePassportRegistration final : public PassportElementType {
 public:

  /**
   * A Telegram Passport element containing the registration page of the user&\#039;s passport.
   */
  passportElementTypePassportRegistration();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -159478209;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A Telegram Passport element containing the user&\#039;s temporary registration.
 */
class passportElementTypeTemporaryRegistration final : public PassportElementType {
 public:

  /**
   * A Telegram Passport element containing the user&\#039;s temporary registration.
   */
  passportElementTypeTemporaryRegistration();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1092498527;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A Telegram Passport element containing the user&\#039;s phone number.
 */
class passportElementTypePhoneNumber final : public PassportElementType {
 public:

  /**
   * A Telegram Passport element containing the user&\#039;s phone number.
   */
  passportElementTypePhoneNumber();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -995361172;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A Telegram Passport element containing the user&\#039;s email address.
 */
class passportElementTypeEmailAddress final : public PassportElementType {
 public:

  /**
   * A Telegram Passport element containing the user&\#039;s email address.
   */
  passportElementTypeEmailAddress();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -79321405;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains information about saved Telegram Passport elements.
 */
class passportElements final : public Object {
 public:
  /// Telegram Passport elements.
  array<object_ptr<PassportElement>> elements_;

  /**
   * Contains information about saved Telegram Passport elements.
   */
  passportElements();

  /**
   * Contains information about saved Telegram Passport elements.
   *
   * \param[in] elements_ Telegram Passport elements.
   */
  explicit passportElements(array<object_ptr<PassportElement>> &&elements_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1264617556;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains information about a Telegram Passport elements and corresponding errors.
 */
class passportElementsWithErrors final : public Object {
 public:
  /// Telegram Passport elements.
  array<object_ptr<PassportElement>> elements_;
  /// Errors in the elements that are already available.
  array<object_ptr<passportElementError>> errors_;

  /**
   * Contains information about a Telegram Passport elements and corresponding errors.
   */
  passportElementsWithErrors();

  /**
   * Contains information about a Telegram Passport elements and corresponding errors.
   *
   * \param[in] elements_ Telegram Passport elements.
   * \param[in] errors_ Errors in the elements that are already available.
   */
  passportElementsWithErrors(array<object_ptr<PassportElement>> &&elements_, array<object_ptr<passportElementError>> &&errors_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1308923044;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains a description of the required Telegram Passport element that was requested by a service.
 */
class passportRequiredElement final : public Object {
 public:
  /// List of Telegram Passport elements any of which is enough to provide.
  array<object_ptr<passportSuitableElement>> suitable_elements_;

  /**
   * Contains a description of the required Telegram Passport element that was requested by a service.
   */
  passportRequiredElement();

  /**
   * Contains a description of the required Telegram Passport element that was requested by a service.
   *
   * \param[in] suitable_elements_ List of Telegram Passport elements any of which is enough to provide.
   */
  explicit passportRequiredElement(array<object_ptr<passportSuitableElement>> &&suitable_elements_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1983641651;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains information about a Telegram Passport element that was requested by a service.
 */
class passportSuitableElement final : public Object {
 public:
  /// Type of the element.
  object_ptr<PassportElementType> type_;
  /// True, if a selfie is required with the identity document.
  bool is_selfie_required_;
  /// True, if a certified English translation is required with the document.
  bool is_translation_required_;
  /// True, if personal details must include the user&\#039;s name in the language of their country of residence.
  bool is_native_name_required_;

  /**
   * Contains information about a Telegram Passport element that was requested by a service.
   */
  passportSuitableElement();

  /**
   * Contains information about a Telegram Passport element that was requested by a service.
   *
   * \param[in] type_ Type of the element.
   * \param[in] is_selfie_required_ True, if a selfie is required with the identity document.
   * \param[in] is_translation_required_ True, if a certified English translation is required with the document.
   * \param[in] is_native_name_required_ True, if personal details must include the user&\#039;s name in the language of their country of residence.
   */
  passportSuitableElement(object_ptr<PassportElementType> &&type_, bool is_selfie_required_, bool is_translation_required_, bool is_native_name_required_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -789019876;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Represents the current state of 2-step verification.
 */
class passwordState final : public Object {
 public:
  /// True, if a 2-step verification password is set.
  bool has_password_;
  /// Hint for the password; may be empty.
  string password_hint_;
  /// True, if a recovery email is set.
  bool has_recovery_email_address_;
  /// True, if some Telegram Passport elements were saved.
  bool has_passport_data_;
  /// Information about the recovery email address to which the confirmation email was sent; may be null.
  object_ptr<emailAddressAuthenticationCodeInfo> recovery_email_address_code_info_;
  /// If not 0, point in time (Unix timestamp) after which the password can be reset immediately using resetPassword.
  int32 pending_reset_date_;

  /**
   * Represents the current state of 2-step verification.
   */
  passwordState();

  /**
   * Represents the current state of 2-step verification.
   *
   * \param[in] has_password_ True, if a 2-step verification password is set.
   * \param[in] password_hint_ Hint for the password; may be empty.
   * \param[in] has_recovery_email_address_ True, if a recovery email is set.
   * \param[in] has_passport_data_ True, if some Telegram Passport elements were saved.
   * \param[in] recovery_email_address_code_info_ Information about the recovery email address to which the confirmation email was sent; may be null.
   * \param[in] pending_reset_date_ If not 0, point in time (Unix timestamp) after which the password can be reset immediately using resetPassword.
   */
  passwordState(bool has_password_, string const &password_hint_, bool has_recovery_email_address_, bool has_passport_data_, object_ptr<emailAddressAuthenticationCodeInfo> &&recovery_email_address_code_info_, int32 pending_reset_date_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -2001619202;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains information about an invoice payment form.
 */
class paymentForm final : public Object {
 public:
  /// The payment form identifier.
  int64 id_;
  /// Full information of the invoice.
  object_ptr<invoice> invoice_;
  /// Payment form URL.
  string url_;
  /// User identifier of the seller bot.
  int53 seller_bot_user_id_;
  /// User identifier of the payment provider bot.
  int53 payments_provider_user_id_;
  /// Information about the payment provider, if available, to support it natively without the need for opening the URL; may be null.
  object_ptr<paymentsProviderStripe> payments_provider_;
  /// Saved server-side order information; may be null.
  object_ptr<orderInfo> saved_order_info_;
  /// Information about saved card credentials; may be null.
  object_ptr<savedCredentials> saved_credentials_;
  /// True, if the user can choose to save credentials.
  bool can_save_credentials_;
  /// True, if the user will be able to save credentials protected by a password they set up.
  bool need_password_;

  /**
   * Contains information about an invoice payment form.
   */
  paymentForm();

  /**
   * Contains information about an invoice payment form.
   *
   * \param[in] id_ The payment form identifier.
   * \param[in] invoice_ Full information of the invoice.
   * \param[in] url_ Payment form URL.
   * \param[in] seller_bot_user_id_ User identifier of the seller bot.
   * \param[in] payments_provider_user_id_ User identifier of the payment provider bot.
   * \param[in] payments_provider_ Information about the payment provider, if available, to support it natively without the need for opening the URL; may be null.
   * \param[in] saved_order_info_ Saved server-side order information; may be null.
   * \param[in] saved_credentials_ Information about saved card credentials; may be null.
   * \param[in] can_save_credentials_ True, if the user can choose to save credentials.
   * \param[in] need_password_ True, if the user will be able to save credentials protected by a password they set up.
   */
  paymentForm(int64 id_, object_ptr<invoice> &&invoice_, string const &url_, int53 seller_bot_user_id_, int53 payments_provider_user_id_, object_ptr<paymentsProviderStripe> &&payments_provider_, object_ptr<orderInfo> &&saved_order_info_, object_ptr<savedCredentials> &&saved_credentials_, bool can_save_credentials_, bool need_password_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1462608358;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Theme colors for a payment form.
 */
class paymentFormTheme final : public Object {
 public:
  /// A color of the payment form background in the RGB24 format.
  int32 background_color_;
  /// A color of text in the RGB24 format.
  int32 text_color_;
  /// A color of hints in the RGB24 format.
  int32 hint_color_;
  /// A color of links in the RGB24 format.
  int32 link_color_;
  /// A color of the buttons in the RGB24 format.
  int32 button_color_;
  /// A color of text on the buttons in the RGB24 format.
  int32 button_text_color_;

  /**
   * Theme colors for a payment form.
   */
  paymentFormTheme();

  /**
   * Theme colors for a payment form.
   *
   * \param[in] background_color_ A color of the payment form background in the RGB24 format.
   * \param[in] text_color_ A color of text in the RGB24 format.
   * \param[in] hint_color_ A color of hints in the RGB24 format.
   * \param[in] link_color_ A color of links in the RGB24 format.
   * \param[in] button_color_ A color of the buttons in the RGB24 format.
   * \param[in] button_text_color_ A color of text on the buttons in the RGB24 format.
   */
  paymentFormTheme(int32 background_color_, int32 text_color_, int32 hint_color_, int32 link_color_, int32 button_color_, int32 button_text_color_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1760030833;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains information about a successful payment.
 */
class paymentReceipt final : public Object {
 public:
  /// Product title.
  string title_;
  /// Product description.
  string description_;
  /// Product photo; may be null.
  object_ptr<photo> photo_;
  /// Point in time (Unix timestamp) when the payment was made.
  int32 date_;
  /// User identifier of the seller bot.
  int53 seller_bot_user_id_;
  /// User identifier of the payment provider bot.
  int53 payments_provider_user_id_;
  /// Information about the invoice.
  object_ptr<invoice> invoice_;
  /// Order information; may be null.
  object_ptr<orderInfo> order_info_;
  /// Chosen shipping option; may be null.
  object_ptr<shippingOption> shipping_option_;
  /// Title of the saved credentials chosen by the buyer.
  string credentials_title_;
  /// The amount of tip chosen by the buyer in the smallest units of the currency.
  int53 tip_amount_;

  /**
   * Contains information about a successful payment.
   */
  paymentReceipt();

  /**
   * Contains information about a successful payment.
   *
   * \param[in] title_ Product title.
   * \param[in] description_ Product description.
   * \param[in] photo_ Product photo; may be null.
   * \param[in] date_ Point in time (Unix timestamp) when the payment was made.
   * \param[in] seller_bot_user_id_ User identifier of the seller bot.
   * \param[in] payments_provider_user_id_ User identifier of the payment provider bot.
   * \param[in] invoice_ Information about the invoice.
   * \param[in] order_info_ Order information; may be null.
   * \param[in] shipping_option_ Chosen shipping option; may be null.
   * \param[in] credentials_title_ Title of the saved credentials chosen by the buyer.
   * \param[in] tip_amount_ The amount of tip chosen by the buyer in the smallest units of the currency.
   */
  paymentReceipt(string const &title_, string const &description_, object_ptr<photo> &&photo_, int32 date_, int53 seller_bot_user_id_, int53 payments_provider_user_id_, object_ptr<invoice> &&invoice_, object_ptr<orderInfo> &&order_info_, object_ptr<shippingOption> &&shipping_option_, string const &credentials_title_, int53 tip_amount_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -401796169;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains the result of a payment request.
 */
class paymentResult final : public Object {
 public:
  /// True, if the payment request was successful; otherwise the verification_url will be non-empty.
  bool success_;
  /// URL for additional payment credentials verification.
  string verification_url_;

  /**
   * Contains the result of a payment request.
   */
  paymentResult();

  /**
   * Contains the result of a payment request.
   *
   * \param[in] success_ True, if the payment request was successful; otherwise the verification_url will be non-empty.
   * \param[in] verification_url_ URL for additional payment credentials verification.
   */
  paymentResult(bool success_, string const &verification_url_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -804263843;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Stripe payment provider.
 */
class paymentsProviderStripe final : public Object {
 public:
  /// Stripe API publishable key.
  string publishable_key_;
  /// True, if the user country must be provided.
  bool need_country_;
  /// True, if the user ZIP/postal code must be provided.
  bool need_postal_code_;
  /// True, if the cardholder name must be provided.
  bool need_cardholder_name_;

  /**
   * Stripe payment provider.
   */
  paymentsProviderStripe();

  /**
   * Stripe payment provider.
   *
   * \param[in] publishable_key_ Stripe API publishable key.
   * \param[in] need_country_ True, if the user country must be provided.
   * \param[in] need_postal_code_ True, if the user ZIP/postal code must be provided.
   * \param[in] need_cardholder_name_ True, if the cardholder name must be provided.
   */
  paymentsProviderStripe(string const &publishable_key_, bool need_country_, bool need_postal_code_, bool need_cardholder_name_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1090791032;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains the user&\#039;s personal details.
 */
class personalDetails final : public Object {
 public:
  /// First name of the user written in English; 1-255 characters.
  string first_name_;
  /// Middle name of the user written in English; 0-255 characters.
  string middle_name_;
  /// Last name of the user written in English; 1-255 characters.
  string last_name_;
  /// Native first name of the user; 1-255 characters.
  string native_first_name_;
  /// Native middle name of the user; 0-255 characters.
  string native_middle_name_;
  /// Native last name of the user; 1-255 characters.
  string native_last_name_;
  /// Birthdate of the user.
  object_ptr<date> birthdate_;
  /// Gender of the user, &quot;male&quot; or &quot;female&quot;.
  string gender_;
  /// A two-letter ISO 3166-1 alpha-2 country code of the user&\#039;s country.
  string country_code_;
  /// A two-letter ISO 3166-1 alpha-2 country code of the user&\#039;s residence country.
  string residence_country_code_;

  /**
   * Contains the user&\#039;s personal details.
   */
  personalDetails();

  /**
   * Contains the user&\#039;s personal details.
   *
   * \param[in] first_name_ First name of the user written in English; 1-255 characters.
   * \param[in] middle_name_ Middle name of the user written in English; 0-255 characters.
   * \param[in] last_name_ Last name of the user written in English; 1-255 characters.
   * \param[in] native_first_name_ Native first name of the user; 1-255 characters.
   * \param[in] native_middle_name_ Native middle name of the user; 0-255 characters.
   * \param[in] native_last_name_ Native last name of the user; 1-255 characters.
   * \param[in] birthdate_ Birthdate of the user.
   * \param[in] gender_ Gender of the user, &quot;male&quot; or &quot;female&quot;.
   * \param[in] country_code_ A two-letter ISO 3166-1 alpha-2 country code of the user&\#039;s country.
   * \param[in] residence_country_code_ A two-letter ISO 3166-1 alpha-2 country code of the user&\#039;s residence country.
   */
  personalDetails(string const &first_name_, string const &middle_name_, string const &last_name_, string const &native_first_name_, string const &native_middle_name_, string const &native_last_name_, object_ptr<date> &&birthdate_, string const &gender_, string const &country_code_, string const &residence_country_code_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1061656137;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A personal document, containing some information about a user.
 */
class personalDocument final : public Object {
 public:
  /// List of files containing the pages of the document.
  array<object_ptr<datedFile>> files_;
  /// List of files containing a certified English translation of the document.
  array<object_ptr<datedFile>> translation_;

  /**
   * A personal document, containing some information about a user.
   */
  personalDocument();

  /**
   * A personal document, containing some information about a user.
   *
   * \param[in] files_ List of files containing the pages of the document.
   * \param[in] translation_ List of files containing a certified English translation of the document.
   */
  personalDocument(array<object_ptr<datedFile>> &&files_, array<object_ptr<datedFile>> &&translation_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1011634661;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains settings for the authentication of the user&\#039;s phone number.
 */
class phoneNumberAuthenticationSettings final : public Object {
 public:
  /// Pass true if the authentication code may be sent via a flash call to the specified phone number.
  bool allow_flash_call_;
  /// Pass true if the authentication code may be sent via a missed call to the specified phone number.
  bool allow_missed_call_;
  /// Pass true if the authenticated phone number is used on the current device.
  bool is_current_phone_number_;
  /// For official applications only. True, if the application can use Android SMS Retriever API (requires Google Play Services &gt;= 10.2) to automatically receive the authentication code from the SMS. See https://developers.google.com/identity/sms-retriever/ for more details.
  bool allow_sms_retriever_api_;
  /// List of up to 20 authentication tokens, recently received in updateOption(&quot;authentication_token&quot;) in previously logged out sessions.
  array<string> authentication_tokens_;

  /**
   * Contains settings for the authentication of the user&\#039;s phone number.
   */
  phoneNumberAuthenticationSettings();

  /**
   * Contains settings for the authentication of the user&\#039;s phone number.
   *
   * \param[in] allow_flash_call_ Pass true if the authentication code may be sent via a flash call to the specified phone number.
   * \param[in] allow_missed_call_ Pass true if the authentication code may be sent via a missed call to the specified phone number.
   * \param[in] is_current_phone_number_ Pass true if the authenticated phone number is used on the current device.
   * \param[in] allow_sms_retriever_api_ For official applications only. True, if the application can use Android SMS Retriever API (requires Google Play Services &gt;= 10.2) to automatically receive the authentication code from the SMS. See https://developers.google.com/identity/sms-retriever/ for more details.
   * \param[in] authentication_tokens_ List of up to 20 authentication tokens, recently received in updateOption(&quot;authentication_token&quot;) in previously logged out sessions.
   */
  phoneNumberAuthenticationSettings(bool allow_flash_call_, bool allow_missed_call_, bool is_current_phone_number_, bool allow_sms_retriever_api_, array<string> &&authentication_tokens_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -421219756;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains information about a phone number.
 */
class phoneNumberInfo final : public Object {
 public:
  /// Information about the country to which the phone number belongs; may be null.
  object_ptr<countryInfo> country_;
  /// The part of the phone number denoting country calling code or its part.
  string country_calling_code_;
  /// The phone number without country calling code formatted accordingly to local rules. Expected digits are returned as &\#039;-&\#039;, but even more digits might be entered by the user.
  string formatted_phone_number_;

  /**
   * Contains information about a phone number.
   */
  phoneNumberInfo();

  /**
   * Contains information about a phone number.
   *
   * \param[in] country_ Information about the country to which the phone number belongs; may be null.
   * \param[in] country_calling_code_ The part of the phone number denoting country calling code or its part.
   * \param[in] formatted_phone_number_ The phone number without country calling code formatted accordingly to local rules. Expected digits are returned as &\#039;-&\#039;, but even more digits might be entered by the user.
   */
  phoneNumberInfo(object_ptr<countryInfo> &&country_, string const &country_calling_code_, string const &formatted_phone_number_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 560180961;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Describes a photo.
 */
class photo final : public Object {
 public:
  /// True, if stickers were added to the photo. The list of corresponding sticker sets can be received using getAttachedStickerSets.
  bool has_stickers_;
  /// Photo minithumbnail; may be null.
  object_ptr<minithumbnail> minithumbnail_;
  /// Available variants of the photo, in different sizes.
  array<object_ptr<photoSize>> sizes_;

  /**
   * Describes a photo.
   */
  photo();

  /**
   * Describes a photo.
   *
   * \param[in] has_stickers_ True, if stickers were added to the photo. The list of corresponding sticker sets can be received using getAttachedStickerSets.
   * \param[in] minithumbnail_ Photo minithumbnail; may be null.
   * \param[in] sizes_ Available variants of the photo, in different sizes.
   */
  photo(bool has_stickers_, object_ptr<minithumbnail> &&minithumbnail_, array<object_ptr<photoSize>> &&sizes_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -2022871583;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Describes an image in JPEG format.
 */
class photoSize final : public Object {
 public:
  /// Image type (see https://core.telegram.org/constructor/photoSize).
  string type_;
  /// Information about the image file.
  object_ptr<file> photo_;
  /// Image width.
  int32 width_;
  /// Image height.
  int32 height_;
  /// Sizes of progressive JPEG file prefixes, which can be used to preliminarily show the image; in bytes.
  array<int32> progressive_sizes_;

  /**
   * Describes an image in JPEG format.
   */
  photoSize();

  /**
   * Describes an image in JPEG format.
   *
   * \param[in] type_ Image type (see https://core.telegram.org/constructor/photoSize).
   * \param[in] photo_ Information about the image file.
   * \param[in] width_ Image width.
   * \param[in] height_ Image height.
   * \param[in] progressive_sizes_ Sizes of progressive JPEG file prefixes, which can be used to preliminarily show the image; in bytes.
   */
  photoSize(string const &type_, object_ptr<file> &&photo_, int32 width_, int32 height_, array<int32> &&progressive_sizes_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1609182352;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A point on a Cartesian plane.
 */
class point final : public Object {
 public:
  /// The point&\#039;s first coordinate.
  double x_;
  /// The point&\#039;s second coordinate.
  double y_;

  /**
   * A point on a Cartesian plane.
   */
  point();

  /**
   * A point on a Cartesian plane.
   *
   * \param[in] x_ The point&\#039;s first coordinate.
   * \param[in] y_ The point&\#039;s second coordinate.
   */
  point(double x_, double y_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 437515705;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Describes a poll.
 */
class poll final : public Object {
 public:
  /// Unique poll identifier.
  int64 id_;
  /// Poll question; 1-300 characters.
  string question_;
  /// List of poll answer options.
  array<object_ptr<pollOption>> options_;
  /// Total number of voters, participating in the poll.
  int32 total_voter_count_;
  /// User identifiers of recent voters, if the poll is non-anonymous.
  array<int53> recent_voter_user_ids_;
  /// True, if the poll is anonymous.
  bool is_anonymous_;
  /// Type of the poll.
  object_ptr<PollType> type_;
  /// Amount of time the poll will be active after creation, in seconds.
  int32 open_period_;
  /// Point in time (Unix timestamp) when the poll will automatically be closed.
  int32 close_date_;
  /// True, if the poll is closed.
  bool is_closed_;

  /**
   * Describes a poll.
   */
  poll();

  /**
   * Describes a poll.
   *
   * \param[in] id_ Unique poll identifier.
   * \param[in] question_ Poll question; 1-300 characters.
   * \param[in] options_ List of poll answer options.
   * \param[in] total_voter_count_ Total number of voters, participating in the poll.
   * \param[in] recent_voter_user_ids_ User identifiers of recent voters, if the poll is non-anonymous.
   * \param[in] is_anonymous_ True, if the poll is anonymous.
   * \param[in] type_ Type of the poll.
   * \param[in] open_period_ Amount of time the poll will be active after creation, in seconds.
   * \param[in] close_date_ Point in time (Unix timestamp) when the poll will automatically be closed.
   * \param[in] is_closed_ True, if the poll is closed.
   */
  poll(int64 id_, string const &question_, array<object_ptr<pollOption>> &&options_, int32 total_voter_count_, array<int53> &&recent_voter_user_ids_, bool is_anonymous_, object_ptr<PollType> &&type_, int32 open_period_, int32 close_date_, bool is_closed_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 116940085;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Describes one answer option of a poll.
 */
class pollOption final : public Object {
 public:
  /// Option text; 1-100 characters.
  string text_;
  /// Number of voters for this option, available only for closed or voted polls.
  int32 voter_count_;
  /// The percentage of votes for this option; 0-100.
  int32 vote_percentage_;
  /// True, if the option was chosen by the user.
  bool is_chosen_;
  /// True, if the option is being chosen by a pending setPollAnswer request.
  bool is_being_chosen_;

  /**
   * Describes one answer option of a poll.
   */
  pollOption();

  /**
   * Describes one answer option of a poll.
   *
   * \param[in] text_ Option text; 1-100 characters.
   * \param[in] voter_count_ Number of voters for this option, available only for closed or voted polls.
   * \param[in] vote_percentage_ The percentage of votes for this option; 0-100.
   * \param[in] is_chosen_ True, if the option was chosen by the user.
   * \param[in] is_being_chosen_ True, if the option is being chosen by a pending setPollAnswer request.
   */
  pollOption(string const &text_, int32 voter_count_, int32 vote_percentage_, bool is_chosen_, bool is_being_chosen_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1473893797;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Describes the type of a poll.
 */
class PollType: public Object {
 public:
};

/**
 * A regular poll.
 */
class pollTypeRegular final : public PollType {
 public:
  /// True, if multiple answer options can be chosen simultaneously.
  bool allow_multiple_answers_;

  /**
   * A regular poll.
   */
  pollTypeRegular();

  /**
   * A regular poll.
   *
   * \param[in] allow_multiple_answers_ True, if multiple answer options can be chosen simultaneously.
   */
  explicit pollTypeRegular(bool allow_multiple_answers_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 641265698;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A poll in quiz mode, which has exactly one correct answer option and can be answered only once.
 */
class pollTypeQuiz final : public PollType {
 public:
  /// 0-based identifier of the correct answer option; -1 for a yet unanswered poll.
  int32 correct_option_id_;
  /// Text that is shown when the user chooses an incorrect answer or taps on the lamp icon; 0-200 characters with at most 2 line feeds; empty for a yet unanswered poll.
  object_ptr<formattedText> explanation_;

  /**
   * A poll in quiz mode, which has exactly one correct answer option and can be answered only once.
   */
  pollTypeQuiz();

  /**
   * A poll in quiz mode, which has exactly one correct answer option and can be answered only once.
   *
   * \param[in] correct_option_id_ 0-based identifier of the correct answer option; -1 for a yet unanswered poll.
   * \param[in] explanation_ Text that is shown when the user chooses an incorrect answer or taps on the lamp icon; 0-200 characters with at most 2 line feeds; empty for a yet unanswered poll.
   */
  pollTypeQuiz(int32 correct_option_id_, object_ptr<formattedText> &&explanation_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 657013913;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Describes a user profile photo.
 */
class profilePhoto final : public Object {
 public:
  /// Photo identifier; 0 for an empty photo. Can be used to find a photo in a list of user profile photos.
  int64 id_;
  /// A small (160x160) user profile photo. The file can be downloaded only before the photo is changed.
  object_ptr<file> small_;
  /// A big (640x640) user profile photo. The file can be downloaded only before the photo is changed.
  object_ptr<file> big_;
  /// User profile photo minithumbnail; may be null.
  object_ptr<minithumbnail> minithumbnail_;
  /// True, if the photo has animated variant.
  bool has_animation_;

  /**
   * Describes a user profile photo.
   */
  profilePhoto();

  /**
   * Describes a user profile photo.
   *
   * \param[in] id_ Photo identifier; 0 for an empty photo. Can be used to find a photo in a list of user profile photos.
   * \param[in] small_ A small (160x160) user profile photo. The file can be downloaded only before the photo is changed.
   * \param[in] big_ A big (640x640) user profile photo. The file can be downloaded only before the photo is changed.
   * \param[in] minithumbnail_ User profile photo minithumbnail; may be null.
   * \param[in] has_animation_ True, if the photo has animated variant.
   */
  profilePhoto(int64 id_, object_ptr<file> &&small_, object_ptr<file> &&big_, object_ptr<minithumbnail> &&minithumbnail_, bool has_animation_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -131097523;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Represents a list of proxy servers.
 */
class proxies final : public Object {
 public:
  /// List of proxy servers.
  array<object_ptr<proxy>> proxies_;

  /**
   * Represents a list of proxy servers.
   */
  proxies();

  /**
   * Represents a list of proxy servers.
   *
   * \param[in] proxies_ List of proxy servers.
   */
  explicit proxies(array<object_ptr<proxy>> &&proxies_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1200447205;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains information about a proxy server.
 */
class proxy final : public Object {
 public:
  /// Unique identifier of the proxy.
  int32 id_;
  /// Proxy server IP address.
  string server_;
  /// Proxy server port.
  int32 port_;
  /// Point in time (Unix timestamp) when the proxy was last used; 0 if never.
  int32 last_used_date_;
  /// True, if the proxy is enabled now.
  bool is_enabled_;
  /// Type of the proxy.
  object_ptr<ProxyType> type_;

  /**
   * Contains information about a proxy server.
   */
  proxy();

  /**
   * Contains information about a proxy server.
   *
   * \param[in] id_ Unique identifier of the proxy.
   * \param[in] server_ Proxy server IP address.
   * \param[in] port_ Proxy server port.
   * \param[in] last_used_date_ Point in time (Unix timestamp) when the proxy was last used; 0 if never.
   * \param[in] is_enabled_ True, if the proxy is enabled now.
   * \param[in] type_ Type of the proxy.
   */
  proxy(int32 id_, string const &server_, int32 port_, int32 last_used_date_, bool is_enabled_, object_ptr<ProxyType> &&type_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 196049779;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Describes the type of a proxy server.
 */
class ProxyType: public Object {
 public:
};

/**
 * A SOCKS5 proxy server.
 */
class proxyTypeSocks5 final : public ProxyType {
 public:
  /// Username for logging in; may be empty.
  string username_;
  /// Password for logging in; may be empty.
  string password_;

  /**
   * A SOCKS5 proxy server.
   */
  proxyTypeSocks5();

  /**
   * A SOCKS5 proxy server.
   *
   * \param[in] username_ Username for logging in; may be empty.
   * \param[in] password_ Password for logging in; may be empty.
   */
  proxyTypeSocks5(string const &username_, string const &password_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -890027341;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A HTTP transparent proxy server.
 */
class proxyTypeHttp final : public ProxyType {
 public:
  /// Username for logging in; may be empty.
  string username_;
  /// Password for logging in; may be empty.
  string password_;
  /// Pass true if the proxy supports only HTTP requests and doesn&\#039;t support transparent TCP connections via HTTP CONNECT method.
  bool http_only_;

  /**
   * A HTTP transparent proxy server.
   */
  proxyTypeHttp();

  /**
   * A HTTP transparent proxy server.
   *
   * \param[in] username_ Username for logging in; may be empty.
   * \param[in] password_ Password for logging in; may be empty.
   * \param[in] http_only_ Pass true if the proxy supports only HTTP requests and doesn&\#039;t support transparent TCP connections via HTTP CONNECT method.
   */
  proxyTypeHttp(string const &username_, string const &password_, bool http_only_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1547188361;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * An MTProto proxy server.
 */
class proxyTypeMtproto final : public ProxyType {
 public:
  /// The proxy&\#039;s secret in hexadecimal encoding.
  string secret_;

  /**
   * An MTProto proxy server.
   */
  proxyTypeMtproto();

  /**
   * An MTProto proxy server.
   *
   * \param[in] secret_ The proxy&\#039;s secret in hexadecimal encoding.
   */
  explicit proxyTypeMtproto(string const &secret_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1964826627;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Describes a type of public chats.
 */
class PublicChatType: public Object {
 public:
};

/**
 * The chat is public, because it has username.
 */
class publicChatTypeHasUsername final : public PublicChatType {
 public:

  /**
   * The chat is public, because it has username.
   */
  publicChatTypeHasUsername();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 350789758;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The chat is public, because it is a location-based supergroup.
 */
class publicChatTypeIsLocationBased final : public PublicChatType {
 public:

  /**
   * The chat is public, because it is a location-based supergroup.
   */
  publicChatTypeIsLocationBased();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1183735952;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Contains content of a push message notification.
 */
class PushMessageContent: public Object {
 public:
};

/**
 * A general message with hidden content.
 */
class pushMessageContentHidden final : public PushMessageContent {
 public:
  /// True, if the message is a pinned message with the specified content.
  bool is_pinned_;

  /**
   * A general message with hidden content.
   */
  pushMessageContentHidden();

  /**
   * A general message with hidden content.
   *
   * \param[in] is_pinned_ True, if the message is a pinned message with the specified content.
   */
  explicit pushMessageContentHidden(bool is_pinned_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -316950436;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * An animation message (GIF-style).
 */
class pushMessageContentAnimation final : public PushMessageContent {
 public:
  /// Message content; may be null.
  object_ptr<animation> animation_;
  /// Animation caption.
  string caption_;
  /// True, if the message is a pinned message with the specified content.
  bool is_pinned_;

  /**
   * An animation message (GIF-style).
   */
  pushMessageContentAnimation();

  /**
   * An animation message (GIF-style).
   *
   * \param[in] animation_ Message content; may be null.
   * \param[in] caption_ Animation caption.
   * \param[in] is_pinned_ True, if the message is a pinned message with the specified content.
   */
  pushMessageContentAnimation(object_ptr<animation> &&animation_, string const &caption_, bool is_pinned_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1034215396;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * An audio message.
 */
class pushMessageContentAudio final : public PushMessageContent {
 public:
  /// Message content; may be null.
  object_ptr<audio> audio_;
  /// True, if the message is a pinned message with the specified content.
  bool is_pinned_;

  /**
   * An audio message.
   */
  pushMessageContentAudio();

  /**
   * An audio message.
   *
   * \param[in] audio_ Message content; may be null.
   * \param[in] is_pinned_ True, if the message is a pinned message with the specified content.
   */
  pushMessageContentAudio(object_ptr<audio> &&audio_, bool is_pinned_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 381581426;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A message with a user contact.
 */
class pushMessageContentContact final : public PushMessageContent {
 public:
  /// Contact&\#039;s name.
  string name_;
  /// True, if the message is a pinned message with the specified content.
  bool is_pinned_;

  /**
   * A message with a user contact.
   */
  pushMessageContentContact();

  /**
   * A message with a user contact.
   *
   * \param[in] name_ Contact&\#039;s name.
   * \param[in] is_pinned_ True, if the message is a pinned message with the specified content.
   */
  pushMessageContentContact(string const &name_, bool is_pinned_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -12219820;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A contact has registered with Telegram.
 */
class pushMessageContentContactRegistered final : public PushMessageContent {
 public:

  /**
   * A contact has registered with Telegram.
   */
  pushMessageContentContactRegistered();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -303962720;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A document message (a general file).
 */
class pushMessageContentDocument final : public PushMessageContent {
 public:
  /// Message content; may be null.
  object_ptr<document> document_;
  /// True, if the message is a pinned message with the specified content.
  bool is_pinned_;

  /**
   * A document message (a general file).
   */
  pushMessageContentDocument();

  /**
   * A document message (a general file).
   *
   * \param[in] document_ Message content; may be null.
   * \param[in] is_pinned_ True, if the message is a pinned message with the specified content.
   */
  pushMessageContentDocument(object_ptr<document> &&document_, bool is_pinned_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -458379775;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A message with a game.
 */
class pushMessageContentGame final : public PushMessageContent {
 public:
  /// Game title, empty for pinned game message.
  string title_;
  /// True, if the message is a pinned message with the specified content.
  bool is_pinned_;

  /**
   * A message with a game.
   */
  pushMessageContentGame();

  /**
   * A message with a game.
   *
   * \param[in] title_ Game title, empty for pinned game message.
   * \param[in] is_pinned_ True, if the message is a pinned message with the specified content.
   */
  pushMessageContentGame(string const &title_, bool is_pinned_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -515131109;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A new high score was achieved in a game.
 */
class pushMessageContentGameScore final : public PushMessageContent {
 public:
  /// Game title, empty for pinned message.
  string title_;
  /// New score, 0 for pinned message.
  int32 score_;
  /// True, if the message is a pinned message with the specified content.
  bool is_pinned_;

  /**
   * A new high score was achieved in a game.
   */
  pushMessageContentGameScore();

  /**
   * A new high score was achieved in a game.
   *
   * \param[in] title_ Game title, empty for pinned message.
   * \param[in] score_ New score, 0 for pinned message.
   * \param[in] is_pinned_ True, if the message is a pinned message with the specified content.
   */
  pushMessageContentGameScore(string const &title_, int32 score_, bool is_pinned_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 901303688;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A message with an invoice from a bot.
 */
class pushMessageContentInvoice final : public PushMessageContent {
 public:
  /// Product price.
  string price_;
  /// True, if the message is a pinned message with the specified content.
  bool is_pinned_;

  /**
   * A message with an invoice from a bot.
   */
  pushMessageContentInvoice();

  /**
   * A message with an invoice from a bot.
   *
   * \param[in] price_ Product price.
   * \param[in] is_pinned_ True, if the message is a pinned message with the specified content.
   */
  pushMessageContentInvoice(string const &price_, bool is_pinned_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1731687492;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A message with a location.
 */
class pushMessageContentLocation final : public PushMessageContent {
 public:
  /// True, if the location is live.
  bool is_live_;
  /// True, if the message is a pinned message with the specified content.
  bool is_pinned_;

  /**
   * A message with a location.
   */
  pushMessageContentLocation();

  /**
   * A message with a location.
   *
   * \param[in] is_live_ True, if the location is live.
   * \param[in] is_pinned_ True, if the message is a pinned message with the specified content.
   */
  pushMessageContentLocation(bool is_live_, bool is_pinned_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1288005709;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A photo message.
 */
class pushMessageContentPhoto final : public PushMessageContent {
 public:
  /// Message content; may be null.
  object_ptr<photo> photo_;
  /// Photo caption.
  string caption_;
  /// True, if the photo is secret.
  bool is_secret_;
  /// True, if the message is a pinned message with the specified content.
  bool is_pinned_;

  /**
   * A photo message.
   */
  pushMessageContentPhoto();

  /**
   * A photo message.
   *
   * \param[in] photo_ Message content; may be null.
   * \param[in] caption_ Photo caption.
   * \param[in] is_secret_ True, if the photo is secret.
   * \param[in] is_pinned_ True, if the message is a pinned message with the specified content.
   */
  pushMessageContentPhoto(object_ptr<photo> &&photo_, string const &caption_, bool is_secret_, bool is_pinned_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 140631122;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A message with a poll.
 */
class pushMessageContentPoll final : public PushMessageContent {
 public:
  /// Poll question.
  string question_;
  /// True, if the poll is regular and not in quiz mode.
  bool is_regular_;
  /// True, if the message is a pinned message with the specified content.
  bool is_pinned_;

  /**
   * A message with a poll.
   */
  pushMessageContentPoll();

  /**
   * A message with a poll.
   *
   * \param[in] question_ Poll question.
   * \param[in] is_regular_ True, if the poll is regular and not in quiz mode.
   * \param[in] is_pinned_ True, if the message is a pinned message with the specified content.
   */
  pushMessageContentPoll(string const &question_, bool is_regular_, bool is_pinned_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -44403654;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A screenshot of a message in the chat has been taken.
 */
class pushMessageContentScreenshotTaken final : public PushMessageContent {
 public:

  /**
   * A screenshot of a message in the chat has been taken.
   */
  pushMessageContentScreenshotTaken();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 214245369;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A message with a sticker.
 */
class pushMessageContentSticker final : public PushMessageContent {
 public:
  /// Message content; may be null.
  object_ptr<sticker> sticker_;
  /// Emoji corresponding to the sticker; may be empty.
  string emoji_;
  /// True, if the message is a pinned message with the specified content.
  bool is_pinned_;

  /**
   * A message with a sticker.
   */
  pushMessageContentSticker();

  /**
   * A message with a sticker.
   *
   * \param[in] sticker_ Message content; may be null.
   * \param[in] emoji_ Emoji corresponding to the sticker; may be empty.
   * \param[in] is_pinned_ True, if the message is a pinned message with the specified content.
   */
  pushMessageContentSticker(object_ptr<sticker> &&sticker_, string const &emoji_, bool is_pinned_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1553513939;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A text message.
 */
class pushMessageContentText final : public PushMessageContent {
 public:
  /// Message text.
  string text_;
  /// True, if the message is a pinned message with the specified content.
  bool is_pinned_;

  /**
   * A text message.
   */
  pushMessageContentText();

  /**
   * A text message.
   *
   * \param[in] text_ Message text.
   * \param[in] is_pinned_ True, if the message is a pinned message with the specified content.
   */
  pushMessageContentText(string const &text_, bool is_pinned_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 274587305;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A video message.
 */
class pushMessageContentVideo final : public PushMessageContent {
 public:
  /// Message content; may be null.
  object_ptr<video> video_;
  /// Video caption.
  string caption_;
  /// True, if the video is secret.
  bool is_secret_;
  /// True, if the message is a pinned message with the specified content.
  bool is_pinned_;

  /**
   * A video message.
   */
  pushMessageContentVideo();

  /**
   * A video message.
   *
   * \param[in] video_ Message content; may be null.
   * \param[in] caption_ Video caption.
   * \param[in] is_secret_ True, if the video is secret.
   * \param[in] is_pinned_ True, if the message is a pinned message with the specified content.
   */
  pushMessageContentVideo(object_ptr<video> &&video_, string const &caption_, bool is_secret_, bool is_pinned_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 310038831;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A video note message.
 */
class pushMessageContentVideoNote final : public PushMessageContent {
 public:
  /// Message content; may be null.
  object_ptr<videoNote> video_note_;
  /// True, if the message is a pinned message with the specified content.
  bool is_pinned_;

  /**
   * A video note message.
   */
  pushMessageContentVideoNote();

  /**
   * A video note message.
   *
   * \param[in] video_note_ Message content; may be null.
   * \param[in] is_pinned_ True, if the message is a pinned message with the specified content.
   */
  pushMessageContentVideoNote(object_ptr<videoNote> &&video_note_, bool is_pinned_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1122764417;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A voice note message.
 */
class pushMessageContentVoiceNote final : public PushMessageContent {
 public:
  /// Message content; may be null.
  object_ptr<voiceNote> voice_note_;
  /// True, if the message is a pinned message with the specified content.
  bool is_pinned_;

  /**
   * A voice note message.
   */
  pushMessageContentVoiceNote();

  /**
   * A voice note message.
   *
   * \param[in] voice_note_ Message content; may be null.
   * \param[in] is_pinned_ True, if the message is a pinned message with the specified content.
   */
  pushMessageContentVoiceNote(object_ptr<voiceNote> &&voice_note_, bool is_pinned_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 88910987;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A newly created basic group.
 */
class pushMessageContentBasicGroupChatCreate final : public PushMessageContent {
 public:

  /**
   * A newly created basic group.
   */
  pushMessageContentBasicGroupChatCreate();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -2114855172;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * New chat members were invited to a group.
 */
class pushMessageContentChatAddMembers final : public PushMessageContent {
 public:
  /// Name of the added member.
  string member_name_;
  /// True, if the current user was added to the group.
  bool is_current_user_;
  /// True, if the user has returned to the group themselves.
  bool is_returned_;

  /**
   * New chat members were invited to a group.
   */
  pushMessageContentChatAddMembers();

  /**
   * New chat members were invited to a group.
   *
   * \param[in] member_name_ Name of the added member.
   * \param[in] is_current_user_ True, if the current user was added to the group.
   * \param[in] is_returned_ True, if the user has returned to the group themselves.
   */
  pushMessageContentChatAddMembers(string const &member_name_, bool is_current_user_, bool is_returned_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1087145158;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A chat photo was edited.
 */
class pushMessageContentChatChangePhoto final : public PushMessageContent {
 public:

  /**
   * A chat photo was edited.
   */
  pushMessageContentChatChangePhoto();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1114222051;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A chat title was edited.
 */
class pushMessageContentChatChangeTitle final : public PushMessageContent {
 public:
  /// New chat title.
  string title_;

  /**
   * A chat title was edited.
   */
  pushMessageContentChatChangeTitle();

  /**
   * A chat title was edited.
   *
   * \param[in] title_ New chat title.
   */
  explicit pushMessageContentChatChangeTitle(string const &title_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1964902749;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A chat theme was edited.
 */
class pushMessageContentChatSetTheme final : public PushMessageContent {
 public:
  /// If non-empty, name of a new theme, set for the chat. Otherwise chat theme was reset to the default one.
  string theme_name_;

  /**
   * A chat theme was edited.
   */
  pushMessageContentChatSetTheme();

  /**
   * A chat theme was edited.
   *
   * \param[in] theme_name_ If non-empty, name of a new theme, set for the chat. Otherwise chat theme was reset to the default one.
   */
  explicit pushMessageContentChatSetTheme(string const &theme_name_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 173882216;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A chat member was deleted.
 */
class pushMessageContentChatDeleteMember final : public PushMessageContent {
 public:
  /// Name of the deleted member.
  string member_name_;
  /// True, if the current user was deleted from the group.
  bool is_current_user_;
  /// True, if the user has left the group themselves.
  bool is_left_;

  /**
   * A chat member was deleted.
   */
  pushMessageContentChatDeleteMember();

  /**
   * A chat member was deleted.
   *
   * \param[in] member_name_ Name of the deleted member.
   * \param[in] is_current_user_ True, if the current user was deleted from the group.
   * \param[in] is_left_ True, if the user has left the group themselves.
   */
  pushMessageContentChatDeleteMember(string const &member_name_, bool is_current_user_, bool is_left_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 598714783;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A new member joined the chat via an invite link.
 */
class pushMessageContentChatJoinByLink final : public PushMessageContent {
 public:

  /**
   * A new member joined the chat via an invite link.
   */
  pushMessageContentChatJoinByLink();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1553719113;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A new member was accepted to the chat by an administrator.
 */
class pushMessageContentChatJoinByRequest final : public PushMessageContent {
 public:

  /**
   * A new member was accepted to the chat by an administrator.
   */
  pushMessageContentChatJoinByRequest();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -205823627;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A forwarded messages.
 */
class pushMessageContentMessageForwards final : public PushMessageContent {
 public:
  /// Number of forwarded messages.
  int32 total_count_;

  /**
   * A forwarded messages.
   */
  pushMessageContentMessageForwards();

  /**
   * A forwarded messages.
   *
   * \param[in] total_count_ Number of forwarded messages.
   */
  explicit pushMessageContentMessageForwards(int32 total_count_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1913083876;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A media album.
 */
class pushMessageContentMediaAlbum final : public PushMessageContent {
 public:
  /// Number of messages in the album.
  int32 total_count_;
  /// True, if the album has at least one photo.
  bool has_photos_;
  /// True, if the album has at least one video.
  bool has_videos_;
  /// True, if the album has at least one audio file.
  bool has_audios_;
  /// True, if the album has at least one document.
  bool has_documents_;

  /**
   * A media album.
   */
  pushMessageContentMediaAlbum();

  /**
   * A media album.
   *
   * \param[in] total_count_ Number of messages in the album.
   * \param[in] has_photos_ True, if the album has at least one photo.
   * \param[in] has_videos_ True, if the album has at least one video.
   * \param[in] has_audios_ True, if the album has at least one audio file.
   * \param[in] has_documents_ True, if the album has at least one document.
   */
  pushMessageContentMediaAlbum(int32 total_count_, bool has_photos_, bool has_videos_, bool has_audios_, bool has_documents_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -748426897;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains a globally unique push receiver identifier, which can be used to identify which account has received a push notification.
 */
class pushReceiverId final : public Object {
 public:
  /// The globally unique identifier of push notification subscription.
  int64 id_;

  /**
   * Contains a globally unique push receiver identifier, which can be used to identify which account has received a push notification.
   */
  pushReceiverId();

  /**
   * Contains a globally unique push receiver identifier, which can be used to identify which account has received a push notification.
   *
   * \param[in] id_ The globally unique identifier of push notification subscription.
   */
  explicit pushReceiverId(int64 id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 371056428;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Describes a recommended chat filter.
 */
class recommendedChatFilter final : public Object {
 public:
  /// The chat filter.
  object_ptr<chatFilter> filter_;
  /// Chat filter description.
  string description_;

  /**
   * Describes a recommended chat filter.
   */
  recommendedChatFilter();

  /**
   * Describes a recommended chat filter.
   *
   * \param[in] filter_ The chat filter.
   * \param[in] description_ Chat filter description.
   */
  recommendedChatFilter(object_ptr<chatFilter> &&filter_, string const &description_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 36048610;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains a list of recommended chat filters.
 */
class recommendedChatFilters final : public Object {
 public:
  /// List of recommended chat filters.
  array<object_ptr<recommendedChatFilter>> chat_filters_;

  /**
   * Contains a list of recommended chat filters.
   */
  recommendedChatFilters();

  /**
   * Contains a list of recommended chat filters.
   *
   * \param[in] chat_filters_ List of recommended chat filters.
   */
  explicit recommendedChatFilters(array<object_ptr<recommendedChatFilter>> &&chat_filters_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -263416880;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains information about the current recovery email address.
 */
class recoveryEmailAddress final : public Object {
 public:
  /// Recovery email address.
  string recovery_email_address_;

  /**
   * Contains information about the current recovery email address.
   */
  recoveryEmailAddress();

  /**
   * Contains information about the current recovery email address.
   *
   * \param[in] recovery_email_address_ Recovery email address.
   */
  explicit recoveryEmailAddress(string const &recovery_email_address_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1290526187;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Represents a remote file.
 */
class remoteFile final : public Object {
 public:
  /// Remote file identifier; may be empty. Can be used by the current user across application restarts or even from other devices. Uniquely identifies a file, but a file can have a lot of different valid identifiers. If the ID starts with &quot;<a href="http://">http://</a>&quot; or &quot;<a href="https://">https://</a>&quot;, it represents the HTTP URL of the file. TDLib is currently unable to download files if only their URL is known. If downloadFile is called on such a file or if it is sent to a secret chat, TDLib starts a file generation process by sending updateFileGenerationStart to the application with the HTTP URL in the original_path and &quot;\#url\#&quot; as the conversion string. Application must generate the file by downloading it to the specified location.
  string id_;
  /// Unique file identifier; may be empty if unknown. The unique file identifier which is the same for the same file even for different users and is persistent over time.
  string unique_id_;
  /// True, if the file is currently being uploaded (or a remote copy is being generated by some other means).
  bool is_uploading_active_;
  /// True, if a remote copy is fully available.
  bool is_uploading_completed_;
  /// Size of the remote available part of the file, in bytes; 0 if unknown.
  int32 uploaded_size_;

  /**
   * Represents a remote file.
   */
  remoteFile();

  /**
   * Represents a remote file.
   *
   * \param[in] id_ Remote file identifier; may be empty. Can be used by the current user across application restarts or even from other devices. Uniquely identifies a file, but a file can have a lot of different valid identifiers. If the ID starts with &quot;<a href="http://">http://</a>&quot; or &quot;<a href="https://">https://</a>&quot;, it represents the HTTP URL of the file. TDLib is currently unable to download files if only their URL is known. If downloadFile is called on such a file or if it is sent to a secret chat, TDLib starts a file generation process by sending updateFileGenerationStart to the application with the HTTP URL in the original_path and &quot;\#url\#&quot; as the conversion string. Application must generate the file by downloading it to the specified location.
   * \param[in] unique_id_ Unique file identifier; may be empty if unknown. The unique file identifier which is the same for the same file even for different users and is persistent over time.
   * \param[in] is_uploading_active_ True, if the file is currently being uploaded (or a remote copy is being generated by some other means).
   * \param[in] is_uploading_completed_ True, if a remote copy is fully available.
   * \param[in] uploaded_size_ Size of the remote available part of the file, in bytes; 0 if unknown.
   */
  remoteFile(string const &id_, string const &unique_id_, bool is_uploading_active_, bool is_uploading_completed_, int32 uploaded_size_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1822143022;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Contains a description of a custom keyboard and actions that can be done with it to quickly reply to bots.
 */
class ReplyMarkup: public Object {
 public:
};

/**
 * Instructs application to remove the keyboard once this message has been received. This kind of keyboard can&\#039;t be received in an incoming message; instead, UpdateChatReplyMarkup with message_id == 0 will be sent.
 */
class replyMarkupRemoveKeyboard final : public ReplyMarkup {
 public:
  /// True, if the keyboard is removed only for the mentioned users or the target user of a reply.
  bool is_personal_;

  /**
   * Instructs application to remove the keyboard once this message has been received. This kind of keyboard can&\#039;t be received in an incoming message; instead, UpdateChatReplyMarkup with message_id == 0 will be sent.
   */
  replyMarkupRemoveKeyboard();

  /**
   * Instructs application to remove the keyboard once this message has been received. This kind of keyboard can&\#039;t be received in an incoming message; instead, UpdateChatReplyMarkup with message_id == 0 will be sent.
   *
   * \param[in] is_personal_ True, if the keyboard is removed only for the mentioned users or the target user of a reply.
   */
  explicit replyMarkupRemoveKeyboard(bool is_personal_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -691252879;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Instructs application to force a reply to this message.
 */
class replyMarkupForceReply final : public ReplyMarkup {
 public:
  /// True, if a forced reply must automatically be shown to the current user. For outgoing messages, specify true to show the forced reply only for the mentioned users and for the target user of a reply.
  bool is_personal_;
  /// If non-empty, the placeholder to be shown in the input field when the reply is active; 0-64 characters.
  string input_field_placeholder_;

  /**
   * Instructs application to force a reply to this message.
   */
  replyMarkupForceReply();

  /**
   * Instructs application to force a reply to this message.
   *
   * \param[in] is_personal_ True, if a forced reply must automatically be shown to the current user. For outgoing messages, specify true to show the forced reply only for the mentioned users and for the target user of a reply.
   * \param[in] input_field_placeholder_ If non-empty, the placeholder to be shown in the input field when the reply is active; 0-64 characters.
   */
  replyMarkupForceReply(bool is_personal_, string const &input_field_placeholder_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1101461919;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains a custom keyboard layout to quickly reply to bots.
 */
class replyMarkupShowKeyboard final : public ReplyMarkup {
 public:
  /// A list of rows of bot keyboard buttons.
  array<array<object_ptr<keyboardButton>>> rows_;
  /// True, if the application needs to resize the keyboard vertically.
  bool resize_keyboard_;
  /// True, if the application needs to hide the keyboard after use.
  bool one_time_;
  /// True, if the keyboard must automatically be shown to the current user. For outgoing messages, specify true to show the keyboard only for the mentioned users and for the target user of a reply.
  bool is_personal_;
  /// If non-empty, the placeholder to be shown in the input field when the keyboard is active; 0-64 characters.
  string input_field_placeholder_;

  /**
   * Contains a custom keyboard layout to quickly reply to bots.
   */
  replyMarkupShowKeyboard();

  /**
   * Contains a custom keyboard layout to quickly reply to bots.
   *
   * \param[in] rows_ A list of rows of bot keyboard buttons.
   * \param[in] resize_keyboard_ True, if the application needs to resize the keyboard vertically.
   * \param[in] one_time_ True, if the application needs to hide the keyboard after use.
   * \param[in] is_personal_ True, if the keyboard must automatically be shown to the current user. For outgoing messages, specify true to show the keyboard only for the mentioned users and for the target user of a reply.
   * \param[in] input_field_placeholder_ If non-empty, the placeholder to be shown in the input field when the keyboard is active; 0-64 characters.
   */
  replyMarkupShowKeyboard(array<array<object_ptr<keyboardButton>>> &&rows_, bool resize_keyboard_, bool one_time_, bool is_personal_, string const &input_field_placeholder_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -64985802;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains an inline keyboard layout.
 */
class replyMarkupInlineKeyboard final : public ReplyMarkup {
 public:
  /// A list of rows of inline keyboard buttons.
  array<array<object_ptr<inlineKeyboardButton>>> rows_;

  /**
   * Contains an inline keyboard layout.
   */
  replyMarkupInlineKeyboard();

  /**
   * Contains an inline keyboard layout.
   *
   * \param[in] rows_ A list of rows of inline keyboard buttons.
   */
  explicit replyMarkupInlineKeyboard(array<array<object_ptr<inlineKeyboardButton>>> &&rows_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -619317658;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Represents result of 2-step verification password reset.
 */
class ResetPasswordResult: public Object {
 public:
};

/**
 * The password was reset.
 */
class resetPasswordResultOk final : public ResetPasswordResult {
 public:

  /**
   * The password was reset.
   */
  resetPasswordResultOk();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1397267463;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The password reset request is pending.
 */
class resetPasswordResultPending final : public ResetPasswordResult {
 public:
  /// Point in time (Unix timestamp) after which the password can be reset immediately using resetPassword.
  int32 pending_reset_date_;

  /**
   * The password reset request is pending.
   */
  resetPasswordResultPending();

  /**
   * The password reset request is pending.
   *
   * \param[in] pending_reset_date_ Point in time (Unix timestamp) after which the password can be reset immediately using resetPassword.
   */
  explicit resetPasswordResultPending(int32 pending_reset_date_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1193925721;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The password reset request was declined.
 */
class resetPasswordResultDeclined final : public ResetPasswordResult {
 public:
  /// Point in time (Unix timestamp) when the password reset can be retried.
  int32 retry_date_;

  /**
   * The password reset request was declined.
   */
  resetPasswordResultDeclined();

  /**
   * The password reset request was declined.
   *
   * \param[in] retry_date_ Point in time (Unix timestamp) when the password reset can be retried.
   */
  explicit resetPasswordResultDeclined(int32 retry_date_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1202200373;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Describes a text object inside an instant-view web page.
 */
class RichText: public Object {
 public:
};

/**
 * A plain text.
 */
class richTextPlain final : public RichText {
 public:
  /// Text.
  string text_;

  /**
   * A plain text.
   */
  richTextPlain();

  /**
   * A plain text.
   *
   * \param[in] text_ Text.
   */
  explicit richTextPlain(string const &text_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 482617702;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A bold rich text.
 */
class richTextBold final : public RichText {
 public:
  /// Text.
  object_ptr<RichText> text_;

  /**
   * A bold rich text.
   */
  richTextBold();

  /**
   * A bold rich text.
   *
   * \param[in] text_ Text.
   */
  explicit richTextBold(object_ptr<RichText> &&text_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1670844268;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * An italicized rich text.
 */
class richTextItalic final : public RichText {
 public:
  /// Text.
  object_ptr<RichText> text_;

  /**
   * An italicized rich text.
   */
  richTextItalic();

  /**
   * An italicized rich text.
   *
   * \param[in] text_ Text.
   */
  explicit richTextItalic(object_ptr<RichText> &&text_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1853354047;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * An underlined rich text.
 */
class richTextUnderline final : public RichText {
 public:
  /// Text.
  object_ptr<RichText> text_;

  /**
   * An underlined rich text.
   */
  richTextUnderline();

  /**
   * An underlined rich text.
   *
   * \param[in] text_ Text.
   */
  explicit richTextUnderline(object_ptr<RichText> &&text_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -536019572;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A strikethrough rich text.
 */
class richTextStrikethrough final : public RichText {
 public:
  /// Text.
  object_ptr<RichText> text_;

  /**
   * A strikethrough rich text.
   */
  richTextStrikethrough();

  /**
   * A strikethrough rich text.
   *
   * \param[in] text_ Text.
   */
  explicit richTextStrikethrough(object_ptr<RichText> &&text_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 723413585;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A fixed-width rich text.
 */
class richTextFixed final : public RichText {
 public:
  /// Text.
  object_ptr<RichText> text_;

  /**
   * A fixed-width rich text.
   */
  richTextFixed();

  /**
   * A fixed-width rich text.
   *
   * \param[in] text_ Text.
   */
  explicit richTextFixed(object_ptr<RichText> &&text_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1271496249;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A rich text URL link.
 */
class richTextUrl final : public RichText {
 public:
  /// Text.
  object_ptr<RichText> text_;
  /// URL.
  string url_;
  /// True, if the URL has cached instant view server-side.
  bool is_cached_;

  /**
   * A rich text URL link.
   */
  richTextUrl();

  /**
   * A rich text URL link.
   *
   * \param[in] text_ Text.
   * \param[in] url_ URL.
   * \param[in] is_cached_ True, if the URL has cached instant view server-side.
   */
  richTextUrl(object_ptr<RichText> &&text_, string const &url_, bool is_cached_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 83939092;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A rich text email link.
 */
class richTextEmailAddress final : public RichText {
 public:
  /// Text.
  object_ptr<RichText> text_;
  /// Email address.
  string email_address_;

  /**
   * A rich text email link.
   */
  richTextEmailAddress();

  /**
   * A rich text email link.
   *
   * \param[in] text_ Text.
   * \param[in] email_address_ Email address.
   */
  richTextEmailAddress(object_ptr<RichText> &&text_, string const &email_address_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 40018679;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A subscript rich text.
 */
class richTextSubscript final : public RichText {
 public:
  /// Text.
  object_ptr<RichText> text_;

  /**
   * A subscript rich text.
   */
  richTextSubscript();

  /**
   * A subscript rich text.
   *
   * \param[in] text_ Text.
   */
  explicit richTextSubscript(object_ptr<RichText> &&text_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -868197812;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A superscript rich text.
 */
class richTextSuperscript final : public RichText {
 public:
  /// Text.
  object_ptr<RichText> text_;

  /**
   * A superscript rich text.
   */
  richTextSuperscript();

  /**
   * A superscript rich text.
   *
   * \param[in] text_ Text.
   */
  explicit richTextSuperscript(object_ptr<RichText> &&text_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -382241437;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A marked rich text.
 */
class richTextMarked final : public RichText {
 public:
  /// Text.
  object_ptr<RichText> text_;

  /**
   * A marked rich text.
   */
  richTextMarked();

  /**
   * A marked rich text.
   *
   * \param[in] text_ Text.
   */
  explicit richTextMarked(object_ptr<RichText> &&text_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1271999614;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A rich text phone number.
 */
class richTextPhoneNumber final : public RichText {
 public:
  /// Text.
  object_ptr<RichText> text_;
  /// Phone number.
  string phone_number_;

  /**
   * A rich text phone number.
   */
  richTextPhoneNumber();

  /**
   * A rich text phone number.
   *
   * \param[in] text_ Text.
   * \param[in] phone_number_ Phone number.
   */
  richTextPhoneNumber(object_ptr<RichText> &&text_, string const &phone_number_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 128521539;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A small image inside the text.
 */
class richTextIcon final : public RichText {
 public:
  /// The image represented as a document. The image can be in GIF, JPEG or PNG format.
  object_ptr<document> document_;
  /// Width of a bounding box in which the image must be shown; 0 if unknown.
  int32 width_;
  /// Height of a bounding box in which the image must be shown; 0 if unknown.
  int32 height_;

  /**
   * A small image inside the text.
   */
  richTextIcon();

  /**
   * A small image inside the text.
   *
   * \param[in] document_ The image represented as a document. The image can be in GIF, JPEG or PNG format.
   * \param[in] width_ Width of a bounding box in which the image must be shown; 0 if unknown.
   * \param[in] height_ Height of a bounding box in which the image must be shown; 0 if unknown.
   */
  richTextIcon(object_ptr<document> &&document_, int32 width_, int32 height_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1480316158;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A reference to a richTexts object on the same web page.
 */
class richTextReference final : public RichText {
 public:
  /// The text.
  object_ptr<RichText> text_;
  /// The name of a richTextAnchor object, which is the first element of the target richTexts object.
  string anchor_name_;
  /// An HTTP URL, opening the reference.
  string url_;

  /**
   * A reference to a richTexts object on the same web page.
   */
  richTextReference();

  /**
   * A reference to a richTexts object on the same web page.
   *
   * \param[in] text_ The text.
   * \param[in] anchor_name_ The name of a richTextAnchor object, which is the first element of the target richTexts object.
   * \param[in] url_ An HTTP URL, opening the reference.
   */
  richTextReference(object_ptr<RichText> &&text_, string const &anchor_name_, string const &url_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1147530634;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * An anchor.
 */
class richTextAnchor final : public RichText {
 public:
  /// Anchor name.
  string name_;

  /**
   * An anchor.
   */
  richTextAnchor();

  /**
   * An anchor.
   *
   * \param[in] name_ Anchor name.
   */
  explicit richTextAnchor(string const &name_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1316950068;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A link to an anchor on the same web page.
 */
class richTextAnchorLink final : public RichText {
 public:
  /// The link text.
  object_ptr<RichText> text_;
  /// The anchor name. If the name is empty, the link must bring back to top.
  string anchor_name_;
  /// An HTTP URL, opening the anchor.
  string url_;

  /**
   * A link to an anchor on the same web page.
   */
  richTextAnchorLink();

  /**
   * A link to an anchor on the same web page.
   *
   * \param[in] text_ The link text.
   * \param[in] anchor_name_ The anchor name. If the name is empty, the link must bring back to top.
   * \param[in] url_ An HTTP URL, opening the anchor.
   */
  richTextAnchorLink(object_ptr<RichText> &&text_, string const &anchor_name_, string const &url_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1541418282;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A concatenation of rich texts.
 */
class richTexts final : public RichText {
 public:
  /// Texts.
  array<object_ptr<RichText>> texts_;

  /**
   * A concatenation of rich texts.
   */
  richTexts();

  /**
   * A concatenation of rich texts.
   *
   * \param[in] texts_ Texts.
   */
  explicit richTexts(array<object_ptr<RichText>> &&texts_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1647457821;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains information about saved card credentials.
 */
class savedCredentials final : public Object {
 public:
  /// Unique identifier of the saved credentials.
  string id_;
  /// Title of the saved credentials.
  string title_;

  /**
   * Contains information about saved card credentials.
   */
  savedCredentials();

  /**
   * Contains information about saved card credentials.
   *
   * \param[in] id_ Unique identifier of the saved credentials.
   * \param[in] title_ Title of the saved credentials.
   */
  savedCredentials(string const &id_, string const &title_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -370273060;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains information about notification settings for several chats.
 */
class scopeNotificationSettings final : public Object {
 public:
  /// Time left before notifications will be unmuted, in seconds.
  int32 mute_for_;
  /// The name of an audio file to be used for notification sounds; only applies to iOS applications.
  string sound_;
  /// True, if message content must be displayed in notifications.
  bool show_preview_;
  /// True, if notifications for incoming pinned messages will be created as for an ordinary unread message.
  bool disable_pinned_message_notifications_;
  /// True, if notifications for messages with mentions will be created as for an ordinary unread message.
  bool disable_mention_notifications_;

  /**
   * Contains information about notification settings for several chats.
   */
  scopeNotificationSettings();

  /**
   * Contains information about notification settings for several chats.
   *
   * \param[in] mute_for_ Time left before notifications will be unmuted, in seconds.
   * \param[in] sound_ The name of an audio file to be used for notification sounds; only applies to iOS applications.
   * \param[in] show_preview_ True, if message content must be displayed in notifications.
   * \param[in] disable_pinned_message_notifications_ True, if notifications for incoming pinned messages will be created as for an ordinary unread message.
   * \param[in] disable_mention_notifications_ True, if notifications for messages with mentions will be created as for an ordinary unread message.
   */
  scopeNotificationSettings(int32 mute_for_, string const &sound_, bool show_preview_, bool disable_pinned_message_notifications_, bool disable_mention_notifications_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -426103745;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Represents a filter for message search results.
 */
class SearchMessagesFilter: public Object {
 public:
};

/**
 * Returns all found messages, no filter is applied.
 */
class searchMessagesFilterEmpty final : public SearchMessagesFilter {
 public:

  /**
   * Returns all found messages, no filter is applied.
   */
  searchMessagesFilterEmpty();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -869395657;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Returns only animation messages.
 */
class searchMessagesFilterAnimation final : public SearchMessagesFilter {
 public:

  /**
   * Returns only animation messages.
   */
  searchMessagesFilterAnimation();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -155713339;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Returns only audio messages.
 */
class searchMessagesFilterAudio final : public SearchMessagesFilter {
 public:

  /**
   * Returns only audio messages.
   */
  searchMessagesFilterAudio();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 867505275;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Returns only document messages.
 */
class searchMessagesFilterDocument final : public SearchMessagesFilter {
 public:

  /**
   * Returns only document messages.
   */
  searchMessagesFilterDocument();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1526331215;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Returns only photo messages.
 */
class searchMessagesFilterPhoto final : public SearchMessagesFilter {
 public:

  /**
   * Returns only photo messages.
   */
  searchMessagesFilterPhoto();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 925932293;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Returns only video messages.
 */
class searchMessagesFilterVideo final : public SearchMessagesFilter {
 public:

  /**
   * Returns only video messages.
   */
  searchMessagesFilterVideo();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 115538222;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Returns only voice note messages.
 */
class searchMessagesFilterVoiceNote final : public SearchMessagesFilter {
 public:

  /**
   * Returns only voice note messages.
   */
  searchMessagesFilterVoiceNote();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1841439357;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Returns only photo and video messages.
 */
class searchMessagesFilterPhotoAndVideo final : public SearchMessagesFilter {
 public:

  /**
   * Returns only photo and video messages.
   */
  searchMessagesFilterPhotoAndVideo();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1352130963;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Returns only messages containing URLs.
 */
class searchMessagesFilterUrl final : public SearchMessagesFilter {
 public:

  /**
   * Returns only messages containing URLs.
   */
  searchMessagesFilterUrl();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1828724341;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Returns only messages containing chat photos.
 */
class searchMessagesFilterChatPhoto final : public SearchMessagesFilter {
 public:

  /**
   * Returns only messages containing chat photos.
   */
  searchMessagesFilterChatPhoto();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1247751329;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Returns only video note messages.
 */
class searchMessagesFilterVideoNote final : public SearchMessagesFilter {
 public:

  /**
   * Returns only video note messages.
   */
  searchMessagesFilterVideoNote();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 564323321;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Returns only voice and video note messages.
 */
class searchMessagesFilterVoiceAndVideoNote final : public SearchMessagesFilter {
 public:

  /**
   * Returns only voice and video note messages.
   */
  searchMessagesFilterVoiceAndVideoNote();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 664174819;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Returns only messages with mentions of the current user, or messages that are replies to their messages.
 */
class searchMessagesFilterMention final : public SearchMessagesFilter {
 public:

  /**
   * Returns only messages with mentions of the current user, or messages that are replies to their messages.
   */
  searchMessagesFilterMention();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 2001258652;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Returns only messages with unread mentions of the current user, or messages that are replies to their messages. When using this filter the results can&\#039;t be additionally filtered by a query, a message thread or by the sending user.
 */
class searchMessagesFilterUnreadMention final : public SearchMessagesFilter {
 public:

  /**
   * Returns only messages with unread mentions of the current user, or messages that are replies to their messages. When using this filter the results can&\#039;t be additionally filtered by a query, a message thread or by the sending user.
   */
  searchMessagesFilterUnreadMention();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -95769149;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Returns only failed to send messages. This filter can be used only if the message database is used.
 */
class searchMessagesFilterFailedToSend final : public SearchMessagesFilter {
 public:

  /**
   * Returns only failed to send messages. This filter can be used only if the message database is used.
   */
  searchMessagesFilterFailedToSend();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -596322564;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Returns only pinned messages.
 */
class searchMessagesFilterPinned final : public SearchMessagesFilter {
 public:

  /**
   * Returns only pinned messages.
   */
  searchMessagesFilterPinned();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 371805512;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains a value representing a number of seconds.
 */
class seconds final : public Object {
 public:
  /// Number of seconds.
  double seconds_;

  /**
   * Contains a value representing a number of seconds.
   */
  seconds();

  /**
   * Contains a value representing a number of seconds.
   *
   * \param[in] seconds_ Number of seconds.
   */
  explicit seconds(double seconds_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 959899022;
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Represents a secret chat.
 */
class secretChat final : public Object {
 public:
  /// Secret chat identifier.
  int32 id_;
  /// Identifier of the chat partner.
  int53 user_id_;
  /// State of the secret chat.
  object_ptr<SecretChatState> state_;
  /// True, if the chat was created by the current user; otherwise false.
  bool is_outbound_;
  /// Hash of the currently used key for comparison with the hash of the chat partner&\#039;s key. This is a string of 36 little-endian bytes, which must be split into groups of 2 bits, each denoting a pixel of one of 4 colors FFFFFF, D5E6F3, 2D5775, and 2F99C9. The pixels must be used to make a 12x12 square image filled from left to right, top to bottom. Alternatively, the first 32 bytes of the hash can be converted to the hexadecimal format and printed as 32 2-digit hex numbers.
  bytes key_hash_;
  /// Secret chat layer; determines features supported by the chat partner&\#039;s application. Nested text entities and underline and strikethrough entities are supported if the layer &gt;= 101.
  in