/* NOTE:  Throughout this file, I refer to SMB servers and SMB generically,
 * though this is Samba-specific stuff.  Also, this release expects that
 * you are running Red Hat Linux (or a derivative) with a "smb" start/stop
 * script in /etc/rc.d/init.d and smb.conf in /etc */

#include <Xm/Xm.h>
#include <Xm/Form.h>
#include <Xm/Label.h>
#include <Xm/TextF.h>
#include <Xm/PushB.h>
#include <Xm/SelectioB.h>
#include <Xm/RowColumn.h>
#include <Xm/ToggleB.h>
#include <Xm/BulletinB.h>
#include <Xm/MessageB.h>
#include <Xm/Notebook.h>

#include <stdio.h>

#include "smbfield.h"

#define SMB_ENABLE  100
#define SMB_DISABLE 200

//extern Widget top_level;
Widget smb_form, smb_notebook;

/* These hold smb.conf entries */
struct smb_global smb_global_section;
struct smb_printer smb_printers_section;
struct smb_share smb_homes_section, smb_tmp_dir, *smb_all_shares;
/* Total number of directories shared, not counting /tmp */
int smb_n_shares = 0;

/* These hold [global] config options:
 * smb_g_t1 is "allow hosts", smb_g_t2 is "encrypt passwords", smb_g_t3 is
 * "load printers", smb_g_t4 is "password server", smb_g_t5[a-c] are
 * "security", smb_g_t6 is "workgroup".
 */ 
Widget smb_g_t1, smb_g_t2, smb_g_t3, smb_g_t4, smb_g_t5a, smb_g_t5b, smb_g_t5c,
       smb_g_t6;

/* These hold [printers] config options:
 * smb_p_t1 is "comment", smb_p_t2 is "browseable", smb_p_t3 is "public",
 * smb_p_t4 is "printable", smb_p_t5 is "writable", smb_p_t6 is "create mode"
 */
Widget smb_p_t1, smb_p_t2, smb_p_t3, smb_p_t4, smb_p_t5, smb_p_t6;

/* These hold [homes] config options:
 * smb_h_t1 is "comment", smb_h_t2 is "browseable", smb_h_t3 is "read only",
 * smb_h_t4 is "create mode"
 */
Widget smb_h_t1, smb_h_t2, smb_h_t3, smb_h_t4;

/* Should we share /tmp ? */
Widget smb_tmp_share;

int smb_is_running = 0;
int smb_conf_exists = 0;
int redhat_init = 0;
int tmp_is_shared = 0;

/* Function prototypes */
void read_smb_config();
void smb_restart();

/* scan_for_smb checks for init scripts and lock files a la Red Hat Linux.
 * Should modify this for better portability... 
 * If smb is there (and running!), read the smb.conf. */
void scan_for_smb() {
  FILE *testf;

  /* Attempt to read smb.conf */
  read_smb_config();

  testf = fopen("/var/lock/subsys/smb","r");
  if (testf != NULL) {
    smb_is_running = 1;
    fclose(testf);
  }
  testf = fopen("/etc/rc.d/init.d/smb","r");
  if (testf != NULL) {
    redhat_init = 1;
    fclose(testf);
  }
}

/* Parse /etc/smb.conf.  It is expected that smb.conf options have spaces
 * before and after the equals sign! */
void read_smb_config() {
  FILE *smbconfig;
  char *thisline, *obfuscate, temp_1[10], temp_2[10], temp_3[10], temp_4[60];
  int n_section = 0, n_shares = -1, this_is_new = 0, x = 0, y = 0;

  thisline = (char *)malloc(sizeof(char) * 80);
  smb_all_shares = malloc(sizeof(struct smb_share));
  smbconfig = fopen("/etc/smb.conf", "r");
  if (smbconfig == NULL)
    return;
  smb_conf_exists = 1;
  /* Load some sort of default to fill in values defined implicitly */
  smb_global_section.security_user = False;
  smb_global_section.security_share = True;
  smb_global_section.security_server = False;
  smb_global_section.password_server = (char *)malloc(sizeof(char) * 2);
  smb_global_section.password_server[0] = '\0';
  smb_global_section.allow_hosts = (char *)malloc(sizeof(char) * 2);
  smb_global_section.allow_hosts[0] = '\0';
  smb_global_section.workgroup = (char *)malloc(sizeof(char) * 2);
  smb_global_section.workgroup[0] = '\0';
  smb_printers_section.comment = (char *)malloc(sizeof(char) * 2);
  smb_printers_section.comment[0] = '\0';
  smb_printers_section.create_mode = (char *)malloc(sizeof(char) * 2);
  smb_printers_section.create_mode[0] = '\0';
  smb_homes_section.comment = (char *)malloc(sizeof(char) * 2);
  smb_homes_section.comment[0] = '\0';
  smb_homes_section.create_mode = (char *)malloc(sizeof(char) * 2);
  smb_homes_section.create_mode[0] = '\0';
  while (!feof(smbconfig)) {
    fgets(thisline, 80, smbconfig);
    /* Strip \n at end of line */
    thisline[strlen(thisline)-1]='\0';

    /* Ignore comments */
    if (thisline[0] == ';') {
      y++;
      continue;
    }
    /* Parse [sharenames].  This sets n_section to an appropriate value,
    * depending on which type of section is being read */
    if (thisline[0] == '[') {
      x++;
      if (strstr(thisline, "[global]") != NULL) {
	n_section = 1;
	this_is_new = 1;
      }
      if (strstr(thisline, "[printers]") != NULL) {
	n_section = 2;
	this_is_new = 1;
      }
      if (strstr(thisline, "[homes]") != NULL) {
	n_section = 3;
	this_is_new = 1;
      }
      if (strstr(thisline, "[tmp]") != NULL) {
	n_section = 4;
	tmp_is_shared = 1;
	this_is_new = 1;
      }
      if (this_is_new == 0) {
	n_section = 5;
	this_is_new = 1;
	smb_n_shares++;
	n_shares++;
	smb_all_shares=realloc(smb_all_shares, 
			       sizeof(struct smb_share)*smb_n_shares);
	smb_all_shares[n_shares].comment = (char *)malloc(sizeof(char) * 2);
	smb_all_shares[n_shares].comment[0] = '\0';
	smb_all_shares[n_shares].path = (char *)malloc(sizeof(char) * 2);
	smb_all_shares[n_shares].path[0] = '\0';
	smb_all_shares[n_shares].create_mode=(char *)malloc(sizeof(char) * 2);
	smb_all_shares[n_shares].create_mode[0] = '\0';
	smb_all_shares[n_shares].valid_users=(char *)malloc(sizeof(char) * 2);
	smb_all_shares[n_shares].valid_users[0] = '\0';
	smb_all_shares[n_shares].write_list=(char *)malloc(sizeof(char) * 2);
	smb_all_shares[n_shares].write_list[0] = '\0';
	smb_all_shares[n_shares].printer=(char *)malloc(sizeof(char) * 2);
	smb_all_shares[n_shares].printer[0] = '\0';
	smb_all_shares[n_shares].share_name = (char *)malloc(sizeof(char) *
							     strlen(thisline));
	strncpy(smb_all_shares[n_shares].share_name, &thisline[1],
		strlen(thisline)-2);
	smb_all_shares[n_shares].share_name[strlen(thisline)-2] = '\0';
#ifdef DEBUG
	printf("%s\n", smb_all_shares[n_shares].share_name);
#endif
      }
      this_is_new = 0;
      continue;
    }
    /* Parse out variable settings */
    /* These ones are in the [global] section */
    /* These variables can be in any section */
    if ((strstr(thisline, "allow hosts") != NULL) || 
	(strstr(thisline, "hosts allow") != NULL)) {
      if (n_section == 1) {
	obfuscate = (char *)malloc(sizeof(char) * (strlen(thisline)-8));
	strcpy(obfuscate, strstr(thisline, "="));
	smb_global_section.allow_hosts=(char *)realloc(smb_global_section.allow_hosts, sizeof(char) * (strlen(thisline)-8));
	strcpy(smb_global_section.allow_hosts, &obfuscate[1]);
	free(obfuscate);
      }
      continue;
    }
    if (strstr(thisline, "encrypt passwords") != NULL) {
      sscanf(thisline, "%s %s %s %s", &temp_1, &temp_2, &temp_3, &temp_4);
      if (n_section == 1)
	if (strstr(temp_4, "yes") != NULL)
	  smb_global_section.encrypt_passwords = True; 
      continue;
    }
    if (strstr(thisline, "load printers") != NULL) {
      sscanf(thisline, "%s %s %s %s", &temp_1, &temp_2, &temp_3, &temp_4);
      if (n_section == 1)
	if (strstr(temp_4, "yes") != NULL)
	  smb_global_section.load_printers = True; 
      continue;
    }
    if (strstr(thisline, "password server") != NULL) {
      sscanf(thisline, "%s %s %s %s", &temp_1, &temp_2, &temp_3, &temp_4);
      if (n_section == 1) {
	obfuscate = (char *)malloc(sizeof(char) * (strlen(thisline)-8));
	strcpy(obfuscate, strstr(thisline, "="));
	smb_global_section.password_server=(char *)realloc(smb_global_section.password_server, sizeof(char) * (strlen(thisline)-8));
	strcpy(smb_global_section.password_server, &obfuscate[1]);
	free(obfuscate);
      }
      continue;
    }    
    if (strstr(thisline, "workgroup") != NULL) {
      sscanf(thisline, "%s %s %s", &temp_1, &temp_2, &temp_4);
      if (n_section == 1) {
	obfuscate = (char *)malloc(sizeof(char) * (strlen(thisline)-8));
	strcpy(obfuscate, strstr(thisline, "="));
	smb_global_section.workgroup=(char *)realloc(smb_global_section.workgroup, sizeof(char) * (strlen(thisline)-8));
	strcpy(smb_global_section.workgroup, &obfuscate[1]);
	free(obfuscate);
      }
      continue;
    }
    if (strstr(thisline, "security") != NULL) {
      sscanf(thisline, "%s %s %s", &temp_1, &temp_2, &temp_4);
      if (n_section == 1) {
	smb_global_section.security_share = False;
	if (strstr(thisline, "user"))
	  smb_global_section.security_user = True;
	if (strstr(thisline, "share"))
	  smb_global_section.security_share = True;
	if (strstr(thisline, "server"))
	  smb_global_section.security_server = True;
      }
      continue;
    }
    /* These variables can be in any section */
    if (strstr(thisline, "comment") != NULL) {
      obfuscate = (char *)malloc(sizeof(char) * (strlen(thisline)-8));
      strcpy(obfuscate, strstr(thisline, "="));
      switch (n_section) {
      case 2:
	smb_printers_section.comment = (char *)realloc(smb_printers_section.comment, sizeof(char) * (strlen(thisline)-8));
	strcpy(smb_printers_section.comment, &obfuscate[2]);
	break;
      case 3:
	smb_homes_section.comment = (char *)realloc(smb_homes_section.comment, sizeof(char) * (strlen(thisline)-8));
	strcpy(smb_homes_section.comment, &obfuscate[2]);
	break;
      case 4:
	smb_tmp_dir.comment = (char *)realloc(smb_tmp_dir.comment, sizeof(char) * (strlen(thisline)-8));
	strcpy(smb_tmp_dir.comment, &obfuscate[2]);
	break;
      case 5:
	smb_all_shares[n_shares].comment = (char *)realloc(smb_all_shares[n_shares].comment, sizeof(char) * (strlen(thisline)-8));
	strcpy(smb_all_shares[n_shares].comment, &obfuscate[2]);
	break;
      }
      free(obfuscate);
      continue;
    }
    if (strstr(thisline, "browseable") != NULL) {
      sscanf(thisline, "%s %s %s", &temp_1, &temp_2, &temp_3);
      switch (n_section) {
      case 2:
	if (strstr(temp_3, "yes") != NULL)
	  smb_printers_section.browseable = True;
	else
	  smb_printers_section.browseable = False;
	break;
      case 3:
	if (strstr(temp_3, "yes") != NULL)
	  smb_homes_section.browseable = True;
	else
	  smb_homes_section.browseable = False;
	break;
      case 4:
	if (strstr(temp_3, "yes") != NULL)
	  smb_tmp_dir.browseable = True;
	else
	  smb_tmp_dir.browseable = False;
	break;
      case 5:
	if (strstr(temp_3, "yes") != NULL)
	  smb_all_shares[n_shares].browseable = True;
	else
	  smb_all_shares[n_shares].browseable = False;
	break;
      }
      continue;
    }
    if (strstr(thisline, "public") != NULL) {
      sscanf(thisline, "%s %s %s", &temp_1, &temp_2, &temp_3);
      switch (n_section) {
      case 2:
	if (strstr(temp_3, "yes") != NULL)
	  smb_printers_section.bpublic = True;
	else
	  smb_printers_section.bpublic = False;
	break;
      case 3:
	if (strstr(temp_3, "yes") != NULL)
	  smb_homes_section.bpublic = True;
	else
	  smb_homes_section.bpublic = False;
	break;
      case 4:
	if (strstr(temp_3, "yes") != NULL)
	  smb_tmp_dir.bpublic = True;
	else
	  smb_tmp_dir.bpublic = False;
	break;
      case 5:
	if (strstr(temp_3, "yes") != NULL)
	  smb_all_shares[n_shares].bpublic = True;
	else
	  smb_all_shares[n_shares].bpublic = False;
	break;
      }
      continue;
    }
    if (strstr(thisline, "writable") != NULL) {
      sscanf(thisline, "%s %s %s", &temp_1, &temp_2, &temp_3);
      switch (n_section) {
      case 2:
	if (strstr(temp_3, "yes") != NULL)
	  smb_printers_section.writable = True;
	else
	  smb_printers_section.writable = False;
	break;
      case 3:
	if (strstr(temp_3, "yes") != NULL)
	  smb_homes_section.writable = True;
	else
	  smb_homes_section.writable = False;
	break;
      case 4:
	if (strstr(temp_3, "yes") != NULL)
	  smb_tmp_dir.writable = True;
	else
	  smb_tmp_dir.writable = False;
	break;
      case 5:
	if (strstr(temp_3, "yes") != NULL)
	  smb_all_shares[n_shares].writable = True;
	else
	  smb_all_shares[n_shares].writable = False;
	break;
      }
      continue;
    }
    if (strstr(thisline, "printable") != NULL) {
      sscanf(thisline, "%s %s %s", &temp_1, &temp_2, &temp_3);
      switch (n_section) {
      case 2:
	if (strstr(temp_3, "yes") != NULL)
	  smb_printers_section.printable = True;
	else
	  smb_printers_section.printable = False;
	break;
      case 3:
	if (strstr(temp_3, "yes") != NULL)
	  smb_homes_section.printable = True;
	else
	  smb_homes_section.printable = False;
	break;
      case 4:
	if (strstr(temp_3, "yes") != NULL)
	  smb_tmp_dir.printable = True;
	else
	  smb_tmp_dir.printable = False;
	break;
      case 5:
	if (strstr(temp_3, "yes") != NULL)
	  smb_all_shares[n_shares].printable = True;
	else
	  smb_all_shares[n_shares].printable = False;
	break;
      }
      continue;
    }
    if (strstr(thisline, "read only") != NULL) {
      sscanf(thisline, "%s %s %s %s", &temp_1, &temp_2, &temp_4, &temp_3);
      switch (n_section) {
      case 3:
	if (strstr(temp_3, "yes") != NULL)
	  smb_homes_section.read_only = True;
	else
	  smb_homes_section.read_only = False;
	break;
      case 4:
	if (strstr(temp_3, "yes") != NULL)
	  smb_tmp_dir.read_only = True;
	else
	  smb_tmp_dir.read_only = False;
	break;
      case 5:
	if (strstr(temp_3, "yes") != NULL)
	  smb_all_shares[n_shares].read_only = True;
	else
	  smb_all_shares[n_shares].read_only = False;
	break;
      }
      continue;
    }
    if (strstr(thisline, "create mode") != NULL) {
      sscanf(thisline, "%s %s %s %s", &temp_1, &temp_2, &temp_3, &temp_4);
      switch (n_section) {
      case 2:
        smb_printers_section.create_mode = (char *)malloc(sizeof(char) * 
							  strlen(temp_4));
        strcpy(smb_printers_section.create_mode, temp_4);
        break;
      case 3:
        smb_homes_section.create_mode = (char *)malloc(sizeof(char) * 
						       strlen(temp_4));
        strcpy(smb_homes_section.create_mode, temp_4);
        break;
      case 4:
        smb_tmp_dir.create_mode = (char *)malloc(sizeof(char) * 
						 strlen(temp_4));
        strcpy(smb_tmp_dir.create_mode, temp_4);
	break;
      case 5:
	smb_all_shares[n_shares].create_mode = (char *)malloc(sizeof(char) * 
							      strlen(temp_4));
	strcpy(smb_all_shares[n_shares].create_mode, temp_4);
	break;
      }
      continue;
    }
    /* These turn up only in smb_all_shares */
    if (strstr(thisline, "path") != NULL) {
      obfuscate = (char *)malloc(sizeof(char) * (strlen(thisline)-8));
      strcpy(obfuscate, strstr(thisline, "="));
      if (n_section == 5) {
	smb_all_shares[n_shares].path = (char *)realloc(smb_all_shares[n_shares].path, sizeof(char) * (strlen(thisline)-8));
	strcpy(smb_all_shares[n_shares].path, &obfuscate[2]);
      }
      free(obfuscate);
      continue;
    }
    if (strstr(thisline, "write list") != NULL) {
      obfuscate = (char *)malloc(sizeof(char) * (strlen(thisline)-8));
      strcpy(obfuscate, strstr(thisline, "="));
      if (n_section == 5) {
	smb_all_shares[n_shares].write_list = (char *)realloc(smb_all_shares[n_shares].write_list, sizeof(char) * (strlen(thisline)-8));
	strcpy(smb_all_shares[n_shares].write_list, &obfuscate[2]);
      }
      free(obfuscate);
      continue;
    }
    if (strstr(thisline, "valid users") != NULL) {
      obfuscate = (char *)malloc(sizeof(char) * (strlen(thisline)-8));
      strcpy(obfuscate, strstr(thisline, "="));
      if (n_section == 5) {
	smb_all_shares[n_shares].valid_users = (char *)realloc(smb_all_shares[n_shares].valid_users, sizeof(char) * (strlen(thisline)-8));
	strcpy(smb_all_shares[n_shares].valid_users, &obfuscate[2]);
      }
      free(obfuscate);
      continue;
    }
    if (strstr(thisline, "printer") != NULL) {
      obfuscate = (char *)malloc(sizeof(char) * (strlen(thisline)-8));
      strcpy(obfuscate, strstr(thisline, "="));
      if (n_section == 5) {
	smb_all_shares[n_shares].printer = (char *)realloc(smb_all_shares[n_shares].printer, sizeof(char) * (strlen(thisline)-8));
	strcpy(smb_all_shares[n_shares].printer, &obfuscate[2]);
      }
      free(obfuscate);
      continue;
    }
  }

#ifdef DEBUG
  printf("%d sections.\n%d comments.\n", x, y);
#endif
  fclose(smbconfig);
  free(thisline);
}

/* read the actual values from the widgets. */
void smb_get_actual() {
  XtVaGetValues(smb_g_t1, XmNvalue, &smb_global_section.allow_hosts, NULL);
  XtVaGetValues(smb_g_t2, XmNset, &smb_global_section.encrypt_passwords, NULL);
  XtVaGetValues(smb_g_t3, XmNset, &smb_global_section.load_printers, NULL);
  XtVaGetValues(smb_g_t4, XmNvalue, &smb_global_section.password_server, NULL);
  XtVaGetValues(smb_g_t5a, XmNset, &smb_global_section.security_user, NULL);
  XtVaGetValues(smb_g_t5b, XmNset, &smb_global_section.security_share, NULL);
  XtVaGetValues(smb_g_t5c, XmNset, &smb_global_section.security_server, NULL);
  XtVaGetValues(smb_g_t6, XmNvalue, &smb_global_section.workgroup, NULL);
  XtVaGetValues(smb_p_t1, XmNvalue, &smb_printers_section.comment, NULL);
  XtVaGetValues(smb_p_t2, XmNset, &smb_printers_section.browseable, NULL);
  XtVaGetValues(smb_p_t3, XmNset, &smb_printers_section.bpublic, NULL);
  XtVaGetValues(smb_p_t4, XmNset, &smb_printers_section.printable, NULL);
  XtVaGetValues(smb_p_t5, XmNset, &smb_printers_section.writable, NULL);
  XtVaGetValues(smb_p_t6, XmNvalue, &smb_printers_section.create_mode, NULL);
  XtVaGetValues(smb_h_t1, XmNvalue, &smb_homes_section.comment, NULL);
  XtVaGetValues(smb_h_t2, XmNset, &smb_homes_section.browseable, NULL);
  XtVaGetValues(smb_h_t3, XmNset, &smb_homes_section.read_only, NULL);
  XtVaGetValues(smb_h_t4, XmNvalue, &smb_homes_section.create_mode, NULL);
  XtVaGetValues(smb_tmp_share, XmNset, &tmp_is_shared, NULL);
}

/* Support function.  Return (char *)"yes" for (Boolean)True.
 * Same for False, but return "no" in that case. */
char *yn(Boolean tf) {
  if (tf)
    return "yes";
  else
    return "no";
}

/* Having said that, let's define a way to write a new smb.conf */
void write_smb_config(int calledby) {
  FILE *new_smb_conf;
  int x;

  system("rm /etc/smb.conf.old");
  system("mv /etc/smb.conf /etc/smb.conf.old");
  new_smb_conf = fopen("/etc/smb.conf", "a");
 
  /* Read widget values, if called from server config widget */
  if (calledby == 0)
    smb_get_actual();
  fprintf(new_smb_conf, "; smb.conf -- Configuration file for Samba SMB server.\n");
  fprintf(new_smb_conf, "; This file generated by bxfm\n");
  fprintf(new_smb_conf, "; You may wish to edit log file settings, etc. to suit\n");
  fprintf(new_smb_conf, "; your own needs.  Odds are good, though, that you should\n");
  fprintf(new_smb_conf, "; look at 'smbconfig.c' in the bxfm source distribution\n");
  fprintf(new_smb_conf, "; and adjust the function 'void write_smb_config()' so\n");
  fprintf(new_smb_conf, "; bxfm doesn't overwrite them when you make changes...\n");
  /* Print the [global] section first */
  fprintf(new_smb_conf, "[global]\n");
  if (strlen(smb_global_section.allow_hosts) > 1)
    fprintf(new_smb_conf, "allow hosts = %s\n", smb_global_section.allow_hosts);
  fprintf(new_smb_conf, "encrypt passwords = %s\n", 
	  yn(smb_global_section.encrypt_passwords));
  fprintf(new_smb_conf, "load printers = %s\n", 
	  yn(smb_global_section.load_printers));
  /* May also need to change this to suit your needs.  This sets Samba to use
   * "lpr -P{printer}", lpq, lprm.  Good enough for Red Hat Linux */
  fprintf(new_smb_conf, "printing = bsd\n");
  if (strlen(smb_global_section.password_server) > 1)
    fprintf(new_smb_conf, "password server = %s\n", smb_global_section.password_server);
  if (strlen(smb_global_section.workgroup) > 1)
    fprintf(new_smb_conf, "workgroup = %s\n", smb_global_section.workgroup);
  if (smb_global_section.security_user)
    fprintf(new_smb_conf, "security = user\n");
  if (smb_global_section.security_share)
    fprintf(new_smb_conf, "security = share\n");
  if (smb_global_section.security_server)
    fprintf(new_smb_conf, "security = server\n");
  /* Change these for different lock file and log settings */
  fprintf(new_smb_conf, "log file = /var/log/samba-log.%%m\n");
  fprintf(new_smb_conf, "lock directory = /var/lock/samba\n");
  fprintf(new_smb_conf, "\n");
  /* Now do the [printers] section */
  fprintf(new_smb_conf, "[printers]\n");
  if (strlen(smb_printers_section.comment) > 1)
    fprintf(new_smb_conf, "comment = %s\n", smb_printers_section.comment);
  fprintf(new_smb_conf, "browseable = %s\n", yn(smb_printers_section.browseable));
  fprintf(new_smb_conf, "printable = yes\n");
  fprintf(new_smb_conf, "public = %s\n", yn(smb_printers_section.bpublic));
  fprintf(new_smb_conf, "writable = %s\n", yn(smb_printers_section.writable));
  if (strlen(smb_printers_section.create_mode) > 1)
    fprintf(new_smb_conf, "create mode = %s\n", smb_printers_section.create_mode);
  fprintf(new_smb_conf, "\n");
  /* Now do the [homes] section */
  fprintf(new_smb_conf, "[homes]\n");
  if (strlen(smb_homes_section.comment) > 1)
    fprintf(new_smb_conf, "comment = %s\n", smb_homes_section.comment);
  fprintf(new_smb_conf, "browseable = %s\n", yn(smb_homes_section.browseable));
  fprintf(new_smb_conf, "printable = no\n");
  fprintf(new_smb_conf, "public = %s\n", yn(smb_homes_section.bpublic));
  fprintf(new_smb_conf, "writable = %s\n", yn(smb_homes_section.writable));
  if (strlen(smb_homes_section.create_mode) > 1)
    fprintf(new_smb_conf, "create mode = %s\n", smb_homes_section.create_mode);
  fprintf(new_smb_conf, "\n");
  /* If /tmp is to be shared, print this standard setup. */
  if (tmp_is_shared) {
    fprintf(new_smb_conf, "[tmp]\n");
    fprintf(new_smb_conf, "comment = Temporary file space\n");
    fprintf(new_smb_conf, "path = /tmp\n");
    fprintf(new_smb_conf, "read only = no\n");
    fprintf(new_smb_conf, "public = yes\n");
    fprintf(new_smb_conf, "create mode = 0755\n");
    fprintf(new_smb_conf, "\n");
  }
  /* Now, iterate through the shares. */
  for (x=0; x<smb_n_shares; x++) {
    fprintf(new_smb_conf, "[%s]\n", smb_all_shares[x].share_name);
    if (strlen(smb_all_shares[x].comment) > 1)
      fprintf(new_smb_conf, "comment = %s\n", smb_all_shares[x].comment);
    if (strlen(smb_all_shares[x].path) > 1)
      fprintf(new_smb_conf, "path = %s\n", smb_all_shares[x].path);
    if (strlen(smb_all_shares[x].printer) > 1)
      fprintf(new_smb_conf, "printer = %s\n", smb_all_shares[x].printer);
    if (strlen(smb_all_shares[x].valid_users) > 1)
      fprintf(new_smb_conf, "valid users = %s\n", smb_all_shares[x].valid_users);
    if (strlen(smb_all_shares[x].write_list) > 1)
      fprintf(new_smb_conf, "write list = %s\n", smb_all_shares[x].write_list);
    fprintf(new_smb_conf, "browseable = %s\n", yn(smb_all_shares[x].browseable));
    fprintf(new_smb_conf, "printable = %s\n", yn(smb_all_shares[x].printable));
    fprintf(new_smb_conf, "public = %s\n", yn(smb_all_shares[x].bpublic));
    fprintf(new_smb_conf, "writable = %s\n", yn(smb_all_shares[x].writable));
    fprintf(new_smb_conf, "read only = %s\n", yn(smb_all_shares[x].read_only));
    if (strlen(smb_all_shares[x].create_mode) > 1)
      fprintf(new_smb_conf, "path = %s", smb_all_shares[x].create_mode);
    fprintf(new_smb_conf, "\n");
  }

  fclose(new_smb_conf);
  /* Need to restart Samba now */
  smb_restart();
}

void write_CB(Widget w, XtPointer client_data, XtPointer call_data) {
  XtDestroyWidget((Widget) client_data);
  write_smb_config(0);
}

/* Load values read from smb.conf (and stored in memory) into widgets. */
void smb_set_actual() {
  XtVaSetValues(smb_g_t1, XmNvalue, smb_global_section.allow_hosts, NULL);
  XtVaSetValues(smb_g_t2, XmNset, smb_global_section.encrypt_passwords, NULL);
  XtVaSetValues(smb_g_t3, XmNset, smb_global_section.load_printers, NULL);
  XtVaSetValues(smb_g_t4, XmNvalue, smb_global_section.password_server, NULL);
  XtVaSetValues(smb_g_t5a, XmNset, smb_global_section.security_user, NULL);
  XtVaSetValues(smb_g_t5b, XmNset, smb_global_section.security_share, NULL);
  XtVaSetValues(smb_g_t5c, XmNset, smb_global_section.security_server, NULL);
  XtVaSetValues(smb_g_t6, XmNvalue, smb_global_section.workgroup, NULL);
  XtVaSetValues(smb_p_t1, XmNvalue, smb_printers_section.comment, NULL);
  XtVaSetValues(smb_p_t2, XmNset, smb_printers_section.browseable, NULL);
  XtVaSetValues(smb_p_t3, XmNset, smb_printers_section.bpublic, NULL);
  XtVaSetValues(smb_p_t4, XmNset, smb_printers_section.printable, NULL);
  XtVaSetValues(smb_p_t5, XmNset, smb_printers_section.writable, NULL);
  XtVaSetValues(smb_p_t6, XmNvalue, smb_printers_section.create_mode, NULL);
  XtVaSetValues(smb_h_t1, XmNvalue, smb_homes_section.comment, NULL);
  XtVaSetValues(smb_h_t2, XmNset, smb_homes_section.browseable, NULL);
  XtVaSetValues(smb_h_t3, XmNset, smb_homes_section.read_only, NULL);
  XtVaSetValues(smb_h_t4, XmNvalue, smb_homes_section.create_mode, NULL);
  XtVaSetValues(smb_tmp_share, XmNset, False, NULL);
}

/* Load default values into widgets.  Note that this doesn't change the
 * actual SMB settings in memory or in smb.conf -- must press OK to commit. */
void smb_set_defaults() {
  XtVaSetValues(smb_g_t2, XmNset, True, NULL);
  XtVaSetValues(smb_g_t3, XmNset, True, NULL);
  XtVaSetValues(smb_g_t5a, XmNset, False, NULL);
  XtVaSetValues(smb_g_t5b, XmNset, True, NULL);
  XtVaSetValues(smb_g_t5c, XmNset, False, NULL);
  XtVaSetValues(smb_g_t6, XmNvalue, "WORKGROUP", NULL);
  XtVaSetValues(smb_p_t1, XmNvalue, "All printers", NULL);
  XtVaSetValues(smb_p_t2, XmNset, False, NULL);
  XtVaSetValues(smb_p_t3, XmNset, True, NULL);
  XtVaSetValues(smb_p_t4, XmNset, True, NULL);
  XtVaSetValues(smb_p_t5, XmNset, True, NULL);
  XtVaSetValues(smb_p_t6, XmNvalue, "0700", NULL);
  XtVaSetValues(smb_h_t1, XmNvalue, "Home directory", NULL);
  XtVaSetValues(smb_h_t2, XmNset, False, NULL);
  XtVaSetValues(smb_h_t3, XmNset, True, NULL);
  XtVaSetValues(smb_h_t4, XmNvalue, "0644", NULL);
  XtVaSetValues(smb_tmp_share, XmNset, False, NULL);
}

/* Load defaults when user presses button. */
void SMB_defaults_CB(Widget w, XtPointer client_data, XtPointer call_data)
  {
    smb_set_defaults();
  }

void SMB_Cancel(Widget w, XtPointer client_data, XtPointer call_data) {
  XtDestroyWidget((Widget)client_data);
}

void SMB_server_toggles(Widget w, XtPointer client_data, XtPointer call_data)
{
  if ((int) client_data == SMB_ENABLE)
    XtSetSensitive(smb_notebook, True);
  else
    XtSetSensitive(smb_notebook, False);
}

void SMB_server_config() {
  Widget smb_shell, smb_rc, smb_radio1, smb_radio2, smb_test, smb_ok,
         smb_cancel, smb_help, smb_p1, smb_p2, smb_p3, smb_t1, smb_t2, smb_t3,
         smb_load_default;
  /* [global] label and container widgets */
  Widget smb_g_l1, smb_g_l2, smb_g_l3, smb_g_rc1, smb_g_l4;
  /* [printers] label and container widgets */
  Widget smb_p_l1, smb_p_l2;
  /* [homes] label and container widgets */
  Widget smb_h_l1, smb_h_l2;
  XmString x1, x2, x3;
  Boolean smb_on, smb_off;

  /* Must call scan_for_smb to find out if Samba or other SMB server is
   * running. */
  if (smb_is_running)
    { smb_on = True; smb_off = False; }
  else
    { smb_on = False; smb_off = True; }

//  smb_shell = XtVaCreatePopupShell("SMB_server_config_popup",
//				   vendorShellWidgetClass, top_level,
//				   XmNtitle, "Configure SMB Server",
//				   XmNwidth, 480, NULL);

  /* The main area is an XmForm to have tight packing */
  smb_form = XtVaCreateManagedWidget("SMB_topform", xmFormWidgetClass,
				    smb_shell, NULL);

  /* Radio buttons (disable/enable) inside a RowColumn.  Callback to
   * SMB_server_toggles to set sensitivity of the configuration XmNotebook */
  smb_rc = XtVaCreateManagedWidget("SMB_server_rc", xmRowColumnWidgetClass,
				   smb_form, XmNorientation, XmVERTICAL,
				   XmNradioBehavior, True,
				   XmNtopAttachment, XmATTACH_FORM,
				   XmNleftAttachment, XmATTACH_FORM,
				   XmNrightAttachment, XmATTACH_FORM,
				   NULL);
  x1 = XmStringCreateSimple("Disable SMB server");
  smb_radio1 = XtVaCreateManagedWidget("SMB_toggle_off", 
				       xmToggleButtonWidgetClass,
				       smb_rc, XmNset, smb_off,
				       XmNlabelString, x1, NULL);
  XtAddCallback(smb_radio1, XmNarmCallback, SMB_server_toggles,
		(XtPointer) SMB_DISABLE);
  x2 = XmStringCreateSimple("Enable SMB server");
  smb_radio2 = XtVaCreateManagedWidget("SMB_toggle_on", 
				       xmToggleButtonWidgetClass,
				       smb_rc, XmNset, smb_on,
				       XmNlabelString, x2, NULL);
  XtAddCallback(smb_radio2, XmNarmCallback, SMB_server_toggles,
		(XtPointer) SMB_ENABLE);

  /* Create XmNotebook and pages (the pages are XmBulletinBoards) */
  smb_notebook = XtVaCreateManagedWidget("SMB_notebook",
					 xmNotebookWidgetClass, smb_form,
					 XmNbackPagePlacement, XmTOP_RIGHT,
					 XmNorientation, XmVERTICAL,
					 XmNbindingType, XmSOLID,
					 XmNbindingWidth, 10, 
					 XmNtopAttachment, XmATTACH_WIDGET,
					 XmNtopWidget, smb_rc,
					 XmNleftAttachment, XmATTACH_FORM,
					 XmNrightAttachment, XmATTACH_FORM,
					 NULL);
  smb_p1 = XtVaCreateManagedWidget("SMB_page", xmBulletinBoardWidgetClass,
				   smb_notebook, XmNnotebookChildType, XmPAGE,
				   XmNheight, 250,
				   NULL);
  smb_t1 = XtVaCreateManagedWidget("Global", xmPushButtonWidgetClass, 
				   smb_notebook, NULL);
  smb_p2 = XtVaCreateManagedWidget("SMB_page", xmBulletinBoardWidgetClass,
				   smb_notebook, XmNnotebookChildType, XmPAGE,
				   XmNheight, 250,
				   NULL);
  smb_t2 = XtVaCreateManagedWidget("Printers", xmPushButtonWidgetClass, 
				   smb_notebook, NULL);
  smb_p3 = XtVaCreateManagedWidget("SMB_page", xmBulletinBoardWidgetClass,
				   smb_notebook, XmNnotebookChildType, XmPAGE,
				   XmNheight, 250,
				   NULL);
  smb_t3 = XtVaCreateManagedWidget("Home Dirs", xmPushButtonWidgetClass, 
				   smb_notebook, NULL);

  /* Page 1: Global options.  Only a subset of available options is
   * supported, sorry. */
  smb_g_l1 = XtVaCreateManagedWidget("Allow hosts:", xmLabelWidgetClass,
				     smb_p1, XmNx, 20, XmNy, 16, NULL);
  smb_g_t1 = XtVaCreateManagedWidget("smb_g_t1", xmTextFieldWidgetClass,
				     smb_p1, XmNx, 140, XmNy, 10, NULL);
  smb_g_t2 = XtVaCreateManagedWidget("Encrypt Passwords",
				     xmToggleButtonWidgetClass, smb_p1,
				     XmNx, 20, XmNy, 42, NULL);
  smb_g_t3 = XtVaCreateManagedWidget("Load printers from /etc/printcap", 
				     xmToggleButtonWidgetClass, smb_p1,
				     XmNx, 20, XmNy, 162, NULL);
  smb_g_l2 = XtVaCreateManagedWidget("Password server:", xmLabelWidgetClass,
				     smb_p1, XmNx, 20, XmNy, 75, NULL);
  smb_g_t4 = XtVaCreateManagedWidget("smb_g_t1", xmTextFieldWidgetClass,
				     smb_p1, XmNx, 140, XmNy, 69, NULL);
  smb_g_l3 = XtVaCreateManagedWidget("Security:", xmLabelWidgetClass, smb_p1,
				     XmNx, 20, XmNy, 105, NULL);
  smb_g_rc1 = XtVaCreateManagedWidget("smb_g_rc1", xmRowColumnWidgetClass,
				      smb_p1, XmNx, 77, XmNy, 98, 
				      XmNorientation, XmHORIZONTAL,
				      XmNradioBehavior, True, NULL);
  smb_g_t5a = XtVaCreateManagedWidget("User", xmToggleButtonWidgetClass,
				      smb_g_rc1, NULL);  
  smb_g_t5b = XtVaCreateManagedWidget("Share", xmToggleButtonWidgetClass,
				      smb_g_rc1, NULL);  
  smb_g_t5c = XtVaCreateManagedWidget("Server", xmToggleButtonWidgetClass,
				      smb_g_rc1, NULL);
  smb_g_l4 = XtVaCreateManagedWidget("Workgroup:", xmLabelWidgetClass,
				     smb_p1, XmNx, 20, XmNy, 135, NULL);
  smb_g_t6 = XtVaCreateManagedWidget("smb_g_t1", xmTextFieldWidgetClass,
				     smb_p1, XmNx, 140, XmNy, 129, NULL);
  smb_tmp_share = XtVaCreateManagedWidget("Share /tmp", 
					  xmToggleButtonWidgetClass, smb_p1,
					  XmNx, 20, XmNy, 187, NULL);

  /* Page 2: [Printers] options.  Again, only a limited subset of options */
  smb_p_l1 = XtVaCreateManagedWidget("Comment:", xmLabelWidgetClass,
				     smb_p2, XmNx, 20, XmNy, 16, NULL);
  smb_p_t1 = XtVaCreateManagedWidget("smb_g_t1", xmTextFieldWidgetClass,
				     smb_p2, XmNx, 120, XmNy, 10, NULL);
  smb_p_t2 = XtVaCreateManagedWidget("Browseable",
				     xmToggleButtonWidgetClass, smb_p2,
				     XmNx, 20, XmNy, 42, NULL);
  smb_p_t3 = XtVaCreateManagedWidget("Public", 
				     xmToggleButtonWidgetClass, smb_p2,
				     XmNx, 180, XmNy, 42, NULL);  
  smb_p_t4 = XtVaCreateManagedWidget("Printable",
				     xmToggleButtonWidgetClass, smb_p2,
				     XmNx, 20, XmNy, 62, NULL);
  smb_p_t5 = XtVaCreateManagedWidget("Writable", 
				     xmToggleButtonWidgetClass, smb_p2,
				     XmNx, 180, XmNy, 62, NULL);
  smb_p_l2 = XtVaCreateManagedWidget("Create mode:", xmLabelWidgetClass,
				     smb_p2, XmNx, 20, XmNy, 93, NULL);
  smb_p_t6 = XtVaCreateManagedWidget("smb_g_t1", xmTextFieldWidgetClass,
				     smb_p2, XmNx, 120, XmNy, 89, NULL);

  /* Page 3: [Homes] options. */
  smb_h_l1 = XtVaCreateManagedWidget("Comment:", xmLabelWidgetClass,
				     smb_p3, XmNx, 20, XmNy, 16, NULL);
  smb_h_t1 = XtVaCreateManagedWidget("smb_g_t1", xmTextFieldWidgetClass,
				     smb_p3, XmNx, 120, XmNy, 10, NULL);
  smb_h_t2 = XtVaCreateManagedWidget("Browseable",
				     xmToggleButtonWidgetClass, smb_p3,
				     XmNx, 20, XmNy, 42, NULL);
  smb_h_t3 = XtVaCreateManagedWidget("Read only", 
				     xmToggleButtonWidgetClass, smb_p3,
				     XmNx, 180, XmNy, 42, NULL);  
  smb_h_l2 = XtVaCreateManagedWidget("Create mode:", xmLabelWidgetClass,
				     smb_p3, XmNx, 20, XmNy, 73, NULL);
  smb_h_t4 = XtVaCreateManagedWidget("smb_g_t1", xmTextFieldWidgetClass,
				     smb_p3, XmNx, 120, XmNy, 69, NULL);

  smb_load_default = XtVaCreateManagedWidget("Load defaults", 
				   xmPushButtonWidgetClass, smb_form,
				   XmNtopAttachment, XmATTACH_WIDGET,
				   XmNtopWidget, smb_notebook,
				   XmNleftAttachment, XmATTACH_FORM,
				   XmNrightAttachment, XmATTACH_FORM,
				   NULL);
  XtAddCallback(smb_load_default, XmNactivateCallback, SMB_defaults_CB, NULL);
  /* Ubiqitous OK/Cancel/Help buttons.  OK button may need to stop or start
   * the server(s) and rewrite the configuration file. */
  smb_ok = XtVaCreateManagedWidget("OK", xmPushButtonWidgetClass, smb_form,
				   XmNwidth, 160,
				   XmNtopAttachment, XmATTACH_WIDGET,
				   XmNtopWidget, smb_load_default,
				   XmNleftAttachment, XmATTACH_FORM,
				   XmNbottomAttachment, XmATTACH_FORM, NULL);
  smb_cancel = XtVaCreateManagedWidget("Cancel", xmPushButtonWidgetClass, 
				       smb_form, XmNwidth, 160,
				       XmNtopAttachment, XmATTACH_WIDGET,
				       XmNtopWidget, smb_load_default,
				       XmNleftAttachment, XmATTACH_WIDGET,
				       XmNleftWidget, smb_ok,
				       XmNbottomAttachment, XmATTACH_FORM, 
				       NULL);
  smb_help = XtVaCreateManagedWidget("Help", xmPushButtonWidgetClass, 
				     smb_form,
				     XmNtopAttachment, XmATTACH_WIDGET,
				     XmNtopWidget, smb_load_default,
				     XmNleftAttachment, XmATTACH_WIDGET,
				     XmNleftWidget, smb_cancel,
				     XmNrightAttachment, XmATTACH_FORM,
				     XmNbottomAttachment, XmATTACH_FORM, 
				     NULL);
  XtAddCallback(smb_ok, XmNactivateCallback, write_CB, 
		(XtPointer)smb_shell);
  XtAddCallback(smb_cancel, XmNactivateCallback, SMB_Cancel, 
		(XtPointer)smb_shell);
  XtSetSensitive(smb_notebook, smb_on);
  
  if (smb_conf_exists)
    smb_set_actual();
  else
    smb_set_defaults();

  XtPopup(smb_shell, XtGrabNone);

  XmStringFree(x1);
  XmStringFree(x2);
}

void smb_restart_cancelCB(Widget w, XtPointer client_data, XtPointer call_data)
{
  XtDestroyWidget(w);
  fprintf(stderr, "OK, not restarting.\n");
}

void smb_restart_okCB(Widget w, XtPointer client_data, XtPointer call_data)
{
  fprintf(stderr, "Restarting...\n");
  system("/etc/rc.d/init.d/smb stop");
  sleep(5);
  system("/etc/rc.d/init.d/smb start");
}

/* Restart Samba deamons, causing them to load new settings. */
void smb_restart() {
  Arg args[5];
  XmString smb_rs_m1, smb_rs_m2, smb_rs_m3;
  Widget smb_info_box;

  smb_rs_m1 = XmStringCreateLtoR("A new configuration has been saved as /etc/smb.conf.  The old\nconfiguration is in /etc/smb.conf.old.\n\nIn order for these changes to take effect, you must restart the\nSMB daemons.  Do you wish to do that now?", XmSTRING_DEFAULT_CHARSET);
  smb_rs_m2 = XmStringCreateSimple("Restart");
  smb_rs_m3 = XmStringCreateSimple("Don't Restart");

  XtSetArg(args[0], XmNmessageString, smb_rs_m1);
  XtSetArg(args[1], XmNokLabelString, smb_rs_m2);
  XtSetArg(args[2], XmNcancelLabelString, smb_rs_m3);
  XtSetArg(args[3], XmNwidth, 500);
//  smb_info_box = XmCreateQuestionDialog(top_level, "smb_rs_now", args, 4);
  XtManageChild(smb_info_box);
  XtAddCallback(smb_info_box, XmNokCallback, smb_restart_okCB, NULL);
  XtAddCallback(smb_info_box, XmNcancelCallback, smb_restart_cancelCB, NULL);
  XmStringFree(smb_rs_m1);
  XmStringFree(smb_rs_m2);
  XmStringFree(smb_rs_m3);
}





