/* getdir.c -- Functions to read a directory into *files.
 *
 * Last revised: 11 June 1996
 * Written (sort of) Bryan Herger
 *
 * See COPYING for copyright and license information.
 */

/* Standard includes */
#include <dirent.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/poll.h>
#include <sys/stat.h>
#include <time.h>
#include <grp.h>
#include <pwd.h>
#include <pthread.h>

#include <Xm/Xm.h>

#include "core.h"
#include "filearea.h"

//#define DEBUG 1
//#define DEBUG_WATCH_DIR 1

#define FILES_ALOCCATED_N 100
#define CHECK_IS_DIR_LNK(mode) 	((mode & S_IFMT) == (S_IFDIR | S_IFLNK))

extern Widget top_level, fm_textfield;

pthread_mutex_t config_mutex;

extern bxfm_ftype ** ftype_list;

extern int count_all;
extern int count_current;

extern ext_ftype ** ext_ftype_list;
extern int ext_count_all;
extern int ext_count_current;

/* This holds the list of files in the current directory */
struct filedata *files;

/* The name of the currently open directory */
char current_wd[120];

/* This is the number of file entries allocated in *files */
/* fileptr is first unused space */
/*int nfiles = 100;
int fileptr;*/

/* A few functions */
int load_dir(IfaceBranche * IbPtr, const char *name);
void clear_dir_list(IfaceBranche * IbPtr);
int compare_name();
void sort_dir(IfaceBranche *IbPtr);
static void CreateFileType(char * fullpath, struct filedata * files, char * bumbuf);

char *getparent(IfaceBranche * IbPtr) {
  int x;
  char *to_return;

  for (x = strlen(IbPtr->current_wd); x > 0; x--)
    if (IbPtr->current_wd[x] == '/')
      break;

  if (x == 0) x++;
  to_return = malloc(sizeof(char) * (x+1));
  strncpy(to_return, IbPtr->current_wd, x);
  to_return[x] = '\0';
  if (x == 0)
    sprintf(to_return, "/");
#ifdef DEBUG
  printf("getparent -> %d:%s\n", x, to_return);
#endif
  return(to_return);
}

int check_parent_dir(IfaceBranche * IbPtr, char * name, mode_t mode_info)
{
	if (1 == IbPtr->ParentDirShow)
		return 0;

	if (!S_ISDIR(mode_info))
		return 0;

	if (0 == strcmp("..", name))
		return 1;
	else
		return 0;
}

int check_current_dir(IfaceBranche * IbPtr, char * name, mode_t mode_info)
{
	if (1 == IbPtr->CurrentDirShow)
		return 0;
	if (0 == strcmp(".", name))
	{
		return 1;
	}
	else
	{
		return 0;
	}
}

int check_hidden_dir(IfaceBranche * IbPtr, char * name, mode_t mode_info)
{
	if (1 == IbPtr->HiddenDirShow)
		return 0;

	if (name[0] == '.')
	{
		if (1 == strlen(name))
			return 0;
			
		if (2 == strlen(name) && name[1] == '.')
			return 0;

		if (S_ISDIR(mode_info) || CHECK_IS_DIR_LNK(mode_info))
			return 1;
	}
	return 0;
}

int check_hidden_file(IfaceBranche * IbPtr, char * name, mode_t mode_info)
{
	if (1 == IbPtr->HiddenFileShow)
		return 0;
		
	if (name[0] == '.')
	{
		if (1 == strlen(name))
			return 0;

		if (!S_ISDIR(mode_info) && !CHECK_IS_DIR_LNK(mode_info))
			return 1;
	}
	return 0;
}

#ifdef USE_INOTIFY

void PutWatchCommand(IfaceBranche * IbPtr, int command)
{
	WatchCommand * ptr1, * ptr2;
	
	ptr1 = calloc(1, sizeof(WatchCommand));
	
#ifdef DEBUG_WATCH_DIR
	printf("PutWatchCommand: start \n");
#endif

	if (!ptr1)
		return;
	
	ptr1->code = command;
	
	if (COMMAND_ADD_WATCH == command)
	{
		ptr1->path = strdup(IbPtr->current_wd);
		ptr1->thread_number = IbPtr->thread_number;
		ptr1->watch_fd = - 1;
		ptr1->IfBrPtr = IbPtr;
	}

	if (COMMAND_DEL_WATCH == command)
	{
		ptr1->path = strdup(IbPtr->current_wd);
		ptr1->thread_number = IbPtr->thread_number;
		ptr1->watch_fd = - 1;
		ptr1->IfBrPtr = IbPtr;
	}

	if (COMMAND_DETACH_IFACE == command)
	{
		ptr1->path = strdup(IbPtr->current_wd);
		ptr1->thread_number = IbPtr->thread_number;
		ptr1->watch_fd = - 1;
		ptr1->IfBrPtr = IbPtr;
	}

	if (COMMAND_TERMINATE_WATCH == command)
	{
		ptr1->path = NULL;
		ptr1->watch_fd = - 1;
		ptr1->IfBrPtr = NULL;
#ifdef DEBUG_WATCH_DIR
		printf("Transmite COMMAND_TERMINATE_WATCH\n");
#endif
	}
	
	pthread_mutex_lock(&watch_mutex);
	if (!WatchCommandQueue)
		WatchCommandQueue = ptr1;
	else
	{
		ptr2 = WatchCommandQueue->next;
		WatchCommandQueue->next = ptr1;
		ptr1->next = ptr2;
	}
	pthread_mutex_unlock(&watch_mutex);
}

static void CleanCancelEventQueue(IfaceBranche * IbPtr)
{
	WatchEvent * ptr1, * ptr2;
	
	pthread_mutex_lock(&IbPtr->branch_mutex);

	ptr1 = IbPtr->WatchEventQueue;

	while (ptr1)
	{
		ptr2 = ptr1->next;
		
		if (ptr1->path)
			free(ptr1->path);
			
		if (ptr1->file_ptr)
		{
			if (ptr1->file_ptr->name)
				free(ptr1->file_ptr->name);
				
			if (ptr1->file_ptr->linkto)
				free(ptr1->file_ptr->linkto);
				
			if (ptr1->file_ptr->info)
				free(ptr1->file_ptr->info);
		}

		free(ptr1);
		ptr1 = ptr2;
	}
	
	ptr1 = calloc(1, sizeof(WatchEvent));
	
	ptr1->code = NOTIFY_WATCH_TERMINATED;
	IbPtr->WatchEventQueue = ptr1;
	pthread_mutex_unlock(&IbPtr->branch_mutex);
#ifdef DEBUG_WATCH_DIR
	printf("Watching: NOTIFY_WATCH_TERMINATED sended to %d thread\n", IbPtr->thread_number);
#endif
}

WatchEvent * GetWatchEvent(IfaceBranche * IbPtr)
{
	WatchEvent * ptr1;

	pthread_mutex_lock(&IbPtr->branch_mutex);

	if (!IbPtr->WatchEventQueue)
	{
		pthread_mutex_unlock(&IbPtr->branch_mutex);
		return NULL;
	}

	ptr1 = IbPtr->WatchEventQueue;

	IbPtr->WatchEventQueue = ptr1->next;
	
	pthread_mutex_unlock(&IbPtr->branch_mutex);
	
	return ptr1;
}


void PutEvent(IfaceBranche * IbPtr, WatchEvent * Ev)
{
	WatchEvent * ptr1;

	pthread_mutex_lock(&IbPtr->branch_mutex);

	ptr1 = IbPtr->WatchEventQueue;
	
	if (!ptr1)
	{
		IbPtr->WatchEventQueue = Ev;
		pthread_mutex_unlock(&IbPtr->branch_mutex);
		return ;
	}
	else
	{
		while(ptr1->next != NULL)
		{
			ptr1 = ptr1->next;
		}
		
		ptr1->next = Ev;
	}
	pthread_mutex_unlock(&IbPtr->branch_mutex);
}

static WatchCommand * GetCommand(void)
{
	WatchCommand * ptr;


	if (!WatchCommandQueue)
		return NULL;
	
	pthread_mutex_lock(&watch_mutex);
	ptr = WatchCommandQueue;
	if (ptr)
		WatchCommandQueue = WatchCommandQueue->next;
	pthread_mutex_unlock(&watch_mutex);
	return ptr;
}

static WatchCommand * CheckWatched(WatchCommand ** IfaceWatch, int n, char * name)
{
	int i;
	WatchCommand * ptr;
	int found = 0;

	for (i = 0; i < n; i++)
	{
		ptr = IfaceWatch[i];
		if (0 == strcmp(ptr->path, name))
		{
			found = 1;
			break;
		}
	}
	
	if (found)
		return ptr;
	else
		return NULL;
}

static struct filedata * CreateFileInfo(char * name, char * path_to_name, struct stat * buf)
{
	char * fullpath;
//	struct stat buf;
	int linkadjust = 0;
//	int filetype;
	char bumbuf[1024];
	struct fileinfo * info;
	char sz[100], own[100], grp[100], prm[100], dt[100];
	bxfm_ftype * magic, * ftype_ptr;
	int i;
	int found = 0;
	int less = 0;
	int magic_finded = 0;
	struct filedata * fnew;

//#ifdef DEBUG
	printf("CreateFileInfo %s\n", name);
//#endif

	fnew = calloc(1, sizeof(struct filedata));

	if (!fnew)
		return NULL;

	fnew->name = calloc(strlen(name) + 1, sizeof(char));
	strcpy(fnew->name, name);

	fullpath = calloc(strlen(name) + strlen(path_to_name) + 3, sizeof(char));
	strcpy(fullpath, path_to_name);
	strcat(fullpath, "/");
	strcat(fullpath, name);


	fnew->filetype = buf->st_mode;

/*	if (S_ISLNK(buf->st_mode)) {
			linkadjust = 1;
	}
	if (S_ISDIR(buf->st_mode))
		fnew->filetype = 0;
	else
		fnew->filetype = 2;

	fnew->filetype += linkadjust;
*/

	info = calloc(1, sizeof(struct fileinfo));
	if (info)
	{
		info->f_size = buf->st_size;
		info->f_mtime = buf->st_mtime;
		info->f_ctime = buf->st_ctime;
		info->f_atime = buf->st_atime;
		info->f_perm = buf->st_mode;
		info->f_owner = buf->st_uid;
		info->f_owngrp = buf->st_gid;
	}

	fnew->info = info;

	magic_get_type(fullpath, bumbuf);

//#ifdef DEBUG
	printf("AddItemToFileView:%s: %s ->\n", fullpath, bumbuf);
//#endif

	CreateFileType(fullpath, fnew, bumbuf);

/*
	pthread_mutex_lock(&config_mutex);
	
	i = count_current - 1;

	while (i >= 0)
	{
		if (strstr((ftype_list[i])->magic, bumbuf)) 
		{
			magic = ftype_list[i];
			fnew->magic = magic;
			magic_finded = 1;
			break ;
		}
		i--;
	}

	if (!magic_finded)
	{
		if (!ftype_list)
		{
			ftype_list = (bxfm_ftype **) calloc(20, sizeof(bxfm_ftype *));
			ftype_ptr = (bxfm_ftype *) calloc(20, sizeof(bxfm_ftype));

			count_all = 20;

			for (i = 0; i < count_all; i++)
			{
				ftype_list[i] = ftype_ptr + i;
			}

			count_current = 0;
				
			ftype_ptr = ftype_list[0];
			ftype_ptr->magic = calloc(strlen(bumbuf) + 4, sizeof(char));
			sprintf(ftype_ptr->magic, "<%s>", bumbuf);
			fnew->magic = ftype_list[0];
		}
		else
		{
			if (count_all == count_current + 1)
			{
				ftype_list = (bxfm_ftype **) realloc(ftype_list, (10 + count_all) * sizeof(bxfm_ftype *));
				ftype_ptr = calloc(10, sizeof(bxfm_ftype));
				for (i = 0; i < 10; i++)
				{
					ftype_list[count_all + i] = ftype_ptr + i;
				}
				count_all += 10;
			}

			ftype_ptr = ftype_list[count_current];
			ftype_ptr->magic = calloc(strlen(bumbuf) + 4, sizeof(char));
			sprintf(ftype_ptr->magic, "<%s>", bumbuf);
			fnew->magic = ftype_ptr;

			count_current ++;
			}
	}		
		
	pthread_mutex_unlock(&config_mutex);
*/

	if (fullpath)
		free(fullpath);
		
	return fnew;
}



static void IntfyAddFile(int fd, char * name, WatchCommand ** IfaceWatch, int n)
{
	IfaceBranche * IbPtr;
	int i, len;
	WatchCommand * ComPtr;
	WatchEvent * EvPtr;
	char * fullpath = NULL;
	struct stat buf, buf2;
	mode_t mode_info;
	char symlink_path[4096];
	ssize_t zt;
	int stat_res;
	
#ifdef DEBUG_WATCH_DIR
	printf("Watching: create event for add  %s\n", name);
#endif
	for (i = 0; i < n; i++)
	{
		ComPtr = IfaceWatch[i];
		if (ComPtr->watch_fd == fd)
		{
			if (fullpath)
				len = strlen(fullpath);
			else
				len = 0;

			if (len < strlen(name) + strlen(ComPtr->path) + 3)
			{
				if (fullpath)
					fullpath = realloc(fullpath, strlen(name) + strlen(ComPtr->path) + 3);
				else
					fullpath = calloc(strlen(name) + strlen(ComPtr->path) + 3, sizeof(char));
			}

			sprintf(fullpath, "%s/%s", ComPtr->path, name);
			lstat(fullpath, &buf);
			
			mode_info = buf.st_mode;

			if (S_ISLNK(mode_info))
			{
				if ((stat_res = stat(fullpath, &buf2)) == 0)
				{
//					memset(symlink_path, 0, 4096);
					zt = readlink(fullpath, symlink_path, 4096);
					if (S_ISDIR(buf2.st_mode))
					{
						mode_info = mode_info | (buf2.st_mode & S_IFMT);
						buf.st_mode = buf.st_mode | (buf2.st_mode & S_IFMT);
					}
				}
			}

			if (check_hidden_dir(ComPtr->IfBrPtr, name, mode_info))
				continue;

			if (check_hidden_file(ComPtr->IfBrPtr, name, mode_info))
				continue;

			if (check_parent_dir(ComPtr->IfBrPtr, name, mode_info))
				continue;

			if (check_current_dir(ComPtr->IfBrPtr, name, mode_info))
				continue;

			EvPtr = calloc(1, sizeof(WatchEvent));
			if (!EvPtr)
				continue;
				
			EvPtr->file_ptr = CreateFileInfo(name, ComPtr->path, &buf);
			
			if (EvPtr->file_ptr)
			{
				if (((buf.st_mode & S_IFMT) == S_IFLNK) && !stat_res && zt > 0)
				{
					EvPtr->file_ptr->linkto = calloc(zt + 1, sizeof(char));
					if (EvPtr->file_ptr->linkto)
						strncpy(EvPtr->file_ptr->linkto, symlink_path, zt);
				}
			}
#ifdef DEBUG_WATCH_DIR
	printf("Watching: event for add created %s\n", name);
#endif
			if (!EvPtr->file_ptr)
			{
				free(EvPtr);
				continue;
			}
			EvPtr->path = strdup(ComPtr->path);
			EvPtr->code = NOTIFY_ADD_FILE;
			EvPtr->next = NULL;
#ifdef DEBUG_WATCH_DIR
	printf("Watching: transmite add event for thread %d\n", IfaceWatch[i]->thread_number);
#endif
			PutEvent(ComPtr->IfBrPtr, EvPtr);
		}
	}
	if (fullpath)
		free(fullpath);
}

static void IntfyChangeAttrib(int fd, char * name, WatchCommand ** IfaceWatch, int n)
{
	IfaceBranche * IbPtr;
	int i, len;
	WatchCommand * ComPtr;
	WatchEvent * EvPtr;
	char * fullpath = NULL;
	struct stat buf, buf2;
	mode_t mode_info;
	char symlink_path[4096];
	ssize_t zt;
	int stat_res;
	
#ifdef DEBUG_WATCH_DIR
	printf("Watching: create event for change attrib  %s\n", name);
#endif
	for (i = 0; i < n; i++)
	{
		ComPtr = IfaceWatch[i];
		if (ComPtr->watch_fd == fd)
		{
			if (fullpath)
				len = strlen(fullpath);
			else
				len = 0;

			if (len < strlen(name) + strlen(ComPtr->path) + 3)
			{
				if (fullpath)
					fullpath = realloc(fullpath, strlen(name) + strlen(ComPtr->path) + 3);
				else
					fullpath = calloc(strlen(name) + strlen(ComPtr->path) + 3, sizeof(char));
			}

			sprintf(fullpath, "%s/%s", ComPtr->path, name);
			lstat(fullpath, &buf);

			mode_info = buf.st_mode;

			if (S_ISLNK(mode_info))
			{
				if ((stat_res = stat(fullpath, &buf2)) == 0)
				{
//					memset(symlink_path, 0, 4096);
					zt = readlink(fullpath, symlink_path, 4096);
					if (S_ISDIR(buf2.st_mode))
					{
						mode_info = mode_info | (buf2.st_mode & S_IFMT);
						buf.st_mode = buf.st_mode | (buf2.st_mode & S_IFMT);
					}
				}
			}


			if (check_hidden_dir(ComPtr->IfBrPtr, name, mode_info))
				continue;

			if (check_hidden_file(ComPtr->IfBrPtr, name, mode_info))
				continue;

			if (check_parent_dir(ComPtr->IfBrPtr, name, mode_info))
				continue;

			if (check_current_dir(ComPtr->IfBrPtr, name, mode_info))
				continue;

			EvPtr = calloc(1, sizeof(WatchEvent));
			if (!EvPtr)
				continue;
				
			EvPtr->file_ptr = CreateFileInfo(name, ComPtr->path, &buf);

			if (EvPtr->file_ptr)
			{
				if (((buf.st_mode & S_IFMT) == S_IFLNK) && !stat_res && zt > 0)
				{
					EvPtr->file_ptr->linkto = calloc(zt + 1, sizeof(char));
					if (EvPtr->file_ptr->linkto)
						strncpy(EvPtr->file_ptr->linkto, symlink_path, zt);
				}
			}


			if (!EvPtr->file_ptr)
			{
				free(EvPtr);
				continue;
			}
			EvPtr->path = strdup(ComPtr->path);
			EvPtr->code = NOTIFY_CHANGE_ATTRIB;
			EvPtr->next = NULL;
#ifdef DEBUG_WATCH_DIR
	printf("Watching: transmite change attrib event for thread %d\n", IfaceWatch[i]->thread_number);
#endif
			PutEvent(ComPtr->IfBrPtr, EvPtr);
		}
	}
	if (fullpath)
		free(fullpath);
}



static void IntfyCloseFile(int fd, char * name, WatchCommand ** IfaceWatch, int n)
{
	IfaceBranche * IbPtr;
	int i, len;
	WatchCommand * ComPtr;
	WatchEvent * EvPtr;
	char * fullpath = NULL;
	struct stat buf, buf2;
	mode_t mode_info;
	char symlink_path[4096];
	ssize_t zt;
	int stat_res;
	
#ifdef DEBUG_WATCH_DIR
	printf("Watching: create event for close  %s\n", name);
#endif
	for (i = 0; i < n; i++)
	{
		ComPtr = IfaceWatch[i];
		if (ComPtr->watch_fd == fd)
		{
			if (fullpath)
				len = strlen(fullpath);
			else
				len = 0;

			if (len < strlen(name) + strlen(ComPtr->path) + 3)
			{
				if (fullpath)
					fullpath = realloc(fullpath, strlen(name) + strlen(ComPtr->path) + 3);
				else
					fullpath = calloc(strlen(name) + strlen(ComPtr->path) + 3, sizeof(char));
			}

			sprintf(fullpath, "%s/%s", ComPtr->path, name);
			lstat(fullpath, &buf);

			mode_info = buf.st_mode;

			if (S_ISLNK(mode_info))
			{
				if ((stat_res = stat(fullpath, &buf2)) == 0)
				{
//					memset(symlink_path, 0, 4096);
					zt = readlink(fullpath, symlink_path, 4096);
					if (S_ISDIR(buf2.st_mode))
					{
						mode_info = mode_info | (buf2.st_mode & S_IFMT);
						buf.st_mode = buf.st_mode | (buf2.st_mode & S_IFMT);
					}
				}
			}


			if (check_hidden_dir(ComPtr->IfBrPtr, name, mode_info))
				continue;

			if (check_hidden_file(ComPtr->IfBrPtr, name, mode_info))
				continue;

			if (check_parent_dir(ComPtr->IfBrPtr, name, mode_info))
				continue;

			if (check_current_dir(ComPtr->IfBrPtr, name, mode_info))
				continue;

			EvPtr = calloc(1, sizeof(WatchEvent));
			if (!EvPtr)
				continue;
				
			EvPtr->file_ptr = CreateFileInfo(name, ComPtr->path, &buf);

			if (EvPtr->file_ptr)
			{
				if (((buf.st_mode & S_IFMT) == S_IFLNK) && !stat_res && zt > 0)
				{
					EvPtr->file_ptr->linkto = calloc(zt + 1, sizeof(char));
					if (EvPtr->file_ptr->linkto)
						strncpy(EvPtr->file_ptr->linkto, symlink_path, zt);
				}
			}


			if (!EvPtr->file_ptr)
			{
				free(EvPtr);
				continue;
			}
			EvPtr->path = strdup(ComPtr->path);
			EvPtr->code = NOTIFY_CLOSE_FILE;
			EvPtr->next = NULL;
#ifdef DEBUG_WATCH_DIR
	printf("Watching: transmite close event for thread %d\n", IfaceWatch[i]->thread_number);
#endif
			PutEvent(ComPtr->IfBrPtr, EvPtr);
		}
	}
	if (fullpath)
		free(fullpath);
}

static void IntfyDeleteFile(int fd, char * name, WatchCommand ** IfaceWatch, int n)
{
	IfaceBranche * IbPtr;
	int i;
	WatchCommand * ComPtr;
	WatchEvent * EvPtr;
	struct filedata * fnew;

#ifdef DEBUG_WATCH_DIR
	printf("Watching: create event for deleting  %s\n", name);
#endif

	for (i = 0; i < n; i++)
	{
		ComPtr = IfaceWatch[i];
		if (ComPtr->watch_fd == fd)
		{
			EvPtr = calloc(1, sizeof(WatchEvent));
			if (!EvPtr)
				continue;
				
			fnew = calloc(1, sizeof(struct filedata));
			if (!fnew)
			{
				free(EvPtr);
				continue;
			}
			fnew->name = strdup(name);
			EvPtr->path = strdup(ComPtr->path);
			EvPtr->code = NOTIFY_DELETE_FILE;
			EvPtr->next = NULL;
			EvPtr->file_ptr = fnew;
#ifdef DEBUG_WATCH_DIR
	printf("Watching: transmite deleting event for thread %d\n", IfaceWatch[i]->thread_number);
#endif
			PutEvent(ComPtr->IfBrPtr, EvPtr);
		}
	}
}


static void InotifyReadProcessing(int fd, WatchCommand ** IfaceWatch, int n)
{
	int length, i = 0;
	IfaceBranche * IbPtr;
	char ievent_buf[EVENT_BUF_LEN];
	struct inotify_event * event;
	
	length = read(fd, ievent_buf, EVENT_BUF_LEN);  

	if (length <= 0) {
#ifdef DEBUG
		perror("Inotify read:");
#endif
		return ;
	}  

#ifdef DEBUG
	printf("Inotify Enter ...\n");
#endif

#ifdef DEBUG_WATCH_DIR
	printf("Watching:  getted message from inotify length %d\n", length);
#endif


	while (i < length ) {
#ifdef DEBUG
		printf("Buf %ld struct %ld i %d\n", EVENT_BUF_LEN, sizeof(struct inotify_event), i);
#endif
		event = (struct inotify_event * ) &(ievent_buf[i]);

#ifdef DEBUG
		printf("Event wd %d\tmask %d\tcookie %d\tlen %d\n", event->wd, event->mask, event->cookie, event->len);
#endif

		if ( event->len ) {
#ifdef DEBUG_WATCH_DIR
		printf("Watching: changed file %s fd = %d\n", event->name, event->wd);
#endif
			if ( event->mask & IN_CREATE ) {
#ifdef DEBUG_WATCH_DIR
				if ( event->mask & IN_ISDIR ) {
					printf( "The directory %s was created.\n", event->name);
				}
				else {
					printf( "The file %s was created.\n", event->name );
				}
#endif
				IntfyAddFile(event->wd, event->name, IfaceWatch, n);
			}
			else if ( event->mask & IN_MOVED_TO ) {
#ifdef DEBUG_WATCH_DIR

					if ( event->mask & IN_ISDIR ) {
						printf( "The directory %s was moved to.\n", event->name );       
					}
					else {
						printf( "The file %s was moved to.\n", event->name );
					}
					if ( event->mask & IN_MASK_ADD ){
						printf( "The %s replacing.\n", event->name );       
					}
#endif
					IntfyAddFile(event->wd, event->name, IfaceWatch, n);
				}
				else if ( event->mask & IN_DELETE ) {
#ifdef DEBUG_WATCH_DIR
						if ( event->mask & IN_ISDIR ) {
							printf( "The directory %s was deleted.\n", event->name );       
						}
						else {
							printf( "The file %s was deleted.\n", event->name );
						}
#endif
						IntfyDeleteFile(event->wd, event->name, IfaceWatch, n);
					}
#ifdef DEBUG_WATCH_DIR
					else if ( event->mask & IN_MODIFY ) {
							if ( event->mask & IN_ISDIR ) {
								printf( "The directory %s was modified.\n", event->name );
							}
							else {
								printf( "The file %s was modified.\n", event->name );
							}
						}
						else if ( event->mask & IN_DELETE_SELF ) {
							if ( event->mask & IN_ISDIR ) {
								printf( "The directory %s was deleted self.\n", event->name );
							}
							else {
								printf( "The file %s was deleted self.\n", event->name );
							}
						}
#endif
							else if ( event->mask & IN_MOVED_FROM ) {
#ifdef DEBUG_WATCH_DIR
								if ( event->mask & IN_ISDIR ) {
									printf( "The directory %s was moved from.\n", event->name );
								}
								else {
									printf( "The file %s was moved from.\n", event->name );
								}
#endif
								IntfyDeleteFile(event->wd, event->name, IfaceWatch, n);
							}
#ifdef DEBUG_WATCH_DIR
							else if ( event->mask & IN_MOVE_SELF ) {
								if ( event->mask & IN_ISDIR ) {
									printf( "The directory %s was moved self.\n", event->name );
								}
								else {
									printf( "The file %s was moved. self\n", event->name );
								}
							}
#endif
								else if ( event->mask & IN_CLOSE_WRITE ) {
#ifdef DEBUG_WATCH_DIR
									if ( event->mask & IN_ISDIR ) {
										printf( "The directory %s closed.\n", event->name );
									}
									else {
										printf( "The file %s closed for writing.\n", event->name );
									}
#endif
									if (!(event->mask & IN_ISDIR))
										IntfyCloseFile(event->wd, event->name, IfaceWatch, n);
								}
									else if ( event->mask & IN_ATTRIB ) {
#ifdef DEBUG_WATCH_DIR
										if ( event->mask & IN_ISDIR ) {
											printf( "For directory %s attributes changed.\n", event->name );
										}
										else {
											printf( "For file %s attributes changed.\n", event->name );
										}
#endif
										IntfyChangeAttrib(event->wd, event->name, IfaceWatch, n);
									}
		}
		i += EVENT_SIZE + event->len;
	}
}


void * WatchDirThread(void * arg)
{
	int watch_fd;
	WatchCommand * ptr, * ptr2, ** ptr3;
	WatchCommand ** IfaceWatch = NULL;
	int i, j, n_iface = 0;
	struct pollfd fdset;
	int QueueNotEmpty = 0;
	int watch_on = 1;
	
	watch_fd = inotify_init();
	
	if (-1 == watch_fd)
		return NULL;
	
	fdset.fd = watch_fd;
	fdset.events = POLLIN;

#ifdef DEBUG_WATCH_DIR
	printf("Watch thread started succesfull\n");
#endif

	while(watch_on)
	{
		ptr = GetCommand();
		
		if (ptr)
		{

#ifdef DEBUG_WATCH_DIR
	printf("Watching: found new command type %d\n", ptr->code);
#endif
			if (ptr->next)
				QueueNotEmpty = 1;
			else
				QueueNotEmpty = 0;
				
/** Add new watch directory **/

			if (COMMAND_ADD_WATCH == ptr->code)
			{
#ifdef DEBUG_WATCH_DIR
	printf("Watching:  COMMAND_ADD_WATCH %s\n", ptr->path);
#endif
				if (!IfaceWatch)
				{
#ifdef DEBUG_WATCH_DIR
	printf("Watching:  adding watch to empty\n");
#endif
					IfaceWatch = calloc(1, sizeof(WatchCommand *));
					if (IfaceWatch)
						n_iface++;
				}
				else
				{
#ifdef DEBUG_WATCH_DIR
	printf("Watching:  adding watch to not empty\n");
#endif
					IfaceWatch = realloc(IfaceWatch, (n_iface + 1) * sizeof(WatchCommand *));
					if (IfaceWatch)
						n_iface++;
				}

				if (IfaceWatch)
				{
					*(IfaceWatch + n_iface - 1) = ptr;

					ptr2 = CheckWatched(IfaceWatch, n_iface - 1, ptr->path);
					if (ptr2)
					{
#ifdef DEBUG_WATCH_DIR
	printf("Watching:  added watcher found in list %s %d %d\n", ptr->path, ptr->thread_number, ptr2->thread_number);
#endif
						ptr->watch_fd = ptr2->watch_fd;
					}
					else
					{
#ifdef DEBUG_WATCH_DIR
	printf("Watching:  added watcher not found in list %s\n", ptr->path);
#endif
						ptr->watch_fd = inotify_add_watch(watch_fd, ptr->path, 
									IN_CREATE | IN_DELETE | IN_MOVED_FROM | IN_MOVED_TO | IN_CLOSE_WRITE | IN_ATTRIB);
#ifdef DEBUG_WATCH_DIR
	printf("Watching:  added path %s to inotify\n", ptr->path);
#endif
					}
				}
			}



/** Delete watch directory **/

			if (COMMAND_DEL_WATCH == ptr->code && IfaceWatch)
			{
				int deleting_fd = -1;
				int found = 0;
#ifdef DEBUG_WATCH_DIR
	printf("Watching:  COMMAND_DEL_WATCH %s\n", ptr->path);
#endif
				
				for (i = 0; i < n_iface; i++)
				{
					ptr2 = IfaceWatch[i];
					if (ptr2->thread_number == ptr->thread_number)
					{
						deleting_fd = ptr2->watch_fd;
						if (ptr2->path)
							free(ptr2->path);
						
						free(ptr2);
						IfaceWatch[i] = NULL;
						found = 1;
						break;
					}
				}
				
				if (found)
				{
#ifdef DEBUG_WATCH_DIR
	printf("Watching:  deleted watcher found in list %s\n", ptr->path);
#endif
					ptr3 = calloc(n_iface - 1, sizeof(WatchCommand *));

					j = 0;
					for (i = 0; i < n_iface; i++)
					{
						ptr2 = IfaceWatch[i];
						
						if (ptr2)
						{
							ptr3[j] = ptr2;
							j++;
						}
					}
					n_iface--;
					if (IfaceWatch)
						free(IfaceWatch);
					
					if (n_iface > 0)
						IfaceWatch = ptr3;
					else
						IfaceWatch = NULL;
				}
				
				found = 0;
				if (n_iface > 0)
				{
					for (i = 0; i < n_iface; i++)
					{
						ptr2 = IfaceWatch[i];
						
						if (0 == strcmp(ptr2->path, ptr->path))
						{
							found = 1;
						}
					}
				}
				
				if (!found && -1 != deleting_fd)
				{
#ifdef DEBUG_WATCH_DIR
	printf("Watching:  deleted watcher is uniqe %s\n", ptr->path);
#endif
					inotify_rm_watch(watch_fd, deleting_fd);
				}
				
				if (ptr->path)
					free(ptr->path);
				
				free(ptr);
			}

/** Detach interface **/

			if (COMMAND_DETACH_IFACE == ptr->code && IfaceWatch)
			{
				int deleting_fd = -1;
				int found = 0;
#ifdef DEBUG_WATCH_DIR
	printf("Watching:  COMMAND_DETACH_WATCH %d %s\n", ptr->thread_number, ptr->path);
#endif
				
				for (i = 0; i < n_iface; i++)
				{
					ptr2 = IfaceWatch[i];
					if (ptr2->thread_number == ptr->thread_number)
					{
						deleting_fd = ptr2->watch_fd;
						CleanCancelEventQueue(ptr2->IfBrPtr);
						if (ptr2->path)
							free(ptr2->path);
						
						free(ptr2);
						IfaceWatch[i] = NULL;
						found = 1;
						break;
					}
				}
				
				if (found)
				{
#ifdef DEBUG_WATCH_DIR
	printf("Watching:  deleted watcher found in list %s\n", ptr->path);
#endif
					ptr3 = calloc(n_iface - 1, sizeof(WatchCommand *));

					j = 0;
					for (i = 0; i < n_iface; i++)
					{
						ptr2 = IfaceWatch[i];
						
						if (ptr2)
						{
							ptr3[j] = ptr2;
							j++;
						}
					}
					n_iface--;
					if (IfaceWatch)
						free(IfaceWatch);
					
					if (n_iface > 0)
						IfaceWatch = ptr3;
					else
						IfaceWatch = NULL;
				}
				
				found = 0;
				if (n_iface > 0)
				{
					for (i = 0; i < n_iface; i++)
					{
						ptr2 = IfaceWatch[i];
						
						if (0 == strcmp(ptr2->path, ptr->path))
						{
							found = 1;
						}
					}
				}
				
				if (!found && -1 != deleting_fd)
				{
#ifdef DEBUG_WATCH_DIR
	printf("Watching:  deleted watcher is uniqe %s\n", ptr->path);
#endif
					inotify_rm_watch(watch_fd, deleting_fd);
				}
				
				if (ptr->path)
					free(ptr->path);
				
				free(ptr);
			}
			
/** Delete watch directory **/

			if (COMMAND_TERMINATE_WATCH == ptr->code)
			{
#ifdef DEBUG_WATCH_DIR
				printf("Watching:  ALL TERMINATING, found %d ifaces\n", n_iface);
#endif
				if (n_iface > 0)
				{
					for (i = 0; i < n_iface; i++)
					{
						ptr2 = IfaceWatch[i];
						
						if (ptr2->watch_fd > 0)
							inotify_rm_watch(watch_fd, ptr2->watch_fd);
					}
				}
				watch_on = 0;
			}
		}
/******************/		
		if (QueueNotEmpty)
			i = poll(&fdset, 1, 0);
		else
			i = poll(&fdset, 1, 250);

//#ifdef DEBUG_WATCH_DIR
//	if (i < 0 || i == 0) printf("Watching: pipe is empty\n");
//	if (i > 0) printf("Watching: pipe is not empty\n");
//#endif
		if (i > 0)
		{
#ifdef DEBUG_WATCH_DIR
			printf("===============\nWatching:\n");
			printf("Poll return %d\n", i);
			if (i == 1)
			{
				printf("Revents %d %d %d\n", fdset.revents, POLLIN, fdset.revents & POLLIN);
			}
			for (i = 0; i < n_iface; i++)
			{
				printf("Thread = %d\t path = %s\t fd = %d\n", IfaceWatch[i]->thread_number, IfaceWatch[i]->path, IfaceWatch[i]->watch_fd);
			}
			printf("===============\n");
#endif
			if (fdset.revents & POLLIN)
				InotifyReadProcessing(watch_fd, IfaceWatch, n_iface);			
		}
//		sleep(1);
	}
	close(watch_fd);
#ifdef DEBUG_WATCH_DIR
	printf("===============\nWatching closed\n");
#endif
	return NULL;
}
#endif


static void CreateFileType(char * fullpath, struct filedata * files, char * bumbuf)
{
	int k, j, l;
	bxfm_ftype * ftype_ptr;
	int magic_finded = 0;


#ifdef DEBUG
	printf("CreateFileType start %s\n", fullpath);
#endif


	magic_finded = 0;

	pthread_mutex_lock(&config_mutex);

	for (j = 0; j < count_current; j++)
	{
		if (0 == strcmp((ftype_list[j])->magic, bumbuf)) 
		{
#ifdef DEBUG
			printf("ok file %s %s %s\n", files->name, (ftype_list[j])->magic, bumbuf);
#endif

			for (k = 0; k < ext_count_current; k++)
			{
				if (0 == strcmp((ftype_list[j])->magic, (ext_ftype_list[k])->type_link))
				{
#ifdef DEBUG
					printf("ok ext type file %s %s %s\n", files->name, (ftype_list[j])->magic, (ext_ftype_list[k])->type_link);
#endif
					if (strlen((ext_ftype_list[k])->pattern) > 1 && *((ext_ftype_list[k])->pattern) == '*')
					{
						if (strstr(files->name, (ext_ftype_list[k])->pattern + 1))
						{
							for (l = 0; l < count_current; l++)
							{
								if (0 == strcmp((ftype_list[l])->magic, (ext_ftype_list[k])->magic))
								{
#ifdef DEBUG
									printf("found alternativ magic for file %s %s %s\n", files->name, (ftype_list[l])->magic, (ext_ftype_list[k])->type_link);
#endif
											
									files->magic = ftype_list[l];
									magic_finded = 1;
									break ;
								}
							}
							break ;
						}
					}
				} 
			}
			if (!files->magic)
				files->magic = ftype_list[j];
			magic_finded = 1;
#ifdef DEBUG
			printf("ok file %s %p %s %lx\n", files->name, files->magic, (files->magic)->magic, (files->magic)->big_pixmap);
#endif
			break ;
		}
#ifdef DEBUG
		printf("\t Type %s\tMagic %s\n", bumbuf, (ftype_list[j])->magic); 
#endif
	}
		
	if (!magic_finded)
	{
		if (!ftype_list)
		{
			ftype_list = (bxfm_ftype **) calloc(20, sizeof(bxfm_ftype *));
			ftype_ptr = (bxfm_ftype *) calloc(20, sizeof(bxfm_ftype));

			count_all = 20;

			for (k = 0; k < count_all; k++)
			{
				ftype_list[k] = ftype_ptr + k;
			}

			count_current = 0;
				
			ftype_ptr = ftype_list[0];
			ftype_ptr->magic = calloc(strlen(bumbuf) + 4, sizeof(char));
			sprintf(ftype_ptr->magic, "%s", bumbuf);
			ftype_ptr->big_pixmap = XmUNSPECIFIED_PIXMAP;
			ftype_ptr->big_mask = XmUNSPECIFIED_PIXMAP;
			ftype_ptr->small_pixmap = XmUNSPECIFIED_PIXMAP;
			ftype_ptr->small_mask = XmUNSPECIFIED_PIXMAP;
			files->magic = ftype_list[0];
		}
		else
		{
			if (count_all == count_current + 1)
			{
				ftype_list = (bxfm_ftype **) realloc(ftype_list, (10 + count_all) * sizeof(bxfm_ftype *));
				ftype_ptr = calloc(10, sizeof(bxfm_ftype));
				for (k = 0; k < 10; k++)
				{
					ftype_list[count_all + k] = ftype_ptr + k;
				}
				count_all += 10;
			}

			ftype_ptr = ftype_list[count_current];
			ftype_ptr->magic = calloc(strlen(bumbuf) + 4, sizeof(char));
			sprintf(ftype_ptr->magic, "%s", bumbuf);
			ftype_ptr->big_pixmap = XmUNSPECIFIED_PIXMAP;
			ftype_ptr->big_mask = XmUNSPECIFIED_PIXMAP;
			ftype_ptr->small_pixmap = XmUNSPECIFIED_PIXMAP;
			ftype_ptr->small_mask = XmUNSPECIFIED_PIXMAP;
			files->magic = ftype_ptr;

			count_current ++;
		}
	}
		
	pthread_mutex_unlock(&config_mutex);

	return;
}
  



void *LoadDirThread(void * arg)
{
	IfaceBranche * IbPtr = (IfaceBranche *) arg;
	register DIR *reading;
	register struct dirent *next;
	char *fullpath, newtitle[128];
	int linkadjust = 0, x;
	struct stat buf, buf2;
	mode_t mode_info;
	const char DirConst[]="DIR";
	const char DateConst[]="Date";
	int i, k, j, l;
	char sz[100], own[100], grp[100], prm[100], dt[100];
	char bumbuf[1024];
	char * tmp_str;
	char symlink_path[4096];
	ssize_t zt;
	int stat_res;

	struct filedata * files;
	int fileptr = 0, dirptr = 0;

	bxfm_ftype * ftype_ptr;
	int magic_finded = 0;

	errno = 0;

#ifdef DEBUG
	printf("Loaddir 1 %s\n", IbPtr->current_wd);
#endif

	reading = opendir(IbPtr->current_wd);
	if (!reading)
	{
#ifdef DEBUG
		printf("Loaddir: erroe openning dir %s\n", IbPtr->current_wd);
#endif
		IbPtr->load_dir_error_set = 1;
		if (errno == EACCES)
			IbPtr->load_dir_error_message = strdup("Permission denied");
		return NULL;
	}

	fullpath = calloc(strlen(IbPtr->current_wd) + 2048, sizeof(char));
	if (!fullpath)
	{
		IbPtr->load_dir_error_set = 1;
		if (errno == EACCES)
			IbPtr->load_dir_error_message = strdup("Memory insufficiency");
		return NULL;
	}
/**** ??????????????? *****/
/*	clear_dir_list(IbPtr);*/

  /*  if (files != NULL) {
    nfiles = 100;
    files = realloc(files, sizeof(struct filedata) * nfiles);
  } */
//printf("Loaddir 3\n");


//printf("Loaddir fileptr %d files\n", fileptr);
	
	IbPtr->files_ptr = calloc(FILES_ALOCCATED_N, sizeof(struct filedata *));
	IbPtr->filesallocated = FILES_ALOCCATED_N;

	IbPtr->dirs_ptr = calloc(FILES_ALOCCATED_N, sizeof(struct filedata *));
	IbPtr->dirsallocated = FILES_ALOCCATED_N;


	while ((next = readdir (reading)) != NULL) 
	{
		sprintf(fullpath, "%s/%s", IbPtr->current_wd, next->d_name);
		lstat(fullpath, &buf);
		
		mode_info = buf.st_mode;
		
		if (S_ISLNK(mode_info))
		{
			memset(symlink_path, 0, 4096);
			zt = readlink(fullpath, symlink_path, 4096);
			if ((stat_res = stat(fullpath, &buf2)) == 0)
			{
				if (S_ISDIR(buf2.st_mode))
				{
					mode_info = mode_info | (buf2.st_mode & S_IFMT);
				}
			}
		}
		
		if (check_hidden_dir(IbPtr, next->d_name, mode_info))
			continue;

		if (check_hidden_file(IbPtr, next->d_name, mode_info))
			continue;

		if (check_parent_dir(IbPtr, next->d_name, mode_info))
			continue;

		if (check_current_dir(IbPtr, next->d_name, mode_info))
			continue;

		files = calloc(1, sizeof(struct filedata));
//		if (S_ISDIR(mode_info))
		if (S_ISDIR(mode_info) || CHECK_IS_DIR_LNK(mode_info))
			IbPtr->dirs_ptr[dirptr] = files;
		else
			IbPtr->files_ptr[fileptr] = files;

		files->name = malloc(sizeof(char) * (strlen(next->d_name) + 1));
		strcpy(files->name, next->d_name);
//		printf("File %d: %s\n", fileptr, files[fileptr].name);
		files->magic = NULL;

		files->info = malloc(sizeof(struct fileinfo));
		if (files->info)
		{
			files->info->f_size = buf.st_size;
			files->info->f_atime = buf.st_atime;
			files->info->f_mtime = buf.st_mtime;
			files->info->f_ctime = buf.st_ctime;
			files->info->f_perm = buf.st_mode;
			files->info->f_owner = buf.st_uid;
			files->info->f_owngrp = buf.st_gid;
		}

//		sprintf(fullpath, "%s/%s", IbPtr->current_wd, next->d_name);
//		lstat(fullpath, &buf);

//		files->filetype = buf.st_mode;
		files->filetype = mode_info;
		if (((files->filetype & S_IFMT) == S_IFLNK) && !stat_res && zt > 0)
		{
			files->linkto = calloc(zt + 1, sizeof(char));
			if (files->linkto)
				strncpy(files->linkto, symlink_path, zt);
		}
/*		if (S_ISLNK(buf.st_mode)) {
			linkadjust = 1;
		}
		if (S_ISDIR(buf.st_mode))
			files->filetype = 0;
		else
			files->filetype = 2;

		files->filetype += linkadjust;
*/	
//		if (S_ISDIR(files->filetype))
		if (S_ISDIR(files->filetype) || CHECK_IS_DIR_LNK(files->filetype))
		{
			dirptr++;
			IbPtr->ndirs++;
			if ((dirptr + 10) > IbPtr->dirsallocated) {
				IbPtr->dirsallocated += 100;
				IbPtr->dirs_ptr = realloc(IbPtr->dirs_ptr, sizeof(struct filedata *) * IbPtr->dirsallocated);
			}
		}
		else
		{
			fileptr++;
			IbPtr->nfiles++;
			if ((fileptr + 10) > IbPtr->filesallocated) {
				IbPtr->filesallocated += 100;
				IbPtr->files_ptr = realloc(IbPtr->files_ptr, sizeof(struct filedata *) * IbPtr->filesallocated);
			}
		}
		linkadjust = 0;
//usleep(100000);		
		if (IbPtr->cancel_reading)
		{
//			IbPtr->cancel_reading = 0;
			closedir(reading);
//			IbPtr->load_dir_joined = 1;
#ifdef DEBUG
			printf("Loaddir stopeed in reading phase\n");
#endif
			if (fullpath)
				free(fullpath);
			return NULL;
		}
	}

	closedir(reading);

	if (IbPtr->nfiles == 0 && IbPtr->ndirs == 0)
	{
#ifdef DEBUG
		printf("Loaddir: readed empty directory, thread terminated\n");
#endif
		if (fullpath)
			free(fullpath);
		return NULL;
	}

#ifdef DEBUG
	printf("Loaddir dirs %d files %d\n", IbPtr->ndirs, IbPtr->nfiles);
/*	for (i = 0; i < IbPtr->ndirs; i++)
	{
		files = IbPtr->dirs_ptr[i];
		printf("dir: %s\n", files->name);
	}

	for (i = 0; i < IbPtr->nfiles; i++)
	{
		files = IbPtr->files_ptr[i];
		printf("file: %s\n", files->name);
	}
	printf("\n");*/
#endif
//	IbPtr->fileptr = fileptr;

	sort_dir(IbPtr);
#ifdef DEBUG
	printf("Loaddir 6\n");

	for (i = 0; i < IbPtr->ndirs; i++)
	{
		files = IbPtr->dirs_ptr[i];
		printf("dir: %s\n", files->name);
	}

	for (i = 0; i < IbPtr->nfiles; i++)
	{
		files = IbPtr->files_ptr[i];
		printf("file: %s\n", files->name);
	}
	printf("\n");
#endif
//usleep(100000);		


	if (IbPtr->cancel_reading)
	{
//		IbPtr->cancel_reading = 0;
//		IbPtr->load_dir_joined = 1;
#ifdef DEBUG
		printf("Loaddir stopeed after sorting\n");
#endif
		if (fullpath)
			free(fullpath);
		return NULL;
	}


/* Set propery for file */



	IbPtr->filesready = IbPtr->ndirs;

	for (i = 0; i < IbPtr->nfiles; i++)
	{
		files = IbPtr->files_ptr[i];
		sprintf(fullpath, "%s/%s", IbPtr->current_wd, files->name);

		magic_get_type(fullpath, bumbuf);

		magic_finded = 0;
		
		CreateFileType(fullpath, files,bumbuf);
		
		IbPtr->filesready++;


		if (IbPtr->cancel_reading)
		{
//			IbPtr->cancel_reading = 0;
//			IbPtr->load_dir_joined = 1;
#ifdef DEBUG
			printf("Loaddir stopeed in properties processing phase\n");
#endif
			if (fullpath)
				free(fullpath);
			return NULL;
		}
	}


// printf("===========\n");
//		i = count_current - 1;
//		while (i >= 0)
//	for (i = 0; i < count_current; i++)
//	{
//		printf("Magic = %s\t%s\t%s\t%s\t%s\n", (ftype_list[i])->magic, (ftype_list[i])->big_icon_name, (ftype_list[i])->small_icon_name, (ftype_list[i])->action, (ftype_list[i])->alter_name); 
//		i--;
//	}
//printf("===========\n");
 
#ifdef DEBUG
	for (i = 0; i < IbPtr->nfiles; i++)
	{
		printf("%s\t%s\t%s\t%lx\n", IbPtr->files_ptr[i]->name, IbPtr->files_ptr[i]->info->f_size, IbPtr->files_ptr[i]->info->f_mtime, IbPtr->files_ptr[i]->info->magic);
	}
#endif

	if (fullpath)
		free(fullpath);
	
	return NULL;
}
  
void clear_dir_list (IfaceBranche * IbPtr)
{
	register int i;
	struct filedata * files;
	
	if (!IbPtr->files_ptr && !IbPtr->dirs_ptr)
		return;

	for (i = 0; i < IbPtr->nfiles; i++)
	{
		files = IbPtr->files_ptr[i];
		if (files->name) 
			free (files->name);
		if (files->linkto)
			free (files->linkto);
		if (files->info)
			free(files->info);
    }

	for (i = 0; i < IbPtr->ndirs; i++)
	{
		files = IbPtr->dirs_ptr[i];
		if (files->name) 
			free (files->name);
		if (files->linkto)
			free (files->linkto);
		if (files->info)
			free(files->info);
    }

    free(IbPtr->files_ptr);
    IbPtr->files_ptr = NULL;
	IbPtr->filesallocated = 0;
	IbPtr->nfiles = 0;

    free(IbPtr->dirs_ptr);
    IbPtr->dirs_ptr = NULL;
	IbPtr->dirsallocated = 0;
	IbPtr->ndirs = 0;


	IbPtr->filesready = 0;
}

int compare_name_ascend(struct filedata ** fp1, struct filedata ** fp2);
int compare_name_descend(struct filedata ** fp1, struct filedata ** fp2);

int compare_date_ascend(struct filedata ** fp1, struct filedata ** fp2);
int compare_date_descend(struct filedata ** fp1, struct filedata ** fp2);

int compare_size_ascend(struct filedata ** fp1, struct filedata ** fp2);
int compare_size_descend(struct filedata ** fp1, struct filedata ** fp2);

void sort_dir(IfaceBranche *IbPtr)
{
	int (*func)();


	switch (IbPtr->SortingCategory) {
	    case SORTING_NAME:
			if (IbPtr->SortingType == SORTING_ASCEND)
				func = compare_name_ascend;
			else
				func = compare_name_descend;
			break ;
	    case SORTING_DATE:
			if (IbPtr->SortingType == SORTING_ASCEND)
				func = compare_date_ascend;
			else
				func = compare_date_descend;
			break ;
	    case SORTING_SIZE:
			if (IbPtr->SortingType == SORTING_ASCEND)
				func = compare_size_ascend;
			else
				func = compare_size_descend;
			break ;
	    default:
			return;
		}
	if (IbPtr->files_ptr && IbPtr->nfiles > 0)
		qsort(IbPtr->files_ptr, IbPtr->nfiles, sizeof (struct filedata **), func);

	if (IbPtr->dirs_ptr && IbPtr->ndirs > 0)
		qsort(IbPtr->dirs_ptr, IbPtr->ndirs, sizeof (struct filedata **), func);
}


int compare_name_ascend(struct filedata ** fp1, struct filedata ** fp2)
{
	struct filedata *file1, *file2;
	int ret;
	
	file1 = *fp1;
	file2 = *fp2;
    /* force .. to be the first item */
	if (!strcmp(file1->name, "."))
		if (strcmp(file2->name, "."))
			return -1;
		else
			return 0;
	else if (!strcmp(file2->name, "."))
		return 1;

	if (!strcmp(file1->name, ".."))
		if (strcmp(file2->name, ".."))
			return -1;
		else
			return 0;
	else if (!strcmp(file2->name, ".."))
		return 1;

    ret = strcmp(file1->name, file2->name);
    return ret;
}

int compare_name_descend(struct filedata ** fp1, struct filedata ** fp2)
{
	struct filedata *file1, *file2;
	int ret;
	
	file1 = *fp1;
	file2 = *fp2;
    /* force .. to be the first item */
	if (!strcmp(file1->name, "."))
		if (strcmp(file2->name, "."))
			return -1;
		else
			return 0;
	else if (!strcmp(file2->name, "."))
		return 1;

	if (!strcmp(file1->name, ".."))
		if (strcmp(file2->name, ".."))
			return -1;
		else
			return 0;
	else if (!strcmp(file2->name, ".."))
		return 1;

    ret = strcmp(file2->name, file1->name);
    return ret;
}

int compare_date_ascend(struct filedata ** fp1, struct filedata ** fp2)
{
	struct filedata *file1, *file2;
	int ret;
	
	file1 = *fp1;
	file2 = *fp2;
    /* force .. to be the first item */
	if (!strcmp(file1->name, "."))
		if (strcmp(file2->name, "."))
			return -1;
		else
			return 0;
	else if (!strcmp(file2->name, "."))
		return 1;

	if (!strcmp(file1->name, ".."))
		if (strcmp(file2->name, ".."))
			return -1;
		else
			return 0;
	else if (!strcmp(file2->name, ".."))
		return 1;

	if (file1->info->f_mtime < file2->info->f_mtime)
		return -1;
	else
		if (file1->info->f_mtime > file2->info->f_mtime)
			return 1;
		else
			return 0;
		
}

int compare_date_descend(struct filedata ** fp1, struct filedata ** fp2)
{
	struct filedata *file1, *file2;
	int ret;
	
	file1 = *fp1;
	file2 = *fp2;
    /* force .. to be the first item */
	if (!strcmp(file1->name, "."))
		if (strcmp(file2->name, "."))
			return -1;
		else
			return 0;
	else if (!strcmp(file2->name, "."))
		return 1;

	if (!strcmp(file1->name, ".."))
		if (strcmp(file2->name, ".."))
			return -1;
		else
			return 0;
	else if (!strcmp(file2->name, ".."))
		return 1;

	if (file1->info->f_mtime > file2->info->f_mtime)
		return -1;
	else
		if (file1->info->f_mtime < file2->info->f_mtime)
			return 1;
		else
			return 0;
		
}

int compare_size_ascend(struct filedata ** fp1, struct filedata ** fp2)
{
	struct filedata *file1, *file2;
	int ret;
	
	file1 = *fp1;
	file2 = *fp2;
    /* force .. to be the first item */
	if (!strcmp(file1->name, "."))
		if (strcmp(file2->name, "."))
			return -1;
		else
			return 0;
	else if (!strcmp(file2->name, "."))
		return 1;

	if (!strcmp(file1->name, ".."))
		if (strcmp(file2->name, ".."))
			return -1;
		else
			return 0;
	else if (!strcmp(file2->name, ".."))
		return 1;

	if (file1->info->f_size < file2->info->f_size)
		return -1;
	else
		if (file1->info->f_size > file2->info->f_size)
			return 1;
		else
			return 0;
		
}

int compare_size_descend(struct filedata ** fp1, struct filedata ** fp2)
{
	struct filedata *file1, *file2;
	int ret;
	
	file1 = *fp1;
	file2 = *fp2;
    /* force .. to be the first item */
	if (!strcmp(file1->name, "."))
		if (strcmp(file2->name, "."))
			return -1;
		else
			return 0;
	else if (!strcmp(file2->name, "."))
		return 1;

	if (!strcmp(file1->name, ".."))
		if (strcmp(file2->name, ".."))
			return -1;
		else
			return 0;
	else if (!strcmp(file2->name, ".."))
		return 1;

	if (file1->info->f_size > file2->info->f_size)
		return -1;
	else
		if (file1->info->f_size < file2->info->f_size)
			return 1;
		else
			return 0;
		
}


int compare_name(struct filedata ** fp1, struct filedata ** fp2)
{
	struct filedata *file1, *file2;
	int ret;
	
	file1 = *fp1;
	file2 = *fp2;
    /* force .. to be the first item */
	if (!strcmp(file1->name, "."))
		if (strcmp(file2->name, "."))
			return -1;
		else
			return 0;
	else if (!strcmp(file2->name, "."))
		return 1;

	if (!strcmp(file1->name, ".."))
		if (strcmp(file2->name, ".."))
			return -1;
		else
			return 0;
	else if (!strcmp(file2->name, ".."))
		return 1;

//	if (dirs_first) {


	if ((S_ISDIR(file1->filetype) || CHECK_IS_DIR_LNK(file1->filetype)) &&
		(!S_ISDIR(file2->filetype) && !CHECK_IS_DIR_LNK(file2->filetype)))
		return -1;

	if ((S_ISDIR(file2->filetype) || CHECK_IS_DIR_LNK(file2->filetype)) &&
		(!S_ISDIR(file1->filetype) && !CHECK_IS_DIR_LNK(file1->filetype)))
		return 1;


/*	if (file1->filetype < 2) {
		if (file2->filetype > 1)
		return -1;
	} 
	else if (file2->filetype < 2)
		return 1;
*/
//	}
    ret = strcmp(file1->name, file2->name);
    return ret;
//    return reverse?(-ret):ret;


//  return strcmp (file1->name, file2->name);
}

