/*	bcm_wdt:	A BCM96352 Hardware Watchdog Device Driver
 *	David Kind, 20.03.2002
 *
 *	Taken and modified from:
 *	SoftDog	0.05:	A Software Watchdog Device
 *
 *	(c) Copyright 1996 Alan Cox <alan@redhat.com>, All Rights Reserved.
 *				http://www.redhat.com
 *
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public License
 *	as published by the Free Software Foundation; either version
 *	2 of the License, or (at your option) any later version.
 *
 *	Neither Alan Cox nor CymruNet Ltd. admit liability nor provide
 *	warranty for any of this software. This material is provided
 *	"AS-IS" and at no charge.
 *
 *	(c) Copyright 1995    Alan Cox <alan@lxorguk.ukuu.org.uk>
 *
 */

#define EXPORT_SYMTAB
#define MODVERSIONS

#include <linux/config.h>
#include <linux/module.h>
#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/fs.h>
#include <linux/miscdevice.h>
#include <linux/watchdog.h>
#include <linux/reboot.h>
#include <linux/smp_lock.h>
#include <linux/init.h>
#include <linux/sched.h>
#include <asm/uaccess.h>
#include <asm/semaphore.h>
#include <linux/tqueue.h>
#include <linux/wait.h>
#include <linux/signal.h>

#include "kthread.h"

/* BCM96352 Specific header files */
#ifndef TYPEDEFS_H
typedef unsigned char   byte;
typedef unsigned char   uint8;
typedef unsigned short  uint16;
typedef unsigned long   uint32;
typedef signed char     int8;
typedef signed short    int16;
typedef signed long     int32;
#endif

typedef struct GpioControl {
  uint16        unused0;
  byte          unused1;
  byte          TBusSel;
  uint16        unused2;
  uint16        GPIODir;
  byte          unused3;
  byte          Leds;           //Only bits [3:0]
  uint16        GPIOio;
  uint32        UartCtl;
} GpioControl;

#define GPIO_BASE   0xfffe0400    /* gpio registers */
#define GPIO        ((volatile GpioControl * const) GPIO_BASE)

/* ** Timer ** */
typedef struct Timer {
  uint16        unused0;
  byte          TimerMask;
  byte          TimerInts;
  uint32        TimerCtl0;
  uint32        TimerCtl1;
  uint32        TimerCtl2;
  uint32        TimerCnt0;
  uint32        TimerCnt1;
  uint32        TimerCnt2;
  uint32        WatchDogDefCount;
  uint32        WatchDogCtl;
  uint32        WDResetCount;
} Timer;

#define TIMR_BASE   0xfffe0200    /* timer registers */
#define WATCHDOG    0x08
#define TIMER       ((volatile Timer * const) TIMR_BASE)

#define TWO_SECOND_TIMEOUT              (4*HZ)
#define WDT_DESCRIPTION					"HW Watchdog Timer BCM96352.\n"
#define ALERT_THREAD_DESCRIPTION        "AlertLEDd"
/* Note: Freq. from //phobos/tools/kernel/mips/linux/include/asm-mips/board-bcm.h */
#define WDT_COUNT                       0x47868C00  // Gives approx 30 seconds.
#define WDT_RESET_COUNT                 0xFFFFFFFF
#define ALWAYS                          1

/* MACRO Defines */
#define WATCHDOG_STOP()                 TIMER->WatchDogCtl = 0xEE00; \
	                                    TIMER->WatchDogCtl = 0x00EE
#define WATCHDOG_START()                TIMER->WatchDogCtl = 0xFF00; \
	                                    TIMER->WatchDogCtl = 0x00FF

/* Local variables & structures */
static int bcmdog_alive;
kthread_t aled_t;

/* Thread prototype */
static void aled_thread(kthread_t *kthread);

/* ************************************************************
 * ************** K T H R E A D  C O D E **********************
 * ************************************************************ */

/* private functions */
static void kthread_launcher(void *data)
{
    kthread_t *kthread = data;
    kernel_thread((int (*)(void *))kthread->function, (void *)kthread, 0);
}

/* public functions */

/* create a new kernel thread. Called by the creator. */
void start_kthread(void (*func)(kthread_t *), kthread_t *kthread)
{
    /*  initialize the semaphore:
        we start with the semaphore locked. The new kernel
        thread will setup its stuff and unlock it. This
        control flow (the one that creates the thread) blocks
        in the down operation below until the thread has reached
        the up() operation.
    */
    init_MUTEX_LOCKED(&kthread->startstop_sem);

    /* store the function to be executed in the data passed to the launcher */
    kthread->function=func;

    /* create the new thread my running a task through keventd */

    /* initialize the task queue structure */
    kthread->tq.sync = 0;
    INIT_LIST_HEAD(&kthread->tq.list);
    kthread->tq.routine =  kthread_launcher;
    kthread->tq.data = kthread;

    /* and schedule it for execution */
    schedule_task(&kthread->tq);

    /* wait till it has reached the setup_thread routine */
    down(&kthread->startstop_sem);
}

/* stop a kernel thread. Called by the removing instance */
void stop_kthread(kthread_t *kthread)
{
    if (kthread->thread == NULL)
    {
        printk("stop_kthread: killing non existing thread!\n");
        return;
    }

    /*  this function needs to be protected with the big
	    kernel lock (lock_kernel()). The lock must be
        grabbed before changing the terminate
	    flag and released after the down() call.
    */
    lock_kernel();

    /*  initialize the semaphore. We lock it here, the
        leave_thread call of the thread to be terminated
        will unlock it. As soon as we see the semaphore
        unlocked, we know that the thread has exited.
    */
    init_MUTEX_LOCKED(&kthread->startstop_sem);

    /*  We need to do a memory barrier here to be sure that
        the flags are visible on all CPUs.
    */
    mb();

    /* set flag to request thread termination */
    kthread->terminate = 1;

    /*  We need to do a memory barrier here to be sure that
        the flags are visible on all CPUs.
    */
    mb();
    kill_proc(kthread->thread->pid, SIGKILL, 1);

    /* block till thread terminated */
    down(&kthread->startstop_sem);

    /* release the big kernel lock */
    unlock_kernel();

    /*  now we are sure the thread is in zombie state. We
        notify keventd to clean the process up.
    */
    kill_proc(2, SIGCHLD, 1);
}

/* initialize new created thread. Called by the new thread. */
void init_kthread(kthread_t *kthread, char *name)
{
    /*  lock the kernel. A new kernel thread starts without
        the big kernel lock, regardless of the lock state
        of the creator (the lock level is *not* inheritated)
    */
    lock_kernel();

    /* fill in thread structure */
    kthread->thread = current;

    /* set signal mask to what we want to respond */
    siginitsetinv(&current->blocked, sigmask(SIGKILL)|sigmask(SIGINT)|sigmask(SIGTERM)|sigmask(SIGUSR1));

    /* initialise wait queue */
    init_waitqueue_head(&kthread->queue);

    /* initialise termination flag */
    kthread->terminate = 0;

    /* set name of this process (max 15 chars + 0 !) */
    sprintf(current->comm, name);

    /* let others run */
    unlock_kernel();

    /* tell the creator that we are ready and let him continue */
    up(&kthread->startstop_sem);
}

/* cleanup of thread. Called by the exiting thread. */
void exit_kthread(kthread_t *kthread)
{
    /* we are terminating */

	/* lock the kernel, the exit will unlock it */
    lock_kernel();
    kthread->thread = NULL;
    mb();

    /* notify the stop_kthread() routine that we are terminating. */
	up(&kthread->startstop_sem);
	/* the kernel_thread that called clone() does a do_exit here. */

	/* there is no race here between execution of the "killer" and real termination
	   of the thread (race window between up and do_exit), since both the
	   thread and the "killer" function are running with the kernel lock held.
	   The kernel lock will be freed after the thread exited, so the code
	   is really not executed anymore as soon as the unload functions gets
	   the kernel lock back.
	   The init process may not have made the cleanup of the process here,
	   but the cleanup can be done safely with the module unloaded.
	*/
}


/* ************************************************************
 * ************** A L E R T L E D  C O D E ********************
 * ************************************************************ */

/* Prototype definitions */
static void aled_thread(kthread_t *kthread)
{
	volatile GpioControl * gpio = (volatile GpioControl *)(GPIO_BASE);
    signed long timeout;

    /* setup the thread environment */
    init_kthread(kthread, ALERT_THREAD_DESCRIPTION);

    /* Alert LED is on GPIO19 - switch it OFF. */
    gpio->Leds |= 8;

    /* Thread - main loop */
    while ( ALWAYS )
    {
        set_current_state(TASK_INTERRUPTIBLE);
        schedule();

        if ( kthread->terminate & 0x01 ) break;

        /* Alert LED is on GPIO19 - switch it ON. */
        gpio->Leds &= ~8;
        timeout = TWO_SECOND_TIMEOUT;
        while ( timeout )
        {
            /*  Remain in this loop for specified time.
             *  Ignore all other SIGUSR1 signals.
             */
            spin_lock(&current->sigmask_lock);
            flush_signals(current);
            spin_unlock(&current->sigmask_lock);

            set_current_state(TASK_INTERRUPTIBLE);
            timeout = schedule_timeout(timeout);

            if ( kthread->terminate & 0x01 ) break;
        }

        if ( kthread->terminate & 0x01 ) break;

        /* Alert LED is on GPIO19 - switch it OFF. */
        gpio->Leds |= 8;
        timeout = TWO_SECOND_TIMEOUT;
        while ( timeout )
        {
            /*  Remain in this loop for specified time.
             *  Ignore all other SIGUSR1 signals.
             */
            spin_lock(&current->sigmask_lock);
            flush_signals(current);
            spin_unlock(&current->sigmask_lock);

            set_current_state(TASK_INTERRUPTIBLE);
            timeout = schedule_timeout(timeout);

            if ( kthread->terminate & 0x01 ) break;
        }

        if ( kthread->terminate & 0x01 ) break;

        spin_lock(&current->sigmask_lock);
        flush_signals(current);
        spin_unlock(&current->sigmask_lock);
    }

    /* Alert LED is on GPIO19 - switch it OFF. */
    gpio->Leds |= 8;

    /* cleanup the thread and leave - have received the stop signal. */
    exit_kthread(kthread);

    /* Returning from the thread here calls the exit functions */
}

void deimos_turn_on_led(void)
{
    /* Send the LED ON signal to the thread. */
    send_sig(SIGUSR1, aled_t.thread, 1);
}

EXPORT_SYMBOL(deimos_turn_on_led);

/* ************************************************************
 * ************** W A T C H D O G  C O D E ********************
 * ************************************************************ */

/* Private */
static void bcmdog_ping()
{
    /* Stop & Reload the timer */
    WATCHDOG_START();
}

/* Public */
/* **** Allow only one person to hold it open **** */
static int bcmdog_open(struct inode *inode, struct file *file)
{
	if ( bcmdog_alive )
		return -EBUSY;

	bcmdog_ping();

	bcmdog_alive=1;
	return 0;
}

static int bcmdog_release(struct inode *inode, struct file *file)
{
	/* **** Shut off the timer. **** */
	/* 	Lock it in if it's a module and we defined ...NOWAYOUT */
    lock_kernel();
#ifndef CONFIG_WATCHDOG_NOWAYOUT
	{
        WATCHDOG_STOP();
	}
#endif
	bcmdog_alive=0;
	unlock_kernel();
	return 0;
}

static ssize_t bcmdog_write(struct file *file, const char *data, size_t len, loff_t *ppos)
{
	/*  Can't seek (pwrite) on this device  */
	if (ppos != &file->f_pos)
		return -ESPIPE;

	/* **** Refresh the timer. **** */
	bcmdog_ping();
	return (1);
}

static int bcmdog_ioctl(struct inode *inode, struct file *file, unsigned int cmd, unsigned long arg)
{
	static struct watchdog_info ident = {
		identity: WDT_DESCRIPTION,
	};

	switch (cmd) {
		default:
			return -ENOIOCTLCMD;
		case WDIOC_GETSUPPORT:
			if(copy_to_user((struct watchdog_info *)arg, &ident, sizeof(ident)))
				return -EFAULT;
			return 0;
		case WDIOC_GETSTATUS:
		case WDIOC_GETBOOTSTATUS:
			return put_user(0,(int *)arg);
		case WDIOC_KEEPALIVE:
			bcmdog_ping();
			return 0;
	}
}

static struct file_operations bcmdog_fops = {
	owner:		THIS_MODULE,
	write:		bcmdog_write,
	ioctl:		bcmdog_ioctl,
	open:		bcmdog_open,
	release:	bcmdog_release,
};

static struct miscdevice bcmdog_miscdev = {
	minor:		WATCHDOG_MINOR,
	name:		"watchdog",
	fops:		&bcmdog_fops,
};

static const char banner[] __initdata = KERN_INFO WDT_DESCRIPTION;

static int __init watchdog_init(void)
{
	int ret;

    /* Also initialising the Thread structure for Alert LED */
    aled_t.thread = NULL;
    start_kthread(aled_thread, &aled_t);

    /* Set the Watchdog Counters */
    TIMER->TimerInts &= (0xFF - WATCHDOG); // Ensure Interrupt disabled.

    TIMER->WatchDogDefCount = WDT_COUNT;
	TIMER->WDResetCount =  WDT_RESET_COUNT;

    WATCHDOG_STOP();

	ret = misc_register(&bcmdog_miscdev);

	if (ret)
		return ret;

	printk(banner);

	return 0;
}

static void __exit watchdog_exit(void)
{
	misc_deregister(&bcmdog_miscdev);
    stop_kthread(&aled_t);
}

module_init(watchdog_init);
module_exit(watchdog_exit);
