/* #(@) $Id: asetup.c 1.2 Sat, 08 Sep 2001 22:26:28 +0200 mxp $
 * asetup - set up audio preferences without running AudioCP
 *
 * One problem (at least with some revisions) of the HP audio system
 * is that the audio preferences (stored in $HOME/.audioCP) are only
 * read when AudioCP is run.  This means that audio applications won't
 * use the preferences unless the user has manually started AudioCP.
 *
 * This program can be used to set up the preferences automatically,
 * without having to run AudioCP.  It is intended to be called from a
 * startup file like $HOME/.profile or $HOME/.dt/sessions/sessionetc.
 *
 * Michael Piotrowski <mxp@dynalabs.de> 2001-08-18
 */

#include <stdlib.h>
#include <stdio.h>
#include <pwd.h>
#include <string.h>
#include <errno.h>
#include <Alib.h>

#define LEN_PREF_LINE 64 /* Maximum length of .audioCP entry */

static char ident[] = "@(#) $Id: asetup.c 1.2 Sat, 08 Sep 2001 22:26:28 +0200 mxp $";

int main(void)
{
   Audio          *audio;
   long            status;
   AOutputDstMask  outputDevMask = 0;
   AGainDB         monitor_gain;
   AGainDB         play_gain;

   uid_t           uid;
   struct passwd  *passwd;
   char           *homedir;

   FILE           *prefs;
   char            line[LEN_PREF_LINE];
   char           *fstatus;
   char           *tok;

   /*
    * Open audio server
    */

   audio = AOpenAudio("", &status);

   if (status != AENoError)
   {
      printf("Could not open audio server, status return == %ld\n", status);
      exit(EXIT_FAILURE);
   }

   /*
    * Try to find out the home directory
    */

   if ((homedir = getenv("HOME")) == NULL)
   {
      if ((passwd = getpwuid(uid = getuid())) == NULL)
      {
	 endpwent();
	 printf("Cannot determine home directory for uid %d\n", uid);
	 
	 return(EXIT_FAILURE);
      }

      endpwent();
      homedir = passwd->pw_dir;
   }

   /*
    * Open ~/.audioCP file to read the user preferences
    */

   prefs = fopen(strcat(homedir, "/.audioCP"), "r");
   if (prefs == NULL)
   {
      printf("Cannot open audio preferences file: %s\n", strerror(errno));
      exit(EXIT_FAILURE);
   }

   /*
    * Read preferences entries
    */

   while ((fstatus = fgets(line, LEN_PREF_LINE - 1, prefs)))
   {
      tok = strtok(line, ":\n ");

      if (strcmp(tok, "outputSpeaker") == 0)
      {
	 tok = strtok(NULL, ":\n ");

	 if (strcmp(tok, "True") == 0)
	    outputDevMask |= ALeftIntSpeakerMask | ARightIntSpeakerMask;
      }
      else if (strcmp(tok, "outputHeadphones") == 0)
      {
	 tok = strtok(NULL, ":\n ");

	 if (strcmp(tok, "True") == 0)
	    outputDevMask |= ALeftHeadphoneMask | ARightHeadphoneMask;
      }
      else if (strcmp(tok, "outputLine") == 0)
      {
	 tok = strtok(NULL, ":\n ");

	 if (strcmp(tok, "True") == 0)
	    outputDevMask |= ALeftLineOutMask | ARightLineOutMask;
      }
      else if (strcmp(tok, "monitoring") == 0)
      {
	 tok = strtok(NULL, ":\n ");

	 if (strcmp(tok, "True") == 0)
	    monitor_gain = AUnityGain;
	 else if (strcmp(tok, "False") == 0)
   	    monitor_gain = -100; /* AZeroGain *should* turn off sound,
				  * but doesn't seem to */
      }
      else if (strcmp(tok, "playVolume") == 0)
      {
	 int min_gain, max_gain, gain_range;
	 int play_volume;

	 tok = strtok(NULL, ":\n");
	 play_volume = atoi(tok);

	 /* The play volume is stored in .audioCP as an integer
	  * between 0 and 100, whereas the Audio library requires a
	  * gain in dB.  The possible gain range is
	  * system-dependent. */
	 min_gain   = AMinOutputGain(audio);
	 max_gain   = AMaxOutputGain(audio);
	 gain_range = abs(min_gain) + abs(max_gain);
         play_gain  = gain_range * play_volume / 100 + min_gain;
      }
      else
      {
	 /* No action defined */
      }
   }

   /*
    * Close preferences file
    */

   fclose(prefs);

   /*
    * Set audio parameters
    */

   ASetDefaultOutputs(audio, outputDevMask, &status);
   if (status != AENoError)
   {
      printf("Could not set default outputs, status return == %ld\n", status);
      exit(EXIT_FAILURE);
   }

   ASetSystemPlayGain(audio, play_gain, &status);
   if (status != AENoError)
   {
      printf("Could not set play gain, status return == %ld\n", status);
      exit(EXIT_FAILURE);
   }

   ASetSystemChannelGain(audio, ASGTMonitor, ACTMono, monitor_gain, &status);
   if (status != AENoError)
   {
      printf("Cannot set monitor gain, status return == %ld\n", status);
      exit(EXIT_FAILURE);
   }

   /*
    * Close audio connection and exit
    */

   ACloseAudio(audio, NULL);
   exit(EXIT_SUCCESS);
}
