/* $Id: nlist.cxx,v 1.3 1997/04/11 14:29:23 cnidr Exp $ */
/************************************************************************
Copyright Notice

Copyright (c) MCNC, Clearinghouse for Networked Information Discovery
and Retrieval, 1994.

Permission to use, copy, modify, distribute, and sell this software and
its documentation, in whole or in part, for any purpose is hereby
granted without fee, provided that

1. The above copyright notice and this permission notice appear in all
copies of the software and related documentation. Notices of copyright
and/or attribution which appear at the beginning of any file included in
this distribution must remain intact.

2. Users of this software agree to make their best efforts (a) to return
to MCNC any improvements or extensions that they make, so that these may
be included in future releases; and (b) to inform MCNC/CNIDR of
noteworthy uses of this software.

3. The names of MCNC and Clearinghouse for Networked Information
Discovery and Retrieval may not be used in any advertising or publicity
relating to the software without the specific, prior written permission
of MCNC/CNIDR.

THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND,
EXPRESS, IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY
WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.

IN NO EVENT SHALL MCNC/CNIDR BE LIABLE FOR ANY SPECIAL, INCIDENTAL,
INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY KIND, OR ANY DAMAGES WHATSOEVER
RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER OR NOT ADVISED OF
THE POSSIBILITY OF DAMAGE, AND ON ANY THEORY OF LIABILITY, ARISING OUT
OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
************************************************************************/


/*@@@
File:		nlist.cxx
Version:	1.00
$Revision: 1.3 $
Description:	Class NUMERICLIST
Author:		Jim Fullton, Jim.Fullton@cnidr.org
@@@*/

#include <stdlib.h>

#include "gdt.h"
#include "defs.hxx"
#include "string.hxx"
#include "nfield.hxx"
#include "nlist.hxx"

NUMERICLIST::NUMERICLIST()
{
  Ncoords = 2;
  table=new NUMERICFLD[100];
  Count=0;
  MaxEntries=100;
  FileName="";
  Pointer=0;
  StartIndex=EndIndex=-1;
}

NUMERICLIST::NUMERICLIST(INT n)
{
  Ncoords    = n;
  table      = new NUMERICFLD[50*Ncoords];
  Count      = 0;
  MaxEntries = 50*Ncoords;
  FileName   = "";
  Pointer    = 0;
  StartIndex = EndIndex = -1;
}

void NUMERICLIST::ResetHitPosition()
{
  if (Relation != 1)
    Pointer = StartIndex;
  else
    Pointer = 0;
}

void NUMERICLIST::SetRelation(INT r)
{
  Relation = r;
}

INT NUMERICLIST::GetRelation()
{
  return(Relation);
}

INT4 NUMERICLIST::GetNextHitPosition()
{
  INT4 Value;

  if(StartIndex == -1)
    return(-1);
  if(Relation != 1) {
    if(Pointer>EndIndex)
      return(-1);
    Value=table[Pointer].GetGlobalStart();
    ++Pointer;
    return(Value);
  } else {
    if(Pointer >= StartIndex)
      while(Pointer <= EndIndex)
	++Pointer;
    if(Pointer >= Count)
      return(-1);
    Value=table[Pointer].GetGlobalStart();
    ++Pointer;
    return(Value);
  }
}

INT NUMERICLIST::GetAttribute()
{
  return(Attribute);
}

void NUMERICLIST::SetAttribute(INT x)
{
  Attribute = x;
}

DOUBLE NUMERICLIST::GetNumericValue(INT i)
{
  return(table[i].GetNumericValue());
}

static int SortCmp(const void* x, const void* y) 
{
  DOUBLE a;
  a=((*((PNUMERICFLD)x)).GetNumericValue()) -
    ((*((PNUMERICFLD)y)).GetNumericValue()) ;
  if (a<0)
    return(-1);
  if (a>0)
    return(1);
  return(0);
}

void NUMERICLIST::SetFileName(STRING s)
{
  FileName = s;
}

void NUMERICLIST::SetFileName(PCHR s)
{
  FileName = s;
}

INT4 NUMERICLIST::GetCount()
{
  return(Count);
}

void NUMERICLIST::Expand()
{
  Resize(Count + (50*Ncoords));
}

void NUMERICLIST::Cleanup()
{
  Resize(Count);
}

void NUMERICLIST::Resize(INT4 Entries)
{
  PNUMERICFLD temp=new NUMERICFLD[Entries];
  INT4 CopyCount;
  INT4 x;

  if(Entries>Count)
    CopyCount=Count;
  else {
    CopyCount=Entries;
    Count=Entries;
  }
  for(x=0; x<CopyCount; x++) {
    temp[x]=table[x];
  }
  if(table)
    delete [] table;
  table=temp;
  MaxEntries=Entries;
}

void NUMERICLIST::Sort()
{
  qsort((void *)table,Count, sizeof(NUMERICFLD),SortCmp);
}

#define GE 4
#define GT 5
#define LE 2
#define LT 1

#define ERROR -1
#define TOO_LOW 0
#define TOO_HIGH 1
#define MATCH 2

INT4 Matcher(DOUBLE Key, DOUBLE A, DOUBLE B, DOUBLE C, 
	INT4 Relation, INT4 Type)
{
	
  switch (Relation)
    {
    case GE:			// greater than or equals
      if ((B>=Key) && (Type==-1 || A<Key))
	return(MATCH);		// exact place - lower boundary
      else if (A>=Key)
	return(TOO_LOW);		// key too low
      else
	return(TOO_HIGH);		// key too high
    case GT:			// greater than
      if ((B>Key) && (Type==-1 || A<=Key))
	return(MATCH);		// exact place - lower boundary
      else if (A>Key)
	return(TOO_LOW);		// key too low
      else
	return(TOO_HIGH);		// key too high
    case LE:			// less than or equals
      if ((B<=Key) && (Type==0 || C>Key))
	return(MATCH);		// exact place - upper boundary
      else if (C<=Key)
	return(TOO_HIGH);
      else
	return(TOO_LOW);
    case LT:			// less than
      if ((B<Key) && (Type==0 || C>=Key))
	return(MATCH);		// exact place - upper boundary
      else if (C<Key)
	return(TOO_HIGH);
      else
	return(TOO_LOW);
    }
  printf("Hideous Matching Error\n");
  return(ERROR);
}

INT4 NUMERICLIST::DiskFind(STRING Fn, DOUBLE Key, INT4 Relation)
{
  //  INT4 B;
  DOUBLE A,C;
  PFILE Fp = fopen(Fn, "r");
  if (!Fp) {
    perror(Fn);
    exit(1);
  }
  else {
    //    fseek(Fp, 0, 2);
    fseek(Fp, 0L, SEEK_END);
    INT Total = ftell(Fp) / ( sizeof(INT4) + sizeof(DOUBLE) );
    INT Low = 0;
    INT High = Total - 1;
    INT X = High / 2;
    INT OX;
    INT State=0;		// 0 = too low 1 = too high 2 = match 
    INT Type;
    INT4 GpS, Dummy;
    DOUBLE NumericValue;
    do {
      OX = X;
      if((X > 0) && (X != High)) {
	fseek(Fp, (long)((X-1) * (sizeof(INT4) + sizeof(DOUBLE))), SEEK_SET);
	Type=1;
      }
      else if (X == 0) {
	fseek(Fp, (long)(X * (sizeof(INT4) + sizeof(DOUBLE))), SEEK_SET);
	Type=-1;
      } else if (X == High) {
	fseek(Fp, (long)((X-1) * (sizeof(INT4) + sizeof(DOUBLE))), SEEK_SET);
	Type=0;
      }
      if (Type > -1) {
#ifndef __SUNPRO_CC
	fread((void *)&GpS, 1, sizeof(INT4), Fp);
	fread((void *)&A, 1, sizeof(DOUBLE), Fp);
#else
	fread((char *)&GpS, 1, sizeof(INT4), Fp);
	fread((char *)&A, 1, sizeof(DOUBLE), Fp);
#endif	
      }

#ifndef __SUNPRO_CC
      fread((void *)&GpS, 1, sizeof(INT4), Fp);
      fread((void *)&NumericValue, 1, sizeof(DOUBLE), Fp);
#else
      fread((char *)&GpS, 1, sizeof(INT4), Fp);
      fread((char *)&NumericValue, 1, sizeof(DOUBLE), Fp);
#endif      

      if(Type != 0) {
#ifndef __SUNPRO_CC
	fread((void *)&Dummy, 1, sizeof(INT4), Fp);
	fread((void *)&C, 1, sizeof(DOUBLE), Fp);
#else
	fread((char *)&Dummy, 1, sizeof(INT4), Fp);
	fread((char *)&C, 1, sizeof(DOUBLE), Fp);
#endif
      }
      
      //   printf("Max=%d Type=%d X=%d High=%d GpS %d Key=%f A=%f B=%f C=%f\n",
      // 	     Total,Type,X,High,GpS,Key,A,NumericValue,C);
      State = Matcher(Key,A,NumericValue,C,Relation,Type);
      if (State == ERROR) {
	exit(0);
      }
      if (State == MATCH) {
	fclose(Fp);
	return X;
      }
      if (State == TOO_LOW) {
	High = X;
      } else {
	Low = X + 1;
      }
      X = (Low + High) / 2;
      if (X < 0) {
	X = 0;
      } else {
	if (X >= Total) {
	  X = Total - 1;
	}
      }
    } while (X != OX);
  }
  fclose(Fp);
  return -1;
}

INT4 NUMERICLIST::GetGlobalStart(INT4 i)
{
  return(table[i].GetGlobalStart());
}

void NUMERICLIST::Dump(INT4 start, INT4 end)
{
  INT4 x;
  
  for(x=0; x<Count; x++)
    printf("start: %d value: %f\n",table[x].GetGlobalStart(),
	   table[x].GetNumericValue());
  
}

void NUMERICLIST::TempLoad()
{
  INT4 x,y=0;
  DOUBLE z;
  FILE *fp=fopen("test.62","w");
  
  for(x=0; x<100; x+=4){
    z=x;
    fwrite(&x,1,sizeof(INT4),fp);
    x+=4;
    fwrite(&z,1,sizeof(DOUBLE),fp);
    printf("Wrote %f\n",z);
    
  }
  fclose(fp);
  fp=fopen("test.62","r");
  while((y=fread(&x,1,sizeof(INT4),fp))!=0) {
    printf("x: %d\n",x);
    fread(&z,1,sizeof(DOUBLE),fp);
    printf("z(double): %f\n",z);
  }
  fclose(fp);
}

void NUMERICLIST::LoadTable(INT4 Start, INT4 End)
{
  FILE *fp;
  GPTYPE high;
  INT4 x,i;
  DOUBLE value;
  
  if(!(FileName.GetLength())) {
    printf("FileName not set\n");
    return;
  }
  
  if(End==-1)
    End=999999999;
  if(Start==-1)
    Start=0;
  
  fp=fopen(FileName,"r");
  if(fp){
    //    fseek(fp,Start*(sizeof(GPTYPE)+sizeof(DOUBLE)),0);
    fseek(fp, (long)(Start*(sizeof(GPTYPE)+sizeof(DOUBLE))), SEEK_SET);
    for(i=Start;i<End;i++){
      x=fread(&high,1,sizeof(GPTYPE),fp);
      if(x==0)
	break;
      x=fread(&value,1,sizeof(DOUBLE),fp);
      table[Count].SetGlobalStart(high);
      table[Count].SetNumericValue(value);
      Count++;
      if(Count==MaxEntries)
	Expand();
    }
    fclose(fp);
    Cleanup();
  }
}

void NUMERICLIST::WriteTable()
{
  PFILE fp;
  INT4 x;
  GPTYPE Val;
  DOUBLE fVal;
  
  if(!(FileName.GetLength())){
    printf("FileName not set\n");
    return;
  }
  fp=fopen(FileName,"wb");
  if(fp){
    for(x=0; x<Count; x++) {
      Val = table[x].GetGlobalStart();
      fVal = table[x].GetNumericValue();
      fwrite(&Val,1,sizeof(GPTYPE),fp);
      fwrite(&fVal,1,sizeof(DOUBLE),fp);
    }
  }
  fclose(fp);
}

void NUMERICLIST::SetCoords(INT n) {
  Ncoords = n;
}

INT NUMERICLIST::GetCoords() {
  return Ncoords;
}

NUMERICLIST::~NUMERICLIST()
{
  if(table)
    delete [] table;
}

#ifdef NEVER
main()
{
  NUMERICLIST list;
  STRING n;
  INT4 Start,End;
  DOUBLE val;
  n="test.62";
  printf("DOUBLE is %d bytes\n",sizeof(DOUBLE));
  list.TempLoad();
  End=list.DiskFind(n, (double)88.0,5); // 5 GT
  printf("\n===\n");
  Start=list.DiskFind(n,(double)88.0,1); // 1 LT
  
  if(End-Start<=1)
    printf("No Match!\n");
  list.SetFileName(n.NewCString());
  list.LoadTable(Start+1,End);
    
}
#endif

