/* $Id: intlist.cxx,v 1.4 1997/04/11 14:29:18 cnidr Exp $ */
/************************************************************************
Copyright Notice

Copyright (c) MCNC, Clearinghouse for Networked Information Discovery
and Retrieval, 1994.

Permission to use, copy, modify, distribute, and sell this software and
its documentation, in whole or in part, for any purpose is hereby
granted without fee, provided that

1. The above copyright notice and this permission notice appear in all
copies of the software and related documentation. Notices of copyright
and/or attribution which appear at the beginning of any file included in
this distribution must remain intact.

2. Users of this software agree to make their best efforts (a) to return
to MCNC any improvements or extensions that they make, so that these may
be included in future releases; and (b) to inform MCNC/CNIDR of
noteworthy uses of this software.

3. The names of MCNC and Clearinghouse for Networked Information
Discovery and Retrieval may not be used in any advertising or publicity
relating to the software without the specific, prior written permission
of MCNC/CNIDR.

THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND,
EXPRESS, IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY
WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.

IN NO EVENT SHALL MCNC/CNIDR BE LIABLE FOR ANY SPECIAL, INCIDENTAL,
INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY KIND, OR ANY DAMAGES WHATSOEVER
RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER OR NOT ADVISED OF
THE POSSIBILITY OF DAMAGE, AND ON ANY THEORY OF LIABILITY, ARISING OUT
OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
************************************************************************/


/*@@@
File:		intlist.cxx
Version:	1.00
$Revision: 1.4 $
Description:	Class INTLIST
Author:		Jim Fullton, Jim.Fullton@cnidr.org
@@@*/

#include <stdlib.h>
#include <time.h>
#include "gdt.h"
#include "defs.hxx"
#include "string.hxx"
#include "nfield.hxx"
#include "nlist.hxx"
#include "intfield.hxx"
#include "intlist.hxx"

static INT SortStartCmp(const void* x, const void* y);
static INT SortEndCmp(const void* x, const void* y);

INTERVALLIST::INTERVALLIST()
{
  Ncoords = 3;
  table=new INTERVALFLD[100];
  Count=0;
  MaxEntries=100;
  FileName="";
  Pointer=0;
  StartIndex=EndIndex=-1;
}

INTERVALLIST::INTERVALLIST(INT n)
{
  Ncoords    = n;
  table      = new INTERVALFLD[50*Ncoords];
  Count      = 0;
  MaxEntries = 50*Ncoords;
  FileName   = "";
  Pointer    = 0;
  StartIndex = EndIndex = -1;
}

void INTERVALLIST::LoadTable(INT4 Start, INT4 End)
// This is only called when the table has been written out during the
// index creation phase, before the file has been sorted (with both
// components dumped)
{
  FILE *fp;
  INT4 high;
  INT4 x,i;
  DOUBLE value;
  
  if(!(FileName.GetLength())) {
    printf("FileName not set\n");
    return;
  }
  
  if(End==-1)
    End=999999999;
  if(Start==-1)
    Start=0;
  
  fp=fopen(FileName,"r");
  if(fp){
    //    fseek(fp,Start*(sizeof(INT4)+sizeof(DOUBLE)),0);
    fseek(fp,(long)(Start*(sizeof(INT4)+sizeof(DOUBLE))),SEEK_SET);
    for(i=Start;i<End;i++){
#ifndef __SUNPRO_CC
      x=fread((void*)&high,1,sizeof(INT4),fp); // explicit cast
#else
      x=fread((char*)&high,1,sizeof(INT4),fp); // explicit cast
#endif
      if(x==0)
	break;
      table[Count].SetGlobalStart(high);
#ifndef __SUNPRO_CC
      x=fread((void*)&value,1,sizeof(DOUBLE),fp); // explicit cast
#else
      x=fread((char*)&value,1,sizeof(DOUBLE),fp); // explicit cast
#endif
      table[Count].SetStartValue(value);
#ifndef __SUNPRO_CC
      x=fread((void*)&value,1,sizeof(DOUBLE),fp); // explicit cast
#else
      x=fread((char*)&value,1,sizeof(DOUBLE),fp); // explicit cast
#endif
      table[Count].SetEndValue(value);
      Count++;
      if(Count==MaxEntries)
	Expand();
    }
    fclose(fp);
    Cleanup();
  }
}

void INTERVALLIST::LoadTable(INT4 Start, INT4 End, INT Offset)
{
  FILE *fp;
  INT4 high;
  INT4 x,i;
  DOUBLE fStart,fEnd;

  INT4 nCount;
  LONG MoveTo;

  if(!(FileName.GetLength())) {
    printf("FileName not set\n");
    return;
  }
  
  fp=fopen(FileName,"rb");
  if (fp) {
    // Bump past Count, then offset to the right starting point
    // Note - there are two tables in the file - one sorted by starting
    // value, one sorted by ending value.  There are Count entries in each
    // version.  
#ifndef __SUNPRO_CC
    x = fread((void*)&nCount,1,sizeof(INT4),fp); // explicit cast
#else
    x = fread((char*)&nCount,1,sizeof(INT4),fp); // explicit cast
#endif

    if(End==-1)
      End=nCount;
    if(Start==-1)
      Start=1;
  
    if (x != 0) {
      if (Offset == 0) {
	MoveTo = sizeof(INT4)
	  +(Start)*(sizeof(INT4)+2*sizeof(DOUBLE));
//	  +(Start-1)*(sizeof(INT4)+2*sizeof(DOUBLE));
      } else {
	MoveTo = sizeof(INT4)
	  +(nCount+Start)*(sizeof(INT4)+2*sizeof(DOUBLE));
//	  +(nCount+Start-1)*(sizeof(INT4)+2*sizeof(DOUBLE));
      }

      fseek(fp,MoveTo,SEEK_SET);
      for(i=Start;i<nCount;i++){
#ifndef __SUNPRO_CC
	x=fread((void*)&high,1,sizeof(INT4),fp); // explicit cast
#else
	x=fread((char*)&high,1,sizeof(INT4),fp); // explicit cast
#endif
	table[Count].SetGlobalStart(high);
	if(x==0)
	  break;
#ifndef __SUNPRO_CC
	x=fread((void*)&fStart,1,sizeof(DOUBLE),fp); // explicit cast
#else
	x=fread((char*)&fStart,1,sizeof(DOUBLE),fp); // explicit cast
#endif
	table[Count].SetStartValue(fStart);
#ifndef __SUNPRO_CC
	x=fread((void*)&fEnd,1,sizeof(DOUBLE),fp); // explicit cast
#else
	x=fread((char*)&fStart,1,sizeof(DOUBLE),fp); // explicit cast
#endif
	table[Count].SetEndValue(fEnd);
	Count++;
	if(Count==MaxEntries)
	  Expand();
      }
    }
    fclose(fp);
    Cleanup();
  }
}

void INTERVALLIST::WriteTable()
{
  PFILE fp;
  INT4 x;
  INT4 Val;
  DOUBLE fStart,fEnd;
  
  if(!(FileName.GetLength())){
    printf("FileName not set\n");
    return;
  }
  fp=fopen(FileName,"wb");
  if(fp){
    for(x=0; x<Count; x++) {
      Val = table[x].GetGlobalStart();
      fStart = table[x].GetStartValue();
      fEnd = table[x].GetEndValue();
#ifndef __SUNPRO_CC
      fwrite((void*)&Val,1,sizeof(INT4),fp); // explicit cast
      fwrite((void*)&fStart,1,sizeof(DOUBLE),fp); // explicit cast
      fwrite((void*)&fEnd,1,sizeof(DOUBLE),fp); // explicit cast
#else
      fwrite((char*)&Val,1,sizeof(INT4),fp); // explicit cast
      fwrite((char*)&fStart,1,sizeof(DOUBLE),fp); // explicit cast
      fwrite((char*)&fEnd,1,sizeof(DOUBLE),fp); // explicit cast
#endif
    }
  }
  fclose(fp);
}

void INTERVALLIST::WriteTable(INT Offset)
{
  PFILE fp;
  INT4 x;
  INT4 Val;
  DOUBLE fStart,fEnd;
  LONG MoveTo,nBytes;
  
  if(!(FileName.GetLength())) {
    printf("FileName not set\n");
    return;
  }
  fp=fopen(FileName,"a+b");

  if (fp) {
    // First, write out the count
    if (Offset == 0) {
#ifndef __SUNPRO_CC
      fwrite((void*)&Count,1,sizeof(INT4),fp); // explicit cast
#else
      fwrite((char*)&Count,1,sizeof(INT4),fp); // explicit cast
#endif
    }

    // Now, go to the specified offset and write the table
    nBytes = sizeof(INT4) + (2 * sizeof(DOUBLE));
    MoveTo = (LONG)((Offset*nBytes) + sizeof(INT4));

//    cout << "Offsetting " << MoveTo << " bytes into the file." << endl;

    if (fseek(fp, MoveTo, SEEK_SET) == 0) {
      for(x=0; x<Count; x++) {
	Val = table[x].GetGlobalStart();
	fStart = table[x].GetStartValue();
	fEnd = table[x].GetEndValue();
#ifndef __SUNPRO_CC
	fwrite((void*)&Val,1,sizeof(INT4),fp); // explicit cast
	fwrite((void*)&fStart,1,sizeof(DOUBLE),fp); // explicit cast
	fwrite((void*)&fEnd,1,sizeof(DOUBLE),fp); // explicit cast
#else
	fwrite((char*)&Val,1,sizeof(INT4),fp); // explicit cast
	fwrite((char*)&fStart,1,sizeof(DOUBLE),fp); // explicit cast
	fwrite((char*)&fEnd,1,sizeof(DOUBLE),fp); // explicit cast
#endif
//	cout << "Wrote " << nBytes << " bytes." << endl;
//	cout << "gp=" << Val;
//	cout << ", [" << fStart;
//	cout << ", " << fEnd << "]" << endl;

      }
    }
  }
  fclose(fp);
}

void INTERVALLIST::SetCoords(INT n) {
  Ncoords = n;
}

INT INTERVALLIST::GetCoords() {
  return Ncoords;
}

INT4 INTERVALLIST::GetGlobalStart(INT4 i)
{
  return(table[i].GetGlobalStart());
}

DOUBLE INTERVALLIST::GetStartValue(INT i)
{
  return(table[i].GetStartValue());
}

DOUBLE INTERVALLIST::GetEndValue(INT i)
{
  return(table[i].GetEndValue());
}

void INTERVALLIST::SortByStart()
{
  qsort((void *)table,Count, sizeof(INTERVALFLD),SortStartCmp);
}

void INTERVALLIST::SortByEnd()
{
  qsort((void *)table,Count, sizeof(INTERVALFLD),SortEndCmp);
}

static INT SortStartCmp(const void* x, const void* y) 
{
  DOUBLE a;
  a=((*((PINTERVALFLD)x)).GetStartValue()) -
    ((*((PINTERVALFLD)y)).GetStartValue()) ;
  if (a<0)
    return(-1);
  if (a>0)
    return(1);
  return(0);
}

static INT SortEndCmp(const void* x, const void* y) 
{
  DOUBLE a;
  a=((*((PINTERVALFLD)x)).GetEndValue()) -
    ((*((PINTERVALFLD)y)).GetEndValue()) ;
  if (a<0)
    return(-1);
  if (a>0)
    return(1);
  return(0);
}
#define GE 4
#define GT 5
#define LE 2
#define LT 1

#define ERROR -1
#define TOO_LOW 0
#define TOO_HIGH 1
#define MATCH 2

INT4 IntervalMatcher(DOUBLE Key, DOUBLE A, DOUBLE B, DOUBLE C, 
	INT4 Relation, INT4 Type)
{
	
  switch (Relation)
    {
    case GE:			// greater than or equals
      if ((B>=Key) && (Type==-1 || A<Key))
	return(MATCH);		// exact place - lower boundary
      else if (A>=Key)
	return(TOO_LOW);		// key too low
      else
	return(TOO_HIGH);		// key too high
    case GT:			// greater than
      if ((B>Key) && (Type==-1 || A<=Key))
	return(MATCH);		// exact place - lower boundary
      else if (A>Key)
	return(TOO_LOW);		// key too low
      else
	return(TOO_HIGH);		// key too high
    case LE:			// less than or equals
      if ((B<=Key) && (Type==0 || C>Key))
	return(MATCH);		// exact place - upper boundary
      else if (C<=Key)
	return(TOO_HIGH);
      else
	return(TOO_LOW);
    case LT:			// less than
      if ((B<Key) && (Type==0 || C>=Key))
	return(MATCH);		// exact place - upper boundary
      else if (C<Key)
	return(TOO_HIGH);
      else
	return(TOO_LOW);
    }
  printf("Hideous Matching Error\n");
  return(ERROR);
}

INT4 INTERVALLIST::DiskFind(STRING Fn, DOUBLE Key, INT4 Relation, 
			    GDT_BOOLEAN ByStart)
{
  INT4 Nrecs;
  DOUBLE A,C;
  PFILE Fp = fopen(Fn, "r");

  if (!Fp) {
    perror(Fn);
    exit(1);

  } else {
    INT ElementSize;

    // These count entries, not offsets
    INT Total;
    INT Low = 0;
    INT High, X, OX;

    INT State=0;	 // 0 = too low 1 = too high 2 = match 
    INT Type;
    INT4 GpS, Dummy;
    DOUBLE NumericValue;
    DOUBLE Hold;         // This is the interval bounday we don't use to search
    INT4 Offset;         // Offset into the file needed to read the element

#ifndef __SUNPRO_CC
    fread((void*)&Total,1,sizeof(INT4),Fp); // explicit cast
#else
    fread((char*)&Total,1,sizeof(INT4),Fp); // explicit cast
#endif

    ElementSize = sizeof(INT4) + 2*sizeof(DOUBLE);
    High = Total - 1;
    X = High / 2;

    if (ByStart) {
      // Search the index sorted by starting interval values
      do {
	OX = X;
	if((X > 0) && (X != High)) {
	  Offset = sizeof(INT4) + (X-1) * ElementSize;
	  fseek(Fp, (long)Offset, SEEK_SET);
	  Type=1;
	}
	else if (X == 0) {
	  Offset = sizeof(INT4) + X * ElementSize;
	  fseek(Fp, (long)Offset, SEEK_SET);
	  Type=-1;
	} else if (X == High) {
	  Offset = sizeof(INT4) + (X-1) * ElementSize;
	  fseek(Fp, (long)Offset, SEEK_SET);
	  Type=0;
	}
	if (Type > -1) {
#ifndef __SUNPRO_CC
	  fread((void *)&GpS, 1, sizeof(INT4), Fp);
	  fread((void *)&A, 1, sizeof(DOUBLE), Fp);
	  fread((void *)&Hold, 1, sizeof(DOUBLE), Fp);
#else
	  fread((char *)&GpS, 1, sizeof(INT4), Fp);
	  fread((char *)&A, 1, sizeof(DOUBLE), Fp);
	  fread((char *)&Hold, 1, sizeof(DOUBLE), Fp);
#endif	
	}
#ifndef __SUNPRO_CC
	fread((void *)&GpS, 1, sizeof(INT4), Fp);
	fread((void *)&NumericValue, 1, sizeof(DOUBLE), Fp);
	fread((void *)&Hold, 1, sizeof(DOUBLE), Fp);
#else
	fread((char *)&GpS, 1, sizeof(INT4), Fp);
	fread((char *)&NumericValue, 1, sizeof(DOUBLE), Fp);
	fread((char *)&Hold, 1, sizeof(DOUBLE), Fp);
#endif
	if(Type != 0) {
#ifndef __SUNPRO_CC
	  fread((void *)&Dummy, 1, sizeof(INT4), Fp);
	  fread((void *)&C, 1, sizeof(DOUBLE), Fp);
	  fread((void *)&Hold, 1, sizeof(DOUBLE), Fp);
#else
	  fread((char *)&Dummy, 1, sizeof(INT4), Fp);
	  fread((char *)&C, 1, sizeof(DOUBLE), Fp);
	  fread((char *)&Hold, 1, sizeof(DOUBLE), Fp);
#endif
	}
      
	//printf("Max=%d Type=%d X=%d High=%d GpS %d Key=%f A=%f B=%f C=%f\n",
	// 	     Total,Type,X,High,GpS,Key,A,NumericValue,C);
	State = IntervalMatcher(Key,A,NumericValue,C,Relation,Type);
	if (State == ERROR) {
	  exit(0);
	}
	if (State == MATCH) {
	  fclose(Fp);
	  return X;
	}
	if (State == TOO_LOW) {
	  High = X;
	} else {
	  Low = X + 1;
	}
	X = (Low + High) / 2;
	if (X < 0) {
	  X = 0;
	} else {
	  if (X >= Total) {
	    X = Total - 1;
	  }
	}
      } while (X != OX);

    } else {
      // Search the index sorted by ending interval values
      do {
	OX = X;
	if((X > 0) && (X != High)) {
	  Offset = sizeof(INT4) + (Total+X-1) * ElementSize;
	  fseek(Fp, (long)Offset, SEEK_SET);
	  Type=1;
	}
	else if (X == 0) {
	  Offset = sizeof(INT4) + (Total+X) * ElementSize;
	  fseek(Fp, (long)Offset, SEEK_SET);
	  Type=-1;
	} else if (X == High) {
	  Offset = sizeof(INT4) + (Total+X-1) * ElementSize;
	  fseek(Fp, (long)Offset, SEEK_SET);
	  Type=0;
	}
	if (Type > -1) {
#ifndef __SUNPRO_CC
	  fread((void *)&GpS, 1, sizeof(INT4), Fp);
	  fread((void *)&Hold, 1, sizeof(DOUBLE), Fp);
	  fread((void *)&A, 1, sizeof(DOUBLE), Fp);
#else
	  fread((char *)&GpS, 1, sizeof(INT4), Fp);
	  fread((char *)&Hold, 1, sizeof(DOUBLE), Fp);
	  fread((char *)&A, 1, sizeof(DOUBLE), Fp);
#endif	
	}
#ifndef __SUNPRO_CC
	fread((void *)&GpS, 1, sizeof(INT4), Fp);
	fread((void *)&Hold, 1, sizeof(DOUBLE), Fp);
	fread((void *)&NumericValue, 1, sizeof(DOUBLE), Fp);
#else
	fread((char *)&GpS, 1, sizeof(INT4), Fp);
	fread((char *)&Hold, 1, sizeof(DOUBLE), Fp);
	fread((char *)&NumericValue, 1, sizeof(DOUBLE), Fp);
#endif
	
	if(Type != 0) {
#ifndef __SUNPRO_CC
	  fread((void *)&Dummy, 1, sizeof(INT4), Fp);
	  fread((void *)&Hold, 1, sizeof(DOUBLE), Fp);
	  fread((void *)&C, 1, sizeof(DOUBLE), Fp);
#else
	  fread((char *)&Dummy, 1, sizeof(INT4), Fp);
	  fread((char *)&Hold, 1, sizeof(DOUBLE), Fp);
	  fread((char *)&C, 1, sizeof(DOUBLE), Fp);
#endif
	}
      
	//printf("Max=%d Type=%d X=%d High=%d GpS %d Key=%f A=%f B=%f C=%f\n",
	// 	     Total,Type,X,High,GpS,Key,A,NumericValue,C);
	State = IntervalMatcher(Key,A,NumericValue,C,Relation,Type);
	if (State == MATCH) {
	  fclose(Fp);
	  return X;
	}
	if (State == TOO_LOW) {
	  High = X;
	} else {
	  Low = X + 1;
	}
	X = (Low + High) / 2;
	if (X < 0) {
	  X = 0;
	} else {
	  if (X >= Total) {
	    X = Total - 1;
	  }
	}
      } while (X != OX);
    }
  }
  fclose(Fp);
  return -1;
}

INT4 INTERVALLIST::DiskDateFind(STRING Fn, DOUBLE Key, INT4 Relation, 
			    GDT_BOOLEAN ByStart)
{
  INT4 Nrecs;
  DOUBLE A,C;
  PFILE Fp = fopen(Fn, "r");
  INT ElementSize;

  // These count entries, not offsets
  INT Total;
  INT Low = 0;
  INT High, X, OX;
  
  INT State=0;	 // 0 = too low 1 = too high 2 = match 
  INT Type;
  INT4 GpS, Dummy;
  DOUBLE NumericValue;
  DOUBLE Hold;         // This is the interval bounday we don't use to search
  INT4 Offset;         // Offset into the file needed to read the element

  time_t today;
  struct tm    *t;
  CHR TodaysDate[1024];
  DOUBLE NumericDate;

  today = time((time_t *)NULL);
  t = localtime(&today);

  strftime(TodaysDate,9,"%Y%m%d",t);
  NumericDate = atof(TodaysDate);

  if (!Fp) {
    perror(Fn);
    exit(1);

  } else {

    fread(&Total,1,sizeof(INT4),Fp);
    ElementSize = sizeof(INT4) + 2*sizeof(DOUBLE);
    High = Total - 1;
    X = High / 2;

    if (ByStart) {
      // Search the index sorted by starting interval values
      do {
	OX = X;
	if ((X > 0) && (X != High)) {
	  Offset = sizeof(INT4) + (X-1) * ElementSize;
	  fseek(Fp, (long)Offset, SEEK_SET);
	  Type=1;

	} else if (X == 0) {
	  Offset = sizeof(INT4) + X * ElementSize;
	  fseek(Fp, (long)Offset, SEEK_SET);
	  Type=-1;

	} else if (X == High) {
	  Offset = sizeof(INT4) + (X-1) * ElementSize;
	  fseek(Fp, (long)Offset, SEEK_SET);
	  Type=0;
	}

	if (Type > -1) {
#ifndef __SUNPRO_CC
	  fread((void *)&GpS, 1, sizeof(INT4), Fp);
	  fread((void *)&A, 1, sizeof(DOUBLE), Fp);
#else
	  fread((char *)&GpS, 1, sizeof(INT4), Fp);
	  fread((char *)&A, 1, sizeof(DOUBLE), Fp);
#endif

	  if (A == 99999999)
	    A = NumericDate;

#ifndef __SUNPRO_CC
	  fread((void *)&Hold, 1, sizeof(DOUBLE), Fp);
#else
	  fread((char *)&Hold, 1, sizeof(DOUBLE), Fp);
#endif
	  if (Hold == 99999999)
	    Hold = NumericDate;
	}

#ifndef __SUNPRO_CC
	fread((void *)&GpS, 1, sizeof(INT4), Fp);
	fread((void *)&NumericValue, 1, sizeof(DOUBLE), Fp);
#else
	fread((char *)&GpS, 1, sizeof(INT4), Fp);
	fread((char *)&NumericValue, 1, sizeof(DOUBLE), Fp);
#endif
	if (NumericValue == 99999999)
	  NumericValue = NumericDate;

#ifndef __SUNPRO_CC
	fread((void *)&Hold, 1, sizeof(DOUBLE), Fp);
#else
	fread((char *)&Hold, 1, sizeof(DOUBLE), Fp);
#endif

	if (Hold == 99999999)
	  Hold = NumericDate;

	if (Type != 0) {
#ifndef __SUNPRO_CC
	  fread((void *)&Dummy, 1, sizeof(INT4), Fp);
	  fread((void *)&C, 1, sizeof(DOUBLE), Fp);
#else
	  fread((char *)&Dummy, 1, sizeof(INT4), Fp);
	  fread((char *)&C, 1, sizeof(DOUBLE), Fp);
#endif
	  if (C == 99999999)
	    C = NumericDate;

#ifndef __SUNPRO_CC
	  fread((void *)&Hold, 1, sizeof(DOUBLE), Fp);
#else
	  fread((char *)&Hold, 1, sizeof(DOUBLE), Fp);
#endif

	  if (Hold == 99999999)
	    Hold = NumericDate;
	}
      
	//printf("Max=%d Type=%d X=%d High=%d GpS %d Key=%f A=%f B=%f C=%f\n",
	// 	     Total,Type,X,High,GpS,Key,A,NumericValue,C);
	State = IntervalMatcher(Key,A,NumericValue,C,Relation,Type);
	if (State == MATCH) {
	  fclose(Fp);
	  return X;
	}

	if (State == TOO_LOW) {
	  High = X;
	} else {
	  Low = X + 1;
	}

	X = (Low + High) / 2;
	if (X < 0) {
	  X = 0;

	} else {
	  if (X >= Total) {
	    X = Total - 1;
	  }
	}
      } while (X != OX);

    } else {
      // Search the index sorted by ending interval values
      do {
	OX = X;
	if ((X > 0) && (X != High)) {
	  Offset = sizeof(INT4) + (Total+X-1) * ElementSize;
	  fseek(Fp, (long)Offset, SEEK_SET);
	  Type=1;

	} else if (X == 0) {
	  Offset = sizeof(INT4) + (Total+X) * ElementSize;
	  fseek(Fp, (long)Offset, SEEK_SET);
	  Type=-1;

	} else if (X == High) {
	  Offset = sizeof(INT4) + (Total+X-1) * ElementSize;
	  fseek(Fp, (long)Offset, SEEK_SET);
	  Type=0;
	}

	if (Type > -1) {
#ifndef __SUNPRO_CC
	  fread((void *)&GpS, 1, sizeof(INT4), Fp);
	  fread((void *)&Hold, 1, sizeof(DOUBLE), Fp);
#else
	  fread((char *)&GpS, 1, sizeof(INT4), Fp);
	  fread((char *)&Hold, 1, sizeof(DOUBLE), Fp);
#endif
	  if (Hold == 99999999)
	    Hold = NumericDate;

#ifndef __SUNPRO_CC
	  fread((void *)&A, 1, sizeof(DOUBLE), Fp);
#else
	  fread((char *)&A, 1, sizeof(DOUBLE), Fp);
#endif
	  if (A == 99999999)
	    A = NumericDate;
	
	}

#ifndef __SUNPRO_CC
	fread((void *)&GpS, 1, sizeof(INT4), Fp);
	fread((void *)&Hold, 1, sizeof(DOUBLE), Fp);
#else
	fread((char *)&GpS, 1, sizeof(INT4), Fp);
	fread((char *)&Hold, 1, sizeof(DOUBLE), Fp);
#endif
	if (Hold == 99999999)
	  Hold = NumericDate;

#ifndef __SUNPRO_CC
	fread((void *)&NumericValue, 1, sizeof(DOUBLE), Fp);
#else
	fread((char *)&NumericValue, 1, sizeof(DOUBLE), Fp);
#endif
	if (NumericValue == 99999999)
	  NumericValue = NumericDate;
	
	if (Type != 0) {
#ifndef __SUNPRO_CC
	  fread((void *)&Dummy, 1, sizeof(INT4), Fp);
	  fread((void *)&Hold, 1, sizeof(DOUBLE), Fp);
#else
	  fread((char *)&Dummy, 1, sizeof(INT4), Fp);
	  fread((char *)&Hold, 1, sizeof(DOUBLE), Fp);
#endif
	  if (Hold == 99999999)
	    Hold = NumericDate;

#ifndef __SUNPRO_CC
	  fread((void *)&C, 1, sizeof(DOUBLE), Fp);
#else
	  fread((char *)&C, 1, sizeof(DOUBLE), Fp);
#endif

	  if (C == 99999999)
	    C = NumericDate;
	}
      
	//printf("Max=%d Type=%d X=%d High=%d GpS %d Key=%f A=%f B=%f C=%f\n",
	// 	     Total,Type,X,High,GpS,Key,A,NumericValue,C);
	State = IntervalMatcher(Key,A,NumericValue,C,Relation,Type);
	if (State == MATCH) {
	  fclose(Fp);
	  return X;
	}
	if (State == TOO_LOW) {
	  High = X;
	} else {
	  Low = X + 1;
	}
	X = (Low + High) / 2;
	if (X < 0) {
	  X = 0;
	} else {
	  if (X >= Total) {
	    X = Total - 1;
	  }
	}
      } while (X != OX);
    }
  }
  fclose(Fp);
  return -1;
}

void INTERVALLIST::Dump(INT4 start, INT4 end)
{
  INT4 x;
  INT4 a;
  DOUBLE b,c;

  for(x=0; x<Count; x++) {
    a = table[x].GetGlobalStart();
    b = table[x].GetStartValue();
    c = table[x].GetEndValue();
    printf("ptr: %d [%f, %f]\n",a,b,c);
  }
}

void INTERVALLIST::SetFileName(STRING s)
{
  FileName = s;
}

void INTERVALLIST::SetFileName(PCHR s)
{
  FileName = s;
}

INT4 INTERVALLIST::GetCount()
{
  return(Count);
}

void INTERVALLIST::Expand()
{
  Resize(Count + (50*Ncoords));
}

void INTERVALLIST::Cleanup()
{
  Resize(Count);
}

void INTERVALLIST::Resize(INT4 Entries)
{
  PINTERVALFLD temp=new INTERVALFLD[Entries];
  INT4 CopyCount;
  INT4 x;

  if(Entries>Count)
    CopyCount=Count;
  else {
    CopyCount=Entries;
    Count=Entries;
  }
  for(x=0; x<CopyCount; x++) {
    temp[x]=table[x];
  }
  if(table)
    delete [] table;
  table=temp;
  MaxEntries=Entries;
}

INTERVALLIST::~INTERVALLIST()
{
  if(table)
    delete [] table;
}

#ifdef NEVER
main()
{
  NUMERICLIST list;
  STRING n;
  INT4 Start,End;
  DOUBLE val;
  n="test.62";
  printf("DOUBLE is %d bytes\n",sizeof(DOUBLE));
  list.TempLoad();
  End=list.DiskFind(n, (double)88.0,5); // 5 GT
  printf("\n===\n");
  Start=list.DiskFind(n,(double)88.0,1); // 1 LT
  
  if(End-Start<=1)
    printf("No Match!\n");
  list.SetFileName(n.NewCString());
  list.LoadTable(Start+1,End);
    
}
#endif

/*
void NUMERICLIST::ResetHitPosition()
{
  if (Relation != 1)
    Pointer = StartIndex;
  else
    Pointer = 0;
}

void NUMERICLIST::SetRelation(INT r)
{
  Relation = r;
}

INT NUMERICLIST::GetRelation()
{
  return(Relation);
}

INT4 NUMERICLIST::GetNextHitPosition()
{
  INT4 Value;

  if(StartIndex == -1)
    return(-1);
  if(Relation != 1) {
    if(Pointer>EndIndex)
      return(-1);
    Value=table[Pointer].GetGlobalStart();
    ++Pointer;
    return(Value);
  } else {
    if(Pointer >= StartIndex)
      while(Pointer <= EndIndex)
	++Pointer;
    if(Pointer >= Count)
      return(-1);
    Value=table[Pointer].GetGlobalStart();
    ++Pointer;
    return(Value);
  }
}

INT NUMERICLIST::GetAttribute()
{
  return(Attribute);
}

void NUMERICLIST::SetAttribute(INT x)
{
  Attribute = x;
}

void NUMERICLIST::TempLoad()
{
  INT4 x,y=0;
  DOUBLE z;
  FILE *fp=fopen("test.62","w");
  
  for(x=0; x<100; x+=4){
    z=x;
    fwrite(&x,1,sizeof(INT4),fp);
    x+=4;
    fwrite(&z,1,sizeof(DOUBLE),fp);
    printf("Wrote %f\n",z);
    
  }
  fclose(fp);
  fp=fopen("test.62","r");
  while((y=fread(&x,1,sizeof(INT4),fp))!=0) {
    printf("x: %d\n",x);
    fread(&z,1,sizeof(DOUBLE),fp);
    printf("z(double): %f\n",z);
  }
  fclose(fp);
}

*/
