########################## -*- Mode: Perl -*- ##########################
##
## File             : Bibtex.pm
##
## Description      : BibTeX converter for WRb format
##
#
# Copyright (C) 1995, 1996 Ulrich Pfeifer, Norbert Goevert
#
# This file is part of SFgate.
#
# SFgate is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# SFgate is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with SFgate; see the file COPYING.  If not, write to
# the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
#
##
## Author           : Norbert Goevert
## Created On       : Wed Mar 27 11:10:08 1996
##
## Last Modified By : Norbert Goevert
## Last Modified On : Tue Jan 28 16:25:05 1997
##
## $State: Exp $
##
## $Id: Bibtex.pm,v 5.1.1.2 1997/02/17 12:55:50 goevert Exp goevert $
##
## $Log: Bibtex.pm,v $
## Revision 5.1.1.2  1997/02/17 12:55:50  goevert
## patch10: global variable problem fixed
##
## Revision 5.1.1.1  1996/12/23 12:49:18  goevert
## patch6: fixed double author field when fetching more than one document
##
## Revision 5.1  1996/11/05 16:54:54  goevert
## *** empty log message ***
##
## Revision 5.0.1.2  1996/05/23 17:07:04  goevert
## patch13: removed no strict 'vars';
##
## Revision 5.0.1.1  1996/05/13 11:27:03  goevert
## patch1:
##
########################################################################


package SFgate::Converter::Bibtex;

use strict;
no strict 'vars';


## #####################################################################
## convert
## #####################################################################
## interface for SFgate to convert one document
##
## (string) $text:     document to convert
## (string) $headline: headline of document to convert
##
## By default every document is printed with its corresponding wais
## headline. If you want to derive another headline just change
## the $headline variable according to your needs.
##
## HTML special characters (&, <, >) should be encoded in text not
## being HTML code. To do this you can use the function
## &::encode_entities, e.g. $ntext = &::encode_entities($ntext)
## encodes the special chars in $ntext.
##
## returns:
## - string: the converted document ($ntext)
## - string: headline of converted document ($headline)
##
sub convert
{
    my($text, $headline) = @_;
    
    $text = &scan($text);
    $text = "<PRE>\n" . &::encode_entities($text) . "</PRE>\n";
    
    return($text, $headline);
}


## #####################################################################
## helper subroutines
## #####################################################################

sub scan
{
    local($_) = @_;
    local($author, $editor, $result, $access_key);
    
    # some globals
    $separator  = "  separator =    ";
    $separator .= "{---------------------------------------------------------},\n";
    $len = 57;             # max line length

    # BibTeX-fields
    if (/^         Autor\(1\): (.*)/m) {
	$author       = "  author =       {$1";
	($access_key, $garbage) = $author =~ /= \{([^, ]*)/m;
    }
    $count = 2;
    while (/^         Autor\($count\): (.*)/m) {
	$author .= " and $1";
	$count++;
    }

    if (/^   Herausgeber\(1\): (.*)/m) {
	$editor       = "  editor =       {$1";
	($access_key, $garbage) = $editor =~ /= \{([^, ]*)/m;
    }
    $count = 2;
    while (/^   Herausgeber\($count\): (.*)/m) {
	$editor .= " and $1";
	$count++;
    }

    $count = 1;
    while (/^     Verfasser\($count\): (.*)/m) {
	$verf = $1;
	if ($count == 1) {
	    ($access_key, $garbage) = $verf =~ /^([^, ]*)/m;
	}
	if ($verf =~ /\[Hrsg\.\]/m) {
	    $verf =~ s/ ?\[Hrsg\.\]//m;
	    if ($editor eq "") {
		$editor = "  editor =       {$verf";
	    }
	    else {
		$editor .= " and $verf";
	    }
	}
	else {
	    if ($author eq "") {
		$author = "  author =       {$verf";
	    }
	    else {
		$author .= " and $verf";
	    }
	}
	$count++;
    }
    
    # let's have a proper length for author and editor lines
    if (length($author) > 77) {
	($first, $tmp) = $author =~ /(.{0,77}) and (.*)/m;
	if (length($first) < 73) {
	    $author = "$first and\n                   $tmp";
	}
	else {
	    $author = "$first\n                   and $tmp";
	}
    }
    if ($author ne "") {
	$author .= "},\n";
    }
    if (length($editor) > 77) {
	($first, $tmp) = $editor =~ /(.{0,77}) and (.*)/m;
	if (length($first) < 73) {
	    $editor = "$first and\n                   $tmp";
	}
	else {
	    $editor = "$first\n                   and $tmp";
	}
    }				
    if ($editor ne "") {
	$editor .= "},\n";
    }

    if (/^           Verlag: (.*)/m) {
	$publisher    = "  publisher =    {$1},\n";
    }
    
    if (/^       Verlagsort: (.*)/m) {
	$address      = "  address =      {$1},\n";
    }

    if (/^            Titel: (.*(\n {19}.*)*)/m) {
	@lines = split(/\n/, $1);
	&breaklines;
	$title        = "  title =        {".shift(@lines);
	while (@lines) {
	    $title .= "\n                  ".shift(@lines);
	}
	$title .= "},\n";
    }
    
    if (/^            Serie: (.*(\n {19}.*)*)/m) {
	@lines = split(/\n/, $1);
	&breaklines;
	$series       = "  series =       {".shift(@lines);
	while (@lines) {
	    $series .=  "\n                  ".shift(@lines);
	}
	$series .= "},\n";
    }
    
    if (/^ Erscheinungsjahr: (.*)/m) {
	$year         = "  year =         {$1},\n";
	$access_key .= ":$1";
    }
    
    if (/^          Ausgabe: (.*)/m) {
	$edition      = "  edition =      {$1},\n";
    }				

    # user-defined fields
    if (/^  Erfassungsdatum: (.*)/m) {
	$add_date     = "  add-date =     {$1},\n";
    }
    
    if (/^   nderungsdatum: (.*)/m) {
	$rev_date     = "  rev-date =     {$1},\n";
    }
    
    if (/^         Signatur: (.*)/m) {
	$signature    = "  signature =    {$1},\n";
    }
    
    if (/^    Zugangsnummer: (.*)/m) {
	$accessno     = "  accessno =     {$1},\n";
    }
    
    if (/^          Kongre: (.*(\n {19}.*)*)/m) {
	@lines = split(/\n/, $1);
	&breaklines;
	$congress     = "  congress =     {".shift(@lines);
	while (@lines) {
	    $congress.= "\n                  ".shift(@lines);
	}
	$congress .= "},\n";
    }
    
    if (/^             ISBN: (.*)/m) {
	$isbn         = "  isbn =         {$1},\n";
    }

    if (/^         Standort: (.*)/m) {
	$location     = "  location =     {$1},\n";
    }

    # what kind of literature?
    if (/^              Art: Monographie/m) {
	&print_book;
    }
    elsif (/^              Art: Proceedings/m) {
	&print_proceedings;
    }
    elsif (/^              Art: Bericht/m) {
	$publisher   =~ /  publisher =    (.*)/m;
	$institution  = "  institution =  $1\n";
	$publisher    = "";
	&print_techreport;
    }
    elsif (/^              Art: (Dissertation|Habilitation)/m) {
	$publisher   =~ /  publisher =    (.*)/m;
	$school       = "  school =       $1\n";
	$publisher    = "";
	&print_phdthesis;
    }
    elsif (/^              Art: Manual/m) {
	$publisher   =~ /  publisher =    (.*)/m;
	$organization = "  organization = $1\n";
	$publisher    = "";
	&print_manual;
    }
    elsif (/^              Art: Verzeichnis/m) {
	&print_book;
    }
    else {
	&print_book;
    }
    
    # print user-defined fields
    &print_every;

    # replace ISO-Umlaute with TeX-Umlaute
    # but leave out citation key
    $result =~ s//\\\"a/g;
    $result =~ s//\\\"A/g;
    $result =~ s//\\\"o/g;
    $result =~ s//\\\"O/g;
    $result =~ s//\\\"u/g;
    $result =~ s//\\\"U/g;
    $result =~ s//\\\"s/g;
    $result =~ s/^(.*\n)//;
    $access_key = $1;
    $access_key =~ s/\\"s/ss/g;    # "
    $access_key =~ s/\\"(.)/$1e/g; # "
    $result = $access_key . $result;

    return $result;
}				


sub print_every {
    
    $result .= $add_date;
    $result .= $rev_date;
    $result .= $signature;
    $result .= $accessno;
    $result .= $congress;
    $result .= $isbn;
    $result .= $location;

    $result .= "}\n";
}
    

sub print_article {

    $result .= "\@Article{$access_key,\n";

    # forced fields
    $result .= $author;
    $result .= $title;
    $result .= $journal;
    $result .= $year;

    # optional fields
    $result .= $volume;
    $result .= $number;
    $result .= $pages;
    $result .= $month;
    $result .= $note;

    # user defined fields
    $result .= $separator;
    $result .= $address;
    $result .= $annote;
    $result .= $booktitle;
    $result .= $chapter;
    $result .= $edition;
    $result .= $howpublished;
    $result .= $institution;
    $result .= $key;
    $result .= $publisher;
    $result .= $organization;
    $result .= $school;
    $result .= $series;
    $result .= $type;
}


sub print_book {

    $result .= "\@Book{$access_key,\n";

    # forced fields
    $result .= $author;
    $result .= $editor;
    $result .= $title;
    $result .= $publisher;
    $result .= $year;

    # optional fields
    $result .= $volume;
    $result .= $number;
    $result .= $series;
    $result .= $address;
    $result .= $edition;
    $result .= $month;
    $result .= $note;

    # user defined fields
    $result .= $separator;
    $result .= $annote;
    $result .= $booktitle;
    $result .= $chapter;
    $result .= $howpublished;
    $result .= $institution;
    $result .= $journal;
    $result .= $key;
    $result .= $organization;
    $result .= $pages;
    $result .= $school;
    $result .= $type;
}


sub print_booklet {

    $result .= "\@Booklet{$access_key,\n";

    # forced fields
    $result .= $title;
    
    # optional fields
    $result .= $author;
    $result .= $howpublished;
    $result .= $address;
    $result .= $year;
    $result .= $month;
    $result .= $note;
    
    # user defined fields
    $result .= $separator;
    $result .= $annote;
    $result .= $booktitle;
    $result .= $chapter;
    $result .= $edition;
    $result .= $editor;
    $result .= $institution;
    $result .= $journal;
    $result .= $key;
    $result .= $organization;
    $result .= $pages;
    $result .= $publisher;
    $result .= $school;
    $result .= $series;
    $result .= $type;
    $result .= $volume;
}


sub print_inbook {
    
    $result .= "\@Inbook{$access_key,\n";
    
    # forced fields
    $result .= $author;
    $result .= $editor;
    $result .= $title;
    $result .= $chapter;
    $result .= $pages;
    $result .= $publisher;
    $result .= $type;
    $result .= $year;
    
    # optional fields
    $result .= $volume;
    $result .= $number;
    $result .= $series;
    $result .= $address;
    $result .= $edition;
    $result .= $month;
    $result .= $note;
    
    # user defined fields
    $result .= $separator;
    $result .= $annote;
    $result .= $booktitle;
    $result .= $howpublished;
    $result .= $institution;
    $result .= $journal;
    $result .= $key;
    $result .= $organization;
    $result .= $school;
}


sub print_incollection {

    $result .= "\@Incollection{$access_key,\n";
    
    # forced fields
    $result .= $author;
    $result .= $title;
    $result .= $booktitle;
    $result .= $publisher;
    $result .= $year;
    
    # optional fields
    $result .= $editor;
    $result .= $volume;
    $result .= $number;
    $result .= $type;
    $result .= $series;
    $result .= $edition;
    $result .= $chapter;
    $result .= $pages;
    $result .= $address;
    $result .= $month;
    $result .= $note;
    
    # user defined fields
    $result .= $separator;
    $result .= $annote;
    $result .= $howpublished;
    $result .= $institution;
    $result .= $journal;
    $result .= $key;
    $result .= $organization;
    $result .= $school;
}			       
				

sub print_inproceedings {

    $result .= "\@Inproceedings{$access_key,\n";

    # forced fields
    $result .= $author;
    $result .= $title;
    $result .= $booktitle;
    $result .= $year;
    
    # optional fields
    $result .= $editor;
    $result .= $volume;
    $result .= $number;
    $result .= $organization;
    $result .= $series;
    $result .= $pages;
    $result .= $publisher;
    $result .= $address;
    $result .= $month;
    $result .= $note;
    
    # user defined fields
    $result .= $separator;
    $result .= $annote;
    $result .= $chapter;
    $result .= $edition;
    $result .= $howpublished;
    $result .= $institution;
    $result .= $journal;
    $result .= $key;
    $result .= $school;
    $result .= $type;
}


sub print_manual {

    $result .= "\@Manual{$access_key,\n";

    # forced fields
    $result .= $title;
    
    # optional fields
    $result .= $author;
    $result .= $organization;
    $result .= $address;
    $result .= $edition;
    $result .= $month;
    $result .= $year;
    $result .= $note;
    
    # user defined fields
    $result .= $separator;
    $result .= $annote;
    $result .= $booktitle;
    $result .= $chapter;
    $result .= $editor;
    $result .= $howpublished;
    $result .= $institution;
    $result .= $journal;
    $result .= $key;
    $result .= $number;
    $result .= $pages;
    $result .= $publisher;
    $result .= $school;
    $result .= $series;
    $result .= $type;
    $result .= $volume;
}				


sub print_masterthesis {

    $result .= "\@Masterthesis{$access_key,\n";

    # forced fields
    $result .= $author;
    $result .= $title;
    $result .= $school;
    $result .= $year;
    
    # optional fields
    $result .= $address;
    $result .= $month;
    $result .= $note;
    $result .= $type;

    # user defined fields
    $result .= $separator;
    $result .= $annote;
    $result .= $booktitle;
    $result .= $chapter;
    $result .= $edition;
    $result .= $editor;
    $result .= $howpublished;
    $result .= $institution;
    $result .= $journal;
    $result .= $key;
    $result .= $number;
    $result .= $organization;
    $result .= $pages;
    $result .= $publisher;
    $result .= $series;
    $result .= $volume;
}


sub print_misc {

    $result .= "\@Misc{$access_key,\n";

    # forced fields

    # optional fields
    $result .= $author;
    $result .= $title;
    $result .= $howpublished;
    $result .= $month;
    $result .= $year;
    $result .= $note;
    
    # user defined fields
    $result .= $separator;
    $result .= $address;
    $result .= $annote;
    $result .= $booktitle;
    $result .= $chapter;
    $result .= $editor;
    $result .= $edition;
    $result .= $institution;
    $result .= $journal;
    $result .= $key;
    $result .= $number;
    $result .= $organization;
    $result .= $pages;
    $result .= $publisher;
    $result .= $school;
    $result .= $series;
    $result .= $type;
    $result .= $volume;
}				
			       

sub print_phdthesis {

    $result .= "\@Phdthesis{$access_key,\n";
    
    # forced fields
    $result .= $author;
    $result .= $title;
    $result .= $school;
    $result .= $year;
    
    # optional fields
    $result .= $address;
    $result .= $month;
    $result .= $note;
    
    # user defined fields
    $result .= $separator;
    $result .= $annote;
    $result .= $booktitle;
    $result .= $chapter;
    $result .= $editor;
    $result .= $edition;
    $result .= $howpublished;
    $result .= $institution;
    $result .= $journal;
    $result .= $key;
    $result .= $number;
    $result .= $organization;
    $result .= $pages;
    $result .= $publisher;
    $result .= $series;
    $result .= $type;
    $result .= $volume;
}				


sub print_proceedings {

  $result .= "\@Proceedings{$access_key,\n";

  # forced fields
  $result .= $title;
  $result .= $year;

  # optional fields
  $result .= $editor;
  $result .= $publisher;
  $result .= $volume;
  $result .= $number;
  $result .= $organization;
  $result .= $address;
  $result .= $month;
  $result .= $note;

  # user defined fields
  $result .= $separator;
  $result .= $author;
  $result .= $annote;
  $result .= $booktitle;
  $result .= $chapter;
  $result .= $edition;
  $result .= $howpublished;
  $result .= $institution;
  $result .= $journal;
  $result .= $key;
  $result .= $pages;
  $result .= $school;
  $result .= $series;
  $result .= $type;
}


sub print_techreport {

  $result .= "\@Techreport{$access_key,\n";

  # forced fields
  $result .= $author;
  $result .= $title;
  $result .= $institution;
  $result .= $year;

  # optional fields
  $result .= $type;
  $result .= $number;
  $result .= $address;
  $result .= $month;
  $result .= $note;

  # user defined fields
  $result .= $separator;
  $result .= $annote;
  $result .= $booktitle;
  $result .= $chapter;
  $result .= $editor;
  $result .= $edition;
  $result .= $howpublished;
  $result .= $journal;
  $result .= $key;
  $result .= $organization;
  $result .= $pages;
  $result .= $publisher;
  $result .= $school;
  $result .= $series;
  $result .= $volume;
}


sub print_unpublished {

    $result .= "\@Unpublished{$access_key,\n";

    # forced fields
    $result .= $author;
    $result .= $title;
    $result .= $note;
    
    # optional fields
    $result .= $month;
    $result .= $year;

    # user defined fields
    $result .= $separator;
    $result .= $address;
    $result .= $annote;
    $result .= $booktitle;
    $result .= $chapter;
    $result .= $editor;
    $result .= $edition;
    $result .= $howpublished;
    $result .= $institution;
    $result .= $journal;
    $result .= $key;
    $result .= $number;
    $result .= $organization;
    $result .= $pages;
    $result .= $publisher;
    $result .= $school;
    $result .= $series;
    $result .= $type;
    $result .= $volume;
}


sub breaklines {

    $line = shift(@lines);
    
    while (@lines) {
	$tmp = shift(@lines);
	$tmp =~ s/ {19}//;
	if ($tmp =~ /^ /) {
	    $line .= $tmp;
	}
	elsif ($line =~ /\.$/) {
	    $line .= " $tmp";
	}
	else {
	    $line .= ". $tmp";
	}
    }

    $line =~ s/\.*$/./;
    
    if (length($line) > $len) {
	($first, $line) = $line =~ /(.{1,$len}) (.*)/o;
	push(@lines, $first);
	while (length($line) > $len) {
	    ($first, $line) = $line =~ /(.{1,$len}) (.*)/o;
	    push(@lines, " $first");
	};
	push(@lines, " $line");
    }
    else {
	push(@lines, $line);
    }
}				


1;
