/*
 * Copyright (c) 1989, 1990, 1991 by the University of Washington
 *
 * For copying and distribution information, please see the file
 * <uw-copyright.h>.
 *
 * Credits:  Originally written by Clifford Neuman (University of Washington)
 *           Syslog support added by Jonathan Kamens (MIT Project Athena)
 */

#include <uw-copyright.h>
#include <sys/time.h>
#include <stdio.h>
#ifdef __STDC__
#include <stdarg.h>
#endif
#ifdef AIX
#include <time.h>
#endif

#include <ardp.h>
#include <pfs.h>
#include <pserver.h>
#include <plog.h>
#include <pmachine.h>
    
/* this array contains info on the type of entries to be printed */
static int 		logtype_array[NLOGTYPE+1] = INITIAL_LOG_VECTOR;

/* info on the filename and open file */
static char 		*log_name = PSRV_LOGFILE;
FILE 			*logfile;
static int 		is_open = 0;
#if 0
/* not currently used */
static int		syslog_open = 0;
#endif

static char 		logtxt[MXPLOGENTRY];

static char *pr_inet_ntoa();
static FILE *plog_additional_outfile = NULL;

/*VARARGS4*/

/*
 * plog - Add entry to logfile
 *
 * 	  PLOG is used to add entries to the logfile.  Note that
 * 	  it is probably not portable since is makes assumptions
 * 	  about what the compiler will do when it is called with
 * 	  less than the correct number of arguments which is the
 * 	  way it is usually called.
 *
 * 	  PLOG returns a pointer to the logged entry.  If an error
 * 	  occurs, vlog returns immediately, but does not indicate 
 * 	  that the log operating failed.
 *
 *    ARGS: type    - Type of entry (to decide if we should log it)
 *	    req     - Pointer to request info including host address,
 *                    and useride. (NULL if should not be printed)
 *          format  - Format as for qsprintf
 *          remaining arguments -- as for qsprintf
 * RETURNS: Pointer to a string containing the log entry
 *
 *    BUGS: The non-ANSI implementation is not portable.  It should really use
 *          another mechanism to support a variable number of arguments.  
 *          Unfortunately, these mechanisms are not easily available.
 *
 *          Currently, the log file is opened and closed on each
 *	    call.
 */
#ifdef __STDC__
#include <stdarg.h>

char *
vplog(int type, RREQ req, char *format, va_list ap)
{
    time_t now, systime, svctime, wttime;
    struct tm *tm;
    int	 log_username = 0;
    int  notfirstfield = 0;
    char *month_sname();
    char usertxt[MXPLOGENTRY];
    char fieldtxt[MXPLOGENTRY];

    (void) vqsprintf(logtxt, sizeof logtxt, format, ap);

    /* If we don't log this type of message, don't write to log */
    if (!logtype_array[type])
	return(logtxt);

    /* get the time */
    (void) time(&now);
    tm = localtime(&now);

    svctime = systime = 0;
    if(req) {
	if(req->rcvd_time.tv_sec) 
	    systime = now - req->rcvd_time.tv_sec;

	if(req->svc_start_time.tv_sec) 
	    svctime = now - req->svc_start_time.tv_sec;

	if(req->rcvd_time.tv_sec && req->svc_start_time.tv_sec) 
	    wttime = req->svc_start_time.tv_sec - req->rcvd_time.tv_sec; 
    }

    if ((type == L_QUEUE_COMP) && (systime < L_COMP_THRESHOLD))
	return(logtxt);

    *usertxt = '\0';

    if(req && req->peer_addr.s_addr &&(logtype_array[L_FIELDS]&L_FIELDS_HADDR))
	strcat(usertxt, pr_inet_ntoa(req->peer_addr.s_addr));

    if(type == L_DIR_UPDATE) {
	if(logtype_array[L_FIELDS] & L_FIELDS_USER_U) log_username++;
    }
    else if(type == L_DIR_REQUEST) {
	if(logtype_array[L_FIELDS] & L_FIELDS_USER_R) log_username++;
    }
    else if(logtype_array[L_FIELDS] & L_FIELDS_USER_I) log_username++;

    if(req && req->client_name && *(req->client_name) && log_username) {
	strcat(usertxt, "(");
	strcat(usertxt, req->client_name);
 	strcat(usertxt, ")");
    }

    if(req && req->sw_id && *(req->sw_id) && 
       (logtype_array[L_FIELDS] & L_FIELDS_SW_ID)) {
	strcat(usertxt, "[");
	strcat(usertxt, req->sw_id);
 	strcat(usertxt, "]");
    }

    if(req && (logtype_array[L_FIELDS] & L_FIELDS_PORT)){
	sprintf(fieldtxt,"[udp/%d]", PEER_PORT(req));
	strcat(usertxt, fieldtxt);
    }

    if(req && req->cid &&(logtype_array[L_FIELDS] & L_FIELDS_CID)) {
	sprintf(fieldtxt,"[cid=%d]", ntohs(req->cid));
	strcat(usertxt, fieldtxt);
    }

    if(req && (logtype_array[L_FIELDS] & L_FIELDS_STIME) &&
       ((systime>=L_SYSTIME_THRESHOLD) || (svctime>=L_SVCTIME_THRESHOLD) ||
	(wttime>=L_WTTIME_THRESHOLD))) {
	strcat(usertxt, "[");
	if(wttime >= L_WTTIME_THRESHOLD) {
	    if(notfirstfield++) strcat(usertxt, ",");
	    sprintf(fieldtxt,"%d:%02dwt", wttime / 60, wttime % 60);
	    strcat(usertxt, fieldtxt);
	}
	if(svctime >= L_SVCTIME_THRESHOLD) {
	    if(notfirstfield++) strcat(usertxt, ",");
	    sprintf(fieldtxt,"%d:%02dsvc", svctime / 60, svctime % 60);
	    strcat(usertxt, fieldtxt);
	}
	if(systime >= L_SYSTIME_THRESHOLD) {
	    if(notfirstfield++) strcat(usertxt, ",");
	    sprintf(fieldtxt,"%d:%02dsys", systime / 60, systime % 60);
	    strcat(usertxt, fieldtxt);
	}
	strcat(usertxt, "]");
    }

#ifdef P_LOGTO_SYSLOG
    if(!syslog_open++) openlog("prospero",LOG_PID|LOG_ODELAY,LOG_PROSPERO);

    if (logtype_array[type] & ~PLOG_TOFILE_ENABLED) {
	syslog(logtype_array[type] & ~PLOG_TOFILE_ENABLED, 
	       "%s%s%s", usertxt, (*usertxt ? " " : ""), logtxt);
    }
#endif

    /* If not printing to file return */
    if (! (logtype_array[type] & PLOG_TOFILE_ENABLED)) return(logtxt);

    if (!is_open) {
	if ((logfile = fopen(log_name,"a")) != NULL)
	    is_open = 1;
    }

    if (is_open) {
	/* print the log entry */
	fprintf(logfile,"%2d-%s-%02d %02d:%02d:%02d ",tm->tm_mday,
		month_sname(tm->tm_mon + 1),tm->tm_year,
		tm->tm_hour, tm->tm_min, tm->tm_sec);
	fprintf(logfile,"%s%s%s\n", usertxt, (*usertxt ? " - " : ""), logtxt);
    }
    /* even if the primary logfile couldn't be opened, go ahead and log to
	the manual (additional) logfile. */
    if(plog_additional_outfile) {
        fprintf(plog_additional_outfile, 
                "%s%s%s\n", usertxt, (*usertxt ? " - " : ""), logtxt);
        fflush(plog_additional_outfile);
    }

    (void) fclose(logfile);
    is_open = 0;

    return(logtxt);
}

char *
plog(int type,                  /* Type of log entry	    */
     RREQ req,                  /* Optional info on request */
     char *format, ...)         /* format string            */
{
    char *retval;
    va_list ap;

    va_start(ap, format);
    retval = vplog(type, req, format, ap);
    va_end(ap);
    return retval;
}
#else
char *
plog(type,req,format,a1,a2,a3,a4,a5,a6,a7,a8,a9,a0)
    int		type;		/* Type of log entry	    */
    RREQ	req;            /* Optional info on request */ 
    char	*format;	/* Format String	    */
    int		a1,a2,a3,a4,a5,a6,a7,a8,a9,a0; /* Args      */
{
    long time(),now;
    struct tm *tm;
    int	 log_username = 0;
    char *month_sname();
    char usertxt[MXPLOGENTRY];

    (void) qsprintf(logtxt, sizeof logtxt, format,a1,a2,a3,a4,a5,a6,
                    a7,a8,a9,a0);

    /* If we don't log this type of message, don't write to log */
    if (!logtype_array[type])
	return(logtxt);

    *usertxt = '\0';

    if(req && req->peer_addr.s_addr && (logtype_array[L_FIELDS] & L_FIELDS_HADDR))
	strcat(usertxt, pr_inet_ntoa(req->peer_addr.s_addr));

    if(type == L_DIR_UPDATE) {
	if(logtype_array[L_FIELDS] & L_FIELDS_USER_U) log_username++;
    }
    else if(type == L_DIR_REQUEST) {
	if(logtype_array[L_FIELDS] & L_FIELDS_USER_R) log_username++;
    }
    else if(logtype_array[L_FIELDS] & L_FIELDS_USER_I) log_username++;

    if(req && req->client_name && *(req->client_name) && log_username) {
	strcat(usertxt, "(");
	strcat(usertxt, req->client_name);
 	strcat(usertxt, ")");
    }

    if(req && req->sw_id && *(req->sw_id) && 
       (logtype_array[L_FIELDS] & L_FIELDS_SW_ID)) {
	strcat(usertxt, "[");
	strcat(usertxt, req->sw_id);
 	strcat(usertxt, "]");
    }

#ifdef P_LOGTO_SYSLOG
    if(!syslog_open++) openlog("prospero",LOG_PID|LOG_ODELAY,LOG_PROSPERO);

    if (logtype_array[type] & ~PLOG_TOFILE_ENABLED) {
	syslog(logtype_array[type] & ~PLOG_TOFILE_ENABLED, 
	       "%s%s%s", usertxt, (*usertxt ? " " : ""), logtxt);
    }
#endif

    /* If not printing to file return */
    if (! (logtype_array[type] & PLOG_TOFILE_ENABLED)) return(logtxt);

    if (!is_open) {
	if ((logfile = fopen(log_name,"a")) == NULL)
	    return(logtxt);
	is_open = 1;
    }

    /* get the time */
    (void) time(&now);
    tm = localtime(&now);

    /* print the log entry */
    fprintf(logfile,"%2d-%s-%02d %02d:%02d:%02d ",tm->tm_mday,
            month_sname(tm->tm_mon + 1),tm->tm_year,
            tm->tm_hour, tm->tm_min, tm->tm_sec);
    fprintf(logfile,"%s%s%s\n", usertxt, (*usertxt ? " - " : ""), logtxt);

    (void) fclose(logfile);
    is_open = 0;

    return(logtxt);
}
#endif

/*
 * set_logfile - Change the name of the logfile
 *
 *		  SET_LOGFILE changes the name of the file to which
 * 		  messages are logged.  If set_logfile is not called,
 * 		  the logfile defaults to PFSLOG.
 *
 *         ARGS:  filename - Name of new logfile
 *      Returns:  success always
 */
set_logfile(filename)
    char *filename;
{
    log_name = filename;
    if(is_open) (void) fclose(logfile);
    is_open = 0;
    return(PSUCCESS);
}

static char *pr_inet_ntoa(a)
    long a;
    {
	static char	astring[20];

#if BYTE_ORDER == BIG_ENDIAN
	sprintf(astring,"%d.%d.%d.%d",(a >> 24) & 0xff,
		(a >> 16) & 0xff,(a >> 8) & 0xff, a & 0xff);
#else
	sprintf(astring,"%d.%d.%d.%d", a & 0xff,
		(a >> 8) & 0xff,(a >> 16) & 0xff, (a >> 24) & 0xff);
#endif
	
	return(astring);
    }
		
close_plog()
    {
	if(is_open) fclose(logfile);
	is_open = 0;

#ifdef P_LOGTO_SYSLOG
	if(syslog_open) closelog();
	syslog_open = 0;
#endif
    }


int
#ifdef __STDC__
plog_manual(FILE *outf)
#else
plog_manual(outf)
    FILE *outf;
#endif
{
    plog_additional_outfile = outf;
}
