/*
 * Copyright (c) 1991 by the University of Washington
 *
 * For copying and distribution information, please see the file
 * <uw-copyright.h>.
 */

#include <uw-copyright.h>

#include <strings.h>

#include <ardp.h>
#include <pfs.h>
#include <psite.h>
#include <pprot.h>
#include <plog.h>
#include <pmachine.h>

static	ACL	find_aclent();
char	*addrights();
char	*subtractrights();

/*
 * change_acl - change access control list
 *
 *       CHANGE_ACL change an access control list (*aclp)
 *       by adding, deleting, or modifying the entry
 *       specified by ae.
 *
 *  ARGS:  aclp - pointer to ACL pointer
 *           ae - a pointer to a single ACL entry to be added or deleted
 *           id - client identification
 *        flags - information on the operation to be performed
 *       diracl - the directory acl (to check if user has access)
 *
 *  NOTE:  This code which automatically adds administer (unless noself
 *         is specified) can be used to upgrade < or [ privs to A.  Dirsrv
 *         include a check which sets noself when admister access is
 *         allows by virtual of a [ or <.
 */
int
change_acl(aclp,ae,req,flags,diracl)
    ACL		*aclp;	/* Pointer to ACL pointer */
    ACL		ae;	/* ACL entry to change    */
    RREQ	req;	/* Client identification  */
    int		flags;  /* Operation and flags    */
    ACL		diracl;/* Directory ACL          */
    {
	ACL	a = *aclp;
	ACL	wacl;
	int	operation;
	int	nosystem;
	int	noself;
	int	dflag;
	int	adminflag;

	operation = flags & EACL_OP;
	nosystem = flags & EACL_NOSYSTEM;
	noself = flags & EACL_NOSELF;
	dflag = !((flags&EACL_OTYPE)^EACL_DIRECTORY);

	switch(operation) {

	case EACL_DEFAULT:
	    aclfree(a);
	    a = NULL;

	    /* If adminflag is clear it means that the user would not be   */
	    /* able to administer the new directory or link, and that      */
	    /* such a situation is undesirable                             */
	    if(noself) adminflag = 1; /* Ok to clear admin rights for user */
	    else if(dflag) adminflag = check_acl(NULL,NULL,req,"A");
	    else adminflag = check_acl(diracl,NULL,req,"a");
	    
	    if(adminflag == 0) {
		/* Must leave user with administer rights */
		if(dflag) {
                    /* XXX This needs to be changed. */
                    /* XXX This MUST be changed.   Grant access according to
                       each access method.  */
		    a = acalloc();
		    a->acetype = ACL_ASRTHOST;
		    a->rights = stcopyr("B",a->rights);
		    a->principals = tkcopy(req->auth_info->principals);
		    wacl = acalloc();
		    wacl->acetype = ACL_DEFAULT;
		    a->next = wacl;
		    wacl->previous = a;
		    wacl->next = acalloc();
		    wacl->next->previous = wacl;
		    wacl = wacl->next;
		    wacl->acetype = ACL_SYSTEM;
		}
		else {
		    a = acalloc();
		    a->acetype = ACL_ASRTHOST;
		    a->rights = stcopyr("a",a->rights);
		    a->principals = tkcopy(req->auth_info->principals);
		    wacl = acalloc();
		    wacl->acetype = ACL_DIRECTORY;
		    a->next = wacl;
		    wacl->previous = a;
		}
	    }
	    acfree(ae);
	    *aclp = a;
	    return(PSUCCESS);

	case EACL_SET:
	    aclfree(a);
	    a = ae;

	    /* Unless the nosystem flag has been specified, add an */
	    /* entry for the SYSTEM ACL                            */
	    if(!nosystem) {
		wacl = acalloc();
		wacl->acetype = ACL_SYSTEM;
		a->next = wacl;
		wacl->previous = a;
	    }

	    /* If adminflag is clear it means that the user would not be    */
	    /* able to administer the new directory or link, and that       */
	    /* such a situation is undesirable                              */
	    if(noself) adminflag = 1;  /* Ok to clear admin rights for user */
	    else if(dflag) adminflag = check_acl(a,NULL,req,"A");
	    else adminflag = check_acl(diracl,a,req,"a");
	    
	    if(adminflag == 0) {
		/* Must leave user with administer rights */
		wacl = acalloc();
		wacl->acetype = ACL_ASRTHOST;
		if(dflag) wacl->rights = stcopyr("B",wacl->rights);
		else wacl->rights = stcopyr("a",wacl->rights);
                /* XXX Change this soon. */
		wacl->principals = tkcopy(req->auth_info->principals);
		wacl->next = a;
		a->previous = wacl;
		a = wacl;
	    }
	    *aclp = a;
	    return(PSUCCESS);

	eacl_insert:
	case EACL_INSERT:
	    /* Must make sure all required fields are specified      */
	    /* Rights must be included for all but NONE, DEFAULT,    */
	    /* SYSTEM, and DIRECTORY                                 */
	    if(!((ae->rights && *(ae->rights)) || (ae->acetype==ACL_NONE) || 
		 (ae->acetype==ACL_DEFAULT) || (ae->acetype==ACL_SYSTEM) || 
		 (ae->acetype==ACL_DIRECTORY))) {
		return(PFAILURE);
	    }

	    /* No need to make sure the user can still access since  */
	    /* we are only adding rights.  This ignores the case of  */
	    /* negative rights, but for now we assume that if the    */
	    /* user is specifying negative rights that they are      */
	    /* intended                                              */

	    /* If NULL, first add DIRECTORY or DEFAULT and SYSTEM    */
	    /* Note that NULL means DIRECTORY, or DEFAULT and SYSTEM */
	    /* so we need to leave SYSTEM even if the user said not  */
	    /* to add it                                             */
	    if(!a && dflag) {
		a = acalloc();
		a->acetype = ACL_DEFAULT;
		wacl = acalloc();
		wacl->acetype = ACL_SYSTEM;
		a->next = wacl;
		wacl->previous = a;
	    }
	    else if(!a) {
		a = acalloc();
		a->acetype = ACL_DIRECTORY;
	    }

	    /* Check to see if entry already exists */
	    wacl = find_aclent(a,ae,1);
	    if(wacl) {
		/* Already there */
		aclfree(ae);
		*aclp = a;
		return(PSUCCESS);
	    }

	    /* New ACL antries are added a head of list.  This means */
	    /* that any negative rights specified will override any  */
	    /* rights that were already present.  Additionaly, any   */
	    /* positive rights specified will override any negative  */
	    /* rights that already existed.                          */
	    wacl = ae;
	    while(wacl->next) wacl = wacl->next;
	    wacl->next = a;
	    a->previous = wacl;
	    a = wacl;
	    *aclp = a;
	    return(PSUCCESS);

	eacl_delete:
	case EACL_DELETE:
	    /* If NULL, first add DIRECTORY or DEFAULT and SYSTEM    */
	    /* Note that NULL means DIRECTORY, or DEFAULT and SYSTEM */
	    /* so we need to leave SYSTEM even if the user said not  */
	    /* to add it                                             */
	    if(!a && dflag) {
		a = acalloc();
		a->acetype = ACL_DEFAULT;
		wacl = acalloc();
		wacl->acetype = ACL_SYSTEM;
		a->next = wacl;
		wacl->previous = a;
	    }
	    else if(!a) {
		a = acalloc();
		a->acetype = ACL_DIRECTORY;
	    }

	    wacl = find_aclent(a,ae,1);
	    if(!wacl) return(PFAILURE);
	    if(a == wacl) {
		a = a->next;
		a->previous = NULL;
		acfree(wacl);
	    }
	    else {
		wacl->previous->next = wacl->next;
		if(wacl->next) wacl->next->previous = wacl->previous;
		acfree(wacl);
	    }

	    /* Make sure that user can fix his mistakes */
	    if(!noself) {
		if(dflag) adminflag = check_acl(a,NULL,req,"A");
		else adminflag = check_acl(diracl,a,req,"a");
                /* CHANGE THIS SOON */
		if(!adminflag) {
		    /* Must leave user with administer rights */
		    wacl = acalloc();
		    wacl->acetype = ACL_ASRTHOST;
		    if(dflag) wacl->rights = stcopyr("B",wacl->rights);
		    else wacl->rights = stcopyr("a",wacl->rights);
		    wacl->principals = tkcopy(req->auth_info->principals);
		    wacl->next = a;
		    a->previous = wacl;
		    a = wacl;
		}
	    }

	    /* If empty, must create a placeholder so that it */
	    /* doesn't revert to nulldir                      */
	    if(!a) {
		a = acalloc();
		a->acetype = ACL_NONE;
	    }
	    
	    acfree(ae);
	    *aclp = a;
	    return(PSUCCESS);

	case EACL_ADD:
	    /* If no rights specified must be insert */
	    if(!(ae->rights)) goto eacl_insert;

	    /* Havn't figured out how to ADD ><][)(, so go to INSERT */
	    if(index("><][)(",*(ae->rights))) goto eacl_insert;

	    /* If NULL, first add DIRECTORY or DEFAULT and SYSTEM    */
	    /* Note that NULL means DIRECTORY, or DEFAULT and SYSTEM */
	    /* so we need to leave SYSTEM even if the user said not  */
	    /* to add it                                             */
	    if(!a && dflag) {
		a = acalloc();
		a->acetype = ACL_DEFAULT;
		wacl = acalloc();
		wacl->acetype = ACL_SYSTEM;
		a->next = wacl;
		wacl->previous = a;
		*aclp = a;
	    }
	    else if(!a) {
		a = acalloc();
		a->acetype = ACL_DIRECTORY;
		*aclp = a;
	    }

	    wacl = find_aclent(a,ae,0);

	    /* If no other entries, then go to insert */
	    if(!wacl) goto eacl_insert;

	    /* Now we must add characters to wacl->rights for */
	    /* any new characters in ae->rights               */
	    wacl->rights = stcopyr(addrights(wacl->rights,ae->rights),wacl->rights);
	    acfree(ae);
	    return(PSUCCESS);

	case EACL_SUBTRACT:
	    /* If no rights specified must delete */
	    if(!(ae->rights)) goto eacl_delete;

	    /* Havn't figured out how to DELETE ><][)(, so go to DELETE */
	    if(index("><][)(",*(ae->rights))) goto eacl_delete;

	    wacl = find_aclent(a,ae,0);

	    /* If no other entries, return error */
	    if(!wacl) {
		acfree(ae);
		return(PFAILURE);
	    }

	    /* Now we must subtract characters from wacl->rights */
	    /* for the characters in ae->rights                  */
	    wacl->rights = subtractrights(wacl->rights,ae->rights);

	    /* If rights are now null, must delete the entry */
	    if(!wacl->rights || !*(wacl->rights)) {
		if(wacl->previous) {
		    wacl->previous->next = wacl->next;
		    if(wacl->next) wacl->next->previous = wacl->previous;
		    acfree(wacl);
		}
		else { /* It is at start of list */
		    a = wacl->next;
		    a->previous = NULL;
		    acfree(wacl);
		}
	    }

	    /* Make sure that user can fix his mistakes */
	    if(!noself) {
		if(dflag) adminflag = check_acl(a,NULL,req,"A");
		else adminflag = check_acl(diracl,a,req,"a");
		if(!adminflag) {
		    /* Must leave user with administer rights */
		    wacl = acalloc();
		    wacl->acetype = ACL_ASRTHOST;
		    if(dflag) wacl->rights = stcopyr("B",wacl->rights);
		    else wacl->rights = stcopyr("a",wacl->rights);
		    wacl->principals = tkcopy(req->auth_info->principals);
		    wacl->next = a;
		    a->previous = wacl;
		    a = wacl;
		}
	    }

	    /* If empty, must create a placeholder so that it */
	    /* doesn't revert to nulldir                      */
	    if(!a) {
		a = acalloc();
		a->acetype = ACL_NONE;
	    }
	    
	    acfree(ae);
	    *aclp = a;
	    return(PSUCCESS);
	}

	return(PFAILURE);
    }


/*
 * Find an ACL entry matching entry e in list a
 *
 *         r is a flag which if set means the rights must match.
 *         if clear, then the rights can be different.
 */
static ACL find_aclent(a,e,r)
    ACL	a;
    ACL e;
    int r;
    {
	ACL w;

	w = a;
	while(w) {
	    if((w->acetype == e->acetype) &&
	       ((!(w->atype) && !(e->atype)) ||
		(w->atype && e->atype && (strcmp(w->atype,e->atype)==0))) &&
	       ((!(w->principals) && !(e->principals)) ||
		(w->principals && e->principals 
                 && equal_sequences(w->principals,e->principals))) &&
	       (!r || (!(w->rights) && !(e->rights)) ||
		(w->rights && e->rights && (strcmp(w->rights,e->rights)==0))))
		return(w);
	    w = w->next;
	}
	return(NULL);
    }



/* 
 * Add rights returns a string containing those rights
 * that are in r or a.  For known rights, addrights
 * will try to place them in canonical order.
 *
 * WARNING: Subtractrights frees r.  It is expected that
 *          the string pointed to by r will be replaced
 *          by the string handed back.
 */
char *addrights(r,a)
    char	*r;
    char	*a;
    {
	static char	*canonical = "-AB><)(][VYLRMDIavlrmd";
	char		*cp;
	char		*newrights;
	char		*nr; /* newrights */

	if(!r) return(stcopy(a));

	nr = newrights = stalloc(strlen(r)+strlen(a)+1);

	/* First check each known right */
	for(cp = canonical;*cp;cp++) {
	    if(index(r,*cp)) *(nr++) = *cp;
	    else if(index(a,*cp)) *(nr++) = *cp;
	}
	/* Now scan r and include anything that is not canonical */
	for(cp = r;*cp;cp++) {
	    if(index(canonical,*cp)==0) *(nr++) = *cp;
	}
	/* Now scan a and include anything that is not canonical */
	/* and isn't in r                                        */
	for(cp = a;*cp;cp++) {
	    if((index(canonical,*cp)==0)&&
	       (index(r,*cp)==0)) *(nr++) = *cp;
	}
	*(nr++) = '\0';
	stfree(r);
	return(newrights);
    }

/* 
 * Subtract rights returns a string containing those rights
 * in r that are not in s. 
 *
 * WARNING: Subtractrights frees r.  It is expected that
 *          the string pointed to by r will be replaced
 *          by the string handed back.
 */
char *subtractrights(r,s)
    char	*r;
    char	*s;
    {
	char		*newrights = stalloc(strlen(r)+1);
	char		*or; /* oldrights */
	char		*nr; /* newrights */

	for(or = r,nr = newrights;*or;or++) {
	    if(index(s,*or)==NULL) *(nr++) = *or;
	}
	*(nr++) = '\0';

	stfree(r);
	return(newrights);
    }
