/*
 * Copyright (c) 1992, 1993 by the University of Southern California
 *
 * For copying and distribution information, please see the file
 * <usc-copyr.h>
 *
 * Written  by swa    1992     to provide common code for starting request
 * Modified by prasad 1992     to add Kerberos support
 * Modified by bcn    1/93     to use new ardp library and rreq structure
 */

#include <usc-copyr.h>

#include <stdio.h>
#include <strings.h>
#include <netdb.h>
#include <pwd.h>

/* For compatability between GCC-2.3.1 running under SunOS 4.1 and the sunOS
 * standard include files (they define size_t and others differently), we MUST 
 * include stddef and stdarg after anything that might include the sun include
 * file <sys/stdtypes.h> (in this case, <pwd.h> includes <sys/stdtypes.h>).
 */ 

#include <stddef.h>
#include <stdarg.h>

#include <ardp.h>
#include <psite.h>              /* must precede pfs.h for def. of KERBEROS  */
#include <pfs.h>
#include <pprot.h>
#include <perrno.h>
#include <pcompat.h>

#ifdef KERBEROS
#include <krb5/krb5.h>
#endif /* KERBEROS */


extern int	pfs_debug;
extern int	perrno;
extern char	p_err_string[];

static PAUTH p__get_pauth(int type, char *hostname);

/* 
 * p__start_req - start a prospero request
 *
 *     p__start_req takes the name of the server (used by Kerberos to 
 *     determine what credentials are needed), it allocates a request
 *     structure, and it fills in the start, including the Prospero
 *     version and software ID, and authentication information.  The
 *     function returns a request of type RREQ, which can be used 
 *     in subsequent calls to p__addreq, and passed to ardp_send.
 */
RREQ
p__start_req(server_hostname)
    char server_hostname[];
{
    RREQ req = ardp_rqalloc(); /* The request to fill in         */
    PAUTH authinfo;      /* Structure containing authentication info */
    TOKEN prin;          /* To Iterate through principal names       */

    if (!req)
        internal_error("Out of memory!"); /* XXX this may be an unwise way of
                                             handling an out-of-memory problem.
                                             Get it working now, fix it later.
                                             */ 

    p__add_req(req, "VERSION %d %s", VFPROT_VNO, PFS_SW_ID);
    authinfo = p__get_pauth(PFSA_UNAUTHENTICATED, server_hostname);
    p__add_req(req, "\nAUTHENTICATE '' UNAUTHENTICATED %'s",
           authinfo->authenticator);
    /* Note that the principals are optional additional information, */
    /* for debugging use -- we don't worry about it here.            */
    for(prin = authinfo->principals; prin; prin = prin->next)
        p__add_req(req, " %'s", prin->token); 
    pafree(authinfo);

#ifdef KERBEROS
    if (authinfo = p__get_pauth(PFSA_KERBEROS, server_hostname)) {
        p__add_req(req, "\nAUTHENTICATE '' KERBEROS %'s", 
                   authinfo->authenticator);
        for (prin = authinfo->principals; prin; prin = prin->next)
            p__add_req(req, " %'s", prin->token); 
        pafree(authinfo);
    }
#endif
    p__add_req(req, "\n");
    return(req);
}

/* Adds stuff to a Prospero request. */
int
p__add_req(RREQ req, char fmt[], ...)
{
    int retval;
    va_list ap;
    va_start(ap,fmt);
    retval = vp__add_req(req, fmt, ap);
    va_end(ap);
    return(retval);
}

int
vp__add_req(RREQ req, char fmt[], va_list ap)
{
    PTEXT	ptmp;

    if(!req->outpkt) {
	ptmp = ardp_ptalloc();
	if(!ptmp) return(PFAILURE);
	APPEND_ITEM(ptmp,req->outpkt);
    }

    /* XXX For now we won't bother adding a second packet if too long */
    req->outpkt->length += vqsprintf(req->outpkt->ioptr,
                  ARDP_PTXT_LEN - req->outpkt->length, fmt, ap) - 1;
    /* -1 because we need to remove trailing null */
    if (req->outpkt->length >= ARDP_PTXT_LEN)
        internal_error("ARDP_PTXT_LEN overflowed!");
    req->outpkt->ioptr = req->outpkt->start + req->outpkt->length;
    return(PSUCCESS);
}

static 
PAUTH p__get_pauth(type, server_hostname)
     int type;
     char *server_hostname;            /* Server hostname  */
{
#ifdef KERBEROS
    struct servent *serv;
    struct hostent *host;
    char *cp;
    char full_hname[255];
    krb5_data buf;
    char **s_realms;			/* server's Kerberos realm(s) */
    krb5_checksum send_cksum;
    krb5_ccache ccdef;
    krb5_principal server;
#endif
    PAUTH auth = paalloc();
    struct passwd *whoiampw;
    int retval = 0;

    auth->principals = NULL;    /* not really needed.  Let's be paranoid. */
    switch(auth->ainfo_type = type) {
      case PFSA_UNAUTHENTICATED:
	/* find out who we are */
	DISABLE_PFS(whoiampw = getpwuid(getuid()));
	if (whoiampw == 0) {
	    auth->authenticator = stcopy("*unknown*");
	}
	else {
	    auth->authenticator = stcopy(whoiampw->pw_name);
	}
        return(auth);
	break;
	
#ifdef KERBEROS                 /* 4/5/93  Made error handling pass errors
                                   to higher layers. */
      case PFSA_KERBEROS:
        qsprintf(full_hname, sizeof full_hname, "%s", server_hostname);
        /* Strip off the trailing (portnum), if any */
        for (cp = full_hname; *cp; ++cp)
            ;
        if (cp > full_hname && *--cp == ')') {
            while (*--cp && cp >= full_hname) {
                if (*cp == '(') {
                    *cp = '\0';
                    break;
                }
                if (!isdigit(*cp)) 
                    break;
            }
        }
	/* Look up server host */
	if ((host = gethostbyname(full_hname)) == (struct hostent *) 0) {
	    if (pfs_debug) 
                fprintf(stderr, "p__get_pauth(): Error in Kerberos \
authentication: %s: unknown host\n", full_hname);
            pafree(auth);
            return NULL;
	}
        qsprintf(full_hname, sizeof full_hname, "%s", host->h_name);

	/* lower-case to get name for "instance" part of service name */
	for (cp = full_hname; *cp; cp++)
	  if (isupper(*cp))
            *cp = tolower(*cp);
	
	/* Get Kerberos realm of server host */
	if (retval = krb5_get_host_realm(full_hname, &s_realms)) {
            if (pfs_debug) 
                fprintf(stderr, "p__get_pauth(): Error getting realm of Kerberos host %s\n",
                        full_hname); 
            pafree(auth);
            return NULL;
	}
	
	/* PREPARE KRB_AP_REQ MESSAGE */
	
	/* compute checksum, using CRC-32 */
	if (!(send_cksum.contents = (krb5_octet *)
	      stalloc(krb5_checksum_size(CKSUMTYPE_CRC32)))) {
            if (pfs_debug)
                fprintf(stderr, "p__get_pauth(): Cannot allocate Kerberos checksum\n");
            pafree(auth);
            return NULL;
	}
	
	/* choose some random stuff to compute checksum from */
	if (retval = krb5_calculate_checksum(CKSUMTYPE_CRC32,
					     server_hostname,
					     strlen(server_hostname),
					     0,
					     0, /* if length is 0, 
						   crc-32 doesn't
						   use the seed */
					     &send_cksum)) {
            if (pfs_debug)
                fprintf(stderr, "p__get_pauth(): Error while computing Kerberos checksum\n");
            stfree(send_cksum.contents);
            pafree(auth);
            return NULL;
	}
	
        /* Get credentials for server, create krb_mk_req message */
	
	if (retval = krb5_cc_default(&ccdef)) {
            
	    if (pfs_debug) 
                fprintf(stderr, "p__get_pauth(): Error while getting default \
Kerberos  cache\n");
            stfree(send_cksum.contents);
            pafree(auth);
            return NULL;
	    exit(1);
	}
	
	/* compose the server name. [0] == realm,
	   [1] == service name (by convention),
	   [2] == FULL host name (by convention)
	   [3] == null ptr */
	
	if (retval = krb5_build_principal(&server,
					  strlen(s_realms[0]), s_realms[0],
					  KERBEROS_SERVICE, full_hname, 0)) {
	    if (pfs_debug)
                fprintf(stderr, "p__get_pauth(): Error while setting up \
Kerberos server principal\n");
            stfree(send_cksum.contents);
            pafree(auth);
            return NULL;
	}
	
	if (retval = krb5_mk_req(server, 0 /* default options */,  &send_cksum,
				 ccdef, &buf)) {
	    if (pfs_debug) fprintf(stderr, "p__get_pauth(): Error while \
preparing Kerberos AP_REQ\n");
            pafree(auth);
            stfree(send_cksum.contents);
            return NULL;
	}
	stfree(send_cksum.contents);
	auth->authenticator = stalloc(AUTHENTICATOR_SZ);
	retval = binencode(buf.data, buf.length, 
			   auth->authenticator, 
			   AUTHENTICATOR_SZ);
	if (retval >= AUTHENTICATOR_SZ) {
	    internal_error("binencode: Buffer too small");
	}
	
	return(auth);
	break;
#endif
      default:
	fprintf(stderr, "p__get_pauth(): Unknown authenticator type: %d\n", type);
        internal_error("Unknown authenticator type!");
        /* NOTREACHED */
    }
}
