/*
  Copyright(c) 1993 by the University of Southern California
  For copying information, see the file <usc-copyr.h>
*/

#include <usc-copyr.h>
#include <ardp.h>
#include <pfs.h>
#include <pparse.h>

extern int pfs_debug;

int (*stdio_fseek)();

/* Returns the character NUL if EOF detected. */
char
in_readc(INPUT in)
{
    int c;

    switch(in->sourcetype) {
    case IO_FILE:
        if((*stdio_fseek)(in->file, in->offset, 0) == -1) {
            /* improper seek */
            if (pfs_debug) 
                fprintf(stderr, "in in_readc() an improper fseek was detected.");
            return '\0';
        }
        if((c = getc(in->file)) == EOF) {
            clearerr(in->file);        /* don't want this to stick around. */
            return '\0';
        }
        return c;
    case IO_STRING:
        return *(in->s);
    case IO_RREQ:
        return *in->ptext_ioptr;
    default:
        internal_error("invalid in->iotype");
    }
}


/* this is not too efficient.  Fix it one day (yeah, right.). */
char 
in_readcahead(INPUT in, int howfar)
{
    INPUT_ST incpy_st;
    INPUT incpy = &incpy_st;

    assert(howfar >= 0);
    input_copy(in, incpy);
    while (howfar-- > 0)
        in_incc(incpy);
    return in_readc(incpy);
}


/* This function may legally be called on a stream which has already reached
   EOF.  In that case, it's a no-op. */
void 
in_incc(INPUT in)
{
    /* this takes advantage of the null termination trick. */
    
    int c;

    switch(in->sourcetype) {
    case IO_FILE:
        ++in->offset;
        break;
    case IO_STRING:
        if (in->s) ++(in->s);   /* don't increment past end of the string. */
        break;
    case IO_RREQ:
        if (in->inpkt) {
            ++in->ptext_ioptr;
            ++in->offset;
            /* Do a loop because there might be a crazy client that
               sends some packets in a sequence with empty length fields. 
               Skip any of them we encounter; go to the next packet with some
               content. */
            while (*in->ptext_ioptr == '\0') {
                in->inpkt = in->inpkt->next;
                if (in->inpkt == NULL)
                    break;
                in->ptext_ioptr = in->inpkt->text;
            }
        }
        break;
    default:
        internal_error("invalid in->iotype");
    }
}


int
in_eof(INPUT in)
{
    switch(in->sourcetype) {
    case IO_FILE:
        if((*stdio_fseek)(in->file, in->offset, 0) == -1) {
            /* improper seek */
            if (pfs_debug) 
                fprintf(stderr, "in in_readc() an improper fseek was detected.");
            return EOF;
        }
        if(getc(in->file) == EOF) {
            clearerr(in->file);        /* don't want this to stick around. */
            return EOF;
        }
        return 0;
    case IO_STRING:
        if (*(in->s)) return 0; 
        else return EOF;
    case IO_RREQ:
        if (in->inpkt) return 0;
        else return EOF;
    }
}
