/*
 * Copyright (c) 1992, 1993 by the University of Southern California
 *
 * For copying and distribution information, please see the file <usc-copyr.h>
 */

#include <usc-copyr.h>
#include <ctype.h>
#include <pfs.h>
#include <pparse.h>
#include <perrno.h>



/* in_line reads quoted Prospero lines from a number of sources, up to the next
   unquoted newline.   Upon return, *thislinep points to the start of a
   NUL-terminated string which contains a Prospero protocol command line.   
   *next_wordp points to the first word in that string after command. 

   in_line passes through the error codes returned by its subsidiary functions.
   The only error they return is PARSE_ERROR.  If pfs_debug is set, then
   explanatory detail describing the malformed line will be reported on 
   stderr. 

   *thislinep and *next_wordp point to a buffer private to in_line().  This
   buffer may be rewritten after each new line is read.
 */

int 
in_line(INPUT in, char **thislinep, char **next_wordp)
{
    int tmp;                    /* temp return value. */
    INPUT_ST eol_st;      /* temp. variable for end of line. */
    INPUT eol = &eol_st;      /* temp. variable for end of line. */
    int linebuflen;                 /* length to copy into linebuf  */
    /* A local static variable.  Points to the line in progress. */
    static char *linebuf = NULL;
    char *cp;            /* temp. index */

    if (!in_nextline(in)) internal_error("in_line() called with no data");
    /* Ok, set EOL to the end of this input line. */
    tmp = qscanf(in, "%~%r%'*(^\n\r)%r", in, eol);
    if (tmp < 2) {
        qsprintf(p_err_string, P_ERR_STRING_SZ, 
                 "Received Prospero packet with an unterminated quote");
        return PARSE_ERROR;
    }
    /* Make sure that linebuf points to a string with enough room to hold the
       current line. */
    if (in->sourcetype == IO_STRING) {
        linebuflen = eol->s - in->s; /* string type doesn't use offset member.
                                        */
    } else {
        linebuflen = eol->offset - in->offset; /* size that strlen() would
                                                  return */ 
    }
    if (!linebuf)
        assert(linebuf = stalloc(linebuflen + 1));
    else if (stsize(linebuf) < linebuflen + 1) {
        stfree(linebuf);
        assert(linebuf = stalloc(linebuflen + 1));
    }
    /* Now copy the bytes from the input stream to the linebuf.  This preserves
       quoting, which is very important. */
    for(cp = linebuf; linebuflen > 0; ++cp, --linebuflen, in_incc(in))
        *cp = in_readc(in);
    *cp = '\0';
    assert((in->sourcetype != IO_STRING) ? in->offset == eol->offset : 1);
    /* I need to trim off trailing spaces here.  Or else there might be
       trouble.  Hmm. */
    while (cp > linebuf && isspace(*--cp))
        *cp = '\0';
    /* push on the read pointer for in to the start of the next line. */
    tmp = qscanf(in, "%R", in);
    if (tmp < 1) {
        qsprintf(p_err_string, P_ERR_STRING_SZ, 
                 "Received Prospero packet with a line that was not LF \
terminated.");
        return PARSE_ERROR;
    }
    /* linebuf now points to a complete line of still quoted text. */
    *thislinep = linebuf;
    if(qsscanf(*thislinep, "%*'s%~%r", next_wordp) < 1
       || *thislinep == *next_wordp) {
        qsprintf(p_err_string, P_ERR_STRING_SZ, 
                 "Received Prospero packet with an empty line.");
        return PARSE_ERROR;
    }
    return PSUCCESS;
}


void
rreqtoin(RREQ rreq, INPUT in)
{
    in->sourcetype = IO_RREQ;
    if(in->rreq = rreq) {       /* might be NULL.  */
        in->inpkt = rreq->rcvd;
        in->ptext_ioptr = rreq->rcvd->start;
        /* Do a loop because there might be a crazy client that sends some
           packets in a sequence with empty length fields.  Skip any of them we
           encounter; go to the next packet with some content. */
        in->offset = 0;         /* on byte 0. */
        while (*in->ptext_ioptr == '\0') {
            in->inpkt = in->inpkt->next;
            if (in->inpkt == NULL)
                break;
            in->ptext_ioptr = in->inpkt->text;
        }
    }
#ifndef NDEBUG
    in->file = NULL;
    in->s = NULL;
#endif
    in->flags = JUST_INITIALIZED; /* is this needed? */
}

