/*
 * Copyright (c) 1991 by the University of Washington
 *
 * For copying and distribution information, please see the file
 * <uw-copyright.h>.
 */

#include <uw-copyright.h>


#include <netdb.h>
#include <sgtty.h>
#include <signal.h>
#include <setjmp.h>
#include <stdio.h>
#include <strings.h>
#include <sys/time.h>
#include <sys/file.h>
#include <sys/ioctl.h>
#include <sys/param.h>
#include <sys/socket.h>

#include <pfs.h>
#include <psite.h>
#include <plog.h>
#include <pauthent.h>
#include <pprot.h>
#include <perrno.h>
#include <pmachine.h>

#define LOG_PACKET(AD,QP,PR) \
    if(PR || (pQlen > 4)) \
      if(pQNlen && PR) \
         plog(L_QUEUE_INFO, AD, NULL, \
	      "Queued: %d of %d (%d) - Priority %d", \
              QP, pQlen, pQNlen, PR, 0); \
      else if(pQNlen) \
         plog(L_QUEUE_INFO, AD, NULL, "Queued: %d of %d (%d)", \
              QP, pQlen, pQNlen, 0); \
      else if(PR) \
         plog(L_QUEUE_INFO, AD, NULL, "Queued: %d of %d - Priority %d", \
               QP, pQlen, PR, 0); \
      else if(QP != pQlen) \
         plog(L_QUEUE_INFO, AD, NULL, "Queued: %d of %d", QP, pQlen, 0); \
      else plog(L_QUEUE_INFO, AD, NULL, "Queued: %d", pQlen, 0); 


static	PREQ	pending = NULL;
int		pQlen = 0;    /* Length of pending queue           */
int		pQNlen = 0;   /* Number of requests that are niced */

static  PREQ	running = NULL;

static	PREQ	incomplete = NULL;
static  PREQ	accepted = NULL;

extern int	errno;

extern int	oldform_count;

static int		srvport = 0;
static int		prvport = -1;
static fd_set		readfds;	/* Used for select		 */
static struct timeval	zerotime; 

static int (*pri_func)() = NULL;  /* Function to compare priorities       */
static int pri_override = 0;	  /* If 1, then oveeride value in request */


struct sockaddr_in sin = {AF_INET};

bind_port(portname)
    char	*portname;
    {
	struct servent 		*sp;
	int     		on = 1;
	int			port_no = 0;

	zerotime.tv_sec = 0;
	zerotime.tv_usec = 0;

	if(*portname == '#') {
	    sscanf(portname+1,"%d",&port_no);
	    if(port_no == 0) {
		fprintf(stderr, "dirsrv: port number must follow #\n");
		exit(1);
	    }
	    sin.sin_port = htons((ushort) port_no);
	}
	else if((sp = getservbyname(portname, "udp")) != NULL) {
	    sin.sin_port = sp->s_port;
	}
	else if(strcmp(portname,"dirsrv") == 0) {
	    fprintf(stderr, "dirsrv: udp/dirsrv unknown service - using %d\n", 
		    DIRSRV_PORT);
	    sin.sin_port = htons((ushort) DIRSRV_PORT);
	}
	else {
	    fprintf(stderr, "dirsrv: udp/%s unknown service\n",portname);
	    exit(1);
	}

	if ((srvport = socket(AF_INET, SOCK_DGRAM, 0)) < 0) {
	    plog(L_STATUS,NULL,NULL,"Startup - Can't open socket",0);
	    fprintf(stderr, "dirsrv: Can't open socket\n");
	    exit(1);
	}
	if (setsockopt(srvport, SOL_SOCKET, SO_REUSEADDR, &on, sizeof(on)) < 0)
	    fprintf(stderr, "dirsrv: setsockopt (SO_REUSEADDR)\n");
	
	if (bind(srvport, &sin, S_AD_SZ) < 0) {
	    plog(L_STATUS,NULL,NULL,"Startup - Can't bind socket",0);
	    fprintf(stderr, "dirsrv: Can not bind socket\n");
	    exit(1);
	}
	return(ntohs(sin.sin_port));
    }

/* 
 * set_queuing_policy allows one to provide a function that will set priorities
 * for requests.  When passed two req structures, r1 and r2, the function 
 * should  return a negative number if r1 should be executed first (i.e. r1
 * has a lower numerical priority) and positive if r2 should be executed
 * first.  If the function returns 0, it means the two have the same 
 * priority and should be executed FCFS.  If override is non-zero, then
 * the priority function is to be applied to all requests.  If non-zero,
 * it is only applied to those with identical a priori priorities (as
 * specified in the datagram itself.
 */
set_queuing_policy(pf,override)
    int (*pf)(); 		/* Function to compare priorities       */
    int	override;		/* If 1, then oveeride value in request */
    {
	pri_func = pf;
	pri_override = override;
	return(PSUCCESS);
    }


set_prvport(port)
    int		port;
    {
	prvport = port;
	zerotime.tv_sec = 0;
	zerotime.tv_usec = 0;
	return(PSUCCESS);
    }


check_for_messages()
    {
	struct sockaddr_in 	from;
	int			fromlen;
	int			n = 0;
	PTEXT			pkt;
	int			cid = 0;
	int			tot_pkt = 0;
	int			recvd_thru = 0;
	unsigned char           flags1,flags2;
	unsigned short		pid;
	short			priority;
	int			keep_looking; /* Keep looking for dup */
	int			tmp;
	int			dpos; /* Position of dupe in queue  */
	PREQ		      	treq; /* Temporary request pointer  */
	PREQ			nreq; /* New request pointer        */
	int			hdr_len;
	char			*ctlptr;
	short			stmp;

    check_for_more:

	/* Check both the prived and unprived ports if necessary */
	FD_ZERO(&readfds);
	FD_SET(srvport, &readfds);
	if(prvport != -1) FD_SET(prvport, &readfds); 
	tmp = select(srvport + 1,&readfds,(fd_set *)0,(fd_set *)0,&zerotime);

	if(tmp == 0) return(PSUCCESS);
	if(tmp < 0) return(PFAILURE);

	pkt = ptalloc();
	recvd_thru = 0;
	priority = 0;

	/* There is a message waiting, add it to the queue */

	fromlen = sizeof(from);
	if((prvport >= 0) && FD_ISSET(prvport,&readfds))
	    n = recvfrom(prvport, pkt->dat, MAX_PTXT_LEN, 0, &from, &fromlen);
	else n = recvfrom(srvport, pkt->dat, MAX_PTXT_LEN, 0, &from, &fromlen);
	if (n <= 0) {
	    plog(L_NET_ERR,NULL,NULL,"Bad recvfrom n = %d errno = %d",n,errno,0);
	    return(PFAILURE);
	}

	pkt->start = pkt->dat;
	pkt->length = n;
	*(pkt->start + pkt->length) = NULL;
	pkt->mbz = 0; /* force zeros to catch runaway strings     */

	pkt->seq = 1;
	if((hdr_len = (unsigned char) *(pkt->start)) < 20) {
	    ctlptr = pkt->start + 1;
	    pkt->seq = 0;
	    if(hdr_len >= 3) { 	/* Connection ID */
		bcopy(ctlptr,&stmp,2);
		if(stmp) cid = ntohs(stmp);
		ctlptr += 2;
	    }
	    if(hdr_len >= 5) {	/* Packet number */
		bcopy(ctlptr,&stmp,2);
		pkt->seq = ntohs(stmp);
		ctlptr += 2;
	    }
	    else { /* No packet number specified, so this is the only one */
		pkt->seq = 1;
		tot_pkt = 1;
	    }
	    if(hdr_len >= 7) {	    /* Total number of packets */
		bcopy(ctlptr,&stmp,2);  /* 0 means don't know      */
		if(stmp) tot_pkt = ntohs(stmp);
		ctlptr += 2;
	    }
	    if(hdr_len >= 9) {	/* Receievd through */
		bcopy(ctlptr,&stmp,2);  /* 0 means don't know      */
		if(stmp) {
		    recvd_thru = ntohs(stmp);
#ifdef DEBUG_RDGRAM
		    plog(L_NET_RDPERR, from.sin_addr.s_addr, NULL, "CID %d Received through %d", cid, recvd_thru, 0);
#endif
		}
		ctlptr += 2;
	    }

	    if(hdr_len >= 11) {	/* Backoff */
		/* Not supported by server */
	    }
	    ctlptr += 2;

	    flags1 = flags2 = 0;
	    if(hdr_len >= 12) flags1 = *ctlptr++; /* Flags */
	    if(hdr_len >= 13) flags2 = *ctlptr++; /* Flags */

	    if(hdr_len >= 15) {	/* Priority or PID */
		bcopy(ctlptr,&stmp,2);
		if(flags1 & 0x1) pid = ntohs(stmp);
		else if (flags1 & 0x2) priority = ntohs(stmp);
	    }
	    ctlptr += 2;

	    if(hdr_len >= 17) {	/* Priority if PID specified above */
		bcopy(ctlptr,&stmp,2);
		if((flags1 & 0x3) == 0x3) priority = ntohs(stmp);
	    }
	    ctlptr += 2;

	    if((priority < 0) && (priority != -42)) {
		plog(L_NET_RDPERR, from.sin_addr.s_addr, NULL, "Priority %d requested - ignored", priority, 0);
		priority = 0;
	    }

	    if(pkt->seq == 0) goto check_for_more;
	    pkt->length -= hdr_len;
	    pkt->start += hdr_len;
	}
	else {
	    /* When we are no longer getting any of these old format  */
	    /* requests, we know that everyone is using the new       */
	    /* reliable datagram protocol, and that they also         */
	    /* have the singleton response bug fixed.  We can then    */
	    /* get rid of the special casing for singleton responses  */
	    oldform_count++;
	    /* Lower the priority to encourage people to upgrade  */
	    priority = 1;
#ifdef LOGOLDFORMAT
	    plog(L_DIR_PWARN,from.sin_addr.s_addr,NULL,"Old RDP format",0);
#endif LOGOLDFORMAT
	    pkt->seq = 1;
	    tot_pkt = 1;
	}

	/* Check to see if it is already on incomplete, accepted, or done */

	/* For now let's assume not - this code will become important */
	/* when the client code is changed to include connection IDs   */

	plog(L_NET_INFO, from.sin_addr.s_addr, NULL, "Received packet", 0);

	/* If this message is complete, insert it at end of pending */
	/* but make sure the request is not already in pending      */
	/* If incomplete, insert it at end of incomplete            */
	/* Incomplete not implemented for the time being            */
	if((pkt->seq == 1) && (tot_pkt == 1)) {
	    nreq = pralloc();
	    treq = pending;
	    dpos = 1;

	    nreq->cid = cid;
	    nreq->recv_tot = 1;
	    nreq->recv = pkt;
	    if(recvd_thru) {
		nreq->rcvd_thru = recvd_thru;
		/* Response is probably cached */
		if(priority > 0) priority = 0;
	    }
	    nreq->priority = priority;
	    bcopy(&from,&(nreq->fromto),sizeof(nreq->fromto));

	    if(running && (running->cid == nreq->cid) && (running->cid != 0) &&
	       (bcmp(running->fromto,nreq->fromto,sizeof(nreq->fromto))==0)) {
		/* Request is running right now */
		plog(L_QUEUE_INFO,nreq->fromto.sin_addr.s_addr,NULL,"Duplicate discarded (presently executing)",0);
		update_req_cfields(running,nreq);
		prfree(nreq);
		nreq = running;
	    }
	    else if(pending) {
		while(treq) {
		  if((treq->cid != 0) && (treq->cid == nreq->cid) &&
		    (bcmp(treq->fromto,nreq->fromto,sizeof(treq->fromto))==0)){
		      /* Request is already on queue */
		      plog(L_QUEUE_INFO,nreq->fromto.sin_addr.s_addr,NULL,"Duplicate discarded (%d of %d)",dpos,pQlen,0);
		      update_req_cfields(treq,nreq);
		      prfree(nreq);
		      nreq = treq;
		      keep_looking = 0;  /* We found the duplicate */
		      break;
		  }
		  /* old if(treq->priority > nreq->priority)  */
		  if((pri_override && pri_func && (pri_func(nreq,treq) < 0)) ||
		     (!pri_override && ((nreq->priority < treq->priority) ||
			    ((treq->priority == nreq->priority) && pri_func &&
			     (pri_func(nreq,treq) < 0))))) {
		      if(pending == treq) {
			  nreq->next = pending;
			  pending->previous = nreq;
			  pending = nreq;
			  nreq->previous = NULL;
		      }
		      else {
			  nreq->next = treq;
			  nreq->previous = treq->previous;
			  nreq->previous->next = nreq;
			  treq->previous = nreq;
		      }
		      pQlen++;
		      if(nreq->priority > 0) pQNlen++;
		      LOG_PACKET(nreq->fromto.sin_addr.s_addr, dpos, priority);
		      keep_looking++;  /* There may still be a duplicate */
		      break;
		  }
		  if(!treq->next) {
		      treq->next = nreq;
		      nreq->previous = treq;
		      pQlen++;
		      if(nreq->priority > 0) pQNlen++;
		      LOG_PACKET(nreq->fromto.sin_addr.s_addr,dpos+1,priority);
		      keep_looking = 0; /* Nothing more to check */
		      break;
		  }
		  treq = treq->next;
		  dpos++;
	        }
		/* If not added at end of queue, check behind packet for dup */
		if(keep_looking) {
		    while(treq) {
			if((treq->cid == nreq->cid) && (treq->cid != 0) && 
			   (bcmp(treq->fromto,nreq->fromto,sizeof(treq->fromto))==0)){
			    /* Found a dup */
			    pQlen--;
			    if(treq->priority > 0) pQNlen--;
			    plog(L_QUEUE_INFO,nreq->fromto.sin_addr.s_addr,NULL,"Duplicate replaced (removed at %d)", dpos, 0);
			    /* treq should never be first at this point */
			    treq->previous->next = treq->next;
			    if(treq->next) treq->next->previous = treq->previous;
			    prfree(treq);
			    break;
			}
			treq = treq->next;
			dpos++;
		    }
		}
	    }
	    else {
		pending = nreq;
		pQlen++;
		if(nreq->priority > 0) pQNlen++;
		LOG_PACKET(nreq->fromto.sin_addr.s_addr, dpos, priority);
	    }

	    /* Here we can optionally send a message telling the  */
	    /* client to back off.  How long should depend on the */
	    /* queue length and the mean service time             */
	    /* 15 min for now - the archie server is overloaded   */
#ifdef ARCHIE
	    transmit_wait(nreq,900); 
#endif ARCHIE
	}
	/* Otherwise add it to incomplete */
	else {
	    /* Note that at this point, no clients generate multi- */
	    /* packet requests.                                    */
	    plog(L_NET_ERR,from.sin_addr.s_addr,NULL,"Multi-packet request received - ignored",0);
	    goto check_for_more;
	}
	
	goto check_for_more;
    }

PREQ get_next_request()
    {
	PREQ	nextreq;
	int	tmp;

    tryagain:
	check_for_messages();
	/* return next message in queue */
	if(pending) {
	    nextreq = pending;
	    pending = pending->next;
	    if(pending) pending->previous = NULL;
	    nextreq->next = NULL;
	    pQlen--;
	    if(nextreq->priority > 0) pQNlen--;
	    running = nextreq;
	    return(nextreq);
	}

	/* if queue is empty, then wait till somethings comes */
	/* in, then go back to start                          */
	FD_ZERO(&readfds);
	FD_SET(srvport, &readfds);
	if(prvport != -1) FD_SET(prvport, &readfds); 
	tmp = select(srvport + 1, &readfds, (fd_set *)0, (fd_set *)0, NULL);
	goto tryagain;
    }


/*
 * transmit_wait - sends a message to the recipient indicating that
 * there may be a delay before the request can be processed.
 * The recipient should backoff and not attempt any retries
 * until the time in the message has elapsed.
 */
transmit_wait(req,timetowait)
    PREQ		req;
    int			timetowait;
    {
        PTEXT		r = ptalloc();
        int		sent;
	int		thisseq;
	short		cid = htons(req->cid);
	short		nseq = 0;
	short		zero = 0;
	short		backoff;

	/* Note that control information follows the null  */
	/* that terminates the text.  Since no text is     */
	/* being sent in this message, we must add a null  */
	/* at the start.                                   */
	     
#if !defined(DONTSUPPORTOLD) && !defined(REALLYNEW)
	/* For now we have to give this message a positive   */
	/* sequence number.  Once all clients have the new   */
	/* code, we will be able to assign a sequence number */
	/* of zero indicating that it is not to be returned  */
	/* to the application                                */
	if(req->cpkt) {
	    thisseq = (req->cpkt->seq)++;
	    nseq = htons((u_short) thisseq);
	}
	else {
	    thisseq = 1;
	    nseq = htons((u_short) thisseq);
	    req->cpkt = ptalloc();
	    req->cpkt->seq = 2;
	}
	/* We also have to include an old style command line */
	/* Eventually we can replace with a single null byte */
	sprintf(r->start,"MULTI-PACKET %d\n",thisseq);
	r->length = strlen(r->start) + 1;
#else
	*r->start = (char) 11;
	r->length = 1;
#endif

	backoff = htons((u_short) timetowait);

	bcopy(&cid,r->start+r->length,2);  	/* Connection ID     */
	bcopy(&nseq,r->start+r->length+2,2);	/* Packet number     */
	bcopy(&zero,r->start+r->length+4,2);	/* Total packets     */
	bcopy(&zero,r->start+r->length+6,2);	/* Received through  */
	bcopy(&backoff,r->start+r->length+8,2);	/* Backoff           */

	r->length += 10; 

	sent = sendto(((prvport != -1) ? prvport : srvport), r->start,
		      r->length, 0, &(req->fromto),S_AD_SZ);

	if(sent != r->length) {
	    plog(L_NET_ERR,req->fromto.sin_addr.s_addr,NULL,"Failed sending backoff packet (length %d) (errno %d)",r->length,errno,0);
	    ptfree(r); r = NULL;
	    return(REPLY_NOTSENT);
	}

	ptfree(r);
	r = NULL;

	return(PSUCCESS);
    }

/*
 * transmit - transmit takes a request block that has been filed in
 * with a messages that have not yet been returned, and it transmits them
 * to the client.  If the packet sequnce number is non-zero, then 
 * apporpirate sequencing information is added.  If the sequence
 * number is negative, then it assumes that this is the last message
 * to be returned and adds additional sequencing information indicating
 * the total number of packets.
 */
transmit(req)
    PREQ		req;
    {
	char 	buf[MAX_DIR_LINESIZE];
	int	sent;
	short	cid = htons(req->cid);
	short	zero = 0;
	short	nseq = 0;
	short	ntotal = 0;
#ifdef ARCHIE
	short	rcvdthru = 0;
	short	bkoff = 0;
#endif ARCHIE

	*buf = '\0';

	if(!req->cpkt) return(PFAILURE);

        if(req->cpkt->seq > 0) {
	    sprintf(buf,"MULTI-PACKET %d\n",req->cpkt->seq);
	    nseq = htons((u_short) req->cpkt->seq);
	}
	else if(req->cpkt->seq < 0) {
	    sprintf(buf, "MULTI-PACKET %d OF %d",
		    -req->cpkt->seq, -req->cpkt->seq);
	    nseq = htons((u_short) -req->cpkt->seq);
	    ntotal = htons((u_short) -req->cpkt->seq);
	}
	else {
	    nseq = htons((u_short) 1);
	    ntotal = htons((u_short) 1);
	}

#ifdef REALLYNEW
	/* Note that in the following, we don't have to make sure  */
	/* there is room for the header in the packet because we   */
	/* are the only one that moves back the start, and ptalloc */
	/* allocates space for us in all packets it creates        */
	
	/* If a single message and no connection ID to return, */
	/* then we can leave the control fields out            */
	if(req->cpkt->seq == 0) {
#ifdef SINGLETONBUGFIXED /* For now, use old format for singleton responses */

	    req->cpkt->start -= 1;
	    req->cpkt->length += 1;
	    *req->cpkt->start = (char) 1;
#endif
	}
	else {	    /* Fill in the control fields */
#ifdef ARCHIE
#ifdef DEBUG_RDGRAM
	    req->cpkt->start -= 13;
	    req->cpkt->length += 13;
	    *req->cpkt->start = (char) 13;
#else
	    req->cpkt->start -= 11;
	    req->cpkt->length += 11;
	    *req->cpkt->start = (char) 11;
#endif
#else
	    req->cpkt->start -= 7;
	    req->cpkt->length += 7;
	    *req->cpkt->start = (char) 7;
#endif 
	    bcopy(&cid,req->cpkt->start+1,2);     /* Conn ID */
	    bcopy(&nseq,req->cpkt->start+3,2);     /* Pkt no  */
	    bcopy(&ntotal,req->cpkt->start+5,2);   /* Total   */

#ifdef ARCHIE
	    /* If archie, set new backoff */
	    rcvdthru = htons((u_short) 0);
	    bcopy(&rcvdthru,req->cpkt->start+7,2); /* Recvd Through */
	    bkoff = htons((u_short) 5);
	    bcopy(&bkoff,req->cpkt->start+9,2);    /* New timeout */
#ifdef DEBUG_RDGRAM
	    /* Request ack on every fourth packet */
	    if((ntohs(nseq) % 4) == 0) *(req->cpkt->start+11) = 0x80;
	    else *(req->cpkt->start+11) = 0x00;
	    *(req->cpkt->start+12) = 0x00;
#endif
#endif ARCHIE
	}
	/* Make room for the trailing null */
	req->cpkt->length += 1;

#else /* Not really new */
#ifndef DONTSUPPORTOLD
	/* Copy the buffer and update length (count the trailing null) */
	strcat(req->cpkt->start + req->cpkt->length, buf);
	req->cpkt->length += strlen(buf) + 1;
#else
	/* Make room for the trailing null */
	req->cpkt->length += 1;
#endif

	/* If a single message and no connection ID to return, */
	/* then we can leave the control fields out            */
	if(req->cpkt->seq != 0) {
	   /* Fill in the control fields */
	   bcopy(&cid,req->cpkt->start+req->cpkt->length,2);     /* Conn ID */
	   bcopy(&nseq,req->cpkt->start+req->cpkt->length+2,2);   /* Pkt no  */
	   bcopy(&ntotal,req->cpkt->start+req->cpkt->length+4,2); /* Total   */
	   req->cpkt->length += 6;
	}
#endif

	/* Only send if packet not yet received */
	/* And send set a window of 75 packets  */
	if((abs(req->cpkt->seq) <= (req->rcvd_thru + 75)) &&
	   ((req->cpkt->seq <= 0) || (req->cpkt->seq > req->rcvd_thru))) {
	    sent = sendto(((prvport != -1) ? prvport : srvport), 
			  req->cpkt->start, req->cpkt->length, 0,
			  &(req->fromto), S_AD_SZ);
	    
	    if(sent != req->cpkt->length) {
		plog(L_NET_ERR,req->fromto.sin_addr.s_addr,NULL,"Failed sending packet (length %d) (errno %d)",req->cpkt->length,errno,0);
		if(req->cpkt->seq <= 0) {
		    if(running == req) running = NULL;
		    prfree(req);
		}
		else { /* Eventually add to req->trns, for now, free */
		    ptfree(req->cpkt);
		    req->cpkt = NULL;
		}
		return(REPLY_NOTSENT);
	    }
	}

	/* If the sequence number was negative, or 0 then get */
	/* rid of the request structure.  Eventually, we will */
	/* leave it around in case we have to retry but we    */
	/* can't tell if we need to do that until client      */
	/* start filing in the connection ID                  */
	if(req->cpkt->seq <= 0) {
	    if(running == req) running = NULL;
	    prfree(req);
	}
	else { /* Eventually add to req->trns, for now, free */
	    ptfree(req->cpkt);
	    req->cpkt = NULL;
	}
	return(PSUCCESS);
    }

static update_req_cfields(existing,newvalues)
    PREQ	existing;
    PREQ 	newvalues;
    {
	if(newvalues->rcvd_thru > existing->rcvd_thru)
	    existing->rcvd_thru = newvalues->rcvd_thru;
	return(TRUE);
    }

