/*
 * Copyright (c) 1991 by the University of Washington
 * Copyright (c) 1993 by the University of Southern California
 *
 * For copying and distribution information, please see the files
 * <uw-copyright.h> and <usc-copyr.h>.
 */

#include <uw-copyright.h>
#include <usc-copyr.h>

#include <strings.h>

#include <ardp.h>
#include <pfs.h>
#include <pserver.h>
#include <pprot.h>
#include <plog.h>
#include <pmachine.h>

#ifndef IPPORT_RESERVED
#define IPPORT_RESERVED 1024
#endif


/* used to initialize ACLs */
struct aclinit {
    int acetype;
    char *atype;
    char *rights;
    char *prin_1;                /* principals list */
    char *prin_2;                /* principals list */
    char *prin_3;                /* principals list */
    char *prin_4;                /* principals list */
    char *prin_5;                /* principals list */
    char *prin_6;                /* principals list */
    char *prin_7;                /* principals list */
    char *prin_8;                /* principals list */
    char *prin_9;                /* principals list */
    char *prin_10;                /* principals list */
    char *prin_11;                /* principals list */
    char *prin_12;                /* principals list */
    char *prin_13;                /* principals list */
    char *prin_14;                /* principals list */
    char *prin_15;                /* principals list */
    char *prin_16;                /* principals list */
    char *prin_17;                /* principals list */
    char *prin_18;                /* principals list */
    char *prin_19;                /* principals list */
    char *prin_20;                /* principals list */
    char *prin_21;                /* principals list */
    char *prin_22;                /* principals list */
    char *prin_23;                /* principals list */
    char *prin_24;                /* principals list */
    char *prin_25;                /* principals list */
    char *prin_26;                /* principals list */
};

static	ACL_ST default_iacl = {ACL_DEFAULT, NULL};
static	ACL_ST system_iacl = {ACL_SYSTEM, NULL};

static struct aclinit default_aclinit[] = DEFAULT_ACL;
static struct aclinit system_aclinit[] = SYSTEM_ACL;
static struct aclinit override_aclinit[] = OVERRIDE_ACL;
static struct aclinit maint_aclinit[] = MAINT_ACL;

static	int	initialized = 0;

static ACL	default_acl = NULL;
static ACL	system_acl = NULL;
static ACL	override_acl = NULL;
static ACL	nulldir_acl = NULL;
ACL             maint_acl = NULL;

static checkl_acl_r();
static check_asserted_principal();
#ifdef __STDC__
static ACL aclinit2acl(struct aclinit init[], int nelems);
static int checkl_acl_r(ACL acl, ACL dacl, RREQ req, char *op, int depth, int ld);
#endif

void
#ifdef __STDC__
initialize_defaults(void)
#else
initialize_defaults()
#endif
{
    initialized++;

    default_acl = aclinit2acl(default_aclinit, 
                              sizeof default_aclinit 
                              / sizeof default_aclinit[0]);
    system_acl = aclinit2acl(system_aclinit, 
                              sizeof system_aclinit 
                              / sizeof system_aclinit[0]);
    override_acl = aclinit2acl(override_aclinit, 
                              sizeof override_aclinit 
                              / sizeof override_aclinit[0]);
    maint_acl = aclinit2acl(maint_aclinit, 
                              sizeof maint_aclinit 
                              / sizeof maint_aclinit[0]);

    nulldir_acl = &default_iacl;
    default_iacl.next = &system_iacl;
    system_iacl.previous = &default_iacl;

}

static ACL aclentryinit2aclentry(struct aclinit *init);

/* constructs a real list of ACL entries out of an array of aclinit
   structures. */
static ACL 
aclinit2acl(struct aclinit init[], int nelems)
{

    ACL rethead = NULL;       /* head of list of AC entries being returned */ 
    int i;

    for (i = 0; i < nelems; ++i) {
        ACL tmp = aclentryinit2aclentry(&init[i]);
        APPEND_ITEM(tmp, rethead);
    }
    return rethead;
}


static ACL
aclentryinit2aclentry(struct aclinit *init)
{
    ACL retval;
    retval = acalloc();
    if (!retval) out_of_memory();
    retval->acetype = init->acetype;
    retval->atype = init->atype;
    retval->rights = init->rights;
    retval->principals = NULL;
    if (init->prin_1) 
        retval->principals = tkappend(init->prin_1, retval->principals);
    if (init->prin_2) 
        retval->principals = tkappend(init->prin_2, retval->principals);
    if (init->prin_3) 
        retval->principals = tkappend(init->prin_3, retval->principals);
    if (init->prin_4) 
        retval->principals = tkappend(init->prin_4, retval->principals);
    if (init->prin_5) 
        retval->principals = tkappend(init->prin_5, retval->principals);
    if (init->prin_6) 
        retval->principals = tkappend(init->prin_6, retval->principals);
    if (init->prin_7) 
        retval->principals = tkappend(init->prin_8, retval->principals);
    if (init->prin_9) 
        retval->principals = tkappend(init->prin_9, retval->principals);
    if (init->prin_10) 
        retval->principals = tkappend(init->prin_10, retval->principals);
    if (init->prin_11) 
        retval->principals = tkappend(init->prin_11, retval->principals);
    if (init->prin_12) 
        retval->principals = tkappend(init->prin_12, retval->principals);
    if (init->prin_13) 
        retval->principals = tkappend(init->prin_13, retval->principals);
    if (init->prin_14) 
        retval->principals = tkappend(init->prin_14, retval->principals);
    if (init->prin_15) 
        retval->principals = tkappend(init->prin_15, retval->principals);
    if (init->prin_16)
        retval->principals = tkappend(init->prin_16, retval->principals);
    if (init->prin_17) 
        retval->principals = tkappend(init->prin_17, retval->principals);
    if (init->prin_18) 
        retval->principals = tkappend(init->prin_18, retval->principals);
    if (init->prin_19) 
        retval->principals = tkappend(init->prin_19, retval->principals);
    if (init->prin_20) 
        retval->principals = tkappend(init->prin_20, retval->principals);
    if (init->prin_21) 
        retval->principals = tkappend(init->prin_21, retval->principals);
    if (init->prin_22) 
        retval->principals = tkappend(init->prin_22, retval->principals);
    if (init->prin_23) 
        retval->principals = tkappend(init->prin_23, retval->principals);
    if (init->prin_24) 
        retval->principals = tkappend(init->prin_24, retval->principals);
    if (init->prin_25) 
        retval->principals = tkappend(init->prin_25, retval->principals);
    if (init->prin_26) 
        retval->principals = tkappend(init->prin_26, retval->principals);
    return retval;
}

/*
 * check_acl - check access control list
 *
 *     CHECK_ACL checks an access control list to see if a particular
 *     user is authorized to perform a particular operation.  It returns
 *     a positive number if the operation is authorized, and zero if
 *     not authorized.  CHECK_ACL actually checks up to three access
 *     control lists.  First, the access control list associated with
 *     a link is checked (if specified).  If the operation would not
 *     be authorized, the directory ACL is checked to see if it overrides
 *     the individual entry.  If sill not authorized,  an override ACL is 
 *     checked to see if the operation is performed.
 *
 *     NOTE on negative rights:  Negative rights apply within
 *     a particular access control list only.  Thus, a negative
 *     entry in the link ACL can override other entries in the
 *     link ACL, but it will not prevent access if the user
 *     is authorized to perform the operation by the directory
 *     or override ACL's.
 */
check_acl(dacl,lacl,req,op)
    ACL		dacl, lacl;  /* Directory and link ACLs          */
    RREQ	req;	     /* Request; used for client identification */
    char	*op;	     /* Operation                        */
{
    int	answer = 0;

    if(!initialized) initialize_defaults();

    if(!dacl) dacl = nulldir_acl;

    /* Check link ACL or dir ACL as link if link ACL is NULL */
    if(!lacl) answer = checkl_acl_r(dacl,dacl,req,op,ACL_NESTING,2);
    else answer = checkl_acl_r(lacl,dacl,req,op,ACL_NESTING,0);

    if(answer) return(answer);

    /* Check to see if directory overrides */
    answer = checkl_acl_r(dacl,dacl,req,op,ACL_NESTING,1);

    if(answer) return(answer);

    /* Check to see if absolute override applies */
    return(checkl_acl_r(override_acl,override_acl,req,op,ACL_NESTING,2));

}


static checkl_acl_r(acl,dacl,req,op,depth,ld)
    ACL		acl;  	     /* Access control list              */
    ACL		dacl;  	     /* Directory access control list    */
    char	*op;	     /* Operation                        */
    RREQ	req;	     /* Client identification            */
    int		depth;	     /* Maximum nesting                  */
    int		ld;	     /* 0 = link, 1 = dir, 2 = both      */
{
    int	retval = 0;    /* Would this entry authorize op  */
    int	answer = 1;    /* How to answer if match         */

    if(depth == 0) return(NOT_AUTHORIZED);

    while(acl) {
        retval = check_permissions(op,acl->rights,ld);
        if(retval||((acl->rights==NULL)&&((acl->acetype == ACL_DEFAULT)
                                     ||(acl->acetype == ACL_SYSTEM)
                                     ||(acl->acetype == ACL_DIRECTORY)))) {
            if(retval == NEG_AUTHORIZED) answer = NOT_AUTHORIZED;
            else answer = AUTHORIZED;

            switch(acl->acetype) {

            case ACL_NONE:
                break;
            case ACL_DEFAULT:
                retval = checkl_acl_r(default_acl,dacl,req,op,depth-1,ld);
                if(retval) return(answer);
                break;
            case ACL_SYSTEM:
                retval = checkl_acl_r(system_acl,dacl,req,op,depth-1,ld);
                if(retval) return(answer);
                break;
            case ACL_OWNER:
                /* Check if user is the owner of the dirtectory */
#if 0
                /* We need to find the owner of the file/directory  */
                /* for which this ACL applies, but unfortunately we */
                /* don't even have the name of the file/directory   */
                ownacl->acetype = ACL_ASRTHOST;
                ownacl->atype = NULL;
                ownacl->rights = acl->rights;
                sprintf(ownaclst,"%s@%%.%%.%%.%%"/*,dirowner*/);
                ownacl->principals = ownaclst;
                ownacl->restrictions = acl->restrictions;
#endif
                break;
            case ACL_DIRECTORY:
                retval = checkl_acl_r(dacl,dacl,req,op,depth-1,ld);
                if(retval) return(answer);
                break;
            case ACL_ANY:
                return(answer);
            case ACL_AUTHENT: /* Only implemented for Kerberos */
                if (strequal(acl->atype, "KERBEROS")) {
                    PAUTH pap;      /* Iteration variable for PAUTH. */
                    TOKEN prin; /* principals */
                    /* Loop through all of the Kerberos authenticated
                       principals.  (This is upward-compatible with future
                       versions of Kerberos that will allow one to be
                       registered as multiple principals simultaneously.) 
                       */
                    for (pap = req->auth_info; pap; pap = pap->next) {
                        if (pap->ainfo_type == PFSA_KERBEROS) {
                            for (prin = pap->principals; prin; 
                                 prin = prin->next) {
                                if (member(prin->token, acl->principals))
                                    return answer;
                            }
                        }
                    }
                }
                break;
            case ACL_LGROUP: /* Not yet implemented */
                break;
            case ACL_GROUP: /* Not yet implemented */
                break;
            case ACL_TRSTHOST:  /* Check host and userid */
                if (!check_prvport(req))
                    break;
                /* DELIBERATE FALLTHROUGH */
            case ACL_ASRTHOST:  /* Check host and asserted userid */
            {
                PAUTH pap;
                TOKEN prin; /* principals */
                /* Loop through all of the asserted principals. */
                for (pap = req->auth_info; pap; pap = pap->next) {
                    if (pap->ainfo_type == PFSA_UNAUTHENTICATED) {
                        for (prin = pap->principals; prin; 
                             prin = prin->next) {
                            if (check_asserted_principal(prin->token,
                                                         acl->principals,
                                                         req->peer_addr))
                                return answer;
                        }
                    }
                }
            }
                break;
            default: /* Not implemented */
                break;
            }
        }
        acl = acl->next;
    }
    return(NOT_AUTHORIZED);
}

/* 
 * check_permissions - Check if operation is authorized
 *
 *           CHECK_PERMISIONS takes a string with letters representing
 *           the permissions required for the current opration, and
 *           a string listing operating authorized by the current ACL
 *           entry.  It returns a 1 if the operation would be
 *           authorized by the current entry, a 0 if not, and a -1
 *           if the ACL entry would have authorized the operation,
 *           but began with a "-" indicating negative authorization.
 *  
 *   ARGS:   op  - String with operations to be performed
 *           p   - Permissions from ACL entry
 *           ld  - Whther ACL entry is for directory or link (or both)
 *
 *   RETURNS: 1 if authorized
 *            0 if not authorized
 *           -1 if negative authorization
 *
 * Protections
 *
 * For directory:                For individual links:      
 *   A Administer                  a Administer           
 *   V View                        v View                 
 *   L List                        l List                 
 *   R Read                        r Read                 
 *   M Modify                      m Modify               
 *   D Delete                      d Delete               
 *   I Insert                               
 *   B Administer                    (does not override link)
 *   Y View                          (does not override link)
 *   > Add rights                  ] Add rights
 *   < Remove rights               [ Remove rights
 *   ) Add rights                    (does not override link)
 *   ( Remove rights                 (does not override link)
 *
 * Special maintenance protections (only make sense in the Maintenance ACL):
 *   S reStart the server
 *   T Terminate the server.
 *   U update User information (e.g., MOTD)
 *   
 *  When a small letter is associated with a directory, it is the default
 *  used for those links in the directory which do not otherwise specify
 *  protections.  The large letter for a directory indicates that the 
 *  right applies to ALL entries in the directory, regardless of the ACLs
 *  associated with the individual link.  
 *
 *  These rights apply to the directory and individual links.  The ability
 *  to read a link does not grant any rights to read the file that the
 *  link points to.  Instead, it grants the right to read the binding
 *  of the link.  The protection mechanisms for the objects themselves
 *  depend on the underlying access mechanism.
 *
 *  The Administer right is required to change the ACL.  "A" allows one
 *  to change the ACLs for the directory as a whole, as well as those 
 *  for individual links.  It does not, however, grant any rights to 
 *  the object pointed to by those links (e.g. it doesn't allow one
 *  to change the permissions on subdirectories.
 *
 *  List allows one to list an entry in a wildcarded search.  Read allows
 *  one to learn the binding of a link.  If one can read a link, but
 *  not list it, then it can only be seen when the user specifies the
 *  exact name of the link in a query.
 *
 *  Modify allows one to change the binding of a link.  It does not
 *  allow one to create a new link or delete an existing one.  Insert
 *  or delete access are necessary for that. 
 *
 *  View allows one to view the contents of the ACL.  Administer implies view.
 *  Add rights and remove rights, ><][)(, allow one to add or remove the
 *  other rights that are specified as part of the same ACL entry.  It is
 *  a limited form of administer.  The other rights included in the
 *  same ACL entry do not apply, but only restrict which rights may be
 *  added or removed.  The add or remove indicators must appear in the
 *  first one or two positions in the rights field, and can not themselves
 *  be added or removed unless they also appear later in the rights field.
 *
 *  If the permission string begins with a "-" and if the specified operation
 *  would otherwise be authorized, check_permissions returns -1 indicating 
 *  that an applicable negative right has been found, and that the operation
 *  should be denied even if subsequent ACL entries authorizing it are found.
 *  If an ACL entry preceding this one has already authorized the operation,
 *  the operation will be performed.
 *
 *  BUGS: For now, only the first character in ><][])( means add or
 *        delete the following rights, and all rights included in the
 *        entry including the first ><][)( may be added or deleted.
 *        Eventually, we will check the first two positions to see
 *        if adding and deleting is allowed, and the matching
 *        characters in those positions will be removed before 
 *        checking subsequent characters.
 */
check_permissions(op,p,ld)
    char	*op;	/* Operation                        */
    char	*p;	/* Permissions                      */
    int		ld;     /* 0 = link, 1 = directory, 2= both */
{
    int	polarity = 1; /* -1 = neg authorization */

    if(!p || !(*p)) return(NOT_AUTHORIZED);

    if(*p == '-') polarity = -1;

    if(index("><][)(",*p) && !index("><][",*op))
        return(NOT_AUTHORIZED);

    while(*op) {
        switch(*(op++)) {
        case 'a':
            if((ld != 1) && index(p,'a')) continue;
            if((ld != 0) && index(p,'A')) continue;
            else return(NOT_AUTHORIZED);
        case 'A': 
            if((ld != 0) && index(p,'A')) continue;
            if((ld != 0) && index(p,'B')) continue;
            else return(NOT_AUTHORIZED);

        case 'v':
            if((ld != 1) && index(p,'v')) continue;
            if((ld != 0) && index(p,'V')) continue;
            if((ld != 1) && index(p,'a')) continue;
            if((ld != 0) && index(p,'A')) continue;
            else return(NOT_AUTHORIZED);
        case 'V': 
            if((ld != 0) && index(p,'V')) continue;
            if((ld != 0) && index(p,'Y')) continue;
            if((ld != 0) && index(p,'A')) continue;
        if((ld != 0) && index(p,'B')) continue;
        else return(NOT_AUTHORIZED);

        case 'l':
            if((ld != 1) && index(p,'l')) continue;
        case 'L': /* and fall through */
            if((ld != 0) && index(p,'L')) continue;
            else return(NOT_AUTHORIZED);

        case 'r':
            if((ld != 1) && index(p,'r')) continue;
        case 'R': /* and fall through */
            if((ld != 0) && index(p,'R')) continue;
            else return(NOT_AUTHORIZED);

        case 'm':
            if((ld != 1) && index(p,'m')) continue;
        case 'M': /* and fall through */
            if((ld != 0) && index(p,'M')) continue;
            else return(NOT_AUTHORIZED);

        case 'd':
            if((ld != 1) && index(p,'d')) continue;
        case 'D': /* and fall through */
            if((ld != 0) && index(p,'D')) continue;
            else return(NOT_AUTHORIZED);

        case 'I':
            if((ld != 0) && index(p,'I')) continue;
            else return(NOT_AUTHORIZED);

        case ']':
            if((ld != 1) && index(p,']')) continue;
            if((ld != 0) && index(p,'>')) continue;
            else return(NOT_AUTHORIZED);
        case '>': 
            if((ld != 0) && index(p,'>')) continue;
            if((ld != 0) && index(p,')')) continue;
            else return(NOT_AUTHORIZED);

        case '[':
            if((ld != 1) && index(p,'[')) continue;
            if((ld != 0) && index(p,'<')) continue;
            else return(NOT_AUTHORIZED);
        case '<': 
            if((ld != 0) && index(p,'<')) continue;
            if((ld != 0) && index(p,'(')) continue;
            else return(NOT_AUTHORIZED);

            /* Maintenance operations. */
        case 'S':
            if (index(p, 'S')) continue;
            else return NOT_AUTHORIZED;
        case 'T':
            if (index(p, 'T')) continue;
            else return NOT_AUTHORIZED;
        case 'U':
            if (index(p, 'U')) continue;
            else return NOT_AUTHORIZED;
        default:
            return(NOT_AUTHORIZED);
        }
    }
    return(polarity);
}

#ifdef  NOTDEF
static char *inet_ntoa(a)
    struct in_addr a;
    {
	static char	astring[20];

#if BYTE_ORDER == BIG_ENDIAN
	sprintf(astring,"%d.%d.%d.%d",(a.s_addr >> 24) & 0xff,
		(a.s_addr >> 16) & 0xff,(a.s_addr >> 8) & 0xff, 
		a.s_addr & 0xff);
#else
	sprintf(astring,"%d.%d.%d.%d", a.s_addr & 0xff,
		(a.s_addr >> 8) & 0xff,(a.s_addr >> 16) & 0xff, 
		(a.s_addr >> 24) & 0xff);
#endif
	
	return(astring);
    }
#endif NOTDEF

static char *inet_def_local(s)
    char	*s;
{
    static char	adstring[50];
    static long	myaddr = 0;
    char		intstring[10];
    static long		o[4];
    char		*ads;
    long		*octet;

    if(!myaddr) {
        myaddr = myaddress();
#if BYTE_ORDER == BIG_ENDIAN
        o[0] = (myaddr >> 24) & 0xff;
        o[1] = (myaddr >> 16) & 0xff;
        o[2] = (myaddr >> 8) & 0xff;
        o[3] = myaddr & 0xff;
#else
        o[0] = myaddr & 0xff;
        o[1] = (myaddr >> 8) & 0xff;
        o[2] = (myaddr >> 16) & 0xff;
        o[3] = (myaddr >> 24) & 0xff;
#endif
    }

    octet = o;

    ads = adstring;
    while(*s && ((ads - adstring) < 40) ) {
        switch(*s) {
        case '.':
            octet++;
            *ads++ = *s;
            break;
        case '%':
            sprintf(intstring,"%d",*octet);
            bcopy(intstring,ads,strlen(intstring));
            ads += strlen(intstring);
            break;
        default:
            *ads++ = *s;
            break;
        }
        s++;
    }
    *ads++ = '\0';
    return(adstring);

}

/* Are we logged into a privileged port?  1 if yes, zero if no. */
int
check_prvport(req)
    RREQ req;
{
    return PEER_PORT(req) <= IPPORT_RESERVED;
}

/*
 * check_asserted_principal - check for membership in list of principals
 * and check optional host ID.  The principals have the format used by ASRTHOST
 * and TRSTHOST.
 */
static
check_asserted_principal(username,principals, hostaddr)
    char username[];
    struct in_addr hostaddr;
    TOKEN principals;
{
    for(; principals; principals = principals->next) {
        char	*host;
        char aclprin[MAX_DIR_LINESIZE]; /* principal specified in the ACL
                                           entry.  We copy it because we can't
                                           modify the principals structure,
                                           since the principals structure might
                                           contain strings that are in readonly
                                           memory. */ 
        strcpy(aclprin, principals->token);

        host = index(aclprin,'@');
        if(host == NULL) {
            if(wcmatch(username,aclprin)) return(TRUE);
            else continue;
        }
        *(host++) = '\0';

        if(!wcmatch(username,aclprin)) continue;

        if(index("%123456789.",*host)) { /* Host address */
            if(wcmatch(inet_ntoa(hostaddr),inet_def_local(host)))
                return(TRUE);
            continue;
        }
        else if(*host == '0') { /* Host address with leading 0 */
            if(wcmatch(inet_ntoa(hostaddr),inet_def_local(host+1)))
                return(TRUE);
            continue;
        }
        else { /* Host name - not implemented */
            continue;
        }
    }
    return(FALSE);
}

