/*
 * Copyright (c) 1989, 1990, 1991 by the University of Washington
 * Copyright (c) 1991, 1992, 1993 by the University of Southern California
 *
 * For copying and distribution information, please see the files
 * <uw-copyright.h> and <usc-copyr.h>.
 */

#include <uw-copyright.h>
#include <usc-copyr.h>

#ifndef PFS_H_INCLUDED
#define PFS_H_INCLUDED

#include <ardp.h>	/* Types defined in ARDP.H are used in the prototypes
			   of functions defined in this file. */
/*
 * PFS_RELEASE identifies the Prospero release.  PFS_SW_ID identifies 
 * the particular software and version of the software being used.  
 * Applications using Prospero must request a string to identify their
 * release.  Software IDs may be obtained from info-prospero@isi.edu.
 */
#define		PFS_RELEASE	"Beta.5.1 ftp-only"
#define		PFS_SW_ID       "B51-f"

/* General Definitions */

#define		MAX_VPATH	1024	 /* Max length of virtual pathname   */

/* Note: In doubly linked lists of the structures used for prospero, the  */
/* ->previous element of the head of the list is the tail of the list and */
/* the ->next element of the tail of the list is NULL.  This allows fast  */
/* addition at the tail of the list.                                      */

/* Definition of structure containing information on virtual link            */
/* Some members of this structure were recently renamed. The  #defined       */
/* aliases exist only for backwards compatability                            */
/* Uncomment the aliases if you need this backwards compatability.           */
#if 0
#define    type		target	    /* compat (will go away)                 */
#define    nametype	hsonametype /* compat (will go away)                 */
#define    filename     hsoname     /* compat                                */
#endif

struct vlink {
    int			dontfree;   /* Flag: don't free this link            */
    char		*name;	    /* Component of pathname: NAME-COMPONENT */
    char		linktype;   /* L = Link, U = Union, N= Native, 
                                       I = Invisible */
    int			expanded;   /* Has a union link been expanded        */
    char		*target;    /* Type of the target of the link *vl1   */
    struct filter       *filters;   /* Filters to be applied when link used  */
    struct vlink	*replicas;  /* Replicas (* see comment below)        */
    char		*hosttype;  /* Type of hostname		             */
    char		*host;	    /* Files physical location	             */
    char		*hsonametype;/* Type of hsoname                      */
    char		*hsoname;   /* Host specific object name             */
    long		version;    /* Version # of dest: OBJECT-VERSION     */
    long		f_magic_no; /* File's magic number.  *vl2            */
    struct pattrib      *oid;       /* Head of a list of object identifiers  */
    struct acl		*acl;       /* ACL for link		             */
    long		dest_exp;   /* Expiration for dest of link. 0=unset  */
    struct pattrib	*lattrib;   /* Attributes associated w/ link         */
    struct pfile	*f_info;    /* Client uses 2 assoc info w/ refed obj */
    struct vlink	*previous;  /* Previous elt in linked list           */
    struct vlink	*next;	    /* Next element in linked list           */
};

typedef struct vlink *VLINK;
typedef struct vlink VLINK_ST;

/* *vl1 Target of link:  For links in a directory that don't point   */
/*      to objects, could be SYMBOLIC or EXTERNAL. For a forwarding  */
/*      pointer it is FP.  Links to objects will have a target field */
/*      indicating the object's BASE-TYPE attribute.  When stored on */
/*      a vlink structure, it may have exactly one of the following  */
/*      forms: "OBJECT", "FILE", "DIRECTORY", "DIRECTORY+FILE".      */
/*      There is no guarantee that the type of the target has not    */
/*      changed.  Thus, this field is the cached value of the        */
/*      object's base type.                                          */

/* *vl2 Note that vlink->replicas is not really a list of replicas   */
/*      of the object.  Instead, it is a list of the objects         */
/*      returned during name resolution that share the same name as  */
/*      the current object.  Such an object should only be           */
/*      considered a replica if it also shares the same non-zero     */
/*      object identifiers.                                          */

/* *vl3 For f_magic_no, zero means unset.  Also, This is the value   */
/*      of the ID REMOTE type.  It must be updated at the same time  */
/*      that the appropriate attribute hanging off of the OID member */
/*      is. This speeds up checks.   Do not write code that depends  */
/*      on this member being present; it will go away Very Soon Now. */

/* The EXPANDED member of the VLINK structure is normally set to either
   FALSE or TRUE or FAILED (all three of which are defined at the end of
   PFS.H). In addition, dsrdir(), ul_insert(), and list_name() work together to
   test and set the following two possible values for it.  These values are
   only used on the server.
 */

/* This virtual link points to the initial directory given dsrdir() */
#define ULINK_PLACEHOLDER 2 
/* This virtual link should not be expanded because the directory it refers to
   has already been expanded.  However, it should be returned in a listing if
   the LEXPAND option was not specified to the LIST command. */
#define ULINK_DONT_EXPAND 3


/* definition for virtual directory structure */
struct vdir {
    int		   version;     /* Version no of directory instance (curr 0) */
    int		   inc_native;	/* Include the native directory              */
    long	   magic_no;	/* Magic number of this directory            */
    struct acl	   *dacl;       /* Default acl for links in dir              */
    struct pfile   *f_info;	/* Directory file info                       */
    struct vlink   *links;	/* The directory entries	             */
    struct vlink   *ulinks;	/* The entries for union links               */
    struct dqstate *dqs;	/* State needed by pending query             */
    int		   status;	/* Status of query                           */
};

typedef struct vdir *VDIR;
typedef struct vdir VDIR_ST;

/* Initialize directory */
#define vdir_init(dir) do { \
			dir->version = 0;     dir->inc_native = 0;   \
                        dir->magic_no = 0;    dir->dacl = NULL;      \
                        dir->f_info = NULL;   dir->links = NULL;     \
    			dir->ulinks = NULL;   dir->dqs = NULL;     \
			dir->status = DQ_INACTIVE; } while (0)

#define vdir_copy(d1,d2) do { \
    			 d2->version = d1->version; \
                         d2->inc_native = d1->inc_native; \
                         d2->magic_no = d1->magic_no; \
                         d2->dacl = d1->dacl; \
    			 d2->f_info = d1->f_info; \
                         d2->links = d1->links; \
                         d2->ulinks = d1->ulinks; \
			 d2->status = d1->status; \
			 dir->dqs = NULL; } while (0)
                         
#define vdir_freelinks(dir) do {                        \
        aclfree(dir->dacl); dir->dacl = NULL;           \
        pflfree(dir->f_info); dir->f_info = NULL;       \
        vllfree(dir->links); dir->links = NULL;         \
        vllfree(dir->ulinks); dir->ulinks = NULL;       \
     } while(0)

/* Values of ->inc_native in vdir structure            */
/* Note: VDIN_NONATIVE has the same value as FALSE     */
/*       VDIN_INCLNATIVE has the same value as TRUE    */
#define VDIN_INCLREAL	-1   /* Include native files, but not . and ..       */
#define VDIN_NONATIVE	 0   /* Do not include files from native directory   */
#define VDIN_INCLNATIVE	 1   /* Include files from native directory          */
#define VDIN_ONLYREAL    2   /* All entries are from native dir (no . and ..)*/
#define VDIN_PSEUDO      3   /* Directory is not real                        */

/* Definition of structure containing filter information             */
struct filter {
    char 		*name;  /* For PREDEFINED filters only  *f1  */
    struct vlink 	*link;  /* For LOADABLE filters only    *f1  */
    short		type;   /* dir, hierarchy, obj, upd     *f2  */
    short   execution_location; /* FIL_SERVER or FIL_CLIENT     *f3  */
    short 	   pre_or_post; /* FIL_PRE or FIL_POST          *f4  */
    short		applied;/* Flag - set means has been applied */
    struct token	*args;  /* Filter arguments                  */
    struct filter	*next;  /* Next filter in linked list        */
    struct filter    *previous; /* Previous filter in linked list    */
};
typedef struct filter *FILTER;
typedef struct filter FILTER_ST;

/* *f1: Exactly one of the following two will be set.  This also lets us */
/*      distinguish between the two basic types of filters, PREDEFINED   */
/*      and LOADABLE.                                                    */ 

/* *f2: Values for the type field */
#define FIL_DIRECTORY       1
#define FIL_HIERARCHY       2
#define FIL_OBJECT          3
#define FIL_UPDATE          4    

/* *f3: Values for the execution_location field */
#define FIL_SERVER          1
#define FIL_CLIENT          2

/* *f4: Values for the pre_or_post field */
#define FIL_PRE             1
#define FIL_POST            2
#define FIL_ALREADY         3   /* already expanded! */

/* This is the structure for attribute values.  */
union avalue {
    struct filter       *filter;   /* A filter                            */
    struct vlink	*link;	   /* A link		                  */
    struct token        *sequence; /* Most general type (subsumes string) */
};

/* definition of structure containing attribute information */
struct pattrib {
    char		precedence;  /* Precedence for link attribute *a1  */
    char                nature;      /* FIELD, APPLICATION or INTRINSIC *a2 */
    char		avtype;	     /* Type of attribute value (sm int) *a3 */
    char		*aname;	     /* Name of the attribute              */
    union avalue	value;	     /* Attribute Value                    */
    struct pattrib	*previous;   /* Previous element in linked list    */
    struct pattrib	*next;	     /* Next element in linked list        */
};

typedef struct pattrib *PATTRIB;
typedef struct pattrib PATTRIB_ST;

/* *a1: values for the precedence field. */
#define         ATR_PREC_UNKNOWN 'X'   /* Default setting. */
#define 	ATR_PREC_OBJECT  'O'   /* Authoritative answer for object */
#define 	ATR_PREC_LINK    'L'   /* Authoritative answer for link   */
#define 	ATR_PREC_CACHED  'C'   /* Object info cached w/ link      */
#define 	ATR_PREC_REPLACE 'R'   /* From link (replaces O)          */
#define 	ATR_PREC_ADD     'A'   /* From link (additional value)    */

/* *a2: values for the nature field. */
#define         ATR_NATURE_UNKNOWN        '\0'
#define         ATR_NATURE_FIELD          'F'
#define         ATR_NATURE_APPLICATION    'A'
#define         ATR_NATURE_INTRINSIC      'I'

/* *a3: Values for the AVTYPE field. */
#define         ATR_UNKNOWN         0  /* error return value and init value */
#define         ATR_FILTER          1
#define         ATR_LINK            2
#define         ATR_SEQUENCE        3


/* Definition of structure containing information on a specific file */
/* **** Incomplete ****                                              */
struct pfile {
    int			version;	  /* Version of local finfo format   */
    long		f_magic_no;	  /* Magic number of current file    */
    long		exp;		  /* Expiration date of timeout      */
    long		ttl;		  /* Time to live after reference    */
    long		last_ref;	  /* Time of last reference          */
    struct vlink	*forward;	  /* List of forwarding pointers     */
    struct vlink	*backlinks;	  /* Partial list of back links      */
    struct pattrib	*attributes;	  /* List of file attributes         */
    struct pfile	*previous;        /* Previous element in linked list */
    struct pfile	*next;		  /* Next element in linked list     */
};

typedef struct pfile *PFILE;
typedef struct pfile PFILE_ST;

/* Definition of structure contining an access control list entry */
struct acl {
    int			acetype;     /* Access Contol Entry type            */
    char		*atype;      /* Authent type if acetyep=ACL_AUTHENT */
    char		*rights;     /* Rights                              */
    struct token        *principals; /* Authorized principals               */
    struct restrict     *restrictions; /* Restrictions on use               */
    struct acl		*previous;     /* Previous elt in linked list       */
    struct acl		*next;	       /* Next element in linked list       */
};
typedef struct acl *ACL;
typedef struct acl ACL_ST;

/* Values for acetype field */
#define ACL_NONE		0         /* Nobody authorized by ths entry */
#define ACL_DEFAULT		1         /* System default                 */
#define ACL_SYSTEM		2         /* System administrator           */
#define ACL_OWNER               3         /* Directory owner                */
#define ACL_DIRECTORY           4         /* Same as directory              */
#define ACL_ANY                 5         /* Any user                       */
#define ACL_AUTHENT             6         /* Authenticated principal        */
#define ACL_LGROUP              7         /* Local group                    */
#define ACL_GROUP               8         /* External group                 */
#define ACL_ASRTHOST            10        /* Check host and asserted userid */
#define ACL_TRSTHOST            11        /* ASRTHOST from privileged port  */


/* Definition of structure contining access restrictions */
/* for future extensions                                 */
struct restrict {
    struct acl		*previous;        /* Previous elt in linked list    */
    struct acl		*next;		  /* Next element in linked list    */
};

/* Definition of "linked list of strings" structure       */
/* Used for ACL principalnames, for values of attributes, */
/* and other uses.                                        */
struct token {
    char *token;            /* string value of the token    */
    struct token *next;     /* next element in linked list  */
    struct token *previous; /* previous element in list     */
};

typedef struct token *TOKEN;
typedef struct token TOKEN_ST;

/* Authorization/authentication information */
struct pfs_auth_info {
    int		ainfo_type;     /* Type of auth info                 *ai1 */
    char	*authenticator; /* For storing auth data                  */
    TOKEN	principals;     /* Principals authenticated by above *ai2 */
    struct pfs_auth_info  *next;/* Next element in linked list            */
};

typedef struct pfs_auth_info *PAUTH;
typedef struct pfs_auth_info PAUTH_ST;

/* ai1: This filed is not necessarily used on the server side         */
/* ai2: At the moment only one principal is specied per authenticator */

/* Values for ainfo_type */
#define PFSA_UNAUTHENTICATED    1
#define PFSA_KERBEROS           2

#define AUTHENTICATOR_SZ     1000 /*  A useful size for holding a Kerberos or

/********************/

#ifndef _TYPES_
#include <sys/types.h>
#endif _TYPES_
#include <stdio.h>              /* Needed for def of FILE, for args to
				   qfprintf.  Also gives us NULL and size_t, 
				   both of which we need. */ 

#if 0
#include <stddef.h>		/* guarantees us NULL and size_t, which ANSI
				   already gives us in <stdio.h> */
#endif
#include <stdarg.h>		/* needed for variable arglist function
				   prototypes (provides va_list type) */


/* Unfortunately, some buggy compilers (in this case, gcc 2.1 under
   VAX BSD 4.3) get upset if <netinet/in.h> is included twice. */
#if !defined(IP_OPTIONS)
#include <netinet/in.h> 
#endif
/* Status for directory queries - the following or a Prospero error code */
#define      DQ_INACTIVE   0   /* Query not presently active             */
#define      DQ_COMPLETE  -1   /* Directory query completed              */
#define      DQ_ACTIVE    -2   /* Directory query in progress            */

/* State information for directory query.  Used only in p_get_vdir(). */
struct dqstate {
    VLINK	dl;       /* Link to queried directory                 */
    char	*comp; /* Name of components to return       */
    int         stfree_comp;    /* should comp be stfreed() when dqstate is? */
    TOKEN	*acomp;	  /* Additional components to resolve          */
    long	flags;	  /* Flags		                       */
    int		fwdcnt;	  /* Number of fwd pointers to chase           */
    int		newlinks; /* Number of new links added this query      */
    int		mcomp;	  /* Multiple components to be resolved (flag) */
    int		unresp;   /* We have unresolved components to process  */
    VLINK	clnk;     /* Current link being filled in              */
    VLINK 	cexp; 	  /* The current ulink being expanded          */
    VLINK	pulnk;    /* Prev union link (insert new one after it) */
    RREQ	preq;	  /* Pending request structure            */
};


#ifdef KERBEROS
#define KERBEROS_SERVICE "prospero" /* The name of the Kerberos Service to
                                       request tickets for, if KERBEROS is
                                       defined in <psite.h> */

#endif

/* Definitions for rd_vlink and rd_vdir */
#define		SYMLINK_NESTING 10       /* Max nesting depth for sym links */

/* Definition fo check_acl */
#define		ACL_NESTING     10       /* Max depth for ACL group nesting */

/* Flags for mk_vdir */
#define	     MKVD_LPRIV     1   /* Minimize privs for creator in new ACL    */

/* Flags for p_get_vdir */
#define	     GVD_UNION       0  /* Do not expand union links 		     */
#define      GVD_EXPAND    0x1  /* Expand union links locally		     */
#define	     GVD_LREMEXP   0x3  /* Request remote expansion of local links   */
#define	     GVD_REMEXP    0x7  /* Request remote expansion of all links     */
#define	     GVD_VERIFY    0x8  /* Only verify args are for a directory      */
#define      GVD_FIND	  0x10  /* Stop expanding when match is found        */
#define	     GVD_ATTRIB   0x20  /* Request attributes from remote server     */
#define	     GVD_NOSORT	  0x40  /* Do not sort links when adding to dir      */
#define      GVD_MOTD     0x80  /* Request motd string from server           */
#define      GVD_MOTD_O  0x100  /* Request motd string from server           */
#define	     GVD_ASYNC   0x200  /* Fill in the directory asynchronously      */

#define	     GVD_EXPMASK  0x7   /* Mask to check expansion flags             */

/* Flags for rd_vdir */
#define	     RVD_UNION      GVD_UNION
#define	     RVD_EXPAND     GVD_EXPAND 
#define	     RVD_LREMEXP    GVD_LREMEXP
#define	     RVD_REMEXP     GVD_REMEXP
#define	     RVD_DFILE_ONLY GVD_VERIFY /* Only return ptr to dir file        */
#define      RVD_FIND       GVD_FIND   
#define      RVD_ATTRIB     GVD_ATTRIB
#define	     RVD_NOSORT	    GVD_NOSORT
#define	     RVD_NOCACHE    128

/* Flags for add_vlink */
#define	     AVL_UNION      1   /* Link is a union link                      */
#define	     AVL_INVISIBLE  2   /* Link is an invisible link.  (Not 
                                   compatible with AVL_UNION)                */

/* Flags for vl_insert */
#define	     VLI_NOCONFLICT 0   /* Do not insert links w/ conflicting names  */
#define      VLI_ALLOW_CONF 1   /* Allow links with conflicting names        */
#define	     VLI_NOSORT     2   /* Allow conflicts and don't sort            */

/* Flags for mapname */
#define      MAP_READWRITE  0   /* Named file to be read and written         */
#define	     MAP_READONLY   1   /* Named file to be read only                */

/* Flags for edit_object_info/pset_at()/pset_linkat() */
#define EOI_ADD 0x01
#define EOI_DELETE 0x02
#define EOI_DELETE_ALL 0x03
#define EOI_REPLACE 0x04

/* Flags for edit_acl */
#define	     EACL_NOSYSTEM   0x01
#define      EACL_NOSELF     0x02
#define      EACL_DEFAULT    0x08
#define      EACL_SET        0x0C
#define      EACL_INSERT     0x14
#define      EACL_DELETE     0x10
#define      EACL_ADD        0x1C
#define      EACL_SUBTRACT   0x18
#define      EACL_LINK       0x00
#define      EACL_DIRECTORY  0x20
#define      EACL_OBJECT     0x60
#define      EACL_INCLUDE    0x40

#define      EACL_OP    (EACL_DEFAULT|EACL_SET|EACL_INSERT|\
			 EACL_DELETE|EACL_ADD|EACL_SUBTRACT)

#define      EACL_OTYPE (EACL_LINK|EACL_DIRECTORY|EACL_OBJECT|EACL_INCLUDE)

#ifdef SERVER_SUPPORT_V1
/* Flags for modify_acl */
#define	     MACL_NOSYSTEM   0x01
#define      MACL_NOSELF     0x02
#define      MACL_DEFAULT    0x08
#define      MACL_SET        0x0C
#define      MACL_INSERT     0x14
#define      MACL_DELETE     0x10
#define      MACL_ADD        0x1C
#define      MACL_SUBTRACT   0x18
#define      MACL_LINK       0x00
#define      MACL_DIRECTORY  0x20
#define      MACL_OBJECT     0x60
#define      MACL_INCLUDE    0x40

#define      MACL_OP    (MACL_DEFAULT|MACL_SET|MACL_INSERT|\
			 MACL_DELETE|MACL_ADD|MACL_SUBTRACT)

#define      MACL_OTYPE (MACL_LINK|MACL_DIRECTORY|MACL_OBJECT|MACL_INCLUDE)

#endif

/* Access methods returned by pget_am() and specified as arguments to it. */
#define P_AM_ERROR			0
#define P_AM_FTP			1
#define P_AM_AFTP			2  /* Anonymous FTP  */
#define P_AM_NFS			4
#define P_AM_KNFS			8  /* Kerberized NFS */
#define P_AM_AFS		       16
#define P_AM_GOPHER                    32
#define P_AM_LOCAL                     64 /* files that can be copied in local
                                             filesystem  */
#define P_AM_RCP                      128 /* Berkeley RCP protocol.  Not yet
                                             implemented. */

/* Flags for dsrdir */
#define DSRD_ATTRIBUTES		       0x1 /* Fill in attributes for links */

/* Return codes */

#define		PSUCCESS	0
#define		PFAILURE	255

/* How many bytes allocated for a string? */
#define stsize(s)   ((s) == NULL ? 0 : (((int *)(s))[-1]))

/* FUNCTION PROTOTYPES */
#ifdef __cplusplus
extern "C" {
#endif

/* Procedures in libpfs.a                                           */
/* These are procedures that call p__start_req() and ardp_send() to */
/* send Prospero  Protocol requests to the server                   */
extern int pget_am(VLINK,TOKEN *,int);
int add_vlink(char direct[], char lname[], VLINK, int);
extern VDIR apply_filters(VDIR, FILTER, VLINK, int);
#ifdef KERBEROS
int binencode(char*,size_t,char*,size_t);
int bindecode(char*,char*,size_t);
int check_krb_auth(char auth[], struct sockaddr_in client, 
                   char ret_client_name[]);
#endif
int del_vlink(char vpath[], int);
ACL get_acl(VLINK,char lname[],int);
#define get_vdir p_get_vdir     /* backwards compatability */
int p_get_vdir(VLINK,char components[], VDIR, long flags, TOKEN *acomp); 
int mk_vdir(char vpath[],int);
int modify_acl(VLINK,char lname[],ACL,int);
PATTRIB pget_at(VLINK,char atname[]);
int pset_at(VLINK,int,PATTRIB);
int pset_linkat(VLINK,char lname[],int, PATTRIB);
int rd_vdir(char dirarg[], char comparg[] ,VDIR,long);
VLINK rd_slink(char path[]);
VLINK rd_vlink(char path[]);
int update_link(char dirname[], char components[]);

/* Internal Prospero utility functions, not intended for use by applications
   programmers but still with external scope (thanks to the accursed C linkage
   model) */
void            initialize_defaults(void);
extern char *pget_wdhost(void), *pget_wdfile(void), *pget_wd(void);
extern char *pget_hdhost(void), *pget_hdfile(void), *pget_hd(void);
char    *pget_rdhost(void),    *pget_rdfile(void); 
char *pget_dhost(void), *pget_dfile(void), *pget_vsname(void), *nlsindex();
extern TOKEN p__slashpath2tkl(char *nextcomp);
extern void p__tkl_back_2slashpath(TOKEN nextcomp_tkl, char *nextcomp);
extern char *p__tkl_2slashpath(TOKEN nextcomp_tkl);
void p__print_shellstring(int which);
int p__get_shellflag(void);


/* UNIX utilities. */
long myaddress(void);
char *myhostname(void);
char *month_sname(int month_number);
int mkdirs(char path[]); 
char *sindex(), *strtok(), *nxtline();
int stequal(char *s1, char *s2); /* test for string equality, accepting either
                             as NULL  */
char *readheader();
char            *timetoasn(time_t time);
time_t		asntotime(char timestring[]);


/* Utility functions that handle Prospero quoting. */
/* Return PSUCCESS or PFAILURE. */
int qfprintf(FILE *outf, char fmt[], ...);
int vqfprintf(FILE *outf, char fmt[], va_list ap);

/* Returns a count of the # of matches. */
int qsscanf(char *input, char *format, ...);

/* Returns the # of bytes it wanted to write. */
size_t qsprintf(char *buf, size_t buflen, char fmt[], ...);
size_t vqsprintf(char *buf, size_t buflen, char fmt[], va_list);
/* Returns a pointer to the position of c within s, respecting quoting. */
char *qindex(char *s, char c);
char *qrindex(char *s, char c); /* like rindex(), but quoting. */

/* Handle User Level Name quoting */
char *p_uln_index(char *s, char c);
char *p_uln_rindex(char *s, char c);

/* Networking utility functions. */
#ifdef ARDP_H_INCLUDED
/* Used by clients to construct request packets.. */
extern RREQ p__start_req(char server_hostname[]);
extern int p__add_req(RREQ request, char format[], ...);
extern int pv__add_req(RREQ request, char fmt[], va_list ap);

RREQ		get_next_request();
#endif /* ARDP_H_INCLUDED */

/* Utility functions handling Prospero memory allocation and data structure 
   manipulation. */
ACL             acalloc();

extern PATTRIB atcopy(PATTRIB at), atlcopy(PATTRIB at);
extern PATTRIB  atalloc(void);
extern int atfree(PATTRIB);
extern int atlfree(PATTRIB);
int             equal_attributes(PATTRIB, PATTRIB);

extern FILTER flalloc(void), flcopy(FILTER source, int r);
int flfree(FILTER);            /* These would be void, but I make them int for
                                   consistency with the other freeing
                                   functions.  XXX */ 
int             equal_filters(FILTER f1, FILTER f2);
int fllfree(FILTER);

PAUTH           paalloc(void);
int             pafree(PAUTH), palfree(PAUTH);

PFILE		pfalloc();

extern char *stcopyr(char *, char *); 
extern stfree(void *s);
extern char *stalloc(int size);
extern char *stcopy(char *s);

TOKEN           qtokenize(char *);
TOKEN           tkalloc(char *), tkcopy(TOKEN);
TOKEN           tkappend(char newstr[], TOKEN toklist);
int             equal_sequences(TOKEN,TOKEN);
int             tkfree(TOKEN), tklfree(TOKEN), member(char *, TOKEN);
int             length(TOKEN s);
char            *elt(TOKEN s, int nth);

extern VLINK	        vl_delete();
extern VLINK            vlalloc();
extern VLINK            vlcopy(VLINK v, int r);
/* End of function prototypes. */
#ifdef __cplusplus
};
#endif  

/* Miscellaneous useful definitions */
#ifndef TRUE
#define TRUE		1
#define FALSE		0
#endif

#define AUTHORIZED      1
#define NOT_AUTHORIZED  0
#define NEG_AUTHORIZED  -1

#define FAILED		-1
#define ZERO(p)		bzero((char *)(p), sizeof(*(p)))
/* Systems without BZERO can use the following:
#define bzero(s, size) memset((s), 0 , (size))
*/

/* A minor C extension that handles a thorn in my side.  --swa */
#define strequal(s1, s2) (strcmp(s1, s2) == 0)
#define strnequal(s1, s2, n) (strncmp(s1, s2, n) == 0)

/* Internal error handling routines used by the pfs code; formerly in */
/* internal_error.h.  These include a replacement for the assert()    */
/* macro, and an interface for internal error handling, better        */
/* documented in internal_error.c                                     */

#ifndef NDEBUG
  #define assert(expr) \
    do { \
        if (!(expr)) \
            internal_error("assertion violated: " #expr); \
    } while(0)
#else /* NDEBUG */
#define assert(expr) do {;} while(0)
#endif /* NDEBUG */



  #define internal_error(msg) do { \
      write(2, "Internal error in file " __FILE__ ": ", \
            sizeof "Internal error in file " __FILE__ ": " -1); \
      write(2, msg, strlen(msg)); \
      write(2, "\n", 1);        \
      if (internal_error_handler)   \
          (*internal_error_handler)(__FILE__, __LINE__, msg);   \
      else  { \
          fprintf(stderr, "line of error: %d\n", __LINE__); \
          abort(); \
      } \
  } while(0)

/* This function may be set to handle internal errors.  Dirsrv handles them in
   this way, by logging to plog.  We make it int instead of void, because
   older versions of the PCC (Portable C Compiler) cannot handle pointers to
   void functions. */
extern int (*internal_error_handler)(char file[], int line, char mesg[]);

/* function form of internal_error. */
int finternal_error(char file[], int line, char mesg[]);
#define interr_buffer_full() \
    finternal_error(__FILE__, __LINE__, "A buffer filled up"); 
#define out_of_memory() \
    finternal_error(__FILE__, __LINE__, "Out of Memory"); 

#endif /* PFS_H_INCLUDED */
