/*
 * Copyright (c) 1989, 1990, 1991 by the University of Washington
 * Copyright (c) 1993 by the University of Southern California
 *
 * For copying and distribution information, please see the files
 * <uw-copyright.h> and <usc-copyr.h>
 */

#include <uw-copyright.h>
#include <usc-copyr.h>

#include <stdio.h>

#include <pfs.h>
#include <perrno.h>

int	perrno;
int	pfs_debug = 0;

PATTRIB		pget_at();

main(argc,argv)
    int		argc;
    char	*argv[];
{
    VDIR_ST		dir_st;
    VDIR		dir= &dir_st;
    VLINK		l;
    VLINK		replica;
    int		flags = 0;
    int		verbose = 0;
    int		dmagic = 0;
    int		show_failed = 0;
    int		show_invisible = 0;
    int		show_at = 0;
    int		replica_flag = 0;
    int		pretty_flag = 0;
    int		conflict_flag = 0;
    int             dirflag = 0;
    int		tmp;
    char		*at_name = NULL; /* will point to allocated memory */
    char		*progname = argv[0];

    flags = RVD_LREMEXP;

    vdir_init(dir);

    argc--;argv++;

    while (argc > 0 && **argv == '-') {
        switch ((*argv)[1]) {

        case 'a':  /* Show attributes */
            show_at = 1; 
            verbose = 1;
            tmp = qsscanf(argv[0],"-a%&s", &at_name);
            if(tmp < 1) at_name = stcopy("#ALL");
            break;

        case 'A':  /* Show only link attributes */
            show_at = 2; /* Show only link attributes */
            verbose = 1;
            tmp = qsscanf(argv[0],"-a%&s", &at_name);
            if(tmp < 1) at_name = stcopy("#ALL");
            break;

        case 'c':  /* Show conflicting links */
            conflict_flag++;
            break;

        case 'd':           /* If target specified, treat it as a link even
                               if it's a directory. */
            dirflag++;
            break;
        case 'D':  /* Debug level */
            pfs_debug = 1; /* Default debug level */
            sscanf(argv[0],"-D%d",&pfs_debug);
            break;

        case 'f':  /* Show failed links */
            show_failed++;
            break;

        case 'i':  /* Show invisible links */
            show_invisible++;
            break;

        case 'm':  /* Display magic number */
            dmagic++;
            break;

        case 'P':               /* Pretty flag -- truncate if too much. */
            pretty_flag++;
            break;
            
        case 'r':  /* Show replicas */
            replica_flag++;
            break;

        case 'u':  /* Do not expand union links */
            flags = RVD_UNION;
            break;

        case 'v':  /* Display link on more than one line (verbose) */
            verbose = 1;
            break;

        default: {
        usage:
            fprintf(stderr,
                    "Usage: vls [-a,-A,-c,-r,-u,-v,-d] \
[<file or directory name>]\n");
            exit(1);
        }
        }
        argc--, argv++;
    }

    if(show_at) flags |= RVD_ATTRIB;
    if (dirflag) flags |= RVD_DFILE_ONLY;

    if (argc > 1) goto usage;

    ardp_abort_on_int();

    tmp = rd_vdir((argc == 1 ? argv[0] : ""), (char *) NULL, dir,flags);

    if(tmp && (tmp != DIRSRV_NOT_DIRECTORY)) {
        fprintf(stderr,"%s",progname);
        perrmesg(" failed: ", tmp, NULL);
        exit(1);
    }

    if(pwarn) pwarnmesg("WARNING: ",0,NULL);

    l = dir->links;

    while(l) {
        if (show_invisible || l->linktype != 'I')
            display_link(l,dmagic,verbose,pretty_flag,show_at,at_name);
        replica = l->replicas;
        while(replica) {
            if((replica_flag && (l->f_magic_no != 0) && 
                (l->f_magic_no == replica->f_magic_no)) || conflict_flag)
                if (show_invisible || l->linktype != 'I')
                    display_link(replica,
                         (dmagic || (l->f_magic_no != replica->f_magic_no)),
                        verbose, pretty_flag, show_at,at_name);
            replica = replica->next;
        }
        l = l->next;
    }

    l = dir->ulinks;

    if((tmp != DIRSRV_NOT_DIRECTORY) || show_failed) {
        while(l) {
            if((l->expanded == FALSE) || (l->expanded == FAILED))
            display_link(l,dmagic,verbose, pretty_flag, show_at,at_name);
            l = l->next;
        }
    }

    vllfree(dir->links);
    exit(0);

}


display_link(VLINK l, int dmagic,int verbose, int pretty_flag, int show_at,
            char *at_name)
{
/* check if a string is a null pointer.  If it is, return an empty string. */ 
#define chknl(s) ((s) ? (s) : "")
    FILTER      fil;
    PATTRIB 	ap;
    char		linkname[MAX_VPATH];
    static  display_fil(FILTER fil);

    if(l->linktype == '-') return;

    if(dmagic) sprintf(linkname,"%s#%d",chknl(l->name),l->f_magic_no);
    else strcpy(linkname,chknl(l->name));

    if (verbose) {
        printf("\n      Name: %s\n",chknl(l->name));
        printf("    Target: %s\n",chknl(l->target));
        printf("  LinkType: %s\n",
               ((l->linktype == 'U') ? "Union" : 
                (l->linktype == 'I' ? "Invisible" : "Standard")));
        if (l->hosttype && !strequal(l->hosttype, "") && 
            !strequal(l->hosttype, "INTERNET-D"))
            printf("  HostType: %s\n",chknl(l->hosttype));
        printf("      Host: %s\n",chknl(l->host));
        if (l->hsonametype && !strequal(l->hsonametype, "") && 
            !strequal(l->hsonametype, "ASCII"))
            printf("  NameType: %s\n",chknl(l->hsonametype));
        printf("   Hsoname: %s\n",chknl(l->hsoname));
        if(l->version) printf("   Version: %d\n",l->version);
        if(dmagic || l->f_magic_no) 
            printf("  RemoteID: %d\n",l->f_magic_no);

        if(show_at) {
            /* If only link attributes, then use l->lattrib */
            /* otherwise get attribues for object.          */
            if(show_at == 2) ap = l->lattrib;
            else {
                /* Eventually we will merge the link attributes with  */
                /* the object attribues based on the precedence field,*/
                /* but for now, if attributes are retruend for the    */
                /* object we use them, otherwisseise we use those     */
                /* associated with the link                           */
                ap = pget_at(l,at_name);
                if(!ap) ap = l->lattrib;
            }
            if(ap) {
                printf("Attributes:\n\n");
                while(ap) {
                    printf("%15s: ",ap->aname);
                    if(ap->avtype == ATR_SEQUENCE) {
                        TOKEN tk;
                        for (tk = ap->value.sequence; tk; tk = tk->next)
                            if (*(tk->token))
                                printf("%s ", tk->token);
                            else
                                fputs("'' ", stdout);
                        putchar('\n');
                    } else if(ap->avtype == ATR_LINK)
                        printf("%s %s %s %d %d\n",
                               ap->value.link->name,ap->value.link->host,
                               ap->value.link->hsoname,
                               ap->value.link->version, 
                               ap->value.link->f_magic_no);
                    else if (ap->avtype == ATR_FILTER)
                        display_fil(ap->value.filter);
                    else printf("<unknown-type %d>\n",ap->avtype);
                    ap = ap->next;
                }
                if(l->filters) printf("\n");
            }
            atlfree(ap);
        }

        if(l->filters) {
            printf("   Filters:\n");
            for(fil = l->filters; fil; fil = fil->next)
                display_fil(fil);
        }
        putchar('\n');
    }
    /* First character is determined by l->target and l->linktype.
       U for a union link (always to a DIRECTORY or DIRECTORY+FILE)
       I for an invisible link (only shown if -i flag specified)
            (could be to a FILE, DIRECTORY, or DIRECTORY+FILE).
       If a normal link, determined by l->target:
          blank (' ') if a normal link to a FILE.
          S for SYMBOLIC
          E for EXTERNAL
          N for NULL (returned if inadequate permissions),
          D for DIRECTORY
          B (Both) for DIRECTORY+FILE
          O for OBJECT 
       
       Second character is F if expanding the union link failed. */

    else {
        printf((pretty_flag ? "%c%c %-20.20s %c%-15.15s %-38.38s\n"
                : "%c%c %-20s %c%-15s %s\n"),
               ((l->linktype != 'L') ? l->linktype :
                (l->target && strequal(l->target, "DIRECTORY+FILE")) ? 'B' :
                (l->target && !strequal(l->target, "FILE")) ? *l->target 
                : ' '),
               (l->expanded ? 'F' : ' '),
               chknl(linkname),
               (l->filters ? '*' : ' '),
               chknl(l->host),chknl(l->hsoname));
    }
}

static
display_fil(fil)
    FILTER fil;
{
    if (fil->name)
        printf("Predefined Filter: %s\n", fil->name);
    else
        printf("Link Filter: %s %s %s %ld %ld\n", chknl(fil->link->name),
               chknl(fil->link->host), 
               chknl(fil->link->hsoname), fil->link->version, fil->link->f_magic_no);
        switch(fil->type) {
    case FIL_DIRECTORY:
        printf(" DIRECTORY");
        break;
    case FIL_HIERARCHY:
        printf(" HIERARCHY");
        break;
    case FIL_OBJECT:
        printf(" OBJECT");
        break;
    case FIL_UPDATE:
        printf(" UPDATE");
        break;
    default:
        internal_error("unknown fil->type value.");
    }
    switch(fil->execution_location) {
    case FIL_SERVER:
        printf(" SERVER");
        break;
    case FIL_CLIENT:
        printf(" CLIENT");
        break;
    default:
        internal_error("unknown fil->execution_location");
    }
    switch(fil->pre_or_post) {
    case FIL_PRE:
        printf(" PRE");
        break;
    case FIL_POST:
        printf(" POST");
        break;
    default:
        internal_error("unknown fil->pre_or_post");
    }
    if (fil->args) {
        TOKEN tk;
        printf(" ARGS");
        for (tk = fil->args; tk; tk = tk->next)
            printf(" %s", tk->token);
    }
    putchar('\n');
}


