/*
 * Copyright (c) 1989, 1990 by the University of Washington
 * Copyright (c) 1993 by the University of Southern California
 *
 * For copying and distribution information, please see the files
 * <uw-copyright.h> and <usc-copyr.h>.
 */

#include <uw-copyright.h>
#include <stdio.h>

#include <pfs.h>

static PATTRIB	lfree = NULL;
int		pattrib_count = 0;
int		pattrib_max = 0;

/*
 * atalloc - allocate and initialize vlink structure
 *
 *    ATALLOC returns a pointer to an initialized structure of type
 *    PATTRIB.  If it is unable to allocate such a structure, it
 *    returns NULL.
 */
PATTRIB
atalloc(void)
{
    PATTRIB	at;
    if(lfree) {
        at = lfree;
        lfree = lfree->next;
    } else {
        at = (PATTRIB) malloc(sizeof(PATTRIB_ST));
        if (!at) out_of_memory();
        pattrib_max++;
    }

    pattrib_count++;

#ifdef ALLOCATOR_CONSISTENCY_CHECK
    at->consistency = INUSE_PATTERN;
#endif
    /* Initialize and fill in default values */
    at->precedence = ATR_PREC_OBJECT;
    at->nature =      ATR_NATURE_UNKNOWN;
    at->avtype = ATR_UNKNOWN;
    at->aname = NULL;
    at->value.sequence = NULL;
    at->previous = NULL;
    at->next = NULL;
    return(at);
}

/*
 * atfree - free a PATTRIB structure
 *
 *    ATFREE takes a pointer to a PATTRRIB structure and adds it to
 *    the free list for later reuse.
 */
atfree(PATTRIB at)
{
#ifdef ALLOCATOR_CONSISTENCY_CHECK
    assert(at->consistency == INUSE_PATTERN);
    at->consistency = FREE_PATTERN;
#endif
    if(at->aname) stfree(at->aname);

    switch(at->avtype) {
    case ATR_UNKNOWN:
        break;
    case ATR_SEQUENCE:
        if (at->value.sequence)
            tklfree(at->value.sequence);
        break;
    case ATR_FILTER:
        if(at->value.filter)
            flfree(at->value.filter);
        break;
    case ATR_LINK:
        if (at->value.link)
            vlfree(at->value.link);
        break;
    default:
        internal_error("Illegal avtype");
    }

    at->next = lfree;
    at->previous = NULL;
    lfree = at;
    pattrib_count--;
}

/*
 * atlfree - free a PATTRIB structure
 *
 *    ATLFREE takes a pointer to a PATTRIB structure frees it and any linked
 *    PATTRIB structures.  It is used to free an entrie list of PATTRIB
 *    structures.
 */
atlfree(PATTRIB at)
{
    PATTRIB	nxt;

    while(at != NULL) {
        nxt = at->next;
        atfree(at);
        at = nxt;
    }
}



PATTRIB
atcopy(PATTRIB at)
{
    PATTRIB newat = atalloc();

#ifdef ALLOCATOR_CONSISTENCY_CHECK
    assert(at->consistency == INUSE_PATTERN);
#endif
    newat->precedence = at->precedence;
    newat->nature = at->nature;
    newat->avtype = at->avtype;
    newat->aname = stcopyr(at->aname, newat->aname);
    newat->value = at->value;
    /* leave previous and next unset. */
    return newat;
}


PATTRIB
atlcopy(PATTRIB atl)
{
    PATTRIB retval = NULL;

    for ( ; atl; atl = atl->next) {
        PATTRIB newat = atcopy(atl);
        APPEND_ITEM(newat, retval);
    }

    return retval;
}
