/*
 * Copyright (c) 1989, 1990, 1991 by the University of Washington
 * Copyright (c) 1992, 1993 by the University of Southern California
 *
 * For copying and distribution information, please see the files
 * <uw-copyright.h> and <usc-copyr.h>
 */

#include <uw-copyright.h>
#include <usc-copyr.h>

#include <sys/param.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/dir.h>
#include <stdio.h>
#include <sgtty.h>
#include <strings.h>
#include <pwd.h>
#include <grp.h>

#define FILE_VNO            5   /* version # of the file format. */

#include <ardp.h>
#include <pfs.h>
#include <plog.h>
#include <pprot.h>
#include <perrno.h>
#include <pmachine.h>
#include <pparse.h>

extern char	shadow[];
extern char	dirshadow[];
extern char	dircont[];
extern char	aftpdir[];

PATTRIB	atalloc();
VLINK check_fwd();

/*
 * dsrfinfo - Read file info from disk
 *
 *   DSRFINFO is used by the directory server to read file information from
 *   disk.  It takes the host specific part of the system level name of the 
 *   file about which information is to be read, and a pointer to the file
 *   structure which it fills in.  It returns 0 or -1 on success, 0 if a file
 *   and -1 if a directory.  On error, it logs the bad data format to the
 *   prospero logfile, and attempts to continue.
 *   It may also return DSRFINFO_FORWARDED.
 */
dsrfinfo(nm,magic,fi)
    char	*nm;
    long	magic;
    PFILE	fi;
{
    FILE 		*pfs_fi = NULL;
    char                *pfs_fi_name = NULL;
    char		name[MAXPATHLEN];
    char		shadow_fname[MAXPATHLEN];
    char		dirshadow_fname[MAXPATHLEN];

    VLINK		cur_fp; /* current forwarding poiner */
    VLINK		lp; /* Temporary link pointer */
    int		tmp;
    char		*ls;  /* Last slash */

    struct passwd   *ownpw;
    struct group    *owngr;
    struct tm	*mt;
    char   		mtime_str[20];
    char   		mode_str[15];

    PATTRIB	at;
    PATTRIB	ap; /* Temp attribute pointer */

    struct stat	file_stat_dat;
    struct stat	*file_stat = &file_stat_dat;

    int		retval = PSUCCESS;
    INPUT_ST        in_st;
    INPUT           in = &in_st;

    /* Expand special file names */
    if((*nm != '/') && *aftpdir && (strncmp(nm,"AFTP",4) == 0)) {
	strcpy(name,aftpdir);
	strcat(name,nm+4);
    }
    else strcpy(name,nm);

    fi->version = -1;       /* sentinel value in case not found. */
    fi->f_magic_no = 0;
    fi->exp = 0;
    fi->ttl = 0;
    fi->last_ref = 0;
    fi->forward = NULL;
    fi->backlinks = NULL;
    fi->attributes = NULL;
    fi->previous = NULL;
    fi->next = NULL;

startover:

    /* Determine name of shadow file */
    strcpy(shadow_fname,shadow);
    strcat(shadow_fname,name);
    sprintf(dirshadow_fname,"%s/%s",shadow_fname,dirshadow);

    /* NOTE: A temporary inefficient shadow file format is*/
    /* in use.  For this reason, the code supporting it   */
    /* is also interim code, and does not do any checking */
    /* to make sure that the file actually follows        */
    /* the format.  Thus, a munged file will result       */
    /* in unpredictable results.			      */

    /* Read the contents of the shadow file */
    /* First find out if it is a directory  */
    if(stat(shadow_fname,file_stat) == 0) {
        if(file_stat->st_mode & S_IFDIR)
            pfs_fi = fopen(pfs_fi_name = dirshadow_fname,"r");
        else pfs_fi = fopen(pfs_fi_name = shadow_fname,"r");
    }

    if(pfs_fi != NULL) {
        char *line, *next_word;
        if(tmp = wholefiletoin(pfs_fi, in))
            return tmp;
        while(in_nextline(in)) {
            in_line(in, &line, &next_word);
            switch(*line) {
            case 'V': /* Version Number */
                tmp = sscanf(line,"VERSION %d",&(fi->version));
                if(tmp != 1) {
                    plog(L_DATA_FRM_ERR,NOREQ,"Bad file info format %s: %s",
                         shadow_fname,line,0);
                }
                if (fi->version != FILE_VNO) {
                    plog(L_DATA_FRM_ERR, NOREQ, "Bad file info format %s: \
unknown version %d, expected %d", shadow_fname, fi->version, FILE_VNO);
                    return PFAILURE;
                }
                break;
            case 'M': /* Magic Number */
                tmp = sscanf(line,"MAGIC-NUMBER %d",&(fi->f_magic_no));
                if(tmp != 1) {
                    fi->f_magic_no = 0;
                    plog(L_DATA_FRM_ERR,NOREQ,"Bad file info format %s: %s",
                         shadow_fname,line);
                }
                break;
            case 'E': /* Expiration Date */
                break;
            case 'T': /* Time to live */
                break;
            case 'L': /* Time of last reference */
                break;
            case 'F': /* Forwarding Pointer */ 
                /* A forwarding pointer has its NAME member set to the old
                   hosname of the object (now being forwarded).  the NAME
                   member may terminate in a *, indicating a wildcarded match.
                   The HOST is the new host of the object and the HSONAME is
                   its new HSONAME.  A trailing * in the HSONAME is replaced
                   by whatever matched the trailing * in the NAME member 
                   of the object being forwarded.  Yes, this means that it is
                   difficult to forward an object whose real HSONAME ends in
                   '*'.  We do not currently create such objects, although we
                   might.  */
                if (strnequal(line, "FORWARD", 7)
                    && in_link(in, line, next_word, 0, &cur_fp, 
                               (TOKEN *) NULL) == PSUCCESS) { 
                    APPEND_ITEM(cur_fp, fi->forward);
                } else {
                    plog(L_DATA_FRM_ERR,NOREQ,"Bad file info format %s: %s",
                         shadow_fname,line);
                }
                break;

            case 'B': /* Back Link */
                break;
            case 'A': /* Attribute or ACL*/ 
		if (strnequal(line, "ACL", 3)) {
		    if(in_ge1_acl(in, line, next_word, &fi->oacl)) {
			plog(L_DATA_FRM_ERR,NOREQ,"Bad file info format %s: %s",
			     shadow_fname,line);
			return PFAILURE;
		    }
		}
		else if (!strnequal(line, "ATTRIBUTE", 9) 
			 || in_ge1_atrs(in, line, next_word, &fi->attributes))
                    plog(L_DATA_FRM_ERR,NOREQ,"Bad file info format %s: %s",
                         shadow_fname,line);
                break;
            }
        }
        fclose(pfs_fi);

        /* Explicitly setting the magic # of an object to a negative number
           indicates that 'this object is not really here; it's just
           a placeholder for forwarding pointers'.  */
        /* Comparing the magic #s is a fast operation; faster than scanning
           though a list of forwarding pointers.  Therefore, if the magic #s
           match, all is well. */
        /* Check_above sets the return value to DSRFINFO_FORWARDED to indicate
           that we're searching up the UNIX directory hierarchy for a matching
           FORWARD message */
        /* Determine if the file has been forwarded, and if so return, */
        /* indicating that fact                                        */
        if((fi->f_magic_no && magic && (fi->f_magic_no != magic)) ||
           (fi->f_magic_no < 0) || (retval == DSRFINFO_FORWARDED)) {
            if(check_fwd(fi->forward,nm,magic))
                return(DSRFINFO_FORWARDED);
            else goto check_above;
        }
    }

    /* Fill in attributes from the real file */

    if((retval == PSUCCESS) && (stat(name,file_stat) == 0)) {
        static char *atval = NULL;
        static size_t atvalsiz = 0;
        int n;

        /*       off_t   st_size;	  /* total size	of file	    */
        at = atalloc();
        at->aname = stcopy("SIZE");
        at->avtype = ATR_SEQUENCE;
        at->nature = ATR_NATURE_INTRINSIC;
    again:
        n = qsprintf(atval, atvalsiz, "%d bytes", file_stat->st_size);
        if (n > atvalsiz) {
            stfree(atval);
            atval = stalloc(atvalsiz = n);
            if (atval == NULL) out_of_memory();
            goto again;
        }
        /* Double STCOPY here; need new interface to tkalloc(). */
        at->value.sequence = tkalloc(atval);
        APPEND_ITEM(at, fi->attributes);
        /*       short   st_uid;	  /* user-id of	owner       */
        ownpw = getpwuid(file_stat->st_uid);
        if(ownpw) {
            at = atalloc();
            at->aname = stcopy("NATIVE-OWNER");
            at->avtype = ATR_SEQUENCE;
            at->nature = ATR_NATURE_INTRINSIC;
            at->value.sequence = tkalloc(ownpw->pw_name);
            APPEND_ITEM(at, fi->attributes);
        }


        /*       short   st_gid;	  /* group-id of owner      */
        owngr = getgrgid(file_stat->st_gid);
        if(owngr) {
            at = atalloc();
            at->aname = stcopy("NATIVE-GROUP");
            at->avtype = ATR_SEQUENCE;
            at->nature = ATR_NATURE_INTRINSIC;
            at->value.sequence = tkalloc(owngr->gr_name);
            APPEND_ITEM(at, fi->attributes);
        }

        /*       time_t  st_atime;    /* file last access time  */

        /*       time_t  st_mtime;    /* file last	modify time */
        /* Should use timetoasn(), when it's written. */
        if(file_stat->st_mtime) {
            at = atalloc();
            at->aname = stcopy("LAST-MODIFIED");
            at->avtype = ATR_SEQUENCE;
            at->nature = ATR_NATURE_INTRINSIC;
            at->value.sequence = tkalloc(timetoasn(file_stat->st_mtime));
            APPEND_ITEM(at, fi->attributes);
        }

        /*       u_short st_mode;	  /* protection	            */
        at = atalloc();
        at->aname = stcopy("UNIX-MODES");
        at->avtype = ATR_SEQUENCE;
        at->nature = ATR_NATURE_INTRINSIC;

        strcpy(mode_str,"----------");
        if((file_stat->st_mode & S_IFLNK) == S_IFLNK) mode_str[0] = 'l';
        if(file_stat->st_mode & S_IREAD) mode_str[1] = 'r';
        if(file_stat->st_mode & S_IWRITE) mode_str[2] = 'w';
        if(file_stat->st_mode & S_IEXEC) mode_str[3] = 'x';
        if(file_stat->st_mode & S_ISUID) mode_str[3] = 's';
        if(file_stat->st_mode & (S_IREAD>>3)) mode_str[4] = 'r';
        if(file_stat->st_mode & (S_IWRITE>>3)) mode_str[5] = 'w';
        if(file_stat->st_mode & (S_IEXEC>>3)) mode_str[6] = 'x';
        if(file_stat->st_mode & S_ISGID) mode_str[6] = 's';
        if(file_stat->st_mode & (S_IREAD>>6)) mode_str[7] = 'r';
        if(file_stat->st_mode & (S_IWRITE>>6)) mode_str[8] = 'w';
        if(file_stat->st_mode & (S_IEXEC>>6)) mode_str[9] = 'x';
        if(file_stat->st_mode & S_IFDIR) {
            mode_str[0] = 'd';
            if(retval == PSUCCESS) retval = -1;
        }

        at->value.sequence = tkalloc(mode_str);
        APPEND_ITEM(at, fi->attributes);
    }
    /* If no native file AND no PFS file info, then look for forwarding */
    else if(fi->version < 0) goto check_above;

    return(retval);

    /* Check above looks for forwarding pointers in directories   */
    /* above the named file.  It is only reached when a file has  */
    /* moved, but a forwarding address does not exist in the      */
    /* corresponding file info.                                   */

 check_above:

    retval = DSRFINFO_FORWARDED;

    ls = rindex(name,'/');
    /* If we have used up all components, return failure */
    if((ls == 0) || (ls == name)) return(PFAILURE);

    *ls = '\0';

    goto startover;
}



/* check_fwd takes a list of forwarding pointers, checks to see */
/* if any apply to the object with the specified magic number   */
/* and returns the appropiate link, or none if not forwarded    */
/* If the match is a widarded match, the returned link will be  */
/* modified so that the matched wildcard is replaced by the     */
/* text that matched it                                         */
/*                                                              */
/* BUGS only a trailing * wildcard is allowed                   */
VLINK 
check_fwd(fl,name,magic)
    VLINK	fl;
    char	*name;
    int		magic;
{
    static char	*ffname = NULL;
    static size_t ffnamesiz = 0;
    char	*sp; /* Star pointer */

    while(fl) {
        if(((magic == 0) || (fl->f_magic_no == 0) ||
            (fl->f_magic_no == magic)) && (wcmatch(name,fl->name))) {

            sp = index(fl->hsoname,'*');
            if(sp) *sp = '\0';

            sp = index(fl->name,'*');
            if(sp) {
                int n;

                sp = name + (sp - fl->name);
                fl->name = stcopyr(name,fl->name);
            again:
                n = qsprintf(ffname, ffnamesiz, "%s%s",fl->hsoname,sp);
                if (n > ffnamesiz) {
                    stfree(ffname);
                    ffname = stalloc(ffnamesiz = n);
                    if (ffname == NULL) out_of_memory();
                    goto again;             /* the test will fail the 2nd time.
                                               */ 
                }
                fl->hsoname = stcopyr(ffname,fl->hsoname);
            }

            return(fl);
        }
        fl = fl->next;
    }
    return(NULL);
}    
    


