/*
 * Copyright (c) 1991       by the University of Washington
 * Copyright (c) 1992, 1993 by the University of Southern California
 *
 * For copying and distribution information, please see the files
 * <uw-copyright.h> and <usc-copyr.h>.
 *
 * Written  by bcn 1991     as check_for_messages in rdgram.c (Prospero)
 * Modified by bcn 1/93     modularized and incorporated into new ardp library
 */

#include <uw-copyright.h>
#include <usc-copyr.h>

#include <netdb.h>
#include <stdio.h>
#include <sys/param.h>
#include <sys/socket.h>
#ifdef AIX
#include <sys/select.h>
#endif

#include <ardp.h>
#include <pserver.h>
#include <plog.h>
#include <pprot.h>
#include <perrno.h>
#include <pmachine.h>

#define LOG_PACKET(REQ,QP) \
    if((REQ)->priority || (pQlen > 4)) \
      if(pQNlen && (REQ)->priority) \
         plog(L_QUEUE_INFO, REQ, "Queued: %d of %d (%d) - Priority %d", \
              QP, pQlen, pQNlen, (REQ)->priority, 0); \
      else if(pQNlen) \
         plog(L_QUEUE_INFO, REQ, "Queued: %d of %d (%d)", \
              QP, pQlen, pQNlen, 0); \
      else if((REQ)->priority) \
         plog(L_QUEUE_INFO, REQ, "Queued: %d of %d - Priority %d", \
               QP, pQlen, (REQ)->priority, 0); \
      else if(QP != pQlen) \
         plog(L_QUEUE_INFO, REQ, "Queued: %d of %d", QP, pQlen, 0); \
      else plog(L_QUEUE_INFO, REQ, "Queued: %d", pQlen, 0); 

RREQ		ardp_pendingQ = NOREQ;   /* Pending requests                */
int		pQlen = 0;               /* Length of pending queue         */
int		pQNlen = 0;              /* Number of niced requests        */

static RREQ	ardp_partialQ = NOREQ;	 /* Incomplete requests             */
int		ptQlen = 0;      	 /* Length of incomplete queue      */
int		ptQmaxlen = 20;		 /* Max length of incomplete queue  */
RREQ		ardp_runQ = NOREQ;       /* Requests currently in progress  */
RREQ		ardp_doneQ = NOREQ;      /* Processed requests              */
int		dQlen = 0;      	 /* Length of reply queue           */
int		dQmaxlen = 20;		 /* Max length of reply queue       */

static struct timeval  zerotime = {0, 0}; /* Used by select                 */

#define max(x,y) (x > y ? x : y)

extern int (*ardp_pri_func)();            /* Function to compare priorities */
extern int ardp_pri_override;             /* If 1, overide value in request */

/* C library error facility */
extern int	errno;
extern int      sys_nerr;
extern char *sys_errlist[];
#define errmesg (errno < sys_nerr ? sys_errlist[errno] : \
                 "(Unprintable error code)")

extern int		ardp_srvport;
extern int		ardp_prvport;

static ardp_update_cfields();

struct sockaddr_in s_in = {AF_INET};

ardp_accept()
{
    struct sockaddr_in 	from;
    int			fromlen;
    int			n = 0;
    PTEXT		pkt;
    PTEXT		tpkt; /* Temporary pkt pointer */
    unsigned char       flags1,flags2;
    unsigned short	pid;
    int			keep_looking; /* Keep looking for dup       */
    int			qpos; /* Position of new req in queue       */
    int			dpos; /* Position of dupe in queue          */
    RREQ		creq; /* Current request                    */
    RREQ		treq; /* Temporary request pointer          */
    RREQ		nreq; /* New request pointer                */
    RREQ		areq = NOREQ; /* Request needing ack        */
    int			hdr_len;
    char		*ctlptr;
    short		stmp;
    int			tmp;
    int			check_for_ack = 1; 
    fd_set		readfds;	               /* Used for select */
    long		now;		      /* Time - used for retries  */
    long		rr_time = 0; /* Time last retrans from done queue */
    
 check_for_more:
    
    now = time(NULL);

    /* Check both the prived and unprived ports if necessary */
    FD_ZERO(&readfds);
    FD_SET(ardp_srvport, &readfds);
    if(ardp_prvport != -1) FD_SET(ardp_prvport, &readfds); 
    tmp = select(max(ardp_srvport,ardp_prvport) + 1,
		 &readfds,(fd_set *)0,(fd_set *)0,&zerotime);
    
    if(tmp == 0) {
	if(areq) ardp_acknowledge(areq); areq = NOREQ;
	return(PSUCCESS);
    }
    if(tmp < 0) return(PFAILURE);
    
    creq = ardp_rqalloc();
    pkt = ardp_ptalloc();
    
    /* There is a message waiting, add it to the queue */
    
    fromlen = sizeof(from);
    if((ardp_prvport >= 0) && FD_ISSET(ardp_prvport,&readfds))
	n = recvfrom(ardp_prvport, pkt->dat, ARDP_PTXT_LEN_R, 0, 
		     (struct sockaddr *) &from, &fromlen);
    else n = recvfrom(ardp_srvport, pkt->dat, ARDP_PTXT_LEN_R, 0, 
		      (struct sockaddr *) &from, &fromlen);
    if (n <= 0) {
	plog(L_NET_ERR,NOREQ,"Bad recvfrom n = %d errno = %d %s",
	     n, errno, errmesg, 0);
	ardp_rqfree(creq); ardp_ptfree(pkt);
	return(PFAILURE);
    }
    
    bcopy(&from,&(creq->peer),sizeof(creq->peer));
    creq->cid = 0; creq->ardp_version = 0;
    creq->rcvd_time.tv_sec = now;
    creq->prcvd_thru = 0;
    
    pkt->start = pkt->dat;
    pkt->length = n;
    *(pkt->start + pkt->length) = '\0';
    pkt->mbz = 0; /* force zeros to catch runaway strings     */
    
    pkt->seq = 1;
    if((hdr_len = (unsigned char) *(pkt->start)) < 32) {
	ctlptr = pkt->start + 1;
	pkt->seq = 0;
	if(hdr_len >= 3) { 	/* Connection ID */
	    bcopy(ctlptr,&stmp,2);
	    if(stmp) creq->cid = ntohs(stmp);
	    ctlptr += 2;
	}
	if(hdr_len >= 5) {	/* Packet number */
	    bcopy(ctlptr,&stmp,2);
	    pkt->seq = ntohs(stmp);
	}
	else { /* No packet number specified, so this is the only one */
	    pkt->seq = 1;
	    creq->rcvd_tot = 1;
	}
	ctlptr += 2;
	if(hdr_len >= 7) {	    /* Total number of packets */
	    bcopy(ctlptr,&stmp,2);  /* 0 means don't know      */
	    if(stmp) creq->rcvd_tot = ntohs(stmp);
	}
	ctlptr += 2;
	if(hdr_len >= 9) {	/* Receievd through */
	    bcopy(ctlptr,&stmp,2);  /* 0 means don't know      */
	    if(stmp) {
		creq->prcvd_thru = ntohs(stmp);
	    }
	}
	ctlptr += 2;
	if(hdr_len >= 11) {	/* Backoff */
	    /* Not supported by server */
	}
	ctlptr += 2;
	
	flags1 = flags2 = 0;
	if(hdr_len >= 12) flags1 = *ctlptr++; /* Flags */
	if(hdr_len >= 13) flags2 = *ctlptr++; /* Flags */
	
	if(flags2 == 1) { /* Cancel request */
	    treq = ardp_pendingQ;
	    while(treq) {
		if((treq->cid == creq->cid) && 
		   (treq->peer_port == creq->peer_port) &&
		   (treq->peer_addr.s_addr == creq->peer_addr.s_addr)){
		    plog(L_QUEUE_INFO,treq,
			 "Request canceled by client - dequeued",0);
		    pQlen--;if(treq->priority > 0) pQNlen--;
		    EXTRACT_ITEM(treq,ardp_pendingQ);
		    ardp_rqfree(treq);
		    ardp_rqfree(creq); ardp_ptfree(pkt);
		    goto check_for_more;
		}
		treq = treq->next;
	    }
	    plog(L_QUEUE_INFO,creq,
		 "Request canceled by client - not on queue",0);
	    ardp_rqfree(creq); ardp_ptfree(pkt);
	    goto check_for_more;
	}
	
	if(hdr_len >= 15) {	/* Priority or PID */
	    bcopy(ctlptr,&stmp,2);
	    if(flags1 & 0x1) pid = ntohs(stmp);
	    else if (flags1 & 0x2) creq->priority = ntohs(stmp);
	}
	ctlptr += 2;
	
	if(hdr_len >= 17) {	/* Priority if PID specified above */
	    bcopy(ctlptr,&stmp,2);
	    if((flags1 & 0x3) == 0x3) creq->priority = ntohs(stmp);
	}
	ctlptr += 2;
	
	if((creq->priority < 0) && (creq->priority != -42)) {
	    plog(L_NET_RDPERR, creq, "Priority %d requested - ignored", creq->priority, 0);
	    creq->priority = 0;
	}
	
	if(pkt->seq == 1) creq->rcvd_thru = max(creq->rcvd_thru, 1);
	
	pkt->length -= hdr_len;
	pkt->start += hdr_len;
        pkt->text = pkt->start;
    }
    else {
	/* When we are no longer getting any of these old format  */
	/* requests, we know that everyone is using the new       */
	/* reliable datagram protocol, and that they also         */
	/* have the singleton response bug fixed.  We can then    */
	/* get rid of the special casing for singleton responses  */
	
	/* Lower the priority to encourage people to upgrade  */
	creq->priority = 1;
	creq->ardp_version = -1;
#ifdef LOGOLDFORMAT
	plog(L_DIR_PWARN,creq,"Old RDP format",0);
#endif LOGOLDFORMAT
	pkt->seq = 1;
	creq->rcvd_tot = 1;
	creq->rcvd_thru = 1;
    }
    
    /* Check to see if it is already on done, partial, or pending */
    
    /* Done queue */
    treq = ardp_doneQ;
    while(treq) {
	if((treq->cid != 0) && (treq->cid == creq->cid) &&
	   (bcmp((char *) &(treq->peer),
		 (char *) &from,sizeof(from))==0)){
	    /* Request is already on doneQ */
	    if(creq->prcvd_thru > treq->prcvd_thru) {
		treq->prcvd_thru = creq->prcvd_thru;
		rr_time = 0; /* made progress, don't supress retransmission */
	    }
	    nreq = treq; 
	    
	    /* Retransmit reply if not completely received */
	    /* and if we didn't retransmit it this second  */
	    if((nreq->prcvd_thru != nreq->trns_tot) &&
	       ((rr_time != now) || (nreq != ardp_doneQ))) {
		plog(L_QUEUE_INFO,nreq,"Retransmitting reply (%d of %d ack)",
		     nreq->prcvd_thru, nreq->trns_tot, 0);
		/* Transmit all outstanding packets */
		tpkt = nreq->trns;
		while(tpkt) {
		    if((abs(tpkt->seq) <= (nreq->prcvd_thru + ARDP_WINDOW_SZ)) &&
		       ((tpkt->seq <= 0) || (tpkt->seq > nreq->prcvd_thru))) 
			ardp_snd_pkt(tpkt,nreq);
		    tpkt = tpkt->next;
		}
		rr_time = now; /* Remember time of retransmission */
	    }
	    /* Move matched request to front of queue */
	    if(nreq != ardp_doneQ) {
		if(nreq == ardp_doneQ->previous) {
		    ardp_doneQ->previous->previous->next = NULL;
		    ardp_doneQ->previous = ardp_doneQ->previous->previous;
		}
		else {
		    nreq->previous->next = nreq->next;
		    nreq->next->previous = nreq->previous;
		}
		nreq->previous = ardp_doneQ->previous;
		nreq->next = ardp_doneQ;
		ardp_doneQ->previous = nreq;
		ardp_doneQ = nreq;
	    }
	    ardp_rqfree(creq); ardp_ptfree(pkt);
	    goto check_for_more;
	}
	/* While we're checking the done queue also see if any    */
	/* replies require follow up requests for acknowledgments */
	if(check_for_ack && (treq->svc_rwait_seq > treq->prcvd_thru) && 
	   (treq->retries_rem > 0) && (now >= treq->wait_till.tv_sec)) {
#ifdef ARDP_LOG_SPONTANEOUS_RETRANSMISSIONS
	    plog(L_QUEUE_INFO,treq,"Requested ack not received - pinging client (%d of %d/%d ack)",
		 treq->prcvd_thru, treq->svc_rwait_seq, treq->trns_tot, 0);
#endif /* ARDP_LOG_SPONTANEOUS_RETRANSMISSIONS */
	    /* Resend the final packet only - to wake the client  */
	    if(treq->trns) ardp_snd_pkt(treq->trns->previous,treq);
	    treq->timeout_adj.tv_sec = ARDP_BACKOFF(treq->timeout_adj.tv_sec);
	    treq->wait_till.tv_sec = now + treq->timeout_adj.tv_sec;
	    treq->retries_rem--;
	}
	treq = treq->next;
    }
    check_for_ack = 0; /* Only check once per call to ardp_accept */
    
    /* If unsequenced control packet free it and check for more */
    if(pkt->seq == 0) {
	ardp_rqfree(creq); ardp_ptfree(pkt);
	goto check_for_more;
    }

    /* Check if incomplete and match up with other incomplete requests */
    if(creq->rcvd_tot != 1) { /* Look for rest of request */
	treq = ardp_partialQ;
	while(treq) {
	    if((treq->cid != 0) && (treq->cid == creq->cid) &&
	       (bcmp((char *) &(treq->peer),
		     (char *) &from,sizeof(from))==0)) {
		/* We found the rest of the request     */
		/* coallesce and log and check_for more */
		ardp_update_cfields(treq,creq);
		if(creq->rcvd_tot) treq->rcvd_tot = creq->rcvd_tot;
		tpkt = treq->rcvd;
		while(tpkt) {
		    if(tpkt->seq == treq->rcvd_thru+1) treq->rcvd_thru++;
		    if(tpkt->seq == pkt->seq) {
			/* Duplicate - discard */
			plog(L_NET_INFO,creq,"Multi-packet duplicate received (pkt %d of %d)",
			     pkt->seq, creq->rcvd_tot, 0);
			if(areq != treq) {
			    if(areq) {
				ardp_acknowledge(areq);
			    }
			    areq = treq;
			}
			ardp_rqfree(creq); ardp_ptfree(pkt);
			goto check_for_more;
		    }
		    if(tpkt->seq > pkt->seq) {
			/* Insert new packet in rcvd */
			pkt->next = tpkt;
			pkt->previous = tpkt->previous;
			if(treq->rcvd == tpkt) treq->rcvd = pkt;
			else tpkt->previous->next = pkt;
			tpkt->previous = pkt;
			if(pkt->seq == treq->rcvd_thru+1) treq->rcvd_thru++;
			while(tpkt && (tpkt->seq == treq->rcvd_thru+1)) {
			    treq->rcvd_thru++;
			    tpkt = tpkt->next;
			}
			pkt = NOPKT;
			break;
		    }
		    tpkt = tpkt->next;
		}
		if(pkt) { /* Append at end of rcvd */
		    APPEND_ITEM(pkt,treq->rcvd); 
		    if(pkt->seq == treq->rcvd_thru+1) treq->rcvd_thru++;
		    pkt = NOPKT;
		}
		if(treq->rcvd_tot && (treq->rcvd_thru == treq->rcvd_tot)) {
		    if(areq == treq) areq = NOREQ;
		    creq = treq; EXTRACT_ITEM(creq, ardp_partialQ); --ptQlen;
		    plog(L_NET_INFO,creq,"Multi-packet request complete",0);
		    goto req_complete;
		}
		else {
		    if(areq != treq) {
			if(areq) {
			    ardp_acknowledge(areq);
			}
			areq = treq;
		    }
		    plog(L_NET_INFO, creq,
			 "Multi-packet request continued (rcvd %d of %d)",
			 creq->rcvd_thru, creq->rcvd_tot, 0);
		    goto check_for_more;
		}
	    }
	    treq = treq->next;
	}
 	/* This is the first packet we received in the request */
	/* log it and queue it and check for more              */
	plog(L_NET_INFO,creq,"Multi-packet request received (pkt %d of %d)",
	     pkt->seq, creq->rcvd_tot, 0);
	APPEND_ITEM(pkt,creq->rcvd);
	APPEND_ITEM(creq,ardp_partialQ); /* Add at end of incomp queue   */
	if(++ptQlen > ptQmaxlen) {       
	    treq = ardp_partialQ;        /* Remove from head of queue    */
	    EXTRACT_ITEM(ardp_partialQ,ardp_partialQ); ptQlen--;
	    plog(L_NET_ERR, treq,
		 "Too many incomplete requests - dropped (rthru %d of %d)",
		 treq->rcvd_thru, treq->rcvd_tot, 0);
	    ardp_rqfree(treq);    
	}
	goto check_for_more;
    }
    
    plog(L_NET_INFO, creq, "Received packet", 0);
    
 req_complete:

    qpos = 0; dpos = 1;

    /* Insert this message at end of pending but make sure the  */
    /* request is not already in pending                        */
    nreq = creq; creq = NOREQ;
    treq = ardp_pendingQ;
	
    if(pkt) {
	nreq->rcvd_tot = 1;
	APPEND_ITEM(pkt,nreq->rcvd);
    }
    
    if(ardp_runQ && (ardp_runQ->cid == nreq->cid) && 
       (ardp_runQ->cid != 0) &&
       (bcmp((char *) &(ardp_runQ->peer),
	     (char *) &(nreq->peer),sizeof(nreq->peer))==0)) {
	/* Request is running right now */
	ardp_update_cfields(ardp_runQ,nreq);
	plog(L_QUEUE_INFO,ardp_runQ,"Duplicate discarded (presently executing)",0);
	ardp_rqfree(nreq);
	nreq = ardp_runQ;
    }
    else if(ardp_pendingQ) {
	while(treq) {
	    if((treq->cid != 0) && (treq->cid == nreq->cid) &&
	       (bcmp((char *) &(treq->peer),
		     (char *) &(nreq->peer),sizeof(treq->peer))==0)){
		/* Request is already on queue */
		ardp_update_cfields(treq,nreq);
		plog(L_QUEUE_INFO,treq,"Duplicate discarded (%d of %d)",dpos,pQlen,0);
		ardp_rqfree(nreq);
		nreq = treq;
		keep_looking = 0;  /* We found the duplicate */
		break;
	    }
	    if((ardp_pri_override && ardp_pri_func && (ardp_pri_func(nreq,treq) < 0)) ||
	       (!ardp_pri_override && ((nreq->priority < treq->priority) ||
				       ((treq->priority == nreq->priority) && ardp_pri_func &&
					(ardp_pri_func(nreq,treq) < 0))))) {
		if(ardp_pendingQ == treq) {
		    nreq->previous = ardp_pendingQ->previous;
		    nreq->next = ardp_pendingQ;
		    ardp_pendingQ->previous = nreq;
		    ardp_pendingQ = nreq;
		}
		else {
		    nreq->next = treq;
		    nreq->previous = treq->previous;
		    nreq->previous->next = nreq;
		    treq->previous = nreq;
		}
		pQlen++;
		if(nreq->priority > 0) pQNlen++;
		LOG_PACKET(nreq, dpos);
		qpos = dpos;
		keep_looking = 1;  /* There may still be a duplicate */
		break;
	    }
	    if(!treq->next) {
		treq->next = nreq;
		nreq->previous = treq;
		ardp_pendingQ->previous = nreq;
		pQlen++;
		if(nreq->priority > 0) pQNlen++;
		LOG_PACKET(nreq,dpos+1);
		qpos = dpos+1;
		keep_looking = 0; /* Nothing more to check */
		break;
	    }
	    treq = treq->next;
	    dpos++;
	}
	/* Save queue position to send to client if appropriate */
	qpos = dpos;
	/* If not added at end of queue, check behind packet for dup */
	if(keep_looking) {
	    while(treq) {
		if((treq->cid == nreq->cid) && (treq->cid != 0) && 
		   (bcmp((char *) &(treq->peer),
			 (char *) &(nreq->peer),
			 sizeof(treq->peer))==0)){
		    /* Found a dup */
		    plog(L_QUEUE_INFO,treq,"Duplicate replaced (removed at %d)", dpos, 0);
		    pQlen--;if(treq->priority > 0) pQNlen--;
		    EXTRACT_ITEM(treq,ardp_pendingQ);
		    ardp_rqfree(treq);
		    break;
		}
		treq = treq->next;
		dpos++;
	    }
	}
    }
    else {
	nreq->next = NULL;
	ardp_pendingQ = nreq;
	ardp_pendingQ->previous = nreq;
	pQlen++;if(nreq->priority > 0) pQNlen++;
	LOG_PACKET(nreq, dpos);
	qpos = dpos;
    }
    
    /* Fill in queue position and system time             */
    nreq->inf_queue_pos = qpos;
    /* nreq->inf_sys_time = **compute-system-time-here**  */

    /* Here we can optionally send a message telling the  */
    /* client to back off.  How long should depend on the */
    /* queue length and the mean service time             */
    /* 15 min for now - the archie server is overloaded, and we expect the
       GOPHER_GW server to be overloaded too.             */
#if defined(ARCHIE) || defined(PSRV_GOPHER_GW)
#ifdef ARCHIE
    if((nreq->ardp_version >= 0) && (sindex(nreq->rcvd->start,"VERSION 1")))
      nreq->ardp_version = -2;
#endif
    ardp_rwait(nreq,900,nreq->inf_queue_pos,nreq->inf_sys_time); 
#endif defined(ARCHIE) || defined(PSRV_GOPHER_GW)

    goto check_for_more;
}


static ardp_update_cfields(RREQ existing,RREQ newvalues)
{
    if(newvalues->prcvd_thru > existing->prcvd_thru)
	existing->prcvd_thru = newvalues->prcvd_thru;
    return(ARDP_SUCCESS);
}

