/*
 * settings.c : Set program parameters on a popup panel
 *
 * George Ferguson, ferguson@cs.rochester.edu, 21 Aug 1991.
 *
 */
#include <stdio.h>
#include <X11/Intrinsic.h>
#include <X11/Shell.h>
#include <X11/StringDefs.h>
#include <X11/Xaw/Form.h>
#include <X11/Xaw/MenuButton.h>
#include <X11/Xaw/Label.h>
#include <X11/Xaw/AsciiText.h>
#include <EzMenu/EzMenu.h>	
#include <X11/Xaw/Cardinals.h>
#include "xarchie.h"
#include "types.h"
#include "appres.h"
#include "alert.h"
#include "confirm.h"

/*
 * Functions declared in this file
 */
void settingsAction();
void applySettingsAction(),defaultSettingsAction(),doneSettingsAction();
void setSearchTypeAction(),setSortTypeAction();
void setHostAction(),setNiceLevelAction();
void setSearchTypeNowAction(),setSortTypeNowAction();
void setHostNowAction(),setNiceLevelNowAction();

static void initSettingsShell(),initSettingsItems();
static void textEventProc();
static void setCurrentSearchType(),setCurrentSortType();
static void setChangedFlag();

/*
 * Data declared in this file
 */
static Widget settingsShell;
static Widget applyButton;
static Widget searchLabel;
static Widget sortLabel;
static Widget hostText;
static Widget maxHitsText;
static Widget timeoutText;
static Widget retriesText;
static Widget niceText;

static SearchType currentSearchType;
static SortType currentSortType;
static Boolean settingsChanged,isPoppedUp;

#define ACTION_PROC(NAME)	void NAME(w,event,params,num_params) \
					Widget w; \
					XEvent *event; \
					String *params; \
					Cardinal *num_params;

/*	-	-	-	-	-	-	-	-	*/
/* External interface (action) procedure */
/*
 * settingsAction() : Pops up (possibly creating) the settings editor,
 *	and fills it with the information from the current values of
 *	the application settings.
 */
/*ARGSUSED*/
ACTION_PROC(settingsAction)
{
    if (settingsShell == NULL)
	initSettingsShell();
    if (isPoppedUp) {
	XRaiseWindow(display,XtWindow(settingsShell));
    } else {
	initSettingsItems();
	XtPopup(settingsShell,XtGrabNone);
	isPoppedUp = True;
    }
}

/*	-	-	-	-	-	-	-	-	*/
/* Initialization procedures */
/*
 * initSettingsShell() : Create the popup settings editor.
 */
static void
initSettingsShell()
{
    isPoppedUp = False;
    initWidgetsFromString(appResources.settingsWidgets,".settingsWidgets");
    /* check for the necessary widgets and set the global variables */
    if ((settingsShell=XtNameToWidget(toplevel,"*settingsShell")) == NULL)
        fail0("didn't create widget \"settingsShell\"");
    /* set globals for optional widgets */
    applyButton = XtNameToWidget(toplevel,"settingsShell*applyButton");
    searchLabel = XtNameToWidget(toplevel,"settingsShell*searchLabel");
    sortLabel = XtNameToWidget(toplevel,"settingsShell*sortLabel");
    hostText = XtNameToWidget(toplevel,"settingsShell*hostText");
    maxHitsText = XtNameToWidget(toplevel,"settingsShell*maxHitsText");
    timeoutText = XtNameToWidget(toplevel,"settingsShell*timeoutText");
    retriesText = XtNameToWidget(toplevel,"settingsShell*retriesText");
    niceText = XtNameToWidget(toplevel,"settingsShell*niceText");
    /* add event handler for detecting changes */
    if (hostText != NULL)
	XtAddEventHandler(hostText,KeyPressMask|ButtonPressMask,False,
			  textEventProc,(XtPointer)NULL);
    if (maxHitsText != NULL)
	XtAddEventHandler(maxHitsText,KeyPressMask|ButtonPressMask,False,
			  textEventProc,(XtPointer)NULL); 
    if (timeoutText != NULL)
	XtAddEventHandler(timeoutText,KeyPressMask|ButtonPressMask,False,
			  textEventProc,(XtPointer)NULL);
    if (retriesText != NULL)
	XtAddEventHandler(retriesText,KeyPressMask|ButtonPressMask,False,
			  textEventProc,(XtPointer)NULL);
    if (niceText != NULL)
	XtAddEventHandler(niceText,KeyPressMask|ButtonPressMask,False,
			  textEventProc,(XtPointer)NULL);
}

/*	-	-	-	-	-	-	-	-	*/
/*
 * initSettingsItems() : Sets the values in the settings editor from the
 *	current state of the application resources.
 */
static void
initSettingsItems()
{
    char buf[8];

    setCurrentSearchType(appResources.searchType);
    setCurrentSortType(appResources.sortType);
    setText(hostText,appResources.archieHost);
    sprintf(buf,"%d",appResources.maxHits);
    setText(maxHitsText,buf);
    setText(hostText,appResources.archieHost);
    sprintf(buf,"%d",appResources.timeout);
    setText(timeoutText,buf);
    sprintf(buf,"%d",appResources.retries);
    setText(retriesText,buf);
    sprintf(buf,"%d",appResources.niceLevel);
    setText(niceText,buf);
    setChangedFlag(False);
}

/*	-	-	-	-	-	-	-	-	*/
/* Callback procedures */
/*
 * applyCallback() : Callback for apply button - Set the application resources
 *	from the items on the settings editor panel. Some of these require
 *	special action when changed, and this routine does that.
 */
/*ARGSUSED*/
ACTION_PROC(applySettingsAction)
{
    Arg args[1];
    char *s;
    int n;

    appResources.searchType = currentSearchType;
    appResources.sortType = currentSortType;
    if (hostText != NULL) {
	XtSetArg(args[0],XtNstring,&s);
	XtGetValues(hostText,args,ONE);
	/* memory leak: we can't free the original, so we can't free any! */
	appResources.archieHost = XtNewString(s);
    }
    if (maxHitsText != NULL) {
	XtSetArg(args[0],XtNstring,&s);
	XtGetValues(maxHitsText,args,ONE);
	appResources.maxHits = atoi(s);
    }
    if (timeoutText != NULL) {
	XtSetArg(args[0],XtNstring,&s);
	XtGetValues(timeoutText,args,ONE);
	appResources.timeout = atoi(s);
    }
    if (retriesText != NULL) {
	XtSetArg(args[0],XtNstring,&s);
	XtGetValues(retriesText,args,ONE);
	appResources.retries = atoi(s);
    }
    if (niceText != NULL) {
	XtSetArg(args[0],XtNstring,&s);
	XtGetValues(niceText,args,ONE);
	n = atoi(s);
	if (n < -32765)		/* leave -32766 to -32768 alone */
	    n = -32765;
	else if (n > 32767)
	    n = 32767;
	appResources.niceLevel = n;
    }
    setChangedFlag(False);
}

/*
 * defaultCallback() : Callback for default button - Reset the items
 *      to their default values.
 */
/*ARGSUSED*/
ACTION_PROC(defaultSettingsAction)
{
    setCurrentSearchType(GfExact);
    setCurrentSortType(GfDefault);
    setText(hostText,"archie.mcgill.ca");
    setText(maxHitsText,"99");
    setText(timeoutText,"4");
    setText(retriesText,"3");
    setText(niceText,"0");
    setChangedFlag(True);
}

/*
 * doneCallback() : Callback for done button - Pop down the editor.
 */
/*ARGSUSED*/
ACTION_PROC(doneSettingsAction)
{
    if (!settingsChanged || confirm0("Discard changes to settings?")) {
	XtPopdown(settingsShell);
	isPoppedUp = False;
    }
}

/*
 * setSearchTypeAction() : Action procedure to set the search type.
 */
/*ARGSUSED*/
ACTION_PROC(setSearchTypeAction)
{
    XrmValue from,to;

    if (*num_params != ONE) {
	alert0("Incorrect number of arguments to set-search-type()");
	return;
    }
    from.addr = *params;
    from.size = sizeof(String);
    to.addr = NULL;
    XtConvertAndStore(w,XtRString,&from,GfRSearchType,&to);
    if (to.addr != NULL)
	setCurrentSearchType((SearchType)(*(to.addr)));
    setChangedFlag(True);
}

/*
 * setSortTypeAction() : Action procedure to set the sort type.
 */
/*ARGSUSED*/
ACTION_PROC(setSortTypeAction)
{
    XrmValue from,to;

    if (*num_params != ONE) {
	alert0("Incorrect number of arguments to set-sort-type()");
	return;
    }
    from.addr = *params;
    from.size = sizeof(String);
    to.addr = NULL;
    XtConvert(w,XtRString,&from,GfRSortType,&to);
    if (to.addr != NULL)
	setCurrentSortType((SortType)(*(to.addr)));
    setChangedFlag(True);
}

/*
 * setHostAction() : Action procedure to set the host.
 */
/*ARGSUSED*/
ACTION_PROC(setHostAction)
{
    if (*num_params != ONE) {
	alert0("Incorrect number of arguments to set-host()");
	return;
    }
    if (hostText == NULL) {
	alert0("set-host() has no effect since hostText was not created");
	return;
    }
    setText(hostText,*params);
    setChangedFlag(True);
}

/*
 * setNiceLevelAction() : Action procedure to set rdgram_priority
 */
/*ARGSUSED*/
ACTION_PROC(setNiceLevelAction)
{
    int n;

    if (*num_params != ONE) {
	alert0("Incorrect number of arguments to set-nice-level()");
	return;
    }
    if (niceText == NULL) {
       alert0("set-nice-level() has no effect since niceText was not created");
	return;
    }
    n = atoi(*params);
    if (n < -32765) {
	alert1("Nice level too negative: %d",n);
	setText(niceText,"-32765");
    } else if (n > 32767) {
	alert1("Nice level too positive: %d",n);
	setText(niceText,"32767");
    } else {
	setText(niceText,*params);
    }
    setChangedFlag(True);
}

/*	-	-	-	-	-	-	-	-	*/
/* These actions are like their non-Now counterparts, expect that
 * (a) they set appResources immediately rather than waiting for
 *     apply-settings() to be called, and
 * (b) they do not set the changedFlag since they have made the change
 *     globally already.
 * Still, they really aren't meant to be used when the settingsPanel is
 * being displayed.
 */

/*ARGSUSED*/
ACTION_PROC(setSearchTypeNowAction)
{
    XrmValue from,to;
    SearchType type;

    if (*num_params != ONE) {
	alert0("Incorrect number of arguments to set-search-type-now()");
	return;
    }
    from.addr = *params;
    from.size = sizeof(String);
    to.addr = NULL;
    XtConvertAndStore(w,XtRString,&from,GfRSearchType,&to);
    if (to.addr != NULL) {
	type = (SearchType)(*(to.addr));
	appResources.searchType = type;
	setCurrentSearchType(type);
    }
}

/*ARGSUSED*/
ACTION_PROC(setSortTypeNowAction)
{
    XrmValue from,to;
    SortType type;

    if (*num_params != ONE) {
	alert0("Incorrect number of arguments to set-sort-type-now()");
	return;
    }
    from.addr = *params;
    from.size = sizeof(String);
    to.addr = NULL;
    XtConvertAndStore(w,XtRString,&from,GfRSortType,&to);
    if (to.addr != NULL) {
	type = (SortType)(*(to.addr));
	appResources.sortType = type;
	setCurrentSortType(type);
    }
}

/*ARGSUSED*/
ACTION_PROC(setHostNowAction)
{
    if (*num_params != ONE) {
	alert0("Incorrect number of arguments to set-host-now()");
	return;
    }
    /* Memory leak */
    appResources.archieHost = XtNewString(*params);
    setText(hostText,*params);
}

/*ARGSUSED*/
ACTION_PROC(setNiceLevelNowAction)
{
    int n;

    if (*num_params != ONE) {
	alert0("Incorrect number of arguments to set-nice-level-now()");
	return;
    }
    n = atoi(*params);
    if (n < -32765) {
	alert1("Nice level too negative: %d -- not set",n);
    } else if (n > 32767) {
	alert1("Nice level too positive: %d -- not set",n);
    } else {
	appResources.niceLevel = n;
	setText(niceText,*params);
    }
}

/*	-	-	-	-	-	-	-	-	*/
/*
 * textEventProc() : Called whenever the user types in any Text item.
 *      Note that this does NOT detect, eg., selection pastes, as
 *	documented in the BUGS section of the man page.
 */
/*ARGSUSED*/
static void
textEventProc(w,client_data,event,continue_flag)
Widget w;
XtPointer client_data;
XEvent *event;
Boolean *continue_flag;
{
    setChangedFlag(True);
}

/*	-	-	-	-	-	-	-	-	*/

static void
setCurrentSearchType(type)
SearchType type;
{
    char *s;

    currentSearchType = type;
    switch (type) {
	case GfExact :   s = GfNExact; break;
	case GfSubstr :  s = GfNSubstr; break;
	case GfSubcase : s = GfNSubcase; break;
	case GfRegexp :  s = GfNRegexp; break;
	case GfExactSubstr :  s = GfNExactSubstr; break;
	case GfExactSubcase : s = GfNExactSubcase; break;
	case GfExactRegexp :  s = GfNExactRegexp; break;
    }
    setLabel(searchLabel,s);
}

static void
setCurrentSortType(type)
SortType type;
{
    char *s;

    currentSortType = type;
    switch (type) {
	case GfDefault : s = GfNDefault; break;
	case GfInvdate : s = GfNInvdate; break;
    }
    setLabel(sortLabel,s);
}

static void
setChangedFlag(value)
Boolean value;
{
    if (applyButton != NULL)
	XtSetSensitive(applyButton,value);
    settingsChanged = value;
}
