#ifndef ObjectList_H
#define ObjectList_H
/* A subclass of the List class that adds the capability to order
	objects based on the informal SortableObject protocol declared
	int the Object(SortableObject) category. */
#import <List_Archival.h>

/*\ ---------------------- Constants ---------------------- \*/
/* The various sort procedures on */
enum {eFirstID, eInsertionSort=0, eShellSort, eHeapSort, eQuickSort,
	eMergeSort, eLastID} SortMethodID;
#define INVALID_SORTMETHODID -1

/* The sort direction enums */
enum {eSmallToLarge, eLargeToSmall} SortDirection;

/* Which direction to choose an object when using the
	findNearestObject: method */
enum {ePreviousObject, eNextObject} Direction;

@interface ObjectList : List
{
	int sortAlgorithm;		// The sorting algorithm used
	int sortDirection;		// Flag indicating if the sort direction is smallest to largest. Not used.
	SEL theKeyMethod;	// The SEL of the object which returns its key value
	int methodTypeID;		// The type returned by theKeyMethod
	int typeSize;			// The size of the type
	int comparisonCount;
	BOOL isHomogeneous;
	BOOL isSorted;		// Flag indicating if the list is sorted
}

/*\ ---------------------- Initialization Methods ---------------------- \*/
- initCount:(int) count sortAlgorithm:(int) algorithm;
- initCount:(int) count sortAlgorithm:(int) algorithm
	typeID:(int) typeID keyMethod:(SEL) keyMethod;
- initCount:(int) count sortAlgorithm:(int) algorithm
	typeID:(int) typeID size:(int) tSize keyMethod:(SEL) keyMethod;
- (BOOL) isHomogeneous;
- changeKeyMethod:(SEL) keyMethod typeID:(int) typeID;
- changeKeyMethod:(SEL) keyMethod typeID:(int) typeID size:(int) tSize
	dir:(int) direction;

/*\ ---------------------- Sort Methods ---------------------- \*/
- sort;
- sortAtLevel:(int) level;
- sortAtLevel:(int) level reportTime:(long *) time
	reportCompares:(int *) comparisons;
- (BOOL) isSorted;
- (BOOL) setSortDirection:(int) direction;
- (int) sortDirection;

/*\ ---------------------- Manipulation Methods ---------------------- \*/
- (int) insertObject: anObject;	// Insert the object into the list
- appendObject: anObject;	// Add the object ot the end of the list

/*\ -------------------------  Search Methods ------------------------- \*/
/* Perform a search for the object given by theObject.  This method performs a
	binary search using the objects compare: method. If found,
	unique indicates if other objects in the list have the same key.
	Returns the object or nil if not found */
- findObject:theObject;
- findObjectWithData:(void *) data;
- findObjectWithData:(void *) data typeString:(const char *) typeString;

/*\ ---------------------- Setting the Sort Method ---------------------- \*/
- setSortMethod:(int) methodID;
+ (int) addSortMethod:(SEL) theMethod delegate: delegate;
+ setDelegate: selDelegate;
+ (BOOL) isValidSortMethod:(int) methodID;

/*\ ------------------------- Private Methods ------------------------- \*/
- (int) binarySearch: anObject canFail:(BOOL) canFail;
- (int) binarySearchUsing:(void *) data canFail:(BOOL) canFail;

/*\ ---------------------- Sort Methods ---------------------- \*/
- insertionSort;
- insertionSort:(int) first inc:(int) step;
- shellSort;
- heapSort;
- (void) insertHeap: keyObj:(int) root :(int) bound;
- mergeSort;		// Not implemented

/*\ ---------------------------- Debugging ---------------------------- \*/
- printHeap;
- print;			// Display the list by asking each object to print itself
- printVerbose;		// Preceed each objects output with ``Element %d''

/*\ ------------------ ObjectArchival Protocol Methods ------------------ \*/
/* The current version for new documents */
#define OBJECTLIST_VERS		1	// Object release version
#define OBJECTLIST_SUBVERS	0	// Object subrelease version
#define OBJECTLIST_TYPE		0	// Object version type
/* Object versions we know about */
#define OBJECTLIST_VERSION_0 1000
- initFromTStream:(NXTypedStream *) stream;
- readFromTStream:(NXTypedStream *) stream;
- writeToTStream:(NXTypedStream *) stream;

@end

/*\ ---------------------- SortableObject Protocol ---------------------- \*/
#import "SortableObject.h"

/* RCS Information:
	$Author: me $;
	$Date: 93/02/23 02:01:43 $;
	$Source: /usr1/me/NeXTSrc/MyClasses/RCS/ObjectList.h,v $;
	$Revision: 1.1 $;
	$Log:	ObjectList.h,v $
 * Revision 1.1  93/02/23  02:01:43  me
 * Begin RCS logging.
 * ;
*/
#endif	// ObjectList_H
