// MOString.h
//
// by Mike Ferris
// Part of MOKit
// Copyright 1993, all rights reserved.

// ABOUT MOKit
// by Mike Ferris (mike@lorax.com)
//
// MOKit is a collection of useful and general objects.  Permission is 
// granted by the author to use MOKit in your own programs in any way 
// you see fit.  All other rights pertaining to the kit are reserved by the 
// author including the right to sell these objects as objects,  as part 
// of a LIBRARY, or as SOURCE CODE.  In plain English, I wish to retain 
// rights to these objects as objects, but allow the use of the objects 
// as pieces in a fully functional program.  Permission is also granted to 
// redistribute the source code of MOKit for FREE as long as this copyright 
// notice is left intact and unchanged.  NO WARRANTY is expressed or implied.  
// The author will under no circumstances be held responsible for ANY 
// consequences from the use of these objects.  Since you don't have to pay 
// for them, and full source is provided, I think this is perfectly fair.

// ABOUT MOString
//
// The MOString class manages a character string.  It allocates memory and 
// frees it as necessary.  MOString's strings are NULL-terminated.
//
// Unless told otherwise a MOString owns the memory it uses to hold the string
// and it will be freed when new contents are set or when the MOString is 
// freed.  This is true unless the contents were set using the 
// -setStringValueUnique: method OR the -makeUnique: method was called 
// OR the -setShouldFree:NO method was called.
// 
// If the MOString is unique it can not be modified.
//
// If you do something weird like slapping a '\0' in the middle of the 
// string with -replaceCharAt:with:, the String forgets about anything  
// that might have been after the index where you put the '\0' (although 
// all the memory will still be freed when it's time).
//
// Also it would not be wise to do something like:
// "[aString setStringValue:([aString string] + 4)]"
// The first thing done by setStringValue: is to free the old string, 
// thus the above snippet is real bad news.  However, this might work OK
// for MOStrings that don't free their contents.

#import <appkit/appkit.h>

@interface MOString:Object <NXTransport>
{
	@protected
		char *str;
		NXAtom uStr;
		size_t len;
		BOOL isUnique;
		BOOL shouldFree;
}

// =-=-=-=-=-=-=-=-=-=-=-=- Initializing  the class -=-=-=-=-=-=-=-=-=-=-=-=

+ initialize;

// =-=-=-=-=-=-= Initializing, copying,  and freeing MOStrings =-=-=-=-=-=-=

- init;
- initStringValue:(const char *)s;
- initStringValueNoCopy:(char *)s;
- initStringValueNoCopy:(char *)s shouldFree:(BOOL)flag;	// DI
- initStringValueUnique:(const char *)s;
- initFromFormat:(const char *)formatStr, ...;
- copyFromZone:(NXZone *)zone;
- deepCopy;
- deepCopyFromZone:(NXZone *)zone;
- shallowCopy;
- shallowCopyFromZone:(NXZone *)zone;
- free;

// =-=-=-=-=-=-=-=-=-=-=-=-= Configuring MOStrings =-=-=-=-=-=-=-=-=-=-=-=-=

- setStringValue:(const char *)s;
- setStringValueNoCopy:(char *)s;
- setStringValueNoCopy:(char *)s shouldFree:(BOOL)flag;
- setStringValueUnique:(const char *)s;
- setFromFormat:(const char *)formatStr, ...;

- setIntValue:(int)val;
- setFloatValue:(float)val;
- setDoubleValue:(double)val;

- setNull;

- makeUnique;
- setShouldFree:(BOOL)flag;

// =-=-=-=-=-=-=-=-=-=-=-=-=- Querying  MOStrings -=-=-=-=-=-=-=-=-=-=-=-=-=

- (const char *)stringValue;
- (int)intValue;
- (float)floatValue;
- (double)doubleValue;
- (char *)getCopyInto:(char *)buf;

- (size_t)length;
- (unsigned int)count;
- (BOOL)isNull;
- (BOOL)isEmpty;
- (BOOL)isUnique;
- (BOOL)shouldFree;

// =-=-=-=-=-=-=-=-=-=-=-=-=-= Archiving methods =-=-=-=-=-=-=-=-=-=-=-=-=-=

- write:(NXTypedStream *)typedStream;
- read:(NXTypedStream *)typedStream;

// =-=-=-=-=-=-=-=-=-=-=-=-= NXTransport  protocol =-=-=-=-=-=-=-=-=-=-=-=-=

- encodeRemotelyFor:(NXConnection *)connection 
			freeAfterEncoding:(BOOL *)flagp isBycopy:(BOOL)isBycopy;
- encodeUsing:(id <NXEncoding>)portal;
- decodeUsing:(id <NXDecoding>)portal;

// =-=-=-=-=-=-=-=-=-=-=-=-=- Hashing and isEqual -=-=-=-=-=-=-=-=-=-=-=-=-=

- (unsigned int)hash;
- (BOOL)isEqual:anObj;

@end

@interface MOString(Parsing)

// =-=-=-=-=-=-=-=-=-=-=-=-=- Parsing  MOStrings -=-=-=-=-=-=-=-=-=-=-=-=-=

- (char)charAt:(int)index;

- substringFrom:(int)start to:(int)end;

- (int)positionOf:(char)aChar nthOccurrence:(int)n;
- (int)countOccurrencesOf:(char)aChar;

- tokenize:(const char *)breakChars into:aList;

@end

@interface MOString(Modifying)

// =-=-=-=-=-=-=-=-=-=-=-=-=- Modifying MOStrings -=-=-=-=-=-=-=-=-=-=-=-=-=

- takeStringValueFrom:sender;
- takeIntValueFrom:sender;
- takeFloatValueFrom:sender;
- takeDoubleValueFrom:sender;

- cat:stringObject;
- catStringValue:(const char *)s;
- catFromFormat:(const char *)format, ...;

- preCat:stringObject;
- preCatStringValue:(const char *)s;
- preCatFromFormat:(const char *)format, ...;

- insert:stringObject at:(int)position;
- insertStringValue:(const char *)s at:(int)position;

- (char)replaceCharAt:(int)index with:(char)newChar;
- (int)replaceAllOccurrencesOfChar:(char)oldChar with:(char)newChar;

- (size_t)recalcLength;

- convertToLower;
- convertToUpper;

@end

@interface MOString(Comparing)

// =-=-=-=-=-=-=-=-=-=-=-=-=- Comparing MOStrings -=-=-=-=-=-=-=-=-=-=-=-=-=

- (int)compare:stringObject;
- (int)compare:stringObject caseSensitive:(BOOL)flag;
- (int)compare:stringObject caseSensitive:(BOOL)flag length:(int)length;
- (int)compare:stringObject caseSensitive:(BOOL)flag length:(int)length 
			withTable:(NXStringOrderTable *)table;
- (int)compareStr:(const char *)s;
- (int)compareStr:(const char *)s caseSensitive:(BOOL)flag;
- (int)compareStr:(const char *)s caseSensitive:(BOOL)flag length:(int)length;
- (int)compareStr:(const char *)s caseSensitive:(BOOL)flag length:(int)length
			withTable:(NXStringOrderTable *)table;

- (int)endCompare:stringObject;
- (int)endCompare:stringObject caseSensitive:(BOOL)flag;
- (int)endCompare:stringObject caseSensitive:(BOOL)flag length:(int)length;
- (int)endCompare:stringObject caseSensitive:(BOOL)flag length:(int)length 
			withTable:(NXStringOrderTable *)table;
- (int)endCompareStr:(const char *)s;
- (int)endCompareStr:(const char *)s caseSensitive:(BOOL)flag;
- (int)endCompareStr:(const char *)s caseSensitive:(BOOL)flag 
			length:(int)length;
- (int)endCompareStr:(const char *)s caseSensitive:(BOOL)flag 
			length:(int)length withTable:(NXStringOrderTable *)table;

@end

@interface MOString(ANSI)

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=- ANSI covers -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//            (strtok is not provided... see -tokenize: above)
//                 (strcoll, and strxfrm are not provided)

- (size_t)strlen;

- (const char *)strcpy:(const char *)s;
- (const char *)strncpy:(const char *)s :(size_t)n;
- (const char *)strcat:(const char *)s;
- (const char *)strncat:(const char *)s :(size_t)n;

- (int)strcmp:(const char *)s;
- (int)strncmp:(const char *)s :(size_t)n;

- (const char *)strchr:(char)aChar;
- (const char *)strrchr:(char)aChar;
- (const char *)strstr:(const char *)searchStr;
- (const char *)strpbrk:(const char *)breakChars;

- (size_t)strspn:(const char *)acceptableChars;
- (size_t)strcspn:(const char *)breakChars;

- sprintf:(const char *)formatStr, ...;

@end

@interface MOString(Debugging)

// =-=-=-=-=-=-=-=-=-=-=-=-=-=- Debug  printing -=-=-=-=-=-=-=-=-=-=-=-=-=-=

- buildInstanceImageIn:(char *)buf;
- printForDebugger:(NXStream *)stream;
- printToStdErr:(const char *)label;

@end

