#-----------------------------------------------------------------------------
# cddb.tcl : An implementation of CDDB based on Net::CDDB
# Author   : straycat000@yahoo.com
# Version  : 1.0
#-----------------------------------------------------------------------------

# package information
set version 1.0
set name "cddb.tcl"
set full_name "$name $version"

# cddb_debugging 
set cddb_debug 0

# cddb information
set cddb_server "freedb.freedb.org"
set cddb_port 888
set cddb_localuser "nobody"
set cddb_localhost "localhost.localdomain"

# _write_line : Write a line to a socket.
proc _write_line { sock line } {
  global cddb_debug

  if $cddb_debug {
    puts "$line"
  }
  puts $sock $line
}

# _read_line : Read a line from a socket.
proc _read_line { sock } {
  global cddb_debug 

  gets $sock line
  if $cddb_debug {
    puts $line
  }
  return $line
}

# _read_lines : Read from the socket until a "." is found.  This is how 
#  	 	CDDB denotes the end of output.
proc _read_lines { sock } {
  global cddb_debug
  while { "[gets $sock line]" != "-1" } {
    if { "$line" == "." } { 
      break
    }
    if $cddb_debug {
      puts "$line"
    }
    lappend lines $line
  }
  return $lines
}

# _connect : Connect to a CDDB server.
proc _connect { host port } {
  if [catch {socket $host $port} s] {
    puts stderr "$s"
    exit
  }
  fconfigure $s -buffering line
  _read_line $s
  return $s
}

# _disconnect : Close the connection to a CDDB server.
proc _disconnect { sock } {
  _write_line $sock "quit"
  close $sock
}

# _cddb_hello : Say hello to a CDDB server.
proc _cddb_hello { sock } {
  global cddb_localuser
  global cddb_localhost
  global full_name
  set msg "cddb hello $cddb_localuser $cddb_localhost $full_name"
  _write_line $sock $msg
  _read_line $sock
}

# _extract_stuff : Extract CDDB information from CDDB read response.
proc _extract_stuff { lines } {
  global cddb_debug 
  set num_tracks 0
  foreach line $lines {
    if { [string first "DTITLE" $line] >= 0 } {
      regexp "DTITLE=(.*) / (.*)" $line all artist album
      if $cddb_debug {
        puts "Artist : $artist"
        puts "Album  : $album"
      }
      set cddb_info(artist) $artist
      set cddb_info(album) $album
    } else {
      if [regexp "TTITLE(\[0-9]\[0-9]*)=(.*)" $line all track title] {
        incr track
        set cddb_info($track) $title 
        incr num_tracks
      }
    }
  }
  set cddb_info(tracks) $num_tracks
  return [array get cddb_info]
}
  
# cddb_query : Perform CDDB query operation.
proc cddb_query { } {
  global cddb_server 
  global cddb_port
  
  set disc_data [exec discid]
  
  set sock [_connect $cddb_server $cddb_port]
  _cddb_hello $sock

  _write_line $sock "cddb query $disc_data"
  set rsp [_read_line $sock]
  set status   [lindex $rsp 0]
  switch "$status" {
    "200" {
      # exact match 
      set category [lindex $rsp 1]
      set disc_id  [lindex $rsp 2]
      set dtitle   [lrange $rsp 3 end]
      regexp "(.*) / (.*)" $dtitle all artist title
    }
    "211" {
      # inexact match - not ready to handle this yet
      set category "inexact"
      #puts "Inexact match found - unimplemented."
      return [cddb_inexact [_read_line $sock]]
    }
    "202" { 
      # not found
      puts "No CDDB information found."
      return
    }
    default {
      puts "Unknown or unimplemented return code." 
      return
    }
  }
  _disconnect $sock
  return [list $category $disc_id $artist $title]
}

# cddb_read : Perform CDDB read operation.
proc cddb_read { category disc_id } {
  global cddb_server
  global cddb_port 
  set sock [_connect $cddb_server $cddb_port]
  _cddb_hello $sock

  _write_line $sock "cddb read $category $disc_id"
  set rsp [_read_lines $sock]

  _disconnect $sock

  array set cddb_info [_extract_stuff $rsp]
  set cddb_info(category) $category
  set cddb_info(disc_id) $disc_id
  return [array get cddb_info]
}

# cddb_lscat : Perforn CDDB lscat operation (list categories).
proc cddb_lscat { } {
  global cddb_server
  global cddb_port

  set sock [_connect $cddb_server $cddb_port]
  _cddb_hello $sock

  _write_line $sock "cddb lscat"
  _read_line $sock
  set rsp [_read_lines $sock]

  _disconnect $sock

  return $rsp
}

# cddb_inexact : Try to match an inexact match.  Use the first match.
proc cddb_inexact { line } {
  set category [lindex $line 0]
  set disc_id  [lindex $line 1]
  set dtitle [lrange $line 2 end]
  regexp "(.*) / (.*)" $dtitle all artist title
  puts "Inexact match found!"
  puts "Using first choice - $artist $title"
  return [list $category $disc_id $artist $title]
}

# cddb_sites : Fetch a list of cddb servers
proc cddb_sites { } {
  global cddb_server
  global cddb_port

  set sock [_connect $cddb_server $cddb_port]
  _cddb_hello $sock

  _write_line $sock "sites"
  set rsp [_read_line $sock]
  set status [lindex $rsp 0]
  if { "$status" == "210" } {
    # Exact match.
    set rsp [_read_lines $sock]
  } else {
    set rsp ""
  }

  _disconnect $sock
  return $rsp
}
