/*
 * Copyright (c) 2010, Johan Thelin
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with 
 * or without modification, are permitted provided that 
 * the following conditions are met:
 *
 *   - Redistributions of source code must retain the above
 *     copyright notice, this list of conditions and the 
 *     following disclaimer.
 *   - Redistributions in binary form must reproduce the
 *     above copyright notice, this list of conditions and
 *     the following disclaimer in the documentation and/or
 *     other materials provided with the distribution.
 *   - The name of Johan Thelin may not be used to endorse
 *     or promote products derived from this software 
 *     without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A 
 * PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL 
 * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 * USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 */

#include "pinchwidget.h"

#include <QEvent>
#include <QGestureEvent>
#include <QPainter>
#include <QPinchGesture>

PinchWidget::PinchWidget(QWidget *parent)
    : QWidget(parent),
    rotationAngle(0),
    scaleFactor(1),
    currentScaleFactor(1)
{
    setMinimumSize(100, 100);
    grabGesture(Qt::PinchGesture);
}

PinchWidget::~PinchWidget()
{
}

bool PinchWidget::event(QEvent *event)
{
    if(event->type() == QEvent::Gesture)
        return gestureEvent(static_cast<QGestureEvent*>(event));
    return QWidget::event(event);
}

bool PinchWidget::gestureEvent(QGestureEvent *event)
{
    if(QGesture *pinch = event->gesture(Qt::PinchGesture))
    {
        pinchGesture(static_cast<QPinchGesture*>(pinch));
        return true;
    }
    return false;
}

void PinchWidget::pinchGesture(QPinchGesture *gesture)
{
    QPinchGesture::ChangeFlags flags = gesture->changeFlags();
    if(flags & QPinchGesture::RotationAngleChanged)
    {
        qreal value = gesture->rotationAngle();
        qreal lastValue = gesture->lastRotationAngle();
        rotationAngle += value - lastValue;
    }
    if(flags & QPinchGesture::ScaleFactorChanged)
    {
        currentScaleFactor = gesture->scaleFactor();
    }
    if(gesture->state() == Qt::GestureFinished)
    {
        scaleFactor *= currentScaleFactor;
        currentScaleFactor = 1;
    }
    update();
}

void PinchWidget::paintEvent(QPaintEvent*)
{
    QPainter p(this);

    // Center
    p.translate(width()/2, height()/2);

    // Draw outline
    p.save();
    p.rotate(rotationAngle);
    p.scale(currentScaleFactor*scaleFactor, currentScaleFactor*scaleFactor);
    p.translate(-50, -50);
    p.fillRect(0, 0, 100, 100, Qt::red);
    p.restore();

    // Draw filled
    p.translate(-50, -50);
    p.fillRect(0, 0, 100, 100, QColor(0,0,0,75));
}

void PinchWidget::resizeEvent(QResizeEvent*)
{
    update();
}
