#include "watermark_dialog.h"
#include <qlayout.h>
#include <qpushbutton.h>
#include <klocale.h>
#include <kdebug.h>
#include <kdialogbase.h>
#include "previewarea.h"
#include <qspinbox.h>
#include <qlabel.h>
#include <qtextedit.h>
#include <qfontdialog.h>
#include <kglobal.h>
#include <kiconloader.h>
#include <libkipi/uploadwidget.h>
#include <kio/netaccess.h>
#include <qpainter.h>
#include <ktempfile.h>
#include <qfileinfo.h>
#include <kimageio.h>
#include <kmessagebox.h>

WatermarkDialog::WatermarkDialog( KIPI::Interface* interface, QWidget* parent, const char* name )
    : KDialogBase( IconList, i18n("Watermark Plugin"), Ok | Cancel, Ok, parent, name, false ), m_interface( interface )
{
    addSettingsPage();
    addDestPage();
}

void WatermarkDialog::reload()
{
    KURL::List imgs = images();
    if ( imgs.count() != 0 ) {
        KURL url = imgs[0];
        m_pixmap = pixmapForURL(url);
        m_previewArea->setPixmap( paintImage( m_pixmap ) );
    }
}

void WatermarkDialog::slotOk()
{

    KURL::List imgs = images();
    for (KURL::List::const_iterator it = imgs.begin(); it != imgs.end(); ++it) {
        QPixmap pix = pixmapForURL( *it );
        pix = paintImage( pix );
        savePixmap( *it, pix );
    }
    KDialogBase::slotOk();
}

void WatermarkDialog::slotChangeFont()
{
    bool ok;
    QFont font = QFontDialog::getFont( &ok, m_font->font(), this );
    if ( ok ) {
        m_font->setFont( font );
        repaintPixmap();
    }
}

void WatermarkDialog::addSettingsPage()
{
    QWidget* top = addPage( i18n( "Settings" ), i18n( "Settings" ),
                            KGlobal::iconLoader()->loadIcon( QString::fromLatin1( "package_settings" ), KIcon::Desktop, 32 ) );
    QHBoxLayout* toplay = new QHBoxLayout( top, 6 );

    QVBoxLayout* vlay = new QVBoxLayout( toplay, 6 );
    QLabel* label = new QLabel( i18n( "Text:" ), top );
    vlay->addWidget( label );

    m_text = new QTextEdit( top );
    vlay->addWidget( m_text );
    connect( m_text, SIGNAL( textChanged() ), this, SLOT( repaintPixmap() ) );

    label = new QLabel( i18n( "Font:" ), top );
    vlay->addWidget( label );

    m_font = new QPushButton( i18n( "Change Font" ), top );
    connect( m_font, SIGNAL( clicked() ), this, SLOT( slotChangeFont() ) );
    vlay->addWidget( m_font );

    m_previewArea = new PreviewArea( top );
    toplay->addWidget( m_previewArea );
}

void WatermarkDialog::addDestPage()
{
    QWidget* top = addPage( i18n( "Destination" ), i18n( "Destination" ),
                            KGlobal::iconLoader()->loadIcon( QString::fromLatin1( "hdd_unmount" ), KIcon::Desktop, 32 ) );
    QHBoxLayout* toplay = new QHBoxLayout( top, 6 );

    m_uploadWidget = new KIPI::UploadWidget( m_interface, top );
    toplay->addWidget( m_uploadWidget );
}

KURL::List WatermarkDialog::images()
{
    KIPI::ImageCollection collection = m_interface->currentSelection();
    if  (!collection.isValid() )
        collection = m_interface->currentAlbum();

    if ( !collection.isValid() )
        return KURL::List();

    return collection.images();
}

QPixmap WatermarkDialog::paintImage( const QPixmap& pixmap )
{
    QPixmap pix = pixmap;
    QPainter painter( &pix );
    painter.setFont( m_font->font() );
    painter.setPen( Qt::gray );
    painter.drawText( QRect( QPoint(0,0), pix.size() ), AlignCenter, m_text->text() );

    return pix;
}

QPixmap WatermarkDialog::pixmapForURL( const KURL& url )
{
    QString tmpFile;
    if ( KIO::NetAccess::download( url, tmpFile, this ) ) {
        QImage img( tmpFile );
        KIO::NetAccess::removeTempFile( tmpFile );

        KIPI::ImageInfo info = m_interface->info( url );
        int angle = info.angle();

        if ( angle != 0 ) {
            QWMatrix matrix;
            matrix.rotate( angle );
            img = img.xForm( matrix );
        }

        return QPixmap(img);
    }
    return QPixmap();
}

void WatermarkDialog::repaintPixmap()
{
    m_previewArea->setPixmap( paintImage( m_pixmap ) );
}

void WatermarkDialog::savePixmap( const KURL& source, const QPixmap& pixmap )
{
    QString ext = QFileInfo( source.fileName() ).extension(false);
    KTempFile tmpFile;
    pixmap.save( tmpFile.name(), KImageIO::type( source.fileName() ) );

    KURL dest = m_uploadWidget->path();
    KURL destFile = dest;
    destFile.addPath( source.fileName() );

    int count = 0;
    while ( KIO::NetAccess::exists( destFile, false, this ) ) {
        destFile = dest;
        QString fileName = source.fileName();
        fileName = QString( "%1-%2.%3" ).arg( QFileInfo( fileName ).baseName() ).arg( count++ ).arg( ext );
        destFile.addPath( fileName );
    }

    KIO::NetAccess::upload( tmpFile.name(), destFile, this );
    tmpFile.unlink();

    QString err;
    if ( !m_interface->addImage( destFile, err ) ) {
        KMessageBox::error( this, QString( "Unable to add file %1 to database. Error was %2" ).arg( destFile.url() ).arg( err ) );
    }
}
