#*************************************************************************
#*
#*  $RCSfile: SpreadSheet.tcl,v $
#*
#*  $Revision: 1.7 $
#*
#*  last change: $Author: grosserw $ $Date: 2005/09/18 10:22:12 $
#*
#*  The Contents of this file are made available subject to the terms of
#*  the following license
#*
#*  - BSD License
#*
#*  see file license.terms in this directory
#*
#*  Copyright 2005 Wolfgang Grosser
#*
#*  Author: Wolfgang Grosser
#*
#*   All Rights Reserved.
#*
#*   Contributor(s): 
#*
#*
#************************************************************************/

package require Itcl
package require tcluno_scalc

itcl::class ::itcluno::SpreadSheet {
	inherit OpenOffice

	public variable delimiter ""
	public variable separator ""

	protected variable sheets ""
	protected variable sheet
	
	protected variable chartIdx 0
	protected variable dataPilotIdx 0
	protected variable controller ""

	constructor {args} {eval OpenOffice::constructor private:factory/scalc $args} {}
	public method getSheetNames {}
	public method insertSheet {sheetName {position end}}
	public method moveSheet {sheetName {position end}}
	public method renameSheet {oldname newname}
	public method deleteSheet {sheetName}
	public method activateSheet {sheetName}
	public method setFreezePosition {sheetName cell}
	public method createChart {targetSheetName graphicRectangle options sourceSheetName sourceRange}
	public method createPivotTable {targetSheetName targetCell sourceSheetName sourceRange columnFields rowFields dataFields}
	public method columnWidth {sheetName columnList width}
	public method getDocumentProperties {{propertyList ""}}
	public method getSheetProperties {sheetName {propertyList ""}}
	public method getColumnProperties {sheetName columnIndex {propertyList ""}}
	public method getRowProperties {sheetName rowIndex {propertyList ""}}
	public method getCellProperties {sheetName cellIndex {propertyList ""}}
	public method getCellRangeProperties {sheetName cellRange {propertyList ""}}
	public method setDocumentProperties {propertyList}
	public method setSheetProperties {sheetName propertyList}
	public method setColumnProperties {sheetName columnIndex propertyList}
	public method setRowProperties {sheetName rowIndex propertyList}
	public method setCellProperties {sheetName cellIndex propertyList}
	public method setCellRangeProperties {sheetName cellRange propertyList}

	public method setCellFormat {sheetName cellRange type {subType ""}}

	public method setCellValue {sheetName cellIndex valueList}
	public method setBorder {sheetName cellRange whichBorder properties}

	public method insertRows {sheetName insertBefore {insertCount 1}}
	public method removeRows {sheetName removeStart {removeCount 1}}
	public method insertColumns {sheetName insertBefore {insertCount 1}}
	public method removeColumns {sheetName removeStart {removeCount 1}}

	public method exportToPdf {filename}

	protected method checkSheetName {sheetName}

	protected method checkForFilter {filename filterSequence}

	protected method getRangeProperties {sheetName func index propertyList}
	protected method getCellRangeAddress {sheetHandle range}
	protected method getCellAddress {sheetHandle cell}

	private method _getController {}
	private method _manipulateColsRows {sheetName start count getFunc manipulateFunc}
}

itcl::body itcluno::SpreadSheet::constructor {args} {
	if {[string length $handle] != 0} {
		set document $handle
	} else {
		if {[string length $filename] == 0} {
			set filename $factory
		}
		if {[catch {checkForFilter $filename $filterSequence} msg]} {
			puts stderr $msg
			#	try opening but without guarantee what happens
			openFile $filename $filterSequence
			return
		}
		openFile $filename $filterSequence
	}
	set sheets [$document getSheets]
# ARNULF
	set model [$desktop getCurrentComponent]
#	if {![string equal $model {TCL_NONE}] && ![string equal $model {???}]} {
#		set controller [$model getCurrentController]
#	}
# ARNULF END
}
itcl::body itcluno::SpreadSheet::getSheetNames {} {
	set count [$sheets getCount]
	set nameList [list]
	for {set i 0} {$i < $count} {incr i} {
		set sheet [$sheets getByIndex $i]
		lappend nameList [$sheet getName]
	}
	return $nameList
}
itcl::body itcluno::SpreadSheet::insertSheet {sheetName {position end}} {
	if {[checkSheetName $sheetName]} {
		return ""
	}
	set sheetList [getSheetNames]
	if {[string equal $position end] || [llength $sheetList] < $position} {
		set position [llength $sheetList]
	}
	if {$position < 0} {
		set position 0
	}
	$sheets insertNewByName $sheetName $position
	return $sheetName
}
itcl::body itcluno::SpreadSheet::moveSheet {sheetName {position end}} {
	if {![checkSheetName $sheetName]} {
		return ""
	}
	set sheetList [getSheetNames]
	if {[string equal $position end] || [llength $sheetList] < $position} {
		set position [llength $sheetList]
	}
	if {$position < 0} {
		set position 0
	}
	$sheets moveByName $sheetName $position
	return $sheetName
}
itcl::body itcluno::SpreadSheet::renameSheet {oldname newname} {
	if {![checkSheetName $oldname]} {
		return ""
	}
	if {[checkSheetName $newname]} {
		return ""
	}
	set sheet [$sheets getByName $oldname]
	$sheet setName $newname
	return $newname
}
itcl::body itcluno::SpreadSheet::deleteSheet {sheetName} {
	if {![checkSheetName $sheetName]} {
		return ""
	}
	$sheets removeByName $sheetName
	return $name
}
itcl::body itcluno::SpreadSheet::activateSheet {sheetName} {
	if {![checkSheetName $sheetName]} {
		return ""
	}
	_getController
	set sheet [$sheets getByName $sheetName]
# ARNULF
	$controller setActiveSheet $sheet
# ARNULF END
	return $sheetName
}
itcl::body itcluno::SpreadSheet::setFreezePosition {sheetName cell} {
	if {![checkSheetName $sheetName]} {
		return ""
	}
	_getController
	if {[string equal $controller {TCL_NONE}]} {
		#	May be we have a SpreadSheet within a Presentation
		return ""
	}
	set sheet [$sheets getByName $sheetName]
	$controller setActiveSheet $sheet
	eval $controller freezeAtPosition [itcluno::OfficeUtilities::cellToRowColumn $cell]
	return $sheetName
}
itcl::body itcluno::SpreadSheet::createChart {targetSheetName graphicRectangle options sourceSheetName sourceRange} {
	if {![checkSheetName $sourceSheetName]} {
		return ""
	}
	set sheetList [getSheetNames]
	if {[lsearch -exact $sheetList $targetSheetName] < 0} {
		set targetSheet [insertSheet $targetSheetName]
	}
	set targetSheet [$sheets getByName $targetSheetName]
	set charts [$targetSheet getCharts]
	if {[string equal $targetSheetName $sourceSheetName]} {
		set sourceSheet $targetSheet
	} else {
		set sourceSheet [$sheets getByName $sourceSheetName]
	}
	set rectangle [::tcluno_soffice::createRectangle $graphicRectangle]
	set cellRangeSequence [$desktop tcluno::createUnoSequence Any]
#	set cellRange [$desktop tcluno::createUnoStructHelper com.sun.star.table.CellRangeAddress [concat $sourceIdx [itcluno::OfficeUtilities::cellRangeToRowColumnList $sourceRange]]] 
	set cellRange [getCellRangeAddress $sourceSheetName $sourceRange]
	$desktop tcluno::appendUnoSequence $cellRangeSequence $cellRange
	array set optionArray $options
	set hasColumnHeader 0
	#	this option is not an OpenOffice property
	if {[info exists optionArray(hasColumnHeader)]} {
		set hasColumnHeader $optionArray(hasColumnHeader)
		unset optionArray(hasColumnHeader)
	}
	set hasRowHeader 0
	#	this option is not an OpenOffice property
	if {[info exists optionArray(hasRowHeader)]} {
		set hasRowHeader $optionArray(hasRowHeader)
		unset optionArray(hasRowHeader)
	}
	$charts addNewByName chart_[incr chartIdx] $rectangle $cellRangeSequence $hasColumnHeader $hasRowHeader
	set chart [$charts getByName chart_$chartIdx]
	set chartObject [$chart getEmbeddedObject]
	foreach {option value} [array get optionArray] {
		#	only these options are supported right now
		switch $option {
			title {
				::tcluno_soffice::scalc::setChartTitle $chartObject $value
			}
			type {
				eval ::tcluno_soffice::scalc::setChartType $chartObject $value
			}
		}
	}
	return chart_$chartIdx
}
itcl::body itcluno::SpreadSheet::createPivotTable {targetSheetName targetCell sourceSheetName sourceRange columnFields rowFields dataFields} {
	if {![checkSheetName $sourceSheetName]} {
		return ""
	}
	set sourceSheet [$sheets getByName $sourceSheetName]
	set dataPilotTableSupplier [$sourceSheet queryInterface [$sourceSheet ::tcluno::getTclunoType {TypeClass com.sun.star.sheet.XDataPilotTablesSupplier INTERFACE}]]
	set dataPilotTables [$dataPilotTableSupplier getDataPilotTables]
	set dataPilotDescription [$dataPilotTables createDataPilotDescriptor]

	set sourceAddress [getCellRangeAddress $sourceSheetName $sourceRange]
	$dataPilotDescription setSourceRange $sourceAddress

	foreach {startColumn startRow endColumn endRow} [itcluno::OfficeUtilities::cellRangeToRowColumnList $sourceRange] break
	set headerList [list]
	for {} {$startColumn <= $endColumn} {incr startColumn} {
		set cell [$sourceSheet getCellByPosition $startColumn $startRow]
		lappend headerList [$cell getString]
	}

	set fields [$dataPilotDescription getDataPilotFields]
	foreach e $columnFields {
		set idx [lsearch -exact $headerList $e]
		if {$idx < 0} {
			puts stderr "column \"$e\" not found in headerlist"
			continue
		}
		set columnField [$fields getByIndex $idx]
		setPropertyValues $columnField [list \
			Orientation [itcluno::OfficeUtilities::enum com.sun.star.sheet.DataPilotFieldOrientation COLUMN] \
		]
	}

	foreach e $rowFields {
		set idx [lsearch -exact $headerList $e]
		if {$idx < 0} {
			puts stderr "row \"$e\" not found in headerlist"
			continue
		}
		set rowField [$fields getByIndex $idx]
		setPropertyValues $rowField [list Orientation [itcluno::OfficeUtilities::enum com.sun.star.sheet.DataPilotFieldOrientation ROW]]
	}

	foreach e $dataFields {
		foreach {text function} $e break
		set idx [lsearch -exact $headerList $text]
		if {$idx < 0} {
			puts stderr "data \"$text\" not found in headerlist"
			continue
		}
		set dataField [$fields getByIndex $idx]
		setPropertyValues $dataField [list \
			Orientation [itcluno::OfficeUtilities::enum com.sun.star.sheet.DataPilotFieldOrientation DATA] \
			Function [itcluno::OfficeUtilities::enum com.sun.star.sheet.GeneralFunction $function] \
		]
	}

	if {![checkSheetName $targetSheetName]} {
		insertSheet $targetSheetName
	}
	set targetAddress [getCellAddress $targetSheetName $targetCell]

	set dataPilotName data_pilot_[incr dataPilotIdx]
	$dataPilotTables insertNewByName $dataPilotName $targetAddress $dataPilotDescription
}
itcl::body itcluno::SpreadSheet::columnWidth {sheetName columnList width} {
	if {![checkSheetName $sheetName]} {
		return ""
	}
	set sheet [$sheets getByName $sheetName]
	set columns [$sheet getColumns]
	foreach col $columnList {
		set column [$columns getByIndex $col]
		if {[string equal $width {optimal}]} {
			$column setPropertyValue OptimalWidth 1
		} else {
			$column setPropertyValue Width $width
		}
	}
	return $sheetName
}
itcl::body itcluno::SpreadSheet::getDocumentProperties {{propertyList ""}} {
	return [getRangeProperties "" "" "" $propertyList]
}
itcl::body itcluno::SpreadSheet::getSheetProperties {sheetName {propertyList ""}} {
	return [getRangeProperties $sheetName "" "" $propertyList]
}
itcl::body itcluno::SpreadSheet::getColumnProperties {sheetName columnIndex {propertyList ""}} {
	return [getRangeProperties $sheetName getColumns [list column $columnIndex] $propertyList]
}
itcl::body itcluno::SpreadSheet::getRowProperties {sheetName rowIndex {propertyList ""}} {
	return [getRangeProperties $sheetName getRows [list row $rowIndex] $propertyList]
}
itcl::body itcluno::SpreadSheet::getCellProperties {sheetName cellIndex {propertyList ""}} {
	return [getRangeProperties $sheetName getCellByPosition [itcluno::OfficeUtilities::cellToRowColumn $cellIndex] $propertyList]
}
itcl::body itcluno::SpreadSheet::getCellRangeProperties {sheetName cellRange {propertyList ""}} {
	return [getRangeProperties $sheetName getCellRangeByPosition [itcluno::OfficeUtilities::cellRangeToRowColumnList $cellRange] $propertyList]
}
itcl::body itcluno::SpreadSheet::setDocumentProperties {propertyList} {
	setPropertyValues $document $propertyList
}
itcl::body itcluno::SpreadSheet::setSheetProperties {sheetName propertyList} {
	if {![checkSheetName $sheetName]} {
		return ""
	}
    set sheetHandle [$sheets getByName $sheetName]
	setPropertyValues $sheetHandle $propertyList
	return $sheetName
}
itcl::body itcluno::SpreadSheet::setColumnProperties {sheetName columnIndex propertyList} {
	if {![checkSheetName $sheetName]} {
		return ""
	}
    set sheetHandle [$sheets getByName $sheetName]
	set columnsHandle [$sheetHandle getColumns]
	set columnHandle [$columnsHandle getByIndex $columnIndex]
	setPropertyValues $columnHandle $propertyList
	return $sheetName
}
itcl::body itcluno::SpreadSheet::setRowProperties {sheetName rowIndex propertyList} {
	if {![checkSheetName $sheetName]} {
		return ""
	}
    set sheetHandle [$sheets getByName $sheetName]
	set rowsHandle [$sheetHandle getRows]
	set rowHandle [$rowsHandle getByIndex $rowIndex]
	setPropertyValues $rowHandle $propertyList
	return $sheetName
}
itcl::body itcluno::SpreadSheet::setCellProperties {sheetName cellIndex propertyList} {
	if {![checkSheetName $sheetName]} {
		return ""
	}
    set sheetHandle [$sheets getByName $sheetName]
	foreach {columnIndex rowIndex} [itcluno::OfficeUtilities::cellToRowColumn $cellIndex] break
	set cellHandle [$sheetHandle getCellByPosition $columnIndex $rowIndex]
	setPropertyValues $cellHandle $propertyList
	return $sheetName
}
itcl::body itcluno::SpreadSheet::setCellRangeProperties {sheetName cellRange propertyList} {
	if {![checkSheetName $sheetName]} {
		return ""
	}
    set sheetHandle [$sheets getByName $sheetName]
    set rangeHandle [eval $sheetHandle getCellRangeByPosition [itcluno::OfficeUtilities::cellRangeToRowColumnList $cellRange]]
	setPropertyValues $rangeHandle $propertyList
	return $sheetName
}
itcl::body itcluno::SpreadSheet::setCellFormat {sheetName cellIndex type {subType ""}} {
	if {![checkSheetName $sheetName]} {
		return ""
	}
	set oFormats [$document getNumberFormats]
	set oLocale [$desktop tcluno::createUnoStructHelper com.sun.star.lang.Locale [list de DE [list]]]
	set formatKey [$oFormats getStandardFormat [itcluno::OfficeUtilities::constant com.sun.star.util.NumberFormat $type] $oLocale]
	if {[string length $subType] > 0} {
		set formatKey [$oFormats getFormatIndex [itcluno::OfficeUtilities::constant com.sun.star.i18n.NumberFormatIndex $subType] $oLocale]
	}
	set sheetHandle [$sheets getByName $sheetName]
	set cellRangeHandle [eval $sheetHandle getCellRangeByPosition [itcluno::OfficeUtilities::cellRangeToRowColumnList $cellIndex]]
	setPropertyValues $cellRangeHandle [list NumberFormat $formatKey]
	return $sheetName
}
itcl::body itcluno::SpreadSheet::setCellValue {sheetName cellIndex valueList} {
	if {![checkSheetName $sheetName]} {
		return ""
	}
    set sheetHandle [$sheets getByName $sheetName]
	foreach {column row} [itcluno::OfficeUtilities::cellToRowColumn $cellIndex] break
	set startColumn $column
	foreach valueRow $valueList {
		foreach value $valueRow {
			if {[string length $value] == 0} {
				incr column
				continue
			}
			set cellHandle [$sheetHandle getCellByPosition $column $row]
			set func setString
			if {[string is double $value]} {
				set func setValue
			}
			if {[string first {=} $value] == 0} {
				set func setFormula
			}
			$cellHandle $func $value
			incr column
		}
		incr row
		set column $startColumn
	}
	return $sheetName
}
itcl::body itcluno::SpreadSheet::setBorder {sheetName cellRange whichBorder properties} {
	if {![checkSheetName $sheetName]} {
		return ""
	}
	set sheetHandle [$sheets getByName $sheetName]
	set cellRangeHandle [eval $sheetHandle getCellRangeByPosition [itcluno::OfficeUtilities::cellRangeToRowColumnList $cellRange]]
	if {[string equal $whichBorder {TableBorder}]} {
		setTableBorder $cellRangeHandle $properties
		return $sheetName
	}
	set border [makeBorderStruct $properties]
	setPropertyValues $cellRangeHandle [list $whichBorder $border]
	return $sheetName
}
itcl::body itcluno::SpreadSheet::checkSheetName {sheetName} {
	set nameList [getSheetNames]
	set idx [lsearch -exact $nameList $sheetName]
	if {$idx < 0} {
		return 0
	}
	return 1
}
itcl::body itcluno::SpreadSheet::insertRows {sheetName insertBefore {insertCount 1}} {
	return [_manipulateColsRows $sheetName $insertBefore $insertCount getRows insertByIndex]
}
itcl::body itcluno::SpreadSheet::removeRows {sheetName removeStart {removeCount 1}} {
	return [_manipulateColsRows $sheetName $removeStart $removeCount getRows removeByIndex]
}
itcl::body itcluno::SpreadSheet::insertColumns {sheetName insertBefore {insertCount 1}} {
	return [_manipulateColsRows $sheetName $insertBefore $insertCount getColumns insertByIndex]
}
itcl::body itcluno::SpreadSheet::removeColumns {sheetName removeStart {removeCount 1}} {
	return [_manipulateColsRows $sheetName $removeStart $removeCount getColumns removeByIndex]
}
itcl::body itcluno::SpreadSheet::exportToPdf {filename} {
	itcluno::OpenOffice::exportToPdf $filename calc_pdf_Export
}
itcl::body itcluno::SpreadSheet::checkForFilter {filename filterSequence} {
	set extension [file extension $filename]
	if {[string equal $extension {.xls}]} {
		set filterProperty [$desktop tcluno::createUnoStructHelper com.sun.star.beans.PropertyValue {FilterName -1 {MS Excel 97} 0}]
		$desktop tcluno::appendUnoSequence $filterSequence $filterProperty
		return
	}
	if {[string equal $extension {.stc}]} {
		return
	}
	if {[string equal $extension {.sxc}]} {
		return
	}
	if {[string equal $extension {.ots}]} {
		return
	}
	if {[string equal $extension {.ods}]} {
		return
	}
	if {[string equal $filename $factory]} {
		return
	}
	#	Standard separator is "\""
	set delim 34
	if {[string length $delimiter] > 0} {
		scan $delimiter {%c} delim
	}
	if {[string equal $extension {.txt}]} {
		set filterProperty [$desktop tcluno::createUnoStructHelper com.sun.star.beans.PropertyValue [list FilterName -1 "Text - txt - csv (StarCalc)" 0]]
		$desktop tcluno::appendUnoSequence $filterSequence $filterProperty
		#	Standard separator is "\t"
		set sep 9
		if {[string length $separator] > 0} {
			scan $separator {%c} sep
		}
		set filterProperty [$desktop tcluno::createUnoStructHelper com.sun.star.beans.PropertyValue [list FilterOptions -1 "$sep,$delim,0,1," 0]]
		$desktop tcluno::appendUnoSequence $filterSequence $filterProperty
		return
	}
	if {[string equal $extension {.csv}]} {
		set filterProperty [$desktop tcluno::createUnoStructHelper com.sun.star.beans.PropertyValue [list FilterName -1 "Text - txt - csv (StarCalc)" 0]]
		$desktop tcluno::appendUnoSequence $filterSequence $filterProperty
		#	Standard separator is ";"
		set sep 59
		if {[string length $separator] > 0} {
			scan $separator {%c} sep
		}
		set filterProperty [$desktop tcluno::createUnoStructHelper com.sun.star.beans.PropertyValue [list FilterOptions -1 "$sep,$delim,0,1," 0]]
		$desktop tcluno::appendUnoSequence $filterSequence $filterProperty
		return
	}
	return -code error -errorinfo "unsupported extension $extension"
}
itcl::body itcluno::SpreadSheet::getRangeProperties {sheetName func index propertyList} {
	if {[string length $sheetName] == 0} {
		set object $document
	} else {
		if {![checkSheetName $sheetName]} {
			return ""
		}
		set object [$sheets getByName $sheetName]
		if {[llength $index] > 0} {
			foreach {idx1 idx2} $index break
			switch $idx1 {
				column -
				row {
					set object [$object $func]
					set object [$object getByIndex $idx2]
				}
				default {
					set object [eval $object $func $index]
				}
			}
		}
	}
	return [getPropertyValues $object $propertyList]
}
itcl::body itcluno::SpreadSheet::getCellRangeAddress {sheetName range} {
	set sheetList [getSheetNames]
	set sourceIdx [lsearch -exact $sheetList $sheetName]
	if {$sourceIdx < 0} {
		return ""
	}
	set address [$desktop tcluno::createUnoStructHelper com.sun.star.table.CellRangeAddress [concat $sourceIdx [itcluno::OfficeUtilities::cellRangeToRowColumnList $range]]]
	return $address
}
itcl::body itcluno::SpreadSheet::getCellAddress {sheetName cell} {
	set sheetList [getSheetNames]
	set sourceIdx [lsearch -exact $sheetList $sheetName]
	if {$sourceIdx < 0} {
		return ""
	}
	set address [$desktop tcluno::createUnoStructHelper com.sun.star.table.CellAddress [concat $sourceIdx [itcluno::OfficeUtilities::cellToRowColumn $cell]]]
	return $address
}
itcl::body itcluno::SpreadSheet::_getController {} {
	if {[string length $controller] > 0} {
		return $controller
	}
	set controller [$document getCurrentController]
	return $controller
}
itcl::body itcluno::SpreadSheet::_manipulateColsRows {sheetName start count getFunc manipulateFunc} {
	if {![checkSheetName $sheetName]} {
		return ""
	}
    set sheetHandle [$sheets getByName $sheetName]
	set colRowHandle [$sheetHandle $getFunc]
	$colRowHandle $manipulateFunc $start $count
	return $sheetName
}
