#*************************************************************************
#*
#*  $RCSfile: Presentation.tcl,v $
#*
#*  $Revision: 1.5 $
#*
#*  last change: $Author: grosserw $ $Date: 2005/09/18 10:21:36 $
#*
#*  The Contents of this file are made available subject to the terms of
#*  the following license
#*
#*  - BSD License
#*
#*  see file license.terms in this directory
#*
#*  Copyright 2005 Wolfgang Grosser
#*
#*  Author: Wolfgang Grosser
#*
#*   All Rights Reserved.
#*
#*   Contributor(s): 
#*
#*
#************************************************************************/

package require Itcl
package require tcluno_simpress

itcl::class itcluno::Presentation {
	inherit OpenOffice

	protected variable drawPages ""
	protected variable drawPage

	protected variable masterPages ""
	protected variable masterPage

	protected variable chartIdx 0
	protected variable controller ""

	protected variable _presentationObject
	protected variable _objectId -1
	protected variable objectPrefix
	protected variable _bitmapId -1
	protected variable bitmapPrefix


	constructor {args} {eval OpenOffice::constructor private:factory/simpress $args} {}
	public method getPageNames {}
	public method insertPage {{position end} {layoutIndex -1}}
	public method renamePage {oldname newname}
	public method deletePage {pageName}
	public method activatePage {pageName}

	public method setMasterText {rectangle text {type Text}}
	public method setText {pageName rectangle text {type Text}}
	public method setPreparedText {pageName index string}
	public method setImage {pageName rectangle graphicURL}
	public method setPreparedImage {pageName index graphicURL}

	public method getDocumentProperties {{propertyList ""}}
	public method getMasterPageProperties {{propertyList ""}}
	public method getPageProperties {pageName {propertyList ""}}
	public method getObjectProperties {objectId {propertyList ""}}
	public method getPreparedObjectProperties {pageName index {propertyList ""}}
	public method setPagePropertyValues {pageName propertyList}
	public method setPropertyValues {objectId propertyList}
	public method setPreparedObjectProperties {pageName index propertyList}

	public method createTable {pageName rectangle}
	public method createChart {pageName rectangle}

	public method exportToPdf {filename}

	protected method checkPageName {pageName}

	protected method getPageByName {pageName}
	protected method getPageByIndex {index}
	protected method checkForFilter {filename filterSequence}

	protected method createOLE2Object {pageName rectangle type}

	private method _getController {}
}

itcl::body itcluno::Presentation::constructor {args} {
	set objectPrefix "_${this}_@@@_"
	set bitmapPrefix "_${this}_bitmap_@@@_"
	if {[string length $filename] == 0} {
		set filename $factory
	}
	if {[catch {checkForFilter $filename $filterSequence} msg]} {
		puts stderr $msg
		#	try opening but without guarantee what happens
		openFile $filename $filterSequence
		return
	}
	openFile $filename $filterSequence
	set drawPages [$document getDrawPages]
	set masterPages [$document getMasterPages]
}
itcl::body itcluno::Presentation::getPageNames {} {
	return [$drawPages getElementNames]
}
itcl::body itcluno::Presentation::insertPage {{position end} {layoutIndex -1}} {
	set pageList [getPageNames]
	if {[string equal $position end] || [llength $pageList] < $position} {
		set position [llength $pageList]
	}
#	if {$position < -1} {
#		set position -1
#	}
	$drawPages insertNewByIndex $position
	set page [getPageByIndex $position]
	if {[string length $page] == 0} {
		return -1
	}
	if {$layoutIndex != -1} {
		$page setPropertyValue Layout $layoutIndex
	}
	return $position
}
itcl::body itcluno::Presentation::renamePage {oldname newname} {
	if {![checkPageName $oldname]} {
		return ""
	}
	if {[checkPageName $newname]} {
		return ""
	}
	set page [$drawPages getByName $oldname]
	$page setName $newname
	return $newname
}
itcl::body itcluno::Presentation::deletePage {pageName} {
	if {![checkPageName $pageName]} {
		return ""
	}
	$drawPages remove $pageName
	return $pageName
}
itcl::body itcluno::Presentation::activatePage {pageName} {
	if {![checkPageName $pageName]} {
		return ""
	}
	_getController
	set page [$drawPages getByName $pageName]
	$controller setCurrentPage $page
	return $pageName
}
itcl::body itcluno::Presentation::setMasterText {rectangle text {type Text}} {
	set page [$masterPages getByIndex 0]
	set text [tcluno::simpress::setText $document $page $rectangle $text com.sun.star.drawing.${type}Shape]
	set _presentationObject($objectPrefix[incr _objectId]) $text
	return $objectPrefix$_objectId
}
itcl::body itcluno::Presentation::setText {pageName rectangle text {type Text}} {
	if {![checkPageName $pageName]} {
		return ""
	}
	set page [$drawPages getByName $pageName]
	set text [tcluno::simpress::setText $document $page $rectangle $text com.sun.star.drawing.${type}Shape]
	set _presentationObject($objectPrefix[incr _objectId]) $text
	return $objectPrefix$_objectId
}
itcl::body itcluno::Presentation::setPreparedText {pageName index string} {
	if {![checkPageName $pageName]} {
		return ""
	}
	set page [$drawPages getByName $pageName]
	set text [$page getByIndex $index]
	$text setString $string
	return [$text getText]
}
itcl::body itcluno::Presentation::setImage {pageName rectangle graphicURL} {
	if {![checkPageName $pageName]} {
		return ""
	}
	set page [$drawPages getByName $pageName]
	foreach {x y w h unit} $rectangle break
	set position [tcluno_soffice::createPoint [list $x $y $unit]]
	set size [tcluno_soffice::createSize [list $w $h $unit]]
	set bitmaps [$document createInstance com.sun.star.drawing.BitmapTable]
	$bitmaps insertByName $bitmapPrefix[incr _bitmapId] $graphicURL
	set newURL [$bitmaps getByName $bitmapPrefix$_bitmapId]
	set tmpImage [$document createInstance com.sun.star.drawing.GraphicObjectShape]
	OpenOffice::setPropertyValues $tmpImage [list GraphicURL $newURL]
	$tmpImage setPosition $position
	$tmpImage setSize $size
#	array set properties [getPropertyValues $tmpImage ""]
#	unset -nocomplain properties(Preview)
#	unset -nocomplain properties(MetaFile)
#	parray properties
	$page add $tmpImage
	set _presentationObject($objectPrefix[incr _objectId]) $tmpImage
	return $objectPrefix$_objectId
}
itcl::body itcluno::Presentation::setPreparedImage {pageName index graphicURL} {
	if {![checkPageName $pageName]} {
		return ""
	}
	set page [$drawPages getByName $pageName]
	set text [$page getByIndex $index]
	set position [lindex [$text getPosition] 1]
	foreach e $position {
		set pos([lindex $e 0]) [lindex $e 1]
	}
	set size [lindex [$text getSize] 1]
	foreach e $size {
		set pos([lindex $e 0]) [lindex $e 1]
	}
	set bitmaps [$document createInstance com.sun.star.drawing.BitmapTable]
	$bitmaps insertByName $bitmapPrefix[incr _bitmapId] $graphicURL
	set newURL [$bitmaps getByName $bitmapPrefix$_bitmapId]
	set tmpImage [$document createInstance com.sun.star.drawing.GraphicObjectShape]
	OpenOffice::setPropertyValues $tmpImage [list GraphicURL $newURL]
	$tmpImage setPosition [tcluno_soffice::createPoint [list [expr {$pos(X)}] [expr {$pos(Y)}]]]
	$tmpImage setSize [tcluno_soffice::createSize [list [expr {$pos(Width)}] [expr {$pos(Height)}]]]
	$page add $tmpImage
	$page remove $text
	return $tmpImage
}
itcl::body itcluno::Presentation::getDocumentProperties {{propertyList ""}} {
	return [getPropertyValues $document $propertyList]
}
itcl::body itcluno::Presentation::getMasterPageProperties {{propertyList ""}} {
    set page [$masterPages getByIndex 0]
	return [getPropertyValues $page $propertyList]
}
itcl::body itcluno::Presentation::getPageProperties {pageName {propertyList ""}} {
	if {![checkPageName $pageName]} {
		return ""
	}
    set page [$drawPages getByName $pageName]
	return [getPropertyValues $page $propertyList]
}
itcl::body itcluno::Presentation::getPreparedObjectProperties {pageName index {propertyList ""}} {
	if {![checkPageName $pageName]} {
		return ""
	}
    set page [$drawPages getByName $pageName]
	set object [$page getByIndex $index]
	return [getPropertyValues $object $propertyList]
}
itcl::body itcluno::Presentation::getObjectProperties {objectId {propertyList ""}} {
	set object $_presentationObject($objectId)
	return [getPropertyValues $object $propertyList]
}
itcl::body itcluno::Presentation::setPagePropertyValues {pageName propertyList} {
	if {![checkPageName $pageName]} {
		return ""
	}
    set page [$drawPages getByName $pageName]
	OpenOffice::setPropertyValues $page $propertyList
}
itcl::body itcluno::Presentation::setPropertyValues {objectId propertyList} {
	OpenOffice::setPropertyValues $_presentationObject($objectId) $propertyList
}
itcl::body itcluno::Presentation::setPreparedObjectProperties {pageName index propertyList} {
	if {![checkPageName $pageName]} {
		return ""
	}
    set page [$drawPages getByName $pageName]
	set object [$page getByIndex $index]
	OpenOffice::setPropertyValues $object $propertyList
}
itcl::body itcluno::Presentation::createTable {pageName rectangle} {
	set model [createOLE2Object $pageName $rectangle table]
	if {[string length $model] == 0} {
		return ""
	}
	return [itcluno::SpreadSheet $this#auto -handle $model]
}
itcl::body itcluno::Presentation::createChart {pageName rectangle} {
	set model [createOLE2Object $pageName $rectangle chart]
	return [itcluno::ChartDocument $this#auto -handle $model]
}
itcl::body itcluno::Presentation::exportToPdf {filename} {
	itcluno::OpenOffice::exportToPdf $filename impress_pdf_Export
}
itcl::body itcluno::Presentation::checkPageName {pageName} {
	set pageList [getPageNames]
	if {[lsearch -exact $pageList $pageName] < 0} {
		return 0
	}
	return 1
}
itcl::body itcluno::Presentation::getPageByName {pageName} {
	if {![checkPageName $pageName]} {
		return ""
	}
	return [$drawPages getByName $pageName]
}
itcl::body itcluno::Presentation::getPageByIndex {index} {
	set nameList [getPageNames]
	if {$index < 0 || $index >= [llength $nameList]} {
		return ""
	}
	return [$drawPages getByIndex $index]
}
itcl::body itcluno::Presentation::checkForFilter {filename filterSequence} {
	if {[string equal [file extension $filename] {.ppt}]} {
		set filterProperty [$desktop tcluno::createUnoStructHelper com.sun.star.beans.PropertyValue {FilterName -1 {MS PowerPoint 97} 0}]
		$desktop tcluno::appendUnoSequence $filterSequence $filterProperty
		return
	}
	if {[string equal [file extension $filename] {.sxi}]} {
		return
	}
	if {[string equal [file extension $filename] {.sti}]} {
		return
	}
	if {[string equal [file extension $filename] {.odp}]} {
		return
	}
	if {[string equal [file extension $filename] {.otp}]} {
		return
	}
	if {[string equal $filename $factory]} {
		return
	}
	return -code error -errorinfo "unsupported extension [file extension $filename]"
}
itcl::body itcluno::Presentation::createOLE2Object {pageName rectangle type} {
	if {![checkPageName $pageName]} {
		return ""
	}
    set page [$drawPages getByName $pageName]
	set msf [$document queryInterface [$document ::tcluno::getTclunoType {TypeClass com.sun.star.lang.XMultiServiceFactory INTERFACE}]]
	set ole2 [$msf createInstance com.sun.star.drawing.OLE2Shape]
	set shape [$ole2 queryInterface [$ole2 ::tcluno::getTclunoType {TypeClass com.sun.star.drawing.XShape INTERFACE}]]
	$page add $shape
	foreach {x y w h unit} $rectangle break
	set point [::tcluno_soffice::createPoint [list $x $y $unit]]
	$shape setPosition $point
	set size [::tcluno_soffice::createSize [list $w $h $unit]]
	$shape setSize $size
	set shapeProp [$shape queryInterface [$shape ::tcluno::getTclunoType {TypeClass com.sun.star.beans.XPropertySet INTERFACE}]]
	$shapeProp setPropertyValue CLSID $clsid($type)
	return [$shapeProp getPropertyValue Model]
}
itcl::body itcluno::Presentation::_getController {} {
	if {[string length $controller] > 0} {
		return $controller
	}
	set controller [$document getCurrentController]
	return $controller
}
