/*
    gui.c - Callback-Funktionen fuer die grafische Oberflaeche
*/

# include <stdio.h>
# include <unistd.h>
# include <locale.h>
# include <signal.h>
# include <fcntl.h>
# include <xview/frame.h>
# include <xview/panel.h>
# include <xview/notice.h>
# include <xview/file_chsr.h>
# include "audio.h"
# include "dft.h"
# include "chirp.h"

# define HELP_PATH "/usr/openwin/lib/help/xdams/"
# define BROWSER1 "konqueror"
# define BROWSER2 "firefox"

# define TMP_DATA_FILE "_tmp"	/* ohne Endung ".dat"! */

# define MAX_DATA 10000

# define FREQ_L 160.0		/* analysierter Frequenzbereich */
# define FREQ_H 15820.0

# define R0	3203.0		/* Ausgangswiderstand fuer Impedanzmessung */

# define KEY_FRAME 1
# define KEY_QUIT_NOTICE 2
# define KEY_SAVE_CHOOSER 3
# define KEY_BUTTON_PLAY 4
# define KEY_BUTTON_PLAY 4
# define KEY_BUTTON_LOOP 5
# define KEY_BUTTON_STOP 6
# define KEY_BUTTON_SAVE 7
# define KEY_BUTTON_PRINT 8
# define KEY_DEVS_NOTICE 9
# define KEY_CAL_NOTICE 10
# define KEY_FRAME_SETUP 11
# define KEY_BUTTON_ZERO 12
# define KEY_MIN_Y_ITEM 13
# define KEY_MAX_Y_ITEM 14
# define KEY_LOAD_CHOOSER 15
# define KEY_REF_CURVE_ITEM 16
# define KEY_SHOW_REF_ITEM 17
# define KEY_TITLE_ITEM 18
# define KEY_PRINTER_ITEM 19
# define KEY_FRAME_PRINT 20
# define KEY_ACQUIRE_NOTICE 21
# define KEY_OVERLOAD_NOTICE 22
# define KEY_CHOICE_MODE 23

/*---------- globale Variablen ----------*/

static int snd_in, snd_out, mixer,		/* File-Handles */
           mode, loop, out_level, rec_level,
	   show_phase, sine_freq, y_min, y_max,
	   show_ref, data_points, help_pid;

FILE *gnuplot;

static char ref_curve_name[256];

static double u_t[MAX_DATA], u0_mag[MAX_DATA], u0_pha[MAX_DATA],
              u1_mag[MAX_DATA], u1_pha[MAX_DATA],
	      magnitude[MAX_DATA], phase[MAX_DATA],
	      dist;


/*---------- Funktion zur Einstellung der Pegel-Stufen ----------*/

void set_levels(void)
 {
  if (mixer > -1)
   {
    switch(out_level)
     {
      case 0:
        set_out_volume(mixer, CHIRP_LEVEL_OUT1, SYNC_LEVEL_OUT);
        break;
      case 1:
        set_out_volume(mixer, CHIRP_LEVEL_OUT2, SYNC_LEVEL_OUT);
        break;
      case 2:
        set_out_volume(mixer, CHIRP_LEVEL_OUT3, SYNC_LEVEL_OUT);
     }
    switch(rec_level)
     {
      case 0:
        set_in_volume(mixer, CHIRP_LEVEL_IN1, SYNC_LEVEL_IN);
        break;
      case 1:
        set_in_volume(mixer, CHIRP_LEVEL_IN2, SYNC_LEVEL_IN);
        break;
      case 2:
        set_in_volume(mixer, CHIRP_LEVEL_IN3, SYNC_LEVEL_IN);
     }
   }
  return;
 }


/*---------- Spektrum in Datei schreiben ----------*/

int write_data(double mag[], double pha[], int f1, int f2, char *name)
 {
  int i, n;
  double f_step;
  FILE *stream;

  if ((stream = fopen(name, "w")) == NULL)
   {
    perror("write_data(): Can't open file");
    return(1);
   }

  fprintf(stream, "# Freq.  Magnitude   Phase\n");

  f_step = 1.0*SAMPLE_FREQ/CHIRP_LENGTH;
  i = 0.5 + f1/f_step;
  n = 0.5 + f2/f_step;
  while (i <= n)
   {
    fprintf(stream, "%7.1f  %10.3e  %6.1f\n", f_step*i, mag[i], pha[i]);
    i++;
   }

  fclose(stream);
  return(0);
 }


/*---------- Uebersteuerung erkennen ----------*/

int overload(double *data, int length)
 {
  int i;

  for (i=0; i<length; i++)
    if ((data[i] > 0.95) || (data[i] < -0.95))
      return(1);
  return(0);
 }


/*---------- Diagramm-Einstellungen ----------*/

void setup_plot(FILE *stream)
 {
  fprintf(stream, "set grid; set data style lines; set grid mxtics mytics\n"
       "set logscale x; set xrange [100:20000]; set xlabel 'Frequenz / Hz'\n"
       "set nolabel\n");

  fprintf(stream, "set yrange [%d:%d]\n", y_min, y_max);

  if (show_phase || show_ref)
    fprintf(stream, "set key\n");
  else
    fprintf(stream, "set nokey\n");

  if (show_phase)
    fprintf(stream, "set y2label 'Phase / '; set y2tics; set y2range [-200:200]\n");
  else
    fprintf(stream, "set y2label ''; set noy2tics\n");

  switch (mode)
   {
    case 0:
      fprintf(stream, "set title 'Frequenzgang der bertragungsfunktion' font 'Helvetica,14'\n"
                      "set ylabel '|H(f)| / dB'\n");
      break;
    case 1:
      fprintf(stream, "set title 'Frequenzgang der Impedanz' font 'Helvetica,14'\n"
                      "set ylabel '|Z(f)| / Ohm'\n");
      break;
    case 2:
      fprintf(stream, "set title 'Klirrfaktormessung: Ausgangsspektrum' font 'Helvetica,14'\n"
                      "set ylabel '|U(f)| / dB'\n");
      if (sine_freq <= 100)
        fprintf(stream, "set xrange [10:20000]\n");
   }

  return;
 }


/*---------- Diagramm (Spektrum) plotten ----------*/

void plot_freq_response(FILE *stream, char name[])
 {
  fprintf(stream, "set nolabel\n");

  if (data_points <= 0)
   {
    if (show_ref)
     {
      fprintf(stream, "plot '%s' using 1:2 title 'Referenz (Betrag)' with lines 3",
              ref_curve_name);
      if (show_phase)
        fprintf(stream, ", '%s' using 1:3 axes x1y2 title 'Referenz (Phase)' with lines 4",
	        ref_curve_name);
      fprintf(stream, "\n");
     }
    else
      fprintf(stream, "set nokey; plot sqrt(-1)\n");
    fflush(stream);
    return;
   }

  if (overload(u_t, data_points))
   {
    fprintf(stream, "set label 'ACHTUNG: Signal bersteuert!' at 1300,%d center font 'Helvetica,16'\n",
            (2*y_max+y_min)/3);
    printf("Uebersteuerung!\a\n");
   }

  if (mode == 2)		/*---- Klirrfaktormessung ----*/
    fprintf(stream, "set label 'Klirrfaktor: %.2f%% @ %d Hz' at 1100,%d font 'Helvetica,14'\n",
            dist*100.0, sine_freq, (4*y_max+y_min)/5);

  fprintf(stream, "plot '%s.dat' using 1:2 title 'Messung (Betrag)'", name);
  if (show_phase)
    fprintf(stream, ", '%s.dat' using 1:3 axes x1y2 title 'Messung (Phase)'", name);
  if (show_ref)
   {
    fprintf(stream, ", '%s' using 1:2 title 'Referenz (Betrag)' with lines 3", ref_curve_name);
    if (show_phase)
      fprintf(stream, ", '%s' using 1:3 axes x1y2 title 'Referenz (Phase)' with lines 4", ref_curve_name);
   }
  fprintf(stream, "\n");
  fflush(stream);

  return;
 }


/*---------- gemessene Kurve u(t) darstellen ----------*/

void plot_u_t(void)
 {
  FILE *stream;
  
  if ((stream = fopen("_tmp2.dat", "w")) == NULL)
   {
    perror("plot_u_t(): Can't open file");
    return;
   }

  write_record_data(stream);
  fclose(stream);

  fprintf(gnuplot, "set title 'gemessener Spannungsverlauf'\n"
          "set nologscale x; set autoscale x; set yrange [-1:1]\n"
	  "set key; set xlabel 't / s'; set data style lines\n"
	  "set ylabel 'u(t)/U_max'\n"
	  "plot '_tmp2.dat' every 2 using 1:2 title 'Messwerte (linker Kanal)', "
	  "'_tmp2.dat' every 2 using 1:3 title 'Sync (rechter Kanal)' with lines 3\n"
	  "!rm -f _tmp2.dat\n");
  fflush(gnuplot);

  return;
 }


/*---------- gemessene + gemittelte Kurve u(t) darstellen ----------*/

void plot_u_t2(int n)
 {
  int i;
  FILE *stream;
  
  if ((stream = fopen("_tmp2.dat", "w")) == NULL)
   {
    perror("plot_u_t2(): Can't open file");
    return;
   }

  fprintf(stream, "# time     value\n");
  for (i=0; i<n; i++)
    fprintf(stream, "%.6f  %7.4f\n", 1.0/SAMPLE_FREQ*i, u_t[i]);

  fclose(stream);

  fprintf(gnuplot, "set title 'gemessener Spannungsverlauf'\n"
          "set nologscale x; set autoscale x; set yrange [-1:1]\n"
	  "set nokey; set xlabel 't / s'; set data style lines\n"
	  "set ylabel 'u(t)/U_max'\n"
	  "plot '_tmp2.dat'; pause 3\n"
	  "!rm -f _tmp2.dat\n");
  fflush(gnuplot);

  return;
 }


/*---------- Funktion fuer Analyse-Auswahl ----------*/

void mode_proc(Panel_item item, int value, Event *event)
 {
  Panel_button_item button;
  Panel_numeric_text_item numeric_item;

  if (value == mode)
    return;
  else
    mode = value;

  if (mode == 2)	/* Klirrfaktor */
   {
    create_sine((double)sine_freq);
    button = xv_get(item, XV_KEY_DATA, KEY_BUTTON_PLAY);
    xv_set(button, PANEL_INACTIVE, FALSE, 0);
    button = xv_get(item, XV_KEY_DATA, KEY_BUTTON_LOOP);
    xv_set(button, PANEL_INACTIVE, FALSE, 0);
    button = xv_get(item, XV_KEY_DATA, KEY_BUTTON_ZERO);
    xv_set(button, PANEL_INACTIVE, TRUE, 0);
   }
  else
   {
    create_testsequence();	/* Chirp-Signal generieren */
    button = xv_get(item, XV_KEY_DATA, KEY_BUTTON_PLAY);
    xv_set(button, PANEL_INACTIVE, TRUE, 0);
    button = xv_get(item, XV_KEY_DATA, KEY_BUTTON_LOOP);
    xv_set(button, PANEL_INACTIVE, TRUE, 0);
    button = xv_get(item, XV_KEY_DATA, KEY_BUTTON_ZERO);
    xv_set(button, PANEL_INACTIVE, FALSE, 0);
   }
  button = xv_get(item, XV_KEY_DATA, KEY_BUTTON_STOP);
  xv_set(button, PANEL_INACTIVE, TRUE, 0);
  button = xv_get(item, XV_KEY_DATA, KEY_BUTTON_SAVE);
  xv_set(button, PANEL_INACTIVE, TRUE, 0);
  button = xv_get(item, XV_KEY_DATA, KEY_BUTTON_PRINT);
  xv_set(button, PANEL_INACTIVE, TRUE, 0);

  switch (mode)
   {
    case 0:		/* Ua/Ue */
      y_min = -20;
      y_max = 20;
      break;
    case 1:		/* Impedanzmessung */
      y_min = 0;
      y_max = 800;
      break;
    case 2:
      y_min = -60;
      y_max = 20;
   }
  numeric_item = xv_get(item, XV_KEY_DATA, KEY_MIN_Y_ITEM);
  xv_set(numeric_item, PANEL_VALUE, y_min, 0);
  numeric_item = xv_get(item, XV_KEY_DATA, KEY_MAX_Y_ITEM);
  xv_set(numeric_item, PANEL_VALUE, y_max, 0);

  setup_plot(gnuplot);
  if ((data_points > 0) || show_ref)
   {
    data_points = 0;
//    show_ref = 0;
    plot_freq_response(gnuplot, TMP_DATA_FILE);
   }

  return;
 }


/*---------- Funktion fuer PLAY-Button (Messung) ----------*/

void play_proc(Panel_item item, Event *event)
 {
  Xv_notice notice;
  Panel_item button;
  int n;

  if (snd_in < 0)
   {
    if (audio_open(&snd_in, &snd_out, &mixer))
     {
      audio_close(&snd_in, &snd_out, &mixer);	/* z. T. geoeffnete Devices wieder schliessen */
      notice = xv_get(item, XV_KEY_DATA, KEY_DEVS_NOTICE);
      xv_set(notice, XV_SHOW, TRUE, 0);
      return;
     }
    audio_init(snd_in, snd_out, mixer);
    set_levels();
   }

  data_points = acquire(snd_in, snd_out, u_t, MAX_DATA);	/* Messung */
  if (data_points <= 0)
   {
    plot_u_t();
    notice = xv_get(item, XV_KEY_DATA, KEY_ACQUIRE_NOTICE);
    xv_set(notice, XV_SHOW, TRUE, 0);
    button = xv_get(item, XV_KEY_DATA, KEY_BUTTON_SAVE);
    xv_set(button, PANEL_INACTIVE, TRUE, 0);
    button = xv_get(item, XV_KEY_DATA, KEY_BUTTON_PRINT);
    xv_set(button, PANEL_INACTIVE, TRUE, 0);
    setup_plot(gnuplot);
    return;
   }

  n = dft(u_t, u1_mag, u1_pha, data_points);

  if (mode == 2)	/* Bei Klirrfaktormessung u(t) fuer 3 Sek. anzeigen */
   {
    plot_u_t2(data_points);
    setup_plot(gnuplot);
   }

  switch (mode)
   {
    case 0:
      calc_gain(u0_mag, u0_pha, u1_mag, u1_pha, magnitude, phase, n);
      write_data(magnitude, phase, FREQ_L, FREQ_H, TMP_DATA_FILE ".dat");
      break;
    case 1:
      calc_Z(u0_mag, u0_pha, u1_mag, u1_pha, magnitude, phase, n, R0);
      write_data(magnitude, phase, FREQ_L, FREQ_H, TMP_DATA_FILE ".dat");
      break;
    case 2:
      calc_bode(u1_mag, u1_pha, magnitude, phase, n);
      write_data(magnitude, phase, 10, 20000, TMP_DATA_FILE ".dat");
      dist = distortion(u1_mag, n, sine_freq*CHIRP_LENGTH/SAMPLE_FREQ);
      break;
   }
  plot_freq_response(gnuplot, TMP_DATA_FILE);

  button = xv_get(item, XV_KEY_DATA, KEY_BUTTON_SAVE);
  xv_set(button, PANEL_INACTIVE, FALSE, 0);
  button = xv_get(item, XV_KEY_DATA, KEY_BUTTON_PRINT);
  xv_set(button, PANEL_INACTIVE, FALSE, 0);

  return;
 }


/*---------- Funktion fuer LOOP-Button (zykl. Messung) ----------*/

void measurement_cycle(void)
 {
  int n;

  data_points = acquire(snd_in, snd_out, u_t, MAX_DATA);	/* Messung */
  if (data_points <= 0)
   {
    fprintf(gnuplot, "set label 'Messung fehlgeschlagen!' at 1200,%d\n",
            (y_max+y_min)/2);
    plot_freq_response(gnuplot, TMP_DATA_FILE);
    setup_plot(gnuplot);
    return;
   }

  n = dft(u_t, u1_mag, u1_pha, data_points);

  switch (mode)
   {
    case 0:
      calc_gain(u0_mag, u0_pha, u1_mag, u1_pha, magnitude, phase, n);
      write_data(magnitude, phase, FREQ_L, FREQ_H, TMP_DATA_FILE ".dat");
      break;
    case 1:
      calc_Z(u0_mag, u0_pha, u1_mag, u1_pha, magnitude, phase, n, R0);
      write_data(magnitude, phase, FREQ_L, FREQ_H, TMP_DATA_FILE ".dat");
      break;
    case 2:
      calc_bode(u1_mag, u1_pha, magnitude, phase, n);
      write_data(magnitude, phase, 10, 20000, TMP_DATA_FILE ".dat");
      dist = distortion(u1_mag, n, sine_freq*CHIRP_LENGTH/SAMPLE_FREQ);
      break;
   }
  plot_freq_response(gnuplot, TMP_DATA_FILE);

  return;
 }

void loop_proc(Panel_item item, Event *event)
 {
  Frame frame;
  Xv_notice notice;
  Panel_item choice, button;
  static struct itimerval timer;

  frame = xv_get(item, XV_KEY_DATA, KEY_FRAME);

  if (snd_in < 0)
   {
    if (audio_open(&snd_in, &snd_out, &mixer))
     {
      audio_close(&snd_in, &snd_out, &mixer);	/* z. T. geoeffnete Devices wieder schliessen */
      notice = xv_get(item, XV_KEY_DATA, KEY_DEVS_NOTICE);
      xv_set(notice, XV_SHOW, TRUE, 0);
      return;
     }
    audio_init(snd_in, snd_out, mixer);
    set_levels();
   }

  choice = xv_get(item, XV_KEY_DATA, KEY_CHOICE_MODE);
  xv_set(choice, PANEL_INACTIVE, TRUE, 0);
  button = xv_get(item, XV_KEY_DATA, KEY_BUTTON_PLAY);
  xv_set(button, PANEL_INACTIVE, TRUE, 0);
  xv_set(item, PANEL_INACTIVE, TRUE, 0);
  button = xv_get(item, XV_KEY_DATA, KEY_BUTTON_ZERO);
  xv_set(button, PANEL_INACTIVE, TRUE, 0);
  button = xv_get(item, XV_KEY_DATA, KEY_BUTTON_STOP);
  xv_set(button, PANEL_INACTIVE, FALSE, 0);
  button = xv_get(item, XV_KEY_DATA, KEY_BUTTON_SAVE);
  xv_set(button, PANEL_INACTIVE, TRUE, 0);
  button = xv_get(item, XV_KEY_DATA, KEY_BUTTON_PRINT);
  xv_set(button, PANEL_INACTIVE, TRUE, 0);

  loop = 1;

  timer.it_value.tv_sec = 0;	/* Timer fuer zyklische Messung */
  timer.it_value.tv_usec = 100000;
  timer.it_interval.tv_sec = 1;
  timer.it_interval.tv_usec = 0;

  notify_set_itimer_func(frame, (Notify_func)measurement_cycle, ITIMER_REAL,
                         &timer, NULL);

  return;
 }


/*---------- Funktion fuer STOP-Button ----------*/

void stop_proc(Panel_item item, Event *event)
 {
  Frame frame;
  Panel_item button, choice;

  frame = xv_get(item, XV_KEY_DATA, KEY_FRAME);

  notify_set_itimer_func(frame, NOTIFY_FUNC_NULL, ITIMER_REAL, NULL, NULL);
  loop = 0;

  choice = xv_get(item, XV_KEY_DATA, KEY_CHOICE_MODE);
  xv_set(choice, PANEL_INACTIVE, FALSE, 0);
  button = xv_get(item, XV_KEY_DATA, KEY_BUTTON_PLAY);
  xv_set(button, PANEL_INACTIVE, FALSE, 0);
  button = xv_get(item, XV_KEY_DATA, KEY_BUTTON_LOOP);
  xv_set(button, PANEL_INACTIVE, FALSE, 0);
  xv_set(item, PANEL_INACTIVE, TRUE, 0);
  button = xv_get(item, XV_KEY_DATA, KEY_BUTTON_ZERO);
  xv_set(button, PANEL_INACTIVE, FALSE, 0);
  if (data_points > 0)
   {
    button = xv_get(item, XV_KEY_DATA, KEY_BUTTON_SAVE);
    xv_set(button, PANEL_INACTIVE, FALSE, 0);
    button = xv_get(item, XV_KEY_DATA, KEY_BUTTON_PRINT);
    xv_set(button, PANEL_INACTIVE, FALSE, 0);
   }

  return;
 }


/*---------- Funktion fuer ZERO-Button (Kalibrierung) ----------*/

void zero_proc(Panel_item item, Event *event)
 {
  int n;
  Xv_notice notice;
  Panel_button_item button;

  if (snd_in < 0)
   {
    if (audio_open(&snd_in, &snd_out, &mixer))
     {
      audio_close(&snd_in, &snd_out, &mixer);	/* z. T. geoeffnete Devices wieder schliessen */
      notice = xv_get(item, XV_KEY_DATA, KEY_DEVS_NOTICE);
      xv_set(notice, XV_SHOW, TRUE, 0);
      return;
     }
    audio_init(snd_in, snd_out, mixer);
    set_levels();
   }

  notice = xv_get(item, XV_KEY_DATA, KEY_CAL_NOTICE);
  xv_set(notice, XV_SHOW, TRUE, 0);

  if (data_points > 0)
   {
    data_points = 0;		/* aktuelle Messung ungueltig machen */
    plot_freq_response(gnuplot, TMP_DATA_FILE);
    button = xv_get(item, XV_KEY_DATA, KEY_BUTTON_SAVE);
    xv_set(button, PANEL_INACTIVE, TRUE, 0);
    button = xv_get(item, XV_KEY_DATA, KEY_BUTTON_PRINT);
    xv_set(button, PANEL_INACTIVE, TRUE, 0);
    usleep(200000);
   }

  n = acquire(snd_in, snd_out, u_t, MAX_DATA);	/* Messung */
  if (n <= 0)
   {
    plot_u_t();
    notice = xv_get(item, XV_KEY_DATA, KEY_ACQUIRE_NOTICE);
    xv_set(notice, XV_SHOW, TRUE, 0);
    setup_plot(gnuplot);
    return;
   }

  if (overload(u_t, n))
   {
    notice = xv_get(item, XV_KEY_DATA, KEY_OVERLOAD_NOTICE);
    xv_set(notice, XV_SHOW, TRUE, 0);
    return;
   }

  dft(u_t, u0_mag, u0_pha, n);

  button = xv_get(item, XV_KEY_DATA, KEY_BUTTON_PLAY);
  xv_set(button, PANEL_INACTIVE, FALSE, 0);
  button = xv_get(item, XV_KEY_DATA, KEY_BUTTON_LOOP);
  xv_set(button, PANEL_INACTIVE, FALSE, 0);

  return;
 }


/*---------- Funktion fuer SETUP-Button ----------*/

void setup_proc(Panel_item item, Event *event)
 {
  Frame frame_cmd;

  frame_cmd = xv_get(item, XV_KEY_DATA, KEY_FRAME_SETUP);
  xv_set(frame_cmd,
    FRAME_CMD_PUSHPIN_IN, TRUE,
    XV_SHOW,              TRUE,
    0);
  return;
 }


/*---------- Funktion fuer Auswahl "Phase anzeigen" ----------*/

void show_phase_proc(Panel_item item, int value, Event *event)
 {
  if (value == show_phase)
    return;
  show_phase = value;
  setup_plot(gnuplot);
  if (!loop && ((data_points > 0) || show_ref))
    plot_freq_response(gnuplot, TMP_DATA_FILE);
  return;
 }


/*---------- Funktion fuer Signalpegel-Auswahl ----------*/

void level_out_proc(Panel_item item, int value, Event *event)
 {
  Panel_button_item button;

  if (value == out_level)
    return;
  else
    out_level = value;

  if (mode == 1)	/* Impedanzmessung */
   {
    button = xv_get(item, XV_KEY_DATA, KEY_BUTTON_STOP);
    if (loop)
      stop_proc(button, event);
    xv_set(button, PANEL_INACTIVE, TRUE, 0);
    button = xv_get(item, XV_KEY_DATA, KEY_BUTTON_PLAY);
    xv_set(button, PANEL_INACTIVE, TRUE, 0);
    button = xv_get(item, XV_KEY_DATA, KEY_BUTTON_LOOP);
    xv_set(button, PANEL_INACTIVE, TRUE, 0);
   }

  set_levels();

  return;
 }


/*---------- Funktion fuer Eingangsverst.-Auswahl ----------*/

void level_in_proc(Panel_item item, int value, Event *event)
 {
  Panel_button_item button;

  if (value == rec_level)
    return;
  else
    rec_level = value;

  if (mode == 1)	/* Impedanzmessung */
   {
    button = xv_get(item, XV_KEY_DATA, KEY_BUTTON_STOP);
    if (loop)
      stop_proc(button, event);
    xv_set(button, PANEL_INACTIVE, TRUE, 0);
    button = xv_get(item, XV_KEY_DATA, KEY_BUTTON_PLAY);
    xv_set(button, PANEL_INACTIVE, TRUE, 0);
    button = xv_get(item, XV_KEY_DATA, KEY_BUTTON_LOOP);
    xv_set(button, PANEL_INACTIVE, TRUE, 0);
   }

  set_levels();

  return;
 }


/*---------- Funktion fuer Frequenz-Eingabe ----------*/

Panel_setting freq_proc(Panel_item item, Event *event)
 {
  int value, n;
  double f_step;

  value = (int)xv_get(item, PANEL_VALUE);
  if (value == sine_freq)
    return(PANEL_NEXT);

  f_step = 1.0*SAMPLE_FREQ/CHIRP_LENGTH;
  if (value < f_step)
    sine_freq = f_step;
  else
   {
    if (value == sine_freq + 1)
      value += f_step;
    n = value/f_step;
    sine_freq = f_step*n;
   }
  value = 0.5 + sine_freq;
  xv_set(item, PANEL_VALUE, value, 0);
  if (mode == 2)	/* Klirrfaktor-Messung ausgewaehlt */
   {
    if (data_points > 0)
     {
      data_points = 0;
      if (!loop)
        plot_freq_response(gnuplot, TMP_DATA_FILE);
     }
    create_sine((double)sine_freq);
   }
  return(PANEL_NEXT);
 }


/*---------- Funktion fuer Grenzen der Y-Achse ----------*/

Panel_setting y_proc(Panel_item item, Event *event)
 {
  int value;
  char *label;

  value = (int)xv_get(item, PANEL_VALUE);
  label = (char *)xv_get(item, PANEL_LABEL_STRING);
  if (strcmp(label, "bis:") == 0)
   {					/* y-Max-Wert */
    if (value == y_max)
      return(PANEL_PREVIOUS);
    if (value <= y_min)
     {
      value = y_min + 1;
      xv_set(item, PANEL_VALUE, value, 0);
     }
    y_max = value;
    setup_plot(gnuplot);
    if ((data_points > 0) || show_ref)
      plot_freq_response(gnuplot, TMP_DATA_FILE);
    return(PANEL_PREVIOUS);
   }
  else
   {					/* y-Min-Wert */
    if (value == y_min)
      return(PANEL_NEXT);
    if (value >= y_max)
     {
      value = y_max - 1;
      xv_set(item, PANEL_VALUE, value, 0);
     }
    y_min = value;
    setup_plot(gnuplot);
    if ((data_points > 0) || show_ref)
      plot_freq_response(gnuplot, TMP_DATA_FILE);
    return(PANEL_NEXT);
   }
 }


/*---------- Funktion fuer QUIT-Button ----------*/

void quit_proc(Panel_item item, Event *event)
 {
  Frame frame;
  Xv_notice notice;
  int choice;

  frame = xv_get(item, XV_KEY_DATA, KEY_FRAME);
  notice = xv_get(item, XV_KEY_DATA, KEY_QUIT_NOTICE);
  xv_set(notice,
    XV_SHOW,       TRUE,
    NOTICE_STATUS, &choice,
    0);
  if (choice == NOTICE_YES)
   {
    audio_close(&snd_in, &snd_out, &mixer);
    if (help_pid > 0)
     {
      kill(help_pid, SIGKILL);
      waitpid(help_pid, NULL, 0);
      help_pid = 0;
     }
    xv_destroy_safe(frame);
   }
  return;
 }

/*---------- Funktion fuer SAVE-Button ----------*/

void save_proc(Panel_item item, Event *event)
 {
  File_chooser save_chooser;

  save_chooser = xv_get(item, XV_KEY_DATA, KEY_SAVE_CHOOSER);
  xv_set(save_chooser, FILE_CHOOSER_UPDATE, XV_SHOW, TRUE, 0);
  return;
 }


/*---------- Funktion fuer SAVE-Button im File Chooser ----------*/

int save_data_proc(File_chooser file_ch, char *filename)
 {
  int len, i;
  static char buffer[128], name[80], *short_name;
  FILE *stream;

  len = strlen(filename);
  if ((len > 4) && (strcmp(&(filename[len-4]), ".dat") == 0))
   {
    if (len > 83)
      len = 83;
    strncpy(name, filename, len-4);
    name[len-4] = '\0';
   }
  else
   {
    strncpy(name, filename, 79);
    name[79] = '\0';
   }

  for (i=strlen(name)-1; i>0; i--)	/* Name ohne Pfad extrahieren */
    if (name[i-1] == '/')
      break;
  short_name = &(name[i]);

  snprintf(buffer, 128, "cp -f %s.dat %s.dat", TMP_DATA_FILE, name);
  system(buffer);

  snprintf(buffer, 128, "%s.plt", name);
  if ((stream = fopen(buffer, "w")) == NULL)
   {
    perror("save_data_proc(): Can't open output file");
    return(XV_ERROR);
   }
  setup_plot(stream);
  plot_freq_response(stream, short_name);
  fprintf(stream, "pause -1\n");
  fclose(stream);
  return(XV_OK);
 }


/*---------- Funktion fuer Referenzkurve-Auswahl ----------*/

void show_ref_proc(Panel_item item, int value, Event *event)
 {
  File_chooser load_chooser;
  Panel_text_item ref_name_item;

  load_chooser = xv_get(item, XV_KEY_DATA, KEY_LOAD_CHOOSER);
  ref_name_item = xv_get(item, XV_KEY_DATA, KEY_REF_CURVE_ITEM);
  if (value)
   {
    if (ref_curve_name[0] == '\0')
     {
      xv_set(item, PANEL_VALUE, 0, 0);
      xv_set(load_chooser,
	XV_KEY_DATA, KEY_REF_CURVE_ITEM, ref_name_item,
	XV_KEY_DATA, KEY_SHOW_REF_ITEM, item,
	FILE_CHOOSER_UPDATE,
	XV_SHOW, TRUE,
	0);
     }
    else
     {
      show_ref = 1;
      setup_plot(gnuplot);
      plot_freq_response(gnuplot, TMP_DATA_FILE);
     }
   }
  else
   {
    show_ref = 0;
    xv_set(load_chooser, XV_SHOW, FALSE, 0);
    setup_plot(gnuplot);
    plot_freq_response(gnuplot, TMP_DATA_FILE);
   }
  return;
 }


/*---------- Funktion fuer BROWSE-Button (Ref.-Kurve) ----------*/

void load_ref_proc(Panel_item item, Event *event)
 {
  File_chooser load_chooser;
  Panel_text_item ref_name_item;
  Panel_choice_item show_ref_item;

  load_chooser = xv_get(item, XV_KEY_DATA, KEY_LOAD_CHOOSER);
  ref_name_item = xv_get(item, XV_KEY_DATA, KEY_REF_CURVE_ITEM);
  show_ref_item = xv_get(item, XV_KEY_DATA, KEY_SHOW_REF_ITEM);
  xv_set(load_chooser,
    XV_KEY_DATA, KEY_REF_CURVE_ITEM, ref_name_item,
    XV_KEY_DATA, KEY_SHOW_REF_ITEM, show_ref_item,
    FILE_CHOOSER_UPDATE,
    XV_SHOW, TRUE,
    0);
  return;
 }


/*---------- Funktion fuer LOAD-Button im File Chooser ----------*/

int set_ref_curve(File_chooser file_ch, char *filename)
 {
  Panel_text_item ref_name_item;
  Panel_choice_item show_ref_choice;
 
  ref_name_item = xv_get(file_ch, XV_KEY_DATA, KEY_REF_CURVE_ITEM);
  show_ref_choice = xv_get(file_ch, XV_KEY_DATA, KEY_SHOW_REF_ITEM);
  xv_set(ref_name_item, PANEL_VALUE, filename, 0);
  xv_set(show_ref_choice, PANEL_VALUE, 1, 0);
  show_ref = 1;
  strncpy(ref_curve_name, filename, 256);
  setup_plot(gnuplot);
  plot_freq_response(gnuplot, TMP_DATA_FILE);
  return(XV_OK);
 }


/*---------- Funktion fuer Referenzkurven-Name ----------*/

Panel_setting ref_name_proc(Panel_item item, Event *event)
 {
  char *value;
  Panel_choice_item show_ref_choice;

  value = (char *)xv_get(item, PANEL_VALUE);
  show_ref_choice = xv_get(item, XV_KEY_DATA, KEY_SHOW_REF_ITEM);
  strncpy(ref_curve_name, value, 256);
  if (value[0] == '\0')
   {
    if (show_ref)
     {
      xv_set(show_ref_choice, PANEL_VALUE, 0, 0);
      show_ref = 0;
      setup_plot(gnuplot);
      if (!loop)
        plot_freq_response(gnuplot, TMP_DATA_FILE);
     }
   }
  else
   {
    xv_set(show_ref_choice, PANEL_VALUE, 1, 0);
    show_ref = 1;
    setup_plot(gnuplot);
    if (!loop)
      plot_freq_response(gnuplot, TMP_DATA_FILE);
   }
  return(PANEL_INSERT);
 }


/*---------- Funktion fuer PRINT-Button ----------*/

void print_setup_proc(Panel_item item, Event *event)
 {
  Frame frame_print;

  frame_print = xv_get(item, XV_KEY_DATA, KEY_FRAME_PRINT);
  xv_set(frame_print, XV_SHOW, TRUE, 0);
  return;
 }


/*---------- Funktion fuer Drucken-Button ----------*/

void print_proc(Panel_item item, Event *event)
 {
  Panel_item title_item, printer_item;
  char *title, *printer, *subtitle;

  title_item = xv_get(item, XV_KEY_DATA, KEY_TITLE_ITEM);
  title = (char *)xv_get(title_item, PANEL_VALUE);
  printer_item = xv_get(item, XV_KEY_DATA, KEY_PRINTER_ITEM);
  printer = (char *)xv_get(printer_item, PANEL_VALUE);
  if (strlen(title) > 0)
   {
    switch (mode)
     {
      case 0:
	subtitle = "Frequenzgang der bertragungsfunktion";
	break;
      case 1:
	subtitle = "Frequenzgang der Impedanz";
	break;
      case 2:
        subtitle = "Klirrfaktormessung: Ausgangsspektrum";
     }
    fprintf(gnuplot, "set title \"%s\\n%s\" font 'Helvetica,14'\n", title, subtitle);
   }
  fprintf(gnuplot, "set encoding iso_8859_1\n"
          "set terminal postscript landscape color solid 'Helvetica' 12\n"
	  "set output '%s'\n", printer);
  fprintf(gnuplot, "replot\nset output\n");
  fflush(gnuplot);
  fprintf(gnuplot, "set terminal x11\n");
  setup_plot(gnuplot);
  plot_freq_response(gnuplot, TMP_DATA_FILE);

  return;
 }


/*---------- Funktion fuer HELP-Button ----------*/

void help_proc(Panel_item item, Event *event)
 {
  if (help_pid > 0)		/* ggf. alten Hilfe-Prozess beenden */
   {
    kill(help_pid, SIGKILL);
    waitpid(help_pid, NULL, 0);
    help_pid = 0;
   }

  if ((help_pid = fork()) == 0)
   {
    int fd_null;

    fd_null = open("/dev/null", O_WRONLY);
    dup2(fd_null, STDOUT_FILENO);
    dup2(fd_null, STDERR_FILENO);
    execlp("dillo", "dillo", "-f", HELP_PATH "help.html", NULL);
    execlp(BROWSER1, BROWSER1, HELP_PATH "help.html", NULL);
    execlp(BROWSER2, BROWSER2, HELP_PATH "help.html", NULL);
    exit(0);
   }
  sleep(1);	/* auf Browser warten */
  return;
 }

/*---------- Funktion fuer Schliessen des CMD-Frame ----------*/

void done_proc(Frame frame_cmd)
 {
  xv_set(frame_cmd, XV_SHOW, FALSE, 0);
  return;
 }

/*---------- XView-Objekte erzeugen ----------*/

Frame create_objects(void)
 {
  Frame frame, frame_setup, frame_print;
  Panel panel, panel_setup, panel_print;
  Xv_notice notice_quit, notice_devs, notice_cal, notice_acquire, notice_overload;
  File_chooser save_chooser, load_chooser;
  Panel_choice_item mode_choice, level_in_choice, level_out_choice,
    show_phase_choice, show_ref_choice;
  Panel_numeric_text_item freq_item, min_y_item, max_y_item;
  Panel_text_item ref_name_item, title_item, printer_item;
  Panel_button_item play_button, loop_button, stop_button, zero_button,
    setup_button, save_button, print_button, help_button, quit_button,
    print2_button, load_ref_button;
  Server_image image_play, image_loop, image_stop, image_zero, image_setup,
    image_save, image_print, image_help, image_quit;
  static unsigned short
    image_bits_play[32] =
     {
      # include "play.icon"
     },
    image_bits_loop[32] =
     {
      # include "loop.icon"
     },
    image_bits_stop[32] =
     {
      # include "stop.icon"
     },
    image_bits_zero[32] =
     {
      # include "zero.icon"
     },
    image_bits_setup[32] =
     {
      # include "setup.icon"
     },
    image_bits_save[32] =
     {
      # include "save.icon"
     },
    image_bits_print[32] =
     {
      # include "print.icon"
     },
    image_bits_help[32] =
     {
      # include "help.icon"
     },
    image_bits_quit[32] =
     {
      # include "quit.icon"
     };

	/*- - - - - - - - - - - - - - - - -*/

  loop = 0;			/* zyklische Messung inaktiv */

	/*- - - - - - - - - - - - - - - - -*/

  frame = xv_create(0, FRAME,
    XV_X, 80,
    XV_Y, 2,
    FRAME_LABEL, "XDAMS - digitales Akustik-Messsystem",
    FRAME_SHOW_RESIZE_CORNER, FALSE,
    0);

	/*- - - - - - - - - - - - - - - - -*/

  notice_quit = xv_create(frame, NOTICE,
    NOTICE_MESSAGE_STRING, "Wollen Sie wirklich beenden?",
    NOTICE_BUTTON_YES,     "  Ja  ",
    NOTICE_BUTTON_NO,      " Nein ",
    0);

  notice_devs = xv_create(frame, NOTICE,
    NOTICE_MESSAGE_STRINGS, "Fehler beim ffnen der Sound-Devices!\n ",
                            "Schlieen Sie die Anwendung, die die Soundkarte",
			    "belegt und versuchen Sie es erneut!", NULL,
    NOTICE_BUTTON,          "  OK  ", 100,
    0);

  notice_cal = xv_create(frame, NOTICE,
    NOTICE_MESSAGE_STRINGS, "Kalibrierung der Messanordnung durchfhren:\n ",
                            "Entfernen Sie die Last (Impedanzmessung) bzw.",
			    "verbinden Sie Ue mit Ua (bertragungsfunktion).", NULL,
    NOTICE_BUTTON,          "  OK  ", 100,
    0);

  notice_acquire = xv_create(frame, NOTICE,
    NOTICE_MESSAGE_STRINGS, "Fehler beim Einlesen von Daten!\n ",
                            "Mglicherweise konnte kein Sync-Signal gefunden werden.",
			    "Bitte die Anschlsse prfen und erneut versuchen.", NULL,
    NOTICE_BUTTON,          "  OK  ", 100,
    0);

  notice_overload = xv_create(frame, NOTICE,
    NOTICE_MESSAGE_STRINGS, "Das Eingangssignal ist bersteuert!\n ",
                            "Bitte den Signalpegel reduzieren und erneut kalibrieren.", NULL,
    NOTICE_BUTTON,          "  OK  ", 100,
    0);

	/*- - - - - - - - - - - - - - - - -*/

  save_chooser = xv_create(frame, FILE_CHOOSER,
    FILE_CHOOSER_TYPE,        FILE_CHOOSER_SAVEAS,
    FILE_CHOOSER_NOTIFY_FUNC, save_data_proc,
    0);

  load_chooser = xv_create(frame, FILE_CHOOSER,
    FILE_CHOOSER_TYPE,        FILE_CHOOSER_OPEN,
    FILE_CHOOSER_NOTIFY_FUNC, set_ref_curve,
    FILE_CHOOSER_FILTER_STRING, ".*[.]dat",
    0);


	/*- - - - - - - - - - - - - - - - -*/

  frame_print = xv_create(frame, FRAME_CMD,
    XV_X,            220,
    XV_Y,            200,
    FRAME_LABEL,     "Diagramm drucken",
    FRAME_DONE_PROC, done_proc,
    XV_SHOW,         FALSE,
    0);

  panel_print = xv_get(frame_print, FRAME_CMD_PANEL);

  title_item = xv_create(panel_print, PANEL_TEXT,
    PANEL_ITEM_Y,              20,
    PANEL_LABEL_STRING,        "berschrift:",
    PANEL_VALUE_DISPLAY_WIDTH, 250,
    PANEL_VALUE,               "",
    PANEL_VALUE_STORED_LENGTH, 256,
    0);

  printer_item = xv_create(panel_print, PANEL_TEXT,
    PANEL_NEXT_ROW,            -1,
    PANEL_LABEL_STRING,        "Drucker:",
    PANEL_VALUE_DISPLAY_WIDTH, 200,
    PANEL_VALUE,               "|lpr",
    PANEL_VALUE_STORED_LENGTH, 256,
    0);

  print2_button = xv_create(panel_print, PANEL_BUTTON,
    PANEL_LABEL_STRING, "Drucken",
    PANEL_NOTIFY_PROC,  print_proc,
    XV_KEY_DATA, KEY_TITLE_ITEM, title_item,
    XV_KEY_DATA, KEY_PRINTER_ITEM, printer_item,
    PANEL_ACCEPT_KEYSTROKE, TRUE,
    0);

  xv_set(panel_print, PANEL_DEFAULT_ITEM, print2_button, 0);

  window_fit(panel_print);
  window_fit(frame_print);

	/*- - - - - - - - - - - - - - - - -*/

  frame_setup = xv_create(frame, FRAME_CMD,
    XV_X,            440,
    XV_Y,            100,
    FRAME_LABEL,     "Einstellungen",
    FRAME_DONE_PROC, done_proc,
    XV_SHOW,         FALSE,
    0);

  panel_setup = xv_get(frame_setup, FRAME_CMD_PANEL);

  xv_create(panel_setup, PANEL_MESSAGE,
    PANEL_LABEL_STRING, "Messung:",
    0);

  level_out_choice = xv_create(panel_setup, PANEL_CHOICE,
    PANEL_DISPLAY_LEVEL,  PANEL_CURRENT,
    PANEL_NEXT_ROW,       -1,
    PANEL_LABEL_STRING,   "Signalpegel:",
    PANEL_CHOICE_STRINGS, "klein (-20 dB)", "mittel (-10 dB)", "hoch", NULL,
    PANEL_VALUE,          2,
    PANEL_DEFAULT_VALUE,  2,
    PANEL_NOTIFY_PROC,    level_out_proc,
    0);
  out_level = 2;

  level_in_choice = xv_create(panel_setup, PANEL_CHOICE,
    PANEL_DISPLAY_LEVEL,  PANEL_CURRENT,
    PANEL_NEXT_ROW,       -1,
    PANEL_LABEL_STRING,   "Eingangsverstrkung:",
    PANEL_CHOICE_STRINGS, "klein (-10 dB)", "mittel", "hoch (+6 dB)", NULL,
    PANEL_VALUE,          1,
    PANEL_DEFAULT_VALUE,  1,
    PANEL_NOTIFY_PROC,    level_in_proc,
    0);
  rec_level = 1;

  xv_create(panel_setup, PANEL_MESSAGE,
    PANEL_NEXT_ROW,     -1,
    PANEL_LABEL_STRING, "_______________________________________",
    0);

  xv_create(panel_setup, PANEL_MESSAGE,
    PANEL_NEXT_ROW,            -1,
    PANEL_LABEL_STRING, "Fr Klirrfaktor-Messung:",
    0);

  freq_item = xv_create(panel_setup, PANEL_NUMERIC_TEXT,
    PANEL_NEXT_ROW,            -1,
    PANEL_LABEL_STRING,        "Frequenz/Hz:",
    PANEL_VALUE_DISPLAY_WIDTH, 50,
    PANEL_MIN_VALUE,           20,
    PANEL_MAX_VALUE,           10000,
    PANEL_VALUE,               500,
    PANEL_NOTIFY_PROC,         freq_proc,
    0);
  sine_freq = 500;

  xv_create(panel_setup, PANEL_MESSAGE,
    PANEL_NEXT_ROW,     -1,
    PANEL_LABEL_STRING, "_______________________________________",
    0);

  xv_create(panel_setup, PANEL_MESSAGE,
    PANEL_NEXT_ROW,            -1,
    PANEL_LABEL_STRING, "Diagramm-Einstellungen:",
    0);

  min_y_item = xv_create(panel_setup, PANEL_NUMERIC_TEXT,
    PANEL_NEXT_ROW,            -1,
    PANEL_LABEL_STRING,        "Y-Achse von:",
    PANEL_VALUE_DISPLAY_WIDTH, 50,
    PANEL_MIN_VALUE,           -80,
    PANEL_MAX_VALUE,           10000,
    PANEL_VALUE,               -20,
    PANEL_NOTIFY_PROC,         y_proc,
    0);
  y_min = -20;

  max_y_item = xv_create(panel_setup, PANEL_NUMERIC_TEXT,
    PANEL_LABEL_STRING,        "bis:",
    PANEL_VALUE_DISPLAY_WIDTH, 50,
    PANEL_MIN_VALUE,           0,
    PANEL_MAX_VALUE,           10000,
    PANEL_VALUE,               20,
    PANEL_NOTIFY_PROC,         y_proc,
    0);
  y_max = 20;

  show_phase_choice = xv_create(panel_setup, PANEL_CHECK_BOX,
    PANEL_NEXT_ROW,       -1,
    PANEL_LABEL_STRING,   "Phase anzeigen:",
    PANEL_VALUE,          0,
    PANEL_NOTIFY_PROC,    show_phase_proc,
    0);
  show_phase = 0;

/*
  xv_create(panel_setup, PANEL_MESSAGE,
    PANEL_NEXT_ROW,     -1,
    PANEL_LABEL_STRING, "_______________________________________",
    0);
*/

  show_ref_choice = xv_create(panel_setup, PANEL_CHECK_BOX,
    PANEL_NEXT_ROW,       -1,
    PANEL_LABEL_STRING,   "Referenzkurve zeigen:",
    PANEL_VALUE,          0,
    PANEL_NOTIFY_PROC,    show_ref_proc,
    XV_KEY_DATA, KEY_LOAD_CHOOSER, load_chooser,
    0);
  show_ref = 0;

  load_ref_button = xv_create(panel_setup, PANEL_BUTTON,
    PANEL_LABEL_STRING, "Datei whlen",
    PANEL_NOTIFY_PROC,  load_ref_proc,
    XV_KEY_DATA,        KEY_LOAD_CHOOSER, load_chooser,
    XV_KEY_DATA,        KEY_SHOW_REF_ITEM, show_ref_choice,
    0);

  ref_name_item = xv_create(panel_setup, PANEL_TEXT,
    PANEL_NEXT_ROW,            -1,
    PANEL_LABEL_STRING,        "Dateiname:",
    PANEL_VALUE_DISPLAY_WIDTH, 220,
    PANEL_VALUE,               "",
    PANEL_VALUE_STORED_LENGTH, 256,
    PANEL_NOTIFY_PROC,         ref_name_proc,
    XV_KEY_DATA,               KEY_SHOW_REF_ITEM, show_ref_choice,
    0);
  ref_curve_name[0] = '\0';

  xv_set(show_ref_choice, XV_KEY_DATA, KEY_REF_CURVE_ITEM, ref_name_item, 0);
  xv_set(load_ref_button, XV_KEY_DATA, KEY_REF_CURVE_ITEM, ref_name_item, 0);

  window_fit(panel_setup);
  window_fit(frame_setup);

	/*- - - - - - - - - - - - - - - - -*/

  panel = xv_create(frame, PANEL, 0);

  mode_choice = xv_create(panel, PANEL_CHOICE,
    PANEL_DISPLAY_LEVEL,  PANEL_CURRENT,
    PANEL_ITEM_Y,         14,
    PANEL_LABEL_STRING,   "Analyse:",
    PANEL_CHOICE_STRINGS, "Ua/Ue", "Impedanz", "Klirrfaktor", NULL,
    PANEL_VALUE,          0,
    PANEL_DEFAULT_VALUE,  0,
    PANEL_NOTIFY_PROC,    mode_proc,
    0);
  mode = 0;			/* Ua/Ue */
  create_testsequence();	/* Chirp-Signal generieren */

  image_play = xv_create(0, SERVER_IMAGE,
    XV_WIDTH,          32,
    XV_HEIGHT,         16,
    SERVER_IMAGE_BITS, image_bits_play,
    0);

  play_button = xv_create(panel, PANEL_BUTTON,
    PANEL_LABEL_IMAGE, image_play,
    PANEL_NOTIFY_PROC, play_proc,
    PANEL_INACTIVE, TRUE,
    XV_KEY_DATA, KEY_DEVS_NOTICE, notice_devs,
    XV_KEY_DATA, KEY_ACQUIRE_NOTICE, notice_acquire,
    0);

  image_loop = xv_create(0, SERVER_IMAGE,
    XV_WIDTH,          32,
    XV_HEIGHT,         16,
    SERVER_IMAGE_BITS, image_bits_loop,
    0);

  loop_button = xv_create(panel, PANEL_BUTTON,
    PANEL_LABEL_IMAGE,  image_loop,
    PANEL_NOTIFY_PROC,  loop_proc,
    XV_KEY_DATA, KEY_FRAME, frame,
    PANEL_INACTIVE, TRUE,
    XV_KEY_DATA, KEY_DEVS_NOTICE, notice_devs,
    0);

  image_stop = xv_create(0, SERVER_IMAGE,
    XV_WIDTH,          32,
    XV_HEIGHT,         16,
    SERVER_IMAGE_BITS, image_bits_stop,
    0);

  stop_button = xv_create(panel, PANEL_BUTTON,
    PANEL_LABEL_IMAGE,  image_stop,
    PANEL_NOTIFY_PROC,  stop_proc,
    PANEL_INACTIVE, TRUE,
    0);

  image_zero = xv_create(0, SERVER_IMAGE,
    XV_WIDTH,          32,
    XV_HEIGHT,         16,
    SERVER_IMAGE_BITS, image_bits_zero,
    0);

  zero_button = xv_create(panel, PANEL_BUTTON,
    PANEL_LABEL_IMAGE,  image_zero,
    PANEL_NOTIFY_PROC,  zero_proc,
    XV_KEY_DATA, KEY_DEVS_NOTICE, notice_devs,
    XV_KEY_DATA, KEY_CAL_NOTICE, notice_cal,
    XV_KEY_DATA, KEY_ACQUIRE_NOTICE, notice_acquire,
    XV_KEY_DATA, KEY_OVERLOAD_NOTICE, notice_overload,
    0);

  image_setup = xv_create(0, SERVER_IMAGE,
    XV_WIDTH,          32,
    XV_HEIGHT,         16,
    SERVER_IMAGE_BITS, image_bits_setup,
    0);

  setup_button = xv_create(panel, PANEL_BUTTON,
    PANEL_LABEL_IMAGE, image_setup,
    PANEL_NOTIFY_PROC, setup_proc,
    XV_KEY_DATA,       KEY_FRAME_SETUP, frame_setup,
    PANEL_ACCEPT_KEYSTROKE, TRUE,
    0);

  image_save = xv_create(0, SERVER_IMAGE,
    XV_WIDTH,          32,
    XV_HEIGHT,         16,
    SERVER_IMAGE_BITS, image_bits_save,
    0);

  save_button = xv_create(panel, PANEL_BUTTON,
    PANEL_LABEL_IMAGE,  image_save,
    PANEL_NOTIFY_PROC,  save_proc,
    XV_KEY_DATA, KEY_SAVE_CHOOSER, save_chooser,
    PANEL_INACTIVE, TRUE,
    0);

  image_print = xv_create(0, SERVER_IMAGE,
    XV_WIDTH,          32,
    XV_HEIGHT,         16,
    SERVER_IMAGE_BITS, image_bits_print,
    0);

  print_button = xv_create(panel, PANEL_BUTTON,
    PANEL_LABEL_IMAGE,  image_print,
    PANEL_NOTIFY_PROC,  print_setup_proc,
    XV_KEY_DATA, KEY_FRAME_PRINT, frame_print,
    PANEL_INACTIVE, TRUE,
    0);

  image_help = xv_create(0, SERVER_IMAGE,
    XV_WIDTH,          32,
    XV_HEIGHT,         16,
    SERVER_IMAGE_BITS, image_bits_help,
    0);

  help_button = xv_create(panel, PANEL_BUTTON,
    PANEL_LABEL_IMAGE,  image_help,
    PANEL_NOTIFY_PROC,  help_proc,
    PANEL_ACCEPT_KEYSTROKE, TRUE,
    0);

  image_quit = xv_create(0, SERVER_IMAGE,
    XV_WIDTH,          32,
    XV_HEIGHT,         16,
    SERVER_IMAGE_BITS, image_bits_quit,
    0);

  quit_button = xv_create(panel, PANEL_BUTTON,
    PANEL_LABEL_IMAGE,  image_quit,
    PANEL_NOTIFY_PROC,  quit_proc,
    XV_KEY_DATA,        KEY_FRAME, frame,
    XV_KEY_DATA,        KEY_QUIT_NOTICE, notice_quit,
    0);

  xv_set(mode_choice,
    XV_KEY_DATA, KEY_BUTTON_PLAY, play_button,
    XV_KEY_DATA, KEY_BUTTON_LOOP, loop_button,
    XV_KEY_DATA, KEY_BUTTON_STOP, stop_button,
    XV_KEY_DATA, KEY_BUTTON_ZERO, zero_button,
    XV_KEY_DATA, KEY_BUTTON_SAVE, save_button,
    XV_KEY_DATA, KEY_BUTTON_PRINT, print_button,
    XV_KEY_DATA, KEY_MIN_Y_ITEM, min_y_item,
    XV_KEY_DATA, KEY_MAX_Y_ITEM, max_y_item,
    0);

  xv_set(play_button,
    XV_KEY_DATA, KEY_BUTTON_SAVE, save_button,
    XV_KEY_DATA, KEY_BUTTON_PRINT, print_button,
    0);

  xv_set(loop_button,
    XV_KEY_DATA, KEY_FRAME, frame,
    XV_KEY_DATA, KEY_CHOICE_MODE, mode_choice,
    XV_KEY_DATA, KEY_BUTTON_PLAY, play_button,
    XV_KEY_DATA, KEY_BUTTON_STOP, stop_button,
    XV_KEY_DATA, KEY_BUTTON_ZERO, zero_button,
    XV_KEY_DATA, KEY_BUTTON_SAVE, save_button,
    XV_KEY_DATA, KEY_BUTTON_PRINT, print_button,
    0);

  xv_set(stop_button,
    XV_KEY_DATA, KEY_FRAME, frame,
    XV_KEY_DATA, KEY_CHOICE_MODE, mode_choice,
    XV_KEY_DATA, KEY_BUTTON_PLAY, play_button,
    XV_KEY_DATA, KEY_BUTTON_LOOP, loop_button,
    XV_KEY_DATA, KEY_BUTTON_ZERO, zero_button,
    XV_KEY_DATA, KEY_BUTTON_SAVE, save_button,
    XV_KEY_DATA, KEY_BUTTON_PRINT, print_button,
    0);

  xv_set(zero_button,
    XV_KEY_DATA, KEY_BUTTON_PLAY, play_button,
    XV_KEY_DATA, KEY_BUTTON_LOOP, loop_button,
    XV_KEY_DATA, KEY_BUTTON_STOP, stop_button,
    XV_KEY_DATA, KEY_BUTTON_SAVE, save_button,
    XV_KEY_DATA, KEY_BUTTON_PRINT, print_button,
    0);

  xv_set(level_out_choice,
    XV_KEY_DATA, KEY_BUTTON_PLAY, play_button,
    XV_KEY_DATA, KEY_BUTTON_LOOP, loop_button,
    XV_KEY_DATA, KEY_BUTTON_STOP, stop_button,
    0);

  xv_set(level_in_choice,
    XV_KEY_DATA, KEY_BUTTON_PLAY, play_button,
    XV_KEY_DATA, KEY_BUTTON_LOOP, loop_button,
    XV_KEY_DATA, KEY_BUTTON_STOP, stop_button,
    0);

  window_fit(panel);
  window_fit(frame);
  return(frame);
 }

/*---------- Hauptschleife des GUI ----------*/

void gui_main(int argc, char *argv[])
 {
  Frame frame;

  xv_init(XV_INIT_ARGS, argc, argv, XV_USE_LOCALE, TRUE, 0);

  if (setlocale(LC_NUMERIC, "en_US") == NULL)
    setlocale(LC_NUMERIC, "UTF-8");

  snd_in = snd_out = mixer = -1;

  if ((gnuplot = popen("gnuplot", "w")) == NULL)
   {
    perror("plot_u_t(): Can't run 'gnuplot'");
    return;
   }
  data_points = 0;

  help_pid = 0;

  frame = create_objects();
  if (frame)
   {
    setup_plot(gnuplot);
    xv_main_loop(frame);
   }

  pclose(gnuplot);
  remove(TMP_DATA_FILE ".dat");
  return;
 }
